/************************************************************************
                        SYST
                Host System Interface
 ************************************************************************
 Copyright (c) SMARTWARE, 1999-2006 All Rights Reserved.
 ************************************************************************
 25/02/2009 - Marc LEGOUX - V2r23p: Add syst__SignalErr
 13/02/2006 - Marc LEGOUX - c6r22 : Add syst_Execute
 06/04/2005 - Marc LEGOUX - c6r19 : Change some unsigned values to ulong
 25/01/2005 - Marc LEGOUX - c6r18 : Add _TaskSuspend, _TaskResume
 25/11/2004 - Marc LEGOUX - c6r18 : Add _TaskInfoGet, _TaskListGet. Hide syst_TaskType
 20/10/2004 - Marc LEGOUX - c6r17 : Add _EventOpenN
 27/01/2004 - Marc LEGOUX - c6r14 : Improve syst_MemInfo
 18/12/2003 - Marc LEGOUX - c6r13 : Add syst_MemReAlloc
 17/12/2003 - Marc LEGOUX - c6r12 : Add syst_MemBlkInfo
 21/05/2003 - Marc LEGOUX - c6r09 : Add syst__Version
 25/02/2003 - Marc LEGOUX - c6r09 : Modify syst_PrintCfgType
 15/11/2002 - Marc LEGOUX - c6r07 : Pragma far-absolute for syst_TaskCur
 15/05/2002 - Marc LEGOUX - c6r07 : Add __Access __AccessExists
 08/02/2002 - Marc LEGOUX - c6r07 : Add _TaskRefDel()
 07/02/2002 - Marc LEGOUX - c6r07 : Add TaskCtx management in MLOS32
 03/02/2002 - Marc LEGOUX - c6r07 : Add _MemAllocX
 31/11/2000 - Marc LEGOUX - c6r05 : Adjust _PrintXXX functions and type
 06/06/2000 - Marc LEGOUX - c6r05 : Add _TaskCtx functions. 
 17/03/2000 - Marc LEGOUX - c6r03 : Add __ExitErr
 04/02/2000 - Marc LEGOUX - c6r03 : Add _TaskPriorSet
 30/01/2000 - Marc LEGOUX - c6r03 : Add _PrintCfgxx
 18/01/2000 - Marc LEGOUX - c6r03 : Add _TaskAbort and some TaskModes
 16/12/1999 - Marc LEGOUX - c6r03 : Hide MLOS32 structures
 12/08/1999 - Marc LEGOUX - c6r01 : restore old syst_TaskType for WIN32
 06/08/1999 - Marc LEGOUX - c6r01 : Add syst_TaskCurGet
 01/01/1999 - Marc LEGOUX - c6r01 : Reorganize
 15/12/1998 - Marc LEGOUX - c5r04 : Move _Pipe to systpipe
 25/11/1998 - Marc LEGOUX - c5r01 : Change syst_Call
 27/10/1997 - Marc LEGOUX - c5r01 : Add syst_MutexOpen and Close
 27/09/1997 - Marc LEGOUX - c4r03 : Add syst_PrintWnd
 17/09/1997 - Marc LEGOUX - c4r02 : Add mode in syst_TaskType
 29/05/1997 - Marc LEGOUX - c4r01 : Update from own module
 ************************************************************************/
#ifndef syst_h
#define syst_h

#include "uid/config.h"	/* Configuration */
#include "err.h"	/* Errors */

#if config_WIN
#include <windows.h>	/* For private types with windows representation*/
#include <windowsx.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*-------------------------- ERRORS -----------------------------------*/
#define syst__escape	(6)
extern err_Type syst__Err;
#define syst__TooLong		(syst__Err+1)	/* Buffer too long */
#define syst__PipeWrite		(syst__Err+2)	/* Pipe write error */
#define syst__Null		(syst__Err+3)	/* Unexpected null pointer */
#define syst__Timeout		(syst__Err+4)	/* Out of time */
#define syst__Memory		(syst__Err+5)	/* Out of memory */
#define syst__Signal		(syst__Err+6)	// Task is signaled
#define syst__SignalAbort	(syst__escape)	// Task asked to abort
#define syst__WaitRV		(syst__Err+7)	/* Return value or wait*/
#define syst__Arg		(syst__Err+8)	/* Wrong arg to appli .exe */
#define syst__Call		(syst__Err+9)
#define syst__FileNotFound	(syst__Err+10)
#define syst__FileMkDir		(syst__Err+11)
#define syst__FileUnlink	(syst__Err+12)
#define syst__Name		(syst__Err+13)	/* Name error */
#define syst__TaskExists	(syst__Err+14)	/* Task struct is already used */
#define syst__TaskCtx		(syst__Err+15)	/* Task Ctx Full */
#define syst__Access		(syst__Err+16)	/* Access denied */
#define syst__AccessExists	(syst__Err+17)	/* Access already allowed */

#define syst__EventOpen	(syst__Err+20)	/* Event not open or already open */
#define syst__MutexOpen	(syst__Err+21)	/* Mutex not open or already open */
#define syst__ObjOpen	(syst__Err+22)	/* Object not open or already open */
#define syst__Image	(syst__Err+23)	/* Invalid image */
// Reserved for semaphore +24..++27
#define syst__MutexCount	(syst__Err+28)	// Mutex count exceeded (same task)
#define syst__MutexRelease	(syst__Err+29)	// Mutex release ignored
#define syst__Version		(syst__Err+30)	// Bad system version
#define syst__LoadAddr		(syst__Err+31)	// Package Load Address
#define syst__NIY		(syst__Err+32)	// Not implemented Yet

#define syst__CallErr		(syst__Err+50)	// Base error of process call
#define syst__CallErrMax	(syst__Err+249)	// Max error of process call

#define syst__SignalErr		(syst__Err+250)	// Base error of signals (2..7F)
#define syst__SignalStop	(syst__Err+252)	// Stop signal
#define syst__SignalErrMax	(syst__Err+250+127)	// Max error of signals (2..7F)

extern err_Type syst__ExitErr;		/* Exit code base */

#if config_WIN
extern err_Type win__Err;		// win code base 
#define syst__NotFound	 (win__Err+ERROR_FILE_NOT_FOUND)	// 2L: Pipe or appli not found 
#define syst__PipeAnswer (win__Err+ERROR_NO_DATA)		// 232L: pipe closed on client side 
#endif

#define syst_NameMax	32		/* Maximum size of names */
/*-------------------------- PRIVATE-----------------------------------*/
#if config_WIN
/*----------------------------------------------------------------------*
                Get Last Windows Error
 Windows error codes are shifted with err_WinMask to let application errors
 starting from 0. 
 *----------------------------------------------------------------------*/
extern 
err_Type GetWinError();
#endif
/************************************************************************
                        REFERENCE
 ************************************************************************
 Only apply to types that own the operation New. 
 ************************************************************************/
#if !config_MLOS8
/*----------------------------------------------------------------------*
                Duplicate a reference
 *----------------------------------------------------------------------*/
extern 
err_Type syst_RefDup(void * ref);

/*----------------------------------------------------------------------*
                Delete a reference
 This may terminate the referenced object and may remove it from memory. 
 *----------------------------------------------------------------------*/
extern 
err_Type syst_RefDel(void * ref);

#define syst_RefDel0(ref) (syst_RefDel(ref), (ref)=(void *)0, 0)
#endif
/************************************************************************
                        TIME
 ************************************************************************/
/*----------------------------------------------------------------------*
                Duration Type
 *----------------------------------------------------------------------*/
#if config_REGSIZE < 32
#if !config_MLOS8
#define syst_TType 	ushort		/* Type of time */
#define syst_TForever	0xFFFF		/* Infinite duration */
#endif
#else
#define syst_TType	ulong		/* Type of time */
#define syst_TForever	0xFFFFFFFFL	/* Infinite duration */
#endif
#define syst_Forever	syst_TForever

#if !config_MLOS8
/*----------------------------------------------------------------------*
                Get Current Time
 The current time is the elapsed time since the start of the system. 
 The time value can be read with different units. 
 The default unit is 1 millisec. 
 The Unit parameter is a power of 10 of the default unit: 
 Unit = 0 : 1	ms
 Unit = 1 : 10	ms
 Unit = 2 : 100	ms
 ...
 Unit =-1 : 0.1	ms
 ...
 NOTE: Depending on the unit, the returned time can overflow and start from 
 0 again. So only differences between times in the same unit must be used. 
 NOTE: The minimum usefull unit depends on the system implementation. 
 For instance when the system counts time with a 100us step a unit parameter 
 smaller than -1 will not give a better precision. 
 *----------------------------------------------------------------------*/
extern 
err_Type syst_TimeGet(
        syst_TType * pTime,	/* (OUT): Time */
        int Unit		/* Time unit: power of 10 of default unit (ms) */
        );
#endif
/************************************************************************
                        SYNCHRONIZATION
 ************************************************************************/
/************************************************************************
                        CRITICAL SECTION
 Reserved to system
 ************************************************************************/
/* Enter critical section: returns old level */
extern 
unsigned int syst_CS();

/* Exit critical section */
extern 
void syst_CSEnd(unsigned int prev_level);
/************************************************************************
                        EVENT
 ************************************************************************
 A statically declared event must be first open. 
 The default mode is AutoReset and the event is initialized to OFF.
 An event can be Set to ON or Reset to OFF. 
 A task can wait for an event: if the event is OFF, the task is suspended
 otherwise the task does not wait. 
 When an event switches from OFF to ON, all waiting tasks are resumed and 
 the event returns to the OFF state if it owns the AutoReset mode. 
 An event must be closed after the last use.
 ************************************************************************/
#if config_WIN
typedef
  HANDLE syst_EventType;	/* Type for declaration and allocation */
#elif config_MLOS32
typedef
  struct {
    void * Private1[46];	// Exact : void * [29] + obj_type[3]
  } syst_EventType;
#else
typedef
  uchar syst_EventType;		/* Type for declaration and allocation */
#endif
typedef
  syst_EventType * syst_EventRef;
/*----------------------------------------------------------------------*
                Init a new AutoReset event
 *----------------------------------------------------------------------*/
extern 
err_Type syst_EventOpen(
        syst_EventRef pEvt	/* (OUT) Event in OFF state */
        );
/*----------------------------------------------------------------------*
                Init a [named] event
 *----------------------------------------------------------------------*/
#define syst_EventModeResetManual	1	// Manual reset 
#define syst_EventModeSetInit		2	// Set when open 

extern 
err_Type syst_EventOpenN(
        syst_EventRef pEvt,	// (OUT) Event in OFF state 
        char * Name,		// Name or 0
        unsigned Mode		// Init Mode
        );
/*----------------------------------------------------------------------*
                Close an event
 *----------------------------------------------------------------------*/
extern 
err_Type syst_EventClose(
        syst_EventRef Evt	/* Event */
        );
/*----------------------------------------------------------------------*
                Set an event to ON (Trig event)
 *----------------------------------------------------------------------*/
extern 
err_Type syst_EventSet(
        syst_EventRef Evt	/* (IN) Event */
        );
/*----------------------------------------------------------------------*
                Reset an event to OFF (UnTrig event)
 *----------------------------------------------------------------------*/
extern 
err_Type syst_EventReset(
        syst_EventRef Evt	/* (IN) Event */
        );
/*----------------------------------------------------------------------*
                Wait for an event to be ON
 *----------------------------------------------------------------------*/
extern 
err_Type syst_EventWait(
        syst_EventRef Evt,	/* (IN) Event */
        syst_TType Timeout
        );
/*----------------------------------------------------------------------*
                Create a New [compound] Event
 *----------------------------------------------------------------------*/
extern 
err_Type syst_EventNew(
        syst_EventRef * pEvt,	/* (OUT) New Event */
        unsigned Mode,		/* Creation mode */
        void * Param		/* Creation parameters */
        );
/************************************************************************
 			MUTEX
 			Mutually exclusive access flag.
 Only one task could own a mutex at a time.
 Ownership is obtained by function syst_MutexTake()
 Ownership is release  by function syst_MutexRelease()
 So execution of the code between Take and Release is garanty to be 
 exclusive, and objects accessed only in that piece of code are
 protected against concurrent accesses.
 
 Mutex must have unique names to be used between different concurrent 
 processes. 
 
 Mutex0 is a simplified and faster mutex (no name nor timeout)
 ************************************************************************/
#if config_WIN
#define syst_MutexModeTakeNew	1	/* Created by Take, deleted by release */
typedef
  struct {
    HANDLE Mutex;
    int Mode;
  } syst_MutexType, syst_Mutex0Type;
#define syst_Mutex0Null {0}

//----------------------------------------
#elif config_MLOS32
typedef
  void * syst_Mutex0Type;
#define syst_Mutex0Null ((void *)0)

typedef
  struct {
    syst_EventType	Private1;
    void *		Private2;
    
  } syst_MutexType;
//----------------------------------------
#else
typedef
  unsigned syst_Mutex0Type;
typedef
  unsigned syst_MutexType;
#define syst_Mutex0Null (0)
#endif

/*----------------------------------------------------------------------*
                Open
 To create [or open] a [named] mutex
 Ownership is NOT granted.
 Must be ended by syst_MutexClose()
 Any task can open a mutex.
 If the mutex is anonymous and is currently used it is overridden. 
 *----------------------------------------------------------------------*/
extern 
err_Type syst_MutexOpen(
        syst_MutexType * pMutex,	/* (OUT): Mutex */
        char * Name			/* name or NULL */
        );
/*----------------------------------------------------------------------*
                Close
 To delete [or close] a [named] mutex
 Any task can close a mutex.
 Waiting tasks are restarted.
 *----------------------------------------------------------------------*/
extern 
err_Type syst_MutexClose(
        syst_MutexType * Mutex		/* Mutex */
        );
/*----------------------------------------------------------------------*
                Take
 Get ownership of a [named] mutex
 If returned code is 0, ownership is granted and the mutex must be 
 released later with syst_MutexRelease()
 Could be called when ownership is already obtained without blocking.
 NOTE: If Name==NULL the ingoing mutex is taken, otherwise it is created.
 *----------------------------------------------------------------------*/
extern 
err_Type syst_MutexTake(
        syst_MutexType * pMutex,	/* (IN): if name==NULL, (OUT): Mutex */
        char * Name,			/* name or NULL */
        syst_TType Timeout		/* Max time to wait (ms) */
        );
/*----------------------------------------------------------------------*
                Release
 Give up ownership of a mutex
 The mutex is released only if it is owned. 
 If mutex has been created by MutexTake, it is also deleted.
 *----------------------------------------------------------------------*/
extern 
err_Type syst_MutexRelease(
        syst_MutexType * Mutex	/* (IN): Mutex */
        );
/*----------------------------------------------------------------------*
                Open
 To create [or open] a basic mutex
 Ownership is NOT granted.
 Must be ended by _Close()
 *----------------------------------------------------------------------*/
#if config_WIN
#define syst_Mutex0Open(m)  syst_MutexOpen((m),NULL)
#define syst_Mutex0Close(m) syst_MutexClose(m)

#else
#define syst_Mutex0Open(m) ((*(m) = (syst_Mutex0Type)0),0)
#define syst_Mutex0Close(m) (0)
#endif
/*----------------------------------------------------------------------*
                Take
 Get ownership of a basic mutex
 Make current task not ready if ownership is not obtained. 
 Could be called when ownership is already obtained without blocking.
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Mutex0Take(
        syst_Mutex0Type * Mutex	/* (IN/OUT): Mutex */
        );
/*----------------------------------------------------------------------*
                Take
 Get ownership of two basic mutex
 Make current task not ready if both ownerships are not obtained. 
 Could be called when ownership is already obtained without blocking.
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Mutex02Take(
        syst_Mutex0Type * Mutex1,	/* Mutex */
        syst_Mutex0Type * Mutex2	/* Mutex */
        );
/*----------------------------------------------------------------------*
                Release
 Give up ownership of a basic mutex
 The mutex is released only if it is owned. 
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Mutex0Release(
        syst_Mutex0Type * Mutex	/* (IN): Mutex */
        );
/************************************************************************
                        EXECUTION
 An independant application could be launched with name and command line.
 Concurrent tasks could be started inside the application.
 ************************************************************************/
#if config_WIN
/*----------------------------------------------------------------------*
                	APPLICATION
 *----------------------------------------------------------------------*/
/* Application instance */
typedef                             
  struct {                          
    STARTUPINFO StartInfo;
    PROCESS_INFORMATION ProcInfo;   
  } syst_AppliType;                   		/* private */
/*----------------------------------------------------------------------*
                Exec Application object
                - Print output
                - Wait for end of execution
 *----------------------------------------------------------------------*
 The return code of the execution is named the Exit code. 
 This code is returned as the error code of syst_Execute except with the 
 NoExitCode mode. 
 *----------------------------------------------------------------------*/
// Exec modes
#define syst_ExecModeOutPrint	1	// Print output with err_Print 
#define syst_ExecModeNoExitCode	2	// Do not return the exit code 
#define syst_ExecModeNoWait	4	// Do not wait for end of execution 
extern
err_Type syst_Execute(
        char * AppliName, 	// 0 or Application name
        char * CmdLine,		// Command line to execute
        syst_AppliType * pProc,	// 0 or OUT: Application instance 
        ulong Mode,		// Exec mode
        void * Param		// 0 or specific parameters
        );
/*----------------------------------------------------------------------*
                Exec Application object
 *----------------------------------------------------------------------*
 Calls syst_Execute()
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Exec(
	char * AppliName, 
        char * CmdLine,
	syst_AppliType * pProc	// 0 or OUT: Application instance 
	);
/*----------------------------------------------------------------------*
                Exit from current application
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Exit(err_Type erno);

#else /* !config_WIN */
#define syst_AppliType unsigned
#endif
/*----------------------------------------------------------------------*
                        Call a program
 Waits for return
 Print the results
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Call(
	char * CmdLine, 
	char *fname);		/* null or print results in file (""=console) */
/************************************************************************
                	TASK
 ************************************************************************
 If a task is devoted to be a server, it must be named and could be 
 signaled for prioritary calls (like End). In that case, 
 the task should stop waiting for any other purpose (if it was) in order 
 to answer the call. To do this, any waiting operation that a server
 could call (other than its own command pipe) must also wait for the 
 task event (syst_TaskCur->Signal).
 ************************************************************************/
// Max entries of task local context 
#define syst_TaskCtxMax		32
#define syst_TaskNull ((syst_TaskRef)(0))

/*----------------------------------------------------------------------*
			MLOSWIN
 *----------------------------------------------------------------------*/
#if config_WIN
// Task object 
typedef                             
  struct {                          
    char	Private1[syst_NameMax+1];
    void * 	Private2[4];
    syst_EventType Private3;
    char 	Private4;
  } syst_TaskType, * syst_TaskRef;

// WARNING: It is obsolete to declare a syst_TaskType variable or struct item. 
// Only syst_TaskRef must be used. 

/* Declarator for task local variables */
#if config_TLS
#define syst_TaskDecl <not allowed>
#else
#define syst_TaskDecl __declspec(thread) // Auto alloc with this declarator
#endif
/*------------------------*/
/* Current Task Instance (ReadOnly) : Get it with syst_TaskCurGet*/
#if config_TLS
extern 
unsigned syst_TaskCurCtxId;
#define syst_TaskCur ((syst_TaskRef)syst_TaskCtxGet(syst_TaskCurCtxId))
#else
extern 
syst_TaskDecl syst_TaskRef syst_TaskCur;
#endif
/*----------------------------------------------------------------------*
                MLOS32 and other systems
 *----------------------------------------------------------------------*/
#else
// Task object 
#define syst_TaskType	void
#define syst_TaskRef	void *

// Task local context : Indexes reserved for system
#define syst_TaskCtxIdS1	(syst_TaskCtxMax+0)
#define syst_TaskCtxIdS2	(syst_TaskCtxMax+1)
#define syst_TaskCtxIdS3	(syst_TaskCtxMax+2)
#define syst_TaskCtxIdS4	(syst_TaskCtxMax+3)

/* No Declarator for task local variables: Use CtxAlloc */
#define syst_TaskDecl <Not allowed>

/* Current Task Instance */
#if config_LIB
// Inside application for relative code
#pragma section syst_VarAbs far-absolute
#pragma use_section syst_VarAbs syst_TaskCur
extern volatile syst_TaskRef syst_TaskCur;
#else
// Inside system 
extern syst_TaskRef syst_TaskCur;
#endif

#endif
/*----------------------------------------------------------------------*/
#if !config_MLOS8
/*----------------------------------------------------------------------*
                Current task Waits for a while
 Timeout = 0 : wait for the end of time slice
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Wait(syst_TType Timeout);	/* Duration in ms */
/*----------------------------------------------------------------------*
                Current task Waits for a while
 The available units are hardware dependent.
 *----------------------------------------------------------------------*/
extern 
err_Type syst_WaitU(
        syst_TType Timeout,	/* Duration in unit */
        int Unit		/* Time unit: power of 10 of default unit (ms) */
        );
/*----------------------------------------------------------------------*
                Signal a Task
 *----------------------------------------------------------------------*/
#define  syst_TaskSignalNull  ((void *)0)	/* Null signal */
#define  syst_TaskSignalAbort ((void *)1)	/* Abort signal */
#define  syst_TaskSignalStop  ((void *)2)	// Stop signal
#define  syst_TaskSignalIntMax  ((void *)0x7F)	// Max numeric signal 
extern 
err_Type syst_TaskSignal(syst_TaskRef task, void * Signal);
/*----------------------------------------------------------------------*
                Kill a task
 *----------------------------------------------------------------------*/
extern 
err_Type syst_TaskKill(syst_TaskRef Task);
/*----------------------------------------------------------------------*
                Abort a task
 Sends an Abort Signal to the task and waits for a timeout and for 
 the end of the task. 
 Errors: 0 if the task ends in time.  
 Deletes in any cases the Task reference: this reference should not
 be used again.
 Can be used to delete an allocated task that is running or already ended.
 *----------------------------------------------------------------------*/
extern 
err_Type syst_TaskAbort(syst_TaskRef Task, syst_TType Timeout);
/*----------------------------------------------------------------------*
                Get Current task 
 WARNING: Do no modify any accessible data from syst_TaskType
 *----------------------------------------------------------------------*/
extern 
err_Type syst_TaskCurGet(
        syst_TaskRef* pTask	/* (OUT): Current task */
        );
/* Get with direct return (null if error) */
extern 
syst_TaskRef syst_TaskCurGet0();
/*----------------------------------------------------------------------*
                Start a task inside current application
 When the start function returns, the task ends.
 If pTask or *pTask is NULL the object is allocated into the heap (and
 returned into *pTask if possible), [otherwise **pTask is directly inited, 
 but it is an obsolete feature that is not recommended]. 
 The name is mandatory for communication with the task (as a server).
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Task(
	syst_TaskRef * pTask,	/* IN: 0, OUT: Task object */
	err_Type (* func)(), 	/* Start function of task */
	void * args,		/* Generic Argument */
	char * name		/* Task name or NULL */
	);
/*----------------------------------------------------------------------*
                Delete a Task ref
 Must be called for an allocated task. After this call the task is 
 removed from memory when it is ended. 
 This reference should not be used again.
 *----------------------------------------------------------------------*/
extern 
err_Type syst_TaskRefDel(
        syst_TaskRef Task	// IN: Task object
        );
/*-----------------------------------------------------------------------*
                        Change Task priority
 *-----------------------------------------------------------------------*/
#define syst_TaskPriorHigh	0xFF	// Default higher than standard
#define syst_TaskPriorLow	0xFE	// Default lower than standard
#define syst_TaskPriorDef	0x00	// Default standard

extern 
err_Type syst_TaskPriorSet(
        syst_TaskRef Task,	/* 0 for current */
        int Prior		/* Use predefined values */
        );
/*----------------------------------------------------------------------*
                        Suspend task
 *----------------------------------------------------------------------*/
extern
err_Type syst_TaskSuspend(
        syst_TaskRef Task,	// 0 for current
        unsigned Mode
        );
/*----------------------------------------------------------------------*
                        Resume task
 *----------------------------------------------------------------------*/
extern
err_Type syst_TaskResume(
        syst_TaskRef Task,	// 0 for current
        unsigned Mode
        );
/*---------------------------------------------------------------------*
                        Get info on a task
 Remark: 
 ModeAbortSignal : Ask to a task to terminate CORRECTLY ITSELF. 
   This mode is not resetable.
   In this mode the task runs normally but receives syst__SignalAbort 
   in place of syst_Timeout. 
 *---------------------------------------------------------------------*/
// task Modes 
#define syst_TaskModeNew		0x01	// Never started (no mcu context)
#define syst_TaskModeSuspended		0x02	// In Ready list if not waiting
#define syst_TaskModeWaiting		0x04	// In Wait list
#define syst_TaskModeStackAlloc		0x08	// Stack was auto allocated
#define syst_TaskModeKilled		0x10	// Killed (in killed list or alone)
#define syst_TaskModeAbortSignal	0x20	// Abort signal has been received
#define syst_TaskModeIrqFast		0x40	// Fast scheduling for Irq
#define syst_TaskModeIrqed		0x80	// Interrupted from Irq

#define syst_TaskModeAlloc	0x100	// The task structure is allocated
#define syst_TaskModeEnded	0x200	// The task activity is dead

#define syst_TaskModeRefMask	0xF000	// Simple Ref counter
#define syst_TaskModeRefBody	0x1000	// Ref of the body
#define syst_TaskModeRefExt	0x2000	// Ref of the creator
#define syst_TaskModeRefExt2	0x4000	// Ref of the creator
#define syst_TaskModeRefExt3	0x8000	// Ref of the creator

#define syst_TaskInfoMask1	0	// First values
typedef
  struct {
    unsigned	Mask;		// IN : Mask of returned values
    
    unsigned	Mode;		// Mode 
    int		Prior;		// Priority 
    ulong 	StackAddr;	// Stack Addr
    ulong 	StackLen;	// Stack Len (IN if copy of stack)
    ulong 	SP;		// Top of Stack
    ulong * 	StackCopy;	// IN or 0, OUT:Allocated copy of stack from SP
    ulong	TimeElp;	// Elapsed time
    err_Type	Erno;		// Restarting error
    char	Name[syst_NameMax+1];	// Task Name or ""
    // End of Mask1 values
    
  } syst_TaskInfoType;
/*----------------------------------------------------------------------*
                        Info Get
 Mode
 syst_TaskModeSuspended : Suspend task (if not current)
 syst_TaskModeStackAlloc: Copy stack. Allocate StackCopy if 0
 *----------------------------------------------------------------------*/
extern
err_Type syst_TaskInfoGet(
        syst_TaskRef Task,	// 0 for current 
        unsigned 	Mode,	// Get mode
        syst_TaskInfoType * pInfo	// IN: Mask, OUT: Other fields
        );
/*---------------------------------------------------------------------*
                        Get Task List
 Refs are counted
 *---------------------------------------------------------------------*/
extern
err_Type syst_TaskListGet(
        syst_TaskRef* pList,		// (OUT): List (null ended)
        unsigned MaxL,
        unsigned Mode,
        void * Pattern
        );
/*----------------------------------------------------------------------*
                        Task Context
 *----------------------------------------------------------------------*
 A task context is an array of (void *) that is owned by the task. 
 But each index of this array represents the same variable for all tasks of 
 a single process. 
 Such a task variable is identified by its index. For each task, the variable 
 has a specific value. 
 The variable id must be allocated by the main task of a process and 
 stored into a standard global variable. 
 When a task is created its context must be initialized to the initial values 
 of the used task variables. 
 *----------------------------------------------------------------------*/
/* Alloc a task variable */
extern 
err_Type syst_TaskCtxAlloc(
        unsigned * pId		/* (OUT): New Task Variable id */
        );

/* Free a task variable */
extern 
err_Type syst_TaskCtxFree(
        unsigned Id		/* Id to free */
        );

/* Get Task variable value */
extern 
void * syst_TaskCtxGet(
        unsigned Id
        );

/* Set Task variable value */
extern 
err_Type syst_TaskCtxSet(
        unsigned Id, 		/* Task variable Id */
        void *   Val		/* New value */
        );
#endif
/************************************************************************
                        HEAP MEMORY
 ************************************************************************/
#if !config_MLOS8
/*----------------------------------------------------------------------*
                Check memory and get Infos
 *----------------------------------------------------------------------*/

#define syst_MemInfoBlkModeEnd	0x80	// After last block
#define syst_MemInfoBlkModeFree	0x40	// Free block else allocated

#define syst_MemInfoBlkModeErr	0x0F	// Error mask (0 = OK)
#define syst_MemErrWrongRef		1
#define syst_MemErrWrongPPrev		2
#define syst_MemErrWrongPNext		3
#define syst_MemErrWrongFree		4
#define syst_MemErrFreeLost		5
#define syst_MemErrWrongAlloc		6
#define syst_MemErrAllocLost		7
#define syst_MemErrWrongLast		8
#define syst_MemErrXXX			15

#pragma pack(1)
typedef
  struct {
    ulong	Addr;		// Block address (Header, Area)
    uchar	Mode;		// Mode and err
  } syst_MemInfoBlkType;

// Mode of syst_MemInfoType
#define syst_MemInfoModeHeap	1	// Get heap infos
#define syst_MemInfoModeBlk	2	// Get block infos
// Mode contains also the flags of syst_MemAreaMask

typedef
  struct {
    ulong Mode;		/* (IN): Mode =0 (RFU) */
    
    ulong AllocSize;	/* (OUT): Current allocated size */
    ulong AllocCalls;	/* (OUT): Total number of calls to alloc */
    ulong FreeCalls;	/* (OUT): Total number ofcalls to free */
    ulong AllocMax;	/* (OUT): 0 or Max allocated address */
    
    // Heap infos
    ulong HeapAddr;		// OUT: Heap area address
    ulong HeapLen;		// OUT: Heap area length
    ulong HeapLenU;		// OUT: Heap area used length
    
    ulong HeapRef;		// OUT: Heap object address
    ulong HeapSize;		// OUT: Heap object size
    ulong HeapHdSize;		// OUT: Heap header size
    ulong HeapMode;		// OUT: Heap mode
    ulong HeapAlign;		// OUT: Heap align mask
    ulong HeapBlkHdSize;	// OUT: Heap block header size
    
    ulong HeapFreeNb;		// OUT: Heap free blocks
    ulong HeapFreeLen;		// OUT: Heap free area length
    ulong HeapAllocNb;		// OUT: Heap allocated blocks
    ulong HeapAllocLen;		// OUT: Heap allocated area length
    
    ulong HeapErno;		// OUT: 0 or one of corrupt heap errors
    
    // Block Infos
    ulong ListNbMax;		// IN : Maximum Number of blocks in list
    ulong ListFirst;		// 0 or IN, OUT: First block num in list 1..
    ulong ListNb;		// OUT: Number of blocks in list
    
    syst_MemInfoBlkType List[1];	// OUT: List of blocks
    
  } syst_MemInfoType;
#pragma pack()

extern 
err_Type syst_MemInfo(
                syst_MemInfoType * pI	// IN/OUT: mem info
                );
/*----------------------------------------------------------------------*
			Check memory
 *----------------------------------------------------------------------*/
extern 
err_Type syst_MemCheck(
                unsigned mode,	/* Mode (default 0) */
                void * param	/* Parameters (default null) */
                );
/*----------------------------------------------------------------------*
                Alloc
 *----------------------------------------------------------------------*/
extern 
err_Type syst_MemAlloc(
                void ** pp,	/* (OUT): Allocated object */
                ulong size);	/* (IN): Size */

extern 
err_Type syst_MemLocalAlloc(
                void ** pp,	/* (OUT): Allocated object */
                ulong size);	/* (IN): Size */

#define syst_MemGlobalAlloc(p,s) syst_MemAlloc(p,s)
/*----------------------------------------------------------------------*
 		Extended Alloc
 *----------------------------------------------------------------------*/
// Area Id
#define syst_MemAreaMask	0xFFFF0000
#define syst_MemAreaRamDef	0		/* Default ram area */
#define syst_MemAreaFlashDef	0x00010000	/* Default flash area */

// Mode flags:
#define syst_MemModeLocal	0x0100	/* Local heap (else Global) */

// Alignment: 0=byte,1=short,2=long,...,fixed addr
#define syst_MemAlignByte	0x00
#define syst_MemAlignShort	0x01
#define syst_MemAlignLong	0x02
#define syst_MemAlignFixed	0xFF	// Fixed Address

extern 
err_Type syst_MemAllocX(
	void ** pp,	// (IN): If fixed address, (OUT) : Allocated block
	ulong size,	// (IN): Size
	ulong mode	// (IN): ORed: Area Id, Mode flags and alignment
	);
/*----------------------------------------------------------------------*
                Free
 *----------------------------------------------------------------------*/
extern 
err_Type syst_MemFree(
        void * p,	/* (IN): Allocated object */
        ulong size	/* (IN): Size or 0*/
        );
/*----------------------------------------------------------------------*
                ReAlloc
 *----------------------------------------------------------------------*/
extern 
err_Type syst_MemReAlloc(
        void * p,	// IN: Allocated object 
        ulong size,	// IN: New Size
        ulong mode	// 0
        );
/*----------------------------------------------------------------------*
                        Info on a block
 *----------------------------------------------------------------------*/
typedef
  struct {
    ulong	State;		// 0: Undef, 1:Free, 2:Alloc
    ulong	Addr;
    ulong	Size;
    ulong	RFU[5];
  } syst_MemBlkInfoType;
#define syst_MemBlkStateUndef	0
#define syst_MemBlkStateFree	1
#define syst_MemBlkStateAlloc	2

extern
err_Type syst_MemBlkInfo(
        void *p, 
        syst_MemBlkInfoType * pInfo	// OUT
        );
#endif
/************************************************************************
                        EVENT LOG
 ************************************************************************/
#if config_WIN
/* Key path of sources in configuration */
#define syst_EvlgCfgPath \
":HKEY_LOCAL_MACHINE:SYSTEM:CurrentControlSet:Services:EventLog:Application"

/* Message modes */
#define syst_EvlgModeError	EVENTLOG_ERROR_TYPE
#define syst_EvlgModeWarning	EVENTLOG_WARNING_TYPE
#define syst_EvlgModeInfo	EVENTLOG_INFORMATION_TYPE
/*----------------------------------------------------------------------*
                General message
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Message(
	unsigned Mode,	/* Message Mode */
	char * Msg);	/* message */
/*----------------------------------------------------------------------*
                Report Event
 *----------------------------------------------------------------------*/
extern 
err_Type syst_EvlgReport(
	long MsgId, 		/* Message id in message file */
	unsigned Mode,		/* Message Mode */
	char ** MsgArray,	/* Array of argument strings */
	unsigned MsgNb		/* number of strings */
	);
#endif	/* config_WIN */
/***********************************************************************
                        CONSOLE
 Only one console per process
 ***********************************************************************/
#if !config_MLOS8
/*----------------------------------------------------------------------*
                        Print string on the current terminal
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Print(char *st);
/*----------------------------------------------------------------------*
                        Get string from the current terminal
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Gets(
        char * st,		/* (OUT): string */
        unsigned maxl		/* Maximum length */
        );
/*----------------------------------------------------------------------*
                        Query
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Query(
        char * title,	/* (IN): Title or NULL */
        char * prompt,	/* (IN): prompt string or NULL */
        char * answer,	/* (IN/OUT): answer string or NULL */
        unsigned maxl	/* Maximum length of answer */
        );
/*----------------------------------------------------------------------*
                        Defines the printing configuration
 Each function of the configuration as the same parameters as the 
 corresponding syst_xxx function. 
 
 NOTE: The Print function has a higher priority than the Print window. 
 If this function returns the error 1, the print message is also posted to 
 the window. 
 *----------------------------------------------------------------------*/
typedef err_Type (* syst_PrintFuncType)();

#define syst_PrintCfgModeMask	0	// Parameters up to PrintWnd
typedef 
  struct {
    unsigned long	Mode;		// Mode and paramater mask
    syst_PrintFuncType PrintFunc;	/* Called by syst_Print */
    syst_PrintFuncType GetsFunc;	/* Called by syst_Gets */
    syst_PrintFuncType QueryFunc;	/* Called by syst_Query */
    void *	       LogObj;		/* Log object or null */
    #if config_WIN
    HWND PrintWnd;
    #endif
  } syst_PrintCfgType;

/*----------------------------------------------------------------------*
			Set 
 *----------------------------------------------------------------------*/
extern 
err_Type syst_PrintCfgSet(
        syst_PrintCfgType * Cfg,	// Null or New config 
        syst_PrintCfgType * pCfg);	// null or IN: Mode, OUT: Old config 
/*----------------------------------------------------------------------*
			Set syst_Print only
 *----------------------------------------------------------------------*/
extern 
err_Type syst_PrintDef(
	err_Type (* PrintFunc)(char *), 
	unsigned Mode);
/*----------------------------------------------------------------------*/
#endif
#if config_WIN
#define WM_SystPrint	(WM_USER+0x201)	/* Print message */
/*----------------------------------------------------------------------*
                        Defines the window for printing
 *----------------------------------------------------------------------*/
extern 
err_Type syst_PrintWndSet(
        HWND hWnd, 		/* null or New window to receive print messages */
        HWND * phWnd);		/* null or (OUT): Old window */
/*----------------------------------------------------------------------*
                        Print into List Box
 The string is appended to the last line of the list box.
 \n produces a new line. 
 *----------------------------------------------------------------------*/
extern 
err_Type syst_PrintLB(
        HWND hWndLB, 		/* List Box */
        char * str, 		/* String to print */
        int LineNbMax,		/* Max line number in LB */
        int ToFree);		/* 1 if str must be freed after print */
#endif
/************************************************************************/
/*----------------------------------------------------------------------*
                        Start Module
 *----------------------------------------------------------------------*/
extern 
err_Type syst_Start(void * Cfg);
/*----------------------------------------------------------------------*
                        End Module
 *----------------------------------------------------------------------*/
extern 
err_Type syst_End(void * Cfg);
/************************************************************************/
#ifdef __cplusplus
}
#endif

#endif
