/************************************************************************
                        SYST
                Host System Interface
                with FILES
 ************************************************************************
 27/09/2009 Marc LEGOUX V2r24b : Define OBN interface
 ************************************************************************/
#ifndef systf_h
#define systf_h

#include <config.h>
#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#if config_MSDOS
#include <io.h>
#include <dos.h>
#endif
#if config_WIN
#include <io.h>
#include <windows.h>
#endif

#include <syst.h>	/* for syst_TType */
/*-------------------------- ERRORS -----------------------------------*/
extern 
err_Type syst__FileErr;
#define syst__FileBadRequest	syst__FileErr
#define syst__FileTooLongName	(syst__FileErr+1)
#ifndef syst__FileNotFound
#define syst__FileNotFound	(syst__FileErr+2)
#endif
#define syst__FileBadName	(syst__FileErr+3)
#define syst__FileFound		(syst__FileErr+4)
#define syst__FileEOF		(syst__FileErr+5)
#define syst__FileError		(syst__FileErr+6)
#define syst__FileCutLine	(syst__FileErr+7)
#define syst__FileOpen		(syst__FileErr+8)
#define syst__FileClose		(syst__FileErr+9)
#define syst__FileRead		(syst__FileErr+10)
#define syst__FileWrite		(syst__FileErr+11)
#define syst__FileGetp		(syst__FileErr+12)
#define syst__FileSetp		(syst__FileErr+13)
#define syst__FileEmptySymb	(syst__FileErr+14)
#define syst__FileUndefSymb	(syst__FileErr+15)
#define syst__FileAccess	(syst__FileErr+16)
#define syst__FileDir		(syst__FileErr+17)
#define syst__FileDirNot	(syst__FileErr+18)
#define syst__FileCreate	(syst__FileErr+19)
#ifndef syst__FileUnlink
#define syst__FileUnlink	(syst__FileErr+20)
#endif
#define syst__FileExists	(syst__FileErr+21)
#define syst__FileRename	(syst__FileErr+22)
#define syst__FileTooBig	(syst__FileErr+23)
#define syst__FileNEOF		(syst__FileErr+24)	// Not EOF

/* DOS attributes are between 0 (_A_NORMAL) and 0x20
	attr:
	1  : _A_RDONLY
	2  : _A_HIDDEN
	4  : _A_SYSTEM
	8  : _A_VOLID
	16 : _A_SUBDIR
	32 : _A_ARCH
*/
/* Type attributes */
#define file_Normal	0x8000
#define file_System	_A_SYSTEM
#define file_Volid	0 /*_A_VOLID*/
#define file_Subdir	_A_SUBDIR
#define file_NoSubdir	(file_Normal|_A_SYSTEM|_A_VOLID)
#define file_Abnormal	(_A_SYSTEM|_A_SUBDIR)
#define file_AType	(_A_SYSTEM|_A_SUBDIR|file_Normal)

/* Mode attributes */
#define file_ROnly	_A_RDONLY
#define file_Hidden	_A_HIDDEN
#define file_Archiv	_A_ARCH
#define file_AMode	(_A_ARCH|_A_HIDDEN|_A_RDONLY)

#define file_DosAtt (_A_ARCH|_A_HIDDEN|_A_RDONLY|_A_SYSTEM|_A_SUBDIR)

/* Modes d'ouverture */
#define file_READ	0
#define file_WRITE	1
#define file_APPEND	2
#define file_READWRITE	3

/* Modes de recherche avec path-list (composables) */
/* ISREAD,ISWRITE,ISEXEC ont les bonnes valeurs pour UNIX et C-VMS*/
#define file_ISNOT	8	/* First where file is not ... */
#define file_READABLE	4	/* First where file is (is not) readable */
#define file_WRITABLE	2	/* First where file is (is not) writable */
#define file_EXECABLE	1	/* First where file is (is not) runable  */
#define file_IS		0	/* First where file exists */
 
#define file_LF 10	/* Terminateur de ligne */
#define file_CR 13	

/* Longueur max de noms de fichiers (avec suffix) */
#if config_MSDOS
#define file_name_max 13
#endif

#if config_WIN
typedef
  HANDLE syst_FileType;	/* private: use to declare/init */
typedef
  HANDLE syst_FileRef;	/* private: use to alloc/create/access */

#define syst_FileNull INVALID_HANDLE_VALUE
#else
typedef
  void syst_FileType;
typedef
  void * syst_FileRef;
#define syst_FileNull 0
#endif
/*----------------------------------------------------------------------*/
// OBN Class syst_File BEGIN Mode=E
/*----------------------------------------------------------------------*
                        Check file access
 Returns 0 if accessible
 *----------------------------------------------------------------------*/
extern 
err_Type syst_FileAccess(char * path, unsigned attr);
/*---------------------------------------------------------------------*
                       Get file attributes
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileAttr(
        char * path,
        unsigned * pAttr	// OUT: Attributes 
        );
/*---------------------------------------------------------------------*
                       Set file attributes
 *---------------------------------------------------------------------*/
extern
err_Type syst_FileAttrSet(
        char * path,
        unsigned AttrMask,	// IN: Attributes to modify
        unsigned Attr 		// IN: Attribute values
        );
/*---------------------------------------------------------------------*
                       Get file infos
 *---------------------------------------------------------------------*/
typedef 
  struct {
    unsigned Attrib;
    long TimeCreate;
    long TimeAccess;
    long TimeWrite;
    unsigned WrTime;
    unsigned WrDate;
    long Size;
    char Name[260];
  } syst_FileInfoType;

extern 
err_Type syst_FileInfo(
        char * path,
        syst_FileInfoType *pInfo	/* (OUT): infos */
        );
/*---------------------------------------------------------------------*
                       Set file time
 *---------------------------------------------------------------------*/
#define syst_FileTimeModeW	0	/* Write time */
#define syst_FileTimeModeA	1	/* Access time */
#define syst_FileTimeModeC	2	/* Create time */

extern 
err_Type syst_FileTimeSet(
        char * path,
        unsigned long tm,
        int mode
        );
/*---------------------------------------------------------------------*
                        Copy file
 *---------------------------------------------------------------------*/
#define syst_FileDirModeCopyInto	1 // If pathto is a dir, copy into
#define syst_FileDirModeCopyAttr	2 // Keep attr as they are
extern 
err_Type syst_FileCopy(
        char * pathfrom,
        char * pathto,
        unsigned mode
        );
/*---------------------------------------------------------------------*
                        Rename file
 *---------------------------------------------------------------------*/
extern
err_Type syst_FileRename(
        char * pathfrom,
        char * pathto,
        unsigned mode
        );
/*---------------------------------------------------------------------*
                        New file
 *---------------------------------------------------------------------*/
extern
err_Type syst_FileNew(
        char * path,
        unsigned Attr
        );
/*---------------------------------------------------------------------*
                        Delete file
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDelete(char *pathname);
/*----------------------------------------------------------------------*
                        Unlink file
 *----------------------------------------------------------------------*/
extern 
err_Type syst_FileUnlink(char * path);
/************************************************************************
                        FILE DATA
 ************************************************************************/
/*----------------------------------------------------------------------*
                Create and or Open file
 *----------------------------------------------------------------------*/
extern 
err_Type syst_FileCreate(
        syst_FileRef * phFile,	/* (OUT): Ref to File */
        char * FPath,		/* File path */
        unsigned Mode		/* Create mode or 0 */
        );
/*----------------------------------------------------------------------*
                Close file
 *----------------------------------------------------------------------*/
extern 
err_Type syst_FileClose(
        syst_FileRef hFile	/* Ref to File */
        );
/*----------------------------------------------------------------------*
                Read file with timeout and Task signal
 The Call is aborted on Timeout or if the task is signaled.
 WARNING: The file must have been CREATED with the flag FILE_FLAG_OVERLAPPED
 and the normal synchronous ReadFile function is NO MORE allowed on this 
 open file (lpOverlapped arg must not be null).
 *----------------------------------------------------------------------*/
extern 
err_Type syst_FileRead(
        syst_FileRef hFile,	/* Ref to File to read */
        void * Buffer, 		/* (IN) Data Buffer */
        ulong Len,		/* Length to read */
        ulong * pLen,		/* (OUT): Read Length */
        syst_TType TimeOut	/* Timeout in ms */
        );
/*----------------------------------------------------------------------*
                Write file
 *----------------------------------------------------------------------*/
extern 
err_Type syst_FileWrite(
        syst_FileRef hFile,	/* Ref to File */
        void * Buffer, 		/* (IN) Data Buffer */
        ulong Len,		/* Length to write */
        ulong * pLen		/* null or (OUT): Written Length */
        );
/***********************************************************************
                        On DIRECTORIES
 ***********************************************************************/
/*----------------------------------------------------------------------*
                        Create directory
 *----------------------------------------------------------------------*/
extern 
err_Type syst_FileDirNew(char * path);
#define syst_FileMkDir syst_FileDirNew
/*----------------------------------------------------------------------*
                        Delete directory
 *----------------------------------------------------------------------*/
extern 
err_Type syst_FileDirDelete(
	char * path	// Wild cards for a file list
	);
/*---------------------------------------------------------------------*
                        Check if exist and is a directory
 Returns 0 if yes
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDirIs(char * fpath);
/*---------------------------------------------------------------------*
                        Copy file list or directory or file
 *---------------------------------------------------------------------*/
extern
err_Type syst_FileDirCopy(
        char * pathfrom,	// wild cards if a list, else a dir or file
        char * pathto,
        unsigned mode
        );
/*---------------------------------------------------------------------*
                       Changement de directorie courante
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDirCurSet(
        char * path
        );
/*---------------------------------------------------------------------*
                       Get current directory
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDirCurGet(
        char * pPath,		/* (OUT) */
        int MaxLen		/* Max returned length */
        );
/*---------------------------------------------------------------------*
                       Iter on directory files
 Returns syst__FileNotFound if no match or end of iteration
 *---------------------------------------------------------------------*/
typedef
  struct {
    /* First fields = FileInfoType */
    unsigned Attrib;		/* Type and mode attributes */
    long TimeCreate;
    long TimeAccess;
    long TimeWrite;
    unsigned WrTime;
    unsigned WrDate;
    long Size;
    char * Name;		/* points on name[] */
    char * Path;
    
    unsigned a_attrib;		/* Type and mode asked attributes */
    unsigned a_notattrib;	/* Type and mode compl. asked attributes */
    
    #if config_MSDOS | config_WIN311
    struct find_t ft;
    /*
    char reserved[21];
    unsigned char attrib;
    unsigned wr_time;
    unsigned wr_date;
    long size;
    char name[13];
    */
    #endif
    
    #if config_WIN32
    long hdl;
    struct _finddata_t fdata;
    /*
    struct _finddata_t {
        unsigned	attrib;
        time_t	time_create;
        time_t	time_access;
        time_t	time_write;
        _fsize_t	size;
        char	name[260];
    };
    */
    #endif
    
    int magic;
  } syst_FileDirIterType;

/*---------------------------------------------------------------------*
                       Init iteration
 Returns syst__FileNotFound if no match or end of iteration
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDirIterInit(
        syst_FileDirIterType *Iter,	/* (OUT): iterator */
        char * path			/* Matching path: Wild card autorized */
        );
/*---------------------------------------------------------------------*
                        Iteration with attribute selection
 one call for each file
 Returns syst__FileNotFound if no match or end of iteration
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDirIterInitAttr(
        syst_FileDirIterType *Iter,	/* (OUT): iterator */
        char * path,			/* Matching path: Wild card autorized */
        unsigned attrib, 		/* Must have those attributes */
        unsigned notattrib 		/* Must NOT have those attributes */
        );
/*---------------------------------------------------------------------*
                        Iteration on all files
 one call for each file
 Returns syst__FileNotFound if no match or end of iteration
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDirIter(
        syst_FileDirIterType *Iter,	/* (IN/OUT): iterator */
        char * pName,			/* null or (OUT): file name (no path) */
        int MaxLen			/* Max file name length */
        );
/*---------------------------------------------------------------------*
                        End Iteration
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDirIterEnd(
        syst_FileDirIterType *Iter);	/* (IN): iterator */
/***********************************************************************
                        On file NAMES and PATHS
 ***********************************************************************/
/*---------------------------------------------------------------------*
                        Extract disk path from full path
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDiskExtract(
        char * fpath,	/* (IN): Full Path */
        char * fdisk,	/* null or (OUT): Disk or current disk (name:) */
        unsigned maxdl,	/* Max disk length */
        char * fpath0	/* null or (OUT): Path without disk */
        );
/*---------------------------------------------------------------------*
                        Extract dir path from full path
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileDirExtract(char * fpath);	/* (IN/OUT) */
/*---------------------------------------------------------------------*
                        Extract file name from full path
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileNameExtract(
        char * fpath,	/* (IN): Full path */
        char ** pname);	/* (OUT): file name and suffix */
/*---------------------------------------------------------------------*
                        Extract file extension from full path
                        (System syntax)
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileSuffixExtract(
        char * fpath,
        char ** psuffix);	/* (OUT) */
/*---------------------------------------------------------------------*
                       Extraction d'un nom de fichier
 en dir name et file name
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileExtract(
        char * fpath,	/* (IN) full pathname (OUT) dir name */
        char * fname,	/* null or (OUT) filename */
        int size	/* filename max size */
        );
/*---------------------------------------------------------------------*
                       Nom absolu
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FilePathAbs(
        char * fpath,	/* (OUT) full absolute pathname */
        char * path,	/* (IN) pathname */
        unsigned maxl,	/* fpath max size */
        char * root	/* (IN) root path for relative path 
                           (null or "": DirCur) */
        );
/*---------------------------------------------------------------------*
                       Nom relatif a un path
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FilePathRel(
        char * rpath,	/* (OUT) relative path */
        char * path,	/* (IN) pathname */
        unsigned maxl,	/* rpath max size */
        char * root	/* (IN) root path for relative path 
                           (null or "": DirCur) */
        );
/*---------------------------------------------------------------------*
                       Extraction d'un path name
 if <fpath> == <root><lpath> returns OK and lpath
 else returns __FileBadName
 *---------------------------------------------------------------------*/
#define syst_FileExtractRoot syst_FileRootExtract
extern 
err_Type syst_FileRootExtract(
        char * fpath,	/* (IN) full pathname */
        char * rpath,	/* (IN) root pathname */
        char * lname,	/* (OUT) local pathname from the root */
        int size	/* lpath max size */
        );
/*---------------------------------------------------------------------*
                       Extraction de la racine commune
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileRootCommon(
        char * rpath,	/* (OUT) root pathname (could == path1)*/
        char * path1,	/* (IN) */
        char * path2,	/* (IN) */
        int size	/* lpath max size */
        );
/*---------------------------------------------------------------------*
                       Compression d'un nom de fichier
 *---------------------------------------------------------------------*
 fpath et hpath peuvent e^tre egaux. 
 fpath <- concat( hpath [\] lpath )
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileCompress(
        char * fpath,	/* (OUT) full pathname */
        char * hpath,	/* high pathname */
        char * lpath,	/* low pathname */
        int size	/* fpath max size */
        );
/*---------------------------------------------------------------------*
                       Expansion d'un nom de fichier
 *---------------------------------------------------------------------*
 Un nom peut etre precede d'un suffixe de deux formes possibles:
    {SYM}, ou SYM est une variable d'env qui contient une suite de 
           pathnames separes par SPACE.
    {|path1 path2 ... pathN}, ou pathi sont des pathnames UNIX-like.
 *---------------------------------------------------------------------*
 lpath et ppath peuvent etre egaux.
 ppath peut e^tre NULL si l'on s'interesse seulement au code d'erreur.
 En cas d'erreur not_found ou found, on retourne le premier nom.
 *---------------------------------------------------------------------*/
extern 
err_Type syst_FileExpand(
        char * lpath,	/* pathname logique */
        char * path,	/* (OUT) pathname physique */
        int size,	/* Taille max de ppath */
        int mode	/* Mode de recherche du fichier */
        );
/*----------------------------------------------------------------------*/
// OBN Class syst_File END
/**********************************************************************
			FICHIERS NON STRUCTURES
 Very OLD interface
 **********************************************************************/
/* Descripteur de fichier non structure' ouvert */
typedef
  int file_opmode;
typedef
   int file_dstype;

/*---------------------------------------------------------------------*/
#define file_exists(fname) (syst_FileAccess((fname),file_IS)==0)
#define file_delete(fname)  syst_FileDelete(fname)
#define file_copy(to, from) syst_FileCopy((from),(to))
/*---------------------------------------------------------------------*
			Ouverture [et Creation]
 *---------------------------------------------------------------------*
 Un nouveau fichier est cre'e' vide et devient accessible par ce nom.
 RETURNS: file_dstype
      -1  - Probleme d'ouverture
      >=0 - Descripteur d'ouverture du fichier.
 *---------------------------------------------------------------------*/
extern 
file_dstype file_open(
        char *pathname,        /* Nom du fichier */
        file_opmode mode,        /* Mode d'ouverture */
        char cflag        /* Creation si n'existe pas */
        );
/*---------------------------------------------------------------------*
			Fermeture
 *---------------------------------------------------------------------*
 RETURNS: 
      0  - Ok.
      -1 - Probleme de fermeture.
 *---------------------------------------------------------------------*/
extern 
int file_close(
        file_dstype fd	/* Descripteur d'ouverture */
        );
/*---------------------------------------------------------------------*
			Lecture ligne
*---------------------------------------------------------------------*
On lit au plus une ligne. 
Le caractere terminal \0 est ajoute' en fin du texte lu.
RETURNS: 
     0  - End-of-file
     -1 - Probleme de lecture
     >0 - Nombre de caracteres lus (terminateur exclu) + 1.
Implantation pas geniale : On lit un "bloc" pour chaque ligne et
on revient en arriere sur la fin de la 1ere ligne de ce bloc.
*---------------------------------------------------------------------*/
extern 
int file_lread(
        file_dstype fd,        /* Descripteur */
        char *buf,                /* Pointeur sur buffer de reception */
        int size                /* Nb max de caracteres a lire (terminateur exclu) */
        );
/*---------------------------------------------------------------------*
			Ecriture ligne
 *---------------------------------------------------------------------*
 On ecrit une ligne terminee par LF.
 Les caracteres LF de la chaine sont transformes en CR.
 Les caracteres NULL peuvent etre ecrit en utilisant size.
 size < 0 => chaine terminee par \0
 RETURNS: 
      -1  - Probleme d'ecriture
      >=0 - Nb de caracteres de la chaine
 *---------------------------------------------------------------------*/
extern 
int file_lwrite(
        file_dstype fd,  /* Descripteur */
        char *buf,       /* Pointeur sur chaine */
        int size        /* Taille de la chaine (si -1, chaine terminee par '\0'). */
        );
/*---------------------------------------------------------------------*
			Lecture bloc
 *---------------------------------------------------------------------*
 Les fins de lignes sont des caracteres LF.
 RETURNS: 
      0  - End-of-file
      -1 - Probleme de lecture
      >0 - Nombre de caracteres effectivement lus.
 *---------------------------------------------------------------------*/
extern 
int file_read(
        file_dstype fd,        /* Descripteur */
        char *buf,                /* Pointeur sur buffer de reception */
        int size                /* Nb max de caracteres a lire */
        );
/*---------------------------------------------------------------------*
			Ecriture bloc
 *---------------------------------------------------------------------*
 Les fins de lignes sont des caracteres LF.
 size < 0 => chaine terminee par \0
 RETURNS: 
      -1  - Probleme de lecture
      >=0 - Nb de caracteres effectivement e'crits.
 *---------------------------------------------------------------------*/
extern 
int file_write(
        file_dstype fd,        /* Descripteur */
        char *buf,                /* Pointeur sur buffer */
        int size                /* Nb de caracteres a ecrire */
        );
/************************************************************************
                        MODULE
 ************************************************************************/
/*----------------------------------------------------------------------*
                        Start
 *----------------------------------------------------------------------*/
extern
err_Type syst_FileStart(void * Cfg);
/*----------------------------------------------------------------------*
                        End
 *----------------------------------------------------------------------*/
extern
err_Type syst_FileEnd(void * Cfg);
/*----------------------------------------------------------------------*/
#endif
