/***********************************************************************
                SL Bus
 ***********************************************************************
 Each group of similar interrupt conditions can be processed by a 
 specific function : a Sub-Irq vector. 
 For instance : ST10 or CDCT P  or CDCT S. 
 
 When an SLBUS interrupt occurs, all sub-vectors are scanned and 
 the defined vectors are executed. 
 
 A vector function must return 0 if it detects one of its own interrupt 
 conditions, otherwise it must return sys__nobj. 
 In case of the detection of a condition : 
   The vector function must suppress the interrupt condition (disable or service)
   The vector function can signal (and schedule) waiting objects. 
   The vector function returns 0. 
 
 A sub-vector entry can be allocated by _IrqNew(). 
 With the _IrqDisabled code, the entry is reserved but not called. 
 With the _IrqNull code, the entry is released. 
 
 WARNING: If this module is used from outside the system (a user driver)
 a vector can not be modified directly. Use the _IrqUpdate function to 
 modify the vector with a user copy. 
 ***********************************************************************
 13/07/2006 - Marc LEGOUX - V2r22i: Add interrupt register definition
 13/05/2003 - Marc LEGOUX - V1r12: Adjusted to be a driver
 06/11/2000 - Marc LEGOUX - V1r08: Add task_IrqEvtType in object
 03/11/2000 - Marc LEGOUX - V1r08: register args for _IrqFuncType
 28/09/2000 - Marc LEGOUX - V1r07: Adapt for definition as a trap
 14/04/1999 - Marc LEGOUX - V1r02
 15/02/1999 - Marc LEGOUX - V1r01: Creation
 ***********************************************************************/
#ifndef slbus_h
#define slbus_h

#include <obj.h>
#include <systirq.h>

/*--------------------------------------------------------------------*
                Sub-IRQ vector type
 *--------------------------------------------------------------------*/
#define slbus_IrqLevel		5

typedef 
  err_Type slbus_IrqFuncType(register void *);

#define slbus_IrqNull		((slbus_IrqFuncType *)0)
#define slbus_IrqDisabled	((slbus_IrqFuncType *)1)
//#define slbus_IrqTask		((slbus_IrqFuncType *)2)

typedef
  struct {
    slbus_IrqFuncType * Irq;	// Vector : slbus_IrqNull if free. 
    void *		Arg;	// Vector parameter 
    syst_IrqEvtType	Evt;	// Irq Event 
    // From Version V2r22i
    ulong		IRAddr;	// Interrupt register address (or 0)
    ulong		IRMask;	// Interrupt register mask
    ulong		RFU[8];
  } slbus_IrqType;

/*--------------------------------------------------------------------*
                Init sub IRQ vector 
 slbus_IrqType * pIrq
 *--------------------------------------------------------------------*/
#define slbus_IrqInit(pIrq) ((pIrq) = (slbus_IrqType *)0)
/*--------------------------------------------------------------------*
                Alloc sub IRQ vector 
 *--------------------------------------------------------------------*
 Make first slbus_IrqInit(*ppIrq)
 or         slbus_IrqRemove(*ppIrq) if already allocated
 *--------------------------------------------------------------------*/
extern
err_Type slbus_IrqNew(
        slbus_IrqType ** ppIrq,		/* (OUT): New vector (*ppIrq must be Null)*/
        slbus_IrqFuncType * func	/* New vector function or Disabled */
        );
/*----------------------------------------------------------------------*
		Get Event in case of a task
 
 slbus_IrqType * Irq
 void         ** pEvt	(OUT): Irq Event
 *----------------------------------------------------------------------*/
#define slbus_IrqEvtGet(Irq, pEvt) ((*(pEvt) = &(Irq)->Evt),0)
/*----------------------------------------------------------------------*
                Update vector with user copy
 Must be used outside the system, because the vector data are not directly
 accessible.
 Call with pIrqUser==0 to get the current value. 
 *----------------------------------------------------------------------*/
extern
err_Type slbus_IrqUpdate(
        slbus_IrqType * pIrq,		/* Vector to update */
        slbus_IrqType * pIrqUser,	/* with this copy or null */
        slbus_IrqType * pIrqPrev	/* null or (OUT): Previous value */
        );
/*--------------------------------------------------------------------*
                Remove sub IRQ vector 
 slbus_IrqType * pIrq
 *--------------------------------------------------------------------*/
#define slbus_IrqRemove(pIrq) if (pIrq) {\
(pIrq)->Irq = slbus_IrqNull; (pIrq) = (slbus_IrqType *)0;}
/*--------------------------------------------------------------------*
                Disable Routine of sub IRQ vector 
 slbus_IrqType * pIrq
 *--------------------------------------------------------------------*/
#define slbus_IrqDisable(pIrq) if (pIrq) {\
(pIrq)->Irq = slbus_IrqDisabled;}
/*--------------------------------------------------------------------*
                Set routine of sub IRQ vector 
 slbus_IrqType * pIrq
 *--------------------------------------------------------------------*/
#define slbus_IrqSet(pIrq,func,arg) if (pIrq) {\
(pIrq)->Irq=(slbus_IrqFuncType *)(func);\
(pIrq)->Arg=(void *)(arg);}

/*--------------------------------------------------------------------*
                Define Interrupt Register of sub IRQ vector 
 slbus_IrqType * pIrq
 *--------------------------------------------------------------------*/
#define slbus_IrqIRDef(pIrq,Addr,Mask) if (pIrq) {\
(pIrq)->IRAddr=(ulong)(Addr);\
(pIrq)->IRMask=(Mask);}
/*--------------------------------------------------------------------*
                Start
 *--------------------------------------------------------------------*/
extern 
err_Type slbus_Start(void * cfg);
/*--------------------------------------------------------------------*
                End
 *--------------------------------------------------------------------*/
extern 
err_Type slbus_End(void * cfg);

#endif
