/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic HAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phbalReg.h>
#include <phhalHwContact.h>
#include <phTools.h>
#include <phToolsAtrParser.h>

#ifdef NXPBUILD__PHHAL_HW_CONTACT_SMARTWARE

#include "phhalHwContact_Smartware.h"
#include "phhalHwContact_Smartware_Int.h"

#include "ex/CARD/card.h"
#include "ex/CARDTM/cardtm.h"
#include "ex/CMT/cmt.H"
#include "ex/MLOS/mloscfg.h"

phStatus_t phhalHwContact_Smartware_Init(
    phhalHwContact_Smartware_DataParams_t * pDataParams,
    uint16_t wSizeOfDataParams,
    void * pBalDataParams,
    uint8_t * pTxBuffer,
    uint16_t  wTxBufSize,
    uint8_t * pRxBuffer,
    uint16_t  wRxBufSize,
    uint8_t * pIntBuffer,
    uint32_t wIntBufferSize
   )
{
    if (sizeof(phhalHwContact_Smartware_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    /* Init. private data */
    pDataParams->wId                = PH_COMP_HAL | PHHAL_HW_CONTACT_SMARTWARE_ID;
    pDataParams->pBalDataParams     = (phbalReg_Smartware_DataParams_t *)pBalDataParams;
    pDataParams->pTxBuffer          = pTxBuffer;
    pDataParams->wTxBufSize         = wTxBufSize;
    pDataParams->wTxBufLen          = 0;
    pDataParams->pRxBuffer          = pRxBuffer;
    pDataParams->wRxBufSize         = wRxBufSize;
    pDataParams->wRxBufLen          = 0;
    pDataParams->pIntBuffer         = pIntBuffer;
    pDataParams->dwIntBufferLen     = 0x00;
    pDataParams->dwIntBufferSize    = wIntBufferSize;
    pDataParams->wRxBufStartPos = 0;
    pDataParams->dwNumExpectedBytes = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_SetConfig32(
    phhalHwContact_Smartware_DataParams_t * pDataParams,
    uint16_t  wConfig,
    uint32_t  dwValue
    )
{
    phStatus_t  statusTmp = PH_ERR_SUCCESS;
    float64_t configValue = 0;
    uint8_t fValue = 0;
    uint8_t dValue = 0;

    switch (wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE:
        if ((dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_DISABLED &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_TX_LRC &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_TX_CRC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        if ((dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_DISABLED &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_RX_LRC &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_RX_CRC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE:
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_FD:
        fValue = (dwValue >> 8) && 0x000000FF;
        dValue = (dwValue && 0x000000FF);
        statusTmp = phhalHwContact_Smartware_Int_ConvertError(
            pDataParams->pBalDataParams->pfCardSetFd(pDataParams->pBalDataParams->CardBuffer, fValue, dValue));

        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_VCC:
        statusTmp = phhalHwContact_Smartware_Int_ConvertError(
            pDataParams->pBalDataParams->pfCardSetVcc(pDataParams->pBalDataParams->CardBuffer, (uint16_t)dwValue));

        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ:
        configValue = (float64_t)(dwValue / 1000.0);
        statusTmp = phhalHwContact_Smartware_Int_ConvertError(
            pDataParams->pBalDataParams->pfCardSetClk(pDataParams->pBalDataParams->CardBuffer, (uint64_t)configValue));

        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_TIMING_MODE:
        pDataParams->wTimingMode = (uint16_t)dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR:
    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
        statusTmp = phhalHwContact_Smartware_Int_ApplyConfiguration(pDataParams, wConfig, dwValue);

        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES:
        pDataParams->dwNumExpectedBytes = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS:
        /* Boundary check */
        if (dwValue > 0xFFFF || dwValue >= pDataParams->wRxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Set start position */
        pDataParams->wRxBufStartPos = (uint16_t)dwValue;
        pDataParams->wRxBufLen = (uint16_t)dwValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_GetConfig32(
    phhalHwContact_Smartware_DataParams_t * pDataParams,
    uint16_t   wConfig,
    uint32_t * pValue
    )
{
    phStatus_t statusTmp;

    switch (wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE:
    case PHHAL_HW_CONTACT_CONFIG_FD:
    case PHHAL_HW_CONTACT_CONFIG_VCC:
    case PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ:
    case PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONTACT_CONFIG_TIMING_MODE:
        *pValue = pDataParams->wTimingMode;
        break;

    case PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR:
    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
        statusTmp = phhalHwContact_Smartware_Int_GetConfiguration(pDataParams, wConfig, pValue);
        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        break;

    case PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES:
        *pValue = pDataParams->dwNumExpectedBytes;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS:
        *pValue = (uint32_t)pDataParams->wRxBufStartPos;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_BUFSIZE:
        *pValue = pDataParams->wRxBufSize - PHHAL_HW_CONTACT_MAX_CRC_BUFFER_LEN;
        break;

    case PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL:
        *pValue = PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD;
        break;

    default:
        *pValue = 0;
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_Wait(
    phhalHwContact_Smartware_DataParams_t * pDataParams,
    uint8_t   bUnit,
    uint16_t  wTimeout
    )
{
    if (bUnit == PHHAL_HW_CONTACT_TIME_MICROSECONDS)
    {
        phTools_Sleep((wTimeout + 999) / 1000);
    }
    else if (bUnit == PHHAL_HW_CONTACT_TIME_MILLISECONDS)
    {
        phTools_Sleep(wTimeout);
    }

    /* just to satisfy compiler */
    if(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_Exchange(
    phhalHwContact_Smartware_DataParams_t * pDataParams,
    uint16_t    wOption,
    uint8_t   * pTxBuffer,
    uint16_t    wTxLength,
    uint8_t  ** ppRxBuffer,
    uint16_t  * pRxLength
    )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t    PH_MEMLOC_REM pCrc[2];
    uint8_t    PH_MEMLOC_REM pLrc[1];
    uint8_t    PH_MEMLOC_REM bLrc;
    uint16_t   PH_MEMLOC_REM wCrc;
    uint32_t   PH_MEMLOC_REM pFdt = 0;
    uint32_t   PH_MEMLOC_REM dwProtocol;
    uint32_t   PH_MEMLOC_REM dwRecvBytes = 256;
    uint16_t   PH_MEMLOC_REM wCrcLength;
    uint8_t *  PH_MEMLOC_REM pCrcData;

    /* Check if caller has provided valid RxBuffer */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pDataParams->pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &pDataParams->wRxBufLen;
    }

    /* Reset received length */
    *pRxLength = 0;

    /* copy data */
    memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pTxBuffer, wTxLength); /* PRQA S 3200 */
    pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;

    /* do not perform real exchange, just fill the global TxBuffer */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_TX) == PHHAL_HW_CONTACT_MODE_TX_CRC)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_CONTACTBASED,
            PH_TOOLS_CRC16_POLY_CONTACTBASED,
            pTxBuffer,
            wTxLength,
            &wCrc));

        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED) == PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED)
        {
            pCrc[0] = (uint8_t)(wCrc >> 8);
            pCrc[1] = (uint8_t)(wCrc);
        }
        else
        {
            pCrc[0] = (uint8_t)(wCrc);
            pCrc[1] = (uint8_t)(wCrc >> 8);
        }

        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pCrc, 2); /* PRQA S 3200 */
        pDataParams->wTxBufLen += 2;
    }
    else if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_TX) == PHHAL_HW_CONTACT_MODE_TX_LRC)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateLrc(pTxBuffer, wTxLength, pLrc));

        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pLrc, 1); /* PRQA S 3200 */
        pDataParams->wTxBufLen += 1;
    }

    /* copy data to pIntTmpBuffer as this buffer is used as RxBuffer afterwards */
    memcpy(pDataParams->pIntBuffer, &pDataParams->pTxBuffer[0], pDataParams->wTxBufLen); /* PRQA S 3200 */
    pDataParams->dwIntBufferLen = (uint32_t)pDataParams->wTxBufLen;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, &dwProtocol));

    if(dwProtocol == PHHAL_HW_CONTACT_PROTOCOLTYPE_T0)
    {
        dwRecvBytes = pDataParams->dwNumExpectedBytes;

        /* exchange data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Smartware_Int_ConvertError(
            pDataParams->pBalDataParams->pfCardExchange(
            pDataParams->pBalDataParams->CardBuffer, card_T0 | card_T1_RxLgDisable, pDataParams->pIntBuffer, pDataParams->dwIntBufferLen, pDataParams->pIntBuffer, &dwRecvBytes)));
    }
    else
    {
        /* exchange data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Smartware_Int_ConvertError(
            pDataParams->pBalDataParams->pfCardExchange(
            pDataParams->pBalDataParams->CardBuffer, card_T1, pDataParams->pIntBuffer, (uint16_t)pDataParams->dwIntBufferLen, pDataParams->pIntBuffer, &dwRecvBytes)));
    }

    pDataParams->dwIntBufferLen = dwRecvBytes;

    /* Reset buffered bytes */
    pDataParams->wTxBufLen = 0;

    /* if success copy data to RxBuffer */
    if(pDataParams->dwIntBufferLen <= pDataParams->wRxBufSize)
    {
        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_CRC)
        {
            /* Smartware wants to receive max +1 byte for LRC so second byte for CRC has to be received manually afterwards */
            dwRecvBytes = 1;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Smartware_Int_ConvertError(
                pDataParams->pBalDataParams->pfCardExchange(
                pDataParams->pBalDataParams->CardBuffer, card_T1 | card_T1_RxLgDisable, 0, 0, &pDataParams->pIntBuffer[pDataParams->dwIntBufferLen], &dwRecvBytes)));
            pDataParams->dwIntBufferLen += dwRecvBytes;
        }

        memcpy(&pDataParams->pRxBuffer[0], pDataParams->pIntBuffer, pDataParams->dwIntBufferLen); /* PRQA S 3200 */
        pDataParams->wRxBufLen = (uint16_t)pDataParams->dwIntBufferLen;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    if(pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_COMM || pDataParams->wTimingMode == PHHAL_HW_TIMING_MODE_FDT)
    {
        /* get execution time of last exchange */
        //PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_RdScr01_Int_ConvertError(
        //    pDataParams->pBalDataParams->pf7816GetFdt(&pFdt)));

        if(pFdt != 0)
        {
            pDataParams->qwTiming_ns = pFdt * 1000;
        }
    }

    /* get data for CRC check */
    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER) == PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER)
    {
        wCrcLength = pDataParams->wRxBufLen;
        pCrcData = pDataParams->pRxBuffer;
    }
    else
    {
        wCrcLength = pDataParams->wRxBufLen - pDataParams->wRxBufStartPos;
        pCrcData = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
    }

    /* Check and remove CRC in software if required */
    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_CRC)
    {
        if (wCrcLength < 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_CONTACTBASED,
            PH_TOOLS_CRC16_POLY_CONTACTBASED,
            pCrcData,
            wCrcLength - 2,
            &wCrc));

        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED) == PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED)
        {
            pCrc[0] = (uint8_t)(wCrc >> 8);
            pCrc[1] = (uint8_t)(wCrc);
        }
        else
        {
            pCrc[0] = (uint8_t)(wCrc);
            pCrc[1] = (uint8_t)(wCrc >> 8);
        }

        /* CRC over all data must be zero */
        if (pCrc[0] != pCrcData[wCrcLength-2] || pCrc[1] != pCrcData[wCrcLength-1])
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        pDataParams->wRxBufLen -= 2;
    }
    else if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_LRC)
    {
        if (wCrcLength < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateLrc(
            pCrcData,
            wCrcLength,
            &bLrc));

        /* LRC over all data must be zero */
        if (bLrc != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        pDataParams->wRxBufLen -= 1;
    }

    *ppRxBuffer = pDataParams->pRxBuffer;
    *pRxLength = pDataParams->wRxBufLen;

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_ApplyProtocolSettings(
    phhalHwContact_Smartware_DataParams_t * pDataParams,
    uint8_t * pAtr,
    uint16_t dwAtrLength,
    uint8_t bProtocolType
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint32_t    PH_MEMLOC_REM wConfig;
    uint8_t     PH_MEMLOC_REM bConfig = 0;
    uint8_t     PH_MEMLOC_REM bSupported;
    uint32_t    PH_MEMLOC_REM wFDIndexValue;

    /* if ATR available check if card supports specified ProtocolType */
    if(dwAtrLength != 0 && pAtr != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(pAtr, dwAtrLength));
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_CheckIfProtocolSupported(bProtocolType, &bSupported));

        if(!bSupported)
        {
            /* not supported so reset to default values */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
    }

    /* Store new protocol type */
    if (bProtocolType != PHHAL_HW_CONTACT_PROTOCOLTYPE_CURRENT)
    {
        pDataParams->bProtocolType = bProtocolType;
    }

    /* Apply global Parameters */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, &wFDIndexValue));

    /* Apply protocol dependent Parameters */
    switch(bProtocolType)
    {
    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T0:
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_WTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_WT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_GTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_GT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, PHHAL_HW_CONTACT_MODE_DISABLED));
        break;

    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T1:
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CGTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CGT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CWTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CWT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_BWTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_BWT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_RedundancyMode(&bConfig));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, (uint32_t)bConfig));

        pDataParams->dwNumExpectedBytes = PHHAL_HW_CONTACT_MAX_T1_BLOCK_SIZE;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, bProtocolType));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_Activate(
                        phhalHwContact_Smartware_DataParams_t * pDataParams,
                        uint8_t * pATRBuffer,
                        uint16_t * pwATRLength
                        )
{
    phStatus_t statusTmp;
    phStatus_t status = PH_ERR_SUCCESS;
    card_Type * CardType = (card_Type *)pDataParams->pBalDataParams->CardBuffer;
    uint8_t bColdReset;

    if(pDataParams->dwAtrLength == 0)
    {
        bColdReset = 1;
    }
    else
    {
        bColdReset = 0;
    }

    /* Reset ATR before doing anything else */
    memset(pDataParams->pAtr, 0x00, PHHAL_HW_CONTACT_MAX_ATR_SIZE);
    pDataParams->dwAtrLength = 0;

    /* Before activation bring reader to default state */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    if(bColdReset)
    {
        status = phhalHwContact_Smartware_Int_ConvertError(
            pDataParams->pBalDataParams->pfCardReset(CardType));
    }
    else
    {
        status = phhalHwContact_Smartware_Int_ConvertError(
            pDataParams->pBalDataParams->pfCardResetWarm(CardType));
    }

    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        if(CardType->ATRLg > PHHAL_HW_CONTACT_MAX_ATR_SIZE)
        {
            *pwATRLength = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        memcpy(pATRBuffer, CardType->ATR, CardType->ATRLg);
        *pwATRLength = (uint16_t)CardType->ATRLg;
        pDataParams->dwAtrLength = (uint32_t)CardType->ATRLg;

        /* Set ATR in phToolsAtrParser to parse certain values later */
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(pATRBuffer, *pwATRLength));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));
    }
    else
    {
        *pwATRLength = 0;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_Deactivate(
                        phhalHwContact_Smartware_DataParams_t * pDataParams
                        )
{
    phStatus_t statusTmp;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Smartware_Int_ConvertError(
        pDataParams->pBalDataParams->pfCardOff((card_Type *)pDataParams->pBalDataParams->CardBuffer)));
    pDataParams->dwAtrLength = 0;
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_ClockStop(
                        phhalHwContact_Smartware_DataParams_t * pDataParams,
                        uint16_t wTgClockCount,
                        uint16_t wThClockCount,
                        uint32_t dwPinState
                        )
{
    /* just to satisfy compiler */
    if(pDataParams || wTgClockCount || wThClockCount || dwPinState);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_InitContactReader(
                                                      phhalHwContact_Smartware_DataParams_t * pDataParams
                                                      )
{
    phStatus_t statusTmp;

    /* Set internal default values for communication */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());

    /* Set default Datarate */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));

    /* Set default Max Clk Atr timing */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR, PHHAL_HW_CONTACT_MAX_CLK_ATR_DEFAULT));

    /* ApplyProtocolSettings to have default values applies */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    /* Deactivate set power and ATR length to zero */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Deactivate(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_CONTACT_SMARTWARE */
