/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software Internal ISO14443-4 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalI14443p4.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHPAL_I14443P4_SW

#include "phpalI14443p4_Sw_Int.h"
#include "phpalI14443p4_Sw.h"

phStatus_t phpalI14443p4_Sw_Int_BuildSParamBlock(
                                        uint8_t bCidEnabled,
                                        uint8_t bCid,
                                        uint8_t * pTxBuffer,
                                        uint16_t * pTxLength)
{
    /* S-Block PCB */
    pTxBuffer[PHPAL_I14443P4_SW_PCB_POS]  = PHPAL_I14443P4_SW_S_BLOCK | PHPAL_I14443P4_SW_PCB_PARAM | PHPAL_I14443P4_SW_S_BLOCK_RFU_BITS;
    *pTxLength = 1;

    /* Append CID if supported */
    if (bCidEnabled != 0)
    {
        pTxBuffer[PHPAL_I14443P4_SW_PCB_POS] |= PHPAL_I14443P4_SW_PCB_CID_FOLLOWING;
        pTxBuffer[(*pTxLength)++] = bCid;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_ExchangeSParamFrame(
                                     phpalI14443p4_Sw_DataParams_t * pDataParams,
                                     uint8_t *pDataIn,
                                     uint16_t wDataInLength,
                                     uint8_t *pDataOut,
                                     uint16_t wDataOutSize,
                                     uint16_t *pwDataOutLength
                                     )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bIsoFrameHeader[10];
    uint16_t    PH_MEMLOC_REM wIsoFrameHeaderLen;
    uint8_t     PH_MEMLOC_REM bTLHeader[10];
    uint16_t    PH_MEMLOC_REM bTLHeaderLen;
    uint8_t *   PH_MEMLOC_REM pResp;
    uint16_t    PH_MEMLOC_REM wRespLen;
    uint8_t     PH_MEMLOC_REM bInvalidBlock;
    uint8_t     PH_MEMLOC_REM bResponseReceived;
    uint16_t    PH_MEMLOC_REM wRetries;
    uint16_t    PH_MEMLOC_REM wOrigTimeout;
    uint16_t    PH_MEMLOC_REM wOrigTimeoutConfig;
    uint32_t    PH_MEMLOC_REM dwDataLength;
    uint16_t    PH_MEMLOC_REM wLengthLength;

    /* Length check fixxme is this needed? or use 2 byte length tag */
    if (wDataInLength >= 0x80)
    {
        PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    /* Build S(DESELECT) frame */
    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4_Sw_Int_BuildSParamBlock(
        pDataParams->bCidEnabled,
        pDataParams->bCid,
        bIsoFrameHeader,
        &wIsoFrameHeaderLen));

    /* Create TL Header (value is added later)*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4_Sw_Int_Attach_TLV_Header(bTLHeader, wDataInLength, sizeof(bTLHeader), PHPAL_I14443P4_SPARAM_BLOCK_INFO_TAG, &bTLHeaderLen));
    memcpy(&bIsoFrameHeader[wIsoFrameHeaderLen], &bTLHeader[sizeof(bTLHeader) - bTLHeaderLen], bTLHeaderLen);
    wIsoFrameHeaderLen += bTLHeaderLen;

    /* Get Orig timeout */
    statusTmp = phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
        &wOrigTimeout);
    if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        wOrigTimeoutConfig = PHHAL_HW_CONFIG_TIMEOUT_VALUE_US;
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,
            &wOrigTimeout));
        wOrigTimeoutConfig = PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS;
    }

    /* Set S(Param) timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
        PHPAL_I14443P4_SW_FWT_SPARAM_US + PHPAL_I14443P4_EXT_TIME_US));

    /* Retrieve max. retry count */
    wRetries = (uint16_t)pDataParams->bMaxRetryCount + 1;

    /* Reset response received flag */
    bResponseReceived = 0;

    /* Do as long as invalid responses are received
    and the retry counter has not reached zero.*/
    do
    {
        /* Send the S(Param) */
        PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_VHBR_Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            bIsoFrameHeader,
            wIsoFrameHeaderLen,
            &pResp,
            &wRespLen));

        status = phpalI14443p4_Sw_Int_VHBR_Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            pDataIn,
            wDataInLength,
            &pResp,
            &wRespLen);

        /* Status --> InvalidBlock mapping */
        bInvalidBlock = (uint8_t)PHPAL_I14443P4_SW_IS_INVALID_BLOCK_STATUS(status);
        if (!bInvalidBlock)
        {
            /* Restore timeout */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                wOrigTimeoutConfig,
                wOrigTimeout));

            /* Check for other errors */
            PH_CHECK_SUCCESS(status);

            /* Signal that we've received something */
            bResponseReceived = 1;

            status = phpalI14443p4_Sw_IsValidSBlock(
                pDataParams->bCidEnabled,
                pDataParams->bCid,
                pResp,
                wRespLen);
            if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                break;
            }

            if (!PHPAL_I14443P4_SW_IS_PARAM(pResp[PHPAL_I14443P4_SW_PCB_POS]))
            {
                status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                break;
            }

            /* Remove S block header */
            pResp++;
            wRespLen--;
            if (pDataParams->bCidEnabled)
            {
                pResp++;
                wRespLen--;
            }

            /* Check S(Param) data */
            if (wRespLen < 2)
            {
                status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                break;
            }

            if (pResp[0] != PHPAL_I14443P4_SPARAM_BLOCK_INFO_TAG)
            {
                status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                break;
            }

            /* Extract data from returned frame */
            PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4_Sw_Int_Parse_TLV_Len(pResp, wRespLen, 1, &dwDataLength, &wLengthLength));
            if (wRespLen != 1 + dwDataLength + wLengthLength)
            {
                status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                break;
            }

            if (dwDataLength > wDataOutSize)
            {
                status = PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
                break;
            }

            memcpy(pDataOut, &pResp[1 + wLengthLength], (uint16_t)dwDataLength);
            *pwDataOutLength = (uint16_t)dwDataLength;
        }
    }
    /* Retry as long as neccessary */
    while ((bInvalidBlock) && (--wRetries));

    /* Restore timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
        pDataParams->pHalDataParams,
        wOrigTimeoutConfig,
        wOrigTimeout));

    /* Operation not successful */
    if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        /* Return ERR_RECOVERY_FAILED if some response has been received before (bMaxRetryCount = 0 suppresses the retry behaviour) */
        if ((pDataParams->bMaxRetryCount > 0) && (bResponseReceived))
        {
            status = PH_ADD_COMPCODE(PHPAL_I14443P4_ERR_RECOVERY_FAILED, PH_COMP_PAL_ISO14443P4);
        }
    }

    return status;
}

phStatus_t phpalI14443p4_Sw_Int_SParamCheckSuppliedParameter(phpalI14443p4_Sw_DataParams_t * pDataParams,
                                           phpalI14443p4_Sw_SParam_Param * pParameter)
{
    uint8_t bPcdStartStopBitSuppression = PH_OFF;
    uint8_t bPcdSofEofSuppression = PH_OFF;
    uint8_t bPcdSyncSuppression = PH_OFF;
    uint8_t bPiccStartStopBitSuppression = PH_OFF;
    uint8_t bPiccSofEofSuppression = PH_OFF;
    uint8_t bPiccSyncSuppression = PH_OFF;
    uint8_t     PH_MEMLOC_REM bSetBitCnt;
    uint16_t    PH_MEMLOC_REM wCounter;


    /* Satisfy compiler */
    if (pDataParams);

    /* Bit Rate tests */

    /* If data is not supplied it will not be used in the S(Param) activate so only check supplied values */
    if (pParameter->BitRatePcd2Picc.bSupplied == PH_ON)
    {
        if ((pParameter->BitRatePcd2Picc.wSuppliedValue & PHPAL_I14443P4_SPARAM_DATARATE_RFU_MASK) != PHPAL_I14443P4_SPARAM_DATARATE_RFU_BITS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
        bSetBitCnt = 0;
        for (wCounter = 0; wCounter < 16; wCounter++)
        {
            if (((((pParameter->BitRatePcd2Picc.wSuppliedValue) & (~PHPAL_I14443P4_SPARAM_DATARATE_RFU_MASK)) >> wCounter) & 0x0001)  == 0x0001)
            {
                bSetBitCnt++;
            }
        }
        if (bSetBitCnt != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    if (pParameter->BitRatePicc2Pcd.bSupplied == PH_ON)
    {
        if ((pParameter->BitRatePicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_DATARATE_RFU_MASK) != PHPAL_I14443P4_SPARAM_DATARATE_RFU_BITS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
        bSetBitCnt = 0;
        for (wCounter = 0; wCounter < 16; wCounter++)
        {
            if (((((pParameter->BitRatePicc2Pcd.wSuppliedValue) & (~PHPAL_I14443P4_SPARAM_DATARATE_RFU_MASK)) >> wCounter) & 0x0001)  == 0x0001)
            {
                bSetBitCnt++;
            }
        }
        if (bSetBitCnt != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    if (pParameter->BitRateFramingOptionsPicc2Pcd.bSupplied == PH_ON)
    {
        bPiccStartStopBitSuppression = ((pParameter->BitRateFramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) ? PH_ON : PH_OFF;
        bPiccSofEofSuppression = ((pParameter->BitRateFramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) ? PH_ON : PH_OFF;
        /*FIXXME add RFU Check */
        if (bPiccStartStopBitSuppression && bPiccSofEofSuppression)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }

    /* FIXXME If start bit and stop bit suppression is selected, SOF and EOF low time of 10 etu and SOF high time of 2 etu shall be used */
    /* FIXXME check if type A is used some combinations are not possible */



    /* Frame format tests */
    if (pParameter->FramePcd2Picc.bSupplied == PH_ON &&
        pParameter->FramePcd2Picc.wSuppliedValue != PHPAL_I14443P4_SPARAM_FRAME_STANDARD &&
        pParameter->FramePcd2Picc.wSuppliedValue != PHPAL_I14443P4_SPARAM_FRAME_WEC)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }
    if (pParameter->FramePicc2Pcd.bSupplied == PH_ON &&
        pParameter->FramePicc2Pcd.wSuppliedValue != PHPAL_I14443P4_SPARAM_FRAME_STANDARD &&
        pParameter->FramePicc2Pcd.wSuppliedValue != PHPAL_I14443P4_SPARAM_FRAME_WEC)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    /* 10.5, Table 6, point e
    When SYNC is suppressed the PCD shall not select both start bit and stop bit suppression and SOF and EOF suppression.
    If start bit and stop bit suppression is selected, SOF and EOF low time of 10 etu and SOF high time of 2 etu shall be used */
    if (pParameter->FramingOptionsPcd2Picc.bSupplied == PH_ON)
    {
        bPcdStartStopBitSuppression = ((pParameter->FramingOptionsPcd2Picc.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) ? PH_ON : PH_OFF;
        bPcdSofEofSuppression = ((pParameter->FramingOptionsPcd2Picc.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) ? PH_ON : PH_OFF;
        bPcdSyncSuppression = ((pParameter->FramingOptionsPcd2Picc.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC) ? PH_ON : PH_OFF;

        /*FIXXME add RFU Check */

        if (bPcdSyncSuppression && bPcdStartStopBitSuppression && bPcdSofEofSuppression)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    if (pParameter->FramingOptionsPicc2Pcd.bSupplied == PH_ON)
    {
        bPiccStartStopBitSuppression = ((pParameter->FramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) ? PH_ON : PH_OFF;
        bPiccSofEofSuppression = ((pParameter->FramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) ? PH_ON : PH_OFF;
        bPiccSyncSuppression = ((pParameter->FramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC) ? PH_ON : PH_OFF;
        /*FIXXME add RFU Check */
        if (bPiccSyncSuppression && bPiccStartStopBitSuppression && bPiccSofEofSuppression)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }

    /* FIXXME If start bit and stop bit suppression is selected, SOF and EOF low time of 10 etu and SOF high time of 2 etu shall be used */
    /* FIXXME check if type A is used some combinations are not possible */


    /* Fixxme at check if Bit Rate Framing Options match Framing options */


    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_SParamExtractIndicatedParam(phpalI14443p4_Sw_DataParams_t * pDataParams,
                                           phpalI14443p4_Sw_SParam_Param * pParameter,
                                           uint8_t *pbSParamDataBuffer,
                                           uint16_t wSParamDataLen,
                                           uint8_t bBitRateIndicationSupported,
                                           uint8_t bFrameFormatIndicationSupported)
{
    uint16_t wCurrentIndicationTagBufferLength;
    uint16_t wTagOffset = 0;
    uint32_t dwDataLength;
    uint16_t wLengthLength;
    phStatus_t status;

    /* Satisfy compiler */
    if (pDataParams);

    /* FIXXME check for double returned values */

    do
    {
        /* At least an empty tag must be indicated */
        if (2 + wTagOffset > wSParamDataLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
        }

        /* Extract data */
        PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Parse_TLV_Len(pbSParamDataBuffer, wSParamDataLen, wTagOffset + 1, &dwDataLength, &wLengthLength));
        wCurrentIndicationTagBufferLength = (uint16_t)(wTagOffset + 1 + dwDataLength + wLengthLength);
        wTagOffset += (1 + wLengthLength);

        switch(pbSParamDataBuffer[wTagOffset - (1 + wLengthLength)])
        {
        case PHPAL_I14443P4_SPARAM_BIT_RATE_INDICATION:
            {
                if (bBitRateIndicationSupported == PH_OFF)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                }
                for (;wTagOffset < wCurrentIndicationTagBufferLength;)
                {
                    PH_CHECK_SUCCESS_FCT(status,
                        phpalI14443p4_Sw_Int_Parse_TLV_Len(pbSParamDataBuffer, wCurrentIndicationTagBufferLength, wTagOffset + 1, &dwDataLength, &wLengthLength));

                    switch(pbSParamDataBuffer[wTagOffset])
                    {
                    case PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_PCD_TAG:
                        {
                            if (dwDataLength != PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_PCD_TAG_LENGTH || wTagOffset + 1 + dwDataLength + wLengthLength > wCurrentIndicationTagBufferLength)
                            {
                                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                            }
                            pParameter->BitRatePcd2Picc.bIndicated = PH_ON;
                            pParameter->BitRatePcd2Picc.wIndicatedValue = pbSParamDataBuffer[wTagOffset + 1 + wLengthLength]*256 + pbSParamDataBuffer[wTagOffset + 1 + wLengthLength + 1];
                            /*FIXXME check for valid data */
                            break;
                        }
                    case PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_PICC_TAG:
                        {
                            if (dwDataLength != PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_PICC_TAG_LENGTH || wTagOffset + 1 + dwDataLength + wLengthLength > wCurrentIndicationTagBufferLength)
                            {
                                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                            }
                            pParameter->BitRatePicc2Pcd.bIndicated = PH_ON;
                            pParameter->BitRatePicc2Pcd.wIndicatedValue = pbSParamDataBuffer[wTagOffset + 1 + wLengthLength]*256 + pbSParamDataBuffer[wTagOffset + 1 + wLengthLength + 1];
                            /*FIXXME check for valid data */
                            break;
                        }
                    case PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_FRAME_OPTION_PICC_TAG:
                        {
                            if (dwDataLength != PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_FRAME_OPTION_PICC_TAG_LENGTH || wTagOffset + 1 + dwDataLength + wLengthLength > wCurrentIndicationTagBufferLength)
                            {
                                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                            }
                            pParameter->BitRateFramingOptionsPicc2Pcd.bIndicated = PH_ON;
                            pParameter->BitRateFramingOptionsPicc2Pcd.wIndicatedValue = pbSParamDataBuffer[wTagOffset + 1 + wLengthLength];
                            /*FIXXME check for valid data */
                            break;
                        }
                    default:
                        {
                            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                        }
                    }
                    wTagOffset += (1 + (uint16_t)dwDataLength + wLengthLength);
                }
                /* Check if the data length of the current indication tag match with the parsed data */
                if (wTagOffset != wCurrentIndicationTagBufferLength)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                }
                break;
            }
        case PHPAL_I14443P4_SPARAM_FRAME_FORMAT_INDICATION:
            {
                /*FIXXME */
                if (bFrameFormatIndicationSupported == PH_OFF)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                }
                for (;wTagOffset < wCurrentIndicationTagBufferLength;)
                {
                    PH_CHECK_SUCCESS_FCT(status,
                        phpalI14443p4_Sw_Int_Parse_TLV_Len(pbSParamDataBuffer, wCurrentIndicationTagBufferLength, wTagOffset + 1, &dwDataLength, &wLengthLength));

                    switch(pbSParamDataBuffer[wTagOffset])
                    {
                    case PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMES_PCD_TAG:
                        {
                            if (dwDataLength != PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMES_PCD_TAG_LENGTH || wTagOffset + 1 + dwDataLength + wLengthLength > wCurrentIndicationTagBufferLength)
                            {
                                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                            }
                            pParameter->FramePcd2Picc.bIndicated = PH_ON;
                            pParameter->FramePcd2Picc.wIndicatedValue = pbSParamDataBuffer[wTagOffset + 1 + wLengthLength];
                            /*FIXXME check for valid data */
                            break;
                        }
                    case PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMES_PICC_TAG:
                        {
                            if (dwDataLength != PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMES_PICC_TAG_LENGTH || wTagOffset + 1 + dwDataLength + wLengthLength > wCurrentIndicationTagBufferLength)
                            {
                                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                            }
                            pParameter->FramePicc2Pcd.bIndicated = PH_ON;
                            pParameter->FramePicc2Pcd.wIndicatedValue = pbSParamDataBuffer[wTagOffset + 1 + wLengthLength];
                            /*FIXXME check for valid data */
                            break;
                        }
                   case PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMING_OPTION_PCD_TAG:
                        {
                            if (dwDataLength != PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMING_OPTION_PCD_TAG_LENGTH || wTagOffset + 1 + dwDataLength + wLengthLength > wCurrentIndicationTagBufferLength)
                            {
                                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                            }
                            pParameter->FramingOptionsPcd2Picc.bIndicated = PH_ON;
                            pParameter->FramingOptionsPcd2Picc.wIndicatedValue = pbSParamDataBuffer[wTagOffset + 1 + wLengthLength];
                            /*FIXXME check for valid data */
                            break;
                        }
                    case PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMING_OPTION_PICC_TAG:
                        {
                            if (dwDataLength != PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMING_OPTION_PICC_TAG_LENGTH || wTagOffset + 1 + dwDataLength + wLengthLength > wCurrentIndicationTagBufferLength)
                            {
                                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                            }
                            pParameter->FramingOptionsPicc2Pcd.bIndicated = PH_ON;
                            pParameter->FramingOptionsPicc2Pcd.wIndicatedValue = pbSParamDataBuffer[wTagOffset + 1 + wLengthLength];
                            /*FIXXME check for valid data */
                            break;
                        }
                    default:
                        {
                            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                        }
                    }
                    wTagOffset += (1 + (uint16_t)dwDataLength + wLengthLength);
                }
                /* Check if the data length of the current indication tag match with the parsed data */
                if (wTagOffset != wCurrentIndicationTagBufferLength)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                }
                break;
            }
        default:
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);

        }
    }while(wTagOffset < wSParamDataLen);
    /* Check if all available data match with parsed data */
    if (wTagOffset != wSParamDataLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_SParamCheckSuppliedParameterAreSupported(phpalI14443p4_Sw_SParam_Param * pParameter)
{
    /* Check if all parameters are supported */
    if (pParameter->BitRatePcd2Picc.bSupplied == PH_ON)
    {
        if (pParameter->BitRatePcd2Picc.bIndicated != PH_ON ||
            (pParameter->BitRatePcd2Picc.wIndicatedValue & pParameter->BitRatePcd2Picc.wSuppliedValue) != pParameter->BitRatePcd2Picc.wSuppliedValue)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    if (pParameter->BitRatePicc2Pcd.bSupplied == PH_ON)
    {
        if (pParameter->BitRatePicc2Pcd.bIndicated != PH_ON ||
            (pParameter->BitRatePicc2Pcd.wIndicatedValue & pParameter->BitRatePicc2Pcd.wSuppliedValue) != pParameter->BitRatePicc2Pcd.wSuppliedValue)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    if (pParameter->BitRateFramingOptionsPicc2Pcd.bSupplied == PH_ON)
    {
        if (pParameter->BitRateFramingOptionsPicc2Pcd.bIndicated != PH_ON ||
            ((uint8_t)pParameter->BitRateFramingOptionsPicc2Pcd.wIndicatedValue & (uint8_t)pParameter->BitRateFramingOptionsPicc2Pcd.wSuppliedValue) != (uint8_t)pParameter->BitRateFramingOptionsPicc2Pcd.wSuppliedValue)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }

    if (pParameter->FramePcd2Picc.bSupplied == PH_ON)
    {
        if (pParameter->FramePcd2Picc.bIndicated != PH_ON ||
            ((uint8_t)pParameter->FramePcd2Picc.wIndicatedValue & (uint8_t)pParameter->FramePcd2Picc.wSuppliedValue) != (uint8_t)pParameter->FramePcd2Picc.wSuppliedValue)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    if (pParameter->FramePicc2Pcd.bSupplied == PH_ON)
    {
        if (pParameter->FramePicc2Pcd.bIndicated != PH_ON ||
            ((uint8_t)pParameter->FramePicc2Pcd.wIndicatedValue & (uint8_t)pParameter->FramePicc2Pcd.wSuppliedValue) != (uint8_t)pParameter->FramePicc2Pcd.wSuppliedValue)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    /* FIXXME Only same frame format in both directions, if bit is set to (1)b */
    if (pParameter->FramingOptionsPcd2Picc.bSupplied == PH_ON)
    {
        if (pParameter->FramingOptionsPcd2Picc.bIndicated != PH_ON ||
            ((uint8_t)pParameter->FramingOptionsPcd2Picc.wIndicatedValue & (uint8_t)pParameter->FramingOptionsPcd2Picc.wSuppliedValue) != (uint8_t)pParameter->FramingOptionsPcd2Picc.wSuppliedValue)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    if (pParameter->FramingOptionsPicc2Pcd.bSupplied == PH_ON)
    {
        if (pParameter->FramingOptionsPicc2Pcd.bIndicated != PH_ON ||
            ((uint8_t)pParameter->FramingOptionsPicc2Pcd.wIndicatedValue & (uint8_t)pParameter->FramingOptionsPicc2Pcd.wSuppliedValue) != (uint8_t)pParameter->FramingOptionsPicc2Pcd.wSuppliedValue)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_SParamGetAutoParameter(phpalI14443p4_Sw_DataParams_t * pDataParams,
                                           phpalI14443p4_Sw_SParam_Param * pParameter)
{
    /* Satisfy compiler */
    if (pDataParams);

    /* Get best parameter */
    /*FIXXME */
    if (pParameter->BitRatePcd2Picc.bIndicated == PH_ON)
    {
        pParameter->BitRatePcd2Picc.bSupplied = PH_ON;
        /*FIXXME */
        pParameter->BitRatePcd2Picc.wSuppliedValue = 0x0100;
    }
    else
    {
        pParameter->BitRatePcd2Picc.bSupplied = PH_OFF;
    }

    if (pParameter->BitRatePicc2Pcd.bIndicated == PH_ON)
    {
        pParameter->BitRatePicc2Pcd.bSupplied = PH_ON;
        /*FIXXME */
        pParameter->BitRatePicc2Pcd.wSuppliedValue = 0x0100;
    }
    else
    {
        pParameter->BitRatePicc2Pcd.bSupplied = PH_OFF;
    }

    if (pParameter->BitRateFramingOptionsPicc2Pcd.bIndicated == PH_ON)
    {
        pParameter->BitRateFramingOptionsPicc2Pcd.bSupplied = PH_ON;
        /*FIXXME */
        pParameter->BitRateFramingOptionsPicc2Pcd.wSuppliedValue = 0x00;
    }
    else
    {
        pParameter->BitRateFramingOptionsPicc2Pcd.bSupplied = PH_OFF;
    }

    if (pParameter->FramePcd2Picc.bIndicated == PH_ON)
    {
        pParameter->FramePcd2Picc.bSupplied = PH_ON;
        /*FIXXME */
        pParameter->FramePcd2Picc.wSuppliedValue = PHPAL_I14443P4_SPARAM_FRAME_STANDARD;
    }
    else
    {
        pParameter->FramePcd2Picc.bSupplied = PH_OFF;
    }

    if (pParameter->FramePicc2Pcd.bIndicated == PH_ON)
    {
        pParameter->FramePicc2Pcd.bSupplied = PH_ON;
        /*FIXXME */
        pParameter->FramePicc2Pcd.wSuppliedValue = PHPAL_I14443P4_SPARAM_FRAME_STANDARD;
    }
    else
    {
        pParameter->FramePicc2Pcd.bSupplied = PH_OFF;
    }

    if (pParameter->FramingOptionsPcd2Picc.bIndicated == PH_ON)
    {
        pParameter->FramingOptionsPcd2Picc.bSupplied = PH_ON;
        /*FIXXME */
        pParameter->FramingOptionsPcd2Picc.wSuppliedValue = 0;
    }
    else
    {
        pParameter->FramingOptionsPcd2Picc.bSupplied = PH_OFF;
    }

    if (pParameter->FramingOptionsPicc2Pcd.bIndicated == PH_ON)
    {
        pParameter->FramingOptionsPicc2Pcd.bSupplied = PH_ON;
        /*FIXXME */
        pParameter->FramingOptionsPicc2Pcd.wSuppliedValue = 0;
    }
    else
    {
        pParameter->FramingOptionsPicc2Pcd.bSupplied = PH_OFF;
    }

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_SParamBuildActivateData(phpalI14443p4_Sw_DataParams_t * pDataParams,
                                           phpalI14443p4_Sw_SParam_Param * pParameter,
                                           uint8_t *pbSParamDataBuffer,
                                           uint16_t wSParamDataSize,
                                           uint16_t *pwSParamDataStartPos,
                                           uint8_t bBitRateIndicationSupported,
                                           uint8_t bFrameFormatIndicationSupported)
{
    uint16_t wSParamDataPos;
    uint16_t wSParamDataLen = 0;
    phStatus_t status;

    /* Satisfy compiler */
    if (pDataParams);

    memset(pbSParamDataBuffer, 0, wSParamDataSize);
    wSParamDataPos = wSParamDataSize;
    if (bBitRateIndicationSupported &&
        ((pParameter->BitRatePcd2Picc.bSupplied == PH_ON) ||
            (pParameter->BitRatePicc2Pcd.bSupplied == PH_ON) ||
            (pParameter->BitRateFramingOptionsPicc2Pcd.bSupplied == PH_ON)))
    {
        if (pParameter->BitRateFramingOptionsPicc2Pcd.bSupplied == PH_ON)
        {
            wSParamDataPos -= PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_FRAME_OPTION_PICC_TAG_LENGTH;
            pbSParamDataBuffer[wSParamDataPos] = (uint8_t)pParameter->BitRateFramingOptionsPicc2Pcd.wSuppliedValue;
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
                pbSParamDataBuffer,
                PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_FRAME_OPTION_PICC_TAG_LENGTH,
                wSParamDataPos,
                PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_FRAME_OPTION_PICC_TAG,
                &wSParamDataLen));
            wSParamDataPos -= wSParamDataLen;
        }
        if (pParameter->BitRatePicc2Pcd.bSupplied == PH_ON)
        {
            wSParamDataPos -= PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PICC_TAG_LENGTH;
            pbSParamDataBuffer[wSParamDataPos] = (uint8_t)(pParameter->BitRatePicc2Pcd.wSuppliedValue/256);
            pbSParamDataBuffer[wSParamDataPos + 1] = (uint8_t)(pParameter->BitRatePicc2Pcd.wSuppliedValue%256);
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
                pbSParamDataBuffer,
                PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PICC_TAG_LENGTH,
                wSParamDataPos,
                PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PICC_TAG,
                &wSParamDataLen));
            wSParamDataPos -= wSParamDataLen;
        }
        if (pParameter->BitRatePcd2Picc.bSupplied == PH_ON)
        {
            wSParamDataPos -= PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PCD_TAG_LENGTH;
            pbSParamDataBuffer[wSParamDataPos] = (uint8_t)(pParameter->BitRatePcd2Picc.wSuppliedValue/256);
            pbSParamDataBuffer[wSParamDataPos + 1] = (uint8_t)(pParameter->BitRatePcd2Picc.wSuppliedValue%256);
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
                pbSParamDataBuffer,
                PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PCD_TAG_LENGTH,
                wSParamDataPos,
                PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PCD_TAG,
                &wSParamDataLen));
            wSParamDataPos -= wSParamDataLen;
        }
        wSParamDataLen = wSParamDataSize - wSParamDataPos;
        PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
            pbSParamDataBuffer,
            wSParamDataLen,
            wSParamDataPos,
            PHPAL_I14443P4_SPARAM_BIT_RATE_ACTIVATION,
            &wSParamDataLen));
        wSParamDataPos -= wSParamDataLen;
    }

    if (bFrameFormatIndicationSupported &&
        ((pParameter->FramePcd2Picc.bSupplied == PH_ON) ||
            (pParameter->FramePcd2Picc.bSupplied == PH_ON) ||
            (pParameter->FramingOptionsPcd2Picc.bSupplied == PH_ON) ||
            (pParameter->FramingOptionsPicc2Pcd.bSupplied == PH_ON)))
    {
        if (pParameter->FramingOptionsPicc2Pcd.bSupplied == PH_ON)
        {
            wSParamDataPos -= PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PICC_TAG_LENGTH;
            pbSParamDataBuffer[wSParamDataPos] = (uint8_t)pParameter->FramingOptionsPicc2Pcd.wSuppliedValue;
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
                pbSParamDataBuffer,
                PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PICC_TAG_LENGTH,
                wSParamDataPos,
                PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PICC_TAG,
                &wSParamDataLen));
            wSParamDataPos -= wSParamDataLen;
        }
        if (pParameter->FramingOptionsPcd2Picc.bSupplied == PH_ON)
        {
            wSParamDataPos -= PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PCD_TAG_LENGTH;
            pbSParamDataBuffer[wSParamDataPos] = (uint8_t)pParameter->FramingOptionsPcd2Picc.wSuppliedValue;
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
                pbSParamDataBuffer,
                PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PCD_TAG_LENGTH,
                wSParamDataPos,
                PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PCD_TAG,
                &wSParamDataLen));
            wSParamDataPos -= wSParamDataLen;
        }
        if (pParameter->FramePicc2Pcd.bSupplied == PH_ON)
        {
            wSParamDataPos -= PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PICC_TAG_LENGTH;
            pbSParamDataBuffer[wSParamDataPos] = (uint8_t)pParameter->FramePicc2Pcd.wSuppliedValue;
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
                pbSParamDataBuffer,
                PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PICC_TAG_LENGTH,
                wSParamDataPos,
                PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PICC_TAG,
                &wSParamDataLen));
            wSParamDataPos -= wSParamDataLen;
        }
        if (pParameter->FramePcd2Picc.bSupplied == PH_ON)
        {
            wSParamDataPos -= PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PCD_TAG_LENGTH;
            pbSParamDataBuffer[wSParamDataPos] = (uint8_t)pParameter->FramePcd2Picc.wSuppliedValue;
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
                pbSParamDataBuffer,
                PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PCD_TAG_LENGTH,
                wSParamDataPos,
                PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PCD_TAG,
                &wSParamDataLen));
            wSParamDataPos -= wSParamDataLen;
        }
        wSParamDataLen = wSParamDataSize - wSParamDataPos;
        PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_Int_Attach_TLV_Header(
            pbSParamDataBuffer,
            wSParamDataLen,
            wSParamDataPos,
            PHPAL_I14443P4_SPARAM_FRAME_FORMAT_ACTIVATION,
            &wSParamDataLen));
        wSParamDataPos -= wSParamDataLen;
    }

    *pwSParamDataStartPos = wSParamDataPos;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_SParamApplyParameter(phpalI14443p4_Sw_DataParams_t * pDataParams,
                                           phpalI14443p4_Sw_SParam_Param * pParameter)
{
    phStatus_t status;

    /* Activate Datarate */
    if (pParameter->BitRateFramingOptionsPicc2Pcd.bSupplied == PH_ON)
    {
        /* Start/Stop Bit Suppression */
        if(((uint8_t)pParameter->BitRateFramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT)
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PICC_2_PCD, PH_ON));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PICC_2_PCD, PH_OFF));
        }
        /* SOF EOF Suppression */
        if(((uint8_t)pParameter->BitRateFramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF)
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PICC_2_PCD, PH_ON));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PICC_2_PCD, PH_OFF));
        }
    }

    if (pParameter->BitRatePcd2Picc.bSupplied == PH_ON)
    {
        switch(pParameter->BitRatePcd2Picc.wSuppliedValue)
        {
        case PHPAL_I14443P4_SPARAM_BIT_RATE_106:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXDATARATE, PHHAL_HW_RF_DATARATE_106));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_212:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXDATARATE, PHHAL_HW_RF_DATARATE_212));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_424:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXDATARATE, PHHAL_HW_RF_DATARATE_424));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_848:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXDATARATE, PHHAL_HW_RF_DATARATE_848));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_1695:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXDATARATE, PHHAL_HW_RF_DATARATE_1695));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_3390:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXDATARATE, PHHAL_HW_RF_DATARATE_3390));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_6780:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXDATARATE, PHHAL_HW_RF_DATARATE_6780));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_3FC4: /* 3fc/4 */
        case PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_FC: /* fc */
        case PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_3FC2: /* 3fc/2 */
        case PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_2FC: /* 2fc */
            /*FIXXME PSK */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }

    if (pParameter->BitRatePicc2Pcd.bSupplied == PH_ON)
    {
        switch(pParameter->BitRatePicc2Pcd.wSuppliedValue)
        {
        case PHPAL_I14443P4_SPARAM_BIT_RATE_106:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXDATARATE, PHHAL_HW_RF_DATARATE_106));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_212:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXDATARATE, PHHAL_HW_RF_DATARATE_212));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_424:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXDATARATE, PHHAL_HW_RF_DATARATE_424));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_848:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXDATARATE, PHHAL_HW_RF_DATARATE_848));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_1695:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXDATARATE, PHHAL_HW_RF_DATARATE_1695));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_3390:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXDATARATE, PHHAL_HW_RF_DATARATE_3390));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_6780:
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXDATARATE, PHHAL_HW_RF_DATARATE_6780));
            break;
        case PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_3FC4: /* 3fc/4 */
        case PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_FC: /* fc */
        case PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_3FC2: /* 3fc/2 */
        case PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_2FC: /* 2fc */
            /*FIXXME PSK */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }

    if (pParameter->FramePcd2Picc.bSupplied == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAME_FORMAT_PCD_2_PICC, pParameter->FramePcd2Picc.wSuppliedValue));
    }

    if (pParameter->FramePicc2Pcd.bSupplied == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAME_FORMAT_PICC_2_PCD, pParameter->FramePicc2Pcd.wSuppliedValue));
    }

    if (pParameter->FramingOptionsPcd2Picc.bSupplied == PH_ON)
    {
        /* Start/Stop Bit Suppression */
        if(((uint8_t)pParameter->FramingOptionsPcd2Picc.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT)
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PCD_2_PICC, PH_ON));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PCD_2_PICC, PH_OFF));
        }
        /* SOF EOF Suppression */
        if(((uint8_t)pParameter->FramingOptionsPcd2Picc.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF)
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PCD_2_PICC, PH_ON));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PCD_2_PICC, PH_OFF));
        }
        /* Sync Suppression */
        if(((uint8_t)pParameter->FramingOptionsPcd2Picc.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC)
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SYNC_SUPPRESSION_PCD_2_PICC, PH_ON));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SYNC_SUPPRESSION_PCD_2_PICC, PH_OFF));
        }
    }

    if (pParameter->FramingOptionsPicc2Pcd.bSupplied == PH_ON)
    {
        /* Start/Stop Bit Suppression */
        if(((uint8_t)pParameter->FramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT)
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PICC_2_PCD, PH_ON));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_START_STOP_BIT_SUPPRESSION_PICC_2_PCD, PH_OFF));
        }
        /* SOF EOF Suppression */
        if(((uint8_t)pParameter->FramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF)
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PICC_2_PCD, PH_ON));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SOF_EOF_SUPPRESSION_PICC_2_PCD, PH_OFF));
        }
        /* Sync Suppression */
        if(((uint8_t)pParameter->FramingOptionsPicc2Pcd.wSuppliedValue & PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC) == PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC)
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SYNC_SUPPRESSION_PICC_2_PCD, PH_ON));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Sw_SetConfig(pDataParams, PHPAL_I14443P4_SW_FRAMING_OPTIONS_SYNC_SUPPRESSION_PICC_2_PCD, PH_OFF));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_Parse_TLV_Len(uint8_t *pData, uint16_t wDataLen, uint16_t wLengthOffset, uint32_t *pdwLengthValue, uint16_t * pwLengthLength)
{
    *pdwLengthValue = 0;
    *pwLengthLength = 0;

    if (wLengthOffset >= wDataLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }
    if (pData[wLengthOffset] < 0x80)
    {
        *pdwLengthValue = pData[wLengthOffset];
        *pwLengthLength = 1;
    }
    else if (pData[wLengthOffset] == 0x81)
    {
        if (wLengthOffset + 1 >= wDataLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
        *pdwLengthValue = pData[wLengthOffset + 1];
        *pwLengthLength = 2;
    }
    else if (pData[wLengthOffset] == 0x82)
    {
        if (wLengthOffset + 2 >= wDataLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
        *pdwLengthValue = pData[wLengthOffset + 1] * 256 + pData[wLengthOffset + 2];
        *pwLengthLength = 3;
    }
    else if (pData[wLengthOffset] == 0x83)
    {
        if (wLengthOffset + 3 >= wDataLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
        *pdwLengthValue = pData[wLengthOffset + 1] * 256 * 256 + pData[wLengthOffset + 2] * 256 + pData[wLengthOffset + 3];
        *pwLengthLength = 4;
    }
    else if (pData[wLengthOffset] == 0x84)
    {
        if (wLengthOffset + 4 >= wDataLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
        *pdwLengthValue = pData[wLengthOffset + 1] * 256 * 256 * 256 + pData[wLengthOffset + 2] * 256 * 256 + pData[wLengthOffset + 3] * 256 + pData[wLengthOffset + 4];
        *pwLengthLength = 5;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_Attach_TLV_Header(
                                    uint8_t *pData,
                                    uint32_t wDataLen,
                                    uint32_t wDataOffset,
                                    uint8_t bTag,
                                    uint16_t *pwDataWritten
                                    )
{
    *pwDataWritten = 0;
    if (wDataLen < 128)
    {
        *pwDataWritten = 2;
    }
    else if (wDataLen < 256)
    {
        *pwDataWritten = 3;
    }
    else if (wDataLen < 65536)
    {
        *pwDataWritten = 4;
    }
    else if (wDataLen < 16777216)
    {
        *pwDataWritten = 5;
    }
    else if (wDataLen <= 4294967295L)
    {
        *pwDataWritten = 6;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    if (*pwDataWritten > wDataOffset)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
    }

    pData[wDataOffset - *pwDataWritten] = bTag;
    if (wDataLen < 128)
    {
        pData[wDataOffset - *pwDataWritten + 1] = (uint8_t)wDataLen;
    }
    else if (wDataLen < 256)
    {
        pData[wDataOffset - *pwDataWritten + 1] = 0x81;
        pData[wDataOffset - *pwDataWritten + 2] = (uint8_t)wDataLen;
    }
    else if (wDataLen < 65536)
    {
        pData[wDataOffset - *pwDataWritten + 1] = 0x82;
        pData[wDataOffset - *pwDataWritten + 2] = (uint8_t)(wDataLen/256);
        pData[wDataOffset - *pwDataWritten + 3] = (uint8_t)(wDataLen % 256);
    }
    else if (wDataLen < 16777216)
    {
        pData[wDataOffset - *pwDataWritten + 1] = 0x83;
        pData[wDataOffset - *pwDataWritten + 2] = (uint8_t)(wDataLen/(256*256));
        pData[wDataOffset - *pwDataWritten + 3] = (uint8_t)((wDataLen/256) % 256);
        pData[wDataOffset - *pwDataWritten + 4] = (uint8_t)(wDataLen % 256);
    }
    else if (wDataLen <= 4294967295L)
    {
        pData[wDataOffset - *pwDataWritten + 1] = 0x84;
        pData[wDataOffset - *pwDataWritten + 2] = (uint8_t)(wDataLen/(256*256*256));
        pData[wDataOffset - *pwDataWritten + 3] = (uint8_t)((wDataLen/(256*256)) % 256);
        pData[wDataOffset - *pwDataWritten + 4] = (uint8_t)((wDataLen/256) % 256);
        pData[wDataOffset - *pwDataWritten + 5] = (uint8_t)(wDataLen % 256);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_Exchange(
        phpalI14443p4_Sw_DataParams_t * pDataParams,
        uint16_t wOption,
        uint8_t * pTxBuffer,
        uint16_t wTxLength,
        uint8_t ** ppRxBuffer,
        uint16_t * pRxLength
        )
{
    phStatus_t statusTmp;
    uint16_t wRxOffset;
    uint8_t bSyncData[6] = {0x55, 0x55, 0x74, 0x74, 0x74, 0x74};

    /* If transmit is enhanced */
    if (pDataParams->bEnhancedBlockStateTx == PH_ON)
    {
        /* Check options */
        if (wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }

        /* clear internal buffer if requested */
        if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
        {
            pDataParams->wTxBufferLength = 0;
        }

        /* set the receive length */
        if (pRxLength != NULL)
        {
            *pRxLength = 0;
        }

        /* Fill the global TxBuffer */
        if (wTxLength != 0)
        {
            /* TxBuffer overflow check */
            if (wTxLength > (sizeof(pDataParams->pbEnhancedBlockBufferTx) - pDataParams->wTxBufferLength))
            {
                pDataParams->wTxBufferLength = 0;
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
            }

            /* copy data */
            memcpy(&pDataParams->pbEnhancedBlockBufferTx[pDataParams->wTxBufferLength], pTxBuffer, wTxLength);  /* PRQA S 3200 */
            pDataParams->wTxBufferLength = pDataParams->wTxBufferLength + wTxLength;
        }

        if (wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            /* Buffer operation -> finished */
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
        }

        /* Format data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4_Sw_Int_VHBR_FormatBlockFWEC(
            pDataParams->pbEnhancedBlockBufferTx, sizeof(pDataParams->pbEnhancedBlockBufferTx), pDataParams->wTxBufferLength,
            pDataParams->pbEnhancedBlockBufferTmp, sizeof(pDataParams->pbEnhancedBlockBufferTmp), &wTxLength));

        if (pDataParams->bFwecSyncEnabledTx == PH_ON)
        {
            statusTmp = phhalHw_Exchange(pDataParams->pHalDataParams, PH_EXCHANGE_BUFFER_FIRST, bSyncData, sizeof(bSyncData), NULL, NULL);
            wOption = PH_EXCHANGE_BUFFER_LAST;
        }
        else
        {
            wOption = PH_EXCHANGE_DEFAULT;
        }

        pTxBuffer = pDataParams->pbEnhancedBlockBufferTmp;
        /* Reset buffered bytes because exchange is performed */
        pDataParams->wTxBufferLength = 0;
    }

    /* Perform Exchange */
    statusTmp = phhalHw_Exchange(pDataParams->pHalDataParams, wOption, pTxBuffer, wTxLength, ppRxBuffer, pRxLength);

    /* If transmit is enhanced */
    if (pDataParams->bEnhancedBlockStateRx == PH_ON)
    {
        /* FIXXME dummy to test */
        *pRxLength = 0;
        *ppRxBuffer = pDataParams->pbEnhancedBlockBufferTx;
        statusTmp = 0;

        if (pDataParams->bFwecSyncEnabledRx == PH_ON)
        {
            memcpy(*ppRxBuffer, bSyncData, sizeof(bSyncData));
            *pRxLength = sizeof(bSyncData);
        }
        memcpy(*ppRxBuffer + *pRxLength, pTxBuffer, wTxLength);
        *pRxLength += wTxLength;
        /* End FIXXME */



        /* Extract result */
        if (ppRxBuffer != NULL && *ppRxBuffer != NULL && pRxLength != NULL)
        {
            /* Check sync pulse */
            if (pDataParams->bFwecSyncEnabledRx == PH_ON)
            {
                if (*pRxLength < sizeof(bSyncData) || memcmp(bSyncData, *ppRxBuffer, sizeof(bSyncData)) != 0)
                {
                    return PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_PAL_ISO14443P4);
                }
            }

            /* Adjust rx buffer */
            *ppRxBuffer += sizeof(bSyncData);
            *pRxLength -= sizeof(bSyncData);

            /* Decode Data */
            PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4_Sw_Int_VHBR_FormatRecvBlockFWEC(
                *ppRxBuffer, *pRxLength,
                pDataParams->pbEnhancedBlockBufferTmp, sizeof(pDataParams->pbEnhancedBlockBufferTmp), pRxLength, &wRxOffset));
            *pRxLength -= wRxOffset;

            /* Start position is always zero in enhanced mode */
            memcpy(&pDataParams->pbEnhancedBlockBufferRx[pDataParams->wRxBufferStartPos], pDataParams->pbEnhancedBlockBufferTmp + wRxOffset, *pRxLength);
            /* Map Data */
            *ppRxBuffer = pDataParams->pbEnhancedBlockBufferRx;
            *pRxLength = pDataParams->wRxBufferStartPos + *pRxLength;
        }
    }

    return statusTmp;
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_GetRxTxBufferSize(
    phpalI14443p4_Sw_DataParams_t * pDataParams,
    uint16_t *pwRxBufferSize,
    uint16_t *pwTxBufferSize
    )
{
    phStatus_t statusTmp;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE, pwRxBufferSize));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE, pwTxBufferSize));

    /* Adjust the buffer if required */
    if (pDataParams->bEnhancedBlockStateRx == PH_ON)
    {
        if (pDataParams->bFwecSyncEnabledRx == PH_ON)
        {
            *pwRxBufferSize -= 6;
        }
        *pwRxBufferSize = (*pwRxBufferSize / 8) * 7;
        *pwRxBufferSize -= 2; /* Length at begin */
        if (*pwRxBufferSize > sizeof(pDataParams->pbEnhancedBlockBufferRx) - 4) /* CRC + Len */
            *pwRxBufferSize = sizeof(pDataParams->pbEnhancedBlockBufferRx) - 4;
    }
    if (pDataParams->bEnhancedBlockStateTx == PH_ON)
    {
        if (pDataParams->bFwecSyncEnabledTx == PH_ON)
        {
            *pwTxBufferSize -= 6;
        }
        *pwTxBufferSize = (*pwTxBufferSize / 8) * 7;
        *pwTxBufferSize -= 2; /* Length at begin */
        if (*pwTxBufferSize > sizeof(pDataParams->pbEnhancedBlockBufferTx) - 4) /* CRC + Len */
            *pwTxBufferSize = sizeof(pDataParams->pbEnhancedBlockBufferTx) - 4;
    }
    return statusTmp;
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_GetNumberOfPreloadedBytes(
    phpalI14443p4_Sw_DataParams_t * pDataParams,
    uint16_t *pwNumberPreloadedBytes
    )
{
    if (pDataParams->bEnhancedBlockStateTx == PH_OFF)
    {
        return phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXBUFFER_LENGTH, pwNumberPreloadedBytes);
    }

    *pwNumberPreloadedBytes = pDataParams->wTxBufferLength;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_GetTxPCB(
    phpalI14443p4_Sw_DataParams_t * pDataParams,
    uint16_t *pwPCB
    )
{
    phStatus_t statusTmp;

    if (pDataParams->bEnhancedBlockStateTx == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_ADDITIONAL_INFO, 0x00));
        return phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXBUFFER, pwPCB);
    }

    *pwPCB = pDataParams->pbEnhancedBlockBufferTx[0];
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_SetTxPCB(
    phpalI14443p4_Sw_DataParams_t * pDataParams,
    uint8_t bPCB
    )
{
    phStatus_t statusTmp;

    if (pDataParams->bEnhancedBlockStateTx == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_ADDITIONAL_INFO, 0x00));
        return phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXBUFFER, bPCB);
    }

    pDataParams->pbEnhancedBlockBufferTx[0] = bPCB;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_SetTxBufferLen(
    phpalI14443p4_Sw_DataParams_t * pDataParams,
    uint16_t wTxBufferLen
    )
{
    if (pDataParams->bEnhancedBlockStateTx == PH_OFF)
    {
        return phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXBUFFER_LENGTH, wTxBufferLen);
    }

    pDataParams->wTxBufferLength = wTxBufferLen;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_SetRxBufferStartPos(
    phpalI14443p4_Sw_DataParams_t * pDataParams,
    uint16_t wRxBufferStartPos
    )
{
    if (pDataParams->bEnhancedBlockStateRx == PH_OFF)
    {
        return phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, wRxBufferStartPos);
    }

    pDataParams->wRxBufferStartPos = wRxBufferStartPos;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_GetRxBufferStartPos(
    phpalI14443p4_Sw_DataParams_t * pDataParams,
    uint16_t * pwRxBufferStartPos
    )
{
    if (pDataParams->bEnhancedBlockStateRx == PH_OFF)
    {
        return phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, pwRxBufferStartPos);
    }

    *pwRxBufferStartPos = pDataParams->wRxBufferStartPos;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sw_Int_VHBR_FormatBlockFWEC(
                                   uint8_t *pTxBuffer,
                                   uint16_t wTxSize,
                                   uint16_t wTxLen,
                                   uint8_t *pEnhancedBlock,
                                   uint16_t wEnhancedBlockSize,
                                   uint16_t *pwEnhancedBlockLen
                                   )
{
    int16_t status = PH_ERR_SUCCESS;
    uint32_t dwCrc;
    uint16_t wLenField;
    uint16_t wPaddingSize = 0;
    uint8_t pFirstBlock[7];



    /*
    Enhanced Block Format

    | Length Field |   Prologue Field    | Information Field | Epilogue Field |
    |     LEN      | PCB | [CID] | [NAD] |       [INF]       |      EDC       |
    |     2 B      | 1B  |  1B   |  1B   |                   |      4B        |

    */

    /* At least a PCB musst supplied */
    if (wTxLen < 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    /* minimum size of dwTxLen + Length Bytes + CRC32 */
    /* divide by 7 to get number of blocks (round up) and multiply by 8 */
    if (wEnhancedBlockSize < (((wTxLen + 2 + 4) + 6)/7)*8)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
    }

    wPaddingSize = 7 - ((wTxLen + 2 /*Len */ + 4 /*CRC*/) % 7);
    if (wPaddingSize == 7)
        wPaddingSize = 0;

    /* Check if CRC fit and also the 0xFF padding */
    if ((uint32_t)wTxLen + 4 + wPaddingSize > (uint32_t)wTxSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
    }

    /* Create first block */
    wLenField = 2 + wTxLen;
    /* attach length bytes */
    pFirstBlock[0] = ( wLenField & 0xFF);
    pFirstBlock[1] = ((wLenField >> 8) & 0xFF);

    /* Calc CRC */
    /* First 2 length bytes */
    PH_CHECK_SUCCESS_FCT(status, phTools_CalculateCrc32(
                PH_TOOLS_CRC_OPTION_MSB_FIRST | PH_TOOLS_CRC32_OPTION_REVERSE_DATA_BYTE,
                PH_TOOLS_CRC32_PRESET_ISO14443,
                PH_TOOLS_CRC32_POLY_ISO14443,
                pFirstBlock,
                2,
                &dwCrc
                ));
    /* Add data */
    PH_CHECK_SUCCESS_FCT(status, phTools_CalculateCrc32(
                PH_TOOLS_CRC_OPTION_MSB_FIRST | PH_TOOLS_CRC32_OPTION_REVERSE_DATA_BYTE | PH_TOOLS_CRC_OPTION_OUPUT_INVERTED | PH_TOOLS_CRC32_OPTION_REVERSE_CRC,
                dwCrc,
                PH_TOOLS_CRC32_POLY_ISO14443,
                pTxBuffer,
                wTxLen,
                &dwCrc
                ));
    /* Add CRC, Padding to pTxBuffer */
    memcpy(pTxBuffer + wTxLen, &dwCrc, 4);
    wTxLen += 4;
    /* Add Padding */
    memset(pTxBuffer + wTxLen, 0xFF, wPaddingSize);
    wTxLen += wPaddingSize;

    /* Add data to first block */
    memcpy(pFirstBlock + 2, pTxBuffer, 5);

    /* Adjust data (size + pointer) */
    wTxLen -= 5;
    pTxBuffer += 5;

    /* Encode first block */
    PH_CHECK_SUCCESS_FCT(status, phTools_HammingEncode(pFirstBlock, pEnhancedBlock, wEnhancedBlockSize, 0));

    /* Adjust offset of output */
    pEnhancedBlock += 8;
    wEnhancedBlockSize -= 8;

    /* Encode remaining data */
    PH_CHECK_SUCCESS_FCT(status, phTools_HammingEncodeMoreBlocks(pTxBuffer, wTxLen, pEnhancedBlock, wEnhancedBlockSize, pwEnhancedBlockLen));

    /* Add again first block */
    *pwEnhancedBlockLen += 8;

    return status;
}


phStatus_t phpalI14443p4_Sw_Int_VHBR_FormatRecvBlockFWEC(
                                   uint8_t *pRxEnhancedBlock,
                                   uint16_t wRxEnhancedBlockLen,
                                   uint8_t *pRxBuffer,
                                   uint16_t wRxBufferSize,
                                   uint16_t *pwRxLen,
                                   uint16_t *pwRxStartPos
                                   )
{
    int16_t status = PH_ERR_SUCCESS;
    uint16_t wLenField;
    uint16_t wPaddingSize;
    uint32_t dwCrc;
    uint32_t *pdwReceivedCrc;

    /*
    Enhanced Block Format

    | Length Field | Prologue Field  | Information Field | Epilogue Field |
    |     LEN      | PCB | CID | NAD |        INF        |      EDC       |
    |     2 B      | 1B  | 1B  | 1B  |                   |      4B        |
    |  HammingSubBlock  | HammingByte  |  HammingSubBlock  | HammmingByte |
    */

    if (wRxEnhancedBlockLen % 8 != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    PH_CHECK_SUCCESS_FCT(status,
        phTools_HammingDecodeMoreBlocks(pRxEnhancedBlock, wRxEnhancedBlockLen, pRxBuffer, wRxBufferSize, pwRxLen));

    /* Check if length fiels match */
    wLenField = pRxBuffer[0] + 256*pRxBuffer[1];
    wPaddingSize = 7 - ((wLenField + 4 /*CRC*/) % 7);
    if (wPaddingSize == 7)
        wPaddingSize = 0;

    if (wLenField + wPaddingSize != *pwRxLen - 4 /*CRC*/)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* Remove Padding */
    *pwRxLen -= wPaddingSize;

    if (*pwRxLen < 7) /* Len + PCB + CRC */
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* Check CRC */
    PH_CHECK_SUCCESS_FCT(status, phTools_CalculateCrc32(
            PH_TOOLS_CRC_OPTION_MSB_FIRST | PH_TOOLS_CRC32_OPTION_REVERSE_DATA_BYTE | PH_TOOLS_CRC_OPTION_OUPUT_INVERTED | PH_TOOLS_CRC32_OPTION_REVERSE_CRC,
            PH_TOOLS_CRC32_PRESET_ISO14443,
            PH_TOOLS_CRC32_POLY_ISO14443,
            pRxBuffer,
            *pwRxLen - 4,
            &dwCrc
            ));

    pdwReceivedCrc = (uint32_t *)&pRxBuffer[*pwRxLen - 4];
    if (dwCrc != *pdwReceivedCrc)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* Remove CRC */
    *pwRxLen -= 4;
    /* Len Offset */
    *pwRxStartPos = 2;

    return status;
}


#endif /* NXPBUILD__PHPAL_I14443P4_SW */
