/*
 * Copyright 2017 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * PCSC ISO14443-4A Component of Reader Library.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phpalI14443p4a.h>
#include <ph_RefDefs.h>
#include <phhalHw.h>

#ifdef NXPBUILD__PHPAL_I14443P4A_PCSC

#include <phhalHw_Pcsc_Cmd.h>
#include "phpalI14443p4a_Pcsc.h"

phStatus_t phpalI14443p4a_Pcsc_Init(
                                    phpalI14443p4a_Pcsc_DataParams_t * pDataParams,
                                    uint16_t wSizeOfDataParams,
                                    void * pHalDataParams
                                    )
{
    if (sizeof(phpalI14443p4a_Pcsc_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_ISO14443P4A);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalDataParams);

    /* Check data parameters */
    if (PH_GET_COMPCODE(pHalDataParams) != PH_COMP_HAL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    /* Extract BAL */
    switch (PH_GET_COMPID(pHalDataParams))
    {
#ifdef NXPBUILD__PHHAL_HW_CALLBACK
        case PHHAL_HW_CALLBACK_ID:
            /* Check if required PCSC HAL is privided with pCbContext and BAL is provided with PCSC HAL */
            if (((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            if (PH_GET_COMPCODE(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext) != PH_COMP_HAL)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            if(PH_GET_COMPID(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext) != PHHAL_HW_PCSC_ID)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            /* Set internal pcsc hal dataparams */
            pDataParams->pHalPcscDataParams = (phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext);

            /* Check if required BAL is provided with HAL */
            if (((phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext))->pBalDataParams == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                break;
            }
            pDataParams->pBalDataParams = (phbalReg_PcscWin_DataParams_t *)
                ((phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext))->pBalDataParams;
            break;
#endif /* NXPBUILD__PHHAL_HW_CALLBACK */

#ifdef NXPBUILD__PHHAL_HW_PCSC
        case PHHAL_HW_PCSC_ID:
            /* Set internal pcsc hal dataparams */
            pDataParams->pHalPcscDataParams = (phhalHw_Pcsc_DataParams_t *)pHalDataParams;

            /* Check if required BAL is provided with HAL */
            if (((phhalHw_Pcsc_DataParams_t *)pHalDataParams)->pBalDataParams == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                break;
            }
            pDataParams->pBalDataParams = (phbalReg_PcscWin_DataParams_t *)
                ((phhalHw_Pcsc_DataParams_t *)pHalDataParams)->pBalDataParams;
            break;
#endif /* NXPBUILD__PHHAL_HW_PCSC */

        default:
            pDataParams->pBalDataParams = NULL;
            pDataParams->pHalPcscDataParams = NULL;
            break;
    }

    /* Initialize data parameters */
    pDataParams->wId            = PH_COMP_PAL_ISO14443P4A | PHPAL_I14443P4A_PCSC_ID;
    pDataParams->pHalDataParams = pHalDataParams;
    pDataParams->bCidSupported  = PH_OFF;
    pDataParams->bNadSupported  = PH_OFF;
    pDataParams->bCid           = 0x00;
    pDataParams->bBitRateCaps   = 0x00;
    pDataParams->bFwi           = PHPAL_I14443P4A_PCSC_FWI_DEFAULT;
    pDataParams->bFsci          = PHPAL_I14443P4A_PCSC_FSCI_DEFAULT;
    pDataParams->bFsdi          = 0x00;
    pDataParams->bDri           = 0x00;
    pDataParams->bDsi           = 0x00;
    pDataParams->bActStatus     = PH_OFF;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI14443p4a_Pcsc_Rats(
                                    phpalI14443p4a_Pcsc_DataParams_t * pDataParams,
                                    uint8_t bFsdi,
                                    uint8_t bCid,
                                    uint8_t * pAts
                                    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM aCmd[2];
    uint8_t     PH_MEMLOC_REM bAtsIndex;
    uint8_t     PH_MEMLOC_REM bSfgi;
    uint32_t    PH_MEMLOC_REM dwSfgt;
    uint32_t    PH_MEMLOC_REM dwFwt;
    uint8_t *   PH_MEMLOC_REM pResp;
    uint16_t    PH_MEMLOC_REM wRespLength;
    uint16_t    PH_MEMLOC_REM wSession = 0U;

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Parameter check */
        if ((bFsdi > 0x0F) || (bCid > 14))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
        }

        /* Set Activation timeout */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
            PHPAL_I14443P4A_PCSC_FWT_ACTIVATION_US + PHPAL_I14443P4A_PCSC_EXT_TIME_US));

        /* Enable CRC */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_TXCRC,
            PH_ON));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_RXCRC,
            PH_ON));

        /* PICC receiving value of FSDI = 9F should interpret it as FSDI = 8 (FSD = 256 bytes)*/
        if (bFsdi > 0x08)
        {
            bFsdi = 0x08;
        }

        /* Rats command */
        aCmd[0] = PHPAL_I14443P4A_PCSC_RATS;
        aCmd[1] = (uint8_t)(((bFsdi << 4) & 0xF0) | (bCid & 0x0F));

        /* Send command using transparent exchange */
        status = phhalHw_Pcsc_Cmd_Transceive(
            pDataParams->pHalPcscDataParams,
            PH_EXCHANGE_DEFAULT,
            aCmd,
            2U,
            &pResp,
            &wRespLength);

        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Check for protocol error */
            if (((uint8_t)wRespLength != pResp[0]) || (wRespLength < 1))
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
            }

            /* Copy ATS buffer */
            memcpy(pAts, pResp, wRespLength);  /* PRQA S 3200 */

            /* Set default values */
            pDataParams->bCidSupported  = PH_ON;
            pDataParams->bCid           = bCid;
            pDataParams->bNadSupported  = PH_OFF;
            pDataParams->bBitRateCaps   = 0x00;
            pDataParams->bFwi           = PHPAL_I14443P4A_PCSC_FWI_DEFAULT;
            pDataParams->bFsdi          = bFsdi;
            pDataParams->bFsci          = PHPAL_I14443P4A_PCSC_FSCI_DEFAULT;
            pDataParams->bDri           = 0x00;
            pDataParams->bDsi           = 0x00;
            bSfgi                       = PHPAL_I14443P4A_PCSC_SFGI_DEFAULT;

            /* Set FSDI in reader */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_PCSC_CONFIG_FSDI,
                bFsdi));

            /* Start parsing with T0 byte */
            bAtsIndex = PHPAL_I14443P4A_PCSC_ATS_T0;

            /* Parse T0/TA/TB/TC */
            if (wRespLength > 1)
            {
                /* Parse T0 */
                pDataParams->bFsci = pAts[bAtsIndex] & 0x0F;
                if (pDataParams->bFsci > 0x08)
                {
                    pDataParams->bFsci = 0x08;
                }

                /* Set FSCI in reader */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_PCSC_CONFIG_FSCI,
                    pDataParams->bFsci));

                bAtsIndex++;

                /* Parse TA(1) */
                if (pAts[PHPAL_I14443P4A_PCSC_ATS_T0] & PHPAL_I14443P4A_PCSC_ATS_TA1_PRESENT)
                {
                    /* Check for protocol error */
                    if (wRespLength <= bAtsIndex)
                    {
                        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
                    }
                    else
                    {
                        /* Store Bitrate capabilities */
                        pDataParams->bBitRateCaps = pAts[bAtsIndex];

                        bAtsIndex++;
                    }
                }

                /* Parse TB(1) */
                if (pAts[PHPAL_I14443P4A_PCSC_ATS_T0] & PHPAL_I14443P4A_PCSC_ATS_TB1_PRESENT)
                {
                    /* Check for protocol error */
                    if (wRespLength <= bAtsIndex)
                    {
                        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
                    }
                    else
                    {
                        /* Store Sfgi */
                        bSfgi = pAts[bAtsIndex] & 0x0F;
                        if (bSfgi == 0x0F)
                        {
                            bSfgi = PHPAL_I14443P4A_PCSC_SFGI_DEFAULT;
                        }

                        /* Store Fwi */
                        pDataParams->bFwi = (pAts[bAtsIndex] >> 4) & 0x0F;

                        if (pDataParams->bFwi == 0x0F)
                        {
                            pDataParams->bFwi = PHPAL_I14443P4A_PCSC_FWI_DEFAULT;
                        }

                        bAtsIndex++;
                    }
                }

                /* Parse TC(1) */
                if (pAts[PHPAL_I14443P4A_PCSC_ATS_T0] & PHPAL_I14443P4A_PCSC_ATS_TC1_PRESENT)
                {
                    /* Check for protocol error */
                    if (wRespLength <= bAtsIndex)
                    {
                        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
                    }
                    else
                    {
                        /* Check NAD Support */
                        if (pAts[bAtsIndex] & PHPAL_I14443P4A_PCSC_ATS_TC1_NAD_SUPPORT)
                        {
                            pDataParams->bNadSupported = 1;
                        }

                        /* Check CID Support */
                        if (!(pAts[bAtsIndex] & PHPAL_I14443P4A_PCSC_ATS_TC1_CID_SUPPORT))
                        {
                            pDataParams->bCidSupported = 0;
                            pDataParams->bCid = 0;
                        }
                    }
                }
            }

            if (pDataParams->bCidSupported)
            {
                /* Set CID in reader */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_PCSC_CONFIG_CID,
                    bCid));
            }

            /* Calculate SFGT in Microseconds */
            dwSfgt = (uint32_t)(PHPAL_I14443P4A_PCSC_FWT_MIN_US * (1 << bSfgi));

            /* Perform SFGT Wait */
            if (dwSfgt > 0xFFFF)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_TIME_MILLISECONDS,
                    (uint16_t)(dwSfgt / 1000)));
            }
            else
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_TIME_MICROSECONDS,
                    (uint16_t)dwSfgt));
            }

            /* Calculate FWT timeout */
            dwFwt = (uint32_t)(PHPAL_I14443P4A_PCSC_FWT_MIN_US * (1 << pDataParams->bFwi));

            /* Add extension time */
            dwFwt = dwFwt + PHPAL_I14443P4A_PCSC_EXT_TIME_US;

            /* Set FWT timeout */
            if (dwFwt > 0xFFFF)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,
                    (uint16_t)(dwFwt / 1000)));
            }
            else
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
                    (uint16_t)dwFwt));
            }

            /* Set FWTI in reader */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_PCSC_CONFIG_FWTI,
                pDataParams->bFwi));

            pDataParams->bActStatus = PH_ON;
        }
        /* Perform protocol error handling */
        else
        {
            if (!I14443P4A_PCSC_IS_INVALID_BLOCK_STATUS(status))
            {
                return status;
            }

            /* Wait FWTmin after RATS */
            if ((status & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_TIME_MICROSECONDS,
                    PHPAL_I14443P4A_PCSC_FWT_MIN_US + PHPAL_I14443P4A_PCSC_EXT_TIME_US));
            }

            /* Send S(DESELECT) command */
            aCmd[0] = PHPAL_I14443P4A_PCSC_S_BLOCK | PHPAL_I14443P4A_PCSC_S_BLOCK_RFU_BITS | PHPAL_I14443P4A_PCSC_PCB_CID_FOLLOWING | PHPAL_I14443P4A_PCSC_PCB_DESELECT;
            aCmd[1] = bCid;

            /* Send command using transparent exchange */
            statusTmp = phhalHw_Pcsc_Cmd_Transceive(
                pDataParams->pHalPcscDataParams,
                PH_EXCHANGE_DEFAULT,
                aCmd,
                2U,
                &pResp,
                &wRespLength);

            /* S(DESELECT) failure */
            if (((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS) || (wRespLength != 2) || (pResp[0] != aCmd[0]) || (pResp[1] != aCmd[1]))
            {
                /* Set HLTA timeout */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
                    PHPAL_I14443P4A_PCSC_HALT_TIME_US));

                /* Send HLTA */
                aCmd[0] = PHPAL_I14443P4A_PCSC_HALT_CMD;
                aCmd[1] = 0x00;

                /* Send command using transparent exchange */
                statusTmp = phhalHw_Pcsc_Cmd_Transceive(
                    pDataParams->pHalPcscDataParams,
                    PH_EXCHANGE_DEFAULT,
                    aCmd,
                    2U,
                    &pResp,
                    &wRespLength);
            }
        }
    }
    else
    {
        /* If bal is available use alternative methode */
        if (pDataParams->pBalDataParams != NULL)
        {
            /* If no ATR is available it seems to be a timeout before so wrong use contition */
            if (pDataParams->pBalDataParams->wAtrBufSize == 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_ISO14443P4A);
            }

            memcpy(pAts, pDataParams->pBalDataParams->pAtrBuffer,
                pDataParams->pBalDataParams->wAtrBufSize);
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
        }
        else
        {
            /* Card already activated to highest supported layer. No control over activation. */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4A);
        }
    }

    return PH_ADD_COMPCODE(status, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI14443p4a_Pcsc_Pps(
                                   phpalI14443p4a_Pcsc_DataParams_t * pDataParams,
                                   uint8_t bDri,
                                   uint8_t bDsi
                                   )
{
    phStatus_t PH_MEMLOC_REM status;
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t    PH_MEMLOC_REM cmd[3] = {0U};
    uint8_t *  PH_MEMLOC_REM pResp = NULL;
    uint16_t   PH_MEMLOC_REM wRespLength = 0U;
    uint16_t   PH_MEMLOC_REM wDataRate = 0U;
    uint16_t   PH_MEMLOC_REM wSession = 0U;

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Check DSI and DRI */
        if ((bDsi > 3U) || (bDri > 3U))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
        }

        /* PPS command */
        cmd[0] = PHPAL_I14443P4A_PCSC_PPSS | pDataParams->bCid;
        cmd[1] = PHPAL_I14443P4A_PCSC_PPS0;
        cmd[2] = ((bDsi << 2U) & 0x0CU) | (bDri & 0x03U);

        /* Send command using transparent exchange */
       PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_Transceive(
            pDataParams->pHalPcscDataParams,
            PH_EXCHANGE_DEFAULT,
            cmd,
            3,
            &pResp,
            &wRespLength));

        /* Check length and start byte of PPS */
        if ((wRespLength != 1) || (pResp[0] != (PHPAL_I14443P4A_PCSC_PPSS | pDataParams->bCid)))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
        }

        /* Store data rates */
        pDataParams->bDri = bDri;
        pDataParams->bDsi = bDsi;

        /* Calculate data rate parameters */
        wDataRate = (bDri == bDsi)? 0x80U : 0x00U;
        wDataRate |= ((bDsi << 3U) & 0x18U) | (bDri & 0x03U);

        /* Apply new data rates */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_CURRENT_SPEED,
            wDataRate));
    }
    else
    {
        /* If bal is available use alternative methode */
        if (pDataParams->pBalDataParams != NULL)
        {
            /* No PPS available for PCSC Reader */
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
        }
        else
        {
            /* Card activated to highest supported layer by default. No control over activation. */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4A);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI14443p4a_Pcsc_ActivateCard(
                                            phpalI14443p4a_Pcsc_DataParams_t * pDataParams,
                                            uint8_t bFsdi,
                                            uint8_t bCid,
                                            uint8_t bDri,
                                            uint8_t bDsi,
                                            uint8_t * pAts
                                            )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint16_t   PH_MEMLOC_REM wSession = 0U;

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Check DSI and DRI */
        if ((bDsi > 3U) || (bDri > 3U))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
        }

        /* Perform RATS */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4a_Pcsc_Rats(
            pDataParams,
            bFsdi,
            bCid,
            pAts));

        /* Perform PPS */
        if ((bDri != PHPAL_I14443P4A_DATARATE_106) || (bDsi != PHPAL_I14443P4A_DATARATE_106))
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4a_Pcsc_Pps(
                pDataParams,
                bDri,
                bDsi));
        }
    }
    else
    {
        /* If bal is available use alternative methode */
        if (pDataParams->pBalDataParams != NULL)
        {
            /* If no ATR is available it seems to be a timeout before so wrong use contition */
            if (pDataParams->pBalDataParams->wAtrBufSize == 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_ISO14443P4A);
            }

            memcpy(pAts, pDataParams->pBalDataParams->pAtrBuffer,
                pDataParams->pBalDataParams->wAtrBufSize);
        }
        else
        {
            /* Card activated to highest supported layer by default. No control over activation. */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4A);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI14443p4a_Pcsc_GetProtocolParams(
                                                 phpalI14443p4a_Pcsc_DataParams_t * pDataParams,
                                                 uint8_t * pCidEnabled,
                                                 uint8_t * pCid,
                                                 uint8_t * pNadSupported,
                                                 uint8_t * pFwi,
                                                 uint8_t * pFsdi,
                                                 uint8_t * pFsci
                                                 )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint16_t    PH_MEMLOC_REM wSession = 0U;
    uint16_t    PH_MEMLOC_REM wFSCI;
    uint16_t    PH_MEMLOC_REM wFSDI;
    uint16_t    PH_MEMLOC_REM wFWI;
    uint16_t    PH_MEMLOC_REM wCid;

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Activated status if not enabled from Rats then necessary Get config of parameters is done
        and then the activated status is enabled*/
        if ( pDataParams->bActStatus == PH_OFF)
        {
            /* Get FSCI  */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                 PHHAL_HW_PCSC_CONFIG_FSCI,
                 &wFSCI));
            pDataParams->bFsci = wFSCI & 0x00FF;

            /* Get FSDI  */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                 PHHAL_HW_PCSC_CONFIG_FSDI,
                 &wFSDI));
            pDataParams->bFsdi = wFSDI & 0x00FF;

            /* Get FWI  */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                 PHHAL_HW_PCSC_CONFIG_FWTI,
                 &wFWI));
            pDataParams->bFwi = wFWI & 0x00FF;

            /* Get Cid  */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                 PHHAL_HW_PCSC_CONFIG_CID,
                 &wCid));
            pDataParams->bCid = wCid & 0x00FF;

            /* Activated Status is Enabled*/
            pDataParams->bActStatus = PH_ON;
        }
        *pCidEnabled    = pDataParams->bCidSupported;
        *pCid           = pDataParams->bCid;
        *pNadSupported  = pDataParams->bNadSupported;
        *pFwi           = pDataParams->bFwi;
        *pFsdi          = pDataParams->bFsdi;
        *pFsci          = pDataParams->bFsci;
    }
    else
    {
        /* If bal is available use alternative methode */
        if (pDataParams->pBalDataParams != NULL)
        {
            /* No GetProtocolParams Function available for PCSC Reader */
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
        }
        else
        {
            /* Card activated to highest supported layer by default. No control over activation. */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4A);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

#endif /* NXPBUILD__PHPAL_I14443P4A_RD710 */

