/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software ISO7816 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHwContact.h>
#include <phpalI7816p4.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_I7816P4_SW

#include "phpalI7816p4_Sw_Int.h"
#include "phpalI7816p4_Sw.h"

static const uint16_t PH_MEMLOC_CONST_ROM bI7816p4_FsTable[9] = {16, 24, 32, 40, 48, 64, 96, 128, 256};

phStatus_t phpalI7816p4_Sw_Init(
                                phpalI7816p4_Sw_DataParams_t * pDataParams,
                                uint16_t wSizeOfDataParams,
                                void * pHalDataParams
                                )
{
    if (sizeof(phpalI7816p4_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_I7816P4);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalDataParams);

    /* Init private data */
    pDataParams->wId                = PH_COMP_PAL_I7816P4 | PHPAL_I7816P4_SW_ID;
    pDataParams->pHalDataParams     = pHalDataParams;

    /* Set default state */
    pDataParams->bT1MainState       = PHDLPROTOCOL_T1_STATE_TX;
    pDataParams->bT1TxState         = PHDLPROTOCOL_T1_TXSUB_READY;

    pDataParams->bT1SeqNumReceive   = 0x00;
    pDataParams->bT1SeqNumSend      = 0x00;

    pDataParams->wT0LengthExpected  = 0x00;
    pDataParams->bT0CurrentCase     = 0x00;

    /* Apply default parameters */
    pDataParams->bNad               = 0x00;
    pDataParams->bIfsc              = PHPAL_I7816P4_SW_FSCI_DEFAULT;
    pDataParams->bIfsd              = PHPAL_I7816P4_SW_FSDI_DEFAULT;

    pDataParams->bAutoGetResponse   = 0x01;
    pDataParams->bProtocol          = 0xFF;
    pDataParams->bAuto6cHandling    = 0x00;

    pDataParams->pIntPayload        = 0x00;
    pDataParams->wIntPayloadLen     = 0x00;
    pDataParams->pIntBuffer[0]      = 0x00;
    pDataParams->wIntBufferLen      = 0x00;
    pDataParams->pRxBuffer[0]       = 0x00;
    memset(pDataParams->pTxBuffer, 0, PHPAL_I7816P4_SW_INT_BUFFER_SIZE);
    pDataParams->wTxBufLen          = 0;
    pDataParams->wRxBufferLen       = 0x00;

    pDataParams->bT1TxRetransmitCount = 0;
    pDataParams->bT1RxRetransmitCount = 0;

    pDataParams->wI2CProtocol = PHPAL_I7816P4_I2C_PROTOCOL_T1OVERI2C_LEN_ONE_CRC;

    pDataParams->bBwtMultiplier = 1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Exchange(
                                    phpalI7816p4_Sw_DataParams_t * pDataParams,
                                    uint16_t wOption,
                                    uint8_t * pTxBuffer,
                                    uint16_t wTxLength,
                                    uint8_t ** ppRxBuffer,
                                    uint16_t * pRxLength
                                    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bBufferOverflow;
    uint32_t    PH_MEMLOC_REM dwProtocol;

    /* Used for Transmission */
    uint8_t *   PH_MEMLOC_REM pUsedTxBuffer;
    uint8_t     PH_MEMLOC_REM bRetryCountRetransmit;

    /* Used for Reception */
    uint16_t    PH_MEMLOC_REM RxLength;
    uint8_t *   PH_MEMLOC_REM pRxBuffer;

    /* Option parameter check */
    if (wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT | PH_EXCHANGE_TXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
    }

    /* clear internal buffer if requested */
    if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        pDataParams->wTxBufLen = 0;
    }

    /* Fill the global TxBuffer */
    /* Either if just buffer or if already data in buffer to appent new data */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT || pDataParams->wTxBufLen != 0)
    {
        if (wTxLength != 0)
        {
            /* TxBuffer overflow check */
            if (wTxLength > (PHPAL_I7816P4_SW_INT_BUFFER_SIZE - pDataParams->wTxBufLen))
            {
                pDataParams->wTxBufLen = 0;
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_I7816P4);
            }

            /* copy data */
            memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pTxBuffer, wTxLength);  /* PRQA S 3200 */
            pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;
        }

        /* Buffer operation -> finished */
        if (wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }
        pUsedTxBuffer = pDataParams->pTxBuffer;
        /* Adjust length and clear buffered size */
        wTxLength = pDataParams->wTxBufLen;
        pDataParams->wTxBufLen = 0;
    }
    else
    {
        pUsedTxBuffer = pTxBuffer;
    }

    /* Check if caller has provided valid RxBuffer */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &RxLength;
    }

    /* use own static buffer*/
    *ppRxBuffer = pDataParams->pIntBuffer;

    /* Reset receive length */
    *pRxLength = 0;

    /* Reset RetryCount */
    bRetryCountRetransmit = 0;

    /* Reset BufferOverflow flag */
    bBufferOverflow = 0;

    /* Set correct Protocol */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, &dwProtocol));

    switch(dwProtocol)
    {
    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T0:
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_Exchange(pDataParams, wOption, pUsedTxBuffer, wTxLength, ppRxBuffer, pRxLength));
        break;

    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T1:
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_Exchange(pDataParams, wOption, pUsedTxBuffer, wTxLength, ppRxBuffer, pRxLength));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_SetConfig(
                                     phpalI7816p4_Sw_DataParams_t * pDataParams,
                                     uint16_t wConfig,
                                     uint16_t wValue
                                     )
{
    phStatus_t statusTmp;

    switch (wConfig)
    {
    case PHPAL_I7816P4_CONFIG_PROTOCOL:
        /* Parameter Check - Protocol Type 0-14 (in our case should be t=0 or t=1) */
        if(wValue < 0x00 || wValue > 0x0E)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }
        pDataParams->bProtocol = (uint8_t)wValue;
        break;

    case PHPAL_I7816P4_CONFIG_IFSC:
        /* The values '00' and 'FF' are rfu */
        if ((wValue > 0xFE) || (wValue < 0x01))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }
        pDataParams->bIfsc = (uint8_t)wValue;
        break;

    case PHPAL_I7816P4_CONFIG_IFSD:
        /* The values '00' and 'FF' are rfu */
        if ((wValue > 0xFE) || (wValue < 0x01))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }
        pDataParams->bIfsd = (uint8_t)wValue;
        break;

    case PHPAL_I7816P4_CONFIG_NAD:
        /* The value 'FF' is for PPS */
        if (wValue >= 0xFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }
        pDataParams->bNad = (uint8_t)(wValue);
        break;

    case PHPAL_I7816P4_CONFIG_AUTO_GETRESPONSE:
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }
        pDataParams->bAutoGetResponse = (uint8_t)(wValue);
        break;

    case PHPAL_I7816P4_CONFIG_AUTO_6C_HANDLING:
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }
        pDataParams->bAuto6cHandling = (uint8_t)(wValue);
        break;

    case PHPAL_I7816P4_CONFIG_TX_RETRANSMIT_COUNT:
        pDataParams->bT1TxRetransmitCount = (uint8_t)wValue;
        break;

    case PHPAL_I7816P4_CONFIG_RX_RETRANSMIT_COUNT:
        pDataParams->bT1RxRetransmitCount = (uint8_t)wValue;
        break;

    case PHPAL_I7816P4_CONFIG_I2C_PROTOCOL:
        if (wValue != PHPAL_I7816P4_I2C_PROTOCOL_T1OVERI2C_LEN_ONE_LRC &&
            wValue != PHPAL_I7816P4_I2C_PROTOCOL_T1OVERI2C_LEN_ONE_CRC &&
            wValue != PHPAL_I7816P4_I2C_PROTOCOL_GP_LEN_ONE_CRC &&
            wValue != PHPAL_I7816P4_I2C_PROTOCOL_GP_LEN_TWO_CRC &&
            wValue != PHPAL_I7816P4_I2C_PROTOCOL_GP_LEN_TWO_CRC_SWAPPED)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }
        pDataParams->wI2CProtocol = wValue;

        /* set according redundancy mode */
        if((wValue & PHPAL_I7816P4_SET_CONFIG_MASK_REDUNDANCY_MODE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LRC)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE,
                PHHAL_HW_CONTACT_MODE_LRC));
        }
        else
        {
            if((wValue & PHPAL_I7816P4_SET_CONFIG_MASK_CRC_SWAPPED) == PHPAL_I7816P4_SW_I2C_PROTOCOL_CRC_NOT_SWAPPED)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE,
                    PHHAL_HW_CONTACT_MODE_CRC));
            }
            else
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE,
                    PHHAL_HW_CONTACT_MODE_CRC_SWAPPED));
            }
        }
        break;

    default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_I7816P4);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_GetConfig(
                                     phpalI7816p4_Sw_DataParams_t * pDataParams,
                                     uint16_t wConfig,
                                     uint16_t * pValue
                                     )
{
    switch (wConfig)
    {
    case PHPAL_I7816P4_CONFIG_PROTOCOL:
        *pValue = (uint16_t)pDataParams->bProtocol;
        break;

    case PHPAL_I7816P4_CONFIG_IFSC:
        *pValue = (uint16_t)pDataParams->bIfsc;
        break;

    case PHPAL_I7816P4_CONFIG_IFSD:
        *pValue = (uint16_t)pDataParams->bIfsd;
        break;

    case PHPAL_I7816P4_CONFIG_NAD:
        *pValue = (uint16_t)pDataParams->bNad;
        break;

    case PHPAL_I7816P4_CONFIG_AUTO_GETRESPONSE:
        *pValue = (uint16_t)pDataParams->bAutoGetResponse;
        break;

    case PHPAL_I7816P4_CONFIG_AUTO_6C_HANDLING:
        *pValue = (uint16_t)pDataParams->bAuto6cHandling;
        break;

    case PHPAL_I7816P4_CONFIG_TX_RETRANSMIT_COUNT:
        *pValue = (uint16_t)pDataParams->bT1TxRetransmitCount;
        break;

    case PHPAL_I7816P4_CONFIG_RX_RETRANSMIT_COUNT:
        *pValue = (uint16_t)pDataParams->bT1RxRetransmitCount;
        break;

    case PHPAL_I7816P4_CONFIG_I2C_PROTOCOL:
        *pValue = pDataParams->wI2CProtocol;
        break;

    default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_I7816P4);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_ResetProtocol(
                                         phpalI7816p4_Sw_DataParams_t * pDataParams
                                         )
{
    phStatus_t statusTmp;

    /* an ATR has to be sent */
    pDataParams->bT1CardIfs = PHPAL_I7816P4_SW_PHDLPROTOCOL_T1_MAX_INF_SIZE;

    pDataParams->bT1SeqNumReceive = 0;
    pDataParams->bT1SeqNumSend = 0;     /*I_BLOCK_SEQ;*/

    pDataParams->wT0LengthExpected  = 0x00;

    /* Apply default parameters */
    pDataParams->bProtocol          = 0xFF;

    pDataParams->pIntPayload        = 0x00;
    pDataParams->wIntPayloadLen     = 0x00;
    pDataParams->pIntBuffer[0]      = 0x00;
    pDataParams->wIntBufferLen      = 0x00;

    pDataParams->pRxBuffer[0]       = 0x00;
    pDataParams->wRxBufferLen       = 0x00;

    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_ResetStates(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

#endif /* NXPBUILD__PHPAL_I7816P4_SW */
