/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software ISO7816 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHwContact.h>
#include <phpalI7816p4.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_I7816P4_SW

#include "phpalI7816p4_Sw_Int.h"
#include "phpalI7816p4_Sw.h"

phStatus_t phpalI7816p4_Sw_Int_T1_Exchange(
                                           phpalI7816p4_Sw_DataParams_t * pDataParams,
                                           uint16_t wOption,
                                           uint8_t * pTxBuffer,
                                           uint16_t wTxLength,
                                           uint8_t ** ppRxBuffer,
                                           uint16_t * pRxLength
                                           )
{
    phStatus_t statusTmp;
    phStatus_t statusExchange;
    phStatus_t statusRxStateMachine;
    uint8_t retransmitCount = 0;

    /* to satisfy compiler */
    if(wOption);

    if(pDataParams->bT1TxState != PHDLPROTOCOL_T1_TXSUB_READY)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
    }

    if(pDataParams->bT1MainState != PHDLPROTOCOL_T1_STATE_TX)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
    }

    pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_I;
    pDataParams->pIntPayload = pTxBuffer;
    pDataParams->wIntPayloadLen = wTxLength;
    pDataParams->wRxBufferLen = 0;

    /* invoke state machine*/
    statusExchange = phpalI7816p4_Sw_Int_T1_TxStateMachine(pDataParams, 0, ((wOption & PH_EXCHANGE_TXCHAINING) == PH_EXCHANGE_TXCHAINING) ? PH_ON : PH_OFF);

    if ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* set to RX-MODE*/
        pDataParams->bT1MainState = PHDLPROTOCOL_T1_STATE_RX;

        /* we want to send chained data */
        while(pDataParams->bT1RxState != PHDLPROTOCOL_T1_RXSUB_READY)
        {
            if (pDataParams->bT1TxState == PHDLPROTOCOL_T1_TXSUB_I)
            {
                pDataParams->bT1SeqNumSend ^= PHPAL_I7816P4_SW_I_BLOCK_SEQ;
            }

            statusRxStateMachine = phpalI7816p4_Sw_Int_T1_RxStateMachine(pDataParams, ((wOption & PH_EXCHANGE_TXCHAINING) == PH_EXCHANGE_TXCHAINING) ? PH_ON : PH_OFF);

            if((statusRxStateMachine & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_ResetStates(pDataParams));
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
            }

            if ((wOption & PH_EXCHANGE_TXCHAINING) == PH_EXCHANGE_TXCHAINING)
            {
                memcpy(*ppRxBuffer, pDataParams->pIntBuffer, pDataParams->wIntBufferLen);
                *pRxLength = pDataParams->wIntBufferLen;

                /*set to TX-MODE*/
                pDataParams->bT1MainState = PHDLPROTOCOL_T1_STATE_TX;

                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
            }
        }
    }
    else if((statusExchange & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_ResetStates(pDataParams));
        return statusExchange;
    }

    if (pDataParams->bT1RxState != PHDLPROTOCOL_T1_RXSUB_READY || pDataParams->bT1MainState != PHDLPROTOCOL_T1_STATE_RX)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_ResetStates(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
    }

    retransmitCount = 0;
    pDataParams->bT1RxState = PHDLPROTOCOL_T1_RXSUB_I;

    do
    {
        /* check size */
        if (pDataParams->wIntBufferLen < PHPAL_I7816P4_SW_PHDLPROTOCOL_T1_MIN_BLOCK_SIZE)
        {
            statusExchange = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
        }

        if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
        {
            /* check if LEN field is correct */
            if ((pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET]+PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE) != pDataParams->wIntBufferLen)
            {
                statusExchange = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
            }
        }
        else
        {
            /* check if LEN field is correct */
            if (((pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] << 8) +
                pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1] +
                PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO) != pDataParams->wIntBufferLen)
            {
                statusExchange = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
            }
        }

        if ((statusExchange & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            if (retransmitCount < pDataParams->bT1RxRetransmitCount)
            {
                retransmitCount++;
                statusExchange = phpalI7816p4_Sw_Int_T1_TxStateMachine(pDataParams, 1, PH_OFF);
            }
            else
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_ResetStates(pDataParams));
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
            }
        }
        else
        {
            if (pDataParams->bT1TxState == PHDLPROTOCOL_T1_TXSUB_I)
            {
                pDataParams->bT1SeqNumSend = pDataParams->bT1SeqNumSend ^ PHPAL_I7816P4_SW_I_BLOCK_SEQ;
            }

            while(pDataParams->bT1TxState != PHDLPROTOCOL_T1_TXSUB_READY)
            {
                statusRxStateMachine = phpalI7816p4_Sw_Int_T1_RxStateMachine(pDataParams, PH_OFF);

                if((statusRxStateMachine & PH_ERR_MASK) != PH_ERR_SUCCESS)
                {
                    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_ResetStates(pDataParams));
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
                }
            }
        }
    } while(statusExchange != PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4) && retransmitCount < pDataParams->bT1RxRetransmitCount);

    if (pDataParams->bT1MainState == PHDLPROTOCOL_T1_STATE_ERROR)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_ResetStates(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
    }

    memcpy(*ppRxBuffer, pDataParams->pIntPayload, pDataParams->wIntPayloadLen);
    *pRxLength = pDataParams->wIntPayloadLen;

    /*set to TX-MODE*/
    pDataParams->bT1MainState = PHDLPROTOCOL_T1_STATE_TX;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T1_TxStateMachine(
                                                 phpalI7816p4_Sw_DataParams_t * pDataParams,
                                                 uint8_t error,
                                                 uint8_t bForceTxChaining
                                                 )
{
    uint16_t length = 0;

    if (!error)
    {
        switch(pDataParams->bT1TxState)
        {
                case PHDLPROTOCOL_T1_TXSUB_I:
                    /* max buffer space minus header and footer */
                    /* length = longest payload length that fits in buffer when packaged in a block. */
                    length = pDataParams->bIfsc;

                    /*prepare block*/
                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_NAD_OFFSET]  =  pDataParams->bNad;
                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET]  =  PHPAL_I7816P4_SW_I_BLOCK_SIGNATURE;
                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] |=  pDataParams->bT1SeqNumSend;

                    /*use chaining if necessary*/
                    if (pDataParams->wIntPayloadLen > length || bForceTxChaining == PH_ON)
                    {
                        /*new rx state (expected block)*/
                        pDataParams->bT1RxState = PHDLPROTOCOL_T1_RXSUB_R_ACK;
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] |= PHPAL_I7816P4_SW_I_BLOCK_CHAINING;

                        /* If chaining is only forced limit the length to data length */
                        if (!(pDataParams->wIntPayloadLen > length))
                        {
                            length = (uint16_t)pDataParams->wIntPayloadLen;
                        }
                    }
                    else
                    {
                        /*new rx state (expected block)*/
                        pDataParams->bT1RxState = PHDLPROTOCOL_T1_RXSUB_READY;
                        length = (uint16_t) pDataParams->wIntPayloadLen;
                    }

                    if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = (uint8_t)length;
                        memcpy(pDataParams->pIntBuffer + PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE, pDataParams->pIntPayload, length);

                        /* NAD, PCB, LEN, length bytes of payload */
                        pDataParams->wIntBufferLen = length + PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE;
                    }
                    else
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = (uint8_t)(length >> 8);
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1] = (uint8_t)(length & 0x00FF);
                        memcpy(pDataParams->pIntBuffer + PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO, pDataParams->pIntPayload, length);

                        /* NAD, PCB, LEN, length bytes of payload */
                        pDataParams->wIntBufferLen = length + PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO;
                    }
                    break;

                case PHDLPROTOCOL_T1_TXSUB_S_IFS_ACK:
                    /*process IFS request*/
                    if (pDataParams->bT1CardIfs < PHPAL_I7816P4_SW_INT_BUFFER_SIZE)
                    {
                        pDataParams->bIfsc = pDataParams->bT1CardIfs;
                    }
                    else
                    {
                        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_I7816P4);
                    }

                    /**new rx state (expected I block)*/
                    pDataParams->bT1RxState = PHDLPROTOCOL_T1_RXSUB_I;

                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_NAD_OFFSET] = pDataParams->bNad;
                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] = PHPAL_I7816P4_SW_T1_S_IFS_ACK;

                    if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = 0x01;
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE] = pDataParams->bT1CardIfs;
                        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE + 1;  /* NAD, PCB, LEN, one byte of payload */
                    }
                    else
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = 0x00;
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1] = 0x01;
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO] = pDataParams->bT1CardIfs;
                        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO + 1;  /* NAD, PCB, 2 Byte LEN, one byte of payload */
                    }
                    break;

                case PHDLPROTOCOL_T1_TXSUB_S_WTX_ACK:
                    /* if reader is chaining previous state was ack */
                    if(pDataParams->bT1RxState != PHDLPROTOCOL_T1_RXSUB_R_ACK)
                    {
                        /* previous state before wtx */
                        pDataParams->bT1RxState = PHDLPROTOCOL_T1_RXSUB_I;
                    }

                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_NAD_OFFSET] = pDataParams->bNad;
                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] = PHPAL_I7816P4_SW_T1_S_WTX_ACK;

                    if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = 0x01;
                        /* payload should contain same INF field as in request */
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE] = pDataParams->bBwtMultiplier;
                        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE + 1;  /* NAD, PCB, LEN, one byte of payload */
                    }
                    else
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = 0x00;
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1] = 0x01;
                        /* payload should contain same INF field as in request */
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO] = pDataParams->bBwtMultiplier;
                        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO + 1;  /* NAD, PCB, 2 Byte LEN, one byte of payload */
                    }
                    break;

                case PHDLPROTOCOL_T1_TXSUB_ACK:
                    /**new rx state (expected block)*/
                    pDataParams->bT1RxState = PHDLPROTOCOL_T1_RXSUB_I;

                    /*prepare ifs request*/
                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_NAD_OFFSET] = pDataParams->bNad;
                    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] = PHPAL_I7816P4_SW_R_BLOCK_SIGNATURE;
                    if (0 != pDataParams->bT1SeqNumReceive)
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] |= PHPAL_I7816P4_SW_R_BLOCK_SEQ;
                    }

                    if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = 0x00;
                        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE;  /* NAD, PCB, LEN, no payload */
                    }
                    else
                    {
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = 0x00;
                        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1] = 0x00;
                        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO;  /* NAD, PCB, 2 Byte LEN, no payload */
                    }
                    break;

                default:
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
        }
    }
    else
    {
        /* errors are no specific states */
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_NAD_OFFSET] = pDataParams->bNad;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] = PHPAL_I7816P4_SW_R_BLOCK_SIGNATURE;

        /*TODO define a specific error*/
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] |= PHPAL_I7816P4_SW_R_BLOCK_ERROR_MASK & 0x01;

        if (pDataParams->bT1SeqNumReceive != 0)
        {
            pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] |= PHPAL_I7816P4_SW_R_BLOCK_SEQ;
        }

        if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
        {
            pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = 0x00;
            pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE;  /* NAD, PCB, LEN, no payload */
        }
        else
        {
            pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = 0x00;
            pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1] = 0x00;
            pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO;  /* NAD, PCB, LEN, no payload */
        }
    }

    return phpalI7816p4_Sw_Int_T1_SendBlock(pDataParams);
}

phStatus_t phpalI7816p4_Sw_Int_T1_RxStateMachine(
                                                 phpalI7816p4_Sw_DataParams_t * pDataParams,
                                                 uint8_t bForceTxChaining
                                                 )
{
    uint8_t length = 0;
    uint8_t retransmitCount = 0;

    switch(pDataParams->bT1RxState)
    {
        case PHDLPROTOCOL_T1_RXSUB_S_IFS_ACK:
            pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_READY;
            break;

        case PHDLPROTOCOL_T1_RXSUB_S_BWT_ACK:
            pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_READY;
            break;

        case PHDLPROTOCOL_T1_RXSUB_I:
            /* received error response, retransmit last packet */
            while ((pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET]&PHPAL_I7816P4_SW_R_BLOCK_MASK) == PHPAL_I7816P4_SW_R_BLOCK_SIGNATURE &&
                 pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET]&PHPAL_I7816P4_SW_R_BLOCK_ERROR_MASK)
            {
                if(retransmitCount == pDataParams->bT1RxRetransmitCount)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
                }
                if(retransmitCount == 0)
                {
                    pDataParams->bT1SeqNumSend = pDataParams->bT1SeqNumSend ^ PHPAL_I7816P4_SW_I_BLOCK_SEQ;
                }
                phpalI7816p4_Sw_Int_T1_TxStateMachine(pDataParams, 0, PH_OFF);
                retransmitCount++;
            }

            if (pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] == PHPAL_I7816P4_SW_T1_S_IFS_REQUEST)
            {
                pDataParams->bT1CardIfs =  pDataParams->pIntBuffer[
                    ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE) ?
                    PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE : PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO];
                    pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_S_IFS_ACK;
            }
            else if (pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] == PHPAL_I7816P4_SW_T1_S_WTX_REQUEST)
            {
                /*need to process ack*/
                pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_S_WTX_ACK;

                if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
                {
                    pDataParams->bBwtMultiplier = pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE];
                }
                else
                {
                    pDataParams->bBwtMultiplier = pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO];
                }
            }
            else
            {
                /*check sequence counter*/
                while ((pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] & PHPAL_I7816P4_SW_I_BLOCK_SEQ) != pDataParams->bT1SeqNumReceive)
                {
                    if(retransmitCount == pDataParams->bT1RxRetransmitCount)
                    {
                        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
                    }
                    if(retransmitCount == 0)
                    {
                        pDataParams->bT1SeqNumSend = pDataParams->bT1SeqNumSend ^ PHPAL_I7816P4_SW_I_BLOCK_SEQ;
                    }
                    phpalI7816p4_Sw_Int_T1_TxStateMachine(pDataParams, 0, PH_OFF);
                    retransmitCount++;
                }

                if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
                {
                    memcpy(pDataParams->pRxBuffer + pDataParams->wRxBufferLen, &pDataParams->pIntBuffer[0] + PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE, pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET]);
                    pDataParams->wRxBufferLen += pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET];
                }
                else
                {
                    memcpy(pDataParams->pRxBuffer + pDataParams->wRxBufferLen, &pDataParams->pIntBuffer[0] + PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO,
                        (pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] << 8) + pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1]);
                    pDataParams->wRxBufferLen += (pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] << 8) + pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1];
                }

                if ((pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] & PHPAL_I7816P4_SW_I_BLOCK_CHAINING) != 0)
                {
                    pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_ACK;
                }
                else
                {
                    pDataParams->pIntPayload = pDataParams->pRxBuffer;
                    pDataParams->wIntPayloadLen = pDataParams->wRxBufferLen;

                    pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_READY;
                }

                /*next expected I-Block*/
                pDataParams->bT1SeqNumReceive=pDataParams->bT1SeqNumReceive^PHPAL_I7816P4_SW_I_BLOCK_SEQ;
            }
            break;

        case PHDLPROTOCOL_T1_RXSUB_R_ACK:
            if (pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] == PHPAL_I7816P4_SW_T1_S_WTX_REQUEST)
            {
                /*need to process ack*/
                pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_S_WTX_ACK;

                if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
                {
                    pDataParams->bBwtMultiplier = pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE];
                }
                else
                {
                    pDataParams->bBwtMultiplier = pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO];
                }
            }
            else
            {
                /* In case the tx chaining was forced just return */
                if (bForceTxChaining == PH_ON)
                {
                    pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_READY;
                    break;
                }
                if(PHPAL_I7816P4_SW_PHDLPROTOCOL_T1_MAX_INF_SIZE > pDataParams->bIfsc)
                {
                    /* length = longest payload length that fits in buffer when packaged in a block. */
                    length = pDataParams->bIfsc;
                }
                else
                {
                    /* length = longest allowed payload length according to spec. */
                    length = PHPAL_I7816P4_SW_PHDLPROTOCOL_T1_MAX_INF_SIZE;
                }

                pDataParams->wIntPayloadLen -= length;
                pDataParams->pIntPayload    += length;
                pDataParams->bT1TxState      = PHDLPROTOCOL_T1_TXSUB_I;
            }
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
    }

    if (PHDLPROTOCOL_T1_TXSUB_READY != pDataParams->bT1TxState)
    {
        /*maybe in extra task*/
        return phpalI7816p4_Sw_Int_T1_TxStateMachine(pDataParams, 0, PH_OFF);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T0_Exchange(
                                           phpalI7816p4_Sw_DataParams_t * pDataParams,
                                           uint16_t wOption,
                                           uint8_t * pTxBuffer,
                                           uint16_t wTxLength,
                                           uint8_t ** ppRxBuffer,
                                           uint16_t * pRxLength
                                           )
{
    phStatus_t statusTmp;

    /* to satisfy compiler */
    if(wOption);

    pDataParams->pIntPayload = pTxBuffer;
    pDataParams->wIntPayloadLen = wTxLength;

    /* Parse APDU and set pDataParams->wT0LengthExpected */
    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_Parse_Apdu(pDataParams, pTxBuffer, wTxLength));

    /* Send Header and Data if available */
    switch(pDataParams->bT0CurrentCase)
    {
    case PHPAL_I7816P4_SW_CASE_1:
        memcpy(pDataParams->pT0Header, pDataParams->pIntPayload, pDataParams->wIntPayloadLen);
        memcpy(&pDataParams->pIntBuffer[0], pDataParams->pIntPayload, pDataParams->wIntPayloadLen);
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] = 0x00;
        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH;
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_SendBlockExpectedBytes(pDataParams, PHPAL_I7816P4_SW_INS_LENGTH));
        break;

    case PHPAL_I7816P4_SW_CASE_2S:
        memcpy(pDataParams->pT0Header, pDataParams->pIntPayload, pDataParams->wIntPayloadLen);
        memcpy(&pDataParams->pIntBuffer[0], pDataParams->pIntPayload, pDataParams->wIntPayloadLen);
        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH;
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_SendBlockExpectedBytes(pDataParams, PHPAL_I7816P4_SW_INS_LENGTH));
        break;

    case PHPAL_I7816P4_SW_CASE_3S:
    case PHPAL_I7816P4_SW_CASE_4S:
        memcpy(pDataParams->pT0Header, pDataParams->pIntPayload, PHPAL_I7816P4_SW_HEADER_LENGTH);
        memcpy(&pDataParams->pIntBuffer[0], pDataParams->pIntPayload, PHPAL_I7816P4_SW_HEADER_LENGTH);
        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH;
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_SendBlockExpectedBytes(pDataParams, PHPAL_I7816P4_SW_INS_LENGTH));

        /* Check the received byte */
        if(pDataParams->pIntBuffer[0] == pDataParams->pT0Header[PHPAL_I7816P4_SW_HEADER_INS_OFFSET])
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_SendRestOfCommand(pDataParams));
        }
        else
        {
            /* pDataParams->pIntBuffer[0] == ~(pDataParams->pT0Header[PHPAL_I7816P4_SW_HEADER_INS_OFFSET] Not implemented */
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
        }
        break;

    case PHPAL_I7816P4_SW_CASE_2E:
    case PHPAL_I7816P4_SW_CASE_3E:
    case PHPAL_I7816P4_SW_CASE_4E:
    default:
        /* extended length not supported for now */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_I7816P4);
    }

    /* We received the procedure byte (NULL byte handling done in send expected function) */
    if(((pDataParams->pIntBuffer[PHPAL_I7816P4_SW_SW1_OFFSET] & 0xF0) == 0x60) ||
       ((pDataParams->pIntBuffer[PHPAL_I7816P4_SW_SW1_OFFSET] & 0xF0) == 0x90))
    {
        /* We got SW1 so receive SW2 */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_ReceiveBytes(pDataParams, 1));

        if(pDataParams->pIntBuffer[PHPAL_I7816P4_SW_SW1_OFFSET] == PHPAL_I7816P4_SW_PROCEDURE_BYTE_61)
        {
            /* 0x61 mean OK - receive data now - SW2 encodes the length */
            if(pDataParams->bAutoGetResponse)
            {
                if(pDataParams->pIntBuffer[PHPAL_I7816P4_SW_SW2_OFFSET] == 0)
                {
                    /* extended length not supported for now */
                    PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_I7816P4);
                }
                else
                {
                    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_ReceiveFullResponseCase4(pDataParams));
                }
            }
            else
            {
                /* skip the GetResponse, just return the SW12 */
                memcpy(&pDataParams->pIntPayload[0], &pDataParams->pIntBuffer[0], PHPAL_I7816P4_SW_SW_LENGTH);
                pDataParams->wIntPayloadLen = PHPAL_I7816P4_SW_SW_LENGTH;
            }
        }
        else if(pDataParams->pIntBuffer[PHPAL_I7816P4_SW_SW1_OFFSET] == PHPAL_I7816P4_SW_PROCEDURE_BYTE_6C)
        {
            if(pDataParams->bAuto6cHandling)
            {
                /* Length we want to get in SW2 */
                PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_ReceiveFullResponseCase2(pDataParams));
            }
            else
            {
                memcpy(&pDataParams->pIntPayload[0], &pDataParams->pIntBuffer[0], PHPAL_I7816P4_SW_SW_LENGTH);
                pDataParams->wIntPayloadLen = PHPAL_I7816P4_SW_SW_LENGTH;
            }
        }
        else
        {
            /* Return SW */
            memcpy(&pDataParams->pIntPayload[0], &pDataParams->pIntBuffer[0], PHPAL_I7816P4_SW_SW_LENGTH);
            pDataParams->wIntPayloadLen = PHPAL_I7816P4_SW_SW_LENGTH;
        }
    }
    else if(pDataParams->pIntBuffer[0] == pDataParams->pT0Header[PHPAL_I7816P4_SW_HEADER_INS_OFFSET])
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_ReceiveFullResponseCase2(pDataParams));
    }
    else
    {
        /* pDataParams->pIntBuffer[0] == ~(pDataParams->pT0Header[PHPAL_I7816P4_SW_HEADER_INS_OFFSET] Not implemented */
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
    }

    /* Copy payload to rx buffer */
    memcpy(*ppRxBuffer, &pDataParams->pIntPayload[0], pDataParams->wIntPayloadLen);
    *pRxLength = pDataParams->wIntPayloadLen;

    /* Reset IntBufferLen after communication */
    pDataParams->wIntBufferLen = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T0_Parse_Apdu(
                                             phpalI7816p4_Sw_DataParams_t * pDataParams,
                                             uint8_t * pTxBuffer,
                                             uint16_t wTxLength
                                             )
{
    /* see table 13 in ISO7816p3 */
    uint8_t C5 = pTxBuffer[4];

    if (wTxLength == 4)
    {
        /* case 1 */
        pDataParams->wT0LengthExpected = 0;
        pDataParams->bT0CurrentCase = PHPAL_I7816P4_SW_CASE_1;
    }
    else if (wTxLength == 5)
    {
        /*  case 2S */
        pDataParams->wT0LengthExpected = C5;
        pDataParams->bT0CurrentCase = PHPAL_I7816P4_SW_CASE_2S;
    }
    else if (wTxLength == 5 + C5)
    {
        /*  case 3S */
        pDataParams->wT0LengthExpected = 0;
        pDataParams->bT0CurrentCase = PHPAL_I7816P4_SW_CASE_3S;
    }
    else if (wTxLength == 6+C5)
    {
        /*  case 4S */
        pDataParams->wT0LengthExpected = pTxBuffer[wTxLength - 1];
        pDataParams->bT0CurrentCase = PHPAL_I7816P4_SW_CASE_4S;
    }
    else
    {
        /* extended length needed from now on */
        uint16_t C6C7 = pTxBuffer[5] * 256 + pTxBuffer[6];
        if (wTxLength == 7)
        {
            /*  case 2E */
            pDataParams->wT0LengthExpected = C6C7;
            pDataParams->bT0CurrentCase = PHPAL_I7816P4_SW_CASE_2E;
        }
        else if (wTxLength == 7+C6C7)
        {
            /*  case 3E */
            pDataParams->wT0LengthExpected = 0;
            pDataParams->bT0CurrentCase = PHPAL_I7816P4_SW_CASE_3E;
        }
        else if (wTxLength == 9+C6C7)
        {
            /*  case 4E */
            pDataParams->wT0LengthExpected = pTxBuffer[wTxLength - 2] * 256 + pTxBuffer[wTxLength - 1];
            pDataParams->bT0CurrentCase = PHPAL_I7816P4_SW_CASE_4E;
        }
        else
        {
            /* malformed APDU */
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_I7816P4);
        }

    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T0_SendCommandExtendedLength(
                                                            phpalI7816p4_Sw_DataParams_t * pDataParams
                                                            )
{
    uint16_t length;
    uint16_t index = 0;
    uint8_t lc;

    length = pDataParams->wIntPayloadLen;

    while(length > 0)
    {
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_CLA_OFFSET] = 0;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_INS_OFFSET] = PHPAL_I7816P4_SW_ENVELOPE_INS;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P1_OFFSET] = 0;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P2_OFFSET] = 0;

        if ( length >= PHPAL_I7816P4_SW_MAX_VALUE_SHORT_PAYLOAD )
        {
            pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] = (PHPAL_I7816P4_SW_MAX_VALUE_SHORT_PAYLOAD - 1);
        }
        else
        {
            pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] = (uint8_t)length;
        }

        lc = pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET];
        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH;

        phpalI7816p4_Sw_Int_T1_SendBlock(pDataParams);
        phpalI7816p4_Sw_Int_T0_ReceiveBlock(pDataParams);

        if(pDataParams->pIntBuffer[0] == PHPAL_I7816P4_SW_ENVELOPE_INS)
        {
            memcpy (&pDataParams->pIntBuffer[0], &pDataParams->pIntPayload[index], lc);
            pDataParams->wIntBufferLen = lc;
            index = index + lc;
            length = length - lc;

            phpalI7816p4_Sw_Int_T1_SendBlock(pDataParams);
            phpalI7816p4_Sw_Int_T0_ReceiveBlock(pDataParams);

            /* Should check here the answer is = 90 00 */
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_I7816P4);
        }
    }

    /* End of data string */
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_CLA_OFFSET] = 0;
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_INS_OFFSET] = PHPAL_I7816P4_SW_ENVELOPE_INS;
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P1_OFFSET] = 0;
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P2_OFFSET] = 0;
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] = 0;

    pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH ;

    phpalI7816p4_Sw_Int_T1_SendBlock(pDataParams);

    return PH_ERR_SUCCESS;
}

phStatus_t phpalI7816p4_Sw_Int_T0_SendRestOfCommand(
                                                    phpalI7816p4_Sw_DataParams_t * pDataParams
                                                    )
{
    phStatus_t statusTmp;
    memcpy(&pDataParams->pIntBuffer[0], &pDataParams->pIntPayload[PHPAL_I7816P4_SW_HEADER_LENGTH], pDataParams->pIntPayload[PHPAL_I7816P4_SW_HEADER_P3_OFFSET]);
    pDataParams->wIntBufferLen = pDataParams->pIntPayload[PHPAL_I7816P4_SW_HEADER_P3_OFFSET];
    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_SendBlockExpectedBytes(pDataParams, PHPAL_I7816P4_SW_INS_LENGTH));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T0_ReceiveBlock(
                                               phpalI7816p4_Sw_DataParams_t * pDataParams
                                               )
{
    /*phdlProtocol_TxRxParameter recParam; */
    /*recParam.pBuffer = control.bufferRxTx.pData;  */
    /*recParam.size = control.bufferRxTx.size; */
    /*recParam.callback = phdlProtocol_T0_ReceiveCallback; */

    /* set receive */
    /*control.receive(&recParam); */

    while(pDataParams->pIntBuffer[0] == PHPAL_I7816P4_SW_PROCEDURE_BYTE_NULL)
    {
        if (pDataParams->wIntBufferLen > 1)
        {
            pDataParams->wIntBufferLen--;
            memcpy(&pDataParams->pIntBuffer[0], &pDataParams->pIntBuffer[1], pDataParams->wIntBufferLen);
        }
        else
        {
            /* no action wait for procedure byte */
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T0_ReceiveExtendedLength(
                                                        phpalI7816p4_Sw_DataParams_t * pDataParams
                                                        )
{
    while(pDataParams->pIntBuffer[pDataParams->wIntBufferLen - 2] == 0x61)
    {
        /* Parse response */
        pDataParams->wT0LengthExpected = pDataParams->pIntBuffer[pDataParams->wIntBufferLen - 1];

        if(pDataParams->wIntBufferLen > PHPAL_I7816P4_SW_SW_LENGTH)
        {
            if(pDataParams->pIntBuffer[0] == PHPAL_I7816P4_SW_GET_RESPONSE_INS)
            {
                memcpy(pDataParams->pIntPayload, &pDataParams->pIntBuffer[1], pDataParams->wIntBufferLen - (PHPAL_I7816P4_SW_SW_LENGTH + 1));
                pDataParams->pIntPayload = pDataParams->pIntPayload + pDataParams->wIntBufferLen - (PHPAL_I7816P4_SW_SW_LENGTH + 1);
                pDataParams->wIntPayloadLen -= pDataParams->wIntBufferLen - (PHPAL_I7816P4_SW_SW_LENGTH + 1);
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_I7816P4);
            }
        }

        /*Prepare next get response command */
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_CLA_OFFSET] = 0;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_INS_OFFSET] = PHPAL_I7816P4_SW_GET_RESPONSE_INS;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P1_OFFSET] = 0;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P2_OFFSET] = 0;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] = (uint8_t)pDataParams->wT0LengthExpected;

        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH;

        phpalI7816p4_Sw_Int_T1_SendBlock(pDataParams);

        /*Receive */
        phpalI7816p4_Sw_Int_T0_ReceiveBlock(pDataParams);

    }

    /*Parse last response */
    if(pDataParams->wIntBufferLen > PHPAL_I7816P4_SW_SW_LENGTH)
    {
        if(pDataParams->pIntBuffer[0] == PHPAL_I7816P4_SW_GET_RESPONSE_INS)
        {
            memcpy(pDataParams->pIntPayload, &pDataParams->pIntBuffer[1], pDataParams->wIntBufferLen - 1);
            pDataParams->pIntPayload = pDataParams->pIntPayload + (pDataParams->wIntBufferLen - 1);
            pDataParams->wIntPayloadLen -= (pDataParams->wIntBufferLen - 1);
            /*control_T0.callback(pDataParams->pIntPayload, PHDL_PROTOCOL_CALLBACK_OK); */
            return PH_ERR_SUCCESS;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_I7816P4);
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_I7816P4);
    }
}

phStatus_t phpalI7816p4_Sw_Int_T0_ReceiveFullResponseCase2(
                                                           phpalI7816p4_Sw_DataParams_t * pDataParams
                                                           )
{
    phStatus_t statusTmp;

    if(pDataParams->pIntBuffer[PHPAL_I7816P4_SW_SW1_OFFSET] == PHPAL_I7816P4_SW_PROCEDURE_BYTE_6C)
    {
        /* 6C means we sent wrong LE field so adjust and receive correct length */
        pDataParams->wT0LengthExpected = pDataParams->pIntBuffer[PHPAL_I7816P4_SW_SW2_OFFSET];
        memcpy(&pDataParams->pIntBuffer[0], pDataParams->pT0Header, PHPAL_I7816P4_SW_HEADER_LENGTH - 1);
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] = (uint8_t)pDataParams->wT0LengthExpected;
        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH;

        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_SendBlockExpectedBytes(pDataParams, pDataParams->wT0LengthExpected + PHPAL_I7816P4_SW_SW_LENGTH + PHPAL_I7816P4_SW_INS_LENGTH));
        memcpy(&pDataParams->pIntPayload[0] , &pDataParams->pIntBuffer[1], pDataParams->wIntBufferLen - 1);
        pDataParams->wIntPayloadLen = pDataParams->wIntBufferLen - 1;
    }
    else if(pDataParams->pIntBuffer[0] == pDataParams->pT0Header[PHPAL_I7816P4_SW_HEADER_INS_OFFSET])
    {
        /* We already know how many bytes we get so we receive them */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_ReceiveBytes(pDataParams, pDataParams->pT0Header[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] + PHPAL_I7816P4_SW_SW_LENGTH));
        memcpy(&pDataParams->pIntPayload[0] , &pDataParams->pIntBuffer[1], pDataParams->wIntBufferLen - 1);
        pDataParams->wIntPayloadLen = pDataParams->wIntBufferLen - 1;
    }

    /* TODOGL: while loop? */
    /* We receive SW1 61 which means there is even more data available */
    if(pDataParams->pIntBuffer[pDataParams->wIntBufferLen - 2] == PHPAL_I7816P4_SW_PROCEDURE_BYTE_61)
    {
        memcpy(pDataParams->pIntPayload , &pDataParams->pIntBuffer[1], pDataParams->wIntBufferLen - (PHPAL_I7816P4_SW_SW_LENGTH + 1));
        pDataParams->pIntPayload = pDataParams->pIntPayload + pDataParams->wIntBufferLen - (PHPAL_I7816P4_SW_SW_LENGTH + 1);
        pDataParams->wIntPayloadLen -= pDataParams->wIntBufferLen - (PHPAL_I7816P4_SW_SW_LENGTH + 1);

        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_CLA_OFFSET] = 0;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_INS_OFFSET] = PHPAL_I7816P4_SW_GET_RESPONSE_INS;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P1_OFFSET] = 0;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P2_OFFSET] = 0;
        pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] = (uint8_t)pDataParams->wT0LengthExpected;

        pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH;

        /*phpalI7816p4_Sw_Int_T1_SendBlock(pDataParams);*/
        phpalI7816p4_Sw_Int_T0_ReceiveBlock(pDataParams);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T0_ReceiveFullResponseCase4(
                                                           phpalI7816p4_Sw_DataParams_t * pDataParams
                                                           )
{
    phStatus_t statusTmp;

    /* TODOGL: if less bytes available isnt it a fail? */
    if (pDataParams->pIntBuffer[1] < pDataParams->wT0LengthExpected || pDataParams->wT0LengthExpected == 0)
    {
        /* less bytes available than requested */
        pDataParams->wT0LengthExpected = pDataParams->pIntBuffer[1];
    }

    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_CLA_OFFSET] = pDataParams->pT0Header[0];
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_INS_OFFSET] = PHPAL_I7816P4_SW_GET_RESPONSE_INS;
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P1_OFFSET] = 0;
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P2_OFFSET] = 0;
    pDataParams->pIntBuffer[PHPAL_I7816P4_SW_HEADER_P3_OFFSET] = (uint8_t)pDataParams->wT0LengthExpected;

    pDataParams->wIntBufferLen = PHPAL_I7816P4_SW_HEADER_LENGTH;

    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T0_SendBlockExpectedBytes(pDataParams, pDataParams->wT0LengthExpected + PHPAL_I7816P4_SW_SW_LENGTH + PHPAL_I7816P4_SW_INS_LENGTH));

    if(pDataParams->pIntBuffer[0] != PHPAL_I7816P4_SW_GET_RESPONSE_INS)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
    }

    /* TODOGL: if SW is 61XX receive more data? */

    memcpy(pDataParams->pIntPayload, &pDataParams->pIntBuffer[1], pDataParams->wT0LengthExpected + PHPAL_I7816P4_SW_SW_LENGTH);
    pDataParams->wIntPayloadLen = pDataParams->wT0LengthExpected + PHPAL_I7816P4_SW_SW_LENGTH;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T1_ResetStates(
                                              phpalI7816p4_Sw_DataParams_t * pDataParams
                                              )
{
    pDataParams->bT1MainState = PHDLPROTOCOL_T1_STATE_TX;
    pDataParams->bT1TxState = PHDLPROTOCOL_T1_TXSUB_READY;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T1_SendBlock(
                                            phpalI7816p4_Sw_DataParams_t * pDataParams
                                            )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    phStatus_t PH_MEMLOC_REM statusExchange;
    uint8_t *  PH_MEMLOC_REM pptmpRxBuffer;
    uint16_t   PH_MEMLOC_REM wtmpRxBufferLen;
    uint32_t   PH_MEMLOC_REM dwCurrentRedundancyMode;
    uint32_t   PH_MEMLOC_REM dwBwtTimeout;
    uint32_t   PH_MEMLOC_REM dwCurrentExpectedBytes;
    uint32_t   PH_MEMLOC_REM dwCommunicationChannel;
    uint32_t   PH_MEMLOC_REM dwRedundancyLength = 0;
    uint32_t   PH_MEMLOC_REM dwRemainingBytes = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* I2C/SPI mode first request the first 3 or 4 bytes and get the length out of this data to request the remaining data */
    if (dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C || dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI)
    {
        if(dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI) /* Set expected NAD for response detection */
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_SPI_RECV_NAD_VALUE, ((pDataParams->bNad & 0xF0) >> 4) | ((pDataParams->bNad & 0x0F) << 4)));
        }

        /* Backup old settings */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE,
            &dwCurrentRedundancyMode));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES,
            &dwCurrentExpectedBytes));

        /* Set Tx only mode */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE,
            (dwCurrentRedundancyMode & (PHHAL_HW_CONTACT_MODE_MASK_TX | PHHAL_HW_CONTACT_MODE_MASK_TX_SWAPPED))));

        if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
        {
            /* Request only 3 bytes */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES,
                PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE));
        }
        else
        {
            /* Request only 4 bytes */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES,
                PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO));
        }

        /* check if bwt extension is needed in case of WTX */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_CheckForBwtExtension(pDataParams, &dwBwtTimeout));

        statusExchange = phhalHwContact_Exchange(
            pDataParams->pHalDataParams,
            PH_EXCHANGE_DEFAULT,
            pDataParams->pIntBuffer,
            pDataParams->wIntBufferLen,
            &pptmpRxBuffer,
            &wtmpRxBufferLen);

        /* restore bwt timeout value in case it was modified */
        PH_CHECK_SUCCESS_FCT(statusTmp,phpalI7816p4_Sw_Int_T1_RestoreBwtTimeoutIfModified(pDataParams, dwBwtTimeout));

        /* If received a valid frame of 3 or 4 bytes get remaining length and request data */
        if(((wtmpRxBufferLen == PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE && (pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE) ||
            (wtmpRxBufferLen == PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO && (pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_TWO)) &&
            (statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Set Full buffer check mode and only rx mode */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE,
                ((dwCurrentRedundancyMode & (PHHAL_HW_CONTACT_MODE_MASK_RX | PHHAL_HW_CONTACT_MODE_MASK_RX_SWAPPED)) | PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER)));

            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS,
                ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE) ?
                PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE : PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO));

            /* Request remaining bytes */
            if ((dwCurrentRedundancyMode & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_LRC)
            {
                dwRedundancyLength = 1;
            }
            else if ((dwCurrentRedundancyMode & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_CRC)
            {
                dwRedundancyLength = 2;
            }
            else
            {
                dwRedundancyLength = 0;
            }

            if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
            {
                dwRemainingBytes = pptmpRxBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET];
            }
            else
            {
                dwRemainingBytes = (pptmpRxBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] << 8) +
                    pptmpRxBuffer[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1];
            }

            dwRemainingBytes += dwRedundancyLength;

            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES,
                dwRemainingBytes));

            statusExchange = phhalHwContact_Exchange(
                pDataParams->pHalDataParams,
                PH_EXCHANGE_DEFAULT,
                NULL,
                0,
                &pptmpRxBuffer,
                &wtmpRxBufferLen);
        }

        /* Restore redundancy Settings */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE,
            dwCurrentRedundancyMode));
        /* Restore expected bytes */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES,
            dwCurrentExpectedBytes));
        /* Restore Rx Buffer startpos */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS,
            0));

        /* Check the status of the exchange after restore settings */
        PH_CHECK_SUCCESS(statusExchange);
    }
    else
    {
        if (dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI_EXCHANGE)
        {
            /* Set Correct GP Mode and Expected NAD for reader response extraction */
            if((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE) /* One byte length is only supported with mode T1SPI */
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_SPI_GP_MODE, PHHAL_HW_CONTACT_CONFIG_SPI_GP_MODE_T1SPI));
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_SPI_RECV_NAD_VALUE, ((pDataParams->bNad & 0xF0) >> 4) | ((pDataParams->bNad & 0x0F) << 4)));
            }
            else /* Use mode GP 2 if two byte length is used (more config posibilites and CRC anyway is done manual) */
            {
                if (pDataParams->bNad != 0x21) /* GP has hardcoded NAD 0x21/0x12 */
                {
                    return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_I7816P4);
                }
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_SPI_GP_MODE, PHHAL_HW_CONTACT_CONFIG_SPI_GP_MODE_GP_MODE_2));
            }
        }

        /* check if bwt extension is needed in case of WTX */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Sw_Int_T1_CheckForBwtExtension(pDataParams, &dwBwtTimeout));

        statusExchange = phhalHwContact_Exchange(
                pDataParams->pHalDataParams,
                PH_EXCHANGE_DEFAULT,
                pDataParams->pIntBuffer,
                pDataParams->wIntBufferLen,
                &pptmpRxBuffer,
                &wtmpRxBufferLen);

        /* restore bwt timeout value in case it was modified */
        PH_CHECK_SUCCESS_FCT(statusTmp,phpalI7816p4_Sw_Int_T1_RestoreBwtTimeoutIfModified(pDataParams, dwBwtTimeout));

        if((statusExchange & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            return statusExchange;
        }

        /* check if received payload is larger than ifsd */
        if((wtmpRxBufferLen - PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE) > pDataParams->bIfsd)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
        }
    }

    if(wtmpRxBufferLen < PHPAL_I7816P4_SW_INT_BUFFER_SIZE)
    {
        memcpy(pDataParams->pIntBuffer, pptmpRxBuffer, wtmpRxBufferLen);
        pDataParams->wIntBufferLen = wtmpRxBufferLen;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_I7816P4);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T1_CheckForBwtExtension(
                                                       phpalI7816p4_Sw_DataParams_t * pDataParams,
                                                       uint32_t * pOldBwtTimeout
                                                       )
{
    phStatus_t statusTmp;

    /* if wtx respose extend timeout */
    if (pDataParams->bT1TxState == PHDLPROTOCOL_T1_TXSUB_S_WTX_ACK)
    {
         /* Backup old setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_BWT_CLK,
            pOldBwtTimeout));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_BWT_CLK,
            *pOldBwtTimeout * pDataParams->bBwtMultiplier));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T1_RestoreBwtTimeoutIfModified(
                                                              phpalI7816p4_Sw_DataParams_t * pDataParams,
                                                              uint32_t dwOldBwtTimeout
                                                              )
{
    phStatus_t statusTmp;

    /* if wtx respose extend timeout */
    if (pDataParams->bT1TxState == PHDLPROTOCOL_T1_TXSUB_S_WTX_ACK)
    {
        /* Restore old setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONTACT_CONFIG_BWT_CLK,
            dwOldBwtTimeout));

        /* reset multiplier */
        pDataParams->bBwtMultiplier = 1;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T0_SendBlockExpectedBytes(
                                                         phpalI7816p4_Sw_DataParams_t * pDataParams,
                                                         uint32_t dwNumExpectedBytes
                                                         )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t    PH_MEMLOC_REM nullByteRec = 0;
    uint8_t    PH_MEMLOC_REM dummy[1];
    uint8_t *  PH_MEMLOC_REM pptmpRxBuffer;
    uint16_t   PH_MEMLOC_REM wtmpRxBufferLen;
    uint32_t   PH_MEMLOC_REM dwOldNumExpectedBytes;
    phStatus_t PH_MEMLOC_REM status;

    /* Set number of expected bytes to speed up communication */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, &dwOldNumExpectedBytes));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, dwNumExpectedBytes));

    status = phhalHwContact_Exchange(
        pDataParams->pHalDataParams,
        PH_EXCHANGE_DEFAULT,
        pDataParams->pIntBuffer,
        pDataParams->wIntBufferLen,
        &pptmpRxBuffer,
        &wtmpRxBufferLen);

    if(wtmpRxBufferLen != 0)
    {
        /* Check for NULL Byte */
        while(pptmpRxBuffer[0] == PHPAL_I7816P4_SW_PROCEDURE_BYTE_NULL)
        {
            nullByteRec = 1;

            if (wtmpRxBufferLen > 1)
            {
                wtmpRxBufferLen--;
                memcpy(&pptmpRxBuffer[0], &pptmpRxBuffer[1], wtmpRxBufferLen);
            }
            else
            {
                /* Exchange nothing - just receive */
                status = phhalHwContact_Exchange(
                    pDataParams->pHalDataParams,
                    PH_EXCHANGE_DEFAULT,
                    dummy,
                    0,
                    &pptmpRxBuffer,
                    &wtmpRxBufferLen);
            }
        }

        if(wtmpRxBufferLen < PHPAL_I7816P4_SW_INT_BUFFER_SIZE)
        {
            memcpy(pDataParams->pIntBuffer, pptmpRxBuffer, wtmpRxBufferLen);
            pDataParams->wIntBufferLen = wtmpRxBufferLen;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_I7816P4);
        }

        if(nullByteRec)
        {
            if((dwNumExpectedBytes - wtmpRxBufferLen) > 0)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, dwNumExpectedBytes - wtmpRxBufferLen));

                /* Exchange nothing - just receive */
                status = phhalHwContact_Exchange(
                    pDataParams->pHalDataParams,
                    PH_EXCHANGE_DEFAULT,
                    dummy,
                    0,
                    &pptmpRxBuffer,
                    &wtmpRxBufferLen);

                if((pDataParams->wIntBufferLen + wtmpRxBufferLen) < PHPAL_I7816P4_SW_INT_BUFFER_SIZE)
                {
                    memcpy(&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], pptmpRxBuffer, wtmpRxBufferLen);
                    pDataParams->wIntBufferLen += wtmpRxBufferLen;
                }
                else
                {
                    return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_I7816P4);
                }
            }
            else if((dwNumExpectedBytes - wtmpRxBufferLen) < 0)
            {
                /* received more than expected */
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_I7816P4);
            }
        }
    }
    else
    {
        return status;
    }

    if(pDataParams->wIntBufferLen != dwNumExpectedBytes)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_I7816P4);
    }

    /* Reset number of expected bytes as it was before */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, dwOldNumExpectedBytes));

    return PH_ADD_COMPCODE(status, PH_COMP_PAL_I7816P4);
}

phStatus_t phpalI7816p4_Sw_Int_T0_ReceiveBytes(
                                               phpalI7816p4_Sw_DataParams_t * pDataParams,
                                               uint32_t dwNumExpectedBytes
                                               )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t    PH_MEMLOC_REM dummy[1] = {0x00};
    uint8_t *  PH_MEMLOC_REM pptmpRxBuffer;
    uint16_t   PH_MEMLOC_REM wtmpRxBufferLen;
    uint32_t   PH_MEMLOC_REM dwOldNumExpectedBytes;
    phStatus_t PH_MEMLOC_REM status;

    /* Set number of expected bytes to speed up communication */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, &dwOldNumExpectedBytes));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, dwNumExpectedBytes));

    /* Exchange nothing - just receive */
    status = phhalHwContact_Exchange(
        pDataParams->pHalDataParams,
        PH_EXCHANGE_DEFAULT,
        dummy,
        0,
        &pptmpRxBuffer,
        &wtmpRxBufferLen);

    if(wtmpRxBufferLen < PHPAL_I7816P4_SW_INT_BUFFER_SIZE)
    {
        memcpy(&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], pptmpRxBuffer, wtmpRxBufferLen);
        pDataParams->wIntBufferLen += wtmpRxBufferLen;
    }

    /* Reset number of expected bytes as it was before */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, dwOldNumExpectedBytes));

    return PH_ADD_COMPCODE(status, PH_COMP_PAL_I7816P4);
}

#endif /* NXPBUILD__PHPAL_I7816P4_SW */
