/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software ISO14443-4A Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHwContact.h>
#include <phpalI7816p4a.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_I7816P4A_SW

#include "phpalI7816p4a_Sw.h"
#include "phpalI7816p4a_Sw_Int.h"
#include <phToolsAtrParser.h>

phStatus_t phpalI7816p4a_Sw_Init(
                                 phpalI7816p4a_Sw_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 void * pHalDataParams
                                 )
{
    if (sizeof(phpalI7816p4a_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_I7816P4A);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalDataParams);

    /* init private data */
    pDataParams->wId            = PH_COMP_PAL_I7816P4A | PHPAL_I7816P4A_SW_ID;
    pDataParams->pHalDataParams = pHalDataParams;
    pDataParams->bIfsd          = PHPAL_I7816P4A_SW_FSDI_DEFAULT;
    pDataParams->bCid           = 0x00;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI7816p4a_Sw_IfsRequest(
                                       phpalI7816p4a_Sw_DataParams_t * pDataParams,
                                       uint8_t bIfsd
                                       )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    phStatus_t  PH_MEMLOC_REM status = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM cmd[4];
    uint8_t *   PH_MEMLOC_REM pResp;
    uint16_t    PH_MEMLOC_REM wRespLength;
    uint16_t    PH_MEMLOC_REM wRetransmitCount = 1;
    uint8_t     PH_MEMLOC_REM bAtrLength = 0;
    uint8_t     PH_MEMLOC_REM bProtocolSupported = 0;
    uint8_t     PH_MEMLOC_REM i = 0;
    uint8_t     PH_MEMLOC_REM counter = 0;
    uint32_t    PH_MEMLOC_REM bCurrentProtocol = 0;
    uint32_t    PH_MEMLOC_REM dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C/SPI communication */
    if (dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C || dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI || dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI_EXCHANGE)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_I7816P4A);
    }

    /* Without Activate the IFS Request should FAIL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetAtrLength(&bAtrLength));
    if(bAtrLength == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4A);
    }

    /* If T=1 is not supported IFS Request should FAIL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_CheckIfProtocolSupported(PHHAL_HW_CONTACT_PROTOCOLTYPE_T1, &bProtocolSupported));
    if(bProtocolSupported == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4A);
    }

    for(i = PHHAL_HW_CONTACT_PROTOCOLTYPE_T0; i <= PH_TOOLS_ATRPARSER_MAX_NUMBER_OF_PROTOCOLS; i++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_CheckIfProtocolSupported(i, &bProtocolSupported));
        if(bProtocolSupported)
        {
            counter++;
        }
    }

    if(counter > 1)
    {
        /* More than one protocol available so PPS has to be already sent which means current protocol should be T=1 */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, &bCurrentProtocol));
        if(bCurrentProtocol != PHHAL_HW_CONTACT_PROTOCOLTYPE_T1)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        }
    }

    /* The values '00' and 'FF' are rfu */
    if ((bIfsd > 0xFE) || (bIfsd < 0x01))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    /* Build IFS Request */
    cmd[0] = pDataParams->bCid;                     /* CID */
    cmd[1] = PHPAL_I7816P4A_SW_IFS_REQUEST_CMD;     /* PCB Byte - C1 */
    cmd[2] = 0x01;                                  /* Length Field */
    cmd[3] = bIfsd;                                 /* Applied Fsdi Value */

    /* If error one retransmit is allowed */
    while(wRetransmitCount != 0)
    {
        /* Perform actual Exchange */
        status = phhalHwContact_Exchange(
            pDataParams->pHalDataParams,
            PH_EXCHANGE_DEFAULT,
            cmd,
            PHPAL_I7816P4A_SW_T1PROLOGUE_LENGTH + 1,
            &pResp,
            &wRespLength);

        /*When S( request) was transmitted and the received response is not S( response) or a BWT
          time-out occurs (only with the interface device), S( request) is retransmitted.*/
        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Check Response - s Block with length 1 */
            if (wRespLength != PHPAL_I7816P4A_SW_T1PROLOGUE_LENGTH + 1 ||
                pResp[0] != pDataParams->bCid ||
                pResp[1] != PHPAL_I7816P4A_SW_IFS_RESPONSE_CMD ||
                pResp[2] != cmd[PHPAL_I7816P4A_SW_T1LENGTHFIELD_OFFSET] ||
                pResp[3] != bIfsd)
            {
                status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4A);
            }
            else
            {
                /* Command successful so save in DataParams */
                pDataParams->bIfsd = bIfsd;
                break;
            }
        }

        wRetransmitCount--;
    }

    return status;
}

phStatus_t phpalI7816p4a_Sw_Pps(
                                phpalI7816p4a_Sw_DataParams_t * pDataParams,
                                uint8_t bProtocolType,
                                uint8_t bFValue,
                                uint8_t bDValue
                                )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM cmd[3];
    uint8_t *   PH_MEMLOC_REM pResp;
    uint16_t    PH_MEMLOC_REM wRespLength;
    uint16_t    PH_MEMLOC_REM wPpsxCounter;
    uint8_t     PH_MEMLOC_REM i;
    uint8_t     PH_MEMLOC_REM bFdValue;
    uint8_t     PH_MEMLOC_REM bProtocolTypeNumber;
    uint32_t    PH_MEMLOC_REM dwOldWt;
    uint32_t    PH_MEMLOC_REM dwNewWt;
    uint32_t    PH_MEMLOC_REM dwCurrentFd;
    uint16_t    PH_MEMLOC_REM wCurrentF;
    uint16_t    PH_MEMLOC_REM wCurrentD;
    uint32_t    PH_MEMLOC_REM dwOldNumExpectedBytes;
    uint32_t    PH_MEMLOC_REM dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C/SPI communication */
    if (dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C || dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI || dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI_EXCHANGE)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_I7816P4A);
    }

    /* To be compatible with HAL CONFIG definitions (T=0 -> 1) */
    bProtocolTypeNumber = bProtocolType - 1;

    /* Parameter Check - Protocol Type 0-14 (in our case should be t=0 or t=1) */
    if(bProtocolTypeNumber < 0x00 || bProtocolTypeNumber > 0x0E)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    if(bFValue > 0xF || bDValue > 0xF)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    bFdValue = (bFValue << 4) | bDValue;

    /* Enable LRC as this works as PCK */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, PHHAL_HW_CONTACT_MODE_LRC));

    /* For PPS WT=9600etu is used */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_WT_CLK, &dwOldWt));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_FD, &dwCurrentFd));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues((uint8_t)dwCurrentFd, &wCurrentF, &wCurrentD));
    dwNewWt = wCurrentF / wCurrentD * 9600;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_WT_CLK, dwNewWt));


    /* Build PPS command */
    cmd[0] = PHPAL_I7816P4A_SW_PPSS;                            /* 0xFF PPS identifier */
    cmd[1] = PHPAL_I7816P4A_SW_PPS0 | bProtocolTypeNumber;      /* Bit8 RFU=0 / presence of PPS1-2-3 (only PPS1) / Protocol */
    cmd[2] = bFdValue;                                          /* 4 bit FValue / 4 Bit DValue for Datarates */

    /* Set number of expected bytes to speed up communication */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, &dwOldNumExpectedBytes));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, PHPAL_I7816P4A_SW_PPS_LENGTH + 1));

    /* Perform actual Exchange */
    status = phhalHwContact_Exchange(
        pDataParams->pHalDataParams,
        PH_EXCHANGE_DEFAULT,
        cmd,
        PHPAL_I7816P4A_SW_PPS_LENGTH,
        &pResp,
        &wRespLength);

    /* Reset number of expected bytes as it was before */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, dwOldNumExpectedBytes));

    /* Error Handling */
    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Check Response */
        if(wRespLength == PHPAL_I7816P4A_SW_PPS_LENGTH)
        {
            if(strncmp((const char*)&cmd[0], (const char*)pResp, PHPAL_I7816P4A_SW_PPS_LENGTH))
            {
                /* Response does not equal Request but lenght matches - ERROR */
                status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4A);
            }

            /* Card has aknowledged new FD so apply */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_FD, bFdValue));
        }
        else if(wRespLength >= 2)
        {
            if(*pResp != PHPAL_I7816P4A_SW_PPSS)
            {
                /* PPS identifier hast to match otherwise - ERROR */
                status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4A);
            }
            else
            {
                /* PPS0 Bits 1 to 4 shall be identical to request */
                if((*(pResp + 1) & 0x1E) != (cmd[1] & 0x1E))
                {
                    status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4A);
                }

                /* Check PPS0 and the availability of PPS1/2/3 */
                wPpsxCounter = 0;
                for(i = 1; i <= 3; i++)
                {
                    if((*(pResp + 1) & (0x10 << i)) != 0)
                    {
                        wPpsxCounter++;
                    }
                }

                if(wPpsxCounter != (wRespLength - 2))
                {
                    /* PPS1/2/3 declared but not available - ERROR */
                    status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4A);
                }
            }
        }
        else
        {
            /* PPS Response smaller than 2 Bytes - ERROR */
            status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4A);
        }
    }

    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Apply ProtocolType */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams->pHalDataParams, 0, 0, bProtocolType));
    }
    else
    {
        /* Reset WT */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_WT_CLK, dwOldWt));
    }

    return status;
}

phStatus_t phpalI7816p4a_Sw_ActivateCard(
                                         phpalI7816p4a_Sw_DataParams_t * pDataParams,
                                         uint8_t bProtocolType,
                                         uint8_t bIfsd,
                                         uint8_t bFValue,
                                         uint8_t bDValue,
                                         uint8_t * pAtr,
                                         uint16_t * pwATRLength
                                         )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM pAtrBuffer[PHHAL_HW_CONTACT_MAX_ATR_SIZE];
    uint16_t    PH_MEMLOC_REM wAtrLength;
    uint32_t    PH_MEMLOC_REM dwCommunicationChannel = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Not supported for I2C/SPI communication */
    if (dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C || dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI || dwCommunicationChannel == PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI_EXCHANGE)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_I7816P4A);
    }

    /* Activate */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Activate(pDataParams->pHalDataParams, &pAtrBuffer[0], &wAtrLength));

    /* Perform PPS */
    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4a_Sw_Pps(pDataParams, bProtocolType, bFValue, bDValue));

    if(bProtocolType == PHHAL_HW_CONTACT_PROTOCOLTYPE_T1 && bIfsd != 0)
    {
        /* Perform IFS Request */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4a_Sw_IfsRequest(pDataParams, bIfsd));
    }

    memcpy(pAtr, &pAtrBuffer[0], wAtrLength);
    *pwATRLength = wAtrLength;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI7816p4a_Sw_GetConfig(
                                      phpalI7816p4a_Sw_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t * pValue
                                      )
{
    phStatus_t statusTmp;
    uint8_t bConfig;

    switch (wConfig)
    {
    case PHPAL_I7816P4A_CONFIG_IFSC:
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_IFSC(&bConfig));
        *pValue = (uint16_t)bConfig;
        break;

    case PHPAL_I7816P4A_CONFIG_IFSD:
        *pValue = (uint16_t)pDataParams->bIfsd;
        break;

    default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_I7816P4A);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4A);
}

#endif /* NXPBUILD__PHPAL_I7816P4A_SW */
