/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software MIFARE(R) Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#ifdef NXPBUILD__PHPAL_I7816P4
#include <phpalI7816p4.h>
#endif /* NXPBUILD__PHPAL_I7816P4 */
#include <ph_RefDefs.h>
#include <phTools.h>

#include "phpalI14443p4.h"

#ifdef NXPBUILD__PHPAL_MIFARE_SW_CONTACT

#include "phpalMifare_SwContact.h"
#include "../phpalMifare_Int.h"

phStatus_t phpalMifare_SwContact_Init(
                               phpalMifare_SwContact_DataParams_t * pDataParams,
                               uint16_t wSizeOfDataParams,
                               void * pHalContactDataParams,
                               void * pPalI7816p4DataParams,
                               uint8_t * pRxBuffer,
                               uint16_t  wRxBufSize
                               )
{
    if (sizeof(phpalMifare_SwContact_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_MIFARE);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalContactDataParams);
    PH_ASSERT_NULL(pRxBuffer);

    /* init private data */
    pDataParams->wId                    = PH_COMP_PAL_MIFARE | PHPAL_MIFARE_SW_CONTACT_ID;
    pDataParams->pHalContactDataParams         = pHalContactDataParams;
    pDataParams->pPalI7816p4DataParams = pPalI7816p4DataParams;
    pDataParams->pRxBuffer = pRxBuffer;
    pDataParams->wRxBufSize = wRxBufSize;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_SwContact_ExchangeL3(
                                     phpalMifare_SwContact_DataParams_t * pDataParams,
                                     uint16_t wOption,
                                     uint8_t * pTxBuffer,
                                     uint16_t wTxLength,
                                     uint8_t ** ppRxBuffer,
                                     uint16_t * pRxLength
                                     )
{
    /* satisfy compiler */
    if (pDataParams || wOption || pTxBuffer || wTxLength || ppRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_SwContact_ExchangeL4(
                                     phpalMifare_SwContact_DataParams_t * pDataParams,
                                     uint16_t wOption,
                                     uint8_t * pTxBuffer,
                                     uint16_t wTxLength,
                                     uint8_t ** ppRxBuffer,
                                     uint16_t * pRxLength
                                     )
{
#ifdef NXPBUILD__PHPAL_I7816P4
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint32_t dwRxStartPos;
    uint8_t *halRxBuffer;
    uint16_t halRxLength;

    /* Not available if no Layer4 has been set */
    if (pDataParams->pPalI7816p4DataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
    }

    /* If PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS is set this needs to be handeled within the PAL because it is also used within 7816 exchange for readering the length byte of the response date for I2C and SPI */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalContactDataParams, PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS, &dwRxStartPos));

    /* Reset offset */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalContactDataParams, PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS, 0));

    /* Perform Exchange */
    PH_CHECK_SUCCESS_FCT(statusTmp, phpalI7816p4_Exchange(
        pDataParams->pPalI7816p4DataParams,
        wOption,
        pTxBuffer,
        wTxLength,
        &halRxBuffer,
        &halRxLength));

    if (!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        /* Copy date to PAL Rx buffer and adjust rx length*/
        memcpy(pDataParams->pRxBuffer + dwRxStartPos, halRxBuffer, halRxLength);
        if (ppRxBuffer != NULL)
        {
            *ppRxBuffer = pDataParams->pRxBuffer;
        }
        if (pRxLength != NULL)
        {
            *pRxLength = (uint16_t)(halRxLength + dwRxStartPos);
        }
    }
    if (!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        /* In case of Tx Chaining it should return the ACK, difference between CB and CL is that CB also has the NAD upfront. so remove this byte and convert CB ack to CL ack */
        if ((wOption & PH_EXCHANGE_TXCHAINING) == PH_EXCHANGE_TXCHAINING)
        {
            if (ppRxBuffer != NULL && *ppRxBuffer != NULL)
            {
                *ppRxBuffer = *ppRxBuffer + 1;

                if ((*ppRxBuffer[0] & 0xEF) == 0x80) /*If error free contact ack -> convert to contactless ack */
                {
                    *ppRxBuffer[0] = 0xA0;
                }
            }
            if (pRxLength != NULL && *pRxLength != 0)
            {
                *pRxLength = *pRxLength - 1;
            }
        }
    }


    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_MIFARE);
#else
    /* satisfy compiler */
    if (pDataParams || wOption || pTxBuffer || wTxLength || ppRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
#endif /* NXPBUILD__PHPAL_I7816P4 */
}

phStatus_t phpalMifare_SwContact_ExchangePc(
                                     phpalMifare_SwContact_DataParams_t * pDataParams,
                                     uint16_t wOption,
                                     uint8_t * pTxBuffer,
                                     uint16_t wTxLength,
                                     uint8_t ** ppRxBuffer,
                                     uint16_t * pRxLength
                                     )
{
    /* satisfy compiler */
    if (pDataParams || wOption || pTxBuffer || wTxLength || ppRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_SwContact_ExchangeRaw(
                                      phpalMifare_SwContact_DataParams_t * pDataParams,
                                      uint16_t wOption,
                                      uint8_t * pTxBuffer,
                                      uint16_t wTxLength,
                                      uint8_t bTxLastBits,
                                      uint8_t ** ppRxBuffer,
                                      uint16_t * pRxLength,
                                      uint8_t * pRxLastBits
                                      )
{
    /* satisfy compiler */
    if (pDataParams || wOption || pTxBuffer || wTxLength || bTxLastBits || ppRxBuffer || pRxLength || pRxLastBits);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_SwContact_MfcAuthenticateKeyNo(
    phpalMifare_SwContact_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint16_t wKeyNo,
    uint16_t wKeyVersion,
    uint8_t * pUid
    )
{
    /* satisfy compiler */
    if (pDataParams || bBlockNo || bKeyType || wKeyNo || wKeyVersion || pUid);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_SwContact_MfcAuthenticate(
    phpalMifare_SwContact_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint8_t * pKey,
    uint8_t * pUid
    )
{
    /* satisfy compiler */
    if (pDataParams || bBlockNo || bKeyType || pKey || pUid);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_SwContact_GetConfig(
                                    phpalMifare_SwContact_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t * pValue
                                    )
{
    /* satisfy compiler */
    if (pDataParams || wConfig || pValue);

    switch(wConfig)
    {
    case PHPAL_I14443P4_CONFIG_FSI:
        *pValue = 0x0707; /* Supported 128 bytes */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_MIFARE);
    }

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
}
#endif /* NXPBUILD__PHPAL_MIFARE_SW_CONTACT */
