/*
 * Copyright 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal definitions for Software ISO15693 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>

#ifdef  NXPBUILD__PHPAL_SLI15693_SW

#include <phhalHw.h>
#include <phpalSli15693.h>
#include <ph_RefDefs.h>

#include "phpalSli15693_Sw_Int.h"

/**
 * Perform all kinds of Inventory commands in normal and extended mode. The parameter bEntended defines whether the command should be
 * used in framed in normal or in extended mode.
 *
 * Input Parameters:
 *      pDataParams         : Pointer to this layer's parameter structure.
 *      bExtended           : Whether the command should be framed in normal or in extended mode. Also same for response frame.
 *      bCmd                : The command code to be framed or performed.
 *                              Inventory
 *                              InventoryRead (Extended)
 *                              FastInventoryRead (Extended)
 *                              InventoryPageRead
 *                              FastInventoryPageRead
 *      bFlags              : Request flags byte.
 *      bAfi                : Application Family Identifier.
 *      pMask               : UID mask, holding known UID bits.
 *      bMaskBitLen         : Number of UID bits within pMask.
 *      bExtendedOptions    : Features of the extended mode.
 *      pCID                : Two byte CID -> if marked in extended options.
 *      bPage_Block_No      : Block number or block number of that page to read.
 *      wNoOfPages_Blocks   : Number of blocks or pages to read.
 *
 * Output Parameters:
 *      pCIDOut             : Received CID
 *      pUid                : Received UID
 *      pUidLen             : Number of received UID bytes.
 *      pData               : Received data will have page information or blocks information or DSFID information.
 *      pDataLen            : Length of bytes available in Data buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid response received.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phpalSli15693_Sw_Int_Inventory(phpalSli15693_Sw_DataParams_t* pDataParams, uint8_t bExtended, uint8_t bCmd, uint8_t bFlags, uint8_t bAfi,
    uint8_t * pMask, uint8_t bMaskBitLen, uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bPage_Block_No, uint16_t wNoOfPages_Blocks, uint8_t * pCIDOut,
    uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[15];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bCmdOffset = 0;
    uint8_t     PH_MEMLOC_REM bMaskByteLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM bStoredUidLen = 0;
    uint16_t    PH_MEMLOC_REM wAsk = 0;
    uint16_t    PH_MEMLOC_REM wTimeout = 0;
    uint16_t    PH_MEMLOC_REM wCurrBlocksToRead = 0;
    uint16_t    PH_MEMLOC_REM wCurrBlockNo = 0;
    uint16_t    PH_MEMLOC_REM wMaxNoBlocks = 0;
    uint8_t     PH_MEMLOC_REM bAllBlocksRead = 0;
    uint8_t     PH_MEMLOC_REM bFirst = 0;

    /* Verify the parameters. */
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);
    if(0U != (bExtendedOptions & PHPAL_SLI15693_EXTENDED_FLAGS_CID_COMPARE))
    {
        PH_ASSERT_NULL_PARAM(pData, PH_COMP_PAL_SLI15693);
        PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_PAL_SLI15693);
    }
    if(0U != (bExtendedOptions & PHPAL_SLI15693_EXTENDED_FLAGS_CID_COMPARE)) PH_ASSERT_NULL_PARAM(pCID, PH_COMP_PAL_SLI15693);
    if(0U != (bExtendedOptions & PHPAL_SLI15693_EXTENDED_FLAGS_CID_RESPONSE)) PH_ASSERT_NULL_PARAM(pCIDOut, PH_COMP_PAL_SLI15693);

    /* First of all we check the bMaskBitLen according to ISO/IEC15693, 8.1, assuming the inventory flag to be set,
     * b6 defines the number of slots.
     * b6 = 1: --> one slot --> mask length = [0 .. 64] bits
     * b6 = 0: --> 16 slots --> mask length = [0 .. 60] bits
     */
    if(((0U != ((bFlags & PHPAL_SLI15693_FLAG_NBSLOTS))) && (bMaskBitLen > 64U)) ||
       ((0U == ((bFlags & PHPAL_SLI15693_FLAG_NBSLOTS))) && (bMaskBitLen > 60U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
    }

    /* Number of Pages/Blocks check */
    if(bCmd != PHPAL_SLI15693_SW_CMD_INVENTORY)
    {
        /* Number of Pages/Blocks can't be zero */
        if(wNoOfPages_Blocks == 0U)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
        }

        /* Check if the block number exceeds the limit. */
        if(((uint16_t) bPage_Block_No + wNoOfPages_Blocks) > PHPAL_SLI15693_SW_MAX_BLOCKS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
        }
    }

    /* Reset UID and data length. */
    pDataParams->bUidBitLen = 0;
    *pUidLen = 0;
    *pDataLen = 0;

    /* Apply flag settings. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams,
        PHPAL_SLI15693_CONFIG_FLAGS,
        bFlags));

    /* Overwrite DataRate for Fast Inventory (Page) Read */
    if((bCmd == PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_READ) || (bCmd == PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_PAGE_READ))
    {
        /* Set high DataRate if HAL is not already configured for it */
        if(0U != (bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_RXDATARATE,
                PHHAL_HW_RF_RX_DATARATE_FAST_HIGH));
        }

        /* Set low DataRate if HAL is not already configured for it */
        else
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_RXDATARATE,
                PHHAL_HW_RF_RX_DATARATE_FAST_LOW));
        }
    }

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Reset command buffer and its length variable. */
    bCmdLen = 0;
    (void) memset(aCmdBuff, 0x00, sizeof(aCmdBuff));

    /* Add command code to command buffer. */
    aCmdBuff[bCmdLen++] = bCmd;

    /* Add AFI to command frame. */
    if((0U != ((bFlags & PHPAL_SLI15693_FLAG_AFI))) && (0U != ((bFlags & PHPAL_SLI15693_FLAG_INVENTORY))))
    {
        aCmdBuff[bCmdLen++] = bAfi;
    }

    /* Making the MSB of mask length byte as 1 to show extended mode and adding the extended information to command buffer. */
    if(0U != bExtended)
    {
        aCmdBuff[bCmdLen] = (uint8_t) (bMaskBitLen | PHPAL_SLI15693_SW_FLAG_INVENTORY_READ_EXTENSION);

        /* In case of extended mode add the extended option in command */
        aCmdBuff[bCmdLen++] = bExtendedOptions;

        /* Add the CID to command buffer. */
        if(0U != (bExtendedOptions & PHPAL_SLI15693_EXTENDED_FLAGS_CID_COMPARE))
        {
            aCmdBuff[bCmdLen++] = pCID[0];
            aCmdBuff[bCmdLen++] = pCID[1];
        }
    }
    else
    {
        aCmdBuff[bCmdLen++] = bMaskBitLen;
    }

    /* Retrieve mask byte length. */
    bMaskByteLen = ((bMaskBitLen % 8U) != 0U) ? ((bMaskBitLen >> 3U) + 1U) : (bMaskBitLen >> 3U);

    /* Prepare mask in DataParams. */
    (void) memcpy(pDataParams->pUid, pMask, bMaskByteLen);
    pDataParams->bUidBitLen = bMaskBitLen;

    /* Prepare mask to send. */
    (void) memcpy(&aCmdBuff[bCmdLen], pMask, bMaskByteLen);
    bCmdLen = (uint8_t) (bCmdLen + bMaskByteLen);

    /* Mask out invalid bits. */
    if(0U != (bMaskBitLen & 0x07U))
    {
        aCmdBuff[bCmdLen - 1U] &= (uint8_t) (0xFFU >> (8U - (bMaskBitLen & 0x07U)));
    }

    /* Get the ASK 100 Condition */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_ASK100,
        &wAsk));

    /* Calculate Timeout based on ask and baud rate */
    if(wAsk != 0U)
    {
        if((bCmd == PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_READ) ||
           (bCmd == PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_PAGE_READ))
        {
            if(0U != (bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_FASTHIGH_SOF_US;
            }
            else
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_FASTLOW_SOF_US;
            }
        }
        else
        {
            if(0U != (bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_HIGH_SOF_US;
            }
            else
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_LOW_SOF_US;
            }
        }
    }
    else
    {
        if((bCmd == PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_READ) ||
           (bCmd == PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_PAGE_READ))
        {
            if(0U != (bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_FASTHIGH_NRT_US;
            }
            else
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_FASTLOW_NRT_US;
            }
        }
        else
        {
            if(0U != (bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_HIGH_NRT_US;
            }
            else
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_LOW_NRT_US;
            }
        }
    }

    /* Set  timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        wTimeout));

    /* Update the command offset variable. */
    bCmdOffset = bCmdLen;

    /* If buffering is set. */
    if((pDataParams->bBuffering) != 0U)
    {
        /* Update the maximum number of blocks with respect to Option flag setting. The value for the blocks is fixed to 60 and 40 to avoid multiple
         * handling of different data in response. RD70x can respond with more amount of data but CM1 cannot. So fixing the blocks count to a lower
         * value.
         */
        wMaxNoBlocks = (uint8_t) (((bFlags & PHPAL_SLI15693_FLAG_OPTION) != 0U) ? 40U : 60U);

        /* Blocks to read. */
        wCurrBlocksToRead = wMaxNoBlocks;

        /* Update the number of blocks to read if its less than the internal required one. */
        if(wNoOfPages_Blocks < wMaxNoBlocks)
        {
            wCurrBlocksToRead = wNoOfPages_Blocks;
            bAllBlocksRead = 1U;
        }
    }
    else
    {
        wCurrBlocksToRead = wNoOfPages_Blocks;
        bAllBlocksRead = 1U;
    }

    if((((pDataParams->bFlags & PHPAL_SLI15693_FLAG_INVENTORY) == 0U) && ((pDataParams->bFlags & PHPAL_SLI15693_FLAG_ADDRESSED) != 0U)) != 0U)
    {
        pDataParams->bExplicitlyAddressed = 1U;
    }

    /* Set First variable. This variable will be used to validate the response only once in case if chaining is enabled. */
    bFirst = PH_ON;

    /* Exchange the information. */
    do
    {
        /* Page (or block) number and count */
        if(bCmd != PHPAL_SLI15693_SW_CMD_INVENTORY)
        {
            /* Adjust number of blocks. Adjustment is made because the User or the application will pass
             * the number of blocks starting from 1 to N. But as per Iso15693 specification the number
             * of blocks ranges from 0 - (N - 1).
             */
            --wCurrBlocksToRead;

            if(!((bExtended != 0U) && ((bExtendedOptions & PHPAL_SLI15693_EXTENDED_FLAGS_SKIP_DATA) != 0U)))
            {
                aCmdBuff[bCmdOffset++] = (uint8_t) ((wCurrBlockNo + bPage_Block_No) & 0x00FFU);
                aCmdBuff[bCmdOffset++] = (uint8_t) wCurrBlocksToRead;
            }
        }

        /* Exchange the command information to Hal layer. */
        wStatus = phpalSli15693_Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            aCmdBuff,
            bCmdOffset,
            &pResponse,
            &wRespLen);

        /* Clear INVENTORY, AFI and NBSLOTS flag */
        if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            pDataParams->bFlags &= (uint8_t) ~(uint8_t) (PHPAL_SLI15693_FLAG_INVENTORY | PHPAL_SLI15693_FLAG_AFI | PHPAL_SLI15693_FLAG_NBSLOTS);
        }

        /* Verify the exchange status. */
        PH_CHECK_SUCCESS(wStatus);

        /* Extracting  the CID as sent by the VICC. */
        if((bExtended != 0U) && ((bExtendedOptions & PHPAL_SLI15693_EXTENDED_FLAGS_CID_RESPONSE) != 0U))
        {
            (void) memcpy(pCIDOut, pResponse, 2U);

            /* Update the response pointer address and length to Skip CID. */
            pResponse += 2U;
            wRespLen -= 2U;
        }

        /* Set the length (amount) of expected UID bytes. */
        if(bCmd == PHPAL_SLI15693_SW_CMD_INVENTORY)
        {
            /* The response length should be exactly the complete UID */
            if(wRespLen != (1U + PHPAL_SLI15693_UID_LENGTH))
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
            }

            /* Copy the DSFID to the data buffer. */
            pData[0] = pResponse[0];
            *pDataLen = 1U;

            /* Store the UID */
            (void) memcpy(pDataParams->pUid, &pResponse[1], PHPAL_SLI15693_UID_LENGTH);
            pDataParams->bUidBitLen = PHPAL_SLI15693_SW_UID_COMPLETE;

            /* Return the UID */
            (void) memcpy(pUid, pDataParams->pUid, PHPAL_SLI15693_UID_LENGTH);
            *pUidLen = PHPAL_SLI15693_UID_LENGTH;
        }
        else
        {
            if((bFlags & PHPAL_SLI15693_FLAG_OPTION) != 0U)
            {
                *pUidLen = PHPAL_SLI15693_UID_LENGTH - bStoredUidLen;

                /* The response length should not be less than the remaining UID. */
                if(bFirst != 0U)
                {
                    bStoredUidLen = pDataParams->bUidBitLen >> 3U;
                    *pUidLen = PHPAL_SLI15693_UID_LENGTH - bStoredUidLen;

                    /* Update the UID length if extended. */
                    if((bExtended != 0U) && ((bExtendedOptions & PHPAL_SLI15693_EXTENDED_FLAGS_UID_MODE) != 0U))
                    {
                        *pUidLen = PHPAL_SLI15693_UID_LENGTH;
                    }
                }

                /* Validate the response length against UID length. */
                if(wRespLen < (*pUidLen))
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
                }

                if(bStoredUidLen < PHPAL_SLI15693_UID_LENGTH)
                {
                    if((*pUidLen) == 0U)
                    {
                        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
                    }

                    /* Return the received (partial) UID */
                    (void) memcpy(pUid, pResponse, *pUidLen);

                    /* We need to merge the contents of the mask buffer and the received data */
                    if((bMaskBitLen % 8U) != 0U)
                    {
                        if(bStoredUidLen < 7U)
                        {
                            /* copy the UID bytes we received from the card */
                            (void) memcpy(&(pDataParams->pUid[bStoredUidLen + 1]), &pResponse[1], (*pUidLen) - 1);
                        }

                        /* merge mask-bits with received bits */
                        pDataParams->pUid[bStoredUidLen] |= pResponse[0];
                    }
                    else
                    {
                        /* Copy the UID bytes we received from the card */
                        (void) memcpy(&(pDataParams->pUid[bStoredUidLen]), pResponse, *pUidLen);
                    }
                }

                /* Update UID length. */
                pDataParams->bUidBitLen = PHPAL_SLI15693_SW_UID_COMPLETE;

                /* Shift pointer and length */
                pResponse += *pUidLen;
                wRespLen = (uint16_t) (wRespLen - *pUidLen);
            }

            /* Copy the received data to internal buffer. */
            (void) memcpy(&pData[*pDataLen], pResponse, wRespLen);
            *pDataLen += wRespLen;
        }

        /* Update the variables to read the remaining data. */
        wCurrBlockNo += wMaxNoBlocks;

        /* Update the Current blocks to read. */
        wCurrBlocksToRead = wMaxNoBlocks;

        /* Reset the command buffer offset. */
        bCmdOffset = bCmdLen;

        /* Set the remaining blocks to read. */
        if((wNoOfPages_Blocks - wCurrBlockNo) < wMaxNoBlocks)
        {
            wCurrBlocksToRead = (uint16_t) (wNoOfPages_Blocks - wCurrBlockNo);
        }

        /* Set the flag to finish the loop. */
        if((wNoOfPages_Blocks * 4) == *pDataLen)
        {
            bAllBlocksRead = 1U;
        }

        /* Set First variable. This variable will be used to validate the response only once in case if chaining is enabled. */
        bFirst = PH_OFF;

    } while(bAllBlocksRead == 0U);

    /* Clear INVENTORY, AFI and NBSLOTS flag */
    pDataParams->bFlags &= (uint8_t) ~(uint8_t) (PHPAL_SLI15693_FLAG_INVENTORY | PHPAL_SLI15693_FLAG_AFI | PHPAL_SLI15693_FLAG_NBSLOTS);

    pDataParams->bExplicitlyAddressed = 0;

    /* Error check */
    PH_CHECK_SUCCESS(wStatus);

    /* Set addressed flag. */
    pDataParams->bFlags |= PHPAL_SLI15693_FLAG_ADDRESSED;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

#endif /* NXPBUILD__PHPAL_SLI15693_SW */
