/*
 * Copyright 2013, 2015, 2017, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Sli15693 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phpalSli15693.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_SLI15693_SW
#include "Sw/phpalSli15693_Sw.h"
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
#include "Rd710/phpalSli15693_Rd710.h"
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

#ifdef NXPBUILD__PHPAL_SLI15693

/* ISO15693 Mandatory commands ------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_Inventory(void * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask, uint8_t bMaskBitLen,
    uint8_t * pDsfid, uint8_t * pUid)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bMaskByteLen;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_Inventory");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFlags);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAfi);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMask);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMaskBitLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDsfid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(0U != bMaskBitLen) PH_ASSERT_NULL_PARAM(pMask, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bFlags), &bFlags);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bAfi), &bAfi);

    /* Compute Mask buffer length */
    bMaskByteLen = bMaskBitLen >> 3U;

    /* Check for incomplete bytes */
    if(bMaskBitLen & 0x07U)
    {
        ++bMaskByteLen;
    }
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMask), pMask, bMaskByteLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bMaskBitLen), &bMaskBitLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_Inventory((phpalSli15693_Sw_DataParams_t *) pDataParams, bFlags, bAfi, pMask, bMaskBitLen,
                pDsfid, pUid);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_Inventory((phpalSli15693_Rd710_DataParams_t *) pDataParams, bFlags, bAfi, pMask, bMaskBitLen,
                pDsfid, pUid);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDsfid), pDsfid, 1);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, 8);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_StayQuiet(void * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_StayQuiet");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_StayQuiet((phpalSli15693_Sw_DataParams_t *) pDataParams);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_StayQuiet((phpalSli15693_Rd710_DataParams_t *) pDataParams);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}





/* ISO15693 Optional commands -------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_Select(void * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_Select");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_Select((phpalSli15693_Sw_DataParams_t*) pDataParams);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_Select((phpalSli15693_Rd710_DataParams_t*) pDataParams);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_ResetToReady(void * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_ResetToReady");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_ResetToReady((phpalSli15693_Sw_DataParams_t*) pDataParams);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_ResetToReady((phpalSli15693_Rd710_DataParams_t*) pDataParams);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}





/* ISO15693 Custom commands ---------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_InventoryRead(void * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask, uint8_t bMaskBitLen,
    uint8_t bBlockNo, uint16_t wNoOfBlocks, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bMaskByteLen = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_InventoryRead");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFlags);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAfi);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMask);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMaskBitLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBlockNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNoOfBlocks);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(0U != bMaskBitLen) PH_ASSERT_NULL_PARAM(pMask, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bFlags), &bFlags);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bAfi), &bAfi);

    /* Compute Mask buffer length */
    bMaskByteLen = bMaskBitLen >> 3U;

    /* Check for incomplete bytes */
    if(bMaskBitLen & 0x07U)
    {
        ++bMaskByteLen;
    }
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMask), &pMask, bMaskByteLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bMaskBitLen), &bMaskBitLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bBlockNo), &bBlockNo);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wNoOfBlocks), &wNoOfBlocks);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Perform operation on active layer */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_InventoryRead((phpalSli15693_Sw_DataParams_t *) pDataParams, bFlags, bAfi, pMask,
                bMaskBitLen, bBlockNo, wNoOfBlocks, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_InventoryRead((phpalSli15693_Rd710_DataParams_t *) pDataParams, bFlags, bAfi, pMask,
                bMaskBitLen, bBlockNo, wNoOfBlocks, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, *pUidLen);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUidLen), pUidLen);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pData), pData, *pDataLen);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDataLen), pDataLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_InventoryReadExtended(void * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask, uint8_t bMaskBitLen,
    uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bBlockNo, uint16_t wNoOfBlocks, uint8_t * pCDIDOut, uint8_t * pUid,
    uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bMaskByteLen = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_InventoryExtended");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFlags);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAfi);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMask);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMaskBitLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bExtendedOptions);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pCID);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBlockNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNoOfBlocks);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pCDIDOut);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(bMaskBitLen != 0x00U) PH_ASSERT_NULL_PARAM(pMask, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bFlags), &bFlags);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bAfi), &bAfi);

    /* Compute Mask buffer length */
    bMaskByteLen = bMaskBitLen >> 3U;

    /* Check for incomplete bytes */
    if(bMaskBitLen & 0x07U)
    {
        ++bMaskByteLen;
    }
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMask), &pMask, bMaskByteLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bMaskBitLen), &bMaskBitLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bExtendedOptions), &bExtendedOptions);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pCID), pCID, 2);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bBlockNo), &bBlockNo);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wNoOfBlocks), &wNoOfBlocks);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_InventoryReadExtended((phpalSli15693_Sw_DataParams_t *) pDataParams, bFlags, bAfi, pMask,
                bMaskBitLen, bExtendedOptions, pCID, bBlockNo, wNoOfBlocks, pCDIDOut, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_InventoryReadExtended((phpalSli15693_Rd710_DataParams_t *) pDataParams, bFlags, bAfi,
                pMask, bMaskBitLen, bExtendedOptions, pCID, bBlockNo, wNoOfBlocks, pCDIDOut, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        if(pCDIDOut != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pCDIDOut), pCDIDOut, 2);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, *pUidLen);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUidLen), pUidLen);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pData), pData, *pDataLen);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDataLen), pDataLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_FastInventoryRead(void * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask, uint8_t bMaskBitLen,
    uint8_t bBlockNo, uint16_t wNoOfBlocks, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bMaskByteLen = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_FastInventoryRead");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFlags);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAfi);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMask);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBlockNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMaskBitLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNoOfBlocks);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(0U != bMaskBitLen) PH_ASSERT_NULL_PARAM(pMask, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bFlags), &bFlags);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bAfi), &bAfi);

    /* Compute Mask buffer length */
    bMaskByteLen = bMaskBitLen >> 3U;

    /* Check for incomplete bytes */
    if(bMaskBitLen & 0x07U)
    {
        ++bMaskByteLen;
    }
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMask), &pMask, bMaskByteLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bMaskBitLen), &bMaskBitLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bBlockNo), &bBlockNo);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wNoOfBlocks), &wNoOfBlocks);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_FastInventoryRead((phpalSli15693_Sw_DataParams_t *) pDataParams, bFlags, bAfi, pMask,
                bMaskBitLen, bBlockNo, wNoOfBlocks, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_FastInventoryRead((phpalSli15693_Rd710_DataParams_t *) pDataParams, bFlags, bAfi,
                pMask, bMaskBitLen, bBlockNo, wNoOfBlocks, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, *pUidLen);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUidLen), pUidLen);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pData), pData, *pDataLen);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDataLen), pDataLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_FastInventoryReadExtended(void * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask, uint8_t bMaskBitLen,
    uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bBlockNo, uint16_t wNoOfBlocks, uint8_t * pCDIDOut, uint8_t * pUid,
    uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bMaskByteLen = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_FastInventoryExtended");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFlags);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAfi);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMask);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMaskBitLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bExtendedOptions);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pCID);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBlockNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNoOfBlocks);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pCDIDOut);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(bMaskBitLen != 0x00U) PH_ASSERT_NULL_PARAM(pMask, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bFlags), &bFlags);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bAfi), &bAfi);

    /* Compute Mask buffer length */
    bMaskByteLen = bMaskBitLen >> 3U;

    /* Check for incomplete bytes */
    if(bMaskBitLen & 0x07U)
    {
        ++bMaskByteLen;
    }
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMask), &pMask, bMaskByteLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bMaskBitLen), &bMaskBitLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bExtendedOptions), &bExtendedOptions);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pCID), pCID, 2);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bBlockNo), &bBlockNo);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wNoOfBlocks), &wNoOfBlocks);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_FastInventoryReadExtended((phpalSli15693_Sw_DataParams_t *) pDataParams, bFlags, bAfi, pMask,
                bMaskBitLen, bExtendedOptions, pCID, bBlockNo, wNoOfBlocks, pCDIDOut, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_FastInventoryReadExtended((phpalSli15693_Rd710_DataParams_t *) pDataParams, bFlags, bAfi,
                pMask, bMaskBitLen, bExtendedOptions, pCID, bBlockNo, wNoOfBlocks, pCDIDOut, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        if(pCDIDOut != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pCDIDOut), pCDIDOut, 2);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, *pUidLen);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUidLen), pUidLen);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pData), pData, *pDataLen);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDataLen), pDataLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_InventoryPageRead(void * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask, uint8_t bMaskBitLen,
    uint8_t bPageNo, uint16_t wNoOfPages, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bMaskByteLen = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_FastInventoryRead");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFlags);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAfi);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMask);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMaskBitLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPageNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNoOfPages);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(0U != bMaskBitLen) PH_ASSERT_NULL_PARAM(pMask, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bFlags), &bFlags);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bAfi), &bAfi);

    /* Compute Mask buffer length */
    bMaskByteLen = bMaskBitLen >> 3U;

    /* Check for incomplete bytes */
    if(bMaskBitLen & 0x07U)
    {
        ++bMaskByteLen;
    }
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMask), &pMask, bMaskByteLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bMaskBitLen), &bMaskBitLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bPageNo), &bPageNo);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wNoOfPages), &wNoOfPages);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_InventoryPageRead((phpalSli15693_Sw_DataParams_t *) pDataParams, bFlags, bAfi,
                pMask, bMaskBitLen, bPageNo, wNoOfPages, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_InventoryPageRead((phpalSli15693_Rd710_DataParams_t *) pDataParams, bFlags,
                bAfi, pMask, bMaskBitLen, bPageNo, wNoOfPages, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, *pUidLen);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUidLen), pUidLen);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pData), pData, *pDataLen);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDataLen), pDataLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_FastInventoryPageRead(void * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask, uint8_t bMaskBitLen,
    uint8_t bPageNo, uint16_t wNoOfPages, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bMaskByteLen = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_FastInventoryRead");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFlags);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAfi);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMask);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMaskBitLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPageNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNoOfPages);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(0U != bMaskBitLen) PH_ASSERT_NULL_PARAM(pMask, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bFlags), &bFlags);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bAfi), &bAfi);

    /* Compute Mask buffer length */
    bMaskByteLen = bMaskBitLen >> 3U;

    /* Check for incomplete bytes */
    if(bMaskBitLen & 0x07U)
    {
        ++bMaskByteLen;
    }
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMask), &pMask, bMaskByteLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bMaskBitLen), &bMaskBitLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bPageNo), &bPageNo);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wNoOfPages), &wNoOfPages);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_FastInventoryPageRead((phpalSli15693_Sw_DataParams_t *) pDataParams, bFlags, bAfi, pMask,
                bMaskBitLen, bPageNo, wNoOfPages, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_FastInventoryPageRead((phpalSli15693_Rd710_DataParams_t *) pDataParams, bFlags, bAfi,
                pMask, bMaskBitLen, bPageNo, wNoOfPages, pUid, pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, *pUidLen);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUidLen), pUidLen);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pData), pData, *pDataLen);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDataLen), pDataLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_StayQuietPersistent(void * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_StayQuietPersistent");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_StayQuietPersistent((phpalSli15693_Sw_DataParams_t *) pDataParams);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_StayQuietPersistent((phpalSli15693_Rd710_DataParams_t *) pDataParams);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}





/* ISO15693 Common commands ---------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_ActivateCard(void * pDataParams, uint8_t  bOption, uint8_t  bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t * pDsfid, uint8_t * pUid, uint8_t * pMoreCardsAvailable)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bMaskByteLen = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_ActivateCard");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFlags);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAfi);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMask);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMaskBitLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDsfid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pMoreCardsAvailable);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(0U != bMaskBitLen) PH_ASSERT_NULL_PARAM(pMask, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDsfid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pMoreCardsAvailable, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bOption), &bOption);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bFlags), &bFlags);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bAfi), &bAfi);

    /* Compute Mask buffer length */
    bMaskByteLen = bMaskBitLen >> 3;

    /* Check for incomplete bytes */
    if(bMaskBitLen & 0x07)
    {
        ++bMaskByteLen;
    }
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMask), &pMask, bMaskByteLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bMaskBitLen), &bMaskBitLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_ActivateCard((phpalSli15693_Sw_DataParams_t *) pDataParams, bOption, bFlags,
                bAfi, pMask, bMaskBitLen, pDsfid, pUid, pMoreCardsAvailable);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_ActivateCard((phpalSli15693_Rd710_DataParams_t *) pDataParams, bOption,
                bFlags, bAfi, pMask, bMaskBitLen, pDsfid, pUid, pMoreCardsAvailable);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDsfid), pDsfid);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, 8);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pMoreCardsAvailable), pMoreCardsAvailable);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_SendEof(void * pDataParams, uint8_t bOption, uint8_t * pDsfid, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData,
    uint16_t * pDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_SendEof");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDsfid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDsfid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bOption), &bOption);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Perform operation on active layer */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_SendEof((phpalSli15693_Sw_DataParams_t *) pDataParams, bOption, pDsfid, pUid, pUidLen,
                pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_SendEof((phpalSli15693_Rd710_DataParams_t *) pDataParams, bOption, pDsfid, pUid,
                pUidLen, pData, pDataLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG

    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDsfid), pDsfid);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, *pUidLen);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUidLen), pUidLen);

        if(bOption == PHPAL_SLI15693_EOF_NEXT_SLOT)
        {
            PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pData), pData, 8U);
        }
        else
        {
            PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pData), pData, 1U);
        }
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pDataLen), pDataLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_Exchange(void * pDataParams, uint16_t wOption, uint8_t * pTxBuffer, uint16_t wTxBuffLen, uint8_t ** ppRxBuffer,
    uint16_t * pRxBuffLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_Exchange");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxBuffLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(ppRxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxBuffLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(wTxBuffLen) PH_ASSERT_NULL_PARAM(pTxBuffer, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wOption), &wOption);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pTxBuffer), pTxBuffer, wTxBuffLen);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wTxBuffLen), &wTxBuffLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_Exchange((phpalSli15693_Sw_DataParams_t *) pDataParams, wOption, pTxBuffer,
                wTxBuffLen, ppRxBuffer, pRxBuffLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_Exchange((phpalSli15693_Rd710_DataParams_t *) pDataParams, wOption, pTxBuffer,
                wTxBuffLen, ppRxBuffer, pRxBuffLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if(((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS) && !(wOption & PH_EXCHANGE_BUFFERED_BIT) && (ppRxBuffer != NULL))
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(ppRxBuffer), *ppRxBuffer, *pRxBuffLen);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pRxBuffLen), pRxBuffLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

#ifdef NXPBUILD__PHPAL_SLI15693_INTERNAL
phStatus_t phpalSli15693_InProcessReplyASync(void * pDataParams, uint8_t * pTxBuffer, uint16_t wTxBuffLen, uint8_t * pResponse1,
    uint16_t * pResp1_Len, uint8_t * pResponse2, uint16_t * pResp2_Len)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_InProcessReplyASync");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxBuffLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pResponse1);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pResp1_Len);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pResponse2);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pResp2_Len);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    if(wTxBuffLen) PH_ASSERT_NULL_PARAM(pTxBuffer, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pResponse1, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pResp1_Len, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pResponse2, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pResp2_Len, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pTxBuffer), pTxBuffer, wTxBuffLen);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wTxBuffLen), &wTxBuffLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_InProcessReplyASync((phpalSli15693_Sw_DataParams_t *) pDataParams, pTxBuffer, wTxBuffLen,
                pResponse1, pResp1_Len, pResponse2, pResp2_Len);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pResponse1), pResponse1, *pResp1_Len);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pResp1_Len), pResp1_Len);

        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pResponse2), pResponse2, *pResp2_Len);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pResp2_Len), pResp2_Len);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}
#endif /* NXPBUILD__PHPAL_SLI15693_INTERNAL */





/* ISO15693 Utility commands --------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_GetSerialNo(void * pDataParams, uint8_t * pUid, uint8_t *pUidLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_GetSerialNo");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUidLen, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_GetSerialNo((phpalSli15693_Sw_DataParams_t *) pDataParams, pUid, pUidLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_GetSerialNo((phpalSli15693_Rd710_DataParams_t *) pDataParams, pUid, pUidLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, *pUidLen);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUidLen), pUidLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_SetSerialNo(void * pDataParams, uint8_t * pUid, uint8_t bUidLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_SetSerialNo");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bUidLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pUid, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pUid), pUid, bUidLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(bUidLen), &bUidLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_SetSerialNo((phpalSli15693_Sw_DataParams_t *) pDataParams, pUid, bUidLen);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_SetConfig");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValue);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wConfig), &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wValue), &wValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_SetConfig((phpalSli15693_Sw_DataParams_t *) pDataParams, wConfig, wValue);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_SetConfig((phpalSli15693_Rd710_DataParams_t *) pDataParams, wConfig, wValue);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

phStatus_t phpalSli15693_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phpalSli15693_GetConfig");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wStatus);

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pValue, PH_COMP_PAL_SLI15693);

    /* Log the information. */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(wConfig), &wConfig);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Component Code Validation */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_PAL_SLI15693)
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return wStatus;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            wStatus = phpalSli15693_Sw_GetConfig((phpalSli15693_Sw_DataParams_t *) pDataParams, wConfig, pValue);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            wStatus = phpalSli15693_Rd710_GetConfig((phpalSli15693_Rd710_DataParams_t *) pDataParams, wConfig, pValue);
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(pValue), pValue);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, PH_LOG_VAR(wStatus), &wStatus);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return wStatus;
}

#endif /* NXPBUILD__PHPAL_SLI15693 */
