/*
 * Copyright 2015 - 2018, 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for MIFARE Plus EVx.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/* Header for this file. */
#include "Example-MfpEVx.h"

#undef NXPBUILD__PH_LOG         /* Disabling Log. To enable comment this line. */

#ifdef NXPBUILD__PH_LOG
    phLog_RegisterEntry_t logRegisterEntries[16];
    phLog_LogEntry_t logEntries[48];
#endif /* NXPBUILD__PH_LOG */

    int __cdecl main()
    {
        phStatus_t wStatus = 0;

        uint16_t wCount = 0;
        int dwChoice = 0;

        uint8_t aAtr[256];
        uint8_t aSeed[16];
        uint8_t aReaderList[512];
        uint8_t aTx_HalBuffer[512];
        uint8_t aRx_HalBuffer[512];

        uint8_t aPortName[] = { '1','2','7','.','0','.','0','.','1','\0','1','5','2','0','0','\0','0','0','0','1','\0' };
        uint8_t aTx_SocketBuffer[0x800];
        uint8_t aRx_SocketBuffer[0x200];
        uint8_t aTx_SimBuffer[0x800];
        uint8_t aRx_SimBuffer[0x800];
        uint8_t aInt_SimBuffer[0xff00];
        uint8_t aInt_MfcCryptoBuffer[0xff00];

        uint8_t aTmi_Buffer[0xFF];

        uint8_t aCryptoASym_Buffer[512];

        uint16_t wLen_IidTableEntries = 0x13;               /**< Number of possible IID table entries in the storage. */
        uint16_t wLen_CardTableEntries = 0x13;              /**< Number of possible Card table entries in the storage. */

        phKeyStore_Sw_KeyEntry_t aKeyEntry[0xFF];
        phKeyStore_Sw_KeyVersionPair_t aKeyVersion[0xFF];
        phKeyStore_Sw_KUCEntry_t aKeyUsageCounter[0xFF];

        phhalHw_Rd70x_DataParams_t stHal_Rd70x;             /* HAL parameter structure for Rd70x component. */
        phhalHw_RdCardSim_DataParams_t stHal_Sim;           /* HAL parameter structure for simulator component. */
        phhalHw_MfcCrypto_DataParams_t stHal_MfcCrypto;     /* HAL parameter structure for MFC Crypto component.*/

        phbalReg_Rd70xUsbWin_DataParams_t stBal_Rd70x;
        phbalReg_SocketWin_DataParams_t stBal_Socket;

        phbalReg_PcscWin_DataParams_t stBal_Pcsc;
        phhalHw_Rd710_DataParams_t    stHal_Rd710;

        phCryptoSym_Sw_DataParams_t stCryptoEnc;            /* CryptoSym parameter structure for Encryption */
        phCryptoSym_Sw_DataParams_t stCryptoMac;            /* CryptoSym parameter structure for MACing */
        phCryptoSym_Sw_DataParams_t stCryptoDiversify;      /* CryptoSym parameter structure for diversify*/
        phCryptoSym_Sw_DataParams_t stCryptoSymRng;         /* CryptoSym parameter structure for Random Number generator */
        phCryptoRng_Sw_DataParams_t stCryptoRng;            /* CryptoSym parameter structure for Random Number generator */

        phalVca_Sw_IidTableEntry_t stIidTableEntry[0x13];   /**< Pointer to the IID Table storage for the layer. */
        phalVca_Sw_CardTableEntry_t stCardTableEntry[0x13]; /**< Pointer to the Card Table storage for the layer. */

        printf("Press %d to run with Reader = CLRd70x, Card = MFPEv1.\n", RD70X);
        printf("Press %d to run with Reader = CLRd710, Card = MFPEv1.\n", RD710);
        printf("Press %d to run with Reader = Simulator, Card = Simulator.\n", SIMULATOR);
        printf("Press x to Quit.\n");
        printf("Enter the option and press Enter to perform the demo - ");
        scanf_s("%d", &dwChoice);

        printf("\n\n");

        /* Exit if requested. */
        EXIT_IF_REQUESTED(dwChoice);

        switch(dwChoice)
        {
            case RD70X:
                printf("Please ensure that a Rd70x reader is connected and is in working condition.\n");
                printf("Firmware Version 2.11 or higher must be used!\n\n");

                /* Initialize the reader BAL component. */
                wStatus = phbalReg_Rd70xUsbWin_Init(&stBal_Rd70x, sizeof(stBal_Rd70x));
                CHECK_SUCCESS(wStatus);

                /* Get list of connected Pegoda readers. */
                wStatus = phbalReg_GetPortList(&stBal_Rd70x, sizeof(aReaderList), aReaderList, &wCount);
                CHECK_SUCCESS(wStatus);

                /* Connect to the first reader out of the list. */
                wStatus = phbalReg_SetPort(&stBal_Rd70x, aReaderList);
                CHECK_SUCCESS(wStatus);

                /* Open the reader port. */
                wStatus = phbalReg_OpenPort(&stBal_Rd70x);
                CHECK_SUCCESS(wStatus);

                /* Initialize the reader HAL component. */
                wStatus = phhalHw_Rd70x_Init(&stHal_Rd70x, sizeof(stHal_Rd70x), &stBal_Rd70x,
                    aTx_HalBuffer, sizeof(aTx_HalBuffer), aRx_HalBuffer, sizeof(aRx_HalBuffer));
                CHECK_SUCCESS(wStatus);

                /* Set the default reader as Rd70x. */
                pHal_Generic = &stHal_Rd70x;

                bIsEnggSample = PH_OFF;
                break;

            case RD710:
                printf("Please ensure that a Rd710 reader is connected and is in working condition.\n");
                printf("Firmware Version 2.11 or higher must be used!\n\n");

                /* Initialize the reader BAL component. */
                wStatus = phbalReg_PcscWin_Init(&stBal_Pcsc, sizeof(stBal_Pcsc), aAtr, 256);
                CHECK_SUCCESS(wStatus);

                /* Set the BAL Communication Configuration values */
                wStatus = phbalReg_SetConfig(&stBal_Pcsc, PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL, PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED);
                CHECK_SUCCESS(wStatus);

                wStatus = phbalReg_SetConfig(&stBal_Pcsc, PHBAL_REG_PCSCWIN_CONFIG_SHARE, PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);
                CHECK_SUCCESS(wStatus);

                /* Select Port to be used (no SAM reader) */
                wStatus = phbalReg_SetPort(&stBal_Pcsc, (uint8_t *) PCSC_READER_NO_SAM_NAME);
                CHECK_SUCCESS(wStatus);

                /* Open the PCSC port */
                wStatus = phbalReg_OpenPort(&stBal_Pcsc);
                CHECK_SUCCESS(wStatus);

                /* Initialize the Rd710 HAL component. */
                wStatus = phhalHw_Rd710_Init(&stHal_Rd710, sizeof(stHal_Rd710), &stBal_Pcsc, 0, aTx_HalBuffer, sizeof(aTx_HalBuffer),
                    aRx_HalBuffer, sizeof(aRx_HalBuffer));
                CHECK_SUCCESS(wStatus);

                /* Standard Rd710 Command Init. */
                wStatus = (phStatus_t) phhalHw_Rd710_Cmd_InitReader(&stHal_Rd710);
                CHECK_SUCCESS(wStatus);

                /*Set default reader as RD710 / CM1. */
                pHal_Generic = &stHal_Rd710;

                bIsEnggSample = PH_OFF;
                break;

            case SIMULATOR:
                printf("Please ensure that Simulator is open and is in waiting state.\n\n");
                printf("Simulator version 5311 or higher must be used!\n\n");

                /* Initialize the reader BAL component */
                wStatus = phbalReg_SocketWin_Init(&stBal_Socket, sizeof(stBal_Socket), 0, sizeof(aTx_SocketBuffer),
                    aTx_SocketBuffer, sizeof(aRx_SocketBuffer), aRx_SocketBuffer);
                CHECK_SUCCESS(wStatus);

                wStatus = phbalReg_SetPort(&stBal_Socket, aPortName);
                CHECK_SUCCESS(wStatus);

                /* Initialize the reader HAL component. */
                wStatus = phhalHw_RdCardSim_Init(&stHal_Sim, sizeof(stHal_Sim), &stBal_Socket, &stKeyStore, aTx_SimBuffer,
                    sizeof(aTx_SimBuffer), aRx_SimBuffer, sizeof(aRx_SimBuffer), aInt_SimBuffer, sizeof(aInt_SimBuffer));
                CHECK_SUCCESS(wStatus);

                /* Open the reader port. */
                wStatus = phbalReg_OpenPort(&stBal_Socket);
                CHECK_SUCCESS(wStatus);

                /* Set the default reader as Rd70x. */
                pHal_Generic = &stHal_Sim;

                bIsEnggSample = PH_ON;
                break;

            default:
                printf(" Invalid option selected.\n\n");
                printf(" Pressing any key will exit the demo. Restart the demo and choose the right option.\n ");
                _getch();
                exit(0);
        }

        /* Update the Sim State. */
        bSimState = bIsEnggSample;

        printf("Build up the library stack... ");

        /* Initialize software KeyStore component. */
        wStatus = phKeyStore_Sw_Init(&stKeyStore, sizeof(stKeyStore), aKeyEntry, 0xFF, aKeyVersion, 0x01, aKeyUsageCounter, 0xFF);
        CHECK_SUCCESS(wStatus);

        /* Initialize CryptoSym for encryption. */
        wStatus = phCryptoSym_Sw_Init(&stCryptoEnc, sizeof(stCryptoEnc), NULL);
        CHECK_SUCCESS(wStatus);

        /* Initialize CryptoSym for macing. */
        wStatus = phCryptoSym_Sw_Init(&stCryptoMac, sizeof(stCryptoMac), NULL);
        CHECK_SUCCESS(wStatus);

        /* Initialize CryptoSym for key diversification. */
        wStatus = phCryptoSym_Sw_Init(&stCryptoDiversify, sizeof(stCryptoDiversify), NULL);
        CHECK_SUCCESS(wStatus);

        /* Initialize CryptoASym for key diversification. */
        wStatus = phCryptoASym_mBedTLS_Init(&stCryptoASym, sizeof(stCryptoASym), &stKeyStore, aCryptoASym_Buffer, sizeof(aCryptoASym_Buffer));
        CHECK_SUCCESS(wStatus);

        /* Initialize CryptoSym for random number generation. */
        wStatus = phCryptoSym_Sw_Init(&stCryptoSymRng, sizeof(stCryptoSymRng), NULL);
        CHECK_SUCCESS(wStatus);

        /* Initialize Crypto for random number generation. */
        wStatus = phCryptoRng_Sw_Init(&stCryptoRng, sizeof(stCryptoRng), &stCryptoSymRng);
        CHECK_SUCCESS(wStatus);

        /* Rest the seed buffer to initiate random number generation. */
        memset(aSeed, 0x00, sizeof(aSeed));

        /* Initiate seed for random number generation. */
        wStatus = phCryptoRng_Seed(&stCryptoRng, aSeed, sizeof(aSeed));
        CHECK_SUCCESS(wStatus);

        /* Initialize TMI utility. */
        memset(&aTmi_Buffer[0], 0x00, sizeof(aTmi_Buffer));
        wStatus = phTMIUtils_Init(&stTMI, &aTmi_Buffer[0], sizeof(aTmi_Buffer));
        CHECK_SUCCESS(wStatus);

        if(bIsEnggSample)
        {
            /* Initialize MFC crypto component. */
            wStatus = phhalHw_MfcCrypto_Init(&stHal_MfcCrypto, sizeof(stHal_MfcCrypto),
                pHal_Generic, aInt_MfcCryptoBuffer, sizeof(aInt_MfcCryptoBuffer));
            CHECK_SUCCESS(wStatus);

            /* Enable MFC crypto configuration in HAL layer. */
            wStatus = phhalHw_SetConfig(&stHal_MfcCrypto, PHHAL_HW_MFCCRYPTO_CONFIG_ENABLE, 0x01);
            CHECK_SUCCESS(wStatus);

            /* Update the generic HAL parameter. */
            pHal_Generic = &stHal_MfcCrypto;
        }

        /* Initialize ISO14443-3A component. */
        wStatus = phpalI14443p3a_Sw_Init(&stI14443p3a, sizeof(stI14443p3a), pHal_Generic);
        CHECK_SUCCESS(wStatus);

        /* Initialize ISO14443-4A component. */
        wStatus = phpalI14443p4a_Sw_Init(&stI14443p4a, sizeof(stI14443p4a), pHal_Generic);
        CHECK_SUCCESS(wStatus);

        /* Initialize ISO14443-4 generic component. */
        wStatus = phpalI14443p4_Sw_Init(&stI14443p4, sizeof(stI14443p4), pHal_Generic);
        CHECK_SUCCESS(wStatus);

        /* Initialize MIFARE component. */
        wStatus = phpalMifare_Sw_Init(&stMifare, sizeof(stMifare), pHal_Generic, &stI14443p4);
        CHECK_SUCCESS(wStatus);

        /* Initialize VCA component. */
        wStatus = phalVca_Sw_Init(&stVca, sizeof(stVca), &stMifare, &stKeyStore, &stCryptoEnc, &stCryptoRng,
            stIidTableEntry, wLen_IidTableEntries, stCardTableEntry, wLen_CardTableEntries);
        CHECK_SUCCESS(wStatus);

        wStatus = phalMfpEVx_Sw_Init(&stMfpEVx, sizeof(stMfpEVx), &stMifare, &stKeyStore, &stCryptoEnc, &stCryptoMac,
            &stCryptoRng, &stCryptoDiversify, &stTMI, &stVca);
        CHECK_SUCCESS(wStatus);

        wStatus = phalVca_SetApplicationType(&stVca, &stMfpEVx);
        CHECK_SUCCESS(wStatus);

            /* Initialize logging */
#ifdef NXPBUILD__PH_LOG
    /* Register the component for logging */
        wStatus = phLog_Init(MfpEv1_Log, logRegisterEntries, (uint16_t) (sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
        CHECK_SUCCESS(wStatus);

        wStatus = phLog_Register(&stMfpEVx, logEntries, (uint16_t) (sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
        CHECK_SUCCESS(wStatus);

        wStatus = phLog_Register(&stVca, logEntries, (uint16_t) (sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
        CHECK_SUCCESS(wStatus);
#endif /* NXPBUILD__PH_LOG */

        printf("Success\n\n");

        /* Print the Firmware version. */


        /* Perform a FieldOn. */
        wStatus = phhalHw_FieldOn(pHal_Generic);
        CHECK_SUCCESS(wStatus);

        /* Configure Hardware for Type-A cards. */
        wStatus = phhalHw_ApplyProtocolSettings(pHal_Generic, PHHAL_HW_CARDTYPE_ISO14443A);
        CHECK_SUCCESS(wStatus);

        /* Add keys to the software KeyStore component. */
        wStatus = UpdateKeystore();
        CHECK_SUCCESS(wStatus);

        /* Update the EEPROM size. */
        wStatus = UpdateEEPROM_Size();
        CHECK_SUCCESS(wStatus);

        do
        {
            /* Clear the console. */
            system("cls");

            /* Restore to security level zero. */
            wStatus = RestoreToSL0();
            CHECK_SUCCESS(wStatus);

            /* Set the current security level of the card. */
            wStatus = SetSecurityLevel();
            CHECK_SUCCESS(wStatus);

            /* Reset the authentication state. */
            wStatus = phalMfpEVx_ResetSecMsgState(&stMfpEVx);
            CHECK_SUCCESS(wStatus);

            printf("Press %d for Demo on Pre-personalization of the card. \n", PERSONALIZATION);
            printf("Press %d for Demo on Security Level 1 in MFC mode. ( ISO14443-L3 ) \n", SL1_MFC);
            printf("Press %d for Demo on Security Level 1 TMAC feature in MFC mode. \n", SL1_MFC_TMAC);
            printf("Press %d for Demo on SL1-SL3 mixed mode. \n", SL1_SL3_MIXED);
            printf("Press %d for Demo on Security Level 1 TMAC feature in MFP mode. \n", SL1_MFP_TMAC);
            printf("Press %d for Demo on switching to Security Level 3. \n", SL3_SWITCH);
            printf("Press %d for Demo on EV0 Secure Messaging Mode in Security Level 3. \n", SL3_EV0_SM);
            printf("Press %d for Demo on EV1 Secure Messaging Mode in Security Level 3. \n", SL3_EV1_SM);
            printf("Press %d for Demo on Security Level 3 TMAC feature. \n", SL3_TMAC);
            printf("Press %d for Demo on ISO 7816 Wrapped mode support in Security Layer 1 or 3. \n", SL1_SL3_ISO7816);
            printf("Press %d for Demo on ISO 7816 Extended Length APDU support in Security Layer 1 or 3. \n", SL1_SL3_ISO7816_EXTENDED);
            printf("Press %d for Originality Check. \n", ORIGINALITY_CHECK);
            printf("Press %d for GetVersion. \n", GET_VERSION);
            printf("Press %d for Virtual Card. \n", VIRTUAL_CARD);
            printf("Press %d for Proximity Check. \n", PROXIMITY_CHECK);
            printf("Press %d for UID Personalization. \n", UID_PERSONALIZATION);
            printf("Press %d for Tearing proof transfer. \n", TEARING_PROOF_TRANSFER);
            printf("Press %d for Multiple Blocks Read. \n", MULIPLE_BLOCK_READ);
            printf("Press x to exit the demonstration. \n");
            printf("Enter the option and press Enter to perform the demo - ");
            scanf_s("%d", &dwChoice);

            printf("\n\n");

            /* Exit if requested by user. */
            EXIT_IF_REQUESTED(dwChoice);

            switch(dwChoice)
            {
                case PERSONALIZATION:
                    printf("Demo on pre-personalization of the card. ---------------------------------------------- ");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_0) == PH_ERR_SUCCESS)
                    {
                        /* Pre-personalize the card. */
                        bIsEnggSample = 1;
                        wStatus = Demo_PrePersonalization(ISO14443_L4, 0x00);
                        CHECK_SUCCESS_BREAK(wStatus);
                        bIsEnggSample = bSimState;

                        /* Set the current security level of the card. */
                        wStatus = SetSecurityLevel();
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL1_MFC:
                    printf("Demo on Security Level 1 in MFC mode. ( ISO14443-L3 ) --------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_SecurityLayer(SECURITY_LEVEL_1_MIFARE, ISO14443_L3, 0x00, bIsEnggSample);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL1_MFC_TMAC:
                    printf("Demo on Security Level 1 TMAC feature in MFC mode. -------------------------------------\n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_TransactionMAC(SECURITY_LEVEL_1_MIFARE);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL1_SL3_MIXED:
                    printf("Demo on SL1 Sl3 mixed mode. ----------------------------------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1) == PH_ERR_SUCCESS)
                    {
                        /* Demo with SL1  . */
                        printf("Demo in security level 1------------------------------------------- \n");
                        wStatus = Demo_SecurityLayer(SECURITY_LEVEL_1_MFP, ISO14443_L4, EV0_SECURE_MESSAGING, bIsEnggSample);
                        CHECK_SUCCESS_BREAK(wStatus);
                        printf("\n");

                        printf("Demo in security level 3------------------------------------------- \n");
                        /* Demo with SL3  . */
                        wStatus = Demo_SecurityLayer(SECURITY_LEVEL_3_MFP, ISO14443_L4, EV0_SECURE_MESSAGING, 0x00);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL1_MFP_TMAC:
                    printf("Demo on Security Level 1 TMAC feature in MFP mode. -------------------------------------\n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_TransactionMAC(SECURITY_LEVEL_1_MFP);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL3_SWITCH:
                    printf("Demo on switching to Security Level 3. ------------------------------------------------ \n\n");

                    /* Pre-personalize the card. */
                    wStatus = Demo_PrePersonalization(ISO14443_L4, 0x00);
                    CHECK_SUCCESS_BREAK(wStatus);

                    /* Set the security Level. */
                    wStatus = SetSecurityLevel();
                    CHECK_SUCCESS_BREAK(wStatus);

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1) == PH_ERR_SUCCESS)
                    {
                        /* Activate to ISO14443 L4. */
                        wStatus = ActivateCard(ISO14443_L4);
                        CHECK_SUCCESS_BREAK(wStatus);

                        /* Switch to security level 3. */
                        wStatus = phalMfpEVx_AuthenticateSL1(&stMfpEVx, ISO14443_L4, FIRST_AUTH, LEVEL_3_SWITCH_KEY, LEVEL_3_SWITCH_KEY_ADDRESS,
                            LEVEL_3_SWITCH_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL3_EV0_SM:
                    printf("Demo on EV0 Secure Messaging Mode in Security Level 3. -------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_3) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_SecurityLayer(SECURITY_LEVEL_3_MFP, ISO14443_L4, EV0_SECURE_MESSAGING, bIsEnggSample);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL3_EV1_SM:
                    printf("Demo on EV1 Secure Messaging Mode in Security Level 3. -------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_3) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_SecurityLayer(SECURITY_LEVEL_3_MFP, ISO14443_L4, EV1_SECURE_MESSAGING, bIsEnggSample);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL3_TMAC:
                    printf("Demo on TMAC feature in Security Level 3. ----------------------------------------------\n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_3) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_TransactionMAC(SECURITY_LEVEL_3_MFP);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL1_SL3_ISO7816:
                    printf("Demo on ISO 7816 Wrapped mode support. ------------------------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1 | SECURITY_LEVEL_3) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_ISO7816((bCurrentSecurityLevel == SECURITY_LEVEL_1) ? SECURITY_LEVEL_1_MFP : SECURITY_LEVEL_3_MFP, ISO14443_L4, 0x00, 0x00);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case SL1_SL3_ISO7816_EXTENDED:
                    printf("Demo on ISO 7816 Extended Length APDU support. ----------------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1 | SECURITY_LEVEL_3) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_ISO7816((bCurrentSecurityLevel == SECURITY_LEVEL_1) ? SECURITY_LEVEL_1_MFP : SECURITY_LEVEL_3_MFP, ISO14443_L4, 0x00, 0x01);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case ORIGINALITY_CHECK:
                    printf("Demo on Originality Signature Check. -------------------------------------------------- \n\n");

                    wStatus = Demo_OriginalityCheck((bCurrentSecurityLevel == SECURITY_LEVEL_0) ? 0x00 :
                        (bCurrentSecurityLevel == SECURITY_LEVEL_1) ? SECURITY_LEVEL_1_MFP : SECURITY_LEVEL_3_MFP);
                    CHECK_SUCCESS_BREAK(wStatus);
                    break;

                case GET_VERSION:
                    printf("Demo on GetVersion.-------------------------------------------------------------------- \n\n");

                    wStatus = Demo_GetVersion((bCurrentSecurityLevel == SECURITY_LEVEL_0) ? 0x00 :
                        (bCurrentSecurityLevel == SECURITY_LEVEL_1) ? SECURITY_LEVEL_1_MFP : SECURITY_LEVEL_3_MFP);
                    CHECK_SUCCESS_BREAK(wStatus);
                    break;

                case VIRTUAL_CARD:
                    printf("Demo on Virtual Card.-------------------------------------------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    //wStatus = IncludeDemo ( bCurrentSecurityLevel );

                    if(wStatus == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_VirtualCardDemo();
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case PROXIMITY_CHECK:
                    printf("Demo on Proximity Check ----------------------------------------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1 | SECURITY_LEVEL_3) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_ProximityCheck();
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case UID_PERSONALIZATION:
                    printf("Demo on UID Personalization ------------------------------------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1))
                    {
                        wStatus = wStatus = Demo_UidPersonalization();
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case TEARING_PROOF_TRANSFER:
                    printf("Demo on Anti-Tearing -------------------------------------------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1 | SECURITY_LEVEL_3) == PH_ERR_SUCCESS)
                    {
                        wStatus = wStatus = Demo_TearingProofTransfer();
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                case MULIPLE_BLOCK_READ:
                    printf("Demo on Multiple block read ------------------------------------------------------------- \n\n");

                    /* Check if this feature is valid in the current security level of the card. */
                    if(IncludeDemo(SECURITY_LEVEL_1 | SECURITY_LEVEL_3) == PH_ERR_SUCCESS)
                    {
                        wStatus = Demo_MultipleBlocksRead(SECURITY_LEVEL_3_MFP, EV0_SECURE_MESSAGING, bIsEnggSample);
                        CHECK_SUCCESS_BREAK(wStatus);
                    }
                    break;

                default:
                    printf("Invalid option selected.\n");
                    break;
            }

            printf("\n\n");
            printf("Press any key to continue with the demonstration.\n");
            printf("Press x to exit the demonstration.\n");
            dwChoice = _getch();
        } while((dwChoice != 'x') && (dwChoice != 'X'));

        return PH_ERR_SUCCESS;
    }

void PrintData ( uint8_t* pBuffer, uint32_t dwLength, char* pFormat, char* pSpecialChar )
{
    uint32_t dwIndex;

    for ( dwIndex = 0; dwIndex < dwLength; ++dwIndex )
        printf ( pFormat, pBuffer[dwIndex] );

    printf ( pSpecialChar );
}

phStatus_t UpdateTMC ( uint8_t bSecurityLevel )
{
    phStatus_t wStatus = 0;

    /* Deactivate TMI Collection. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phTMIUtils_ActivateTMICollection ( &stTMI, PH_TMIUTILS_PAUSE_TMI ) );

    /* Read the data from the TMAC block based on security level and state. */
    switch ( bSecurityLevel )
    {
        case SECURITY_LEVEL_1_MIFARE:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_ReadSL1TMBlock ( &stMfpEVx, TMAC_BLOCK_1, aRx_Data ) );
            break;

        default:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Read ( &stMfpEVx, ENCRYPTED_OFF, MAC_RESPONSE_OFF, MAC_COMMAND_ON, TMAC_BLOCK_1, 1, aRx_Data ) );
            break;
    }

    /* Deactivate TMI Collection. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phTMIUtils_ActivateTMICollection ( &stTMI, PH_TMIUTILS_RESUME_TMI ) );

    /* Extract the TMC. */
    memcpy ( &dwTMC, aRx_Data, 4 );

    if ( dwTMC != 0x00000000 )
        dwTMC = dwTMC - 1;

    memcpy ( aTMC, &dwTMC, 4 );

    return PH_ERR_SUCCESS;
}

phStatus_t ActivateCard ( uint8_t bIsoLayer )
{
    phStatus_t wStatus;

    uint8_t bCardPlaced = 0;
    uint8_t bMoreCardsAvailable;    /* more card available */
    uint8_t bCidEnabled;            /* Unequal '0' if Card Identifier is enabled. */
    uint8_t bCid = 1;               /* Card Identifier. */
    uint8_t bNadSupported;          /* Node Address Support; Unequal '0' if supported. */
    uint8_t bFwi;                   /* Frame Waiting Integer. */
    uint8_t bFsdi = 8;              /* PCD Frame Size Integer; 0-8 */
    uint8_t bFsci;                  /* PICC Frame Size Integer; 0-8. */

    uint8_t aAts[256];              /* Answer to Select; uint8_t[1-254] */

    /* Reset the field */
    PH_CHECK_SUCCESS_FCT ( wStatus, phhalHw_FieldReset ( pHal_Generic ) );

    PH_CHECK_SUCCESS_FCT ( wStatus, phhalHw_Wait(pHal_Generic, PHHAL_HW_TIME_MILLISECONDS, 5));

    /* Activate Layer 3 card. In loop till a card is detected. */
    do
    {
        wStatus = phpalI14443p3a_ActivateCard ( &stI14443p3a, NULL, 0x00, aUid, &bUid_Len, &bSak, &bMoreCardsAvailable );

        if ( ( wStatus != PH_ERR_SUCCESS ) && ( !bCardPlaced ) )
        {
            printf ( "Place MIFARE Plus Ev1 card into field.\n" );
            bCardPlaced = 1;
        }
    } while ( wStatus != PH_ERR_SUCCESS );
    CHECK_SUCCESS ( wStatus );

    /* Set Auth mode as Iso14443 Level 3. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( &stMfpEVx, PHAL_MFPEVX_AUTH_MODE, PHAL_MFPEVX_NOT_AUTHENTICATED_L3 ) );

    /* Activate L4 */
    if ( bIsoLayer )
    {
        PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p4a_ActivateCard ( &stI14443p4a, bFsdi, bCid, PHPAL_I14443P4A_DATARATE_106, PHPAL_I14443P4A_DATARATE_106, aAts ) );

        /* Retrieve the ISO14443-4A protocol parameters. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p4a_GetProtocolParams( &stI14443p4a, &bCidEnabled, &bCid, &bNadSupported, &bFwi,
            &bFsdi, &bFsci ) );

        /* Set the ISO14443-4 protocol parameters. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p4_SetProtocol( &stI14443p4, bCidEnabled, bCid, bNadSupported, 0, bFwi, bFsdi, bFsci ) );

        /* Set Auth mode as Iso14443 Level 3. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( &stMfpEVx, PHAL_MFPEVX_AUTH_MODE, PHAL_MFPEVX_NOT_AUTHENTICATED_L4 ) );
    }

    return PH_ERR_SUCCESS;
}

phStatus_t RestoreToSL0 ()
{
    phStatus_t wStatus = 0;
    uint16_t wRxLen = 0;
    uint8_t aTx_Data[] =  { 0x68, 0x00, 0x80 };
    uint8_t* pRxData = NULL;

    /* Check if the demo is performed on the test firmware. */
    if ( bIsEnggSample )
    {
        /* Activate in Layer 4. */
        PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

        /* Exchange the command to restore the security level zero. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phpalMifare_ExchangeL4 ( &stMifare, PH_EXCHANGE_DEFAULT, aTx_Data, 0x01, &pRxData, &wRxLen ) );

        /* Activate in Layer 4. */
        //PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

        /* Exchange the command to restore the test mode configuration. */
        //aTx_Data[0] = 0x66;
        //PH_CHECK_SUCCESS_FCT ( wStatus, phpalMifare_ExchangeL4 ( &stMifare, PH_EXCHANGE_DEFAULT, aTx_Data, 0x03, &pRxData, &wRxLen ) );
    }

    return PH_ERR_SUCCESS;
}

phStatus_t SetSecurityLevel ()
{
    phStatus_t wStatus = 0;

    /* Activate the card in ISO14443 L3 to get the SAK value. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L3 ) );

    switch ( bSak )
    {
        case 0x18:
        case 0x08:
            bCurrentSecurityLevel = SECURITY_LEVEL_1;
            break;

        case 0x20:
            /* Activate the card in ISO14443 L4. */
            wStatus = ActivateCard ( ISO14443_L4 );
            if ( wStatus != PH_ERR_SUCCESS )
            {
                printf ( " Not a valid MFP EV1 card\n" );
                return wStatus;
            }

            /* Perform Write perso to identify for Security Level 0. */
            wStatus = phalMfpEVx_WritePerso ( &stMfpEVx, ISO14443_L4, CARD_MASTER_KEY, 1, gaCardMasterKey );
            if ( wStatus == PH_ERR_SUCCESS )
                bCurrentSecurityLevel = SECURITY_LEVEL_0;
            else
            {
                wStatus = phalMfpEVx_AuthenticateSL3 ( &stMfpEVx, FIRST_AUTH, CARD_MASTER_KEY, CARD_MASTER_KEY_ADDRESS, CARD_MASTER_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 );

                if ( ( wStatus == 0x2180 ) ||  ( wStatus == PH_ERR_SUCCESS ) )
                    bCurrentSecurityLevel = SECURITY_LEVEL_3;
                else
                {
                    printf ( " Not a valid MFP EV1 card\n" );
                    return wStatus;
                }
            }
            break;

        default:
            printf ( " Not a valid MFP EV1 card\n" );
            return PH_ERR_UNSUPPORTED_PARAMETER;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t IncludeDemo ( uint8_t bSecurityLevel )
{
    if ( !bIsEnggSample )
    {
        if ( ( ( bCurrentSecurityLevel & bSecurityLevel ) != SECURITY_LEVEL_0 ) &&
            ( ( bCurrentSecurityLevel & bSecurityLevel ) != SECURITY_LEVEL_1 ) &&
            ( ( bCurrentSecurityLevel & bSecurityLevel ) != SECURITY_LEVEL_3 ) )
        {
            printf ( "\n\n" );
            printf ( "This demo feature is not supported in Security Level %d\n", ( ( bCurrentSecurityLevel == SECURITY_LEVEL_0 ) ? 0x00 : ( bCurrentSecurityLevel == SECURITY_LEVEL_1 ) ? 0x01 : 0x03) );
            printf ( "\n\n" );

            return PH_ERR_UNSUPPORTED_COMMAND;
        }
    }

    return PH_ERR_SUCCESS;
}

phStatus_t AddKeyToKeyStore ( uint8_t bKeyNo, uint8_t bKeyVersion, uint8_t* pKey, uint16_t wKeyType)
{
    phStatus_t wStatus = 0;

    PH_CHECK_SUCCESS_FCT ( wStatus, phKeyStore_FormatKeyEntry ( &stKeyStore, bKeyNo, wKeyType ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, phKeyStore_SetKeyAtPos ( &stKeyStore, bKeyNo, 0x00, wKeyType, pKey, bKeyVersion ) );

    return PH_ERR_SUCCESS;
}

phStatus_t UpdateEEPROM_Size ()
{
    phStatus_t wStatus = 0;
    uint8_t aVerInfo[33];

    /* Activate Layer 4 card. */
    wStatus = ActivateCard ( ISO14443_L4 );
    CHECK_SUCCESS ( wStatus );

    /* Get the version. */
    wStatus = phalMfpEVx_GetVersion ( &stMfpEVx, aVerInfo );
    CHECK_SUCCESS ( wStatus );

    bEEPROMSize = (uint8_t) ( ( aVerInfo[5] == 0x16 ) ? EEPROM_2KB : EEPROM_4KB );

    return wStatus;
}

phStatus_t UpdateKeystore ()
{
    phStatus_t wStatus = 0;
    uint8_t aMifareKey[12];

    /* Add the required keys to KeyStore. */
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( CARD_MASTER_KEY_ADDRESS, CARD_MASTER_KEY_VERSION, gaCardMasterKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( CARD_CONFIGURATION_KEY_ADDRESS, CARD_CONFIGURATION_KEY_VERSION, gaCardConfigKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( LEVEL_3_SWITCH_KEY_ADDRESS, LEVEL_3_SWITCH_KEY_VERSION, gaLevel3SwitchKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( SL_1_CARD_AUTH_KEY_ADDRESS, SL_1_CARD_AUTH_KEY_VERSION, gaSL1CardAuthKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( LEVEL_3_SECTOR_SWITCH_KEY_ADDRESS, LEVEL_3_SECTOR_SWITCH_KEY_VERSION, gaLevel3SectorSwitchKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( L1_L3_MIX_SECTOR_SWITCH_KEY_ADDRESS, L1_L3_MIX_SECTOR_SWITCH_KEY_VERSION, gaLevel13SMixSectorSwitchKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );

    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( VC_PROXIMITY_KEY_ADDRESS, VC_PROXIMITY_KEY_VERSION, gaVcProximityKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( VC_SELECT_ENC_KEY_ADDRESS, VC_SELECT_ENC_KEY_VERSION, gaVcSelectEncKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( VC_SELECT_MAC_KEY_ADDRESS, VC_SELECT_MAC_KEY_VERSION, gaVcSelectMacKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );

    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( TMAC_KEY_1_ADDRESS, TMAC_KEY_1_VERSION, gaTMAC_Key_1, PH_KEYSTORE_KEY_TYPE_AES128 ) );

    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( TMAC_CONFIG_KEY_1_ADDRESS, TMAC_CONFIG_KEY_1_VERSION, gaTMAC_ConfigKey_1, PH_KEYSTORE_KEY_TYPE_AES128 ) );

    /* Update MIFARE keys to software KeyStore. */
    memcpy ( aMifareKey, gaMifare_SectorKey_FF, 6);
    memcpy ( &aMifareKey[6], &gaMifare_SectorKey_FF[10], 6);
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aMifareKey, PH_KEYSTORE_KEY_TYPE_MIFARE ) );

    /* Update AES keys to software KeyStore. */
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( AES_KEY_ADDRESS, AES_KEY_VERSION, gaAES_SectorKey_FF, PH_KEYSTORE_KEY_TYPE_AES128 ) );

    /* Update Originality keys to software KeyStore. */
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( ORIGINALITY_KEY_1_ADDRESS, ORIGINALITY_KEY_1_VERSION, gaOriginalityKey1, PH_KEYSTORE_KEY_TYPE_AES128 ) );

    return PH_ERR_SUCCESS;
}

uint8_t GetNumberofSectors ()
{

    switch ( bEEPROMSize )
    {
        case EEPROM_2KB:
            return ( 32 );

        case EEPROM_4KB:
            return ( 40 );

        default:
            return 0;
    }
}

phStatus_t GetAESSectorBlock_A ( uint16_t wSectorNr )
{
    return ( uint16_t ) ( ( wSectorNr * 2 ) + PHYSICAL_AES_START_ADDRESS );
}

phStatus_t GetAESSectorBlock_B ( uint16_t wSectorNr )
{
    return ( uint16_t ) ( ( ( wSectorNr * 2 ) + PHYSICAL_AES_START_ADDRESS ) + 1 );
}

phStatus_t ResetCommitReaderIDBlock ( uint8_t bSecurityLevel )
{
    phStatus_t wStatus          = 0;
    uint8_t aCommitReaderID []  = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

    /* Clear the variables. */
    memset ( aTMC, 0x00, 4 );
    memset ( aTMV, 0x00, 8 );

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

    switch ( bSecurityLevel )
    {
        case SECURITY_LEVEL_1_MIFARE:
        case SECURITY_LEVEL_1_MFP:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( &stMfpEVx, ISO14443_L4, FIRST_AUTH, TMAC_CONFIG_KEY_1, TMAC_CONFIG_KEY_1_ADDRESS, TMAC_CONFIG_KEY_1_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        case SECURITY_LEVEL_3_MFP:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( &stMfpEVx, FIRST_AUTH, TMAC_CONFIG_KEY_1, TMAC_CONFIG_KEY_1_ADDRESS, TMAC_CONFIG_KEY_1_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        default:
            break;
    }
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Write ( &stMfpEVx, ENCRYPTED_ON, MAC_COMMAND_OFF, TMAC_COMMIT_READER_ID_1, 1, aCommitReaderID, aTMC, aTMV ) );

    return PH_ERR_SUCCESS;
}

phStatus_t ConfigureValueBlock ( uint8_t bBlockNo, uint8_t bSecurityLevel, uint8_t bISOLayer )
{
    phStatus_t wStatus      = 0;

    /* Clear the variables. */
    memset ( aTMC, 0x00, 4 );
    memset ( aTMV, 0x00, 8 );

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( bISOLayer ) );

    /* Update the address field. */
    gaValueBlock[12] = bBlockNo;
    gaValueBlock[13] = ( uint8_t ) ( ~bBlockNo );
    gaValueBlock[14] = bBlockNo;
    gaValueBlock[15] = ( uint8_t ) ( ~bBlockNo );

    /* Authenticate according to the layer option. */
    switch ( bSecurityLevel )
    {
        case SECURITY_LEVEL_1_MIFARE:
            /* Authenticate the block using MIFARE key A. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateMfc ( &stMfpEVx, bBlockNo, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len ) );
            break;

        case SECURITY_LEVEL_1_MFP:
            /* Switch sector 0 and 1 to SL1 ans SL3 mixed mode. */
            wStatus = phalMfpEVx_SSAuthenticate ( &stMfpEVx, 0x00, L1_L3_MIX_SECTOR_SWITCH_KEY, L1_L3_MIX_SECTOR_SWITCH_KEY_ADDRESS, L1_L3_MIX_SECTOR_SWITCH_KEY_VERSION,
                0, NULL, ( sizeof ( aSectorNo ) / 2 ), aSectorNo, aKeyB_Key, aKeyB_Ver, 0, NULL );

            /* Activate the card. */
            PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

            /* Perform AES Sector authentication key. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( &stMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A ( bBlockNo / 4 ), AES_KEY_ADDRESS, AES_KEY_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        case SECURITY_LEVEL_3_MFP:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( &stMfpEVx, FIRST_AUTH, GetAESSectorBlock_A ( bBlockNo / 4 ), AES_KEY_ADDRESS, AES_KEY_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;
    }

    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Write ( &stMfpEVx, ENCRYPTED_OFF, MAC_COMMAND_OFF, bBlockNo, 1, gaValueBlock, aTMC, aTMV ) );

    return PH_ERR_SUCCESS;
}

phStatus_t ConfigureTmProtectedBlock ( uint8_t bBlockNo, uint8_t bSecurityLevel )
{
    phStatus_t wStatus                          = 0;

    uint16_t bTMAC_ConfigurationBlockNr         = 0;
    uint16_t bByteNrInTmacConfigurationBlock    = 0;
    uint16_t bBitNrInTmacConfigurationBlock     = 0;
    uint8_t bTMAC_ConfigurationBlockData[]      = { 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

    /* Clear the variables. */
    memset ( aTMC, 0x00, 4 );
    memset ( aTMV, 0x00, 8 );

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

    /* Authenticate according to the layer option. */
    switch ( bSecurityLevel )
    {
        case SECURITY_LEVEL_1_MIFARE:
        case SECURITY_LEVEL_1_MFP:
            /* Switch sector 0 and 1 to SL1 ans SL3 mixed mode. */
            wStatus = phalMfpEVx_SSAuthenticate ( &stMfpEVx, 0x00, L1_L3_MIX_SECTOR_SWITCH_KEY, L1_L3_MIX_SECTOR_SWITCH_KEY_ADDRESS, L1_L3_MIX_SECTOR_SWITCH_KEY_VERSION,
                0, NULL, ( sizeof ( aSectorNo ) / 2 ), aSectorNo, aKeyB_Key, aKeyB_Ver, 0, NULL );

            /* Activate the card. */
            PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

            /* Perform authentication using TMAC config key. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( &stMfpEVx, ISO14443_L4, FIRST_AUTH, TMAC_CONFIG_KEY_1, TMAC_CONFIG_KEY_1_ADDRESS, TMAC_CONFIG_KEY_1_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        case SECURITY_LEVEL_3_MFP:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( &stMfpEVx, FIRST_AUTH, TMAC_CONFIG_KEY_1, TMAC_CONFIG_KEY_1_ADDRESS, TMAC_CONFIG_KEY_1_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;
    }

    /* Modify appropriate TMACConfigurationBlock to set TMC/TMV returned and CRI Required. */
    bTMAC_ConfigurationBlockData[0x00] = 0xAA;
    if ( bSecurityLevel != SECURITY_LEVEL_1_MIFARE )
        bTMAC_ConfigurationBlockData[0x01] = 0xAA;

    /* Update the configuration block data for CRI Required, TMC/TMV Returned flags set. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Write( &stMfpEVx, ENCRYPTED_ON, MAC_COMMAND_OFF, TMAC_CONFIG_BLOCK_1_0, 0x01, bTMAC_ConfigurationBlockData, aTMC, aTMV ) );

    /* Configuration a data block as TM Protected block. */
    bTMAC_ConfigurationBlockNr      = ( uint8_t ) ( ( bBlockNo / 128 ) + 1 );
    bByteNrInTmacConfigurationBlock = ( uint8_t ) ( bBlockNo % 128 / 8 );
    bBitNrInTmacConfigurationBlock  = ( uint8_t ) ( bBlockNo % 8 );

    /* Modify appropriate TMACConfigurationBlock to set blockNr as TMAC protected. */
    bTMAC_ConfigurationBlockData[bByteNrInTmacConfigurationBlock] = (uint8_t) (1 << bBitNrInTmacConfigurationBlock);
    bTMAC_ConfigurationBlockData[1] = 0x00;

    /* Update the configuration block data to configure a block as TMAC protected block. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Write( &stMfpEVx, ENCRYPTED_ON, MAC_COMMAND_OFF, TMAC_CONFIG_BLOCK_1_1, 0x01, bTMAC_ConfigurationBlockData, aTMC, aTMV ) );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_PrePersonalization ( uint8_t bISOLayer, uint8_t bSwitch_SL3 )
{
    phStatus_t wStatus = 0;
    uint16_t wSector = 0;

    if ( bIsEnggSample )
    {
        /* Activate to the respective ISO14443 layer. */
        PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( bISOLayer ) );

        /* Write the Card Master key. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, CARD_MASTER_KEY, 1, gaCardMasterKey ) );

        /* Write the Card configuration key. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, CARD_CONFIGURATION_KEY, 1, gaCardConfigKey ) );

        if ( !bSwitch_SL3 )
        {
            /* Write the Level 3 Switch key. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, LEVEL_3_SWITCH_KEY, 1, gaLevel3SwitchKey ) );
        }

        /* Write the SL1 Card Auth key. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, SL_1_CARD_AUTHENTICATION_KEY, 1, gaSL1CardAuthKey ) );

        /* Write the Level 3 Sector Switch key. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, LEVEL_3_SECTOR_SWITCH_KEY, 1, gaLevel3SectorSwitchKey ) );

        /* Write the Level 1, Level 3 Mix Sector Switch key. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, L1_L3_MIX_SECTOR_SWITCH_KEY, 1, gaLevel13SMixSectorSwitchKey ) );

        /* Write the VC keys. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, VC_PROXIMITY_KEY, 1, gaVcProximityKey ) );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, VC_SELECT_ENC_KEY, 1, gaVcSelectEncKey ) );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, VC_SELECT_MAC_KEY, 1, gaVcSelectMacKey ) );

        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, MFP_CONFIGURATION_BLOCK, 1, gaMfpCongifBlock ) );
        /*PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, FIELD_CONFIGURATION_BLOCK, 1, gaFieldConfigBlock ) );*/
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, ANTI_TEARING_CONFIGURATION_BLOCK, 1, gaAntiTearingConfigBlock ) );

        /* Write the transaction MAC details. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, TMAC_KEY_1, 1, gaTMAC_Key_1 ) );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, TMAC_CONFIG_KEY_1, 1, gaTMAC_ConfigKey_1 ) );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, TMAC_BLOCK_1, 1, gaTMAC_Block_1 ) );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, TMAC_COMMIT_READER_ID_1, 1, gaTMAC_Commit_Reader_ID_1 ) );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, TMAC_CONFIG_BLOCK_1_0, 1, gaTMAC_Config_Block_1_0 ) );

        /* Write the AES sector keys. */
        for ( wSector = 0; wSector < GetNumberofSectors () * 2; wSector++ )
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( &stMfpEVx, bISOLayer, (uint16_t) ( 0x4000 + wSector ), 1, gaAES_SectorKey_FF ) );

        /* Set the hardware timeout to 30ms. */
        wStatus = phhalHw_SetConfig(pHal_Generic, PHHAL_HW_CONFIG_TIMEOUT_VALUE_US, 30000);
        CHECK_SUCCESS ( wStatus );

        if ( bSwitch_SL3 )
        {
            /* Commit Perso to switch to security Level 3. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_CommitPerso ( &stMfpEVx, 0x03, bISOLayer ) );
        }
        else
        {
            /* Commit Perso to switch to security Level 1. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_CommitPerso ( &stMfpEVx, 0x01, bISOLayer ) );
        }

        /* Activate to the respective ISO14443 layer. */
        PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( bISOLayer ) );
    }

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_GetVersion ( uint8_t bSecurityLevel )
{
    phStatus_t wStatus = 0;
    uint8_t bAdditionalBytes = 0;
    uint8_t aVerInfo[33];

    /* Pre-personalize the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, ( bSecurityLevel >> 0x01 ) ) );

    /* Set the security level. */
    PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );
    bSecurityLevel = bCurrentSecurityLevel;

    /* Activate to the respective ISO14443 layer. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

    printf ( "Not Authenticated state: " );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_GetVersion ( &stMfpEVx, aVerInfo ) );
    PrintData ( aVerInfo, ( bUid_Len == 4 ) ? 27 : ( bUid_Len == 7 ) ? 28 : 33, "%02X", "\n" );

    if ( bSecurityLevel != SECURITY_LEVEL_0 )
    {
        /* Authenticate according to the layer option. */
        switch ( bSecurityLevel )
        {
            case SECURITY_LEVEL_1_MFP:
                /* Perform authentication using AES Sector. */
                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( &stMfpEVx, ISO14443_L4, FIRST_AUTH, CARD_MASTER_KEY, CARD_MASTER_KEY_ADDRESS, CARD_MASTER_KEY_VERSION,
                    0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
                break;

            case SECURITY_LEVEL_3_MFP:
                /* Update the secure messaging. */
                aPcdCap2In[0] = 0x00;

                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( &stMfpEVx, FIRST_AUTH, CARD_MASTER_KEY, CARD_MASTER_KEY_ADDRESS, CARD_MASTER_KEY_VERSION,
                    0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
                break;
        }

        printf ( "Authenticated state    : " );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_GetVersion ( &stMfpEVx, aVerInfo ) );
        PrintData ( aVerInfo, ( bUid_Len == 4 ) ? 27 : ( bUid_Len == 7 ) ? 28 : 33, "%02X", "\n" );
    }

    if ( ( bUid_Len == 4 ) || ( bUid_Len == 10 ) )
        bAdditionalBytes = 2;

    printf ( "Hardware Information ----------------------------\n" );
    printf ( "Vendor ID         : %02X\n", aVerInfo[0] );
    printf ( "Type              : %02X\n", aVerInfo[1] );
    printf ( "Sub Type          : %02X\n", aVerInfo[2] );
    printf ( "Major Version     : %02X\n", aVerInfo[3] );
    printf ( "Minor Version     : %02X\n", aVerInfo[4] );
    printf ( "Storage Size      : %02X\n", aVerInfo[5] );
    printf ( "Protocol          : %02X\n", aVerInfo[6] );
    printf ( "\n" );

    printf ( "Software Information ----------------------------\n" );
    printf ( "Vendor ID         : %02X\n", aVerInfo[7] );
    printf ( "Type              : %02X\n", aVerInfo[8] );
    printf ( "Sub Type          : %02X\n", aVerInfo[9] );
    printf ( "Major Version     : %02X\n", aVerInfo[10] );
    printf ( "Minor Version     : %02X\n", aVerInfo[11] );
    printf ( "Storage Size      : %02X\n", aVerInfo[12] );
    printf ( "Protocol          : %02X\n", aVerInfo[13] );
    printf ( "\n" );

    printf ( "Product Information ----------------------------\n" );
    printf ( "UID               : " );
    PrintData ( &aVerInfo[14 + bAdditionalBytes], bUid_Len, "%02X", "\n" );

    printf ( "Batch Number      : " );
    PrintData ( &aVerInfo[14 + bUid_Len + bAdditionalBytes], 5, "%02X", "\n" );

    printf ( "CW Production     : %02X\n", aVerInfo[14 + bUid_Len + 5 + bAdditionalBytes] );
    printf ( "Year of Production: %02X\n", aVerInfo[14 + bUid_Len + 5 + 1 + bAdditionalBytes] );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_TransactionMAC(uint8_t bSecurityLevel)
{
    phStatus_t wStatus = 0;
    uint32_t dwTMI_Len = 0;
    uint8_t bAddress = 0;
    uint8_t bData_BlockNo = 0x04;
    uint8_t bValue_BlockNo = 0x05;

    uint8_t aTMRIPrev[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    uint8_t aEncTMRI[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

    uint8_t aTx_Data[] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10 };

    uint8_t aRx_Value[] = { 0x00, 0x00, 0x00, 0x00 };
    uint8_t aTx_Value[] = { 0x00, 0x00, 0x00, 0x00 };

    uint8_t* pTMI = NULL;

    /* Pre-personalize the card. */
    PH_CHECK_SUCCESS_FCT(wStatus, Demo_PrePersonalization(ISO14443_L4, (bSecurityLevel >> 0x01)));

    /* Set the security level. */
    PH_CHECK_SUCCESS_FCT(wStatus, SetSecurityLevel());

    printf("\tTMAC feature on Data block %04X ----------------------------------------------\n", bData_BlockNo);

    /* Configure data block as TM protected block. */
    PH_CHECK_SUCCESS_FCT(wStatus, ConfigureTmProtectedBlock(bData_BlockNo, bSecurityLevel));

    /* Enable TMI Collection. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_RESET_TMI));
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_ACTIVATE_TMI));

    /* Authenticate according to the layer option. */
    switch(bSecurityLevel)
    {
        case SECURITY_LEVEL_1_MIFARE:
            /* Activate the card. */
            PH_CHECK_SUCCESS_FCT(wStatus, ActivateCard(ISO14443_L3));

            /* Authenticate the block using MIFARE key A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateMfc(&stMfpEVx, bData_BlockNo, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len));
            break;

        case SECURITY_LEVEL_1_MFP:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateSL1(&stMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A((uint16_t) (bData_BlockNo / 4)),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2));
            break;

        case SECURITY_LEVEL_3_MFP:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateSL3(&stMfpEVx, FIRST_AUTH, GetAESSectorBlock_A((uint16_t) (bData_BlockNo / 4)),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2));
            break;
    }

    if(bSecurityLevel != SECURITY_LEVEL_1_MIFARE)
    {
        gaTMAC_Commit_Reader_ID_1[0]++;
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_CommitReaderID(&stMfpEVx, bData_BlockNo, gaTMAC_Commit_Reader_ID_1, aEncTMRI));

        /* Update TMC. */
        UpdateTMC(bSecurityLevel);

        /* Decrypt Reader ID. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_DecryptReaderID(&stMfpEVx, 0xFFFF, TMAC_KEY_1_ADDRESS, TMAC_CONFIG_KEY_1_VERSION, 0, 0, NULL,
            0, aTMC, aUid, bUid_Len, aEncTMRI, aTMRIPrev));

        printf("\t\tCommitReaderID (TMRICur)          : ");
        PrintData(gaTMAC_Commit_Reader_ID_1, sizeof(gaTMAC_Commit_Reader_ID_1), "%02X ", "\n");

        printf("\t\tEncrypted TMRI (EncTMRI)          : ");
        PrintData(aEncTMRI, sizeof(aEncTMRI), "%02X ", "\n");

        printf("\t\tDecrypted Reader ID (TMRIPrev)    : ");
        PrintData(aTMRIPrev, sizeof(aTMRIPrev), "%02X ", "\n");
    }

    /* Write the data to the block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Write(&stMfpEVx, ENCRYPTED_ON, MAC_COMMAND_OFF, bData_BlockNo, 0x01, aTx_Data, aTMC, aTMV));

    /* Get the collected TMI data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetTMI(&stTMI, &pTMI, &dwTMI_Len));

    /* Read the data from the block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Read(&stMfpEVx, ENCRYPTED_OFF, MAC_RESPONSE_OFF, MAC_COMMAND_ON, bData_BlockNo, 1, aRx_Data));

    printf("\t\tTx Data                           : ");
    PrintData(aTx_Data, sizeof(aTx_Data), "%02X ", "\n");

    printf("\t\tRx Data                           : ");
    PrintData(aRx_Data, sizeof(aRx_Data), "%02X ", "\n");

    printf("\t\tTransaction MAC Counter (TMC)     : ");
    PrintData(aTMC, sizeof(aTMC), "%02X ", "\n");

    printf("\t\tTransaction MAC Value (TMV)       : ");
    PrintData(aTMV, sizeof(aTMV), "%02X ", "\n");

    printf("\t\tTransaction MAC Input (TMI)       : ");
    PrintData(pTMI, dwTMI_Len, "%02X ", "\n");

    /* Update TMC. */
    UpdateTMC(bSecurityLevel);

    /* Calculate TMV with calculated TMI and TMC. */
    memset(aTMV, 0x00, sizeof(aTMV));
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_CalculateTMV(&stMfpEVx, 0xFFFF, TMAC_KEY_1_ADDRESS, TMAC_KEY_1_VERSION, 0, 0,
        NULL, 0, aTMC, aUid, bUid_Len, pTMI, (uint8_t) dwTMI_Len, aTMV));

    printf("\t\tCalculated TMV                    : ");
    PrintData(aTMV, sizeof(aTMV), "%02X ", "\n");
    printf("\n");

    /* Deactivate TMI Collection. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_PAUSE_TMI));

    printf("\tTMAC feature on Value block %04X ----------------------------------------------\n", bValue_BlockNo);

    /* Clear the buffers. */
    memset(aTMV, 0x00, sizeof(aTMV));
    memset(aTMRIPrev, 0x00, sizeof(aTMRIPrev));
    memset(aEncTMRI, 0x00, sizeof(aEncTMRI));

    /* Configure a block as value block. */
    PH_CHECK_SUCCESS_FCT(wStatus, ConfigureValueBlock(bValue_BlockNo, bSecurityLevel, (bSecurityLevel == SECURITY_LEVEL_1_MIFARE) ? ISO14443_L3 : ISO14443_L4));

    /* Configure data block as TM protected block. */
    PH_CHECK_SUCCESS_FCT(wStatus, ConfigureTmProtectedBlock(bValue_BlockNo, bSecurityLevel));

    /* Enable TMI Collection. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_RESET_TMI));
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_ACTIVATE_TMI));

    /* Authenticate according to the layer option. */
    switch(bSecurityLevel)
    {
        case SECURITY_LEVEL_1_MIFARE:
            /* Activate the card. */
            PH_CHECK_SUCCESS_FCT(wStatus, ActivateCard(ISO14443_L3));

            /* Authenticate the block using MIFARE key A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateMfc(&stMfpEVx, bValue_BlockNo, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len));
            break;

        case SECURITY_LEVEL_1_MFP:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateSL1(&stMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A((uint16_t) (bValue_BlockNo / 4)),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2));
            break;

        case SECURITY_LEVEL_3_MFP:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateSL3(&stMfpEVx, FIRST_AUTH, GetAESSectorBlock_A((uint16_t) (bValue_BlockNo / 4)), AES_KEY_ADDRESS, AES_KEY_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2));
            break;
    }

    if(bSecurityLevel != SECURITY_LEVEL_1_MIFARE)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_CommitReaderID(&stMfpEVx, bValue_BlockNo, gaTMAC_Commit_Reader_ID_1, aEncTMRI));

        /* Update TMC. */
        UpdateTMC(bSecurityLevel);

        /* Decrypt Reader ID. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_DecryptReaderID(&stMfpEVx, 0xFFFF, TMAC_KEY_1_ADDRESS, TMAC_CONFIG_KEY_1_VERSION, 0, 0,
            NULL, 0, aTMC, aUid, bUid_Len, aEncTMRI, aTMRIPrev));

        printf("\t\tCommitReaderID (TMRICur)          : ");
        PrintData(gaTMAC_Commit_Reader_ID_1, sizeof(gaTMAC_Commit_Reader_ID_1), "%02X ", "\n");

        printf("\t\tEncrypted TMRI (EncTMRI)          : ");
        PrintData(aEncTMRI, sizeof(aEncTMRI), "%02X ", "\n");

        printf("\t\tDecrypted Reader ID (TMRIPrev)    : ");
        PrintData(aTMRIPrev, sizeof(aTMRIPrev), "%02X ", "\n");
    }

    /* Increment transfer the data to the block. */
    aTx_Value[0] = 0x01;
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_IncrementTransfer(&stMfpEVx, MAC_COMMAND_OFF, bValue_BlockNo, bValue_BlockNo, aTx_Value, aTMC, aTMV));
    printf("\t\t(IncrementTransfer Value) Tx Value: ");
    PrintData(aTx_Value, sizeof(aTx_Value), "%02X ", "\n");

    /* Get the collected TMI data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetTMI(&stTMI, &pTMI, &dwTMI_Len));

    /* Read the value written from the block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_ReadValue(&stMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bValue_BlockNo, aRx_Value, &bAddress));

    printf("\t\t(Read Value) Rx Value             : ");
    PrintData(aRx_Value, sizeof(aRx_Value), "%02X ", "\n");

    printf("\t\tTransaction MAC Counter (TMC)     : ");
    PrintData(aTMC, sizeof(aTMC), "%02X ", "\n");

    printf("\t\tTransaction MAC Value (TMV)       : ");
    PrintData(aTMV, sizeof(aTMV), "%02X ", "\n");

    printf("\t\tTransaction MAC Input (TMI)       : ");
    PrintData(pTMI, dwTMI_Len, "%02X ", "\n");

    /* Update TMC. */
    UpdateTMC(bSecurityLevel);

    /* Calculate TMV with calculated TMI and TMC. */
    memset(aTMV, 0x00, sizeof(aTMV));
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_CalculateTMV(&stMfpEVx, 0xFFFF, TMAC_KEY_1_ADDRESS, TMAC_KEY_1_VERSION, 0, 0,
        NULL, 0, aTMC, aUid, bUid_Len, pTMI, (uint8_t) dwTMI_Len, aTMV));

    printf("\t\tCalculated TMV                    : ");
    PrintData(aTMV, sizeof(aTMV), "%02X ", "\n");
    printf("\n");

    /* Deactivate TMI Collection. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_PAUSE_TMI));

    /* Enable TMI Collection. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_RESET_TMI));
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_ACTIVATE_TMI));

    /* Authenticate according to the layer option. */
    switch(bSecurityLevel)
    {
        case SECURITY_LEVEL_1_MIFARE:
            /* Activate L3 */
            ActivateCard(ISO14443_L3);

            /* Authenticate the block using MIFARE key A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateMfc(&stMfpEVx, bValue_BlockNo, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len));
            break;

        case SECURITY_LEVEL_1_MFP:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateSL1(&stMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A((uint16_t) (bValue_BlockNo / 4)),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2));
            break;

        case SECURITY_LEVEL_3_MFP:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateSL3(&stMfpEVx, FIRST_AUTH, GetAESSectorBlock_A((uint16_t) (bValue_BlockNo / 4)),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2));
            break;
    }

    if(bSecurityLevel != SECURITY_LEVEL_1_MIFARE)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_CommitReaderID(&stMfpEVx, bValue_BlockNo, gaTMAC_Commit_Reader_ID_1, aEncTMRI));

        /* Update TMC. */
        UpdateTMC(bSecurityLevel);

        /* Decrypt Reader ID. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_DecryptReaderID(&stMfpEVx, 0xFFFF, TMAC_KEY_1_ADDRESS, TMAC_CONFIG_KEY_1_VERSION, 0, 0,
            NULL, 0, aTMC, aUid, bUid_Len, aEncTMRI, aTMRIPrev));
    }

    /* Decrement transfer the data to the block. */
    aTx_Value[0] = 0x01;
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_DecrementTransfer(&stMfpEVx, MAC_COMMAND_OFF, bValue_BlockNo, bValue_BlockNo, aTx_Value, aTMC, aTMV));
    printf("\t\t(DecrementTransfer Value) Tx Value: ");
    PrintData(aTx_Value, sizeof(aTx_Value), "%02X ", "\n");

    /* Get the collected TMI data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetTMI(&stTMI, &pTMI, &dwTMI_Len));

    /* Read the value written from the block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_ReadValue(&stMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bValue_BlockNo, aRx_Value, &bAddress));

    printf("\t\t(Read Value) Rx Value             : ");
    PrintData(aRx_Value, sizeof(aRx_Value), "%02X ", "\n");

    printf("\t\tTransaction MAC Counter (TMC)     : ");
    PrintData(aTMC, sizeof(aTMC), "%02X ", "\n");

    printf("\t\tTransaction MAC Value (TMV)       : ");
    PrintData(aTMV, sizeof(aTMV), "%02X ", "\n");

    printf("\t\tTransaction MAC Input (TMI)       : ");
    PrintData(pTMI, dwTMI_Len, "%02X ", "\n");

    /* Update TMC. */
    UpdateTMC(bSecurityLevel);

    /* Calculate TMV with calculated TMI and TMC. */
    memset(aTMV, 0x00, sizeof(aTMV));
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_CalculateTMV(&stMfpEVx, 0xFFFF, TMAC_KEY_1_ADDRESS, TMAC_KEY_1_VERSION, 0, 0,
        NULL, 0, aTMC, aUid, bUid_Len, pTMI, (uint8_t) dwTMI_Len, aTMV));

    printf("\t\tCalculated TMV                    : ");
    PrintData(aTMV, sizeof(aTMV), "%02X ", "\n");

    /* Deactivate TMI Collection. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(&stTMI, PH_TMIUTILS_PAUSE_TMI));

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_OriginalityCheck(uint8_t bSecurityLevel)
{
    phStatus_t wStatus = 0;
    uint16_t wOption = 0;
    uint8_t * pSignature = NULL;

    /* Pre-personalize the card. */
    PH_CHECK_SUCCESS_FCT(wStatus, Demo_PrePersonalization(ISO14443_L4, (bSecurityLevel >> 0x01)));

    /* Set the security level. */
    PH_CHECK_SUCCESS_FCT(wStatus, SetSecurityLevel());
    bSecurityLevel = bCurrentSecurityLevel;

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT(wStatus, ActivateCard(ISO14443_L4));

    /* Read the signature from the PICC */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_ReadSign(&stMfpEVx, ISO14443_L4, 0x00, &pSignature));

    printf("\tASymmetric Signature\n");
    printf("\t\tPublic Key: ");
    PrintData(gaPublicKey, (uint32_t) sizeof(gaPublicKey), "%02X", "\n");
    printf("\t\tSignature : ");
    PrintData(pSignature, 56, "%02X", "\n");

    /* Load the Public Key. */
    wOption = (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | PH_CRYPTOASYM_CURVE_ID_SECP224R1);
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKeyDirect(&stCryptoASym, wOption, gaPublicKey, (uint16_t) sizeof(gaPublicKey)));

    /* Signature verification for the UID sent. */
    printf("\t\tECDSA verification of Asymmetric Originality Check ");
    wStatus = phCryptoASym_ECC_Verify(&stCryptoASym, PH_EXCHANGE_DEFAULT, PH_CRYPTOASYM_HASH_ALGO_NOT_APPLICABLE, aUid, bUid_Len, pSignature,
        (uint16_t) sizeof(pSignature));
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        printf("Passed\n");
    else
        printf("Failed\n");
    printf("\n");

    printf("\tSymmetric Signature\n");
    printf("\t\tBlock No : %04X\n", ORIGINALITY_KEY_1);

    /* Authenticate with originality key 1. */
    switch(bSecurityLevel)
    {
        case SECURITY_LEVEL_1_MFP:
            /* Perform authentication using AES Sector. */
            wStatus = phalMfpEVx_AuthenticateSL1(&stMfpEVx, ISO14443_L4, FIRST_AUTH, ORIGINALITY_KEY_1, ORIGINALITY_KEY_1_ADDRESS, ORIGINALITY_KEY_1_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2);
            break;

        case SECURITY_LEVEL_3_MFP:
            /* Update the secure messaging. */
            aPcdCap2In[0] = 0x00;

            wStatus = phalMfpEVx_AuthenticateSL3(&stMfpEVx, FIRST_AUTH, ORIGINALITY_KEY_1, ORIGINALITY_KEY_1_ADDRESS, ORIGINALITY_KEY_1_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2);
            break;
    }

    /* Signature verification for the UID sent. */
    printf("\t\tAuthentication using Symmetric Originality block 0x8000 ");
    if(!wStatus)
        printf("Passed\n");
    else
        printf("Failed\n");
    printf("\n");

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_ISO7816 ( uint8_t bSecurityLevel, uint8_t bISOLayer, uint8_t bSecureMessaging, uint8_t bExtended )
{
    phStatus_t wStatus  = 0;

    /* Enable Wrapped mode support. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( &stMfpEVx, PHAL_MFPEVX_WRAPPED_MODE, 0x01 ) );

    /* Enable Extended Length APDU support. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( &stMfpEVx, PHAL_MFPEVX_EXTENDED_APDU, bExtended ) );

    /* Perform layer operation. */
    PH_CHECK_SUCCESS_FCT ( wStatus, Demo_SecurityLayer ( bSecurityLevel, bISOLayer, bSecureMessaging, 0x01 ) );

    /* Disable Wrapped mode support. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( &stMfpEVx, PHAL_MFPEVX_WRAPPED_MODE, 0x00 ) );

    /* Disable Extended Length APDU support. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( &stMfpEVx, PHAL_MFPEVX_EXTENDED_APDU, 0x00 ) );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_SecurityLayer(uint8_t bSecurityLevel, uint8_t bISOLayer, uint8_t bSecureMessaging, uint8_t bPrepersonalize)
{
    phStatus_t wStatus = 0;
    uint8_t bData_BlockNo = 0x01;
    uint8_t bValue_BlockNo = 0x02;
    uint8_t bAddress = 0;

    uint8_t aTx_Data[] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10 };

    uint8_t aRx_Value[] = { 0x00, 0x00, 0x00, 0x00 };
    uint8_t aTx_Value[] = { 0x00, 0x00, 0x00, 0x00 };

    /* Clear the variables. */
    memset(aTMC, 0x00, 4);
    memset(aTMV, 0x00, 8);

    if(bPrepersonalize)
    {
        /* Pre-personalize the card. */
        PH_CHECK_SUCCESS_FCT(wStatus, Demo_PrePersonalization(ISO14443_L4, (bSecurityLevel >> 0x01)));

        /* Set the security level. */
        PH_CHECK_SUCCESS_FCT(wStatus, SetSecurityLevel());
    }

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT(wStatus, ActivateCard(bISOLayer));

    /* Configure a block as value block. */
    PH_CHECK_SUCCESS_FCT(wStatus, ConfigureValueBlock(bValue_BlockNo, bSecurityLevel, bISOLayer));

    printf("\tNon TMAC feature on Data block %04X ------------------------------------------\n", bData_BlockNo);

    /* Authenticate according to the layer option. */
    switch(bSecurityLevel)
    {
        case SECURITY_LEVEL_1_MIFARE:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateMfc(&stMfpEVx, bData_BlockNo, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len));
            break;

        case SECURITY_LEVEL_1_MFP:
            /* Perform authentication using AES Sector. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateSL1(&stMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A((uint16_t) (bData_BlockNo / 4)),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2));
            break;

        case SECURITY_LEVEL_3_MFP:
            /* Update the secure messaging. */
            aPcdCap2In[0] = bSecureMessaging;

            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_AuthenticateSL3(&stMfpEVx, FIRST_AUTH, GetAESSectorBlock_A((uint16_t) (bData_BlockNo / 4)),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2));
            break;
    }

    /* Write a random data to the block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Write(&stMfpEVx, ENCRYPTED_OFF, MAC_COMMAND_OFF, bData_BlockNo, 1, aTx_Data, aTMC, aTMV));
    printf("\t\t(Write Data) Tx Value             : ");
    PrintData(aTx_Data, sizeof(aTx_Data), "%02X ", "\n");

    /* Read the data written from the block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Read(&stMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bData_BlockNo, 1, aRx_Data));
    printf("\t\t(Read Data) Rx Value              : ");
    PrintData(aRx_Data, sizeof(aRx_Data), "%02X ", "\n");
    printf("\n");

    printf("\tNon TMAC feature on Value block %04X -----------------------------------------\n", bValue_BlockNo);

    /* Increment transfer the data to the block. */
    aTx_Value[0] = 0x01;
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_IncrementTransfer(&stMfpEVx, MAC_COMMAND_OFF, bValue_BlockNo, bValue_BlockNo, aTx_Value, aTMC, aTMV));
    printf("\t\t(IncrementTransfer Value) Tx Value: ");
    PrintData(aTx_Value, sizeof(aTx_Value), "%02X ", "\n");

    /* Read the value written from the block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_ReadValue(&stMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bValue_BlockNo, aRx_Value, &bAddress));
    printf("\t\t(Read Value) Rx Value             : ");
    PrintData(aRx_Value, sizeof(aRx_Value), "%02X ", "\n");

    /* Decrement transfer the data to the block. */
    aTx_Value[0] = 0x01;
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_DecrementTransfer(&stMfpEVx, MAC_COMMAND_OFF, bValue_BlockNo, bValue_BlockNo, aTx_Value, aTMC, aTMV));
    printf("\t\t(DecrementTransfer Value) Tx Value: ");
    PrintData(aTx_Value, sizeof(aTx_Value), "%02X ", "\n");

    /* Read the value written from the block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_ReadValue(&stMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bValue_BlockNo, aRx_Value, &bAddress));
    printf("\t\t(Read Value) Rx Value             : ");
    PrintData(aRx_Value, sizeof(aRx_Value), "%02X ", "\n");

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_VirtualCardDemo ()
{
    phStatus_t wStatus;

    uint8_t bSelector           = 0;
    uint8_t bOption             = 0;
    uint8_t bExtendedLenApdu    = 0;
    uint16_t wFCILen            = 0;

    uint8_t aFCI[40];
    uint8_t aRandChal[16];
    uint8_t aVCData[16];

    /* Pre-Personalize the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, bCurrentSecurityLevel >> 0x02 ) );

    /* Set the current security level. */
    PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

    /* ISO Select for Virtual Card */
    bOption = 0x00;             /* bOption == 0x00 FCI is returned. If 0x0C no FCI returned */
    bSelector = 0x04;           /* bSelector equals 0x04 for selecting the virtual card only with ISO DF Name */
    bExtendedLenApdu = 0x00;    /* Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or any other value for Short APDUs. */

    memset ( aFCI, 0, sizeof ( aFCI ) );

    printf ( "IsoSelect              : " );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_IsoSelect ( &stVca, bSelector, bOption, sizeof ( gaVCIID ), gaVCIID, NULL, 0x00, 0x00, 0x00, 0x00, 0x00, aFCI, &wFCILen ) );
    printf ( "Success\n" );

    /* AuthVCMandatory is true so IsoExternalAuthenticate is compulsory and if AuthVCMandatory is false then IsoExternalAuthenticate is not required */
    if( wFCILen == VCA_AUTH_RND_LEN )
    {
        /* Authenticating the Selected Virtual Card */
        /* Decrypting the response data got from ISOSelect API */
        printf ( "Decryption of Response : " );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_DecryptResponse ( &stVca, VC_SELECT_ENC_KEY_ADDRESS, VC_SELECT_ENC_KEY_VERSION, aFCI, aRandChal, aVCData ) );
        printf ( "Success\n" );

        memcpy ( &aFCI[16], aVCData, 16 );

        printf ( "IsoExternalAuthenticate: " );
        PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_IsoExternalAuthenticate ( &stVca, aFCI, VC_SELECT_MAC_KEY_ADDRESS, VC_SELECT_MAC_KEY_VERSION ) );
        printf ( "Success\n" );
    }
    printf ( "\n" );

    if ( bCurrentSecurityLevel != SECURITY_LEVEL_0 )
        PH_CHECK_SUCCESS_FCT ( wStatus, Demo_SecurityLayer ( ( bCurrentSecurityLevel == SECURITY_LEVEL_0 ) ? 0x00 :
                    ( bCurrentSecurityLevel == SECURITY_LEVEL_1 ) ? SECURITY_LEVEL_1_MFP : SECURITY_LEVEL_3_MFP, ISO14443_L4, EV0_SECURE_MESSAGING, 0x00 ) );

   return wStatus;
}

phStatus_t Demo_ProximityCheck ()
{
    phStatus_t wStatus;
    uint8_t     bGenerateRndC = 0x01;
    uint8_t     aRndC[8];
    uint8_t     bOption = 0;
    uint8_t     aPubRespTime[2];
    uint8_t     aResponse[30];
    uint16_t    wRespLen = 0;
    uint8_t     bNumSteps = 0x08;
    uint8_t     aCumRndC[16];

    /* Pre-Personalize the card and switch to security level 1. */
    PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, bCurrentSecurityLevel >> 0x02 ) );

    /* Set the current security level. */
    PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

    printf ( "Proximity Check in Not Authenticated state ------------------------------------------\n" );

    printf ( "\tPerform Prepare PC\n" );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_PrepareProximityCheckNew ( &stVca, &bOption, aPubRespTime, aResponse, &wRespLen ) );
    printf ( "\t\tStatus                 : Success\n" );
    printf ( "\t\tPublished Response Time: " );
    PrintData ( aPubRespTime, sizeof ( aPubRespTime ), "%02X ", "\n" );
    if ( bOption & 0x01 )
        printf ( "\t\tPPs1                   : %02X\n", aResponse[0] );
    else
    {
        printf ( "\t\tActBitRate             : ");
        PrintData ( aResponse, wRespLen, "%02X", "");
    }
    printf ( "\n" );

    printf ( "\tPerform ProximityCheck\n" );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_ExecuteProximityCheckNew ( &stVca, bGenerateRndC, aRndC, bNumSteps, aPubRespTime, aCumRndC ) );
    printf ( "\t\tStatus                 : Success\n" );
    printf ( "\t\tPCD Random number      : " );
    PrintData ( aCumRndC, 8, "%02X ", "\n" );
    printf ( "\t\tPD Random number       : " );
    PrintData ( &aCumRndC[8], 8, "%02X ", "\n" );
    printf ( "\n" );

    printf ( "\tPerform Verify PC\n" );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_VerifyProximityCheckNew ( &stVca, bOption, aPubRespTime, aResponse, wRespLen, VC_PROXIMITY_KEY_ADDRESS, VC_PROXIMITY_KEY_VERSION, aCumRndC ) );
    printf ( "\t\tStatus                 : Success\n" );
    printf ( "\n" );

    printf ( "Proximity Check in Authenticated state ----------------------------------------------\n" );

    /* Authenticate according to the layer option. */
    printf ( "\tAuthentication using AES Sector Keys\n" );
    switch ( bCurrentSecurityLevel )
    {
        case SECURITY_LEVEL_1_MFP:
            /* Perform authentication using AES Sector. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( &stMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A ( 0x00 ), AES_KEY_ADDRESS, AES_KEY_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        case SECURITY_LEVEL_3_MFP:
            /* Update the secure messaging. */
            aPcdCap2In[0] = EV0_SECURE_MESSAGING;

            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( &stMfpEVx, FIRST_AUTH, GetAESSectorBlock_A ( 0x00 ), AES_KEY_ADDRESS, AES_KEY_VERSION,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;
    }
    printf ( "\t\tStatus                 : Success\n\n" );

    printf ( "\tPerform Prepare PC\n" );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_PrepareProximityCheckNew ( &stVca, &bOption, aPubRespTime, aResponse, &wRespLen ) );
    printf ( "\t\tStatus                 : Success\n" );
    printf ( "\t\tPublished Response Time: " );
    PrintData ( aPubRespTime, sizeof ( aPubRespTime ), "%02X ", "\n" );
    if ( bOption & 0x01 )
        printf ( "\t\tPPs1                   : %02X\n", aResponse[0] );
    else
    {
        printf ( "\t\tActBitRate             : ");
        PrintData ( aResponse, wRespLen, "%02X", "");
    }
    printf ( "\n" );

    printf ( "\tPerform ProximityCheck\n" );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_ExecuteProximityCheckNew ( &stVca, bGenerateRndC, aRndC, bNumSteps, aPubRespTime, aCumRndC ) );
    printf ( "\t\tStatus                 : Success\n" );
    printf ( "\t\tPCD Random number      : " );
    PrintData ( aCumRndC, 8, "%02X ", "\n" );
    printf ( "\t\tPD Random number       : " );
    PrintData ( &aCumRndC[8], 8, "%02X ", "\n" );
    printf ( "\n" );

    printf ( "\tPerform Verify PC\n" );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalVca_VerifyProximityCheckNew ( &stVca, bOption, aPubRespTime, aResponse, wRespLen, AES_KEY_ADDRESS, AES_KEY_VERSION, aCumRndC ) );
    printf ( "\t\tStatus                 : Success\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_UidPersonalization ()
{
    phStatus_t wStatus          = 0;

    uint8_t bChoice             = 0;
    uint8_t bNVBUid             = 0;
    uint8_t aAtqA[2]            = {0x00, 0x00 };

    printf ( "Press 1 to perform UID Personalization using Option 0 (UIDF0)\n" );
    printf ( "Press 2 to perform UID Personalization using Option 1 (UIDF1)\n" );
    printf ( "Press 3 to perform UID Personalization using Option 2 (UIDF2)\n" );
    printf ( "Press 4 to perform UID Personalization using Option 3 (UIDF3)\n" );
    printf ( "Enter your choice - " );
    scanf ( "%c", &bChoice );

    printf ( "\n" );

    if ( ( bChoice != 'x' ) || ( bChoice != 'X' ) )
    {
        switch ( bChoice )
        {
            case '1':
                /* UID Personalization with Option 0 (UIDF0) --------------------------------------------------------------------------------------------------- */
                printf ( "\tUID Personalization with UIDF0 option -------------------------------------------\n" );

                /* Pre-Personalize the card and switch to security level 1. */
                PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, SECURITY_LEVEL_1_MIFARE ) );

                /* Set the current security level. */
                PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

                printf ( "\t\tActivate Iso14443 Layer3                 : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L3 ) );
                printf ( "Success\n" );

                /* Perform MIFARE Classic authentication. */
                printf ( "\t\tMifareClassic authentication             : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateMfc ( &stMfpEVx, 0x00, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len ) );
                printf ( "Success\n" );

                /* Personalize the UID with UIDF0 option.  */
                printf ( "\t\tUID Personalization with Option 0        : " );
                wStatus = phalMfpEVx_PersonalizeUid ( &stMfpEVx, PHAL_MFPEVX_UID_TYPE_UIDF0 );
                if ( wStatus != PH_ERR_SUCCESS )
                {
                    printf ( "\n\t\tUID Personalization with Option 0 (UIDF0) is not possible because the PICC is locked.\n\n" );
                    return wStatus;
                }
                else
                {
                    printf ( "Success\n" );

                    printf ( "\t\tUID                                      : " );
                    PrintData ( aUid, bUid_Len, "%02X", "\n\n" );
                }
                break;

            case '2':
                /* UID Personalization with Option 1 (UIDF1) --------------------------------------------------------------------------------------------------- */
                printf ( "\tUID Personalization with UIDF1 option ------------------------------------------- \n" );

                /* Restore to SL0. */
                PH_CHECK_SUCCESS_FCT ( wStatus, RestoreToSL0 () );

                /* Set the current security level of the card. */
                PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

                /* Reset the authentication state. */
                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_ResetSecMsgState ( &stMfpEVx ) );

                /* Pre-Personalize the card and switch to security level 1. */
                PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, SECURITY_LEVEL_1_MIFARE ) );

                /* Set the current security level. */
                PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

                printf ( "\t\tActivate Iso14443 Layer3                 : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L3 ) );
                printf ( "Success\n" );

                /* Perform MIFARE Classic authentication. */
                printf ( "\t\tMifareClassic authentication             : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateMfc ( &stMfpEVx, 0x00, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len ) );
                printf ( "Success\n" );

                /* Personalize the UID with UIDF0 option.  */
                printf ( "\t\tUID Personalization with Option 1        : " );
                wStatus = phalMfpEVx_PersonalizeUid ( &stMfpEVx, PHAL_MFPEVX_UID_TYPE_UIDF1 );
                if ( wStatus != PH_ERR_SUCCESS )
                {
                    printf ( "\n\t\tUID Personalization with Option 1 (UIDF1) is not possible because the PICC is locked.\n\n" );
                    return wStatus;
                }
                else
                {
                    printf ( "Success\n" );

                    printf ( "\t\tUID                                      : " );
                    PrintData ( aUid, bUid_Len, "%02X", "\n\n" );
                }

                printf ( "\t\tShortcut selection demo ------------------------------------------------- \n" );
                /* Reset the field */
                PH_CHECK_SUCCESS_FCT ( wStatus, phhalHw_FieldReset ( pHal_Generic ) );

                /* Perform ReqA */
                printf ( "\t\t\t\tRequest A                : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p3a_RequestA ( &stI14443p3a, aAtqA ) );
                printf ( "Success\n" );

                /* Perform Anti-Collision with cascade 1. */
                printf ( "\t\t\t\tCascade 1 Anti-Collision : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p3a_Anticollision ( &stI14443p3a, 0x93, NULL, 0, aUid, &bNVBUid ) );
                printf ( "Success\n" );

                /* Perform Select. */
                printf ( "\t\t\t\tSelect                   : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p3a_Select ( &stI14443p3a, 0x93, aUid, &bSak ) );
                printf ( "Success\n\n" );

                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( &stMfpEVx, PHAL_MFPEVX_AUTH_MODE, PHAL_MFPEVX_NOT_AUTHENTICATED_L3 ) );

                /* Read the data written from the block 0. */
                //printf ( "\t\t\t\tRead                     : " );
                //wStatus = phalMfpEVx_Read ( &stMfpEVx, ENCRYPTED_OFF, MAC_RESPONSE_OFF, MAC_COMMAND_OFF, 0x00, 1, aRx_Data );
                //PrintData ( aRx_Data, sizeof ( aRx_Data ), "%02X ", "\n\n" );
                break;

            case '3':
                /* UID Personalization with Option 2 (UIDF2) --------------------------------------------------------------------------------------------------- */
                printf ( "\tUID Personalization with UIDF2 option ------------------------------------------- \n" );

                /* Restore to SL0. */
                PH_CHECK_SUCCESS_FCT ( wStatus, RestoreToSL0 () );

                /* Set the current security level of the card. */
                PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

                /* Reset the authentication state. */
                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_ResetSecMsgState ( &stMfpEVx ) );

                /* Pre-Personalize the card and switch to security level 1. */
                PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, SECURITY_LEVEL_1_MIFARE ) );

                /* Set the current security level. */
                PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

                printf ( "\t\tActivate Iso14443 Layer3                 : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L3 ) );
                printf ( "Success\n" );

                printf ( "\t\tUID                                      : " );
                PrintData ( aUid, bUid_Len, "%02X", "\n" );

                /* Perform MIFARE Classic authentication. */
                printf ( "\t\tMifareClassic authentication             : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateMfc ( &stMfpEVx, 0x00, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len ) );
                printf ( "Success\n" );

                /* Personalize the UID with UIDF0 option.  */
                printf ( "\t\tUID Personalization with Option 1        : " );
                wStatus = phalMfpEVx_PersonalizeUid ( &stMfpEVx, PHAL_MFPEVX_UID_TYPE_UIDF2 );
                if ( wStatus != PH_ERR_SUCCESS )
                {
                    printf ( "\n\t\tUID Personalization with Option 2 (UIDF2) is not possible because the PICC is locked.\n\n" );
                    return wStatus;
                }
                else
                {
                    printf ( "Success\n" );

                    printf ( "\t\tActivate Iso14443 Layer3                 : " );
                    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L3 ) );
                    printf ( "Success\n" );

                    printf ( "\t\tUID                                      : " );
                    PrintData ( aUid, bUid_Len, "%02X", "\n\n" );
                }
                break;

            case '4':
                /* UID Personalization with Option 3 (UIDF3) --------------------------------------------------------------------------------------------------- */
                printf ( "\tUID Personalization with UIDF3 option ------------------------------------------- \n" );

                /* Restore to SL0. */
                PH_CHECK_SUCCESS_FCT ( wStatus, RestoreToSL0 () );

                /* Set the current security level of the card. */
                PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

                /* Reset the authentication state. */
                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_ResetSecMsgState ( &stMfpEVx ) );

                /* Pre-Personalize the card and switch to security level 1. */
                PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, SECURITY_LEVEL_1_MIFARE ) );

                /* Set the current security level. */
                PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

                printf ( "\t\tActivate Iso14443 Layer3                 : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L3 ) );
                printf ( "Success\n" );

                printf ( "\t\tUID                                      : " );
                PrintData ( aUid, bUid_Len, "%02X", "\n" );

                /* Perform MIFARE Classic authentication. */
                printf ( "\t\tMifareClassic authentication             : " );
                PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateMfc ( &stMfpEVx, 0x00, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len ) );
                printf ( "Success\n" );

                /* Personalize the UID with UIDF0 option.  */
                printf ( "\t\tUID Personalization with Option 1        : " );
                wStatus = phalMfpEVx_PersonalizeUid ( &stMfpEVx, PHAL_MFPEVX_UID_TYPE_UIDF3 );
                if ( wStatus != PH_ERR_SUCCESS )
                {
                    printf ( "\n\t\tUID Personalization with Option 3 (UIDF3) is not possible because the PICC is locked.\n\n" );
                    return wStatus;
                }
                else
                {
                    printf ( "Success\n" );

                    printf ( "\t\tActivate Iso14443 Layer3                 : " );
                    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L3 ) );
                    printf ( "Success\n" );

                    printf ( "\t\tUID                                      : " );
                    PrintData ( aUid, bUid_Len, "%02X", "\n\n" );
                }
                break;
        }
    }

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_TearingProofTransfer ()
{
    phStatus_t wStatus          = 0;
    uint8_t aTx_Data[]          = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

    /* Pre-Personalize the card and switch to security level 1 or 3. */
    PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, bCurrentSecurityLevel >> 0x02 ) );

    /* Set the current security level. */
    PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

    /* Print the current Anti-Tearing configuration block. */
    printf ( "Current value of Anti-Tearing block (0xB004): " );
    PrintData ( gaAntiTearingConfigBlock, sizeof ( gaAntiTearingConfigBlock ), "%02X", "\n" );

    printf ( "Authentication using Configuration key      : " );
    /* Authenticate according to the layer option. */
    switch ( bCurrentSecurityLevel )
    {
        case SECURITY_LEVEL_1:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( &stMfpEVx, ISO14443_L4, FIRST_AUTH, CARD_MASTER_KEY,
                CARD_MASTER_KEY_ADDRESS, CARD_MASTER_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        case SECURITY_LEVEL_3:

            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( &stMfpEVx, FIRST_AUTH, CARD_MASTER_KEY,
                CARD_MASTER_KEY_ADDRESS, CARD_MASTER_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;
    }
    printf ( "Success\n" );

    /* Update the last byte of AntiTearing block. */
    memcpy ( aTx_Data, gaAntiTearingConfigBlock, sizeof ( gaAntiTearingConfigBlock ) );
    aTx_Data[15] = 0x00;
    printf ( "Writing to Anti-Tearing block               : " );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Write ( &stMfpEVx, ENCRYPTED_ON, MAC_COMMAND_ON, ANTI_TEARING_CONFIGURATION_BLOCK, 1, aTx_Data, aTMC, aTMV ) );
    printf ( "Success\n" );
    printf ( "Updated value of Anti-Tearing block (0xB004): " );
    PrintData ( aTx_Data, sizeof ( aTx_Data ), "%02X", "\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_MultipleBlocksRead ( uint8_t bSecurityLevel, uint8_t bSecureMessaging, uint8_t bPrepersonalize )
{
    phStatus_t wStatus          = 0;
    uint8_t bData_BlockNo       = 0x01;
    uint8_t bIteration          = 0;

    uint8_t aRx_Data1[15 * 16];

    if( bPrepersonalize )
    {
        /* Pre-personalize the card. */
        PH_CHECK_SUCCESS_FCT ( wStatus, Demo_PrePersonalization ( ISO14443_L4, ( bSecurityLevel >> 0x01 ) ) );

        /* Set the security level. */
        PH_CHECK_SUCCESS_FCT ( wStatus, SetSecurityLevel () );
    }

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

    /* Authenticate according to the layer option. */
    switch ( bSecurityLevel )
    {
        case SECURITY_LEVEL_1_MFP:
            /* Perform authentication using AES Sector. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( &stMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A ( ( uint16_t) ( bData_BlockNo / 4 ) ),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        case SECURITY_LEVEL_3_MFP:
            /* Update the secure messaging. */
            aPcdCap2In[0] = bSecureMessaging;

            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( &stMfpEVx, FIRST_AUTH, GetAESSectorBlock_A ( ( uint16_t) ( bData_BlockNo / 4 ) ),
                AES_KEY_ADDRESS, AES_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;
    }

    /* Read the data written from the block. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Read ( &stMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bData_BlockNo, 15, aRx_Data1 ) );
    printf ( "(Read Data) Rx Value              :" );
    for ( bIteration = 0; bIteration < 15; bIteration++)
    {
        PrintData ( &aRx_Data1[bIteration * 16], 16, "%02X ", "\n" );
        printf ( "\t\t\t\t   " );
    }
    printf ( "\n" );

    return PH_ERR_SUCCESS;
}

/* Logging function. */
#ifdef NXPBUILD__PH_LOG
void MfpEv1_Log ( void* pDataParams, uint8_t bOption, phLog_LogEntry_t* pEntries, uint16_t wEntryCount )
{
    uint16_t wIteration;
    UNUSED_VARIABLE ( bOption );
    UNUSED_VARIABLE ( pDataParams );

    for ( wIteration = 0; wIteration < wEntryCount; wIteration++ )
    {
        if ( wIteration != 0 )
        {
            printf ( "%s : ", pEntries[wIteration].pString );
        }
        else
        {
            printf ( "-----------------------------------------\n" );
            printf ( "%s\n", pEntries[0].pString );
            printf ( "-----------------------------------------\n" );
        }
        if ( pEntries[wIteration].wDataLen != 0 )
            PrintData ( ( uint8_t * ) ( pEntries[wIteration].pData ), pEntries[wIteration].wDataLen, "%02X ", "\t" );
        else
            PrintData ( ( uint8_t * ) ( pEntries[wIteration].pData ), pEntries[wIteration].wDataLen, "%02X ", "" );
    }
    printf("\n\n");
}
#endif /* NXPBUILD__PH_LOG */
