/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header
 */
#include "Example-Rc663_Felica.h"
int __cdecl main(int argc, char *argv[])
{
    /* ******************************************************************************* */
    /* DECLARATION */
    /* ******************************************************************************* */
    /* data parameter storage */
    phbalReg_SerialWin_DataParams_t balSerial;
    phhalHw_Rc663_DataParams_t      halReader;
    phpalFelica_Sw_DataParams_t     palFelica;
    phalFelica_Sw_DataParams_t      alFelica;

    /* common variables */
    uint8_t bIndex;
    char * pArgument;
    phStatus_t status;

    /* variables used by the BAL component */
    uint8_t * pComPort = (uint8_t*)"COM1";

    /* variables used by the HAL component */
    uint8_t bHalBufferReader[256];

    /* variables used by the PAL component */
    uint8_t pIDmPMm[16], bIDmPMmLength;
    uint8_t pSysCode[2];
    uint8_t bNumTimeSlots;
    uint8_t pRxBuffer[16], bRxLength;
    uint8_t bMoreCardsAvaliable;
    uint8_t pUidOut[16];

    /* variables used by the AL component */
    uint8_t pServiceList[2];
    uint8_t bNumServices;
    uint8_t bTxNumBlocks, bRxNumBlocks;
    uint8_t pBlockList[3], bBlockListLength;
    uint8_t pBlockData[16];

    /* ******************************************************************************* */
    /* INITIALISATION */
    /* ******************************************************************************* */
    /* iterate through all arguments */
    for (bIndex = 0; bIndex < (uint8_t)argc; ++bIndex);
    {
        /* Retrive next argument */
        pArgument = argv[bIndex];

        /* */
        /* Insert argument handling here */
        /* */
    }

    /* initialise the 'bus abstraction layer' BAL: */
    /* use the serial interface */
    status = phbalReg_SerialWin_Init(&balSerial, sizeof(phbalReg_SerialWin_DataParams_t), 0, NULL);
    CHECK_SUCCESS(status);

    status = phbalReg_SetPort(&balSerial, pComPort);
    CHECK_SUCCESS(status);

    /* initialise the 'hardware abstraction layer' HAL: */
    /* use the RC663 Reader IC, glue it together with the BAL component */
    status = phhalHw_Rc663_Init(
        &halReader,
        sizeof(phhalHw_Rc663_DataParams_t),
        &balSerial,
        NULL,
        bHalBufferReader,
        sizeof(bHalBufferReader),
        bHalBufferReader,
        sizeof(bHalBufferReader));
    CHECK_SUCCESS(status);

    /* initialise the 'protocol abstraction layer' PAL: */
    /* use the the Felica protocol. */
    status = phpalFelica_Sw_Init(&palFelica, sizeof(palFelica), &halReader);
    CHECK_SUCCESS(status);

    /* initialise the 'application layer' AL: */
    /* use the Felica application, glue it together with the PAL component */
    status = phalFelica_Sw_Init(&alFelica, sizeof(alFelica), &palFelica);

    status = phbalReg_OpenPort(&balSerial);
    CHECK_SUCCESS(status);

    /* set the PCs baud rate to 115200 */
    status = phbalReg_SetConfig(&balSerial, PHBAL_REG_SERIALWIN_CONFIG_BITRATE, PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200);
    CHECK_SUCCESS(status);

    /* On some machines the RS232 physical interface jitters when the port is opened. For that case, */
    /* send a dummy command in order to the resynchronise the interface. */
    phhalHw_WriteRegister(&halReader, 0x37, 0xFF);

    /* Configure HAL for Felica cards */
    status = phhalHw_ApplyProtocolSettings(&halReader, PHHAL_HW_CARDTYPE_FELICA);
    CHECK_SUCCESS(status);

    /* ******************************************************************************* */
    /* CARD COMMUNICATION */
    /* ******************************************************************************* */
    /* Switch on the field */
    status = phhalHw_FieldReset(&halReader);
    CHECK_SUCCESS(status);

    /* Activate card */
    bIDmPMmLength = 0;
    bNumTimeSlots = PHPAL_FELICA_NUMSLOTS_1;
    pSysCode[0] = 0xFF;
    pSysCode[1] = 0xFF;
    status = phpalFelica_ActivateCard(&palFelica, pIDmPMm, bIDmPMmLength,
                                        pSysCode, bNumTimeSlots, pRxBuffer,
                                        &bRxLength, &bMoreCardsAvaliable);
    printf("phpalFelica_ActivateCard:  %04X\n", status);

    if(PH_ERR_SUCCESS == status)
    {
        status = phpalFelica_GetSerialNo(&palFelica, pUidOut, &bRxLength);
        printf("phpalFelica_GetSerialNo:   %04X\nUID: ", status);
        printBufferHex(pUidOut, bRxLength);

        /* Read one block */
        bTxNumBlocks = 1;
        bNumServices = 1;
        pServiceList[0] = 0x0b;
        pServiceList[1] = 0x00;
        bBlockListLength = 2;
        pBlockList[0] = 0x80;
        pBlockList[1] = 0x01;
        status = phalFelica_Read(&alFelica, bNumServices, pServiceList,
                                            bTxNumBlocks, pBlockList, bBlockListLength,
                                            &bRxNumBlocks, pBlockData);
        printf("phalFelica_Read:           %04X\n", status);

        if(PH_ERR_SUCCESS == status)
        {
            printf("Block Data: ");
            printBufferHex(pBlockData, 16);
        }

        /* Write one block */
        pServiceList[0] = 0x09;
        pServiceList[1] = 0x00;
        memcpy(pBlockData, "\xD0\xD1\xD2\xD3\xD4\xD5\xD6\xD7\xD8\xD9\xDA\xDB\xDC\xDD\xDE\xDF", 16);
        pBlockData[4] = 0xD3;
        status = phalFelica_Write(&alFelica, bNumServices, pServiceList,
                                             bTxNumBlocks, pBlockList,
                                             bBlockListLength, pBlockData);
        printf("phalFelica_Write:          %04X\n", status);
    }


    status = phbalReg_ClosePort(&balSerial);
    fprintf(stdout, "\n press any key to exit\n");
    _getch();
    return 0;
}

/**
 * Helper functions
 */
void printBufferHex(uint8_t * pBuffer, uint8_t bLength)
{
    uint8_t bIndex;

    for (bIndex = 0; bIndex < bLength; ++bIndex)
    {
        printf("%02X ", pBuffer[bIndex]);
    }
    printf("\n");
}
