/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header
 */
#include "Example-Rc663_Mfp.h"
int __cdecl main(int argc, char *argv[])
{
    /* ******************************************************************************* */
    /* DECLARATION */
    /* ******************************************************************************* */
    /* data parameter storage */
    phbalReg_SerialWin_DataParams_t balSerial;
    phhalHw_Rc663_DataParams_t      halReader;
    phpalI14443p3a_Sw_DataParams_t  palI14443p3a;
    phpalI14443p4a_Sw_DataParams_t  palI14443p4a;
    phpalI14443p4_Sw_DataParams_t   palI14443p4;
    phpalMifare_Sw_DataParams_t     palMifare;
    phalMfc_Sw_DataParams_t         alMifareC;  /* needed for MifarePlus Cards in Security Level1 */
    phalMfp_Sw_DataParams_t         alMifareP;

    phKeyStore_Sw_DataParams_t      KeyStore;
    phCryptoSym_Sw_DataParams_t     CryptoEnc;
    phCryptoSym_Sw_DataParams_t     CryptoMac;
    phCryptoSym_Sw_DataParams_t     CryptoDiv;
    phCryptoSym_Sw_DataParams_t     CryptoRngCrypto;
    phCryptoRng_Sw_DataParams_t     CryptoRng;

    /* common variables */
    uint8_t bIndex;
    char * pArgument;
    phStatus_t status;

    /* variables used by the BAL component */
    uint8_t * pComPort = (uint8_t*)"COM1";

    /* variables used by the HAL component */
    uint8_t bHalBufferReader[256];

    /* variables used by the PAL (ISO14443-3) component */
    uint8_t pUidIn[10], bLenUidIn, pUidOut[10], bLenUidOut;
    uint8_t bSak, bMoreCardsAvaliable;

    /* variables used by the PAL (ISO14443-4) component */
    uint8_t bCid, bFsdi, bDri, bDsi, pAts[32];
    uint8_t bCidEnabled, bNadSupported;
    uint8_t bNad, bFwi, bFsci;

    /* variables used by the AL (Mifare) component */
    uint16_t wKeyNumber, wKeyVersion, wBlockNo;
    uint8_t bLayer4Comm, bFirstAuth;
    uint8_t bLenDivInput, bLenPcdCap2;
    uint8_t pPcdCap2[8], pDivInput[4], pPcdCap2In[4], pPcdCap2Out[8], pKmf[6];
    uint8_t pOriginalBlockData[PHAL_MFC_DATA_BLOCK_LENGTH];
    uint8_t pNewBlockData[PHAL_MFC_DATA_BLOCK_LENGTH];
    uint8_t pReadBlockData[PHAL_MFC_DATA_BLOCK_LENGTH];
    uint8_t bBlockNo, bKeyType;
    uint8_t bEncrypted, bReadMaced, bWriteMaced, bMacOnCmd, bNumOfBlocks;

    /* variables used by the KeyStore component */
    uint16_t wKeyStoreKeyType;
    uint8_t pNewKey[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    phKeyStore_Sw_KeyEntry_t        pKeyEntries[NUMBER_OF_KEYENTRIES];
    uint16_t wNoOfKeyEntries =      NUMBER_OF_KEYENTRIES;
    phKeyStore_Sw_KeyVersionPair_t  pKeyVersionPairs[NUMBER_OF_KEYVERSIONPAIRS * NUMBER_OF_KEYENTRIES];
    uint16_t wNoOfKeyVersionPairs = NUMBER_OF_KEYVERSIONPAIRS;
    phKeyStore_Sw_KUCEntry_t        pKUCEntries[NUMBER_OF_KUCENTRIES];
    uint16_t wNoOfKUCEntries =      NUMBER_OF_KUCENTRIES;

    /* variables used by the Crypto component */
    uint8_t pSeed[16], bSeedLength;

    /* ******************************************************************************* */
    /* INITIALISATION */
    /* ******************************************************************************* */
    /* iterate through all arguments */
    for (bIndex = 0; bIndex < (uint8_t)argc; ++bIndex);
    {
        /* Retrive next argument */
        pArgument = argv[bIndex];

        /* */
        /* Insert argument handling here */
        /* */
    }

    /* initialise the 'bus abstraction layer' BAL: */
    /* use the serial interface */
    status = phbalReg_SerialWin_Init(&balSerial, sizeof(phbalReg_SerialWin_DataParams_t), 0, NULL);
    CHECK_SUCCESS(status);

    status = phbalReg_SetPort(&balSerial, pComPort);
    CHECK_SUCCESS(status);

    /* initialise the 'hardware abstraction layer' HAL: */
    /* use the RC663 Reader IC, glue it together with the BAL component */
    status = phhalHw_Rc663_Init(
        &halReader,
        sizeof(phhalHw_Rc663_DataParams_t),
        &balSerial,
        NULL,
        bHalBufferReader,
        sizeof(bHalBufferReader),
        bHalBufferReader,
        sizeof(bHalBufferReader));
    CHECK_SUCCESS(status);

    /* initialise the 'protocol abstraction layer' PAL: */
    /* use the the ISO14443-3 TypeA protocol. */
    status = phpalI14443p3a_Sw_Init(&palI14443p3a, sizeof(palI14443p3a), &halReader);
    CHECK_SUCCESS(status);

    /* initialise the 'protocol abstraction layer' PAL: */
    /* use the the ISO14443-4 TypeA protocol, glue it togeter with the underlaying PAL component. */
    status = phpalI14443p4a_Sw_Init(&palI14443p4a, sizeof(palI14443p4a),&halReader);
    CHECK_SUCCESS(status);

    /* initialise the 'protocol abstraction layer' PAL: */
    /* use the the ISO14443-4 protocol, glue it togeter with the underlaying PAL component. */
    status = phpalI14443p4_Sw_Init(&palI14443p4, sizeof(palI14443p4), &halReader);
    CHECK_SUCCESS(status);

    /* initialise the 'protocol abstraction layer' PAL: */
    /* use the the Mifare protocol, glue it togeter with the underlaying PAL component. */
    status = phpalMifare_Sw_Init(&palMifare, sizeof(palMifare),&halReader, &palI14443p4);
    CHECK_SUCCESS(status);

    /* initialise Crypto components */
    status = phCryptoSym_Sw_Init(&CryptoEnc, sizeof(CryptoEnc), &KeyStore);
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&CryptoMac, sizeof(CryptoMac), &KeyStore);
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&CryptoDiv, sizeof(CryptoDiv), &KeyStore);
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&CryptoRngCrypto, sizeof(CryptoRngCrypto), &KeyStore);
    CHECK_SUCCESS(status);

    /* initialise Crypto Rng component */
    status = phCryptoRng_Sw_Init(&CryptoRng, sizeof(CryptoRng), &CryptoRngCrypto);
    CHECK_SUCCESS(status);

    /* initialise the Key Store: */
    status = phKeyStore_Sw_Init(&KeyStore, sizeof(KeyStore), pKeyEntries, wNoOfKeyEntries,
                                pKeyVersionPairs, wNoOfKeyVersionPairs, pKUCEntries, wNoOfKUCEntries);
    CHECK_SUCCESS(status);

    /* load 2 Keys to the Store ;-), one MIFARE(R) key for classic authentications, and one AES key. */
    /* Note: If You use Key number 0x00, be aware that in SAM this Key is the 'Host authentication key' !!! */
    wKeyNumber = 1;
    wKeyVersion = 0;
    wKeyStoreKeyType = PH_KEYSTORE_KEY_TYPE_MIFARE;
    status = phKeyStore_FormatKeyEntry(&KeyStore, wKeyNumber, wKeyStoreKeyType);
    CHECK_SUCCESS(status);

    /* Set the Key and store it in the Key Store */
    memcpy(pNewKey, FF_Key, PH_KEYSTORE_KEY_TYPE_MIFARE_SIZE);
    status = phKeyStore_SetKey(&KeyStore, wKeyNumber, wKeyVersion, wKeyStoreKeyType, pNewKey, wKeyVersion);
    CHECK_SUCCESS(status);

    wKeyNumber = 2;
    wKeyVersion = 0;
    wKeyStoreKeyType = PH_KEYSTORE_KEY_TYPE_AES128;
    status = phKeyStore_FormatKeyEntry(&KeyStore, wKeyNumber, wKeyStoreKeyType);
    CHECK_SUCCESS(status);

    /* Set the Key and store it in the Key Store */
    memcpy(pNewKey, FF_Key, PH_KEYSTORE_KEY_TYPE_AES128_SIZE);
    status = phKeyStore_SetKey(&KeyStore, wKeyNumber, wKeyVersion, wKeyStoreKeyType, pNewKey, wKeyVersion);
    CHECK_SUCCESS(status);

    /* Seed the Random Generator */
    bSeedLength = sizeof(pSeed);
    memset(pSeed, 0x00, sizeof(pSeed));
    status = phCryptoRng_Seed(&CryptoRng, pSeed, bSeedLength);
    CHECK_SUCCESS(status);

    /* initialise the 'application layer' AL: */
    /* use the Mifare Classic application, glue it together with the PAL component */
    status = phalMfc_Sw_Init(&alMifareC, sizeof(alMifareC), &palMifare, &KeyStore);
    CHECK_SUCCESS(status);

    /* initialise the 'application layer' AL: */
    /* use the Mifare Plus application, glue it together with the PAL component */
    status = phalMfp_Sw_Init(&alMifareP, sizeof(alMifareP), &palMifare, &KeyStore,
                             &CryptoEnc, &CryptoMac, &CryptoRng, &CryptoDiv);
    CHECK_SUCCESS(status);

    status = phbalReg_OpenPort(&balSerial);
    CHECK_SUCCESS(status);

    /* set the PCs baud rate to 115200 */
    status = phbalReg_SetConfig(&balSerial, PHBAL_REG_SERIALWIN_CONFIG_BITRATE, PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200);
    CHECK_SUCCESS(status);

    /* On some machines the RS232 physical interface jitters when the port is opened. For that case, */
    /* send a dummy command in order to the resynchronise the interface. */
    phhalHw_WriteRegister(&halReader, 0x37, 0xFF);

    /* Configure HAL for ICode EPC HF G2 cards */
    status = phhalHw_ApplyProtocolSettings(&halReader, PHHAL_HW_CARDTYPE_ISO14443A);
    CHECK_SUCCESS(status);

    /* ******************************************************************************* */
    /* CARD COMMUNICATION */
    /* ******************************************************************************* */
    /* Switch on the field */
    status = phhalHw_FieldReset(&halReader);
    CHECK_SUCCESS(status);

    /* Activate card */
    bLenUidIn = 0;
    status = phpalI14443p3a_ActivateCard(&palI14443p3a, pUidIn, bLenUidIn,
                                          pUidOut, &bLenUidOut,
                                           &bSak, &bMoreCardsAvaliable);
    printf("phpalI14443p3a_ActivateCard:  %04X\n", status);

    if(PH_ERR_SUCCESS == status)
    {
        status = phpalI14443p3a_GetSerialNo(&palI14443p3a, pUidOut, &bLenUidOut);
        printf("phpalI14443p3a_GetSerialNo:   %04X\nUID: ", status);
        if(PH_ERR_SUCCESS == status)
            printBufferHex(pUidOut, bLenUidOut);

        /* Do the appropriate Authentication for the Card, depending on its Security Level. */
        /* Read one MIFARE(R) Block, write one Block and restore the written Block. */
        memcpy(pNewBlockData, "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10",
            sizeof(pNewBlockData));

        switch(bSak)
        {
        case 0x18: /* Security Level L1 */
            {
                printf("\n   MIFARE Plus Card is in Security Level L1");
                printf("\n   ----------------------------------------\n");

                /* Authenticate with Mifare crypto */
                bBlockNo = 0;
                wKeyNumber = 1;
                bKeyType = PHAL_MFC_KEYA;
                status = phalMfc_Authenticate(&alMifareC, bBlockNo, bKeyType, wKeyNumber, wKeyVersion,
                                               pUidOut, bLenUidOut);
                printf("phalMfc_Authenticate:         %04X\n", status);

                if(PH_ERR_SUCCESS == status)
                {
                    /* Read one block */
                    bBlockNo = 1;
                    status = phalMfc_Read(&alMifareC, bBlockNo, pOriginalBlockData);
                    printf("phalMfc_Read:                 %04X\n", status);
                    if(PH_ERR_SUCCESS == status)
                    {
                            printf("Block %d Data: ", bBlockNo);
                        printBufferHex(pOriginalBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                    }

                    /* Write one block */
                    status = phalMfc_Write(&alMifareC, bBlockNo, pNewBlockData);
                    printf("phalMfc_Write:                %04X\n", status);

                    status = phalMfc_Read(&alMifareC, bBlockNo, pReadBlockData);
                    printf("phalMfc_Read:                 %04X\n", status);
                    if(PH_ERR_SUCCESS == status)
                    {
                            printf("Block %d Data: ", bBlockNo);
                        printBufferHex(pReadBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                    }

                    /* Restore old Block Data */
                    status = phalMfc_Write(&alMifareC, bBlockNo, pOriginalBlockData);
                    printf("phalMfc_Write:                %04X\n", status);

                    status = phalMfc_Read(&alMifareC, bBlockNo, pReadBlockData);
                    printf("phalMfc_Read:                 %04X\n", status);
                    if(PH_ERR_SUCCESS == status)
                    {
                            printf("Block %d Data: ", bBlockNo);
                        printBufferHex(pReadBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                    }
                }
                break;
            }
        case 0x11: /* Security Level L2 */
            {
                printf("\n   MIFARE Plus Card is in Security Level L2\n");
                printf("\n   ----------------------------------------\n");

                /* first we do a 'nonfirst' (also refered to as 'following') authentication ( --> customer-focused passion to win) */
                bLayer4Comm = 0;
                bFirstAuth = 0;
                wBlockNo = PHYSICAL_AES_START_ADDRESS;
                wKeyNumber = 2;
                wKeyVersion = 0;
                bLenDivInput = 0;
                bLenPcdCap2 = 0;
                status = phalMfp_AuthenticateSL2(&alMifareP,bLayer4Comm, bFirstAuth, wBlockNo, wKeyNumber, wKeyVersion,
                    bLenDivInput, pDivInput, bLenPcdCap2, pPcdCap2In, pPcdCap2Out, pPcdCap2, pKmf);
                printf("phalMfp_AuthenticateSL2:        %04X\n", status);

                bBlockNo = 0;
                wKeyNumber = 1;
                bKeyType = PHPAL_MIFARE_KEYA;
                status = phalMfp_AuthenticateClassicSL2(&alMifareP, bBlockNo, bKeyType, wKeyNumber, wKeyVersion,
                                                        pUidOut, bLenUidOut);
                printf("phalMfp_AuthenticateClassicSL2: %04X\n", status);

                if(PH_ERR_SUCCESS == status)
                {
                    /* Read one block */
                    bEncrypted = 0;
                    bReadMaced = 0;
                    bMacOnCmd = 0;
                    bBlockNo = 1;
                    bNumOfBlocks = 1;
                    status = phalMfc_Read(&alMifareC, bBlockNo, pOriginalBlockData);
                    printf("phalMfc_Read:                 %04X\n", status);
                    if(PH_ERR_SUCCESS == status)
                    {
                        printf("Block %d Data: ", bBlockNo);
                        printBufferHex(pOriginalBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                    }

                    /* Write one block */
                    status = phalMfc_Write(&alMifareC, bBlockNo, pNewBlockData);
                    printf("phalMfc_Write:                %04X\n", status);

                    status = phalMfc_Read(&alMifareC, bBlockNo, pReadBlockData);
                    printf("phalMfc_Read:                 %04X\n", status);
                    if(PH_ERR_SUCCESS == status)
                    {
                            printf("Block %d Data: ", bBlockNo);
                        printBufferHex(pReadBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                    }

                    /* Restore old Block Data */
                    status = phalMfc_Write(&alMifareC, bBlockNo, pOriginalBlockData);
                    printf("phalMfc_Write:                %04X\n", status);

                    status = phalMfc_Read(&alMifareC, bBlockNo, pReadBlockData);
                    printf("phalMfc_Read:                 %04X\n", status);
                    if(PH_ERR_SUCCESS == status)
                    {
                            printf("Block %d Data: ", bBlockNo);
                        printBufferHex(pReadBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                    }
                }
                break;
            }
        case 0x20: /* Security Level L3 */
            {
                printf("\n   MIFARE Plus Card is in Security Level L3\n");
                printf("\n   ----------------------------------------\n");

                bFsdi = 8;
                bCid = 1;
                bDri = PHPAL_I14443P4A_DATARATE_212;
                bDsi = PHPAL_I14443P4A_DATARATE_212;
                status = phpalI14443p4a_ActivateCard(&palI14443p4a, bFsdi, bCid, bDri, bDsi, pAts);
                printf("phpalI14443p4a_ActivateCard:  %04X\n", status);

                if(PH_ERR_SUCCESS == status)
                {
                    /* Set protocol parameters for ISO14443 Layer4 */
                    status = phpalI14443p4a_GetProtocolParams(&palI14443p4a, &bCidEnabled, &bCid,
                                                              &bNadSupported, &bFwi, &bFsdi, &bFsci);
                    CHECK_SUCCESS(status);

                    bNad = 0;
                    status = phpalI14443p4_SetProtocol(&palI14443p4, bCidEnabled, bCid,
                                                        bNadSupported, bNad, bFwi, bFsdi, bFsci );
                    CHECK_SUCCESS(status);

                    /* Authenticate */
                    bFirstAuth = 1;
                    wBlockNo = PHYSICAL_AES_START_ADDRESS + 2;
                    bLenDivInput = 0;
                    bLenPcdCap2 = 0;
                    status = phalMfp_AuthenticateSL3(&alMifareP, bFirstAuth, wBlockNo, wKeyNumber,
                                                      wKeyVersion, bLenDivInput, pDivInput,
                                                      bLenPcdCap2, pPcdCap2In, pPcdCap2Out, pPcdCap2);
                    printf("phalMfp_AuthenticateSL3:      %04X\n", status);


                    if(PH_ERR_SUCCESS == status)
                    {
                        /* Read one block */
                        bEncrypted = 0;
                        bReadMaced = 0;
                        bMacOnCmd = 1;
                        wBlockNo = 4;
                        bNumOfBlocks = 1;
                        status = phalMfp_Read(&alMifareP, bEncrypted, bReadMaced, bMacOnCmd,
                                              wBlockNo, bNumOfBlocks, pOriginalBlockData);
                        printf("phalMfp_Read:                 %04X\n", status);
                        if(PH_ERR_SUCCESS == status)
                        {
                            printf("Block %d Data: ", wBlockNo);
                            printBufferHex(pOriginalBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                        }

                        /* Write one block */
                        bWriteMaced = 0;
                        status = phalMfp_Write(&alMifareP, bEncrypted, bWriteMaced,
                                                wBlockNo, bNumOfBlocks, pNewBlockData);
                        printf("phalMfp_Write:                %04X\n", status);

                        status = phalMfp_Read(&alMifareP, bEncrypted, bReadMaced, bMacOnCmd,
                                              wBlockNo, bNumOfBlocks, pReadBlockData);
                        printf("phalMfp_Read:                 %04X\n", status);
                        if(PH_ERR_SUCCESS == status)
                        {
                            printf("Block %d Data: ", wBlockNo);
                            printBufferHex(pReadBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                        }

                        /* Restore old Block Data */
                        status = phalMfp_Write(&alMifareP, bEncrypted, bWriteMaced,
                                                wBlockNo, bNumOfBlocks, pOriginalBlockData);

                        printf("phalMfp_Write:                %04X\n", status);
                        status = phalMfp_Read(&alMifareP, bEncrypted, bReadMaced, bMacOnCmd,
                                              wBlockNo, bNumOfBlocks, pReadBlockData);
                        printf("phalMfp_Read:                 %04X\n", status);
                        if(PH_ERR_SUCCESS == status)
                        {
                            printf("Block %d Data: ", wBlockNo);
                            printBufferHex(pReadBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
                        }
                    }
                }
                break;
            }
        default:
            break;
        }
    }

    status = phbalReg_ClosePort(&balSerial);
    fprintf(stdout, "\n press any key to exit\n");
    _getch();
    return 0;
}

/**
 * Helper functions
 */
void printBufferHex(uint8_t * pBuffer, uint8_t bLength)
{
    uint8_t bIndex;

    for (bIndex = 0; bIndex < bLength; ++bIndex)
    {
        printf("%02X ", pBuffer[bIndex]);
    }
    printf("\n");
}