/*
 * Copyright 2015 - 2018, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Icode DNA.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/* Header for this file. */
#include "Example-Rd70x_Rd710_CM1_IcodeDna.h"

#undef NXPBUILD__PH_LOG			/* Disabling Log. To enable comment this line. */

#ifdef NXPBUILD__PH_LOG
	phLog_RegisterEntry_t logRegisterEntries[16];
	phLog_LogEntry_t logEntries[48];
#endif /* NXPBUILD__PH_LOG */

int __cdecl main (void)
{
	phStatus_t wStatus = 0;

	uint16_t wIteration = 0;
	uint16_t wCount = 0;
	uint32_t dwChoice = 0;

	uint8_t bReaderType = 0; /* 0-> Rd70x, 1 => Rd710 / CM1 */
	uint8_t bKeyData = 0;
	uint32_t dwBlockNo = 0, dwNoOfBlocks = 0;
	uint8_t bOption = 0, bFlags = 0, bAfi = 0x00, bMaskLength = 0;
	uint8_t bDsfid = 0, bMoreCardsAvaliable = 0;

	uint8_t aAtr[256];
	uint8_t aSeed[10];
	uint8_t aReaderList[512];
	uint8_t aMask[PHPAL_SLI15693_UID_LENGTH];
	uint8_t aUid[PHPAL_SLI15693_UID_LENGTH];
	uint8_t aTx_HalBuffer[GLOBAL_BUFFER_SIZE];
    uint8_t aRx_HalBuffer[GLOBAL_BUFFER_SIZE];

	uint8_t	aReadConfigBuffer[4];
	uint8_t aReadData[64 * 4];
	uint8_t * pReadBuffer = NULL;
	uint16_t wReadBufLen = 0;
	uint16_t wReadDataLen = 0;

	phKeyStore_Sw_KeyEntry_t aKeyEntry[0xFF];
	phKeyStore_Sw_KeyVersionPair_t aKeyVersion[0xFF];
	phKeyStore_Sw_KUCEntry_t aKeyUsageCounter[0xFF];

	phbalReg_Rd70xUsbWin_DataParams_t stBal_Rd70x;
	phhalHw_Rd70x_DataParams_t stHal_Rd70x;

	phbalReg_PcscWin_DataParams_t stBal_Pcsc;
    phhalHw_Rd710_DataParams_t	  stHal_Rd710;

	phCryptoSym_Sw_DataParams_t stCryptoEnc;			/* CryptoSym parameter structure for ENC */
	phCryptoSym_Sw_DataParams_t stCryptoSymRng;			/* CryptoSym parameter structure for SymRng */
	phCryptoRng_Sw_DataParams_t stCryptoRng;			/* CryptoSym parameter structure for Rng */

	printf ( "Press 1 To run with Rd70x reader.\n" );
	printf ( "Press 2 To run with Rd710 / CM1 reader.\n" );
	printf ( "Press x To Quit.\n\n");

	dwChoice = _getch();
	system ( "cls");

	switch(dwChoice)
	{
		case '1':
			printf ( "Please ensure that a Rd70x reader is connected and is in working condition.\n" );
			printf ( "Firmware Version 2.11 or higher must be used!\n\n" );

			/* Initialize the reader BAL component. */
			wStatus = phbalReg_Rd70xUsbWin_Init ( &stBal_Rd70x, sizeof ( stBal_Rd70x ) );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Get list of connected pegoda readers. */
			wStatus = phbalReg_GetPortList ( &stBal_Rd70x, sizeof ( aReaderList ), aReaderList, &wCount );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Connect to the first reader out of the list. */
			wStatus = phbalReg_SetPort ( &stBal_Rd70x, aReaderList );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Open the reader port. */
			wStatus = phbalReg_OpenPort ( &stBal_Rd70x );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Initialize the reader HAL component. */
			wStatus = phhalHw_Rd70x_Init ( &stHal_Rd70x, sizeof ( stHal_Rd70x ), &stBal_Rd70x,
				aTx_HalBuffer, sizeof ( aTx_HalBuffer ), aRx_HalBuffer, sizeof ( aRx_HalBuffer ) );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Set the default reader as Rd70x. */
			pHal_Generic = &stHal_Rd70x;

			wStatus = phpalSli15693_Sw_Init ( &stPal15693_Sw, sizeof ( stPal15693_Sw ), &stHal_Rd70x );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Set the default PAL component as Rd710. */
			pPal_Generic = &stPal15693_Sw;

			/* Set the reader type to Rd70x. */
			bReaderType = 0;
			break;

		case '2':
			printf ( "NxpRdLib ANSI-C Rd710 / CM1 reader Example Program \n\n" );
			printf ( "Please ensure that a Pegoda-2 CM1 reader is connected and in working condition.\n\n" );

			/* Inititialize BAL component. */
			wStatus = phbalReg_PcscWin_Init ( &stBal_Pcsc, sizeof ( stBal_Pcsc ), aAtr, 256 );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Set the BAL Communication Configuration values */
			wStatus = phbalReg_SetConfig ( &stBal_Pcsc, PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL, PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED );
			CHECK_SUCCESS ( wStatus, 0 );

			wStatus = phbalReg_SetConfig ( &stBal_Pcsc, PHBAL_REG_PCSCWIN_CONFIG_SHARE, PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Select Port to be used (no SAM reader) */
			wStatus = phbalReg_SetPort ( &stBal_Pcsc,  ( uint8_t * ) PCSC_READER_NO_SAM_NAME );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Initialize the Rd710 HAL component. */
			wStatus = phhalHw_Rd710_Init ( &stHal_Rd710, sizeof ( stHal_Rd710 ), &stBal_Pcsc, 0, aTx_HalBuffer, sizeof ( aTx_HalBuffer ),
							aRx_HalBuffer, sizeof ( aRx_HalBuffer ) );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Open the PCSC port */
			wStatus = phbalReg_OpenPort ( &stBal_Pcsc );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Standard Rd710 Cmd Init. */
			wStatus = ( phStatus_t ) phhalHw_Rd710_Cmd_InitReader ( &stHal_Rd710 );
			CHECK_SUCCESS ( wStatus, 0 );

			/*Set default reader as RD710 / CM1. */
			pHal_Generic = &stHal_Rd710;

			wStatus = phpalSli15693_Sw_Init ( &stPal15693_Sw, sizeof ( stPal15693_Sw ), &stHal_Rd710 );
			CHECK_SUCCESS ( wStatus, 0 );

			/* Set the default PAL component as Sw. */
			pPal_Generic = &stPal15693_Sw;

			/* Set the reader type to Rd710. */
			bReaderType = 1;
			break;

		case 'x':
		case 'X':
			exit ( 0 );

		default:
			printf ( " Invalid option selected.\n\n" );
			printf ( " Pressing any key will exit the demo. Restart the demo and choose the right option.\n ");
			_getch ();
			exit ( 0 );
	}

	printf ( "Build up the library stack... " );

	/* Initialize software keystore component. */
	wStatus = phKeyStore_Sw_Init ( &stKeyStore, sizeof ( stKeyStore ), aKeyEntry, 0xFF, aKeyVersion, 0x01,
		aKeyUsageCounter, 0xFF );
	CHECK_SUCCESS ( wStatus, 0 );

	/* Initialize CryptoSym for encryption. */
	wStatus = phCryptoSym_Sw_Init ( &stCryptoEnc, sizeof ( stCryptoEnc ), NULL );
	CHECK_SUCCESS ( wStatus, 0 );

	/* Initialize CryptoSym for random number generation. */
	wStatus = phCryptoSym_Sw_Init ( &stCryptoSymRng, sizeof ( stCryptoSymRng ), NULL );
	CHECK_SUCCESS ( wStatus, 0 );

	/* Initialize Crypto for random number generation. */
	wStatus = phCryptoRng_Sw_Init ( &stCryptoRng, sizeof ( stCryptoRng ), &stCryptoSymRng );
	CHECK_SUCCESS ( wStatus, 0 );

	/* Rest the seed buffer to initiate random number generation. */
	memset ( aSeed, 0x00, sizeof ( aSeed ) );

	/* Initiate seed for random number generation. */
	wStatus = phCryptoRng_Seed ( &stCryptoRng, aSeed, sizeof ( aSeed ) );
	CHECK_SUCCESS ( wStatus, 0 );

	/* Initialize the Icode Dna application layer. */
	wStatus = phalICode_Sw_Init ( &stICode, sizeof ( stICode ), pPal_Generic, &stCryptoEnc, &stCryptoRng, &stKeyStore );
	CHECK_SUCCESS ( wStatus, 0 );

	printf ( "Success\n" );

	/* Initialize logging */
#ifdef NXPBUILD__PH_LOG
    /* Register the component for logging */
	wStatus = phLog_Init(ICode_Log, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS ( wStatus, 0 );

    wStatus = phLog_Register(&stICode, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS ( wStatus, 0 );
#endif /* NXPBUILD__PH_LOG */

	/* Configure HAL for Sli15693 tags */
	wStatus = phhalHw_ApplyProtocolSettings ( pHal_Generic, PHHAL_HW_CARDTYPE_ISO15693 );
	CHECK_SUCCESS ( wStatus, 0 );

	/* Clear the console. */
	system ( "cls" );

	/* Get the status if engineering sample is placed on the reader. */
	printf ( "Press 1 if engineering sample is placed on the the reader.\n" );
	printf ( "Press 0 if plastc card is placed on the the reader.\n" );
	printf ( "Enter your choice: " );
	dwChoice = _getch ();
	printf ( "%c\n\n", dwChoice );

	bIsEnggSample = ( uint8_t ) ( dwChoice - 0x30 );

	if ( bIsEnggSample > 1 )
	{
		printf ( "Restart the application and select valid option. \n" );
		printf ( "Pressing any key will exit the demo. \n" );
		_getch ();
		exit ( 0 );
	}

	if ( !bIsEnggSample )
	{
		printf ("If Key 0 data is known, Enter the same in the below mentioned format or else enter.\n" );
		printf ( "Format should be XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX (Hex format)\n" );
		printf ( "If key 0 is not know the default values (i.e. all zero's) will be used. \n" );
		printf ( "Key 0 Data: " );

		for ( wIteration = 0; wIteration < 16; wIteration++)
		{
			bKeyData = ( uint8_t ) _getch ();
			printf ( "%c", bKeyData );
			if ( bKeyData == 0x0D )
				break;

			bKeyData = HexString2Hex ( bKeyData );
			if ( bKeyData == 0xFF )
			{
				printf ( "\n\n" );
				printf ( "Invalid key data.\n" );
				printf ( "Pressing any key will exit the application. \n" );
				printf ( "Restart the application and enter the data in hex format.\n\n" );
				_getch ();
				exit ( 0 );
			}
			gaKey_0[wIteration] = ( uint8_t ) ( bKeyData << 4 );

			bKeyData = ( uint8_t ) _getch ();
			printf ( "%c", bKeyData );
			if ( bKeyData == 0x0D )
				break;

			bKeyData = HexString2Hex ( bKeyData );
			if ( bKeyData == 0xFF )
			{
				printf ( "\n\n" );
				printf ( "Invalid key data.\n" );
				printf ( "Pressing any key will exit the application. \n" );
				printf ( "Restart the application and enter the data in hex format.\n\n" );
				_getch ();
				exit ( 0 );
			}
			gaKey_0[wIteration] |= bKeyData;
		}
	}

	/* Add keys to the software keystore component. */
	wStatus = UpdateKeystore ();
	CHECK_SUCCESS ( wStatus, 0 );

	do
	{
		/* Clear the console. */
		system ( "cls" );

		printf ( "Press 1 for Demo on Authenticate command with TAM method. \n" );
		printf ( "Press 2 for Demo on Authenticate command with MAM method. \n" );
		printf ( "Press 3 for Demo on Challenge command. \n" );
		printf ( "Press 4 for Demo on ReadBuffer command without verification. \n" );
		printf ( "Press 5 for Demo on ReadBuffer command with verification. \n" );
		printf ( "Press 6 for Demo on WriteConfig command. \n" );
		printf ( "Press 7 for Demo on ReadConfig command. \n" );
		printf ( "Press 8 for Demo on PickRandomID command. \n" );
		printf ( "Press 9 for Demo on ReadMultipleBlocks command. \n" );
		printf ( "Press x to exit the demonstration. \n" );
		printf ( "Enter your choice: " );
		dwChoice = _getch ();
		printf ( "%c\n\n", dwChoice );

		if ( (dwChoice == 'x') || ( dwChoice == 'X' ) )
			exit ( 0 );

		/* Switch on the field */
		wStatus = phhalHw_FieldReset ( pHal_Generic );
		CHECK_SUCCESS ( wStatus, 0 );

		/* Activate card */
		printf ( "Activate Card -  " );
		bOption = PHPAL_SLI15693_ACTIVATE_ADDRESSED;
		bFlags = (PHPAL_SLI15693_FLAG_DATA_RATE | PHPAL_SLI15693_FLAG_INVENTORY | PHPAL_SLI15693_FLAG_NBSLOTS);
		wStatus = phpalSli15693_ActivateCard ( stICode.pPalSli15693DataParams, bOption, bFlags,
					bAfi, aMask, bMaskLength, &bDsfid, aUid, &bMoreCardsAvaliable );
		CHECK_SUCCESS ( wStatus, 1 );

		/* Reset Memory if engineering sample is used. */
		ResetConfigMemory();

		/* Print the UID and DSFID. */
		printf ( "UID  : ");
		PrintData ( aUid, sizeof ( aUid ), "%02X ", "\n" );
		printf ( "DSFID: %02X\n\n", bDsfid );

		switch ( dwChoice )
		{
			case '1':
				printf ( "Demo for Authenticate command with TAM method ----------------------------------------\n" );

				/* Reset configuration memory if engineering sample is used. */
				UpdateConfigMemory();

				printf ( "\tAuthenticate : " );
				wStatus = phalICode_AuthenticateTAM1 ( &stICode, PHAL_ICODE_OPTION_OFF, AES_KEY_0_ADDRESS, AES_KEY_0_VERSION, CARD_KEY_0, NULL, 0 );
				CHECK_SUCCESS ( wStatus, 1 );
				break;

			case '2':
				printf ( "Demo for Authenticate command with MAM method ----------------------------------------\n" );

				/* Reset configuration memory if engineering sample is used. */
				UpdateConfigMemory();

				printf ( "\tAuthenticate : " );
				wStatus = phalICode_AuthenticateMAM ( &stICode, PHAL_ICODE_OPTION_OFF, AES_KEY_0_ADDRESS, AES_KEY_0_VERSION, CARD_KEY_0, 0x00, NULL, 0 );
				CHECK_SUCCESS ( wStatus, 1 );
				break;

			case '3':
				printf ( "Demo for Challenge command -----------------------------------------------------------\n" );

				/* Reset configuration memory if engineering sample is used. */
				UpdateConfigMemory();

				printf ( "\tNon Addressed mode configuration : " );
				bFlags = PHPAL_SLI15693_FLAG_DATA_RATE;
				wStatus = phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags );
				CHECK_SUCCESS ( wStatus, 1 );

				phhalHw_FieldReset ( pHal_Generic );
				CHECK_SUCCESS ( wStatus, 0 );

				phhalHw_FieldOn ( pHal_Generic );
				CHECK_SUCCESS ( wStatus, 0 );

				printf ( "\tMoving to Ready state            : " );
				wStatus = phpalSli15693_ResetToReady ( stICode.pPalSli15693DataParams );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tChallenge                        : " );
				wStatus = phalICode_Challenge ( &stICode, PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL, PHAL_ICODE_CHALLENGE_TAM_1,
                    CARD_KEY_0, NULL, 0 );
				CHECK_SUCCESS ( wStatus, 1 );
				break;

			case '4':
				printf ( "Demo for ReadBuffer command without verification -------------------------------------\n" );

				/* Reset configuration memory if engineering sample is used. */
				UpdateConfigMemory();

				printf ( "\tNon Addressed mode configuration : " );
				bFlags = PHPAL_SLI15693_FLAG_DATA_RATE;
				wStatus = phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags );
				CHECK_SUCCESS ( wStatus, 1 );

				phhalHw_FieldReset ( pHal_Generic );
				CHECK_SUCCESS ( wStatus, 0 );

				phhalHw_FieldOn ( pHal_Generic );
				CHECK_SUCCESS ( wStatus, 0 );

				printf ( "\tMoving to Ready state            : " );
				wStatus = phpalSli15693_ResetToReady ( stICode.pPalSli15693DataParams );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tChallenge                        : " );
                wStatus = phalICode_Challenge(&stICode, PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL, PHAL_ICODE_CHALLENGE_TAM_1,
                    CARD_KEY_0, NULL, 0);
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tAddressed mode configuration     : " );
				bFlags = PHPAL_SLI15693_FLAG_DATA_RATE | PHPAL_SLI15693_FLAG_ADDRESSED;
				wStatus = phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tReadBuffer                       : " );
				wStatus = phalICode_ReadBuffer ( &stICode, 0x00, AES_KEY_0_ADDRESS, AES_KEY_0_VERSION, &pReadBuffer, &wReadBufLen );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tResponse Data                    : ");
				PrintData ( pReadBuffer, wReadBufLen, "%02X ", "\n" );
				break;

			case '5':
				printf ( "Demo for ReadBuffer command with verification ----------------------------------------\n" );

				/* Reset configuration memory if engineering sample is used. */
				UpdateConfigMemory();

				printf ( "\tNon Addressed mode configuration : " );
				bFlags = PHPAL_SLI15693_FLAG_DATA_RATE;
				wStatus = phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags );
				CHECK_SUCCESS ( wStatus, 1 );

				phhalHw_FieldReset ( pHal_Generic );
				CHECK_SUCCESS ( wStatus, 0 );

				phhalHw_FieldOn ( pHal_Generic );
				CHECK_SUCCESS ( wStatus, 0 );

				printf ( "\tMoving to Ready state            : " );
				wStatus = phpalSli15693_ResetToReady ( stICode.pPalSli15693DataParams );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tChallenge                        : " );
                wStatus = phalICode_Challenge(&stICode, PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL, PHAL_ICODE_CHALLENGE_TAM_1,
                    CARD_KEY_0, NULL, 0);
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tAddressed mode configuration     : " );
				bFlags = PHPAL_SLI15693_FLAG_DATA_RATE | PHPAL_SLI15693_FLAG_ADDRESSED;
				wStatus = phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tReadBuffer                       : " );
				wStatus = phalICode_ReadBuffer ( &stICode, 0x01, AES_KEY_0_ADDRESS, AES_KEY_0_VERSION, &pReadBuffer, &wReadBufLen );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tResponse Data                    : ");
				PrintData ( pReadBuffer, wReadBufLen, "%02X ", "\n" );
				break;

			case '6':
				printf ( "Demo for WriteConfig command ---------------------------------------------------------\n" );

				WriteConfig ( (uint8_t * ) PRINT_MACRO ( AUTHENTICATION_LIMIT ), AUTHENTICATION_LIMIT, gaAuthLimit, sizeof ( gaAuthLimit ) );
				WriteConfig ( (uint8_t * ) PRINT_MACRO ( KEY_0 ), KEY_0, gaKey_0, sizeof ( gaKey_0 ) );
				WriteConfig ( (uint8_t * ) PRINT_MACRO ( KEY_PREVILAGE_0 ), KEY_PREVILAGE_0, gaKeyPrevilage_0, sizeof ( gaKeyPrevilage_0 ) );
				WriteConfig ( (uint8_t * ) PRINT_MACRO ( KEY_HEADER_0 ), KEY_HEADER_0, gaKeyHeader_0, sizeof ( gaKeyHeader_0 ) );

				WriteConfig ( (uint8_t * ) PRINT_MACRO ( CRYPTO_CONFIGURATION_HEADER ), CRYPTO_CONFIGURATION_HEADER, gaCyptoConfigHeader, sizeof ( gaCyptoConfigHeader ) );
				WriteConfig ( (uint8_t * ) PRINT_MACRO ( GLOBAL_CRYPTO_HEADER ), GLOBAL_CRYPTO_HEADER, gaGlobalCryptoHeader, sizeof ( gaGlobalCryptoHeader ) );
				break;

			case '7':
				printf ( "Demo for ReadConfig command ----------------------------------------------------------\n" );

				ReadConfig ( (uint8_t * ) PRINT_MACRO ( GLOBAL_CRYPTO_HEADER ), GLOBAL_CRYPTO_HEADER, aReadConfigBuffer, sizeof ( aReadConfigBuffer ) );
				ReadConfig ( (uint8_t * ) PRINT_MACRO ( CRYPTO_CONFIGURATION_HEADER ), CRYPTO_CONFIGURATION_HEADER, aReadConfigBuffer, sizeof ( aReadConfigBuffer ) );
				ReadConfig ( (uint8_t * ) PRINT_MACRO ( AUTHENTICATION_LIMIT ), AUTHENTICATION_LIMIT, aReadConfigBuffer, sizeof ( aReadConfigBuffer ) );
				ReadConfig ( (uint8_t * ) PRINT_MACRO ( KEY_HEADER_0 ), KEY_HEADER_0, aReadConfigBuffer, sizeof ( aReadConfigBuffer ) );
				ReadConfig ( (uint8_t * ) PRINT_MACRO ( KEY_PREVILAGE_0 ), KEY_PREVILAGE_0, aReadConfigBuffer, sizeof ( aReadConfigBuffer ) );
				ReadConfig ( (uint8_t * ) PRINT_MACRO ( KEY_0 ), KEY_0, pReadBuffer, 16 );
				break;

			case '8':
				printf ( "Demo for PickRandomID command --------------------------------------------------------\n" );

				/* Reset configuration memory if engineering sample is used. */
				UpdateConfigMemory ();

				/* Print the UID. */
				printf ( "\tUID before PickRandomID Command      : ");
				PrintData ( aUid, sizeof ( aUid ), "%02X ", "\n" );

				printf ( "\tMutual Authenticate (Privacy Enable) : " );
				wStatus = phalICode_AuthenticateMAM ( &stICode, PHAL_ICODE_OPTION_OFF, AES_KEY_0_ADDRESS, AES_KEY_0_VERSION, CARD_KEY_0,
					PHAL_ICODE_PURPOSE_MAM2_ENABLE_PRIVACY, NULL, 0 );
				CHECK_SUCCESS ( wStatus, 1 );

				/* Switch on the field */
				wStatus = phhalHw_FieldReset ( pHal_Generic );
				CHECK_SUCCESS ( wStatus, 0 );

				/* Perform Inventory. */
				bOption = PHPAL_SLI15693_ACTIVATE_ADDRESSED;
				bFlags = ( PHPAL_SLI15693_FLAG_NBSLOTS | PHPAL_SLI15693_FLAG_INVENTORY | PHPAL_SLI15693_FLAG_DATA_RATE );
				wStatus = phpalSli15693_ActivateCard ( stICode.pPalSli15693DataParams, bOption, bFlags, bAfi, aMask, bMaskLength, &bDsfid, aUid, &bMoreCardsAvaliable );
				CHECK_SUCCESS ( wStatus, 0 );

				/* Print the UID. */
				printf ( "\tUID after Privacy enable             : ");
				PrintData ( aUid, sizeof ( aUid ), "%02X ", "\n" );

				printf ( "\tEnable Pick Random ID                : " );
				wStatus = phalICode_PickRandomID ( &stICode );
				CHECK_SUCCESS ( wStatus, 1 );

				/* Activate card */
				bOption = PHPAL_SLI15693_ACTIVATE_ADDRESSED;
				bFlags = ( PHPAL_SLI15693_FLAG_NBSLOTS | PHPAL_SLI15693_FLAG_INVENTORY | PHPAL_SLI15693_FLAG_DATA_RATE );
				wStatus = phalICode_Inventory( &stICode, bFlags, bAfi, aMask, bMaskLength, &bDsfid, aUid );
				CHECK_SUCCESS ( wStatus, 0 );

				printf ( "\tUID after PickRandomID Command       : ");
				PrintData ( aUid, sizeof ( aUid ), "%02X ", "\n" );

				printf ( "\tMutual Authenticate (Privacy Disable): " );
				wStatus = phalICode_AuthenticateMAM ( &stICode, PHAL_ICODE_OPTION_OFF, AES_KEY_0_ADDRESS, AES_KEY_0_VERSION, CARD_KEY_0, PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY,
					NULL, 0 );
				CHECK_SUCCESS ( wStatus, 1 );

				/* Switch on the field */
				wStatus = phhalHw_FieldReset ( pHal_Generic );
				CHECK_SUCCESS ( wStatus, 0 );

				/* Activate card */
				bOption = PHPAL_SLI15693_ACTIVATE_ADDRESSED;
				bFlags = ( PHPAL_SLI15693_FLAG_NBSLOTS | PHPAL_SLI15693_FLAG_INVENTORY | PHPAL_SLI15693_FLAG_DATA_RATE );
				wStatus = phalICode_Inventory( &stICode, bFlags, bAfi, aMask, bMaskLength, &bDsfid, aUid );
				CHECK_SUCCESS ( wStatus, 0 );

				/* Print the UID. */
				printf ( "\tUID after Privacy disable            : ");
				PrintData ( aUid, sizeof ( aUid ), "%02X ", "\n" );
				break;

			case '9':
				printf ( "Demo for ReadMultipleBlocks command --------------------------------------------------\n" );

				/* Reset configuration memory if engineering sample is used. */
				UpdateConfigMemory();

				/* Disable bufferring by default. */
				wStatus = phalICode_SetConfig ( &stICode, PHAL_ICODE_CONFIG_ENABLE_BUFFERING, PH_OFF );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tEnter Starting Block No       : " );
				scanf ( "%d", &dwBlockNo );

				printf ( "\tEnter Number of Blocks to Read: " );
				scanf ( "%d", &dwNoOfBlocks );
				printf ( "\n" );

				printf ( "\tAuthenticate       : " );
				wStatus = phalICode_AuthenticateTAM1 ( &stICode, PHAL_ICODE_OPTION_OFF, AES_KEY_0_ADDRESS, AES_KEY_0_VERSION, CARD_KEY_0, NULL, 0 );
				CHECK_SUCCESS ( wStatus, 1 );

				/* Enable bufferring if reader is CM1 and the number of blocks is more that 60. */
				if ( ( dwNoOfBlocks > 60 ) && (bReaderType) )
				{
					wStatus = phalICode_SetConfig ( &stICode, PHAL_ICODE_CONFIG_ENABLE_BUFFERING, PH_ON );
					CHECK_SUCCESS ( wStatus, 1 );
				}

				printf ( "\tReadMultipleBlocks : " );
				wStatus = phalICode_ReadMultipleBlocks ( &stICode, PHAL_ICODE_OPTION_OFF, (uint8_t )dwBlockNo, (uint8_t ) dwNoOfBlocks, aReadData, &wReadDataLen );
				CHECK_SUCCESS ( wStatus, 1 );

				printf ( "\tNo Of Blocks Read  : %02X (%03d)\n", ( wReadDataLen / 4 ), ( wReadDataLen / 4 ) );
				printf ( "\tData Length        : %02X (%03d)\n", wReadDataLen, wReadDataLen );
				printf ( "\tData\n");

				for ( wIteration = 0; wIteration < ( wReadDataLen / 4 ); wIteration++ )
				{
					printf ( "\t         Block %03d: ", wIteration );
					PrintData ( &aReadData[wIteration * 4], 4, "%02X ", "\n");
				}
				break;

			default:
				printf ( "\n\n" );
				printf ( "Invalid option selected.\n" );
				break;
		}

		printf ( "\n\n" );
		printf ( "Press any key to continue with the demonstration.\n" );
		printf ( "Press x to exit the demonstration.\n" );
		dwChoice = _getch ();
	}while( ( dwChoice != 'x' ) && ( dwChoice != 'X' ) );

	return 0;
}

phStatus_t UpdateKeystore ()
{
	phStatus_t wStatus = 0;

	/* Add the required keys to keystore. */
	PH_CHECK_SUCCESS_FCT ( wStatus, phKeyStore_FormatKeyEntry ( &stKeyStore, AES_KEY_0_ADDRESS, PH_KEYSTORE_KEY_TYPE_AES128 ) );
	PH_CHECK_SUCCESS_FCT ( wStatus, phKeyStore_SetKeyAtPos ( &stKeyStore, AES_KEY_0_ADDRESS, 0x00, PH_KEYSTORE_KEY_TYPE_AES128, gaKey_0, AES_KEY_0_VERSION ) );

	return PH_ERR_SUCCESS;
}

phStatus_t ResetConfigMemory()
{
	phStatus_t wStatus = 0;
	uint8_t bFlags = 0;
	uint8_t bIteration = 0;
	uint16_t wRxLength = 0;

	uint8_t aTxData[7];
	uint8_t* pRxData = NULL;

	if ( bIsEnggSample )
	{
		bFlags = PHPAL_SLI15693_FLAG_DATA_RATE;
		aTxData[0] = 0xF2; aTxData[1] = 0x1E; aTxData[2] = 0x00; aTxData[3] = 0x00;	aTxData[4] = 0x81; aTxData[5] = 0x00; aTxData[6] = 0x81;
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_TIMEOUT_MS, 10 ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_Exchange( &stPal15693_Sw, 0x00, aTxData, sizeof ( aTxData ), &pRxData, &wRxLength ) );

		aTxData[0] = 0xF2; aTxData[1] = 0x1F; aTxData[2] = 0x00; aTxData[3] = 0x00;	aTxData[4] = 0x00; aTxData[5] = 0x00; aTxData[6] = 0x00;
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_TIMEOUT_MS, 10 ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_Exchange( &stPal15693_Sw, 0x00, aTxData, sizeof ( aTxData ), &pRxData, &wRxLength ) );

		aTxData[0] = 0xF2; aTxData[1] = 0x20; aTxData[2] = 0x00; aTxData[3] = 0x00;	aTxData[4] = 0x81; aTxData[5] = 0x00; aTxData[6] = 0x00;
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_TIMEOUT_MS, 10 ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_Exchange( &stPal15693_Sw, 0x00, aTxData, sizeof ( aTxData ), &pRxData, &wRxLength ) );

		for ( bIteration = 0; bIteration < 4; bIteration++)
		{
			aTxData[0] = 0xF2; aTxData[1] = (0x24 + bIteration); aTxData[2] = 0x00;
			memcpy ( &aTxData[3], &gaKey_0[4 * bIteration], 4 );
			PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_TIMEOUT_MS, 10 ) );
			PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_SetConfig ( stICode.pPalSli15693DataParams, PHPAL_SLI15693_CONFIG_FLAGS, bFlags ) );
			PH_CHECK_SUCCESS_FCT ( wStatus, phpalSli15693_Exchange( &stPal15693_Sw, 0x00, aTxData, sizeof ( aTxData ), &pRxData, &wRxLength ) );
		}
	}

	return PH_ERR_SUCCESS;
}

phStatus_t UpdateConfigMemory()
{
	phStatus_t wStatus = 0;

	if ( bIsEnggSample )
	{
		PH_CHECK_SUCCESS_FCT ( wStatus, phalICode_WriteConfig ( &stICode, PHAL_ICODE_OPTION_OFF, AUTHENTICATION_LIMIT, gaAuthLimit ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phalICode_WriteConfig ( &stICode, PHAL_ICODE_OPTION_OFF, KEY_PREVILAGE_0, gaKeyPrevilage_0 ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phalICode_WriteConfig ( &stICode, PHAL_ICODE_OPTION_OFF, KEY_HEADER_0, gaKeyHeader_0 ) );

		PH_CHECK_SUCCESS_FCT ( wStatus, phalICode_WriteConfig ( &stICode, PHAL_ICODE_OPTION_OFF, CRYPTO_CONFIGURATION_HEADER, gaCyptoConfigHeader ) );
		PH_CHECK_SUCCESS_FCT ( wStatus, phalICode_WriteConfig ( &stICode, PHAL_ICODE_OPTION_OFF, GLOBAL_CRYPTO_HEADER, gaGlobalCryptoHeader ) );
	}

	return PH_ERR_SUCCESS;
}

uint8_t HexString2Hex ( uint8_t bData )
{
	if ( ( bData >= '0' ) &&  ( bData <= '9' ) )
		return (bData - 48);
	else
	{
		if ( ( bData == 'A') || ( bData == 'a' ) )
			return 10;
		if ( ( bData == 'B') || ( bData == 'b' ) )
			return 11;
		if ( ( bData == 'C') || ( bData == 'c' ) )
			return 12;
		if ( ( bData == 'D') || ( bData == 'd' ) )
			return 13;
		if ( ( bData == 'E') || ( bData == 'e' ) )
			return 14;
		if ( ( bData == 'F') || ( bData == 'f' ) )
			return 15;
		else
			return 0xFF;
	}
}

void PrintData ( uint8_t * pBuffer, uint32_t dwLength, char* pFormat, char* pSpecialChar )
{
    uint32_t dwIndex;

    for ( dwIndex = 0; dwIndex < dwLength; ++dwIndex )
        printf ( pFormat, pBuffer[dwIndex] );

    printf ( pSpecialChar );
}

phStatus_t ReadConfig ( uint8_t* pName, uint8_t bBlock, uint8_t* pBuffer, uint16_t wBuffLen )
{
	phStatus_t wStatus = 0;
	uint16_t wBuffLenTmp = 0;

	memset ( pBuffer, 0x00, wBuffLen );
	printf ( "\tBlock : %s ( 0x%02X, %02d )\n", pName, bBlock, bBlock );
	wStatus = phalICode_ReadConfig ( &stICode, PHAL_ICODE_OPTION_OFF, bBlock, ( uint8_t ) ( wBuffLen / 4 ), &pBuffer, &wBuffLenTmp );
	printf ( "\tStatus: " );
	CHECK_SUCCESS ( wStatus, 1 );
	printf ( "\tData  : " );
	PrintData ( pBuffer, wBuffLen, "%02X ", "\n\n" );

	return PH_ERR_SUCCESS;
}

phStatus_t WriteConfig ( uint8_t* pName, uint8_t bBlock, uint8_t* pBuffer, uint8_t bBuffLen )
{
	phStatus_t wStatus = 0;
	uint8_t bIteration = 0;

	printf ( "\tBlock : %s ( 0x%02X, %02d )\n", pName, bBlock, bBlock );

	for ( bIteration = 0; bIteration < ( bBuffLen / 4); bIteration++ )
		wStatus = phalICode_WriteConfig ( &stICode, PHAL_ICODE_OPTION_OFF, ( bBlock + bIteration ), (pBuffer + ( 4 * bIteration ) ) );
	printf ( "\tStatus: " );
	CHECK_SUCCESS ( wStatus, 1 );
	printf ( "\tData  : " );
	PrintData ( pBuffer, bBuffLen, "%02X ", "\n\n" );

	return PH_ERR_SUCCESS;
}

/* Logging function. */
#ifdef NXPBUILD__PH_LOG
void IcodeDna_Log ( void* pDataParams, uint8_t bOption, phLog_LogEntry_t* pEntries, uint16_t wEntryCount )
{
    uint16_t wIteration;
    UNUSED_VARIABLE ( bOption );
    UNUSED_VARIABLE ( pDataParams );

    for ( wIteration = 0; wIteration < wEntryCount; wIteration++ )
    {
        if ( wIteration != 0 )
        {
            printf ( "%s : ", pEntries[wIteration].pString );
        }
        else
        {
            printf ( "-----------------------------------------\n" );
            printf ( "%s\n", pEntries[0].pString );
            printf ( "-----------------------------------------\n" );
        }
		if ( pEntries[wIteration].wDataLen != 0 )
			PrintData ( ( uint8_t * ) ( pEntries[wIteration].pData ), pEntries[wIteration].wDataLen, "%02X ", "\t" );
		else
			PrintData ( ( uint8_t * ) ( pEntries[wIteration].pData ), pEntries[wIteration].wDataLen, "%02X ", "" );
    }
    printf("\n\n");
}
#endif /* NXPBUILD__PH_LOG */
