/*
 * Copyright 2015 - 2017, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header
 */
#include "Example-Rd710_CM1_ICode.h"

#define PCSC_READER_P2_NO_SAM_NAME  "NXP Pegoda N CL 0 0"     /**< Pegoda 2: no SAM reader */
#define GLOBAL_BUFFER_SIZE          0x012CU                 /**< global size of transmit and recieve buffer */
#define PWD_IDENTIFIER_PRIVACY 0x04
/* This 33 bytes string is used for Originality check.
 * NOTE: This should be provided by product team once the key is finalized.
 */
const char *publickey_str = "048878A2A2D3EEC336B4F261A082BD71F9BE11C4E2E896648B32EFA59CEA6E59F0";

int __cdecl main()
{
    /* ******************************************************************************* */
    /* DECLARATION */
    /* ******************************************************************************* */
    /* data parameter storage */
	phbalReg_PcscWin_DataParams_t balPcsc;            /**< PCSC (Windows) BAL parameter structure */
    phhalHw_Rd710_DataParams_t	  halRd710;            /**< Rd710 HAL parameter structure */
    phbalReg_SerialWin_DataParams_t balSerial;
    phhalHw_Rc663_DataParams_t      halRc663;
    phpalSli15693_Rd710_DataParams_t   palSli_Rd710;
    phpalSli15693_Sw_DataParams_t   palSli_Sw;
    phalICode_Sw_DataParams_t   alICode;

	void * pHal = NULL;			 /**< Pointer to the parameter structure of the PAL layer. */
	void * palSli = NULL;		 /**< Pointer to the parameter structure of the PAL layer. */
    /* variables used by the BAL copmonent */
    uint8_t * pComPort = (uint8_t*)"COM4";
	uint8_t pEasIdValue[2] = {0x00, 0x00};
	uint8_t bEasIdMaskLength = 0x00;

	uint8_t pProtectionStates[60] = {'\0'};
	uint16_t wNumReceivedStates = 0;

	uint8_t * pSign = NULL;
	uint16_t wSignLen = 0;

	uint8_t * pEas = NULL;
	uint16_t wEasLen = 0;

    phStatus_t status;
    uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader receive buffer */
    uint8_t aAtr[256];                /**< Atr */
    uint8_t bIndex;
    int iChoice, choice;                      /**< choice */

    /* variables used by the PAL component */
    uint8_t bOption, bFlags, bAfi;
    uint8_t pMask[PHPAL_SLI15693_UID_LENGTH], bMaskLength;
    uint8_t bDsfid, pUid[PHPAL_SLI15693_UID_LENGTH];
    uint8_t pUid2[PHPAL_SLI15693_UID_LENGTH];
    uint8_t bRxLength, bMoreCardsAvaliable;

    uint8_t  bBlockNo;
    uint16_t wNoOfBlocks;
    uint8_t  bUidLength;
	uint8_t  bExtendedOption;
	uint8_t  bCID[2] = { 0x00, 0x00};
	uint8_t  bCDIDOut[20];
	uint8_t  aData[PHAL_ICODE_BLOCK_SIZE * 80] = {'\0'};
    uint16_t wDataLength;

    /* variables used by the AL (ISO15693) component */
    uint8_t pOriginalData[PHAL_ICODE_BLOCK_SIZE];
    uint8_t** ppData = (uint8_t**)&aData;
	uint8_t* pData = aData;

	printf("Press 1 To run with Rc663 reader.\n\n");
	printf("Press 2 To run with Rd710 CM1 reader.\n\n");
	printf("Press x To Quit.\n\n");
	choice=_getch();
	switch(choice)
	{
	 case('1') :
		printf("\nNxpRdLib ANSI-C Rc663 reader Example Program \n\n");
		printf("Please ensure that a Rc663 reader is connected and in working condition.\n\n");
		printf("Performing startup...\n\n");
		printf("-------------------------\n");

		/* initialise the 'bus abstraction layer' BAL: */
		/* use the serial interface */
		status = phbalReg_SerialWin_Init(&balSerial, sizeof(phbalReg_SerialWin_DataParams_t), 0, NULL);
		CHECK_SUCCESS(status);

		status = phbalReg_SetPort(&balSerial, pComPort);
		CHECK_SUCCESS(status);

		/* initialise the 'hardware abstraction layer' HAL: */
		/* use the RC663 Reader IC, glue it together with the BAL component */
		status = phhalHw_Rc663_Init(
			&halRc663,
			sizeof(phhalHw_Rc663_DataParams_t),
			&balSerial,
			NULL,
			aHalBufferReaderTx,
			sizeof(aHalBufferReaderTx),
			aHalBufferReaderRx,
			sizeof(aHalBufferReaderRx));
		CHECK_SUCCESS(status);

		/**< Set HAL pointer */
		pHal = &halRc663;

		/* Opening the port */
		status = phbalReg_OpenPort(&balSerial);
		CHECK_SUCCESS(status);

		/* set the PCs baud rate to 115200 */
		status = phbalReg_SetConfig(&balSerial, PHBAL_REG_SERIALWIN_CONFIG_BITRATE, PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200);
		CHECK_SUCCESS(status);

		/* On some machines the RS232 physical interface jitters when the port is opened. For that case, */
		/* send a dummy command in order to the resynchronise the interface. */
		phhalHw_WriteRegister(&halRc663, 0x37, 0xFF);

		/* Initialise the Sli15693 protocol. */
		status = phpalSli15693_Sw_Init(&palSli_Sw, sizeof(palSli_Sw), &halRc663);
		CHECK_SUCCESS(status);
		palSli = &palSli_Sw;		/**< Set PAL pointer */

		break;
	 case('2'):
		printf("\nNxpRdLib ANSI-C Rd710 CM1 reader Example Program \n\n");
		printf("Please ensure that a Pegoda-2 CM1 reader is connected and in working condition.\n\n");
		printf("Performing startup...\n\n");
		printf("-------------------------\n");

		/* Init. PCSC BAL */
		status = phbalReg_PcscWin_Init(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
			sizeof(phbalReg_PcscWin_DataParams_t),  /**< [In] Specifies the size of the data parameter structure. */
			aAtr,                                   /**< [In] Pointer to buffer used for storing the ATR after card activation. */
			256);                                   /**< [In] Size of the ATR buffer. */
		CHECK_SUCCESS(status);

		/* Set the BAL Communication Configuration values */
		status = phbalReg_SetConfig(&balPcsc,           /**< [In] Pointer to this layer's parameter structure. */
			PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,          /**< BAL Communication Configs */
			PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED);/**< BAL Communication Configuration values */
		CHECK_SUCCESS(status);

		/* set the BAL Communication Configs */
		status = phbalReg_SetConfig(&balPcsc,       /**< [In] Pointer to this layer's parameter structure. */
			PHBAL_REG_PCSCWIN_CONFIG_SHARE,         /**< BAL Communication Configs */
			PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);  /**< BAL Communication Configuration values */
		CHECK_SUCCESS(status);

		/* Select Port to be used (no SAM reader) */
		status = phbalReg_SetPort(&balPcsc,         /**< [In] Pointer to this layer's parameter structure. */
			(uint8_t*)PCSC_READER_P2_NO_SAM_NAME);  /**< [In] Port Name as String. */
		CHECK_SUCCESS(status);

		/**
		* Initialise the Rd710 HAL component:
		*   RD710 with GLOBAL_BUFFER_SIZE bytes send/receive buffer
		*/
		status = phhalHw_Rd710_Init(&halRd710,  /**< [In] Pointer to this layer's parameter structure. */
			sizeof(phhalHw_Rd710_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
			&balPcsc,                           /**< [In] Pointer to the lower layers parameter structure */
			0,                                  /**< [In] Slot number */
			aHalBufferReaderTx,                 /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
			sizeof(aHalBufferReaderTx),         /**< [In] Size of the global transmit buffer. */
			aHalBufferReaderRx,                 /**< [In] Pointer to global receive buffer used by the Exchange() function. */
			sizeof(aHalBufferReaderRx));        /**< [In] Size of the global receive buffer. */
		CHECK_SUCCESS(status);

		/**< Set HAL pointer */
		pHal = &halRd710;

		/* Open the PCSC port */
		status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
		CHECK_SUCCESS(status);

		/* Standard Rd710 Cmd Init. */
		status = (phStatus_t)phhalHw_Rd710_Cmd_InitReader(&halRd710);  /**< [In] Pointer to this layer's parameter structure. */
		CHECK_SUCCESS(status);

		/* Initialise the Sli15693 protocol. */
		status = phpalSli15693_Rd710_Init(&palSli_Rd710, sizeof(palSli_Rd710), &halRd710);
		CHECK_SUCCESS(status);
		palSli = &palSli_Rd710;		/**< Set PAL pointer */

		/* Initialise the Sli15693 protocol. */
		status = phpalSli15693_Sw_Init(&palSli_Sw, sizeof(palSli_Sw), &halRd710);
		CHECK_SUCCESS(status);
		palSli = &palSli_Sw;		/**< Set PAL pointer */
		break;

	default:
		printf(" Quitting ... \n");
		exit(0);
	}

	/* Configure HAL for Sli15693 tags */
	status = phhalHw_ApplyProtocolSettings(pHal, PHHAL_HW_CARDTYPE_ISO15693);
	CHECK_SUCCESS(status);

	/* Switch on the field */
	status = phhalHw_FieldReset(pHal);
	CHECK_SUCCESS(status);

    /* initialise the 'application layer' AL: */
    /* use the ICODE application, glue it together with the PAL component */
    status = phalICode_Sw_Init(&alICode, sizeof(alICode), palSli, NULL, NULL, NULL);

    /* ******************************************************************************* */
    /* CARD COMMUNICATION */
    /* ******************************************************************************* */
	do
	{
	printf("Press 1 for Activate Card, Get Serial Number and Inventory Read \n");
	printf("Press 2 for Read & Write Block \n");
	printf("Press 3 for Set Password & Write Password \n");
	printf("Press 4 for Un-addressed Mode Read/Write \n");
	printf("Press 5 for Selecting one-by-one card (changing context) \n");
	printf("Press 6 for EAS Alarm \n");
	printf("Press 7 for PWD Protect AFI/EAS \n");
	printf("Press 8 for Read Signature for SLIX-2 \n");
	printf("Press 9 for Inventory Read Extended \n");
	printf("Press a for Stay Quiet Persistent \n");
	printf("Press b for Read/Write Multiple Blocks \n");
	printf("Press c for Check Multiple Block Security Status \n");
	printf("Press d for Enable Privacy \n");
	printf("Press x to quit \n");
	iChoice = _getch();
	/* To See The Complete Log ,right Click on Console Window,Go to Properties->Layout->Screen Buffer Size->999 */
	switch(iChoice)
	{
	case('1'):
		/* Activate card */
		bOption = PHPAL_SLI15693_ACTIVATE_ADDRESSED;
		bFlags =  PHPAL_SLI15693_FLAG_NBSLOTS | PHPAL_SLI15693_FLAG_DATA_RATE;
		bAfi = 0xFF; /* Application Family: 0xFF --> all Applications */
		bMaskLength = 0;
		status = phpalSli15693_ActivateCard(palSli, bOption, bFlags,
											bAfi, pMask, bMaskLength,
											&bDsfid, pUid, &bMoreCardsAvaliable);
		printf("phpalSli15693_ActivateCard:  %04X\n", status);
        status = phpalSli15693_GetSerialNo(palSli, pUid, &bRxLength);
        printf("phpalSli15693_GetSerialNo:   %04X\nUID: ", status);
        printBufferHex(pUid, bRxLength);

        if(bRxLength == PHPAL_SLI15693_UID_LENGTH)
        {
            memcpy(pMask, pUid, PHPAL_SLI15693_UID_LENGTH);
            bMaskLength = PHPAL_SLI15693_UID_LENGTH * 8;
        }

        /* Sli Inventory Read (PAL) */
        bFlags = 0x32;
        bAfi = 0x00;
        bBlockNo = 0;
        wNoOfBlocks = 28;
        status = phpalSli15693_InventoryRead(palSli, bFlags, bAfi, pMask, bMaskLength,
                                              bBlockNo, wNoOfBlocks,
                                              pUid, &bUidLength,
                                              aData, &wDataLength);
        printf("phpalSli15693_InventoryRead: %04X\n", status);
		if(PH_ERR_SUCCESS == status)
		{
			printf("Block Data, %3d bytes:\n", wDataLength);
			for(bIndex = 0; bIndex < wNoOfBlocks; bIndex++)
			{
				printf("Block %2d: ",bIndex);
				printBufferHex(aData+(bIndex*PHPAL_SLI15693_BLOCK_LENGTH),
									(uint8_t)PHPAL_SLI15693_BLOCK_LENGTH);
			}
		}
		break;

	case('2'):
        /* Store the blocks Data in pOriginalData, then write the block, read it back and
         * restore it finally. */
        /* Write one block (AL) */
        bOption = PHAL_ICODE_OPTION_DEFAULT;
        bBlockNo = 0x0a;

        status = phalICode_ReadSingleBlock(&alICode, bOption, bBlockNo, ppData, &wDataLength);
        CHECK_SUCCESS(status);
        /*memcpy(pOriginalData, (uint8_t*)*ppData, wDataLength);*/

        printf("Block %2d Data, %3d bytes:\n", bBlockNo, wDataLength);
        printBufferHex((uint8_t*)*ppData, (uint8_t)wDataLength);

        memcpy(aData, "\xAB\xCD\xEF\xBA", 4);
        wDataLength = 4;
        status = phalICode_WriteSingleBlock(&alICode, bOption, bBlockNo, aData, (uint8_t) wDataLength);
		CHECK_SUCCESS(status);
        printf("phalI15693_WriteSingleBlock:          %04X\n", status);
        if(PH_ERR_SUCCESS == status)
        {
            printf("Wrote Block %2d with data Data: ", bBlockNo);
            printBufferHex(aData, (uint8_t)wDataLength);
        }

        /* Read one block (AL) */
        status = phalICode_ReadSingleBlock(&alICode, bOption, bBlockNo, ppData, &wDataLength);
        printf("phalI15693_ReadSingleBlock:          %04X\n", status);


        if(PH_ERR_SUCCESS == status)
        {
            printf("Block %2d Data, %3d bytes:\n", bBlockNo, wDataLength);
            printBufferHex((uint8_t*)*ppData, (uint8_t)wDataLength);
        }
        status = phalICode_WriteSingleBlock(&alICode, bOption, bBlockNo, pOriginalData, (uint8_t) wDataLength);
        CHECK_SUCCESS(status);
		break;

	case('3'):
		/* Set Password and Write Password */
		status = SetPWD(&alICode, 0x10);
		CHECK_SUCCESS(status);
		/* Write the Password */
		//status = phalICode_WritePassword(&alICode, bPwdIdentifier, pXorPwd);
        //CHECK_SUCCESS(status);
		break;

	case('4'):
		/* Un-addressed Mode */
		bFlags = 0x02;
		bAfi = 0xFF; /* Application Family: 0xFF --> all Applications */
		bMaskLength = 0;
		/* Setting the address flag */
		phpalSli15693_SetConfig(palSli, PHPAL_SLI15693_CONFIG_FLAGS, bFlags);
        /* Read one block (AL) */
        bOption = PHAL_ICODE_OPTION_DEFAULT;
        bBlockNo = 0x0A;
        status = phalICode_ReadSingleBlock(&alICode, bOption, bBlockNo, ppData, &wDataLength);
        CHECK_SUCCESS(status);
		break;

	case('5'):
		/* Selecting cards (changing context) */
		bOption = PHPAL_SLI15693_ACTIVATE_ADDRESSED;
		bFlags =  PHPAL_SLI15693_FLAG_NBSLOTS | PHPAL_SLI15693_FLAG_DATA_RATE;
		bAfi = 0xFF; /* Application Family: 0xFF --> all Applications */
		bMaskLength = 0;
		status = phpalSli15693_ActivateCard(palSli, bOption, bFlags,
											bAfi, pMask, bMaskLength,
											&bDsfid, pUid2, &bMoreCardsAvaliable);
		printf("phpalSli15693_ActivateCard:  %04X\n", status);
        status = phpalSli15693_GetSerialNo(palSli, pUid2, &bRxLength);
        printf("phpalSli15693_GetSerialNo:   %04X\nUID: ", status);
        printBufferHex(pUid2, bRxLength);

		while(bMoreCardsAvaliable)
		{
			status = phpalSli15693_StayQuiet(palSli);
			CHECK_SUCCESS(status);
			status = phpalSli15693_ActivateCard(palSli, bOption, bFlags,
									bAfi, pMask, bMaskLength,
									&bDsfid, pUid, &bMoreCardsAvaliable);
			printf("phpalSli15693_ActivateCard:  %04X\n", status);
			status = phpalSli15693_GetSerialNo(palSli, pUid, &bRxLength);
			printf("phpalSli15693_GetSerialNo:   %04X\nUID: ", status);
			printBufferHex(pUid, bRxLength);

			if(bRxLength == PHPAL_SLI15693_UID_LENGTH)
			{
				memcpy(pMask, pUid, PHPAL_SLI15693_UID_LENGTH);
				bMaskLength = PHPAL_SLI15693_UID_LENGTH * 8;
			}
		}
        /* Read single block */
        bOption = PHAL_ICODE_OPTION_DEFAULT;
        bBlockNo = 0x0A;

        status = phalICode_ReadSingleBlock(&alICode, bOption, bBlockNo, ppData, &wDataLength);
        CHECK_SUCCESS(status);
		printf("Block %2d Data, %3d bytes:\n", bBlockNo, wDataLength);
        printBufferHex((uint8_t*)*ppData, (uint8_t)wDataLength);
		status = phpalSli15693_StayQuiet(palSli);
		CHECK_SUCCESS(status);
		status = phpalSli15693_SetSerialNo(palSli, pUid2, PHPAL_SLI15693_UID_LENGTH);
		CHECK_SUCCESS(status);
		status = phpalSli15693_Select(palSli);
		CHECK_SUCCESS(status);
        status = phalICode_ReadSingleBlock(&alICode, bOption, bBlockNo, ppData, &wDataLength);
        CHECK_SUCCESS(status);
		printf("Block %2d Data, %3d bytes:\n", bBlockNo, wDataLength);
        printBufferHex((uint8_t*)*ppData, (uint8_t)wDataLength);
		break;

	case('6'):
		/* EAS Alarm */
		/* Set Password and Write Password (Required only if the card is password protected) */
		status = SetPWD(&alICode, 0x10);
		CHECK_SUCCESS(status);
		bOption = 0x00;
		bFlags =  PHPAL_SLI15693_FLAG_NBSLOTS | PHPAL_SLI15693_FLAG_DATA_RATE;
		/* Setting up the EAS */
		status = phalICode_SetEAS(&alICode, PHAL_ICODE_OPTION_DEFAULT);
		CHECK_SUCCESS(status);
		/* Calling EAS Alarm with option 0 and EAS mask length as 0 */
		status = phalICode_EASAlarm(&alICode, bOption, pEasIdValue, bEasIdMaskLength, &pEas, &wEasLen);
		CHECK_SUCCESS(status);
		printBufferHex(pEas, (uint8_t) wEasLen);
		/* Calling EAS Alarm with option 1 and EAS mask length as 0 */
		bOption = 0x01;
		status = phalICode_EASAlarm(&alICode, bOption, pEasIdValue, bEasIdMaskLength, &pEas, &wEasLen);
		CHECK_SUCCESS(status);
		printBufferHex(pEas, 2);
		/* Calling EAS Alarm with option 1, EAS mask length as 0x10 (2 Bytes)
		 * and pEasIdValue should be 2 byte EAS (out EAS of the above last EASAlarm run) */
		bOption = 0x01;
		bEasIdMaskLength = 0x10;
		memcpy(pEasIdValue, pEas, 2);
		status = phalICode_EASAlarm(&alICode, bOption, pEasIdValue, bEasIdMaskLength, &pEas, &wEasLen);
		CHECK_SUCCESS(status);
		printBufferHex(pEas, (uint8_t) wEasLen);
		break;

	case('7'):
		/* PWD Protect AFI/EAS */
		/* Set Password and Write Password */
		status = SetPWD(&alICode, 0x10);
		CHECK_SUCCESS(status);
		bFlags = 0x0062;	/* Setting flag value for ProtectAFI */
		/* Setting the Option flag */
		status = phpalSli15693_SetConfig(palSli, PHPAL_SLI15693_CONFIG_FLAGS, bFlags);
        CHECK_SUCCESS(status);
        /* Protect AFI */
		status = phalICode_PasswordProtectAFI(&alICode);
        CHECK_SUCCESS(status);
		break;

	case('8'):
		/* Read Signature and Verify Signature */
		status = phpalSli15693_GetSerialNo(palSli, pUid, &bRxLength);
        printf("phpalSli15693_GetSerialNo:   %04X\nUID: ", status);
        printBufferHex(pUid, bRxLength);
		/* Read Signature */
		status = phalICode_ReadSignature(&alICode, &pSign, &wSignLen);
        CHECK_SUCCESS(status);
		printBufferHex(pSign, (uint8_t) wSignLen);
		/* Signature Validation */
		status = SignatureValidation(pUid, pSign);
		CHECK_SUCCESS(status);
		break;

	case('9'):
		/* Activate card */
		bOption = PHPAL_SLI15693_ACTIVATE_ADDRESSED;
		bFlags =  PHPAL_SLI15693_FLAG_NBSLOTS | PHPAL_SLI15693_FLAG_DATA_RATE;
		bAfi = 0xFF; /* Application Family: 0xFF --> all Applications */
		bMaskLength = 0x00;
		status = phpalSli15693_ActivateCard(palSli, bOption, bFlags,
											bAfi, pMask, bMaskLength,
											&bDsfid, pUid, &bMoreCardsAvaliable);
		CHECK_SUCCESS(status);
		printf("phpalSli15693_ActivateCard:  %04X\n", status);
        status = phpalSli15693_GetSerialNo(palSli, pUid, &bRxLength);
        printf("phpalSli15693_GetSerialNo:   %04X\nUID: ", status);
        printBufferHex(pUid, bRxLength);

        if(bRxLength == PHPAL_SLI15693_UID_LENGTH)
        {
            memcpy(pMask, pUid, PHPAL_SLI15693_UID_LENGTH);
            bMaskLength = PHPAL_SLI15693_UID_LENGTH * 8;
        }

        /* Sli Inventory Read Extended(PAL) */
        bFlags = 0x22;
        bAfi = 0xFF;
        bBlockNo = 0;
        wNoOfBlocks = 28;
		bExtendedOption = 0x12;
		bMaskLength = 0x40;
        status = phpalSli15693_InventoryReadExtended(palSli, bFlags, bAfi, pMask, bMaskLength,
                                              bExtendedOption, bCID, bBlockNo, wNoOfBlocks,
                                              bCDIDOut, pUid, &bUidLength,
                                              aData, &wDataLength);
		CHECK_SUCCESS(status);
        printf("phpalSli15693_InventoryReadExtended: %04X\n", status);
		if(PH_ERR_SUCCESS == status)
		{
			printf("Block Data, %3d bytes:\n", wDataLength);
			for(bIndex = 0; bIndex < wNoOfBlocks; bIndex++)
			{
				printf("Block %2d: ",bIndex);
				printBufferHex(aData+(bIndex*PHPAL_SLI15693_BLOCK_LENGTH),
									(uint8_t)PHPAL_SLI15693_BLOCK_LENGTH);
			}
		}
		break;

	case('a'):
		/* Stay Quiet Persistent */
		/* Set Flags */
		bOption = PHPAL_SLI15693_ACTIVATE_ADDRESSED;
		bFlags =  PHPAL_SLI15693_FLAG_NBSLOTS | PHPAL_SLI15693_FLAG_DATA_RATE;
		status = phpalSli15693_SetConfig(palSli, PHPAL_SLI15693_CONFIG_FLAGS, bFlags);
        CHECK_SUCCESS(status);
		status = phpalSli15693_StayQuietPersistent(palSli);
		CHECK_SUCCESS(status);
		status = phalICode_EASAlarm(&alICode, bOption, pEasIdValue, bEasIdMaskLength, &pEas, &wEasLen);
		CHECK_SUCCESS(status);
		printBufferHex(pEas, (uint8_t) wEasLen);
		break;

	case('b'):
        /* Read and Write multiple block */
        bOption = PHAL_ICODE_OPTION_DEFAULT;
        bBlockNo = 0x00;
		wNoOfBlocks = 63;

        status = phalICode_ReadMultipleBlocks(&alICode, bOption, bBlockNo, (uint8_t) wNoOfBlocks, pData, &wDataLength);
        CHECK_SUCCESS(status);
        printBufferHex_Large(pData, wDataLength);

        /*wDataLength = (uint16_t)(wNoOfBlocks * 4);
        status = phalICode_WriteMultipleBlocks(&alICode, bOption, bBlockNo, wNoOfBlocks, aData, wDataLength);
		CHECK_SUCCESS(status);
        printf("phalI15693_WriteSingleBlock:          %04X\n", status);
        if(PH_ERR_SUCCESS == status)
        {
            printf("Wrote Block %2d with data Data: ", bBlockNo);
            printBufferHex_Large(aData, wDataLength);
        }*/
		break;

	case('c'):
        /* Block Security Status */
        bOption = PHAL_ICODE_OPTION_DEFAULT;
        bBlockNo = 0x02;
		wNoOfBlocks = 0x03;

		status = phalICode_GetMultipleBlockProtectionStatus(&alICode, bBlockNo, (uint8_t) wNoOfBlocks, pProtectionStates, &wNumReceivedStates);
		CHECK_SUCCESS(status);
        printBufferHex(pProtectionStates, (uint8_t) wNumReceivedStates);
		break;

	case('d'):
		/* Enable Privacy */
		status = EnablePrivacy(&alICode);
		CHECK_SUCCESS(status);
		/* Set Password with PWD Identifier as 0x04 for Privacy */
		status = SetPWD(&alICode, PWD_IDENTIFIER_PRIVACY);
		CHECK_SUCCESS(status);
		break;

	default:
		printf(" Quitting ... \n");
		break;
	}

	printf("\n\n----------------\n\n");
	printf("Do you like to Continue(1=Yes, 0=No)\n\n");
	iChoice = _getch();
	}while(iChoice != '0');
}

/**
 * Helper functions
 */
void printBufferHex(uint8_t * pBuffer, uint8_t bLength)
{
    uint8_t bIndex;

    for (bIndex = 0; bIndex < bLength; ++bIndex)
    {
        printf("%02X ", pBuffer[bIndex]);
    }
    printf("\n");
}

void printBufferHex_Large(uint8_t * pBuffer, uint16_t bLength)
{
    uint16_t bIndex;

    for (bIndex = 0; bIndex < bLength; ++bIndex)
    {
        printf("%02X ", pBuffer[bIndex]);
    }
    printf("\n");
}

phStatus_t SetPWD(phalICode_Sw_DataParams_t * alICode, uint8_t  bPwdIdentifier)
{
	phStatus_t status;
    uint8_t bIndex;
	uint8_t * pRnd = NULL;
	uint16_t wRndLen = 0;
	uint8_t  pXorPwd[4] = {0x00, 0x00, 0x00, 0x00};
	uint8_t  pXorPwdPrivacy[4] = {0x0F, 0x0F, 0x0F, 0x0F};
	/* Get the random number */
	status = phalICode_GetRandomNumber(alICode, &pRnd, &wRndLen);
    CHECK_SUCCESS(status);
	memcpy(&pRnd[2], pRnd, 2);
	printBufferHex(pRnd, 4);
	for (bIndex = 0; bIndex < 4; ++bIndex)
	{
		pXorPwd[bIndex] ^= pRnd[bIndex];
		pXorPwdPrivacy[bIndex] ^= pRnd[bIndex];
	}
	printBufferHex((bPwdIdentifier==0x04) ? pXorPwdPrivacy : pXorPwd, 4);
	/* Set the Password */
	status = phalICode_SetPassword(alICode, PHAL_ICODE_OPTION_OFF, bPwdIdentifier, (bPwdIdentifier == 0x04)
        ? pXorPwdPrivacy : pXorPwd);
	return status;
}

phStatus_t EnablePrivacy(phalICode_Sw_DataParams_t * alICode)
{
	phStatus_t status;
    uint8_t bIndex;
	uint8_t * pRnd = NULL;
	uint16_t wRndLen = 0;
	uint8_t  pXorPwd[4] = {0x0F, 0x0F, 0x0F, 0x0F};
	/* Get the random number */
	status = phalICode_GetRandomNumber(alICode, &pRnd, &wRndLen);
    CHECK_SUCCESS(status);
	memcpy(&pRnd[2], pRnd, 2);
	printBufferHex(pRnd, 4);
	for (bIndex = 0; bIndex < 4; ++bIndex)
	{
		pXorPwd[bIndex] ^= pRnd[bIndex];
	}
	printBufferHex(pXorPwd, 4);
	/* Set the Password */
	status = phalICode_EnablePrivacy(alICode, PHAL_ICODE_OPTION_OFF, pXorPwd);
	return status;
}

phStatus_t SignatureValidation(uint8_t * pUid, uint8_t * aSignature)
{
    BIGNUM *pk_bignum = BN_new();
    EC_POINT *public_key = NULL;
	unsigned char r[16];
    unsigned char s[16];
    char r_dest[33];
    char s_dest[33];
    int loop = 0;
	/* Create a EC_KEY for NID_secp128r1 */
	EC_KEY *pubKey = EC_KEY_new_by_curve_name(NID_secp128r1);
    const EC_GROUP  *ecgroup = EC_KEY_get0_group(pubKey);
    ECDSA_SIG *signature = ECDSA_SIG_new();
    if (signature == NULL)
    {
        return 1;
    }
    if (pubKey == NULL)
    {
        printf("Creation of PubKey failed \n");
        return 1;
    }
	/* Convert the hex public key x,y co-ordinates to BIGNUM */
    BN_hex2bn(&pk_bignum, publickey_str);
    public_key = EC_POINT_bn2point(ecgroup, pk_bignum, NULL, NULL);
	/* Set the public key point to EC_KEY */
    EC_KEY_set_public_key(pubKey, public_key);

	/* Extract the r and s part of the signature*/
    memcpy(r, aSignature, 16);
    memcpy(s, aSignature+16, 16);
	/* BIGNUM conversion function expects r in ASCII value */
    for(loop = 0;loop < 16; loop++)
    {
        sprintf_s((r_dest+(loop*2)), 3, "%02X", r[loop]);
        sprintf_s((s_dest+(loop*2)), 3, "%02X", s[loop]);
    }
    BN_hex2bn(&signature->r, r_dest);
    BN_hex2bn(&signature->s, s_dest);
	/* Signature Validation for the UID sent */
    if (ECDSA_do_verify(pUid, 0x08, signature, pubKey) == 1)
    {
        printf("Signature Validation SUCCESS!! ... \n");
        return PH_ERR_SUCCESS;
    }
    else
    {
        printf("Signature Validation FAILED!! ... \n");
        return 1;
    }
}

