/*
 * Copyright 2013, 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Rd710 MF0EV1 for ECDSA verification.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
* Header for this file
*/
#include "Example-Rd710_MF0EV1_ECDSA.h"

#pragma warning(disable:4996)

#define PCSC_READER_P2_NO_SAM_NAME  "NXP Pegoda N CL 0 0"     /**< Pegoda 2: no SAM reader */
#define GLOBAL_BUFFER_SIZE          0x012CU /**< global size of transmit and recieve buffer */
#define PHAL_MFUL_UNUSED_VARIABLE(x)  {for( ( x ) = ( x ) ; ( x ) != ( x ) ; );}

#define CHECK_SUCCESS(x)                                                \
    if ((x) != PH_ERR_SUCCESS)                                          \
{                                                                       \
    printf("An error occured (0x%04X), press any key to exit...", (x)); \
    _getch();                                                           \
    return 0;                                                           \
}

/* Utility function to print a buffer */
void PRINT_BUFFER(uint8_t * pBuffer, uint16_t wLength)
{
    uint16_t wIndex;

    for (wIndex = 0; wIndex < wLength; ++wIndex)
    {
        printf("%02X ", pBuffer[wIndex]);
    }
    printf("\n");
}

/* Logging function. Turned OFF currently */
#ifdef NXPBUILD__PH_LOG
void phalMful_log(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
    uint16_t i;
    PHAL_MFUL_UNUSED_VARIABLE(bOption);
    PHAL_MFUL_UNUSED_VARIABLE(pDataParams);

    for (i = 0; i < wEntryCount; i++ )
    {
        if ( i != 0 )
        {
            printf("%s : ", pEntries[i].pString);
        }
        else
        {
            printf("-----------------------------------------\n");
            printf("%s \n", pEntries[0].pString);
            printf("-----------------------------------------\n");
        }
        PRINT_BUFFER((uint8_t *)(pEntries[i].pData), pEntries[i].wDataLen);
    }
    printf("\n\n");
}
#endif /* NXPBUILD__PH_LOG */

 /**< complete Uid; uint8_t[4/7/10]. */
static uint8_t aUid[10];
/* Password for pwdAuth */
static uint8_t aPwd[PHAL_MFUL_WRITE_BLOCK_LENGTH] = {0xFF, 0xFF, 0xFF, 0xFF};
/* 33 Byte Public key string.
 * Supplied in X(16 bytes),Y(16 bytes) co-ordinate with "04" prefixed
 */
const char *publickey_str = "0490933bdcd6e99b4e255e3da55389a827564e11718e017292faf23226a96614b8";

#ifdef NXPBUILD__PH_LOG
phLog_RegisterEntry_t logRegisterEntries[16];
phLog_LogEntry_t logEntries[48];
#endif /* NXPBUILD__PH_LOG */

/**
* Reset card : Function to reset and activate the PICC
*
*/
uint8_t resetCard(phalMful_Sw_DataParams_t * palMful,             /**< Pointer to MIFARE Ultralight parameter structure. */
                  phpalMifare_Sw_DataParams_t * palMifare,        /**< Pointer to MIFARE parameter structure. */
                  phhalHw_Rd70x_DataParams_t * pHal,              /**< Pointer to HAL parameter structure */
                  phpalI14443p3a_Sw_DataParams_t * palI14443p3a)  /**< PAL-ISO14443P3A parameter structure */
{
    phStatus_t status;
    uint8_t aSak[1];                /**< Select Acknowledge; uint8_t. */
    uint8_t bMoreCardsAvaliable;    /**< more card available */
    uint8_t bLength;                /**< Length */

    /**
    * init. Ultralight component
    */
    status = phalMful_Sw_Init(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phalMful_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
        palMifare,                        /**< [In] Pointer to palMifare parameter structure. */
        NULL,                             /**< [In] Pointer to phKeystore parameter structure. */
        NULL,                             /**< [In] Pointer to phCrypto data parameters structure. */
        NULL);                            /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
    CHECK_SUCCESS(status);

    /**
    * Field Reset
    */
    status = phhalHw_FieldReset(pHal);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    /**
    * Activate Layer 3 card
    */
    status = phpalI14443p3a_ActivateCard(palI14443p3a,  /**< [In] PAL-ISO14443P3A parameter structure */
        NULL,                                           /**< [In] unknown UID */
        0x00,                                           /**< [In] UID length 0 */
        aUid,                                           /**< [Out] complete Uid; uint8_t[4/7/10]. */
        &bLength,                                       /**< [Out] Length of Uid; 4/7/10. */
        aSak,                                           /**< [Out] Select Acknowledge; uint8_t. */
        &bMoreCardsAvaliable);                          /**< [Out] Whether there are more cards in the field or not; uint8_t. */
    CHECK_SUCCESS(status);


    return PH_ERR_SUCCESS;
}

/**
* Originality Check Demo App
*
*/
uint8_t OriginalityCheckDemo(phalMful_Sw_DataParams_t * palMful)

{
    phStatus_t status;
    uint8_t bAddr = 0x0;
    uint8_t *aSignature;
    BIGNUM *pk_bignum = BN_new();
    EC_POINT *public_key = NULL;
    int len = 7;
    /* Create a EC_KEY for secp128r1 */
    EC_KEY *pubKey = EC_KEY_new_by_curve_name(NID_secp128r1);
    const EC_GROUP  *ecgroup = EC_KEY_get0_group(pubKey);
    ECDSA_SIG *signature = ECDSA_SIG_new();
    unsigned char r[16];
    unsigned char s[16];
    char r_dest[33];
    char s_dest[33];
    int loop = 0;

    if (signature == NULL)
    {
        return 1;
    }
    if (pubKey == NULL)
    {
        printf("Creation of PubKey failed \n");
        return 1;
    }
    /* Convert the hex public key x,y co-ordinates to BIGNUM */
    BN_hex2bn(&pk_bignum, publickey_str);
    public_key = EC_POINT_bn2point(ecgroup, pk_bignum, NULL, NULL);
    /* Set the public key point to EC_KEY */
    EC_KEY_set_public_key(pubKey, public_key);

    /* Read the signature from the PICC */
    status = phalMful_ReadSign(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Value is always 00. Present for forward compatibility reasons */
        &aSignature);                   /**< [Out] Pointer to a 32 byte signature read from the card */
    CHECK_SUCCESS(status);

    /* Extract the r and s part of the signature*/
    memcpy(r, aSignature, 16);
    memcpy(s, aSignature+16, 16);

    /* BIGNUM conversion function expects r in ASCII value */
    for(loop = 0;loop < 16; loop++)
    {
        sprintf_s((r_dest+(loop*2)), 3, "%02X", r[loop]);
        sprintf_s((s_dest+(loop*2)), 3, "%02X", s[loop]);
    }

    BN_hex2bn(&signature->r, r_dest);
    BN_hex2bn(&signature->s, s_dest);

    /*Signature verification for the UID sent*/
    if (ECDSA_do_verify(aUid, len, signature, pubKey) == 1)
    {
        printf("Signature verified successfully ... \n");
        return PH_ERR_SUCCESS;
    }
    else
    {
        printf("Signature verification failed... \n");
        return 1;
    }
}

int __cdecl main()
{
    phStatus_t status;
    uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader receive buffer */
    uint8_t aSak[1];                  /**< Select Acknowledge; uint8_t. */
    uint8_t aAtr[256];                /**< Atr */
    uint8_t bMoreCardsAvailable;      /**< more card available */
    uint8_t bLength;                  /**< Length */
    int iChoice;                       /**< choice */
    /**
    * Bfl data parameter storage
    */
    phbalReg_PcscWin_DataParams_t balPcsc;          /**< PCSC (Windows) BAL parameter structure */
    phhalHw_Rd710_DataParams_t halRd710;            /**< Rd710 HAL parameter structure */
    phpalI14443p3a_Sw_DataParams_t palI14443p3a;    /**< PAL-ISO14443P3A parameter structure */
    phpalI14443p4a_Sw_DataParams_t palI14443p4a;    /**< PAL-ISO14443P4A parameter structure */
    phpalI14443p4_Sw_DataParams_t palI14443p4;      /**< ISO14443-4 parameter structure */
    phpalMifare_Sw_DataParams_t palMifare;          /**< PAL-MIFARE parameter structure */
    phalMful_Sw_DataParams_t alMful;                /**< Pointer to MIFARE Ultralight parameter structure. */
    void * pHal = NULL;                             /**< Pointer to the parameter structure of the HAL layer. */
    uint8_t flag = 1;

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /**
    * Init. PCSC BAL
    */
    status = phbalReg_PcscWin_Init(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phbalReg_PcscWin_DataParams_t),  /**< [In] Specifies the size of the data parameter structure. */
        aAtr,                                   /**< [In] Pointer to buffer used for storing the ATR after card activation. */
        256);                                   /**< [In] Size of the ATR buffer. */
    CHECK_SUCCESS(status);

    /**
    * set the BAL Communication Configuration values
    */
    status = phbalReg_SetConfig(&balPcsc,           /**< [In] Pointer to this layer's parameter structure. */
        PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,          /**< BAL Communication Configs */
        PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED);/**< BAL Communication Configuration values */
    CHECK_SUCCESS(status);
    /**
    * set the BAL Communication Configs
    */
    status = phbalReg_SetConfig(&balPcsc,       /**< [In] Pointer to this layer's parameter structure. */
        PHBAL_REG_PCSCWIN_CONFIG_SHARE,         /**< BAL Communication Configs */
        PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);  /**< BAL Communication Configuration values */
    CHECK_SUCCESS(status);

    /**
    * Select Port to be used (no SAM reader)
    */
    status = phbalReg_SetPort(&balPcsc,         /**< [In] Pointer to this layer's parameter structure. */
        (uint8_t*)PCSC_READER_P2_NO_SAM_NAME);  /**< [In] Port Name as String. */
    CHECK_SUCCESS(status);

    /**
    * Initialise the Rd710 HAL component:
    *   RD710 with GLOBAL_BUFFER_SIZE bytes send/receive buffer
    */
    status = phhalHw_Rd710_Init(&halRd710,  /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phhalHw_Rd710_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
        &balPcsc,                           /**< [In] Pointer to the lower layers parameter structure */
        0,                                  /**< [In] Slot number */
        aHalBufferReaderTx,                 /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        sizeof(aHalBufferReaderTx),         /**< [In] Size of the global transmit buffer. */
        aHalBufferReaderRx,                 /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        sizeof(aHalBufferReaderRx));        /**< [In] Size of the global receive buffer. */
    CHECK_SUCCESS(status);

    pHal = &halRd710;  /**< Set HAL pointer */

    /**
    * Open the PCSC port
    */
    status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    /**
    * Standard Rd710 Cmd Init.
    */
    status = (phStatus_t)phhalHw_Rd710_Cmd_InitReader(&halRd710);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    /**
    * init the 14443-3A component
    */
    status = phpalI14443p3a_Sw_Init(&palI14443p3a,  /**< [In] PAL-ISO14443P3A parameter structure */
        sizeof(palI14443p3a),                       /**< [In] Specifies the size of the data parameter structure */
        pHal);                                      /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
    * init the 14443-4A component
    */
    status = phpalI14443p4a_Sw_Init(&palI14443p4a,  /**< [In]  PAL-ISO14443P4A parameter structure */
        sizeof(palI14443p4a),                       /**< [In] Specifies the size of the data parameter structure */
        pHal);                                      /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
    * init the 14443-4 component
    */
    status = phpalI14443p4_Sw_Init(&palI14443p4,    /**< [In] ISO14443-4 parameter structure */
        sizeof(palI14443p4),                        /**< [In] Specifies the size of the data parameter structure */
        pHal);                                      /**< [In] Pointer to the parameter structure of the underlying HAL layer.*/
    CHECK_SUCCESS(status);

    /**
    * init. MIFARE
    */
    status = phpalMifare_Sw_Init(&palMifare,    /**< [In] PAL-MIFARE parameter structure */
        sizeof(palMifare),                      /**< [In] Specifies the size of the data parameter structure */
        pHal,                                   /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
        &palI14443p4);                          /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
    CHECK_SUCCESS(status);

    /**
    * init. Ultralight component
    */
    status = phalMful_Sw_Init(&alMful,      /**< [In] Pointer to MIFARE Ultralight parameter structure. */
        sizeof(phalMful_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
        &palMifare,                         /**< [In] PAL-MIFARE parameter structure */
        NULL,                               /**< [In] Pointer to a key store structure */
        NULL,                               /**< [In] CryptoSym parameter structure for ENC */
        NULL);                              /**< [In] CryptoRng parameter structure for Rng */
    CHECK_SUCCESS(status);

    /* Initialize logging */
#ifdef NXPBUILD__PH_LOG
    /* Register the component for logging */
    status = phLog_Init(phalMful_log, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
    status = phLog_Register(&alMful, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
#endif /* NXPBUILD__PH_LOG */

    /**
    * Field Reset
    */
    status = phhalHw_FieldReset(pHal);  /**< [In] Pointer to the parameter structure of the HAL layer. */
    CHECK_SUCCESS(status);

    /**
    * Configure HAL for Type-A cards
    */
    status = phhalHw_ApplyProtocolSettings(pHal,    /**< [In] Pointer to this layer's parameter structure. */
        PHHAL_HW_CARDTYPE_ISO14443A);               /**< [In] Type of card for which the hal should be configured for. */
    CHECK_SUCCESS(status);

    printf("Please insert a UL_EV1 card for the examples!\n");

    /**
    * Activate Layer 3 card
    */
    status = phpalI14443p3a_ActivateCard(&palI14443p3a, /**< [In] PAL-ISO14443P3A parameter structure */
        NULL,                                           /**< [In] unknown UID */
        0x00,                                           /**< [In] UID length 0 */
        aUid,                                           /**< [Out] complete Uid; uint8_t[4/7/10]. */
        &bLength,                                       /**< [Out] Length of Uid; 4/7/10. */
        aSak,                                           /**< [Out] Select Acknowledge; uint8_t. */
        &bMoreCardsAvailable);                          /**< [Out] Whether there are more cards in the field or not; uint8_t. */
    CHECK_SUCCESS(status);

    printf("Perform the examples: \n");

    printf("---------------------------\n");
    while(flag)
    {
        printf("MIFARE Ultralight_EV1 Example\n");
        printf("[1] Reset Card \n");
        printf("[2] Originality Check Demo \n");
        printf("[x] Exit\n");
        printf("----------------\n");
        printf("Choice:");
        iChoice = _getch();
        printf(" %c", iChoice);
        printf("\n\n");
        printf("---------------------------\n");
        printf("\n\n");

        switch (iChoice)
        {
        case '1':
            printf("Resetting the card\n");
            resetCard(&alMful,   /**< MIFARE Ultralight parameter structure. */
                &palMifare,      /**< PAL-MIFARE parameter structure */
                pHal,            /**< Pointer to the parameter structure of the HAL layer. */
                &palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
            break;
        case '2':
            printf("Originality Check Demo \n");
            OriginalityCheckDemo(&alMful);  /**< MIFARE Ultralight parameter structure. */
            break;
        default:
            flag = 0;
            break;
        }
    }
    return 0;
}