/*
 * Copyright 2017 -2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for MIFARE DESFire Light Application.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * Demonstrates:
 * Two applications on same card:
 *   AFC Application with AES Crypto
 *   NextPACs application with 3KEY DES crypto
 * Card personalization and Check out.
 */

/*******************************************/
/* PLEASE NOTE*/
/*Tested On Firmware Wk1408 Tag */
/*Not all the MIFARE DESFire Light features are in.
/* Tested following:*/
/*case 1: creating an Application ,Authenticating using Ev2SecureMessaging,creating a value file and transactions on value file*/
/*case 2: creation of a  Delegated Application,creating a standard file and checking the Free& Format commands*/
/*case 3: creating an  Application with keysetparameters ,creating a standard data file and  read &write of data into that file*/
/*case 4: creation of  a Delegated Application*/
/*case 5: creation of two applications and accessing files in each application by selecting both applications at a time*/
/*case 6: Updating the records in Linear Record File
/*case 7: Demonstrating Originality Check in Symmetric mode */
/*case 8: Demonstrating Originality Check in ASymmetric mode */
/*case 9: Demonstrating ISO 7816 commands with Extended APDU Lengths(Lc > 1 and Le > 1).*/
/*******************************************/

/* Header for this file */


/* Reader Library Headers */
#include <phbalReg.h>
#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phpalMifare.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include <time.h>
#include <phalMfdfLight.h>
#include "../../intfs/phTMIUtils.h"
#include "Example-RdSim_MfdfLight.h"

#pragma warning(disable:4996)

#define KEYCOUNT 21
#define KEYVERSIONS 1
#define PCSC_READER_P2_NO_SAM_NAME  "NXP Pegoda N CL 0 0"     /**< Pegoda 2: no SAM reader */
#ifdef NXPBUILD__PH_LOG


phLog_RegisterEntry_t logRegisterEntries[16];
phLog_LogEntry_t logEntries[48];
#endif /* NXPBUILD__PH_LOG */

/* Utility function to check return status */
#define CHECK_SUCCESS(x)                        \
	if ((x) != PH_ERR_SUCCESS)                  \
{                                               \
	printf("An error occurred (0x%04X)", (x));	\
	return 0;                                   \
}

static uint8_t gaHalRxReader[512U];
static uint8_t gaHalTxReader[512U];
static uint8_t gbGenBuffer[1024];
static uint8_t pISOFileName[16];
static uint8_t bISOFileLen;
static uint8_t bRunOnMifareApplet;

/* Utility function to print a buffer */
void PRINT_BUFFER(uint8_t * pBuffer, uint16_t wLength)
{
	uint16_t wIndex;

	for (wIndex = 0; wIndex < wLength; ++wIndex)
	{
		printf("%02X ", pBuffer[wIndex]);
	}
	printf("\n");
}

/* Logging function. Turned OFF currently */
#ifdef NXPBUILD__PH_LOG
void phalMfdfLight_log(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
	uint16_t i;
	PHAL_MFDFLIGHT_UNUSED_VARIABLE(bOption);
	PHAL_MFDFLIGHT_UNUSED_VARIABLE(pDataParams);

	for (i = 0; i < wEntryCount; i++ )
	{
		if ( i != 0 )
		{
			printf("%s : ", pEntries[i].pString);
		}
		else
		{
			printf("-----------------------------------------\n");
			printf("%s \n", pEntries[0].pString);
			printf("-----------------------------------------\n");
		}
		PRINT_BUFFER((uint8_t *)(pEntries[i].pData), pEntries[i].wDataLen);
	}
	printf("\n\n");
}
#endif /* NXPBUILD__PH_LOG */

#define AMKEY           0       /* Application Master Key number. */
#define ACKEY           1       /* Application Common Key number. */
#define ARKEY           2       /* Application Read Key number. */
#define AWKEY           3       /* Application Write Key number. */
#define AAKEY           4       /* Application Accreditation Key number. */
#define PICCKEY         5       /* PICC Key entry number in key store. */
#define PICCAESKEY      6       /* New PICC (AES128) Key entry number in key store. */
#define AMKEY_NEW       7       /* New Application Master Key entry number in Key store. */
#define SYSTEM_KEY      8       /* System key entry number in key store. */
#define NPKEY0          9       /* Next PACS App Key 0. */
#define NPKEY1          10      /* Next PACS App Key 1. */
#define SYSTEM_KEY_NP   11      /* Next PACS System key. */
#define DEFAULT_3K3DES  12      /* Default 3K3DES KEY. All zeros. */
#define NPKEY_FOR_UID   13      /* Used to retrieve UID */
#define AFCKEY_FOR_UID  14      /* Used to retrieve UID */
#define PICCDAMAUTHKEY  15      /* PICC DAM Auth Key for DAM */
#define PICCORIGINALITYKEY 16	/* PICC Originality Key */
#define VCCONFIGKEY        17	/* PICC VC Configuration Key */
#define VCPROXIMITYKEY     18	/* PICC VC Proximity Key */
#define VCSELECTMACKEY     19	/* PICC VC Select MAC Key */
#define VCSELECTENCKEY     20	/* PICC VC Select ENC Key */

#define STDDATAFILE1    1       /* File number of Standard data file 1. */
#define STDDATAFILE2    2       /* File number of Standard data file 2. */
#define BCKUPDATAFILE1  6       /* File number od Backup data file.*/
#define BCKUPDATAFILE2  7       /* File number of backup datafile. */
#define VALUEFILE       3       /* File number of Value file. */
#define RECORDFILE      4       /* File number of Record file. */
#define TMACFILE        5       /* File number of Record file. */

#define NPCARDOBJ       1       /* File number of NP Card object. */
#define NPPACSOBJ       2       /* File number of NP PACS object. */
#define TMI_BUFFER_SIZE 255     /* TMI Buffer Size */
#define IID_KEY_COUNT   0x13U   /* number of IID keys */
#define VCA_AUTH_RND_LEN   36   /* Size of the Virtual Card and Proximity Check Random numbers */

/* Default PICC key */
static uint8_t bDefaultPICCKey[16] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

/* AES PICC key */
static uint8_t bAESPICCKey[16] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Application Master key of AFC App */
static uint8_t bAMKey[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Application Master key of AFC App */
static uint8_t bACKey[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static uint8_t bPICCKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static uint8_t bPICCORIGNINALITYKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC VC Configuration Key */
static uint8_t bVCCONFIGKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC VC Proximity Key */
static uint8_t bVCPROXIMITYKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC VC Select MAC Key */
static uint8_t bVCSELECTMACKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC VC Select ENC Key */
static uint8_t bVCSELECTENCKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC DAM Auth Key for DAM */
static uint8_t  bPICCDAMAUTHKEY[16]= {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Application Common Key of AFC App */
static uint8_t bACKEY[16] =     {
	0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
	0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11
};
/* Application Read Key of AFC App */
static uint8_t bARKey[16] =     {
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
};
/* Application Write Key of AFC App */
static uint8_t bAWKey[16] =     {
	0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33,
	0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33
};
/* Application Accreditation Key of AFC App */
static uint8_t bAAKey[16] =     {
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44
};
/* New Application Master key of AFC App */
static uint8_t bAMKey_New[16] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* AFC App Key to retrieve UID */
static uint8_t bAfcKeyN[24]  =   {
	0x1A, 0x2B, 0x3C, 0x4D, 0x5E, 0x6F, 0x70, 0x81,
	0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
	0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98
};

/* System key of AFC App */
static uint8_t bSysKey[16] =    {
	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	0x09, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16
};
/* Next PACs System key */
static uint8_t bDEFAULT_3K3DES[24] =  {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Default 3K 3DES Key */
static uint8_t bSysKeyNP[24] =  {
	0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xCA, 0xCB,
	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	0x09, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16
};
/* Data for AFC App data file 1 */
static uint8_t bData1[32] =     {
	0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8,
	0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8,
	0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8,
	0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8
};
/* Data for AFC App data file 2 */
static uint8_t bData2[28] =     {
	0x4D, 0x55, 0x53, 0x54, 0x45, 0x52, 0x3C, 0x53,
	0x41, 0x4D, 0x50, 0x4C, 0x45, 0x3C, 0x3C, 0x01,
	0x07, 0x80, 0x01, 0x07, 0x09, 0x30, 0x06, 0x10,
	0x00, 0x00, 0x00, 0x00
};
/* NextPac key 0 */
static uint8_t bNPKey0[24]  =   {
	0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF, 0x00, 0x11,
	0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
	0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80
};
/* NextPac key 1 */
static uint8_t bNPKey1[24]  =   {
	0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF, 0xE0, 0xE1,
	0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9,
	0xC0, 0xD0, 0xE0, 0xF0, 0x20, 0x30, 0x40, 0x50
};

/* NP Key to retrieve UID */
static uint8_t bNPKey2[24]  =   {
	0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xC0, 0xA1,
	0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9,
	0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80
};


/* MASTER APPLICATION ID */
static uint8_t bAfcMApp[3] = {0x00, 0x00, 0x00};
/* AFC APPLICATION id */
static uint8_t bAfcApp[3] = {0x2A, 0x8C, 0xF1};
/* AFC APPLICATION2 id */
static uint8_t bAfcApp2[3] = {0x02, 0x02, 0x02};

/* Delegated AFC APPLICATION id */
static uint8_t bAfcDelApp[3] = {0x01, 0x01, 0x01};

static uint8_t bCardUid[10];

static uint8_t bVersionString[28];

/* This 33 bytes string is used for Originality check.
 * NOTE: This should be provided by product team once the key is finalized.
 */
const char *publickey_str = "046F70AC557F5461CE5052C8E4A7838C11C7A236797E8A0730A101837C004039C";
/* TMC & TMV */
uint8_t bTMC[4] = {0x00,0x00,0x00,0x00};
uint8_t bTMV[8] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
uint8_t bTMIBuffer[TMI_BUFFER_SIZE];

phStatus_t StoreKeysInKeyStore(phKeyStore_Sw_DataParams_t * pKeyStore)
{
	phStatus_t status;
	phKeyStore_Sw_DataParams_t keyStore;
	uint8_t bDefKey[24];

	keyStore = *pKeyStore;

	/* Set the default PICC Master key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, PICCKEY, PH_CRYPTOSYM_KEY_TYPE_2K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		PICCKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_2K3DES,
		bPICCKEY,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Master key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AMKEY, PH_CRYPTOSYM_KEY_TYPE_2K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AMKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_2K3DES,
		bAMKey,
		0x00);
	CHECK_SUCCESS(status);

	/* PICC DAM Auth Key for DAM */
	status = phKeyStore_FormatKeyEntry(&keyStore, PICCDAMAUTHKEY, PH_CRYPTOSYM_KEY_TYPE_DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		PICCDAMAUTHKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_DES,
		bPICCDAMAUTHKEY,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Common key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, ACKEY, PH_CRYPTOSYM_KEY_TYPE_DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		ACKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_DES,
		bACKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Read key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, ARKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		ARKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bARKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Write key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AWKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AWKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAWKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Accreditation key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AAKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AAKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAAKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the new PICC AES128 key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, PICCAESKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		PICCAESKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAESPICCKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the new PICC ORIGINALITY CHECK AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, PICCORIGINALITYKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		PICCORIGINALITYKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bPICCORIGNINALITYKEY,
		0x00);
	CHECK_SUCCESS(status);
	/* Set the new PICC VC Configuration Key AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, VCCONFIGKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		VCCONFIGKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bVCCONFIGKEY,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the new PICC VC PROXIMITY KEY AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, VCPROXIMITYKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		VCPROXIMITYKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bVCPROXIMITYKEY,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the new PICC VC Select MAC KEY AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, VCSELECTMACKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		VCSELECTMACKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bVCSELECTMACKEY,
		0x00);
	CHECK_SUCCESS(status);
	/* Set the new PICC VC Select ENC KEY AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, VCSELECTENCKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		VCSELECTENCKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bVCSELECTENCKEY,
		0x00);
	CHECK_SUCCESS(status);
	/* Set the new Application Master key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AMKEY_NEW, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AMKEY_NEW,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAMKey_New,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the SYSTEM key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, SYSTEM_KEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		SYSTEM_KEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bSysKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the UID retriever key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AFCKEY_FOR_UID, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AFCKEY_FOR_UID,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAfcKeyN,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the NPKEY0 in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY0, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		NPKEY0,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bNPKey0,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the NPKEY1 in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY1, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		NPKEY0,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bNPKey1,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the NP system KEY in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, SYSTEM_KEY_NP, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		SYSTEM_KEY_NP,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bSysKeyNP,
		0x00);
	CHECK_SUCCESS(status);

	memset(bDefKey, 0, 24);
	/* Set the default 3K3DES KEY in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, DEFAULT_3K3DES, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		DEFAULT_3K3DES,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bDEFAULT_3K3DES,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the UID retriever KEY for NP App in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY_FOR_UID, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		NPKEY_FOR_UID,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bNPKey2,
		0x00);
	CHECK_SUCCESS(status);

	return 0;
}

phStatus_t ISO_SelectPICCFile(phalMfdfLight_Sw_DataParams_t * pAlMfdfLight)
{
	phStatus_t status;
	uint8_t pFCI[256] = {'\0'};
	uint8_t *ppFCI = pFCI;
	uint16_t 	pwFCILen = 0;
	uint8_t pFid[2] = {0x00,0x00};  // 0xE1,0x10
	uint8_t bExtendedLenApdu = 0x00; /* 0x00 = short apdu, 0x01 = extended apdu */
#if 1
	/* DF Name = 0xD2760000850100*/
	bISOFileLen = 7;
	pISOFileName[0] = 0xD2;
	pISOFileName[1] = 0x76;
	pISOFileName[2] = 0x00;
	pISOFileName[3] = 0x00;
	pISOFileName[4] = 0x85;
	pISOFileName[5] = 0x01;
	pISOFileName[6] = 0x00;
#endif

	status = phalMfdfLight_IsoSelectFile(
                                     pAlMfdfLight,
                                     0x0C,  /* 0x00: Return No FCI */
									 0x04, /* Select by DF name */
                                     pFid,
                                     pISOFileName,
                                     bISOFileLen,
									 bExtendedLenApdu,
									 &ppFCI,
                                     &pwFCILen
                                     );
	CHECK_SUCCESS(status);
	printf("***** IsoSelectFile SUCCESS!!*******\n");
	return status;
}

phStatus_t LRP_Demo(phalMfdfLight_Sw_DataParams_t * pAlMfdfLight )
{
	phStatus_t status;
	uint8_t pFCI[256] = {'\0'};
	uint8_t *ppFCI = pFCI;
	uint16_t pwFCILen = 0;
	uint8_t pFid[2] = {0x01, 0xdf};
	uint8_t bOption, bLength;
	uint8_t bExtendedLenApdu = 0x00; /* 0x00 = short apdu, 0x01 = extended apdu */
	uint8_t bLenPcdCapsIn=6;                                    /**< [In] Length of PcdCapsIn. Always zero for following authentication */
	uint8_t bPcdCapsIn[6] = {0x00,0x00,0x00,0x00,0x00,0x00};    /**< [In] PCD Capabilities. Upto 6 bytes. */
	uint8_t bPcdCapsInLrp[6] = {0x02,0x00,0x00,0x00,0x00,0x00};    /**< [In] PCD Capabilities. Upto 6 bytes. */
	uint8_t bPcdCapsOut[6] ;                                    /**< [Out] PCD Capabilities. 6 bytes. */
	uint8_t bPdCapsOut[6] ;                                     /**< [Out] PD Capabilities. 6 bytes. */

	/* Do ISO Select First */
	status = phalMfdfLight_IsoSelectFile(
                                     pAlMfdfLight,
                                     0x0C, /* Select by DF name */
                                     0x00, /* Return FCI */
                                     pFid,
                                     pISOFileName,
                                     bISOFileLen,
									 bExtendedLenApdu,
									 &ppFCI,
                                     &pwFCILen
                                     );
	CHECK_SUCCESS(status);

	/* EV2 First Authenticate */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		0x01,                                         /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsIn,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);
#if 1
	/* Perform Set Configuration to enable LRP Mode */
	bOption = PHAL_MFDFLIGHT_SET_CONFIG_OPTION5; /* option to set default key and version */
	memset(gbGenBuffer, 0x00, sizeof(gbGenBuffer));             /* Default key */

	/* Prepare data */
	bLength = 0;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x02;  /* Set LRP Mode*/
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;

	status = phalMfdfLight_SetConfiguration(pAlMfdfLight,
		bOption,
		gbGenBuffer,
		bLength);
	CHECK_SUCCESS(status);

	/* Perform LRP Authentication */
	/* EV2 First Authenticate */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	printf("====== LRP First Authenticate is success ======\n");
#endif

	/* Perform LRP Authentication */
	/* EV2 First Authenticate */
#if 0
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHNONFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);
	printf("====== LRP Authenticate Non First is success ======\n");
#endif

	status = phalMfdfLight_GetCardUID(pAlMfdfLight,      /* [In] Pointer to parameters data structure */
						bCardUid);                          /* [Out] 7 Byte Card UID */
	CHECK_SUCCESS(status);

	printf("====== Get Card UID is success ======\n");
	return status;
}

phStatus_t ReferenceTransaction_1(phalMfdfLight_Sw_DataParams_t * pAlMfdfLight )
{
	phStatus_t status;
	uint8_t pFCI[256] = {'\0'};
	uint8_t *ppFCI = pFCI;
	uint16_t pwFCILen = 0;

	uint8_t pFid[2] = {0x01, 0xdf};
	uint8_t bOption, bLength;
	uint8_t bExtendedLenApdu = 0x00; /* 0x00 = short apdu, 0x01 = extended apdu */
	uint8_t bLenPcdCapsIn=6;                                    /**< [In] Length of PcdCapsIn. Always zero for following authentication */
	uint8_t bPcdCapsIn[6] = {0x00,0x00,0x00,0x00,0x00,0x00};    /**< [In] PCD Capabilities. Upto 6 bytes. */
	uint8_t bPcdCapsInLrp[6] = {0x02,0x00,0x00,0x00,0x00,0x00};    /**< [In] PCD Capabilities. Upto 6 bytes. */
	uint8_t bPcdCapsOut[6] ;                                    /**< [Out] PCD Capabilities. 6 bytes. */
	uint8_t bPdCapsOut[6] ;                                     /**< [Out] PD Capabilities. 6 bytes. */
	uint8_t  pOffset[3];
	uint8_t  *pRxData;
	uint16_t pRxDataLen;
	uint8_t pTxDataLen[3];
	uint8_t bData_Write[128];
	uint8_t bVerInfo[30] = {'\0'};
	uint8_t bFileIDs[16] = {'\0'};
	uint8_t bISOFileIDs[8] = {'\0'};
	uint8_t bNumofFileIds = 0;
	uint8_t bFileSettingsBuffer[20] = {'\0'};
	uint8_t bFileSettingsBufLen = 0;
	uint8_t bValue[4] = {'\0'};
	uint8_t bCreditValue[4] = {100, 0, 0, 0};
	uint8_t pTMRI[16] = {0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33};
	uint8_t pOutTMRI[16] = {0x00};

	status = ISO_SelectPICCFile(pAlMfdfLight);
	CHECK_SUCCESS(status);

	status = phalMfdfLight_IsoSelectFile(
                                     pAlMfdfLight,
                                     0x00, /* Select by DF name */
                                     0x00, /* Return FCI */
                                     pFid,
                                     pISOFileName,
                                     bISOFileLen,
									 bExtendedLenApdu,
									 &ppFCI,
                                     &pwFCILen
                                     );
	CHECK_SUCCESS(status);
	printf("====== ISOSelectFile is success ======\n\n\n");

	/* EV2 First Authenticate with App Key No 0*/
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		0x01,                                         /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsIn,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	/* Perform Set Configuration to enable LRP Mode */
	bOption = PHAL_MFDFLIGHT_SET_CONFIG_OPTION5; /* option to set default key and version */
	memset(gbGenBuffer, 0x00, sizeof(gbGenBuffer));             /* Default key */

	/* Prepare data */
	bLength = 0;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x02;  /* Set LRP Mode*/
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;

	status = phalMfdfLight_SetConfiguration(pAlMfdfLight,
		bOption,
		gbGenBuffer,
		bLength);
	CHECK_SUCCESS(status);

	/* Perform LRP Authentication */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x01,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	printf("====== LRP First Authenticate is success ======\n\n");
	status = phalMfdfLight_GetCardUID(pAlMfdfLight,      /* [In] Pointer to parameters data structure */
						bCardUid);                         /* [Out] 7 Byte Card UID */
	CHECK_SUCCESS(status);
	printf("====== Get Card UID is success ======\n\n");


	status = phalMfdfLight_GetVersion(pAlMfdfLight,      /* [In] Pointer to parameters data structure */
						bVerInfo);                         /* [Out] 28/30 Byte version string */
	CHECK_SUCCESS(status);
	printf("====== Get Version is success ======\n\n");

	status = phalMfdfLight_GetFileIDs(pAlMfdfLight,      /* [In] Pointer to parameters data structure */
						bFileIDs,					/* [out] List of 4 File IDs */
						&bNumofFileIds);		     /* [out] Num of file IDs */
	CHECK_SUCCESS(status);
	printf("====== Get File IDs  is success ======\n\n");

	status = phalMfdfLight_GetISOFileIDs(pAlMfdfLight,      /* [In] Pointer to parameters data structure */
						bISOFileIDs,					/* [Out] List of 4 ISO File IDs */
						&bNumofFileIds);				 /*[Out] Num of file IDs */
	CHECK_SUCCESS(status);
	printf("====== Get ISO File IDs  is success ======\n\n");

	/* Retreive File settings for all Files */
	for(bOption = 0; bOption < 4; bOption ++)
	{
		status =  phalMfdfLight_GetFileSettings(pAlMfdfLight, bFileIDs[bOption], bFileSettingsBuffer, &bFileSettingsBufLen);
		CHECK_SUCCESS(status);
		printf("====== Get file settings for file id [%x]  is success ======\n",bFileIDs[bOption]);
	}

#if 1
	/* Get Current Value */
	status = phalMfdfLight_GetValue(pAlMfdfLight,         /* [In] Pointer to this layers param structure. */
		PHAL_MFDFLIGHT_COMMUNICATION_PLAIN,             /* [In] Communication option. */
		VALUEFILE,                                    /* [In] 1 byte file number. */
		bValue);                                      /* [Out] 4 Byte array to store the value read out. LSB First. */
	CHECK_SUCCESS(status);

	printf(" The amount in your account  Before Credit is %x %x %x %x \n", bValue[3], bValue[2], bValue[1], bValue[0]);

		status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x03,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	/* Add money (100) to the account */
	status = phalMfdfLight_Credit(pAlMfdfLight,     /* [In] Pointer to this layers param structure. */
		PHAL_MFDFLIGHT_COMMUNICATION_ENC,         /* [In] Communication option. */
		VALUEFILE,                              /* [In] 1 byte file number. */
		bCreditValue);                                /* [In] 4 byte value array. LSB first. */
	CHECK_SUCCESS(status);

		/* As per the CRS, Auth with Key 0x01 before CommitreaderID. Perform LRP Authentication Again */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x01,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	/* Commit Reader ID */
	status = phalMfdfLight_CommitReaderID(pAlMfdfLight, pTMRI, pTMRI);
	CHECK_SUCCESS(status);
	printf("====== CommitReaderID  is success ======\n\n");

	/* Commit after credit*/
	status = phalMfdfLight_CommitTransaction( pAlMfdfLight,  /* [In] Pointer to this layers param structure. */
		0x01,                                            /* [In] Set to 1 to get TMC & TMV */
		bTMC,                                            /* [Out] 4 byte TMAC counter. */
		bTMV);                                           /* [Out] 8 byte TMAC value. */
	CHECK_SUCCESS(status);

	/* Performing Get Value After Credit */
	status = phalMfdfLight_GetValue(pAlMfdfLight,    /* [In] Pointer to this layers param structure. */
		PHAL_MFDFLIGHT_COMMUNICATION_PLAIN,        /* [In] Communication option. */
		VALUEFILE,                               /* [In] 1 byte file number. */
		bValue);                                 /* [Out] 4 Byte array to store the value read out. LSB First. */
	CHECK_SUCCESS(status);

	printf(" The amount in your account  After credit is %x %x %x %x \n", bValue[3], bValue[2], bValue[1], bValue[0]);
#endif

	/* Perform LRP Authentication Again */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x03,                                         /* [In] Key number on Card: Write Key for 0x00 */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	printf("====== LRP First Authenticate is success ======\n\n");

	/* Test Non First Authenticate as well once */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHNONFIRST_LRP,                /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x03,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);
	printf("====== LRP Authenticate Non First is success ======\n");

	/* Set Short Length APDU as Prime supports only short length */
	status= phalMfdfLight_SetConfig(pAlMfdfLight,
					PHAL_MFDFLIGHT_SHORT_LENGTH_APDU,
						0x01);
	CHECK_SUCCESS(status);

	/* Set the write data to 16 Bytes 0xFF*/
	memset(bData_Write, 0x33, 128);

	/* Writing  The data */
	pTxDataLen[0]=128;
	pTxDataLen[1]=0x00;
	pTxDataLen[2]=0x00;

	/* Offset address */
	pOffset[0]=0x00;
	pOffset[1]=0x00;
	pOffset[2]=0x00;

	status= phalMfdfLight_WriteData(pAlMfdfLight,   /* [In] Pointer to this layers param structure. */
		//PHAL_MFDFLIGHT_COMMUNICATION_PLAIN,         /* [In] Is either plain or encrypted or MAC'd. */
		PHAL_MFDFLIGHT_COMMUNICATION_ENC,
		0x01,										/* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		0x00,										/* [In] 1 byte file number. 7th bit indicates SAI. */
		pOffset,									/* [In] 3 bytes offset. LSB First. */
		bData_Write,                                /* [Out] Pointer to HAL Tx buffer For writing data. */
		pTxDataLen									/* [Out] Pointer to Length of RxData. */
		);
	CHECK_SUCCESS(status);
	printf("====== WriteData  is success ======\n\n");

	/* Reading The data back */
	/* Data Length -LSB First */
	pTxDataLen[0]=128;
	pTxDataLen[1]=0x00;
	pTxDataLen[2]=0x00;

	/* Offset address */
	pOffset[0]=0x00;
	pOffset[1]=0x00;
	pOffset[2]=0x00;

	/* Perform LRP Authentication Again */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x01,                                         /* [In] Key number on Card: Read Key for 0x00 */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	/* Do the Read For second time. This would test EnCCtr maintained in the crypto structure */
	status= phalMfdfLight_ReadData(pAlMfdfLight,      /* [In] Pointer to this layers param structure. */
		//PHAL_MFDFLIGHT_COMMUNICATION_PLAIN,           /* [In] Is either plain or encrypted or MAC'd. */
		PHAL_MFDFLIGHT_COMMUNICATION_ENC,
		0x01,                                     /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		0x00,								/* [In] 1 byte file number. 7th bit indicates SAI. */
		pOffset,                                  /* [In] 3 bytes offset. LSB First. */
		pTxDataLen,                               /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
		&pRxData,                                 /* [Out] Pointer to HAL Rx buffer returned back to user. */
		&pRxDataLen                               /* [Out] Pointer to Length of RxData. */
		);

	CHECK_SUCCESS(status);
	printf("====== ReadData  is success ======\n\n");

#if 1
	/* As per the CRS, Auth with Key 0x01 before CommitreaderID. Perform LRP Authentication Again */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x01,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	/* Commit Reader ID */
	status = phalMfdfLight_CommitReaderID(pAlMfdfLight, pTMRI, pTMRI);
	CHECK_SUCCESS(status);
	printf("====== CommitReaderID  is success ======\n\n");
#endif

	/* Decrypt Reader ID */
	status = phalMfdfLight_DecryptReaderID(
                                       pAlMfdfLight,      /**< [In] Pointer to this layers parameter structure. */
                                       PHAL_MFDFLIGHT_NO_DIVERSIFICATION,      /**< [In] Diversification option. */
                                       PICCAESKEY,  /**< [In] Key number in key store of DAM MAC key. */
                                       0x00, /**< [In] Key version in key store of DAM MAC key. */
									   0,
									   0,
                                       NULL,     /**< [In] Diversification input for TMACKey. */
                                       0,    /**< [In] Length of diversification input. */
                                       bTMC,           /**< [In] 4 bytes Transaction MAC Counter. */
                                       bCardUid,          /**< [In] 7 (or 10) byte UID of the card. */
                                       0x07,         /**< [In] Length of UID supplied. */
                                       pTMRI,      /**< [In] Encrypted TMAC Reader Id (16 bytes). */
                                       pOutTMRI     /**< [Out] Reader ID of the last successful transaction. */
                                       );
	CHECK_SUCCESS(status);

	/* Commit the write data */
	status =   phalMfdfLight_CommitTransaction(
				pAlMfdfLight,
				0x01,
				bTMC,
				bTMV
			);
	CHECK_SUCCESS(status);
	printf("====== CommitTransaction  is success ======\n\n");

	/* Reset Authentication */
	status =   phalMfdfLight_ResetAuthentication(pAlMfdfLight);
	CHECK_SUCCESS(status);

	return status;
}

phStatus_t ReferenceTransaction_2(phalMfdfLight_Sw_DataParams_t * pAlMfdfLight )
{
	phStatus_t status;
	uint8_t pFCI[256] = {'\0'};
	uint8_t *ppFCI = pFCI;
	uint16_t pwFCILen = 0;
	uint8_t pFid[2] = {0x01, 0xdf};
	uint8_t bOption, bLength;
	uint8_t bExtendedLenApdu = 0x00; /* 0x00 = short apdu, 0x01 = extended apdu */
	uint8_t bLenPcdCapsIn=6;                                    /**< [In] Length of PcdCapsIn. Always zero for following authentication */
	uint8_t bPcdCapsIn[6] = {0x00,0x00,0x00,0x00,0x00,0x00};    /**< [In] PCD Capabilities. Upto 6 bytes. */
	uint8_t bPcdCapsInLrp[6] = {0x02,0x00,0x00,0x00,0x00,0x00};    /**< [In] PCD Capabilities. Upto 6 bytes. */
	uint8_t bPcdCapsOut[6] ;                                    /**< [Out] PCD Capabilities. 6 bytes. */
	uint8_t bPdCapsOut[6] ;                                     /**< [Out] PD Capabilities. 6 bytes. */

	uint8_t  pOffset[3];
	uint8_t  *pRxData;
	uint16_t pRxDataLen;
	uint8_t pTxDataLen[3];
	uint8_t bData_Write[128];
	uint8_t RecNo[3]= {0x00,0x00,0x00};
	uint8_t RecCount[3] = {0x01,0x00,0x00};
	uint8_t pRecordSize[3];
	uint8_t pTMRI[16] = {0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33};

	status = ISO_SelectPICCFile(pAlMfdfLight);
	CHECK_SUCCESS(status);

	/* Do ISO Select First */
	status = phalMfdfLight_IsoSelectFile(
                                     pAlMfdfLight,
                                     0x0C, /* Select by DF name */
                                     0x00, /* Return FCI */
                                     pFid,
                                     pISOFileName,
                                     bISOFileLen,
									 bExtendedLenApdu,
									 &ppFCI,
                                     &pwFCILen
                                     );
	CHECK_SUCCESS(status);
	printf("====== ISOSelect File  is success ======\n\n");

	/* EV2 First Authenticate */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_NON_LRP,               /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsIn,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	/* Perform Set Configuration to enable LRP Mode */
	bOption = PHAL_MFDFLIGHT_SET_CONFIG_OPTION5; /* option to set default key and version */
	memset(gbGenBuffer, 0x00, sizeof(gbGenBuffer));             /* Default key */

	/* Prepare data */
	bLength = 0;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] = 0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x02;  /* Set LRP Mode*/
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;
	gbGenBuffer[bLength++] =0x00;

	status = phalMfdfLight_SetConfiguration(pAlMfdfLight,
		bOption,
		gbGenBuffer,
		bLength);
	CHECK_SUCCESS(status);

	/* Auth with LRP First */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	printf("====== LRP First Authenticate is success ======\n");

	status = phalMfdfLight_GetCardUID(pAlMfdfLight,      /* [In] Pointer to parameters data structure */
						bCardUid);                         /* [Out] 7 Byte Card UID */
	CHECK_SUCCESS(status);
	printf("====== Get Card UID is success ======\n\n");

	/* Change Key testing */
	status = phalMfdfLight_ChangeKey(pAlMfdfLight,      /* [In] Pointer to this layer's parameter structure. */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,            /* [In] Diversification option. */
		PICCAESKEY,                                    /* [In] Old key number in keystore. */
		0,                                          /* [In] Old key version in keystore. */
		AWKEY,                                    /* [In] New key number in keystore. */
		0,                                          /* [In] New key version in keystore. */
		0x00,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
		NULL,                                       /* [In] Diversification input. Can be NULL. */
		0);                                         /* [In] Length of diversification input max 31B. */
	CHECK_SUCCESS(status);

	/* Authenticate with Changed Key */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,               /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		AWKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	/* Revert back the old key */
	status = phalMfdfLight_ChangeKey(pAlMfdfLight,      /* [In] Pointer to this layer's parameter structure. */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,            /* [In] Diversification option. */
		AWKEY,                                    /* [In] Old key number in keystore. */
		0,                                          /* [In] Old key version in keystore. */
		PICCAESKEY,                                    /* [In] New key number in keystore. */
		0,                                          /* [In] New key version in keystore. */
		0x00,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
		NULL,                                       /* [In] Diversification input. Can be NULL. */
		0);                                         /* [In] Length of diversification input max 31B. */
	CHECK_SUCCESS(status);


	/* Perform LRP Authentication with Key No 0x02(RW Access)*/
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x02,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	status= phalMfdfLight_SetConfig(pAlMfdfLight,
					PHAL_MFDFLIGHT_SHORT_LENGTH_APDU,
						0x01);

	/* Writing The record data */
	/* Data Length -LSB First */
	pTxDataLen[0]=0x10;
	pTxDataLen[1]=0x00;
	pTxDataLen[2]=0x00;

	/* Offset address */
	pOffset[0]=0x00;
	pOffset[1]=0x00;
	pOffset[2]=0x00;

	memset(bData_Write, 0x33, 16);
	/* Perform Write Record in Full Enc Mode */
	status =  phalMfdfLight_WriteRecord(
                                      pAlMfdfLight,
                                      PHAL_MFDFLIGHT_COMMUNICATION_ENC,
                                      0x01, /* 1= 14444 chaining */
                                      0x01, /* File No */
                                      pOffset,
                                      bData_Write,
                                      pTxDataLen
                                      );
	CHECK_SUCCESS(status);
	printf("======WriteRecord  is success ======\n\n");

	/* As per the CRS, Auth with Key 0x01 before CommitreaderID. Perform LRP Authentication Again */
	status = phalMfdfLight_AuthenticateEv2(pAlMfdfLight,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFLIGHT_AUTHFIRST_LRP,                   /* first authentication */
		PHAL_MFDFLIGHT_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x01,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsInLrp,                                /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);

	/* Commit Reader ID */
	status = phalMfdfLight_CommitReaderID(pAlMfdfLight, pTMRI, pTMRI);
	CHECK_SUCCESS(status);
	printf("====== CommitReaderID  is success ======\n\n");

	/* Commit the write data */
	status =   phalMfdfLight_CommitTransaction(
				pAlMfdfLight,
				0x01,
				bTMC,
				bTMV
			);
	CHECK_SUCCESS(status);
	printf("====== CommitTransaction is success ======\n\n");

	/* Record size */
	pRecordSize[0] = 0x10;
	pRecordSize[1] = 0x00;
	pRecordSize[2] = 0x00;

	/* Offset address */
	pOffset[0]=0x00;
	pOffset[1]=0x00;
	pOffset[2]=0x00;

	status= phalMfdfLight_SetConfig(pAlMfdfLight,
					PHAL_MFDFLIGHT_SHORT_LENGTH_APDU,
						0x01);

	CHECK_SUCCESS(status);
	status =   phalMfdfLight_ReadRecords(
                                   pAlMfdfLight,
                                   PHAL_MFDFLIGHT_COMMUNICATION_ENC,
                                   0x01, /* 1= 14444 chaining */
                                   0x01, /* File No */
                                   RecNo,
                                   RecCount,
                                   pRecordSize,
                                   &pRxData,                                 /* [Out] Pointer to HAL Rx buffer returned back to user. */
								   &pRxDataLen                               /* [Out] Pointer to Length of RxData. */
                                   );
	CHECK_SUCCESS(status);
	printf("====== ReadRecord  is success ======\n\n");

	return status;
}


int __cdecl main(int argc, char *argv[])
{
	phStatus_t status;

	uint8_t bCidEnable, bCid, bNadSupported, bFwi, bFsdi, bFsci;
	uint8_t bSak[1];
	uint8_t bUid[10];
	uint8_t pSeed[8];
	uint8_t bMoreCardsAvaliable;
	uint8_t bLength;

	int iChoice,choice;
	phKeyStore_Sw_KeyEntry_t keyEntry[KEYCOUNT];
	phKeyStore_Sw_KeyVersionPair_t keyVersion[KEYCOUNT * KEYVERSIONS];
	phKeyStore_Sw_KUCEntry_t keyUsage[KEYCOUNT];

	phpalI14443p3a_Sw_DataParams_t I14443p3a;
	phpalI14443p4a_Sw_DataParams_t I14443p4a;
	phpalI14443p4_Sw_DataParams_t I14443p4;
	phpalMifare_Sw_DataParams_t palMifare;
	phCryptoSym_Sw_DataParams_t cryptoEnc;
	phCryptoSym_Sw_DataParams_t cryptoSymRnd;
	phCryptoSym_Sw_DataParams_t cryptoMac;
	phCryptoRng_Sw_DataParams_t cryptoRng;
	phKeyStore_Sw_DataParams_t keyStore;
	phalMfdfLight_Sw_DataParams_t alMfdfLight;
	phTMIUtils_t TMIDataParams;
	void * pHal= NULL;
	void * pKeyStore;                                 /**< Pointer to a KeyStore parameter structure */
	uint8_t pAts[255];
	phbalReg_SocketWin_DataParams_t balSocket;
	phhalHw_RdCardSim_DataParams_t  halRdSim;
	uint8_t pPortName[] = {'1','2','7','.','0','.','0','.','1','\0','1','5','2','0','0','\0','0','0','0','1','\0'};
	uint8_t TxBufferSocket[0x800];
	uint8_t RxBufferSocket[0x200];
	uint8_t TxBufferCardSim[0x800];
	uint8_t RxBufferCardSim[0x800];
	uint8_t IntBufferCardSim[0xff00];

	PHAL_MFDFLIGHT_UNUSED_VARIABLE(argc);
	PHAL_MFDFLIGHT_UNUSED_ARRAY(argv);
	printf("\nNxpRdLib ANSI-C Example MIFARE DesFire EV2 Program\n\n");
	printf("Performing startup...\n\n");
	memset(&bTMIBuffer[0],0x00,TMI_BUFFER_SIZE);

	/* init. keystore */
	status = phKeyStore_Sw_Init(&keyStore,
		sizeof(phKeyStore_Sw_DataParams_t),
		keyEntry,
		KEYCOUNT,
		keyVersion,
		KEYVERSIONS,
		keyUsage,
		KEYCOUNT);
	CHECK_SUCCESS(status);
	pKeyStore = &keyStore;
	/* init. crypto */
	status = phCryptoSym_Sw_Init(&cryptoEnc, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
	CHECK_SUCCESS(status);
	status = phCryptoSym_Sw_Init(&cryptoMac, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
	CHECK_SUCCESS(status);
	status = phCryptoSym_Sw_Init(&cryptoSymRnd, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
	CHECK_SUCCESS(status);
	status = phCryptoRng_Sw_Init(&cryptoRng, sizeof(phCryptoRng_Sw_DataParams_t), &cryptoSymRnd);
	CHECK_SUCCESS(status);
	memset(pSeed, 0x00, 0x08);
	status = phCryptoRng_Seed(&cryptoRng, pSeed, 8);
	CHECK_SUCCESS(status);

	printf("Press 1 To run with  Reader = RdCardSim, Card = C# Simulator.\n\n");
	printf("Press 2 To run with CLRd70x.\n\n");
	printf("Press 3 To run with  Reader CM-1 or Rd710.\n\n");
	printf("Press x To Quit.\n\n");
	choice=_getch();
	switch(choice)
	{
	 case('2') :
		break;

	case('1'):
		printf("Please ensure that Simulator is open and is in waiting state.\n\n");
		printf("Simulator version 5311 must be used!\n\n");

		/* Initialise the Reader BAL component */
		status =  phbalReg_SocketWin_Init(&balSocket, sizeof(phbalReg_SocketWin_DataParams_t), 0, sizeof(TxBufferSocket), TxBufferSocket,sizeof(RxBufferSocket), RxBufferSocket);
		CHECK_SUCCESS(status);

		status = phbalReg_SetPort(&balSocket, pPortName);
		CHECK_SUCCESS(status);
		/* init the Reader HAL component */
		status =   phhalHw_RdCardSim_Init(&halRdSim,
			sizeof(halRdSim),
			&balSocket,
			&keyStore,
			TxBufferCardSim,
			sizeof(TxBufferCardSim),
			RxBufferCardSim,
			sizeof(RxBufferCardSim),
			IntBufferCardSim,
			sizeof(IntBufferCardSim));

		pHal = &halRdSim;

		/* open the reader port */
		status = phbalReg_OpenPort(&balSocket);
		CHECK_SUCCESS(status);
		break;

	case('3'):
		printf("\nNxpRdLib ANSI-C Rd710 CM1 reader Example Program \n\n");
		printf("Please ensure that a Pegoda-2 CM1 reader is connected and in working condition.\n\n");
		printf("Performing startup...\n\n");
		printf("-------------------------\n");

		/* Init. PCSC BAL */
		break;

	default:
		printf(" Quitting ... \n");
		exit(0);

	}

	/* init the 14443-3A component */
	status = phpalI14443p3a_Sw_Init(&I14443p3a, sizeof(phpalI14443p3a_Sw_DataParams_t), pHal);
	CHECK_SUCCESS(status);

	/* init the 14443-4A component */
	status = phpalI14443p4a_Sw_Init(&I14443p4a, sizeof(phpalI14443p4a_Sw_DataParams_t), pHal);
	CHECK_SUCCESS(status);

	/* init the 14443-4 component */
	status = phpalI14443p4_Sw_Init(&I14443p4, sizeof(phpalI14443p4_Sw_DataParams_t), pHal);
	CHECK_SUCCESS(status);

	/* init. mifare pal */
	status = phpalMifare_Sw_Init(&palMifare, sizeof(phpalMifare_Sw_DataParams_t), pHal, &I14443p4);
	CHECK_SUCCESS(status);

	status = phTMIUtils_Init(&TMIDataParams,&bTMIBuffer[0],TMI_BUFFER_SIZE);
	CHECK_SUCCESS(status);


	/* Initialize logging */
#ifdef NXPBUILD__PH_LOG
	/* Register the component for logging */
	status = phLog_Init(phalMfdfLight_log, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
	/*status = phLog_Register(&cryptoRng, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
	CHECK_SUCCESS(status);
	status = phLog_Register(&cryptoRng, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
	CHECK_SUCCESS(status);*/
	status = phLog_Register(&alMfdfLight, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
	CHECK_SUCCESS(status);
#endif /* NXPBUILD__PH_LOG */

	/* ####################Card Initialization########################### */
	do
	{
		/* Switch on the field */
		status = phhalHw_FieldOn(pHal);
		CHECK_SUCCESS(status);

		/* Configure HAL for Type-A cards */
		status = phhalHw_ApplyProtocolSettings(pHal, PHHAL_HW_CARDTYPE_ISO14443A);
		CHECK_SUCCESS(status);

		/* Activate Layer 3 card. In loop till a card is detected. */
		do
		{
			status = phpalI14443p3a_ActivateCard(&I14443p3a, NULL, 0x00, bUid, &bLength, bSak, &bMoreCardsAvaliable);

		} while (status != PH_ERR_SUCCESS);


		/* switch on time mesaurement */
		/* status = phhalHw_SetConfig(&halRdSim, PHHAL_HW_CONFIG_TIMING_MODE, PHHAL_HW_TIMING_MODE_COMM);
		CHECK_SUCCESS(status);*/

		/* Send RATS */
		status = phpalI14443p4a_Rats(&I14443p4a, 0x08, 0x01, pAts);
		CHECK_SUCCESS(status);

		/* Obtain the protocol parameters from layer 4a and then store them in layer 4 */
		/*
		bCidEnable indicates if the card supports Cid. if 0, then no cid is supported.
		bNadSupported indicates if card supports Node address. if 0, NAD is not supported.
		bFwi => Framw waiting integer. can be zero to maximum value can be 0 TO 15.
		bFsdi -> Frame size indicator of pcd 0 to 8
		bFsci -> Frame size indicator of Card 0 - 8
		*/
		status = phpalI14443p4a_GetProtocolParams(&I14443p4a, &bCidEnable, &bCid, &bNadSupported, &bFwi, &bFsdi, &bFsci);
		CHECK_SUCCESS(status);

		status = phpalI14443p4_SetProtocol(&I14443p4, bCidEnable, bCid, bNadSupported, 0, bFwi, bFsdi, bFsci);
		CHECK_SUCCESS(status);
		/* ####################End Card Initialization########################### */

		/* Initialize the MF DesFire EV1 component */
		status = phalMfdfLight_Sw_Init(&alMfdfLight,    /* [In] Pointer to this layer's parameter structure. */
			sizeof(phalMfdfLight_Sw_DataParams_t),    /* [In] Specifies the size of the data parameter structure */
			&palMifare,                             /* [In] Pointer to a palMifare component context. */
			&keyStore,                              /* [In] Pointer to Key Store data parameters. */
			&cryptoEnc,                             /* [In] Pointer to a Crypto component context for encryption. */
			&cryptoMac,                             /* [In] Pointer to a CryptoMAC context */
			&cryptoRng,                             /* [In] Pointer to a CryptoRng component context. */
			&TMIDataParams,                         /* [In] Pointer to TMIUtils dataparams */
			pHal);                                  /* [In] Pointer to the HAL parameters structure. */
		CHECK_SUCCESS(status);

		/* Set all the keys in the S/W key store */
		status = StoreKeysInKeyStore(&keyStore);
		CHECK_SUCCESS(status);

		/* Clear the console. */
		system ( "cls" );
		printf("A Ticket is discovered \n");
		printf("Press 1 for LRP Demo\n");
		printf("Press 2 for Reference transaction 1 demo(ReadData)\n");
		printf("Press 3 for Reference transaction 2 demo(ReadRecords)\n");
		printf("Press x to quit \n");
		iChoice = _getch();

		/* To See The Complete Log ,right Click on Console Window,Go to Properties->Layout->Screen Buffer Size->999 */
		switch(iChoice)
		{
		case '1':
			/* Authenticate with the PICC Master key */
			status = LRP_Demo(&alMfdfLight);   /* [In] Pointer to parameters data structure */
			CHECK_SUCCESS(status);
			break;
		case '2':
			/* Reference transaction 1  */
			status = ReferenceTransaction_1(&alMfdfLight);   /* [In] Pointer to parameters data structure */
			CHECK_SUCCESS(status);
			break;

		case '3':
			/* Reference transaction 2 */
			status = ReferenceTransaction_2(&alMfdfLight);   /* [In] Pointer to parameters data structure */
			CHECK_SUCCESS(status);
			break;

		case 'x':
		case 'X':
			exit ( 0 );

		default:
			printf ( "\n\n" );
			printf ( "Invalid option selected.\n" );
			break;
		}

		printf("\n");
		printf ( "Press any key to continue with the demonstration.\n" );
		printf ( "Press x to exit the demonstration.\n" );
		iChoice = _getch ();
	}while( ( iChoice != 'x' ) && ( iChoice != 'X' ) );

	/* Returns zero if the procedure was successful. */
	return 0;
}