/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Tools Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHTOOLS_ATRPARSER_H
#define PHTOOLS_ATRPARSER_H

#include <ph_Status.h>
#include <phhalHwContact.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

/** \defgroup phToolsAtrParser
*
* \brief This component implements functionality to parse contactbased ATRs
* @{
*/

/**
* \name AtrParser defines
*/
/*@{*/
#define PH_TOOLS_ATRPARSER_TS_OFFSET                        0x00U       /**< Offset of TS in ATR Buffer */
#define PH_TOOLS_ATRPARSER_T0_OFFSET                        0x01U       /**< Offset of T0 in ATR Buffer */
#define PH_TOOLS_ATRPARSER_MAX_NUMBER_OF_PROTOCOLS          0x0EU       /**< Maximum Number of Protocols */
#define PH_TOOLS_ATRPARSER_DEFAULT_F_VALUE                  372         /**< Fd which is the Default F Value needed for BWT calc */
/*@}*/

/**
* \brief Global Atr Parameters
*/
typedef struct
{
    uint8_t bTS;                    /**< First Bytes of ATR encodes convention 3F or 3B */
    uint8_t bFdValue;               /**< TA1 Fi and Di and TA2 Bit 5 encodes which values are used */
    uint8_t bGTetu;                 /**< TC1 GuardTime for PPS only */
    uint8_t bSupportedVCCs;         /**< First TA for T=15 Bits 1-6 encode supported VCC Classes */
    uint8_t bSupportedProtocols[PH_TOOLS_ATRPARSER_MAX_NUMBER_OF_PROTOCOLS];
} phToolsAtrParser_GlobalParameters;

/**
* \brief T=0 Atr Parameters
*/
typedef struct
{
    uint8_t bGTetu;                 /**< TC1 GuardTime for T=0 */
    uint8_t bWI;                    /**< TC2 encodes waiting time integer */
} phToolsAtrParser_T0Parameters;

/**
* \brief T=1 Atr Parameters
*/
typedef struct
{
    uint8_t bCGTetu;                /**< TC1 CharacterGuardTime for T=1 (CGT=GT) */
    uint8_t bIFSC;                  /**< First TA for T=1 - max size blocks that can be received by the card */
    uint8_t bCWI;                   /**< First TB for T=1 Bit 1-4 - character waittime integer */
    uint8_t bBWI;                   /**< First TB for T=1 Bit 5-8 - block waittime integer */
    uint8_t bRedundancyCode;        /**< First TC for T=1 Bit 1 - 1 equals CRC & 0 equals LRC */
} phToolsAtrParser_T1Parameters;

/**
* \brief Set internal ATR Buffer
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_SetInternalAtr(
                                           uint8_t * pAtrBuffer,
                                           uint32_t  wAtrSize
                                           );

/**
* \brief Check if internal ATR is valid (value ranges, lenghts)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_CheckInternalAtr();

/**
* \brief Parses Global Parameters of internal ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_ParseGlobalParameters();

/**
* \brief Parses T=0 Parameters of internal ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_ParseT0Parameters();

/**
* \brief Parses T=1 Parameters of internal ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_ParseT1Parameters();

/**
* \brief Returns first TD Byte with specified ProtocolType
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetProtocolTypeOffset(
                                                  uint8_t bProtocolType,
                                                  uint8_t * pOffsetTD
                                                  );

/**
* \brief Parse Protocol dependent GuardTime in ETU
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_ParseGT(
                                    uint8_t bProtocolType,
                                    uint8_t * pGuardTimeEtu
                                    );
/**
* \brief Initialise internal structs with default values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_SetDefaultValues();

/**
* \brief Get Character Guard Time for T=0 of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT0_GTclk(
                                        uint32_t * pGTclk,
                                        uint32_t bFdIndexValue
                                        );

/**
* \brief Get Waittime Integer of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT0_WI(
                                     uint8_t * pWI
                                     );

/**
* \brief Get Waiting Time according to formula (ISO7816)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT0_WTclk(
                                        uint32_t * pWTclk,
                                        uint32_t bFdIndexValue
                                        );

/**
* \brief Get Character Guard for T=1 Time of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT1_CGTclk(
                                         uint32_t * pCGTetu,
                                         uint32_t bFdIndexValue
                                         );

/**
* \brief Get IFSC of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT1_IFSC(
                                       uint8_t * pIFSC
                                       );

/**
* \brief Get Character Waittime Integer of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT1_CWI(
                                      uint8_t * pCWI
                                      );

/**
* \brief Get Block Waittime Integer of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT1_BWI(
                                      uint8_t * pBWI
                                      );

/**
* \brief Get Redundancy Mode of current ATR (LRC or CRC)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT1_RedundancyMode(
                                                 uint8_t * pRedundancyMode
                                                 );

/**
* \brief Get CWT according to formula (ISO7816)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT1_CWTclk(
                                         uint32_t * pCWTclk,
                                         uint32_t bFdIndexValue
                                         );

/**
* \brief Get BWT according to formula (ISO7816)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetT1_BWTclk(
                                         uint32_t * pBWTclk,
                                         uint32_t bFdIndexValue
                                         );

/**
* \brief Get TS byte of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetGlobal_TS(
                                         uint8_t * pTS
                                         );

/**
* \brief Get FD Values current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetGlobal_FD(
                                         uint8_t * pFD
                                         );

/**
* \brief Get Guard Time byte of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetGlobal_GTclk(
                                            uint32_t * pGTclk,
                                            uint32_t bFdIndexValue
                                            );

/**
* \brief Get Byte which specifies supported VCCs of current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetGlobal_SupportedVCCs(
                                                    uint8_t * pSupportedVCCs
                                                    );

/**
* \brief Checks if given ProtocolType is supported in current ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetGlobal_CheckIfProtocolSupported(
                                                               uint8_t bProtocolType,
                                                               uint8_t * pSupported
                                                               );

/**
* \brief Get direct or inverse Convention Type
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetGlobal_Convention(
                                                 uint8_t * pConvention
                                                 );

/**
* \brief Convert Index to real Values according to ISO7816 table
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_ConvertFdIndexesToValues(
                                                     uint32_t bFdIndexes,
                                                     uint16_t * fValue,
                                                     uint16_t * dValue
                                                     );
/**
* \brief Get the length of the internal ATR
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phToolsAtrParser_GetAtrLength(
                                         uint8_t * bAtrLength
                                         );


/** @}
* end of phToolsAtrParser group
*/

#ifdef __cplusplus
}/*Extern C*/
#endif

#endif /* PHTOOLS_ATRPARSER_H */
