/*
* Copyright (c), NXP Semiconductors Bangalore / India
*
*               (C)NXP Semiconductors
* All rights are reserved. Reproduction in whole or in part is
* prohibited without the written consent of the copyright owner.
* NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
* particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
* arising from its use.
*/

/** \file
* Generic MIFARE IDENTITY MFIDENTITY Application Component of Reader Library Framework.
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7230 $
* $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
*
* History:
*  Santosh Araballi: Generated 31. August 2010
*
*/

#ifndef PHALMFIDENTITY_H
#define PHALMFIDENTITY_H

#include <ph_Status.h>
#include <phTMIUtils.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <ph_TypeDefs.h>
#include <ph_RefDefs.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/***************************************************************************************************************************************/
/* Software Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW

/**
 * \defgroup phalMfIdentity_Sw Component : Software
 * @{
 */

#define PHAL_MFIDENTITY_SW_ID											0x01U   /**< ID for Software MFIDENTITY layer. */

/** \name Sizes */
/** @{ */
#define PHAL_MFIDENTITY_SIZE_TI											4U		/**< Size of Transaction Identifier. */
#define PHAL_MFIDENTITY_SIZE_MAC										16U		/**< Size of (untruncated) MAC. */
#define PHAL_MFIDENTITY_BLOCK_SIZE										16U		/**< Block Size */
#define PHAL_MFIDENTITY_MAX_WRITE_SIZE									0xFFFDU	/**< Max size in a Write function. */
/** @} */

/** \brief MIFARE Identity Software parameter structure  */
typedef struct
{
	uint16_t wId;																/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;                                				/**< Pointer to the parameter structure of the palMifare component. */
	void * pKeyStoreDataParams;                                 				/**< Pointer to the parameter structure of the KeyStore layer. */
	void * pCryptoDataParamsEnc;                                				/**< Pointer to the parameter structure of the Crypto layer for encryption. */
	void * pCryptoDataParamsMac;                                				/**< Pointer to the parameter structure of the CryptoMAC. */
	void * pCryptoRngDataParams;                                				/**< Pointer to the parameter structure of the CryptoRng layer. */
	void * pHalDataParams;                                      				/**< Pointer to the HAL parameters structure. */
	uint8_t bSesAuthENCKey[24];                                 				/**< Session key for this authentication */
	uint8_t bKeyNo;                                             				/**< key number against which this authentication is done */
	uint8_t bIv[16];                                            				/**< Max size of IV can be 16 bytes */
	uint8_t bAuthMode;                                          				/**< Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA) */
	uint8_t pAid[3];                                            				/**< Aid of the currently selected application */
	uint8_t bCryptoMethod;                                      				/**< DES,3DES, 3K3DES or AES, LRP  */
	uint8_t bWrappedMode;                                       				/**< Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs. */
	uint16_t wCrc;                                              				/**< 2 Byte CRC initial value in Authenticate mode. */
	uint32_t dwCrc;                                             				/**< 4 Byte CRC initial value in 0x1A, 0xAA mode. */
	uint16_t wAdditionalInfo;                                   				/**< Specific error codes for Desfire generic errors. */
	uint32_t dwPayLoadLen;                                       				/**< Amount of data to be read. Required for Enc read to verify CRC. */
	uint16_t wCmdCtr ;                                          				/**< Command count within transaction. */
    uint8_t bTi[PHAL_MFIDENTITY_SIZE_TI];                      					/**< Transaction Identifier. */
	uint8_t bSesAuthMACKey[16];                                 				/**< Authentication MAC key for the session. */
	uint8_t bKeySessionAuthMaster[16];                          				/**< Session Auth master key. */
    uint8_t pUnprocByteBuff[PHAL_MFIDENTITY_SIZE_MAC];        					/**< Buffer containing unprocessed bytes for read mac answer stream. */
	uint8_t bNoUnprocBytes;                                     				/**< Amount of data in the pUnprocByteBuff. */
	uint8_t bLastBlockBuffer[16];                               				/**< Buffer to store last Block of encrypted data in case of chaining. */
	uint8_t bLastBlockIndex;                                    				/**< Last Block Buffer Index. */
	void * pTMIDataParams;                                      				/**< Pointer to the parameter structure for collecting TMI. */
	uint8_t bShortLenApdu;	     												/**< Parameter for force set Short Length APDU in case of BIG ISO read. */
	uint8_t bUid[7];	     													/**< Store the UID of the card. This get updated if GetCardUID commmand is called. */
	uint8_t bUidLength;	     													/**< Length of bytes available in UID buffer. */
	uint8_t bFileOption;	     												/**< The bit settings of the FileOption parameter. */
	uint8_t bSDMOption;	     													/**< Bit setting of SDM option. */
	uint8_t bSDMKeyNo;	     													/**< Reference key number used for SDM operation. */
	uint8_t bSDMKeyVer;	     													/**< Reference Key version used for SDM operation. */
	uint8_t aSDMENCOffset[3];	     											/**< Buffer to represent the SDM ENC Offset. */
	uint8_t aSDMENCLength[3];	     											/**< Buffer to represent the SDM ENC Length. */
	uint8_t aSDMReadCTR[3];		     											/**< Buffer to represent the SDM Read Counter value. */
	uint8_t bKeyType;															/**< Distinguish between the LRP_AES or AES key. */
} phalMfIdentity_Sw_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfIdentity_Sw_Init(
		phalMfIdentity_Sw_DataParams_t * pDataParams,							/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,												/**< [In] Specifies the size of the data parameter structure */
		void * pPalMifareDataParams,											/**< [In] Pointer to a palMifare component context. */
		void * pKeyStoreDataParams,												/**< [In] Pointer to Key Store data parameters. */
		void * pCryptoDataParamsEnc,											/**< [In] Pointer to a Crypto component context for encryption. */
		void * pCryptoDataParamsMac,											/**< [In] Pointer to a CryptoMAC component context. */
		void * pCryptoRngDataParams,											/**< [In] Pointer to a CryptoRng component context. */
		void * pTMIDataParams,													/**< [In] Pointer to a TMI component. */
		void * pHalDataParams													/**< [In] Pointer to the HAL parameters structure. */
	);

/**
 * end of group phalMfIdentity_Sw
 * @}
 */

#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

/***************************************************************************************************************************************/
/* Sam NonX Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX

/**
 * \defgroup phalMfIdentity_Sam_NonX Component : Sam_NonX
 * @{
 */

#define PHAL_MFIDENTITY_SAMAV3_NONX_ID									0x02U	/**< ID for SamAV3 Non X MF Identity layer. */

/** \brief MIFARE Identity SAM AV3 NonX parameter structure  */
typedef struct{
	uint16_t wId;																/**< Layer ID for this component, NEVER MODIFY! */
	phhalHw_SamAV3_DataParams_t * pHalSamDataParams;    						/**< Pointer to the parameter structure of the hal of the SAM layer. */
	void * pPalMifareDataParams;                        						/**< Pointer to the parameter structure of the palMifare component. */
	phTMIUtils_t * pTMIDataParams;                                      		/**< Pointer to the parameter structure for collecting TMI. */
	uint8_t bAuthMode;                                  						/**< Auth. Mode 0x0A or Ox1A or OxAA. */
	uint8_t bAuthType;                                  						/**< AuthType. Whether its EV2 or LRP. */
	uint8_t bKeyNo;                                     						/**< Key number against which authenticated. */
	uint8_t aAid[3];                                    						/**< Currently selected application Id. */
	uint16_t wAdditionalInfo;                           						/**< Specific error codes for Desfire generic errors or error codes returned by the card. */
	uint8_t aUid[7];	     													/**< Store the UID of the card. This get updated if GetCardUID commmand is called. */
	uint8_t bUidLength;	     													/**< Length of bytes available in UID buffer. */
	uint8_t bFileOption;	     												/**< The bit settings of the FileOption parameter. */
	uint8_t bSDMOption;	     													/**< Bit setting of SDM option. */
	uint8_t bSDMKeyNo;	     													/**< Reference key number used for SDM operation. */
	uint8_t bSDMKeyVer;	     													/**< Reference Key version used for SDM operation. */
	uint8_t bRAMKeyNo;	     													/**< Reference RAM key number used for SDM operation. */
	uint8_t bRAMKeyVer;	     													/**< Reference RAM Key version used for SDM operation. */
	uint8_t aSDMENCOffset[3];	     											/**< Buffer to represent the SDM ENC Offset. */
	uint8_t aSDMENCLen[3];	     												/**< Buffer to represent the SDM ENC Length. */
	uint8_t aSDMReadCtr[3];		     											/**< Buffer to represent the SDM Read Counter value. */
	uint8_t bKeyType;															/**< Distinguish between the LRP_AES or AES key. */
} phalMfIdentity_SamAV3_NonX_DataParams_t;

/**
 * \brief Initializes the MIFARE Identity Sam AV3 component in Non X mode (S mode).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS for Invaid Dataparams
 * \retval #PH_ERR_INVALID_PARAMETER for invalid parameter values.
 */
phStatus_t phalMfIdentity_SamAV3_NonX_Init(
		phalMfIdentity_SamAV3_NonX_DataParams_t * pDataParams,					/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,												/**< [In] Specifies the size of the data parameter structure */
		phhalHw_SamAV3_DataParams_t * pHalSamDataParams,						/**< [In] Pointer to the HAL parameter structure of the SAM. */
		void * pPalMifareDataParams,											/**< [In] Pointer to Pal Mifare parameter structure. */
		phTMIUtils_t * pTMIDataParams											/**< [In] Pointer to the parameter structure for collecting TMI. */
	);

/**
 * end of group phalMfIdentity_Sam_NonX
 * @}
 */
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

/***************************************************************************************************************************************/
/* SAM X Dataparams and Initialization Interface.																				       */
/***************************************************************************************************************************************/

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X

/**
 * \defgroup phalMfIdentity_Sam_X Component : Sam_X
 * @{
 */

#define PHAL_MFIDENTITY_SAMAV3_X_ID										0x03U   /**< ID for SamAV3 X-Mode MF Identity layer. */


/** \brief MIFARE Identity SAM AV3 X parameter structure  */
typedef struct{
	uint16_t wId;																/**< Layer ID for this component, NEVER MODIFY! */
	phhalHw_SamAV3_DataParams_t * pHalSamDataParams;    						/**< Pointer to the parameter structure of the hal of the SAM layer. */
	phTMIUtils_t * pTMIDataParams;                                      		/**< Pointer to the parameter structure for collecting TMI. */
	uint8_t bAuthMode;                                  						/**< Auth Mode. 0x0A or 0x1A or 0xAA. */
	uint8_t bAuthType;                                  						/**< AuthType. Whether its EV2 or LRP. */
	uint8_t bKeyNo;                                     						/**< Key number against which authenticated. */
	uint8_t aAid[3];                                    						/**< Currently selected application Id. */
	uint16_t wAdditionalInfo;                           						/**< Specific error codes for Desfire generic errors. */
	uint8_t aUid[7];	     													/**< Store the UID of the card. This get updated if GetCardUID commmand is called. */
	uint8_t bUidLength;	     													/**< Length of bytes available in UID buffer. */
	uint8_t bFileOption;	     												/**< The bit settings of the FileOption parameter. */
	uint8_t bSDMOption;	     													/**< Bit setting of SDM option. */
	uint8_t bSDMKeyNo;	     													/**< Reference key number used for SDM operation. */
	uint8_t bSDMKeyVer;	     													/**< Reference Key version used for SDM operation. */
	uint8_t bRAMKeyNo;	     													/**< Reference RAM key number used for SDM operation. */
	uint8_t bRAMKeyVer;	     													/**< Reference RAM Key version used for SDM operation. */
	uint8_t aSDMENCOffset[3];	     											/**< Buffer to represent the SDM ENC Offset. */
	uint8_t aSDMENCLen[3];	     												/**< Buffer to represent the SDM ENC Length. */
	uint8_t aSDMReadCtr[3];		     											/**< Buffer to represent the SDM Read Counter value. */
	uint8_t bKeyType;															/**< Distinguish between the LRP_AES or AES key. */
} phalMfIdentity_SamAV3_X_DataParams_t;

/**
 * \brief Initialise SamAV3 X-Mode layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfIdentity_SamAV3_X_Init(
		phalMfIdentity_SamAV3_X_DataParams_t * pDataParams,						/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,												/**< [In] Specifies the size of the data parameter structure */
		phhalHw_SamAV3_DataParams_t * pHalSamDataParams,						/**< [In] Pointer to the HAL parameter structure of the SAM. */
		phTMIUtils_t * pTMIDataParams											/**< [In] Pointer to the parameter structure for collecting TMI. */
	);

/**
 * end of group phalMfIdentity_Sam_X
 * @}
 */

#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */



/***************************************************************************************************************************************/
/* MIFARE Identity Generic command declarations.     																				   */
/***************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_MFIDENTITY

/** \defgroup phalMfIdentity MIFARE Identity
 * \brief These Functions implement the MIFARE Identity commands.
 * @{
 */

/** \name Other Options for various Functions */
/** @{ */
#define PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST								0x01U	/**< Originality key for AES */
#define PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST								0x04U	/**< Originality key for AES  */
#define PHAL_MFIDENTITY_APP_KEY_LAST										0x04U	/**< Numebr of application keys. */
#define PHAL_MFIDENTITY_MAC_DATA_INCOMPLETE									0x01U	/**< Option for indicating more data to come in next call for MAC calculation */
#define PHAL_MFIDENTITY_COMMUNICATION_MAC_ON_RC								0x02U	/**< MAC is appended over response  */
#define PHAL_MFIDENTITY_AUTHENTICATE_RESET									0x08U	/**< authentication state shall be reset */
/** @} */

/** \name The communication mode to be used. */
/** @{ */
#define PHAL_MFIDENTITY_COMMUNICATION_PLAIN									0x00U	/**< Plain mode of communication. */
#define PHAL_MFIDENTITY_COMMUNICATION_PLAIN_1								0x20U	/**< Plain mode of communication. */
#define PHAL_MFIDENTITY_COMMUNICATION_MACD									0x10U	/**< MAC mode of communication. */
#define PHAL_MFIDENTITY_COMMUNICATION_ENC									0x30U	/**< Enciphered mode of communication. */
/** @} */

/** \name Error Codes mapping to custom values. */
/** @{ */
#define PHAL_MFIDENTITY_ERR_FORMAT						(PH_ERR_CUSTOM_BEGIN + 0)	/**< MF Identity Response - No changes done to backup files. */
#define PHAL_MFIDENTITY_ERR_OUT_OF_EEPROM_ERROR			(PH_ERR_CUSTOM_BEGIN + 1)	/**< MF Identity Response - Insufficient NV-Memory. */
#define PHAL_MFIDENTITY_ERR_NO_SUCH_KEY					(PH_ERR_CUSTOM_BEGIN + 2)	/**< MF Identity Invalid key number specified. */
#define PHAL_MFIDENTITY_ERR_PERMISSION_DENIED			(PH_ERR_CUSTOM_BEGIN + 3)	/**< MF Identity Current configuration/status does not allow the requested command. */
#define PHAL_MFIDENTITY_ERR_APPLICATION_NOT_FOUND		(PH_ERR_CUSTOM_BEGIN + 4)	/**< MF Identity Requested AID not found on PICC. */
#define PHAL_MFIDENTITY_ERR_BOUNDARY_ERROR				(PH_ERR_CUSTOM_BEGIN + 5)	/**< MF Identity Attempt to read/write data from/to beyond the files/record's limits. */
#define PHAL_MFIDENTITY_ERR_COMMAND_ABORTED				(PH_ERR_CUSTOM_BEGIN + 6)	/**< MF Identity Previous cmd not fully completed. Not all frames were requested or provided by the PCD. */
#define PHAL_MFIDENTITY_ERR_COUNT						(PH_ERR_CUSTOM_BEGIN + 7)	/**< MF Identity Num. of applns limited to 28. No additional applications possible. */
#define PHAL_MFIDENTITY_ERR_DUPLICATE					(PH_ERR_CUSTOM_BEGIN + 8)	/**< MF Identity File/Application with same number already exists. */
#define PHAL_MFIDENTITY_ERR_FILE_NOT_FOUND				(PH_ERR_CUSTOM_BEGIN + 9)	/**< MF Identity Specified file number does not exist. */
#define PHAL_MFIDENTITY_ERR_PICC_CRYPTO					(PH_ERR_CUSTOM_BEGIN + 10)	/**< MF Identity Crypto error returned by PICC. */
#define PHAL_MFIDENTITY_ERR_PARAMETER_ERROR				(PH_ERR_CUSTOM_BEGIN + 11)	/**< MF Identity Parameter value error returned by PICC. */
#define PHAL_MFIDENTITY_ERR_DF_GEN_ERROR				(PH_ERR_CUSTOM_BEGIN + 12)	/**< MF Identity DesFire Generic error. Check additional Info. */
#define PHAL_MFIDENTITY_ERR_DF_7816_GEN_ERROR			(PH_ERR_CUSTOM_BEGIN + 13)	/**< MF Identity ISO 7816 Generic error. Check Additional Info. */
#define PHAL_MFIDENTITY_ERR_CMD_INVALID					(PH_ERR_CUSTOM_BEGIN + 14)	/**< MF Identity ISO 7816 Generic error. Check Additional Info. */
/** @} */

/** \name Key Types */
/** @{ */
#define PHAL_MFIDENTITY_KEY_TYPE_2K3DES										0x00U	/**< 2 Key Triple Des */
#define PHAL_MFIDENTITY_KEY_TYPE_3K3DES										0x01U	/**< 3 Key Triple Des */
#define PHAL_MFIDENTITY_KEY_TYPE_AES128										0x02U	/**< AES 128 Key */
/** @} */




#ifdef NXPBUILD__PH_CRYPTOSYM
/* MIFARE Identity secure messaging related commands. -------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfIdentity_SecureMessaging Commands_SecureMessaging
 * \brief These Components implement the MIFARE Identity Secure Messaging related commands.
 * @{
 */

/** \name Authentication Modes */
/** @{ */
#define PHAL_MFIDENTITY_NOT_AUTHENTICATED									0xFFU	/**< No authentication. */
#define PHAL_MFIDENTITY_AUTHENTICATE										0x0AU	/**< D40 Authentication; 0x0A. */
#define PHAL_MFIDENTITY_AUTHENTICATEISO										0x1AU	/**< ISO Authentication; 0x1A. */
#define PHAL_MFIDENTITY_AUTHENTICATEAES										0xAAU	/**< AES Authentication; 0xAA. */
#define PHAL_MFIDENTITY_AUTHENTICATEEV2										0x71U	/**< EV2 First Authentication; 0x71. */
#define PHAL_MFIDENTITY_CMD_AUTHENTICATE_PART2								0x77U	/**< EV2 NonFirst Authenticate; 0x77. */
#define PHAL_MFIDENTITY_AUTH_TYPE_EV2										0x00U	/**< Auth Type as EV2. */
#define PHAL_MFIDENTITY_AUTH_TYPE_LRP										0x01U	/**< Auth Type as LRP. */
#define PHAL_MFIDENTITY_AUTH_TYPE_UNKNOWN									0xFFU	/**< Auth Type as Unknown. */
#define PHAL_MFIDENTITY_AUTHNONFIRST_NON_LRP								0x0000U	/**< Non First Auth in regular EV2 auth Mode. */
#define PHAL_MFIDENTITY_AUTHFIRST_NON_LRP									0x0001U	/**< First Auth in regular EV2 auth Mode. */
#define PHAL_MFIDENTITY_AUTHNONFIRST_LRP									0x0002U	/**< Non First Auth in LRP mode. */
#define PHAL_MFIDENTITY_AUTHFIRST_LRP										0x0003U	/**< First Auth in LRP mode. */
/** @} */

/** \name Diversification options to be used with Authenticate command. */
/** @{ */
#define PHAL_MFIDENTITY_NO_DIVERSIFICATION            						0xFFFFU	/**< No diversification. */
#define PHAL_MFIDENTITY_DIV_METHOD_ENCR				PH_CRYPTOSYM_DIV_MODE_DESFIRE	/**< Encryption based method of diversification. */
#define PHAL_MFIDENTITY_DIV_METHOD_CMAC			PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS	/**< CMAC based method of diversification. */
/** @} */

/**
 * \brief Performs an First or Non First Authentication depending upon bFirstAuth Parameter. This will be using the AES128 keys and will
 * generate and verify the contents based on generic AES algorithm.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_AuthenticateEv2(
		void *pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFirstAuth,     												/**< [In] One of the below options.
																				 *			\arg #PHAL_MFIDENTITY_AUTHNONFIRST_NON_LRP
																				 *			\arg #PHAL_MFIDENTITY_AUTHFIRST_NON_LRP
																				 *			\arg #PHAL_MFIDENTITY_AUTHNONFIRST_LRP
																				 *			\arg #PHAL_MFIDENTITY_AUTHFIRST_LRP
																				 */
		uint16_t wOption,       												/**< [In] Diversification option can be one of
																				 *			\arg #PHAL_MFIDENTITY_NO_DIVERSIFICATION
																				 *			\arg #PHAL_MFIDENTITY_DIV_METHOD_ENCR
																				 *			\arg #PHAL_MFIDENTITY_DIV_METHOD_CMAC
																				 */
		uint16_t wKeyNo,        												/**< [In] Key number in keystore of software or SAM. */
		uint16_t wKeyVer,       												/**< [In] Key version in keystore of software or SAM. */
		uint8_t bKeyNoCard,     												/**< [In] Key number on card. */
		uint8_t * pDivInput,    												/**< [In] Diversification input. Can be NULL. */
		uint8_t bDivLen,        												/**< [In] Length of diversification input max 31B. */
		uint8_t bLenPcdCapsIn,  												/**< [In] Length of PcdCapsIn. Always zero for following authentication */
		uint8_t *bPcdCapsIn,    												/**< [In] PCD Capabilities. */
		uint8_t *bPcdCapsOut,   												/**< [Out] PCD Capabilities. */
		uint8_t *bPdCapsOut     												/**< [Out] PD Capabilities. */
	);

/**
 * end of group phalMfIdentity_SecureMessaging
 * @}
 */
#endif /* NXPBUILD__PH_CRYPTOSYM */




/* MIFARE Identity Memory and Configuration mamangement commands. -------------------------------------------------------------------- */
/**
 * \defgroup phalMfIdentity_MemoryConfiguration Commands_MemoryConfiguration
 * \brief These Components implement the MIFARE Identity Memory and Configuration Management related commands.
 * @{
 */

#ifdef NXPBUILD__PH_CRYPTOSYM
/** \name Options for SetConfiguration command. */
/** @{ */
#define PHAL_MFIDENTITY_SET_CONFIG_OPTION0									0x00U	/**< Option 0 for PICC configuration. */
#define PHAL_MFIDENTITY_SET_CONFIG_OPTION2									0x02U	/**< Option 2 for ATS update. */
#define PHAL_MFIDENTITY_SET_CONFIG_OPTION3									0x03U	/**< Option 3 for SAK update. */
#define PHAL_MFIDENTITY_SET_CONFIG_OPTION4									0x04U	/**< Option 4 for Secure Messaging Configuration. */
#define PHAL_MFIDENTITY_SET_CONFIG_OPTION5									0x05U	/**< Option 5 for Capability Data. */
#define	PHAL_MFIDENTITY_SET_CONFIG_OPTION10									0x0AU	/**< Option 10 for Failed Authentication Counter Configuration. */
#define	PHAL_MFIDENTITY_SET_CONFIG_OPTION11									0x0BU	/**< Option 11 for Hardware Configuration. */
/** @} */

/**
 * \brief Configures the card and pre personalizes the card with a key, defines if the UID or the random ID is sent back during communication setup and
 * configures the ATS string.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_SetConfiguration(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Configuration Option. Define length and content of the Data parameter.
																				 *			\arg #PHAL_MFIDENTITY_SET_CONFIG_OPTION0
																				 *			\arg #PHAL_MFIDENTITY_SET_CONFIG_OPTION2
																				 *			\arg #PHAL_MFIDENTITY_SET_CONFIG_OPTION3
																				 *			\arg #PHAL_MFIDENTITY_SET_CONFIG_OPTION4
																				 *			\arg #PHAL_MFIDENTITY_SET_CONFIG_OPTION5
																				 *			\arg #PHAL_MFIDENTITY_SET_CONFIG_OPTION10
																				 *			\arg #PHAL_MFIDENTITY_SET_CONFIG_OPTION11
																				 */
		uint8_t * pData,														/**< [In] Data for the option specified.*/
		uint8_t bDataLen														/**< [In] Length of bytes available in Data buffer. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * \brief Returns manufacturing related data of the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_GetVersion(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pVerInfo														/**< [Out] The version information of the product. */
	);

#ifdef NXPBUILD__PH_CRYPTOSYM
/**
 * \brief Returns the Unique ID of the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_GetCardUID(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pUid															/**< [Out] The complete UID of the PICC. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * end of group phalMfIdentity_MemoryConfiguration
 * @}
 */




#ifdef NXPBUILD__PH_CRYPTOSYM
/* MIFARE Identity Key mamangement commands. ----------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfIdentity_KeyManagement Commands_KeyManagement
 * \brief These Components implement the MIFARE Identity Key Management related commands.
 * @{
 */

/** \name Diversification options to be used with ChangeKey command. */
/** @{ */
#define PHAL_MFIDENTITY_CHGKEY_NO_DIVERSIFICATION            				0xFFFFU	/**< No diversification. */
#define PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY            						0x0002U	/**< Bit 1. Indicating diversification of new key requred. */
#define PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY            						0x0004U	/**< Bit 2 indicating old key was diversified. */
#define PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY_ONERND     						0x0008U	/**< Bit 3 indicating new key diversification using one rnd. Default is two rounds. */
#define PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY_ONERND     						0x0010U	/**< Bit 4 indicating old key diversification using one rnd. Default is two rounds. */
#define PHAL_MFIDENTITY_CHGKEY_DIV_METHOD_CMAC								0x0020U	/**< Bit 5 indicating key diversification method based on CMAC. Default is Encryption method */
/** @} */

/**
 * \brief Changes any key on the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_ChangeKey(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,      													/**< [In] One of the below options.
																				 *			\arg #PHAL_MFIDENTITY_NO_DIVERSIFICATION
																				 *			\arg #PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY | #PHAL_MFIDENTITY_CHGKEY_DIV_METHOD_CMAC
																				 *			\arg #PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY | #PHAL_MFIDENTITY_CHGKEY_DIV_METHOD_CMAC
																				 *			\arg #PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY | #PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY_ONERND
																				 *			\arg #PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY | #PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY_ONERND
																				 *			\arg #PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY | #PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY
																				 *			\arg #PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY | #PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY |
																				 *				 #PHAL_MFIDENTITY_CHGKEY_DIV_METHOD_CMAC
																				 *			\arg #PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY | #PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY |
																				 *				 #PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY_ONERND | #PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY_ONERND
																				 */
		uint16_t wOldKeyNo,														/**< [In] Old key number in keystore of software or SAM. */
		uint16_t wOldKeyVer,    												/**< [In] Old key number in keystore of software or SAM. */
		uint16_t wNewKeyNo,     												/**< [In] New key number in keystore of software or SAM. */
		uint16_t wNewKeyVer,    												/**< [In] New key number in keystore of software or SAM. */
		uint8_t bKeyNoCard,     												/**< [In] Key number on card. */
		uint8_t * pDivInput,    												/**< [In] Diversification input. Can be NULL. */
		uint8_t bDivLen         												/**< [In] Length of diversification input max 31B. */
	);

/**
 * \brief Reads out the current key version of any key stored on the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_GetKeyVersion(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bKeyNo,															/**< [In] Key number on card. */
		uint8_t bKeySetNo,														/**< [In] 1 byte Key set number. Optional as it is passed only when bit[6] of bKeyNo is set. */
		uint8_t * pKeyVersion,													/**< [Out] The version of the specified key. */
		uint8_t * bRxLen														/**< [Out] Length of bytes available in KeyVersion buffer. */
	);

/**
 * end of group phalMfIdentity_KeyManagement
 * @}
 */
#endif /* NXPBUILD__PH_CRYPTOSYM */




/* MIFARE DESFire Light File mamangement commands. ------------------------------------------------------------------------------------ */
/**
 * \defgroup phalMfIdentity_FileManagement Commands_FileManagement
 * \brief These Components implement the MIFARE Identity File Management related commands.
 * @{
 */

#ifdef NXPBUILD__PH_CRYPTOSYM
/** \name Diversification options to be used with CreateTransactionMacFile command.
 * These macros are valid only if the layer is initialized for SAM.
 */
/** @{ */
#define PHAL_MFIDENTITY_CRTMACFILE_DIVERSIFICATION_OFF            			0x00U	/**< No diversification. */
#define PHAL_MFIDENTITY_CRTMACFILE_DIVERSIFICATION_ON            			0x01U	/**< Diversification is enabled.*/
/** @} */

/**
 * \brief Creates a Transaction MAC file. An application can have only one Transaction MAC File.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_CreateTransactionMacFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo,														/**< [In] File number of the file to be created. */
		uint8_t bCommMode,          											/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN_1
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_ENC
																				 */
		uint8_t * pAccessRights,    											/**< [In] The access right to be applied for the file. Should be 2 byte. */
		uint8_t bKeyType,           											/**< [In] Bit0-1 indicates key type and it should be always #PHAL_MFIDENTITY_KEY_TYPE_AES128 */
		uint8_t * bTMKey,           											/**< [In] The 16 byte key value to be used as TMKey. */
		uint16_t wKeyNo,														/**< [In] The reference key number in SAM keystore. Only valid if initialized for SAM. */
		uint8_t bTMKeyVer, 														/**< [In] The version of the TMKey. */
		uint8_t * pDivInput,													/**< [In] Diversification input to diversify the TMKey. */
		uint8_t bDivInputLen													/**< [In] Length of bytes available in DivInput buffer. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * \brief Permanently deactivates a file within the file directory of the currently selected application.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_DeleteFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo															/**< [In] File number of the file to be deleted. */
	);

/**
 * \brief Get informtion on the properties of a specific file
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_GetFileSettings(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo,														/**< [In] The file number for which the setting to be retrieved. */
		uint8_t * pFSBuffer,													/**< [Out] The buffer containing the settings. The buffer should be 17 bytes. */
		uint8_t * bBufferLen													/**< [Out] The length of bytes available in FSBuffer buffer. */
	);

#ifdef NXPBUILD__PH_CRYPTOSYM
/**
 * \brief Returns manufacturing related data of the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_GetFileCounters(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Indicates the mode of communication to be used while exchanging the data to PICC.
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_ENC
																				 */
		uint8_t bFileNo,														/**< [In] File number for which the Counter information need to be received. */
		uint8_t * pFileCounters,												/**< [Out] The SDMReadCounter information returned by the PICC. */
		uint8_t * pRxLen														/**< [Out] Length of bytes available in FileCounters buffer. */
	);

/** \name The File option and other flags for ChangeFileSettings command. */
/** @{ */
#define PHAL_MFIDENTITY_FILE_OPTION_PLAIN								0x00U	/**< Plain mode of communication. */
#define PHAL_MFIDENTITY_FILE_OPTION_PLAIN_1								0x02U	/**< Plain mode of communication. */
#define PHAL_MFIDENTITY_FILE_OPTION_MACD								0x01U	/**< MAC mode of communication. */
#define PHAL_MFIDENTITY_FILE_OPTION_ENC									0x03U	/**< Enciphered mode of communication. */
#define PHAL_MFIDENTITY_EXCLUNAUTHCONFIG								0x10U	/**< 4th Bit of FileOption indicating Excude Unauthentication config. */
#define PHAL_MFIDENTITY_TMCLIMITCONFIG									0x20U	/**< 5th Bit of FileOption indicating TMC limit config. */
#define PHAL_MFIDENTITY_FILE_OPTION_SDM_ENABLED							0x40U	/**< 6th Bit of FileOption indicating Secure Dynamin Messaging and Mirroring. */

#define PHAL_MFIDENTITY_VCUID_PRESENT									0x80U	/**< Only VCUID is considred for SDM MAC calculation */
#define PHAL_MFIDENTITY_RDCTR_PRESENT									0x40U	/**< Only RDCTR  is considred for SDM MAC calculation */
#define PHAL_MFIDENTITY_RDCTR_LIMIT_PRESENT								0x20U	/**< Indicated the presence of SDM Read Counter Limit. */
#define PHAL_MFIDENTITY_SDM_ENC_FILE_DATA_PRESENT						0x10U	/**< Indicated the presence of SDM ENC File data. */
#define PHAL_MFIDENTITY_MFID_SPECIFICS_ENABLED							0x01U	/**< MFID Specific feature to  be used. */
/** @} */

/**
 * \brief Changes the access parameters of an existing file.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_ChangeFileSettings(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommMode,														/**< [In] Indicates the mode of communication to be used while exchanging the data to PICC.
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_ENC
																				 *		  Ored with the below option
																				 *			\arg #PHAL_MFIDENTITY_MFID_SPECIFICS_ENABLED
																				 */
		uint8_t bFileNo,														/**< [In] File number for which the setting need to be updated. */
		uint8_t bFileOption,      												/**< [In] New communication settings for the file.
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_PLAIN_1
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_MACD
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_ENC
																				 *		  \n
																				 *		  Ored with one of the below flags if required.
																				 *			\arg #PHAL_MFIDENTITY_EXCLUNAUTHCONFIG
																				 *			\arg #PHAL_MFIDENTITY_TMCLIMITCONFIG
																				 */
		uint8_t * pAccessRights,												/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t bAdditionalInfoLen,												/**< [In] Length of the Additional Info to be sent (2 Bytes or 4 Bytes).
																				 *		  This is used to set TMC Limit in case of TMAC File
																				 */
		uint8_t * bAdditionalInfo												/**< [In] One of the below inforamtion. \n
																				 *			\c If Standard AES: 4 byte TMC Limit value. \n
																				 *			\c If LRP         : 2 byte TMC Limit value.
																				 */
		);

/**
 * \brief Changes the access parameters of an existing file.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_ChangeFileSettingsSDM(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Indicates the mode of communication to be used while exchanging the data to PICC.
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_ENC
																				 */
		uint8_t bFileNo,														/**< [In] File number for which the setting need to be updated. */
		uint8_t bFileOption,      												/**< [In] New communication settings for the file.
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_PLAIN_1
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_MACD
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_ENC
																				 *		  \n
																				 *		  Ored with one of the below flags if required.
																				 *			\arg #PHAL_MFIDENTITY_TMCLIMITCONFIG
																				 *			\arg #PHAL_MFIDENTITY_EXCLUNAUTHCONFIG
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION_SDM_ENABLED
																				 */
		uint8_t * pAccessRights,												/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t bSdmOptions,													/**< [In] One of the below values to be used. Can be ORed.
																				 *			\arg #PHAL_MFIDENTITY_VCUID_PRESENT
																				 *			\arg #PHAL_MFIDENTITY_RDCTR_PRESENT
																				 *			\arg #PHAL_MFIDENTITY_RDCTR_LIMIT_PRESENT
																				 *			\arg #PHAL_MFIDENTITY_SDM_ENC_FILE_DATA_PRESENT
																				 */
		uint8_t * pSdmAccessRights,												/**< [In] The SDM access rights to be applied. Should be 2 bytes. */
		uint8_t * pPICCDataOffset,												/**< [In] Mirror position (LSB first) for encrypted PICCData. Should be 3 bytes. */
		uint8_t * pSDMMACInputOffset,											/**< [In] Offset in the file where the SDM MAC computation starts (LSB first). Should be 3 bytes. */
		uint8_t * pSDMENCOffset,												/**< [In] SDMENCFileData mirror position (LSB first). Should be 3 bytes. */
		uint8_t * pSDMENCLen,													/**< [In] Length of the SDMENCFileData (LSB first). Should be 3 bytes. */
		uint8_t * pSDMMACOffset ,												/**< [In] SDMMAC mirror position (LSB first). Should be 3 bytes. */
		uint8_t * pSDMReadCtrOffset,											/**< [In] SDMReadCtrLimit value (LSB first). Should be 3 bytes. */
		uint8_t * pTMCLimitValue												/**< [In] TMC Limit to mbe updated. Should be one of the below values. \n
																				 *			\c If EV2 Authenticated, then 4 bytes else \n
																				 *			\c If LRP Authenticated, then 2 bytes
																				 */

	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * end of group phalMfIdentity_FileManagement
 * @}
 */




/* MIFARE Identity Data mamangement commands. ---------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfIdentity_DataManagement Commands_DataManagement
 * \brief These Components implement the MIFARE Identity Data Management related commands.
 * @{
 */

/**
 * \brief Reads data from standard data files or backup data files
 *
 * \remarks
 *
 * Chaining upto the size of the HAL Rx buffer is handled within this function. If more data is to be read, the user has to call
 * this function again with bOption = #PH_EXCHANGE_RXCHAINING | [one of the communication options]
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING indicating more data to be read.
 * \retval Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_ReadData(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_ENC
																				 *
																				 *			\c #PH_EXCHANGE_RXCHAINING: To be Or'd with the above option
																				 *				flag if Chaining status is returned.
																				 */
		uint8_t bIns,															/**< [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		uint8_t bFileNo,														/**< [In] The file number from where the data to be read. */
		uint8_t * pOffset,														/**< [In] The offset from where the data should be read. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 need to be offset then it will be 10 00 00.
																				 */
		uint8_t * pLength,														/**< [In] The number of bytes to be read. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be read then it will be 10 00 00. \n
																				 *			\c If complete file need to be read then it will be 00 00 00.
																				 */
		uint8_t ** ppRxdata,													/**< [Out] The data retuned by the PICC. */
		uint16_t * pRxdataLen													/**< [Out] Length of bytes available in RxData buffer. */
	);

/**
 * \brief Writes data to standard data files or backup data files
 *
 * \remarks
 * Implements chaining to the card.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_WriteData(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_ENC
																				 */
		uint8_t bIns,															/**< [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		uint8_t bFileNo,														/**< [In] The file number from where the data to be read. */
		uint8_t * pOffset,														/**< [In] The offset from where the data should be written. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 need to be offset then it will be 10 00 00.
																				 */
		uint8_t * pTxData,														/**< [In] The data to be written to the PICC. */
		uint8_t * pTxDataLen													/**< [In] The number of bytes to be written. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be written then it will be 10 00 00.
																				 */
	);

/**
 * \brief Reads out a set of complete records from a Cyclic or Linear Record File.
 *
 * \remarks
 * The readrecords command reads and stores data in the rxbuffer upto the rxbuffer size before returning
 * to the user. The rxbuffer is configured during the HAL init and this is specified by the user.
 *
 * Chaining upto the size of the HAL Rx buffer is handled within this function.
 * If more data is to be read, the user has to call this function again with
 * bCommOption = PH_EXCHANGE_RXCHAINING | [one of the communication options]
 *
 * If TMI collection is ON, if pRecCount is zero then pRecSize is madatory parameter.
 * If pRecSize and RecCount are zero and TMI collection is ON, then  PH_ERR_INVALID_PARAMETER error returned.
 * If TMI collection is ON; and if wrong pRecSize is provided, then wrong RecCount value will be calculated and updated for TMI collection.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING indicating more data to be read.
 * \retval Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_ReadRecords(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_ENC
																				 *
																				 *			\c #PH_EXCHANGE_RXCHAINING: To be Or'd with the above option
																				 *				flag if Chaining status is returned.
																				 */
		uint8_t bIns,															/**< [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		uint8_t bFileNo,														/**< [In] The file number from where the data to be read. */
		uint8_t * pRecNo,														/**< [In] 3 bytes LSB first. Record number of the newest record targeted, starting to count
																				 *		  from the latest record written.
																				 */
		uint8_t * pRecCount,													/**< [In] 3 bytes LSB first. Number of records to be read. If 0x00 00 00, then all the records are read. */
		uint8_t * pRecSize,														/**< [In] The number of bytes to be read. Will be of 3 bytes with LSB first. */
		uint8_t ** ppRxdata,													/**< [Out] The data retuned by the PICC. */
		uint16_t * pRxdataLen													/**< [Out] Length of bytes available in RxData buffer. */
	);

/**
 * \brief Writes data to a record in a Cyclic or Linear Record File.
 *
 * \remarks
 * Implements chaining to the card.
 * The data provided on pData will be chained to the card by sending data upto the frame size of the DESFire PICC, at a time.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_WriteRecord(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFIDENTITY_COMMUNICATION_ENC
																				 */
		uint8_t bIns,															/**< [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		uint8_t bFileNo,														/**< [In] The file number to which the data should be written. */
		uint8_t * pOffset,														/**< [In] The offset from where the data should be written. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 need to be offset then it will be 10 00 00.
																				 */
		uint8_t * pData,														/**< [In] The data to be written to the PICC. */
		uint8_t * pDataLen														/**< [In] The number of bytes to be written. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be written then it will be 10 00 00.
																				 */
	);

/**
 * \brief Resets a Cyclic or Linear Record File.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_ClearRecordFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo															/**< [In] The file number to be cleared. */
	);

/**
 * end of group phalMfIdentity_DataManagement
 * @}
 */




/* MIFARE Identity Transaction mamangement commands. --------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfIdentity_TransactionnManagement Commands_TransactionManagement
 * \brief These Components implement the MIFARE Identity Transaction Management related commands.
 * @{
 */

/** \name Options for CommitTransaction command. */
/** @{ */
#define PHAL_MFIDENTITY_COMMIT_TXN_NO_TMC_TMV_RETURNED          			0x00	/**< Option byte is not exchanged to the PICC. */
#define PHAL_MFIDENTITY_COMMIT_TXN_TMC_TMV_RETURNED          				0x01	/**< Option byte exchanged to PICC and represent return of TMC and TMV. */
#define PHAL_MFIDENTITY_COMMIT_TXN_INCLUDE_OPTION          					0x80	/**< Option byte exchanged to PICC. This is to support the exchange of Option byte to PICC. */
/** @} */

/**
 * \brief Validates all previous write access' on Backup Data files, value files and record files within one application.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_CommitTransaction(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] One of the below options.
																				 *			\arg #PHAL_MFIDENTITY_COMMIT_TXN_NO_TMC_TMV_RETURNED
																				 *			\arg #PHAL_MFIDENTITY_COMMIT_TXN_TMC_TMV_RETURNED
																				 *		  \n
																				 *		  Can be ored with one of the above values.
																				 *			\arg #PHAL_MFIDENTITY_COMMIT_TXN_INCLUDE_OPTION
																				 */
		uint8_t * pTMC,															/**< [Out] The increased value as stored in FileType.TransactionMAC. \n
																				 *			\c If Standard AES: 4 byte Transaction MAC counter value. \n
																				 *			\c If LRP         : 2 byte of Actual TMC followed by 2 bytes of Session TMC
																				 */
		uint8_t * pTMV 															/**< [Out] 8 bytes Transaction MAC value. */
	);

/**
 * \brief Invalidates all previous write access' on Backup Data files, value files and record files within one application.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_AbortTransaction(
		void * pDataParams														/**< [In] Pointer to this layer's parameter structure. */
	);

/**
 * \brief Secures the transaction by commiting the application to ReaderID specified
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_CommitReaderID(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pTMRI,														/**< [In] 16 bytes of Transaction MAC reader ID. */
		uint8_t * pEncTMRI														/**< [Out] 16 bytes of Encrypted Transaction MAC ReaderID of the latest successful transaction. */
	);

/**
 * end of group phalMfIdentity_TransactionnManagement
 * @}
 */




/* MIFARE Identity ISO7816-4 commands. ----------------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfIdentity_ISO7816 Commands_ISO7816
 * \brief These Components implement the MIFARE Identity ISO/IEC 7816-4 Basic commands.
 * @{
 */

/** \name Options for ISOSelectFile command. */
/** @{ */
#define PHAL_MFIDENTITY_FCI_RETURNED									0x00U	/**< Option to indicate the return of FCI. */
#define PHAL_MFIDENTITY_FCI_NOT_RETURNED								0x0CU	/**< Option to indicate the no return of FCI. */
#define PHAL_MFIDENTITY_SELECTOR_0										0x00U	/**< Option to indicate Selection by 2 byte file Id. */
#define PHAL_MFIDENTITY_SELECTOR_1										0x01U	/**< Option to indicate Selection by child DF. */
#define PHAL_MFIDENTITY_SELECTOR_2										0x02U	/**< Option to indicate Select EF under current DF. Fid = EF id. */
#define PHAL_MFIDENTITY_SELECTOR_3										0x03U	/**< Option to indicate Select parent DF of the current DF. */
#define PHAL_MFIDENTITY_SELECTOR_4										0x04U	/**< Option to indicate Selection by DF Name. DFName and len is then valid. */
/** @} */

/**
 * \brief ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */

phStatus_t phalMfIdentity_IsoSelectFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Option for return / no return of FCI.
																				 *			\arg #PHAL_MFIDENTITY_FCI_RETURNED
																				 *			\arg #PHAL_MFIDENTITY_FCI_NOT_RETURNED
																				 */
		uint8_t bSelector,														/**< [In] The selector to be used.
																				 *			\arg #PHAL_MFIDENTITY_SELECTOR_0
																				 *			\arg #PHAL_MFIDENTITY_SELECTOR_1
																				 *			\arg #PHAL_MFIDENTITY_SELECTOR_2
																				 *			\arg #PHAL_MFIDENTITY_SELECTOR_3
																				 *			\arg #PHAL_MFIDENTITY_SELECTOR_4
																				 */
		uint8_t * pFid,															/**< [In] The ISO File number to be selected. */
		uint8_t * pDFname,														/**< [In] The ISO DFName to be selected. Valid only when bOption = 0x04. */
		uint8_t bDFnameLen,														/**< [In] Length of bytes available in DFname buffer. */
		uint8_t bExtendedLenApdu,												/**< [In] Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or
																				 *		  any other value for Short APDUs.
																				 */
		uint8_t ** ppFCI,														/**< [Out] The FCI information returned by the PICC. */
		uint16_t * pwFCILen														/**< [Out] Length of bytes available in FCI buffer. */
	);

/**
 * end of group phalMfIdentity_ISO7816
 * @}
 */




/* MIFARE Identity Originality Check functions. -------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfIdentity_OriginalityCheck Commands_OriginalityCheck
 * \brief These Components implement the MIFARE Identity Originality Check functions.
 * @{
 */

/**
 * \brief Performs the originality check to verify the genuineness of chip
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_ReadSign(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bAddr,															/**< [In] Value is always 00. Present for forward compatibility reasons */
		uint8_t ** pSignature													/**< [Out] The plain 56 bytes originality signature of the PICC. */
	);

/**
 * end of group phalMfIdentity_OriginalityCheck
 * @}
 */




/* MIFARE Identity Miscellaneous functions. ------------------------------------------------------------------------------------------ */
/**
 * \defgroup phalMfIdentity_Miscellaneous Commands_Miscellaneous
 * \brief These Components implement the MIFARE DESFire Light Miscellaneous functions. These are not part of actual Identity data sheet
 * rather its for internal purpose.
 * @{
 */

/** \name The configuration to be used for SetConfig / GetConfig. */
/** @{ */
#define PHAL_MFIDENTITY_ADDITIONAL_INFO										0x00A1U	/**< Option for getconfig to get additional info of a generic error. */
#define PHAL_MFIDENTITY_WRAPPED_MODE										0x00A2U	/**< Option to get / set current status of command wrapping in ISO 7816-4 APDUs. */
#define PHAL_MFIDENTITY_SHORT_LENGTH_APDU									0x00A3U	/**< Option to get / set Short Length APDU wrapping in ISO 7816-4 APDUs. */
#define PHAL_MFIDENTITY_FILE_OPTION											0x00A4U	/**< The settings of the File. */
#define PHAL_MFIDENTITY_SDM_OPTION											0x00A5U	/**< The SDM settings of the file. */
#define PHAL_MFIDENTITY_KEY_INFO											0x00A6U	/**< Key Information. This should be used to set Key number and version for decryption the SDMENCFileData. */
#define PHAL_MFIDENTITY_SDM_ENC_OFFSET										0x00A7U	/**< SDM ENC offset. This should be set to identify the start position of SDM Encrypted file data. */
#define PHAL_MFIDENTITY_SDM_ENC_LENGTH										0x00A8U	/**< SDM ENC Length. This should be set for the total encrypted data available. */
#define PHAL_MFIDENTITY_SDM_READ_CTR										0x00A9U	/**< SDM Read Counter information. */
#define PHAL_MFIDENTITY_SDM_KEY_TYPE										0x00AAU	/**< Distinguish between the LRP_AES or AES key. The values to be used for this configuration are
																					 *		\c AES128		: 0x00
																					 *		\c LRP_AES128	: 0x01
																					 */
#define PHAL_MFIDENTITY_RAM_KEY_INFO										0x00ABU	/**< Key Information. This should be used to set Key number and version for decryption the SDMENCFileData. */
/** @} */

/**
 * \brief Perform a GetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_GetConfig(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wConfig,														/**< [In] Configuration to read. Will be one of the below values.
																				 *			\arg #PHAL_MFIDENTITY_ADDITIONAL_INFO
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION
																				 *			\arg #PHAL_MFIDENTITY_SDM_OPTION
																				 *			\arg #PHAL_MFIDENTITY_KEY_INFO
																				 *			\arg #PHAL_MFIDENTITY_SDM_ENC_OFFSET
																				 *			\arg #PHAL_MFIDENTITY_SDM_ENC_LENGTH
																				 *			\arg #PHAL_MFIDENTITY_SDM_READ_CTR
																				 *			\arg #PHAL_MFIDENTITY_SDM_KEY_TYPE
																				 *			\arg #PHAL_MFIDENTITY_RAM_KEY_INFO
																				 *		  \n
																				 *		  Support for Software mode only including the above ones.
																				 *			\arg #PHAL_MFIDENTITY_WRAPPED_MODE
																				 *			\arg #PHAL_MFIDENTITY_SHORT_LENGTH_APDU
																				 */
		uint8_t * pValue														/**< [Out] The value for the mentioned configuration. */
	);

/**
 * \brief Perform a SetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_SetConfig(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wConfig,														/**< [In] Configuration to set. Will be one of the below values.
																				 *			\arg #PHAL_MFIDENTITY_ADDITIONAL_INFO
																				 *			\arg #PHAL_MFIDENTITY_FILE_OPTION
																				 *			\arg #PHAL_MFIDENTITY_SDM_OPTION
																				 *			\arg #PHAL_MFIDENTITY_KEY_INFO
																				 *			\arg #PHAL_MFIDENTITY_SDM_ENC_OFFSET
																				 *			\arg #PHAL_MFIDENTITY_SDM_ENC_LENGTH
																				 *			\arg #PHAL_MFIDENTITY_SDM_READ_CTR
																				 *			\arg #PHAL_MFIDENTITY_SDM_KEY_TYPE
																				 *			\arg #PHAL_MFIDENTITY_RAM_KEY_INFO
																				 *		  \n
																				 *		  Support for Software mode only including the above ones.
																				 *			\arg #PHAL_MFIDENTITY_WRAPPED_MODE
																				 *			\arg #PHAL_MFIDENTITY_SHORT_LENGTH_APDU
																				 */
		uint8_t * pValue														/**< [In] The value for the mentioned configuration. */
	);

/**
 * \brief Reset the authentication
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_ResetAuthentication(
		void * pDataParams														/**< [In] Pointer to this layer's parameter structure. */
	);

#ifdef NXPBUILD__PH_CRYPTOSYM
/**
 * \brief Computed the Transaction MAC input as performed on the PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_CalculateTMV(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,														/**< [In] Diversification option. 0xFFFF is for No Diversification. */
		uint16_t wKeyNoTMACKey,													/**< [In] Key number in key store of Software. */
		uint16_t wKeyVerTMACKey,												/**< [In] Key version in key store of Software. */
		uint16_t wRamKeyNo,														/**< [In] Key number of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint16_t wRamKeyVer,													/**< [In] Key version of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint8_t * pDivInput,													/**< [In] Diversification input to diversify TMACKey. */
		uint8_t bDivInputLen,													/**< [In] Length of byte available in DivInput buffer. */
		uint8_t * pTMC,															/**< [In] 4 bytes Transaction MAC Counter. It should be 1 time subtracted from
																				 *		  the actual value and should be LSB first.
																				 */
		uint8_t * pUid,															/**< [In] UID of the card. */
		uint8_t bUidLen,														/**< [In] Length of UID supplied. */
		uint8_t * pTMI,															/**< [In] Transaction MAC Input. */
		uint32_t wTMILen,														/**< [In] Length of bytes available in TMI buffer. */
		uint8_t * pTMV															/**< [Out] The computed Transaction MAC Value. */
	);

/**
 * \brief Decrypted the reader ID as performed on the PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_DecryptReaderID(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,														/**< [In] Diversification option. 0xFFFF is for No Diversification. */
		uint16_t wKeyNoTMACKey,													/**< [In] Key number in key store of Software. */
		uint16_t wKeyVerTMACKey,												/**< [In] Key version in key store of Software. */
		uint16_t wRamKeyNo,														/**< [In] Key number of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint16_t wRamKeyVer,													/**< [In] Key version of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint8_t * pDivInput,													/**< [In] Diversification input to diversify TMACKey. */
		uint8_t bDivInputLen,													/**< [In] Length of byte available in DivInput buffer. */
		uint8_t * pTMC,															/**< [In] 4 bytes Transaction MAC Counter. It should be 1 time subtracted from
																				 *		  the actual value and should be LSB first.
																				 */
		uint8_t * pUid,															/**< [In] UID of the card. */
		uint8_t bUidLen,														/**< [In] Length of UID supplied. */
		uint8_t * pEncTMRI,														/**< [In] Encrypted Transaction MAC ReaderID of the latest successful transaction. */
		uint8_t * pTMRIPrev														/**< [Out] Decrypted Reader ID of the last successful transaction. */
	);

/**
 * \brief Calculate MAC for SDM information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_CalculateMACSDM(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,														/**< [In] Diversification option. 0xFFFF is for No Diversification. */
		uint8_t bSdmOption,														/**< [In] SDM Option to indicate which parameters to be considered. Can be ORed.
																				 *			\arg #PHAL_MFIDENTITY_VCUID_PRESENT
																				 *			\arg #PHAL_MFIDENTITY_RDCTR_PRESENT
																				 */
		uint16_t wSDMMacKeyNo,													/**< [In] Key number in Software or SAM keystore. */
		uint16_t wSDMMacKeyVer,    												/**< [In] Key version in Software or SAM keystore. */
		uint16_t wRamKeyNo,														/**< [In] Key number of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint16_t wRamKeyVer,													/**< [In] Key version of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint8_t * pDivInput,													/**< [In] Diversification input to diversify the SDMMACKey. */
		uint8_t bDivInputLen,													/**< [In] Length of byte available in DivInput buffer. */
		uint8_t * pUid,         												/**< [In] UID of the card. */
		uint8_t bUidLen,														/**< [In] Length of bytes available in UID buffer. */
		uint8_t * pSDMReadCtr,	    											/**< [In] SDM Read Counter Input. */
		uint8_t * pInData,														/**< [In] Data read out of PICC from SDMMacInputoffset for ReadLength bytes */
		uint16_t wInDataLen,													/**< [In] Length of bytes available in InData buffer. */
		uint8_t * pRespMac														/**< [Out] The computed MAC information. */
	);

/**
 * \brief Decrypt the SDM File data information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_DecryptSDMENCFileData(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,														/**< [In] Diversification option. 0xFFFF is for No Diversification. */
		uint16_t wEncKeyNo,														/**< [In] Key number in keystore of Software or SAM. */
		uint16_t wEncKeyVer,													/**< [In] Key version in keystore of Software or SAM. */
		uint16_t wRamKeyNo,														/**< [In] Key number of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint16_t wRamKeyVer,													/**< [In] Key version of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint8_t * pDivInput,													/**< [In] Diversification input to diversify the EncKey. */
		uint8_t bDivInputLen,													/**< [In] Length of byte available in DivInput buffer. */
		uint8_t * pUid,         												/**< [In] UID of the card. */
		uint8_t bUidLen,														/**< [In] Length of bytes available in UID buffer. */
		uint8_t * pSDMReadCtr,	    											/**< [In] SDM Read Counter Input. */
		uint8_t * pEncdata,														/**< [In] Input Enciphered data. Caller has to pass only the Enciphered data which is between
																				 *		  SDMEncOffset for SDMEncLength bytes.
																				 */
		uint16_t wEncDataLen,		      										/**< [In] Length of bytes available in Encdata buffer */
		uint8_t * pPlainData													/**< [Out] The decrypted SDM ENC file data. */
	);

/**
 * \brief Decrypt SDM PICC Data
 * 
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfIdentity_DecryptSDMPICCData(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,														/**< [In] Diversification option. 0xFFFF is for No Diversification. */
		uint16_t wKeyNo,														/**< [In] Key number in Software or SAM keystore. */
		uint16_t wKeyVer,														/**< [In] Key version in Software or SAM keystore. */
		uint8_t * pDivInput,													/**< [In] Diversification input to diversify the Key. */
		uint8_t bDivInputLen,													/**< [In] Length of byte available in DivInput buffer. */
		uint8_t * pEncdata,														/**< [In] The Encrypted PICC data. */
		uint16_t wEncDataLen,		      										/**< [In] Length of bytes available in EncData buffer. */
		uint8_t * pPlainData													/**< [Out] The decrypted SDM PICC data. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * end of group phalMfIdentity_Miscellaneous
 * @}
 */

/** 
 * end of froup phalMfIdentity
 * @} 
 */
#endif /* NXPBUILD__PHAL_MFIDENTITY */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHALMFIDENTITY_H */
