/*
 * Copyright 2013, 2017 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */


/** \file
 * Generic MIFARE DESFire(R) EV1 Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  Santosh Araballi: Generated 31. August 2010
 *
 */

#ifndef PHALMFDF_H
#define PHALMFDF_H

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <ph_TypeDefs.h>
#include <ph_RefDefs.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef NXPBUILD__PHAL_MFDF_SW
/***************************************************************************************************************************************/
/* Software Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/

/**
 * \defgroup phalMfdf_Sw Component : Software
 * @{
 */

#define PHAL_MFDF_SW_ID													0x01U	/**< ID for Software MF DesFire layer. */

/** \brief MIFARE DESfire Software parameter structure  */
typedef struct
{
	uint16_t wId;																/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;                                				/**< Pointer to the parameter structure of the palMifare component. */
	void * pKeyStoreDataParams;                                 				/**< Pointer to the parameter structure of the KeyStore layer. */
	void * pCryptoDataParamsEnc;                                				/**< Pointer to the parameter structure of the Crypto layer for encryption. */
	void * pCryptoRngDataParams;                                				/**< Pointer to the parameter structure of the CryptoRng layer. */
	void * pHalDataParams;                                      				/**< Pointer to the HAL parameters structure. */
	uint8_t bSessionKey[24];                                    				/**< Session key for this authentication */
	uint8_t bKeyNo;                                             				/**< key number against which this authentication is done */
	uint8_t bIv[16];                                            				/**< Max size of IV can be 16 bytes */
	uint8_t bAuthMode;                                          				/**< Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA) */
	uint8_t pAid[3];                                            				/**< Aid of the currently selected application */
	uint8_t bCryptoMethod;                                      				/**< DES,3DES, 3K3DES or AES */
	uint8_t bWrappedMode;                                       				/**< Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs. */
	uint16_t wCrc;                                              				/**< 2 Byte CRC initial value in Authenticate mode. */
	uint32_t dwCrc;                                             				/**< 4 Byte CRC initial value in 0x1A, 0xAA mode. */
	uint16_t wAdditionalInfo;                                   				/**< Specific error codes for Desfire generic errors. */
	uint16_t wPayLoadLen;                                       				/**< Amount of data to be read. Required for Enc read to verify CRC. */
	uint8_t bLastBlockBuffer[16];                               				/**< Buffer to store last Block of encrypted data in case of chaining. */
	uint8_t bLastBlockIndex;                                   					/**< Last Block Buffer Index. */
} phalMfdf_Sw_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfdf_Sw_Init(
		phalMfdf_Sw_DataParams_t * pDataParams,									/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,             								/**< [In] Specifies the size of the data parameter structure */
		void * pPalMifareDataParams,            								/**< [In] Pointer to a palMifare component context. */
		void * pKeyStoreDataParams,             								/**< [In] Pointer to Key Store data parameters. */
		void * pCryptoDataParamsEnc,            								/**< [In] Pointer to a Crypto component context for encryption. */
		void * pCryptoRngDataParams,            								/**< [In] Pointer to a CryptoRng component context. */
		void * pHalDataParams                   								/**< [In] Pointer to the HAL parameters structure. */
	);

/**
 * end of group phalMfdf_Sw
 * @}
 */
#endif /* NXPBUILD__PHAL_MFDF_SW */



#ifdef NXPBUILD__PHAL_MFDF_SAM_NONX
/***************************************************************************************************************************************/
/* Sam NonX Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/

/**
 * \defgroup phalMfdf_Sam_NonX Component : Sam_NonX
 * @{
 */

#define PHAL_MFDF_SAMAV2_ID												0x02U	/**< ID for SamAV2 MF DesFire layer. */

/** \brief MIFARE DESfire SAM AV2 NonX parameter structure  */
typedef struct{
	uint16_t wId;																/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;                        						/**< Pointer to the parameter structure of the palMifare component. */
	phhalHw_SamAV2_DataParams_t * pHalSamDataParams;    						/**< Pointer to the parameter structure of the hal of the SAM layer. */
	void * pHalDataParams;                              						/**< Pointer to the HAL layer of the contact less reader. */
	uint8_t bAuthMode;                                  						/**< Auth. Mode 0x0A or Ox1A or OxAA. */
	uint8_t bKeyNo;                                     						/**< Key number against which authenticated. */
	uint8_t bWrappedMode;                               						/**< Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus */
	uint8_t pAid[3];                                    						/**< Currently selected application Id. */
	uint16_t wAdditionalInfo;                           						/**< Specific error codes for Desfire generic errors. */
} phalMfdf_SamAV2_DataParams_t;

/**
 * \brief Initialise SAM AV2 NonX layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfdf_SamAV2_Init(
		phalMfdf_SamAV2_DataParams_t * pDataParams,								/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,                         					/**< [In] Specifies the size of the data parameter structure */
		phhalHw_SamAV2_DataParams_t * pHalSamDataParams,    					/**< [In] Pointer to the HAL parameter structure of the SAM. */
		void * pHalDataParams,                              					/**< [In] Pointer to HAL param structure of contactless reader. */
		void * pPalMifareDataParams                        						/**< [In] Pointer to the parameter structure of the underlying palMIFARE layer. */
	);


#ifdef NXPBUILD__PHAL_MFDF_SAMAV3_NONX
#define PHAL_MFDF_SAMAV3_NONX_ID										0x04U	/**< ID for SamAV3 MF DesFire layer. */

/** \brief MIFARE DESfire SAM AV3 NonX parameter structure  */
typedef struct{
	uint16_t wId;																/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;                        						/**< Pointer to the parameter structure of the palMifare component. */
	phhalHw_SamAV3_DataParams_t * pHalSamDataParams;    						/**< Pointer to the parameter structure of the hal of the SAM layer. */
	uint8_t bAuthMode;                                  						/**< Auth. Mode 0x0A or Ox1A or OxAA. */
	uint8_t bKeyNo;                                     						/**< Key number against which authenticated. */
	uint8_t bWrappedMode;                               						/**< Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus */
	uint8_t pAid[3];                                    						/**< Currently selected application Id. */
	uint16_t wAdditionalInfo;                           						/**< Specific error codes for Desfire generic errors. */
} phalMfdf_SamAV3_NonX_DataParams_t;

/**
 * \brief Initialise SAM AV3 NonX layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfdf_SamAV3_NonX_Init(
		phalMfdf_SamAV3_NonX_DataParams_t * pDataParams,						/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,                         					/**< [In] Specifies the size of the data parameter structure */
		phhalHw_SamAV3_DataParams_t * pHalSamDataParams,    					/**< [In] Pointer to the HAL parameter structure of the SAM. */
		void * pPalMifareDataParams                        						/**< [In] Pointer to the parameter structure of the underlying palMIFARE layer. */
	);

#endif /* NXPBUILD__PHAL_MFDF_SAMAV3_NONX */

/**
 * end of group phalMfdf_Sam_NonX
 * @}
 */
#endif /* NXPBUILD__PHAL_MFDF_SAM_NONX */



#ifdef NXPBUILD__PHAL_MFDF_SAM_X
/***************************************************************************************************************************************/
/* Sam X Dataparams and Initialization Interface.																				       */
/***************************************************************************************************************************************/

/**
 * \defgroup phalMfdf_Sam_X Component : Sam_X
 * @{
 */

#define PHAL_MFDF_SAMAV2_X_ID											0x03U	/**< ID for SamAV2_X MF DesFire layer. */

/** \brief MIFARE DESfire SAM AV2 X parameter structure  */
typedef struct{
	uint16_t wId;																/**< Layer ID for this component, NEVER MODIFY! */
	phhalHw_SamAV2_DataParams_t * pHalSamDataParams;    						/**< Pointer to the parameter structure of the hal of the SAM layer. */
	uint8_t bAuthMode;                                  						/**< Auth Mode. 0x0A or 0x1A or 0xAA. */
	uint8_t bKeyNo;                                     						/**< Key number against which authenticated. */
	uint8_t bWrappedMode;                               						/**< Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus. */
	uint8_t pAid[3];                                    						/**< Currently selected application Id. */
	uint16_t wAdditionalInfo;                           						/**< Specific error codes for Desfire generic errors. */
} phalMfdf_SamAV2_X_DataParams_t;

/**
 * \brief Initialise SAM AV2 X layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfdf_SamAV2_X_Init(
		phalMfdf_SamAV2_X_DataParams_t * pDataParams,							/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,												/**< [In] Specifies the size of the data parameter structure. */
		phhalHw_SamAV2_DataParams_t * pHalSamDataParams							/**< [In] Pointer to the HAL parameter structure of the SAM. */
	);

#ifdef NXPBUILD__PHAL_MFDF_SAMAV3_X
#define PHAL_MFDF_SAMAV3_X_ID											0x05U	/**< ID for SamAV3_X MF DesFire layer. */

/** \brief MIFARE DESfire SAM AV2 X parameter structure  */
typedef struct{
	uint16_t wId;																/**< Layer ID for this component, NEVER MODIFY! */
	phhalHw_SamAV3_DataParams_t * pHalSamDataParams;							/**< Pointer to the parameter structure of the hal of the SAM layer. */
	uint8_t bAuthMode;															/**< Auth Mode. 0x0A or 0x1A or 0xAA. */
	uint8_t bKeyNo;																/**< Key number against which authenticated. */
	uint8_t bWrappedMode;														/**< Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus. */
	uint8_t pAid[3];															/**< Currently selected application Id. */
	uint16_t wAdditionalInfo;													/**< Specific error codes for Desfire generic errors. */
} phalMfdf_SamAV3_X_DataParams_t;

/**
 * \brief Initialise SAM AV3 X layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfdf_SamAV3_X_Init(
		phalMfdf_SamAV3_X_DataParams_t * pDataParams,							/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,												/**< [In] Specifies the size of the data parameter structure. */
		phhalHw_SamAV3_DataParams_t * pHalSamDataParams							/**< [In] Pointer to the HAL parameter structure of the SAM. */
	);

#endif /* NXPBUILD__PHAL_MFDF_SAMAV3_X */

/**
 * end of group phalMfdf_Sam_X
 * @}
 */
#endif /* NXPBUILD__PHAL_MFDF_SAM_X */



/***************************************************************************************************************************************/
/* MIFARE DESFire Generic command declarations.     																				   */
/***************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_MFDF

/** \defgroup phalMfdf MIFARE DESFire (R)
 * \brief These Functions implement the MIFARE DESFire(R) EV1 commands.
 * @{
 */

/** \defgroup phalMfdf_Errors ErrorCodes
 * \brief These component implement the MIFARE DESFire Error codes.
 * @{
 */

/**
 * \defgroup phalMfdf_PICC_Errors PICC ErrorCodes
 * \brief These Components implement the actual PICC error codes.
 * These are the values that will not be returned to the user from the interface in case of error.
 * @{
 */

/** \name MIFARE Desfire PICC Response Codes */
/** @{ */
#define PHAL_MFDF_RESP_OPERATION_OK										0x00U	/**< MF DF Response - Successful operation. */
#define PHAL_MFDF_RESP_NO_CHANGES										0x0CU	/**< MF DF Response - No changes done to backup files. */
#define PHAL_MFDF_RESP_ERR_OUT_OF_EEPROM_ERROR							0x0EU	/**< MF DF Response - Insufficient NV-Memory. */
#define PHAL_MFDF_RESP_ILLEGAL_COMMAND_CODE								0x1CU	/**< MF DF command code not supported. */
#define PHAL_MFDF_RESP_ERR_INTEGRITY_ERROR								0x1EU	/**< MF DF CRC or MAC does not match data padding bytes not valid. */
#define PHAL_MFDF_RESP_NO_SUCH_KEY										0x40U	/**< MF DF Invalid key number specified. */
#define PHAL_MFDF_RESP_ERR_LENGTH_ERROR									0x7EU	/**< MF DF Length of command string invalid. */
#define PHAL_MFDF_RESP_PERMISSION_DENIED								0x9DU	/**< MF DF Current configuration/status does not allow the requested command. */
#define PHAL_MFDF_RESP_ERR_PARAMETER_ERROR								0x9EU	/**< MF DF Value of params invalid. */
#define PHAL_MFDF_RESP_APPLICATION_NOT_FOUND							0xA0U	/**< MF DF Requested AID not found on PICC. */
#define PHAL_MFDF_RESP_ERR_APPL_INTEGRITY_ERROR							0xA1U	/**< MF DF Unrecoverable error within application, appln will be disabled. */
#define PHAL_MFDF_RESP_ERR_AUTHENTICATION_ERROR							0xAEU	/**< MF DF Current authentication status does not allow the requested cmd. */
#define PHAL_MFDF_RESP_ADDITIONAL_FRAME									0xAFU	/**< MF DF Additional data frame is expected to be sent. */
#define PHAL_MFDF_RESP_ERR_BOUNDARY_ERROR								0xBEU	/**< MF DF Attempt to read/write data from/to beyond the files/record's limits. */
#define PHAL_MFDF_RESP_ERR_PICC_INTEGRITY								0xC1U	/**< MF DF Unrecoverable error within PICC. PICC will be disabled. */
#define PHAL_MFDF_RESP_ERR_COMMAND_ABORTED								0xCAU	/**< MF DF Previous cmd not fully completed. Not all frames were requested or provided by the PCD. */
#define PHAL_MFDF_RESP_ERR_PIC_DISABLED									0xCDU	/**< MF DF PICC was disabled by an unrecoverable error. */
#define PHAL_MFDF_RESP_ERR_COUNT										0xCEU	/**< MF DF Num. of applns limited to 28. No additional applications possible. */
#define PHAL_MFDF_RESP_ERR_DUPLICATE									0xDEU	/**< MF DF File/Application with same number already exists. */
#define PHAL_MFDF_RESP_ERR_EEPROM										0xEEU	/**< MF DF Could not complete NV-Write operation due to loss of power. */
#define PHAL_MFDF_RESP_ERR_FILE_NOT_FOUND								0xF0U	/**< MF DF Specified file number does not exist. */
#define PHAL_MFDF_RESP_ERR_FILE_INTEGRITY								0xF1U	/**< MF DF Unrecoverable error within file. File will be disabled. */
/** @} */

/** \name MIFARE Desfire PICC ISO7816 Response Codes */
/** @{ */
#define PHAL_MFDF_ISO7816_SUCCESS										0x9000U /**< Correct execution. */
#define PHAL_MFDF_ISO7816_ERR_WRONG_LENGTH								0x6700U /**< Wrong length. */
#define PHAL_MFDF_ISO7816_ERR_INVALID_APPLN								0x6A82U /**< Application / file not found. */
#define PHAL_MFDF_ISO7816_ERR_WRONG_PARAMS								0x6A86U /**< Wrong parameters P1 and/or P2. */
#define PHAL_MFDF_ISO7816_ERR_WRONG_LC									0x6A87U /**< Lc inconsistent with P1/p2. */
#define PHAL_MFDF_ISO7816_ERR_WRONG_LE									0x6C00U /**< Wrong Le. */
#define PHAL_MFDF_ISO7816_ERR_NO_PRECISE_DIAGNOSTICS					0x6F00U /**< No precise diagnostics. */
#define PHAL_MFDF_ISO7816_ERR_EOF_REACHED								0x6282U /**< End of File reached. */
#define PHAL_MFDF_ISO7816_ERR_FILE_ACCESS								0x6982U	/**< File access not allowed. */
#define PHAL_MFDF_ISO7816_ERR_FILE_EMPTY								0x6985U	/**< File empty or access conditions not satisfied. */
#define PHAL_MFDF_ISO7816_ERR_FILE_NOT_FOUND							0x6A82U /**< File not found. */
#define PHAL_MFDF_ISO7816_ERR_MEMORY_FAILURE							0x6581U /**< Memory failure (unsuccessful update). */
#define PHAL_MFDF_ISO7816_ERR_INCORRECT_PARAMS							0x6B00U /**< Wrong parameter p1 or p2. READ RECORDS. */
#define PHAL_MFDF_ISO7816_ERR_WRONG_CLA									0x6E00U /**< Wrong Class byte. */
#define PHAL_MFDF_ISO7816_ERR_UNSUPPORTED_INS							0x6D00U /**< Instruction not supported. */
/** @} */

/**
 * end of group phalMfdf_PICC_Errors
 * @}
 */

/** \defgroup phalMfdf_Cust_Errors Custom ErrorCodes
 * \brief These Components implement the custom error codes mapped to PICC return codes.
 * These are the values that will be returned to the user from the interface in case of error.
 * @{
 */

/** \name Custom Error Codes mapping for PICC erro codes */
/** @{ */
#define PHAL_MFDF_NO_CHANGES						(PH_ERR_CUSTOM_BEGIN + 0)	/**< DESFire Response - No changes done to backup files. */
#define PHAL_MFDF_ERR_OUT_OF_EEPROM_ERROR			(PH_ERR_CUSTOM_BEGIN + 1)	/**< DESFire Response - Insufficient NV-Memory. */
#define PHAL_MFDF_ERR_NO_SUCH_KEY					(PH_ERR_CUSTOM_BEGIN + 2)	/**< DESFire Invalid key number specified. */
#define PHAL_MFDF_ERR_PERMISSION_DENIED				(PH_ERR_CUSTOM_BEGIN + 3)	/**< DESFire Current configuration/status does not allow the requested command. */
#define PHAL_MFDF_ERR_APPLICATION_NOT_FOUND			(PH_ERR_CUSTOM_BEGIN + 4)	/**< DESFire Requested AID not found on PICC. */
#define PHAL_MFDF_ERR_BOUNDARY_ERROR				(PH_ERR_CUSTOM_BEGIN + 5)	/**< DESFire Attempt to read/write data from/to beyond the files/record's limits. */
#define PHAL_MFDF_ERR_COMMAND_ABORTED				(PH_ERR_CUSTOM_BEGIN + 6)	/**< DESFire Previous cmd not fully completed. Not all frames were requested or provided by the PCD. */
#define PHAL_MFDF_ERR_COUNT							(PH_ERR_CUSTOM_BEGIN + 7)	/**< DESFire Num. of applns limited to 28. No additional applications possible. */
#define PHAL_MFDF_ERR_DUPLICATE						(PH_ERR_CUSTOM_BEGIN + 8)	/**< DESFire File/Application with same number already exists. */
#define PHAL_MFDF_ERR_FILE_NOT_FOUND				(PH_ERR_CUSTOM_BEGIN + 9)	/**< DESFire Specified file number does not exist. */
#define PHAL_MFDF_ERR_PICC_CRYPTO					(PH_ERR_CUSTOM_BEGIN + 10)	/**< DESFire Crypto error returned by PICC. */
#define PHAL_MFDF_ERR_PARAMETER_ERROR				(PH_ERR_CUSTOM_BEGIN + 11)	/**< DESFire Parameter value error returned by PICC. */
#define PHAL_MFDF_ERR_DF_GEN_ERROR					(PH_ERR_CUSTOM_BEGIN + 12)	/**< DESFire DesFire Generic error. Check additional Info. */
#define PHAL_MFDF_ERR_DF_7816_GEN_ERROR				(PH_ERR_CUSTOM_BEGIN + 13)	/**< DESFire ISO 7816 Generic error. Check Additional Info. */
/** @} */

/**
 * end of group phalMfdf_Cust_Errors
 * @}
 */

/**
 * end of group phalMfdf_Errors
 * @}
 */

/** \defgroup phalMfdf_Common_Defs Defines
 * \brief These are common definitions for complete command features of DESFire.
 * @{
 */

/** \name Options for application / file creation commands command. */
/** @{ */
#define PHAL_MFDF_ISO_FILE_INFO_NOT_AVAILABLE							0x00U	/**< Option to indicate no ISO File ID or ISODFName are present. */
#define PHAL_MFDF_ISO_FILE_ID_AVAILABLE									0x01U	/**< Option to indicate the presence of ISO FileID. */
#define PHAL_MFDF_ISO_DF_NAME_AVAILABLE									0x02U	/**< Option to indicate the presence of ISO DFName. */
#define PHAL_MFDF_ISO_FILE_ID_DF_NAME_AVAILABLE							0x03U	/**< Option to indicate the presence of both ISO FileID and ISO DFName. */
/** @} */

/** \name The communication mode to be used. */
/** @{ */
#define PHAL_MFDF_COMMUNICATION_PLAIN									0x00U	/**< Plain mode of communication. */
#ifdef NXPBUILD__PHAL_MFDF_NDA
#define PHAL_MFDF_COMMUNICATION_MACD									0x10U	/**< MAC mode of communication. */
#define PHAL_MFDF_COMMUNICATION_ENC										0x30U	/**< Enciphered mode of communication. */
#endif /* NXPBUILD__PHAL_MFDF_NDA */
/** @} */

/**
 * end of group phalMfdf_Common_Defs
 * @}
 */

/* MIFARE DESFire security related commands. ----------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfdf_Security Commands_Security
 * \brief These Components implement the MIFARE DESFire security related commands.
 * @{
 */
#ifdef NXPBUILD__PHAL_MFDF_NDA

/** \defgroup phalMfdf_Defs_Div Defines
 * \brief These are common diversification definitions for ChangeKey and Authenticate.
 * @{
 */

/** \name Diversification options to be used with ChangeKey */
/** @{ */
#define PHAL_MFDF_NO_DIVERSIFICATION            						0xFFFFU	/**< No diversification. */
#define PHAL_MFDF_CHGKEY_DIV_NEW_KEY            						0x0002U	/**< Bit 1. Indicating diversification of new key requred. */
#define PHAL_MFDF_CHGKEY_DIV_OLD_KEY            						0x0004U	/**< Bit 2 indicating old key was diversified. */
#define PHAL_MFDF_CHGKEY_DIV_NEW_KEY_ONERND     						0x0008U	/**< Bit 3 indicating new key diversification using one rnd. Default is two rounds. */
#define PHAL_MFDF_CHGKEY_DIV_OLD_KEY_ONERND     						0x0010U	/**< Bit 4 indicating old key diversification using one rnd. Default is two rounds. */
#define PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC								0x0020U	/**< Bit 5 indicating key diversification method based on CMAC. Default is Encryption method */
/** @} */

/** \name Diversification options to be used with Authenticate */
/** @{ */
#define PHAL_MFDF_DIV_METHOD_ENCR			PH_CRYPTOSYM_DIV_MODE_DESFIRE		/**< Encryption based method of diversification. */
#define PHAL_MFDF_DIV_METHOD_CMAC			PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS	/**< CMAC based method of diversification. */
#define PHAL_MFDF_DIV_OPTION_2K3DES_FULL	PH_CRYPTOSYM_DIV_OPTION_2K3DES_FULL	/**< Encryption based method, full key diversification. */
#define PHAL_MFDF_DIV_OPTION_2K3DES_HALF	PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF	/**< Encryption based method, half key diversification. */
/** @} */

/**
 * end of group phalMfdf_Defs_Div
 * @}
 */

/**
 * \brief Performs an Authentication with the specified key number. This command supports the backward compatiable D40 authentication.
 * The command can be used with DES and 2K3DES keys and performs DESFire4 native authentication.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_Authenticate(
		void *pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,      													/**< [In] One of the below options.
																				 *			\arg #PHAL_MFDF_NO_DIVERSIFICATION
																				 *			\arg #PHAL_MFDF_DIV_METHOD_ENCR
																				 *			\arg #PHAL_MFDF_DIV_METHOD_ENCR | #PHAL_MFDF_DIV_OPTION_2K3DES_HALF
																				 *			\arg #PHAL_MFDF_DIV_METHOD_ENCR | #PHAL_MFDF_DIV_OPTION_2K3DES_FULL
																				 *			\arg #PHAL_MFDF_DIV_METHOD_CMAC
																				 */
		uint16_t wKeyNo,       													/**< [In] Key number in keystore to authenticate with. */
		uint16_t wKeyVer,      													/**< [In] Key version in the key store. */
		uint8_t bKeyNoCard,    													/**< [In] Key number on card. */
		uint8_t * pDivInput,   													/**< [In] Diversification input. Can be NULL. */
		uint8_t bDivLen        													/**< [In] Length of diversification input max 31B. */
	);

/**
 * \brief Performs an DES Authentication in ISO CBS send mode. The command can be used with DES,3DES and 3K3DES keys.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_AuthenticateISO(
		void *pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,      													/**< [In] One of the below options.
																				 *			\arg #PHAL_MFDF_NO_DIVERSIFICATION
																				 *			\arg #PHAL_MFDF_DIV_METHOD_ENCR
																				 *			\arg #PHAL_MFDF_DIV_METHOD_ENCR | #PHAL_MFDF_DIV_OPTION_2K3DES_HALF
																				 *			\arg #PHAL_MFDF_DIV_METHOD_ENCR | #PHAL_MFDF_DIV_OPTION_2K3DES_FULL
																				 *			\arg #PHAL_MFDF_DIV_METHOD_CMAC
																				 */
		uint16_t wKeyNo,       													/**< [In] Key number in keystore to authenticate with. */
		uint16_t wKeyVer,      													/**< [In] Key version in the key store. */
		uint8_t bKeyNoCard,    													/**< [In] Key number on card. */
		uint8_t * pDivInput,   													/**< [In] Diversification input. Can be NULL. */
		uint8_t bDivLen        													/**< [In] Length of diversification input max 31B. */
	);

/**
 * \brief Performs an AES Authentication. The command should be used with AES128 keys.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_AuthenticateAES(
		void *pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,      													/**< [In] One of the below options.
																				 *			\arg #PHAL_MFDF_NO_DIVERSIFICATION
																				 *			\arg #PHAL_MFDF_DIV_METHOD_ENCR
																				 *			\arg #PHAL_MFDF_DIV_METHOD_CMAC
																				 */
		uint16_t wKeyNo,       													/**< [In] Key number in keystore to authenticate with. */
		uint16_t wKeyVer,      													/**< [In] Key version in the key store. */
		uint8_t bKeyNoCard,    													/**< [In] Key number on card. */
		uint8_t * pDivInput,   													/**< [In] Diversification input. Can be NULL. */
		uint8_t bDivLen        													/**< [In] Length of diversification input max 31B. */
	);


/**
 * \brief Changes the master key settings on PICC and application level.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_ChangeKeySettings(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bKeySettings													/**< [In] New key settings to be changed. */
	);
#endif /* NXPBUILD__PHAL_MFDF_NDA */

/**
 * \brief Gets information on the PICC and application master key settings.
 *
 * \note
 * In addition it returns the maximum number of keys which are configured for the selected application.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetKeySettings(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pKeySettings													/**< [Out] Key setting of the currently selected application identifier. */
	);

#ifdef NXPBUILD__PHAL_MFDF_NDA
/**
 * \brief Changes any key on the PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_ChangeKey(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,      													/**< [In] One of the below options.
																				 *			\arg #PHAL_MFDF_NO_DIVERSIFICATION
																				 *			\arg #PHAL_MFDF_CHGKEY_DIV_NEW_KEY | #PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC
																				 *			\arg #PHAL_MFDF_CHGKEY_DIV_OLD_KEY | #PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC
																				 *			\arg #PHAL_MFDF_CHGKEY_DIV_NEW_KEY | #PHAL_MFDF_CHGKEY_DIV_NEW_KEY_ONERND
																				 *			\arg #PHAL_MFDF_CHGKEY_DIV_OLD_KEY | #PHAL_MFDF_CHGKEY_DIV_OLD_KEY_ONERND
																				 *			\arg #PHAL_MFDF_CHGKEY_DIV_NEW_KEY | #PHAL_MFDF_CHGKEY_DIV_OLD_KEY
																				 *			\arg #PHAL_MFDF_CHGKEY_DIV_NEW_KEY | #PHAL_MFDF_CHGKEY_DIV_OLD_KEY |
																				 *				 #PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC
																				 *			\arg #PHAL_MFDF_CHGKEY_DIV_NEW_KEY | #PHAL_MFDF_CHGKEY_DIV_OLD_KEY |
																				 *				 #PHAL_MFDF_CHGKEY_DIV_NEW_KEY_ONERND | #PHAL_MFDF_CHGKEY_DIV_OLD_KEY_ONERND
																				 */
		uint16_t wOldKeyNo,														/**< [In] Old key number in keystore of software or SAM. */
		uint16_t wOldKeyVer,    												/**< [In] Old key number in keystore of software or SAM. */
		uint16_t wNewKeyNo,     												/**< [In] New key number in keystore of software or SAM. */
		uint16_t wNewKeyVer,    												/**< [In] New key number in keystore of software or SAM. */
		uint8_t bKeyNoCard,     												/**< [In] Key number on card. */
		uint8_t * pDivInput,    												/**< [In] Diversification input. Can be NULL. */
		uint8_t bDivLen         												/**< [In] Length of diversification input max 31B. */
	);
#endif /* NXPBUILD__PHAL_MFDF_NDA */

/**
 * \brief Reads out the current key version of any key stored on the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetKeyVersion(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bKeyNo,															/**< [In] Key number on card. */
		uint8_t * pKeyVersion													/**< [Out] The version of the specified key. */
);

/**
 * end of group phalMfdf_Security
 * @}
 */




/* MIFARE DESFire PICC level commands. ----------------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfdf_PICC_Level Commands_PiccLevel
 * \brief These Components implement the MIFARE DESFire PICC level commands.
 * @{
 */

/**
 * \brief Creates new applications on the PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_CreateApplication(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Option to represent the present of ISO information.
																				 *			\arg #PHAL_MFDF_ISO_FILE_INFO_NOT_AVAILABLE
																				 *			\arg #PHAL_MFDF_ISO_FILE_ID_AVAILABLE
																				 *			\arg #PHAL_MFDF_ISO_DF_NAME_AVAILABLE
																				 *			\arg #PHAL_MFDF_ISO_FILE_ID_DF_NAME_AVAILABLE
																				 */
		uint8_t * pAid,															/**< [In] The application identifier to be used. Will be of 3 bytes with LSB first.
																				 *			\c If application 01 need to be created then the Aid will be 01 00 00.
																				 */
		uint8_t bKeySettings1,													/**< [In] Application Key settings. */
		uint8_t bKeySettings2,													/**< [In] Additional key settings.  */
		uint8_t * pISOFileId,													/**< [In] ISO File ID to be used. Will be two bytes. */
		uint8_t * pISODFName,													/**< [In] ISO DF Name to be used. Will be 16 bytes and can also be NULL based on the option. */
		uint8_t bISODFNameLen													/**< [In] Length of bytes available in ISODFName buffer. */
	);

#ifdef NXPBUILD__PHAL_MFDF_NDA
/**
 * \brief Permanently deletes the applications on the PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_DeleteApplication(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pAid															/**< [In] The application identifier to be used. Will be of 3 bytes with LSB first. \n
																				 *			\c If application 01 need to be deleted then the Aid will be 01 00 00.
																				 */
	);

/**
 * \brief Returns application identifiers of all applications on the PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetApplicationIDs(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pAidBuff,														/**< [Out] The available identifiers of the application(s). */
		uint8_t * pNumAid														/**< [Out] The number of available application(s). */
	);

/**
 * \brief Returns the Dedicated File(DF) names.
 *
 * \remarks
 * The pDFBuffer will be filled with 3 byte AID + 2 byte ISO Fid + one DF Name at a time.If there are more DFs, then status PH_ERR_SUCCESS_CHAINING
 * is returned. The caller should call this again with bOption = PH_EXCHANGE_RXCHAINING.
 *
 * CAUTION: This should not be called with AES or ISO authentication. DOING SO WILL DAMAGE THE DESFIRE Card
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING more DF Names to be returned.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetDFNames(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] One of the below options.
																				 *			\arg #PH_EXCHANGE_DEFAULT   : To exchange the initial command to the PICC.
																				 *			\arg #PH_EXCHANGE_RXCHAINING: To exchange the chaining command to PICC in
																				 *										  case PICC returns AF as the status.
																				 */
		uint8_t * pDFBuffer,													/**< [Out] The ISO information about the application. At a time the information of one application
																				 *		   will be returned. The buffer should be 21 (3 + 2 + 16) bytes long.
																				 */
		uint8_t * bDFInfoLen													/**< [Out] Length of bytes available in DFBuffer buffer. */
	);
#endif /* NXPBUILD__PHAL_MFDF_NDA */

/**
 * \brief Selects one particular application on the PICC for further access
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_SelectApplication(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pAid															/**< [In] The application identifier to be used. Will be of 3 bytes with LSB first. \n
																				 *			\c If application 01 need to be selected then the Aid will be 01 00 00.
																				 */
	);

#ifdef NXPBUILD__PHAL_MFDF_NDA
/**
 * \brief Releases the PICC user memory
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_FormatPICC(
		void * pDataParams														/**< [In] Pointer to this layer's parameter structure. */
	);
#endif /* NXPBUILD__PHAL_MFDF_NDA */

/**
 * \brief Returns manufacturing related data of the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetVersion(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pVerInfo														/**< [Out] The version information of the product. */
	);

/**
 * \brief Returns free memory available on the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_FreeMem(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pMemInfo														/**< [Out] Current free memory available. Will be of 3 bytes with LSB first. */
	);

#ifdef NXPBUILD__PHAL_MFDF_NDA

/** \defgroup phalMfdf_Defs_PiccLevel Defines
 * \brief These are definitions for \ref phalMfdf_SetConfiguration.
 * @{
 */
#define PHAL_MFDF_SET_CONFIG_OPTION1									0x00U	/**< Option 0 for updating the PICC Configuration. */
#define PHAL_MFDF_SET_CONFIG_OPTION2									0x01U	/**< Option 1 for updating the Default Keys. */
#define PHAL_MFDF_SET_CONFIG_OPTION3									0x02U	/**< Option 2 for updating the ATS. */
/**
 * end of group phalMfdf_Defs_PiccLevel
 * @}
 */

/**
 * \brief Configures the card and pre personalizes the card with a key, defines if the UID or the random ID is sent back during
 * communication setup and configures the ATS string.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_SetConfiguration(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Configuration Option. Define length and content of the Data parameter.
																				 *			\arg #PHAL_MFDF_SET_CONFIG_OPTION1
																				 *			\arg #PHAL_MFDF_SET_CONFIG_OPTION2
																				 *			\arg #PHAL_MFDF_SET_CONFIG_OPTION3
																				 */
		uint8_t * pData,														/**< [In] Data for the option specified.*/
		uint8_t bDataLen														/**< [In] Length of bytes available in Data buffer. */
	);

/**
 * \brief Returns the Unique ID of the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetCardUID(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pUid															/**< [Out] The complete UID of the PICC. */
	);
#endif /* NXPBUILD__PHAL_MFDF_NDA */

/**
 * end of group phalMfdf_PICC_Level
 * @}
 */




/* MIFARE DESFire Application level commands. ---------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfdf_AppLevel Commands_ApplicationLevel
 * \brief These Components implement the MIFARE DESFire Application level commands.
 * @{
 */

/**
 * \brief Returns the file IDs of all active files within the currently selected application
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetFileIDs(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pFid,															/**< [Out] The buffer containing the available file ID(s). Buffer should be 32 bytes. */
		uint8_t * bNumFid														/**< [Out] The length of bytes available in Fid buffer. */
	);

/**
 * \brief Get the ISO File IDs.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetISOFileIDs(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pFidBuffer,													/**< [Out] The buffer containing the available ISO file ID(s). Buffer should be 64 bytes. */
		uint8_t * bNumFid														/**< [Out] The number of ISO File ID's read. */
	);

/**
 * \brief Get informtion on the properties of a specific file
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetFileSettings(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo,														/**< [In] The file number for which the setting to be retrieved. */
		uint8_t * pFSBuffer,													/**< [Out] The buffer containing the settings. The buffer should 17 bytes. */
		uint8_t * bBufferLen													/**< [Out] The length of bytes available in FSBuffer buffer. */
	);

#ifdef NXPBUILD__PHAL_MFDF_NDA
/**
 * \brief Changes the access parameters of an existing file.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_ChangeFileSettings(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Indicates whether the settings to be sent enciphered or plain.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 */
		uint8_t bFileNo,														/**< [In] File number for which the setting need to be updated. */
		uint8_t bCommSett,														/**< [In] New communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 */
		uint8_t * pAccessRights													/**< [In] The new access right to be applied for the file. Should be 2 byte. */
	);
#endif /* NXPBUILD__PHAL_MFDF_NDA */

/**
* \brief Creates files for storage of plain unformatted user data within an existing application on the PICC.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phalMfdf_CreateStdDataFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Option to represent the present of ISO information.
																				 *			\arg #PHAL_MFDF_ISO_FILE_INFO_NOT_AVAILABLE
																				 *			\arg #PHAL_MFDF_ISO_FILE_ID_AVAILABLE
																				 */
		uint8_t bFileNo,														/**< [In] The file number to be created. */
		uint8_t * pISOFileId,													/**< [In] The ISO File number to be applied. */
		uint8_t bCommSett,														/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t * pAccessRights,												/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t * pFileSize														/**< [In] The size of the file. Will be of 3 bytes with LSB first.
																				 *			\c If size 0x10 need to be created then the FileSize will be 10 00 00.
																				 */
	);

/**
 * \brief Creates files for the storage of plain unformatted user data within an existing application, additionally supporting the
 * feature of an integrated backup mechanism.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_CreateBackupDataFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Option to represent the present of ISO information.
																				 *			\arg #PHAL_MFDF_ISO_FILE_INFO_NOT_AVAILABLE
																				 *			\arg #PHAL_MFDF_ISO_FILE_ID_AVAILABLE
																				 */
		uint8_t bFileNo,														/**< [In] The file number to be created. */
		uint8_t * pISOFileId,													/**< [In] The ISO File number to be applied. */
		uint8_t bCommSett,														/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t * pAccessRights,												/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t * pFileSize														/**< [In] The size of the file. Will be of 3 bytes with LSB first.
																				 *			\c If size 0x10 need to be created then the FileSize will be 10 00 00.
																				 */
	);

/** \defgroup phalMfdf_Defs_AppLevel Defines
 * \brief These are definitions for \ref phalMfdf_CreateValueFile.
 * @{
 */
#define PHAL_MFDF_ENABLE_LIMITEDCREDIT									0x01U	/**< Bit 0 set to 1 to enable Limited credit. */
#define PHAL_MFDF_ENABLE_FREE_GETVALUE									0x02U	/**< Bit 1 set to 1 to enable free getvalue. */
/**
 * end of group phalMfdf_Defs_AppLevel
 * @}
 */

/**
* \brief Creates files for the storage and manipulation of 32bit signed integer values within an existing application on the PICC.
* User provides the entire information in the ValInfo buffer.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phalMfdf_CreateValueFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo,														/**< [In] The file number to be created. */
		uint8_t bCommSett,														/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t * pAccessRights,												/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t * pLowerLmit,													/**< [In] The lower limit for the file. Will be of 4 bytes with LSB first.
																				 *			\c If size 0x10 need to be created then the value will be 10 00 00 00.
																				 */
		uint8_t * pUpperLmit,													/**< [In] The upper limit for the file. Will be of 4 bytes with LSB first.
																				 *			\c If size 0x20 need to be created then the value will be 20 00 00 00.
																				 */
		uint8_t * pValue,														/**< [In] The initial value. Will be of 4 bytes with LSB first.
																				 *			\c If size 0x10 need to be created then the value will be 10 00 00 00.
																				 */
		uint8_t bLimitedCredit													/**< [In] One of the below values.
																				 *			\arg #PHAL_MFDF_ENABLE_LIMITEDCREDIT
																				 *			\arg #PHAL_MFDF_ENABLE_FREE_GETVALUE
																				 */
	);

/**
 * \brief Creates files for multiple storage of structural similar data, for example for layalty programs within an existing application.
 * Once the file is filled, further writing is not possible unless it is cleared.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_CreateLinearRecordFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Option to represent the present of ISO information.
																				 *			\arg #PHAL_MFDF_ISO_FILE_INFO_NOT_AVAILABLE
																				 *			\arg #PHAL_MFDF_ISO_FILE_ID_AVAILABLE
																				 */
		uint8_t bFileNo,														/**< [In] The file number to be created. */
		uint8_t * pISOFileId,													/**< [In] The ISO File number to be applied. */
		uint8_t bCommSett,														/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t * pAccessRights,												/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t * pRecordSize,													/**< [In] The size of the file. Will be of 3 bytes with LSB first.
																				 *			\c If size 0x10 need to be created then the RecordSize will be 10 00 00.
																				 */
		uint8_t * pMaxNoOfRec													/**< [In] The maximum number of record in the file. Will be of 3 bytes with LSB first.
																				 *			\c If size 0x04 need to be created then the value will be 04 00 00.
																				 */
	);

/**
 * \brief Creates files for multiple storage of structural similar data, for example for logging transactions, within an existing application.
 * Once the file is filled, the PICC automatically overwrites the oldest record with the latest written one.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_CreateCyclicRecordFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Option to represent the present of ISO information.
																				 *			\arg #PHAL_MFDF_ISO_FILE_INFO_NOT_AVAILABLE
																				 *			\arg #PHAL_MFDF_ISO_FILE_ID_AVAILABLE
																				 */
		uint8_t bFileNo,														/**< [In] The file number to be created. */
		uint8_t * pISOFileId,													/**< [In] The ISO File number to be applied. */
		uint8_t bCommSett,														/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t * pAccessRights,												/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t * pRecordSize,													/**< [In] The size of the file. Will be of 3 bytes with LSB first.
																				 *			\c If size 0x10 need to be created then the RecordSize will be 10 00 00.
																				 */
		uint8_t * pMaxNoOfRec													/**< [In] The maximum number of record in the file. Will be of 3 bytes with LSB first.
																				 *			\c If size 0x04 need to be created then the value will be 04 00 00.
																				 */
	);

/**
 * \brief Permanently deactivates a file within the file directory of the currently selected application.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_DeleteFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo															/**< [In] The file number to be deleted. */
	);

/**
 * end of group phalMfdf_AppLevel
 * @}
 */




/* MIFARE DESFire Data Manipulation commands. ---------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfdf_Data Commands_DataManipulation
 * \brief These Components implement the MIFARE DESFire Data Manipulation commands.
 * @{
 */

/**
 * \brief Reads data from standard data files or backup data files.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING indicating more data to be read.
 * \retval Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_ReadData(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 *
																				 *			\c #PH_EXCHANGE_RXCHAINING: To be Or'd with the above option
																				 *				flag if the PICC returns AF followed by data.
																				 */
		uint8_t bFileNo,														/**< [In] The file number from where the data to be read. */
		uint8_t * pOffset,														/**< [In] The offset from where the data should be read. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 need to be offset then it will be 10 00 00.
																				 */
		uint8_t * pLength,														/**< [In] The number of bytes to be read. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be read then it will be 10 00 00. \n
																				 *			\c If complete file need to be read then it will be 00 00 00.
																				 */
		uint8_t ** ppRxdata,													/**< [Out] The data retuned by the PICC. */
		uint16_t * pRxdataLen													/**< [Out] Length of bytes available in RxData buffer. */
	);

/**
 * \brief Writes data to standard data files or backup data files
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */


phStatus_t phalMfdf_WriteData(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t bFileNo,														/**< [In] The file number to which the data should be written. */
		uint8_t * pOffset,														/**< [In] The offset from where the data should be written. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 need to be offset then it will be 10 00 00.
																				 */
		uint8_t * pTxData,														/**< [In] The data to be written to the PICC. */
		uint8_t * pTxDataLen													/**< [In] The number of bytes to be written. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be written then it will be 10 00 00.
																				 */
	);


/**
 * \brief Reads the currently stored value from value files.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_GetValue(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t bFileNo,														/**< [In] The file number for which the value to the retrieved. */
		uint8_t * pValue														/**< [Out] The value returned by the PICC. The buffer should be 4 bytes. */
	);

/**
 * \brief Increases a value stored in a Value File
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_Credit(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t bFileNo,														/**< [In] The file number to which the value should be credited. */
		uint8_t * pValue														/**< [Out] The value to be credited. Will be of 4 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be credited then it will be 10 00 00 00.
																				 */
	);

/**
 * \brief Decreases a value stored in a Value File
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_Debit(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t bFileNo,														/**< [In] The file number to which the value should be debited. */
		uint8_t * pValue														/**< [Out] The value to be debited. Will be of 4 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be debited then it will be 10 00 00 00.
																				 */
	);

/**
 * \brief Allows a limited increase of a value stored in a Value File without having full credit permissions to the file.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_LimitedCredit(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t bFileNo,														/**< [In] The file number to which the value should be credited. */
		uint8_t * pValue														/**< [Out] The value to be credited. Will be of 4 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be credited then it will be 10 00 00 00.
																				 */
	);

/**
 * \brief Writes data to a record in a Cyclic or Linear Record File.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_WriteRecord(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t bFileNo,														/**< [In] The file number to which the data should be written. */
		uint8_t * pOffset,														/**< [In] The offset from where the data should be written. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 need to be offset then it will be 10 00 00.
																				 */
		uint8_t * pData,														/**< [In] The data to be written to the PICC. */
		uint8_t * pDataLen														/**< [In] The number of bytes to be written. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 bytes need to be written then it will be 10 00 00.
																				 */
	);

/**
 * \brief Reads out a set of complete records from a Cyclic or Linear Record File.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING indicating more data to be read.
 * \retval Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_ReadRecords(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bCommOption,													/**< [In] Communication settings for the file.
																				 *			\arg #PHAL_MFDF_COMMUNICATION_PLAIN
																				 *			\cond NXPBUILD__PHAL_MFDF_NDA
																				 *			\arg #PHAL_MFDF_COMMUNICATION_MACD
																				 *			\arg #PHAL_MFDF_COMMUNICATION_ENC
																				 *			\endcond
																				 */
		uint8_t bFileNo,														/**< [In] The file number from where the data to be read. */
		uint8_t * pOffset,														/**< [In] The offset from where the data should be read. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 need to be offset then it will be 10 00 00.
																				 */
		uint8_t * pNumRec,														/**< [In] The number of records to be read. Will be of 3 bytes with LSB first.
																				 *			\c If 0x10 record need to be read then it will be 10 00 00. \n
																				 *			\c If all records need to be read then it will be 00 00 00.
																				 */
		uint8_t * pRecSize,														/**< [In] The number of bytes to be read. Will be of 3 bytes with LSB first. */
		uint8_t ** ppRxdata,													/**< [Out] The data retuned by the PICC. */
		uint16_t * pRxdataLen													/**< [Out] Length of bytes available in RxData buffer. */
	);

/**
 * \brief Resets a Cyclic or Linear Record File.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_ClearRecordFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo															/**< [In] The file number to be cleared. */
	);

/**
 * \brief Validates all previous write access' on Backup Data files, value files and record files within one application.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_CommitTransaction(
		void * pDataParams														/**< [In] Pointer to this layer's parameter structure. */
	);

/**
 * \brief Invalidates all previous write access' on Backup Data files, value files and record files within one application.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_AbortTransaction(
		void * pDataParams														/**< [In] Pointer to this layer's parameter structure. */
	);

/**
 * end of group phalMfdf_Data
 * @}
 */




/* MIFARE DESFire ISO7816 commands. -------------------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfdf_ISO7816 Commands_ISO7816
 * \brief These Components implement the MIFARE DESFire ISO/IEC 7816-4 Basic inter-industry commands.
 * @{
 */

/** \defgroup phalMfdf_Defs_ISO7816 Defines
 * \brief These are definitions for \ref phalMfdf_IsoSelectFile.
 * @{
 */
#define PHAL_MFDF_FCI_RETURNED											0x00U	/**< Option to indicate the return of FCI. */
#define PHAL_MFDF_FCI_NOT_RETURNED										0x0CU	/**< Option to indicate the no return of FCI. */
#define PHAL_MFDF_SELECTOR_0											0x00U	/**< Option to indicate Selection by 2 byte file Id. */
#define PHAL_MFDF_SELECTOR_2											0x02U	/**< Option to indicate Select EF under current DF. Fid = EF id. */
#define PHAL_MFDF_SELECTOR_4											0x04U	/**< Option to indicate Selection by DF Name. DFName and len is then valid. */
/**
 * end of group phalMfdf_Defs_ISO7816
 * @}
 */

/**
 * \brief ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */

phStatus_t phalMfdf_IsoSelectFile(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,														/**< [In] Option for return / no return of FCI.
																				 *			\arg #PHAL_MFDF_FCI_RETURNED
																				 *			\arg #PHAL_MFDF_FCI_NOT_RETURNED
																				 */
		uint8_t bSelector,														/**< [In] The selector to be used.
																				 *			\arg #PHAL_MFDF_SELECTOR_0
																				 *			\arg #PHAL_MFDF_SELECTOR_2
																				 *			\arg #PHAL_MFDF_SELECTOR_4
																				 */
		uint8_t * pFid,															/**< [In] The ISO File number to be selected. */
		uint8_t * pDFname,														/**< [In] The ISO DFName to be selected. Valid only when bOption = 0x04. */
		uint8_t bDFnameLen,														/**< [In] Length of bytes available in DFname buffer. */
		uint8_t ** ppFCI,														/**< [Out] The FCI information returned by the PICC. */
		uint16_t * pwFCILen														/**< [Out] Length of bytes available in FCI buffer. */
	);

/**
 * \brief ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING operation success with chaining.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_IsoReadBinary(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,														/**< [In] One of the below options.
																				 *			\arg #PH_EXCHANGE_DEFAULT   : To exchange the initial command to the PICC.
																				 *			\arg #PH_EXCHANGE_RXCHAINING: To exchange the chaining command to PICC in
																				 *										  case PICC returns AF as the status.
																				 */
		uint8_t bOffset,														/**< [In] The offset from where the data should be read. */
		uint8_t bSfid,															/**< [In] Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
																				 *		  Else it is treated as MSB of 2Byte offset.
																				 */
		uint8_t bBytesToRead,													/**< [In] Number of bytes to read. If 0, then entire file to be read. */
		uint8_t ** ppRxBuffer,													/**< [Out] The data retuned by the PICC. */
		uint16_t * pBytesRead													/**< [Out] Length of bytes available in RxData buffer. */
	);

/**
 * \brief Iso Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_IsoUpdateBinary(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOffset,														/**< [In] The offset from where the data should be updated. */
		uint8_t bSfid,															/**< [In] Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
																				 *		  Else it is treated as MSB of 2Byte offset.
																				 */
		uint8_t * pData,														/**< [In] Data to be updated. */
		uint8_t bDataLen														/**< [In] Length of bytes available in Data buffer. */
	);

/**
 * \brief Iso Read Records. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_IsoReadRecords(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,														/**< [In] One of the below options.
																				 *			\arg #PH_EXCHANGE_DEFAULT   : To exchange the initial command to the PICC.
																				 *			\arg #PH_EXCHANGE_RXCHAINING: To exchange the chaining command to PICC in
																				 *										  case PICC returns AF as the status.
																				 */
		uint8_t bRecNo,															/**< [In] Record to read / from where to read. */
		uint8_t bReadAllFromP1,													/**< [In] Whether to read all records from P1 or just one. */
		uint8_t bSfid,															/**< [In] Short ISO File Id. */
		uint8_t bBytesToRead,													/**< [In] Number of bytes to read. */
		uint8_t ** ppRxBuffer,													/**< [Out] The data retuned by the PICC. */
		uint16_t * pBytesRead													/**< [Out] Length of bytes available in RxData buffer. */
	);

/**
 * \brief Iso Append record. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_IsoAppendRecord(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bSfid,															/**< [In] Short ISO File Id. */
		uint8_t * pData,														/**< [In] Data to be appended. */
		uint8_t bDataLen														/**< [In] Length of bytes available in Data buffer. */
	);

/**
 * \brief ISOGetChallenge. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \remarks
 * THIS COMMAND IS NOT SUPPORTED IN SAM-X Configuration.
 *
 * Returns the random number from the PICC. Size depends on the key type referred by wKeyNo and wKeyVer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_IsoGetChallenge(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wKeyNo,														/**< [In] Key number to be used form keystore. */
		uint16_t wKeyVer,														/**< [In] Key version to be used from key store. */
		uint8_t bLe,															/**< [In] Length of expected challenge RPICC1. */
		uint8_t * pRPICC1														/**< [Out] RPICC1 returned from PICC. */
	);

#ifdef NXPBUILD__PHAL_MFDF_NDA
/**
 * \brief Iso External Authenticate. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \remarks
 * THIS COMMAND IS NOT SUPPORTED IN SAM-X Configuration.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_IsoExternalAuthenticate(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pInput,														/**< [In] Input data should be as mentioned below.
																				 *			\c Reference to crypto algorigthm - 1 Byte: 00 => context defined, 02 => 2K3DES,
																				 *													    04 => 3k3DES, 09 => AES128 \n
																				 *			\c Card master key flag - 1 Byte:  0x00 if card master key, 0x01 otherwise. \n
																				 *			\c Key number on card - 1 Byte: 0x0 to 0xD \n
																				 *			\c Length of random number: 1 Byte \n
																				 *			\c Random number generated by PCD : 8 or 16 bytes. Not required for Sam non X mode. \n
																				 *			\c Random number returned by GetChallenge command : 8 Bytes or 16 Bytes \n
																				 *			\c Key number : 2 bytes \n
																				 *			\c key Version: 2 bytes
																				 */
		uint8_t bInputLen,														/**< [In] Length of bytes available in Input buffer. */
		uint8_t * pDataOut,														/**< [Out] Returns Rnd number PCD2 in sam non x mode. Nothing in S/W mode. */
		uint8_t * pOutLen														/**< [Out] Length of data returned in pDataOut. */
	);

/**
 * \brief Iso Internal Authenticate. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_IsoInternalAuthenticate(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pInput,														/**< [In] Input data should be as mentioned below.
																				 *			\c Reference to crypto algorigthm - 1 Byte: 00 => context defined, 02 => 2K3DES,
																				 *													    04 => 3k3DES, 09 => AES128 \n
																				 *			\c Card master key flag - 1 Byte:  0x00 if card master key, 0x01 otherwise. \n
																				 *			\c Key number on card - 1 Byte: 0x0 to 0xD \n
																				 *			\c Length of random number: 1 Byte \n
																				 *			\c Random number Rpcd2 : 8 Bytes or 16 Bytes. \n
																				 *			\c Key number : 2 bytes \n
																				 *			\c key Version: 2 bytes
																				 */
		uint8_t bInputLen,														/**< [In] Length of bytes available in Input buffer. */
		uint8_t * pDataOut,														/**< [Out] RRPICC2 || RPCD2 after decryption in S/W mode. Nothing in Sam non x mode. */
		uint8_t * pOutLen														/**< [Out] Length of data available in DataOut buffer. */
	);

/**
 * \brief Perform Iso authentication GetChallenge, External Authenticate & Internal Authenticate of a DESFire PICC
 *
 * Internally performs the three pass Iso authentication by calling
 * GetChallenge \n
 * External Authenticate \n
 * Internal Authenticate \n
 * Generates and stores the session key \n
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_IsoAuthenticate(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wKeyNo,														/**< [In] DESFire key number or SAM Key entry number. */
		uint16_t wKeyVer,														/**< [In] Key version. */
		uint8_t bKeyNoCard,														/**< [In] Key number on card. 0x0 to 0xD. */
		uint8_t bIsPICCkey														/**< [In] Is it PICC Master key ? 1 = YES. */
	);
#endif /* NXPBUILD__PHAL_MFDF_NDA */

/**
 * end of group phalMfdf_ISO7816
 * @}
 */




/* MIFARE DESFire Miscellaneous functions. ------------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfdf_Miscellaneous Commands_Miscellaneous
 * \brief These Components implement the MIFARE DESFire Miscellaneous functions. These are not part of actual DESFire data sheet
 * rather its for internal purpose.
 * @{
 */

/** \defgroup phalMfdf_Miscellaneous_Defs Defines
 * \brief These are common definitions for \ref phalMfdf_GetConfig and \ref phalMfdf_SetConfig utility interfaces.
 * @{
 */
#define PHAL_MFDF_ADDITIONAL_INFO										0x00A1	/**< Option for getconfig to get additional info of a generic error. */
#define PHAL_MFDF_WRAPPED_MODE											0x00A2	/**< Option for getconfig to get additional info of a generic error. */
/**
 * end of group phalMfdf_Miscellaneous_Defs
 * @}
 */

/**
 * \brief Perform a GetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfdf_GetConfig(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wConfig,														/**< [In] Configuration to read. Will be one of the below values.
																				 *			\arg #PHAL_MFDF_ADDITIONAL_INFO
																				 *			\arg #PHAL_MFDF_WRAPPED_MODE
																				 */
		uint16_t * pValue														/**< [Out] The value for the mentioned configuration. */
	);

/**
 * \brief Perform a SetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfdf_SetConfig(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wConfig,														/**< [In] Configuration to be updated. Will be one of the below values.
																				 *			\arg #PHAL_MFDF_ADDITIONAL_INFO
																				 *			\arg #PHAL_MFDF_WRAPPED_MODE
																				 */
		uint16_t wValue															/**< [In] The value for the mentioned configuration. */
	);

/**
 * \brief Resets Authentication status.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfdf_ResetAuthStatus(
		void * pDataParams														/**< [In] Pointer to this layer's parameter structure. */
	);

/**
 * end of defgroup phalMfdf_Miscellaneous
 * @}
 */

/**
 * end of defgroup phalMfdf
 * @}
 */
#endif /* NXPBUILD__PHAL_MFDF */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHALMFDF_H */
