/*
 * Copyright 2013, 2016 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic MIFARE(R) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALMFP_H
#define PHALMFP_H

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalMifare.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHAL_MFP

/** \defgroup phalMfp MIFARE Plus (R)
 * \brief These Components implement the MIFARE Plus (R) commands.
 * @{
 */

/** \name Authentication Parameters */
/** @{ */
#define PHAL_MFP_KEYA												0x0AU	/**< MIFARE(R) Key A. */
#define PHAL_MFP_KEYB												0x0BU	/**< MIFARE(R) Key B. */
/** @} */

/** \name Custom Error Codes */
/** @{ */
#define PHAL_MFP_ERR_AUTH						(PH_ERR_CUSTOM_BEGIN + 0)	/*< MFP Authentication Error. */
#define PHAL_MFP_ERR_CMD_OVERFLOW				(PH_ERR_CUSTOM_BEGIN + 1)	/*< MFP Command Overflow Error. */
#define PHAL_MFP_ERR_MAC_PCD					(PH_ERR_CUSTOM_BEGIN + 2)	/*< MFP MAC Error. */
#define PHAL_MFP_ERR_BNR						(PH_ERR_CUSTOM_BEGIN + 3)	/*< MFP Blocknumber Error. */
#define PHAL_MFP_ERR_EXT						(PH_ERR_CUSTOM_BEGIN + 4)	/*< MFP Extension Error. */
#define PHAL_MFP_ERR_CMD_INVALID				(PH_ERR_CUSTOM_BEGIN + 5)	/*< MFP Invalid Command Error. */
#define PHAL_MFP_ERR_FORMAT						(PH_ERR_CUSTOM_BEGIN + 6)	/*< MFP Authentication Error. */
#define PHAL_MFP_ERR_GEN_FAILURE				(PH_ERR_CUSTOM_BEGIN + 7)	/*< MFP Generic Error. */
/** @} */

/** \name Sizes */
/** @{ */
#define PHAL_MFP_SIZE_TI												4U	/**< Size of Transaction Identifier. */
#define PHAL_MFP_SIZE_KEYMODIFIER										6U	/**< Size of MIFARE KeyModifier. */
#define PHAL_MFP_SIZE_MAC												16U	/**< Size of (untruncated) MAC. */
/** @} */

/**
 * end of group phalMfp
 * @}
 */
#endif /* NXPBUILD__PHAL_MFP */



/***************************************************************************************************************************************/
/* Software Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/

#ifdef NXPBUILD__PHAL_MFP_SW

/** \defgroup phalMfp_Sw Component : Software
 * @{
 */

#define PHAL_MFP_SW_ID          0x01    /**< ID for Software MIFARE Plus layer. */

/** \brief MIFARE Plus Software parameter structure */
typedef struct
{
	uint16_t wId;															/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;											/**< Pointer to the parameter structure of the palMifare component. */
	void * pKeyStoreDataParams;												/**< Pointer to the parameter structure of the KeyStore layer. */
	void * pCryptoDataParamsEnc;											/**< Pointer to the parameter structure of the Crypto layer for encryption. */
	void * pCryptoDataParamsMac;											/**< Pointer to the parameter structure of the Crypto layer for macing. */
	void * pCryptoRngDataParams;                            				/**< Pointer to the parameter structure of the CryptoRng layer. */
	void * pCryptoDiversifyDataParams;                      				/**< Pointer to the parameter structure of the CryptoDiversify layer (can be NULL). */
	uint8_t bKeyModifier[PHAL_MFP_SIZE_KEYMODIFIER];        				/**< Key Modifier for MIFARE Plus SL2 authentication. */
	uint16_t wRCtr;                                         				/**< R_CTR (read counter); The PICC's read counter is used for a following authentication. */
	uint16_t wWCtr;                                         				/**< W_CTR (write counter); The PICC's write counter is used for a following authentication. */
	uint8_t bTi[PHAL_MFP_SIZE_TI];                          				/**< Transaction Identifier; unused if 'bFirstAuth' = 1; uint8_t[4]. */
	uint8_t bNumUnprocessedReadMacBytes;                    				/**< Amount of data in the pUnprocessedReadMacBuffer. */
	uint8_t pUnprocessedReadMacBuffer[PHAL_MFP_SIZE_MAC];   				/**< Buffer containing unprocessed bytes for read mac answer stream. */
	uint8_t pIntermediateMac[PHAL_MFP_SIZE_MAC];            				/**< Intermediate MAC for Read Calculation. */
	uint8_t bFirstRead;                                     				/**< Indicates whether the next read is a first read in a read (MACed) sequence or not. */
} phalMfp_Sw_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfp_Sw_Init(
		phalMfp_Sw_DataParams_t * pDataParams,								/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,											/**< [In] Specifies the size of the data parameter structure. */
		void * pPalMifareDataParams,										/**< [In] Pointer to a palMifare component context. */
		void * pKeyStoreDataParams,											/**< [In] Pointer to a KeyStore component context. */
		void * pCryptoDataParamsEnc,										/**< [In] Pointer to a Crypto component context for encryption. */
		void * pCryptoDataParamsMac,										/**< [In] Pointer to a Crypto component context for Macing. */
		void * pCryptoRngDataParams,										/**< [In] Pointer to a CryptoRng component context. */
		void * pCryptoDiversifyDataParams									/**< [In] Pointer to the parameter structure of the CryptoDiversify layer (can be NULL). */
	);

/**
 * end of group phalMfp_Sw
 * @}
 */
#endif /* NXPBUILD__PHAL_MFP_SW */



/***************************************************************************************************************************************/
/* Sam NonX Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/

#ifdef NXPBUILD__PHAL_MFP_SAM_NONX

/** \defgroup phalMfp_Sam_NonX Component : Sam_NonX
 * @{
 */

#define PHAL_MFP_SAMAV2_ID											0x02	/**< ID for SamAV2 MIFARE Plus layer. */

/** \brief MIFARE Plus SamAV2 parameter structure */
typedef struct
{
	uint16_t wId;															/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;                        					/**< Pointer to the parameter structure of the palMifare component. */
	phhalHw_SamAV2_DataParams_t * pHalSamDataParams;    					/**< Pointer to the parameter structure of the hal of the SAM layer. */
	uint8_t bKeyModifier[PHAL_MFP_SIZE_KEYMODIFIER];    					/**< Key Modifier for MIFARE Plus SL2 authentication */
} phalMfp_SamAV2_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfp_SamAV2_Init(
		phalMfp_SamAV2_DataParams_t * pDataParams,							/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,											/**< [In] Specifies the size of the data parameter structure. */
		phhalHw_SamAV2_DataParams_t * pHalSamDataParams,					/**< [In] Pointer to the parameter structure of the hal of the SAM layer. */
		void * pPalMifareDataParams											/**< [In] Pointer to the parameter structure of the palMifare component. */
	);

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX

#define PHAL_MFP_SAMAV3_NONX_ID										0x04	/**< ID for SamAV3_NonX MIFARE Plus layer. */

/** \brief MIFARE Plus SamAV3_NonX parameter structure */
typedef struct
{
	uint16_t wId;															/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;                        					/**< Pointer to the parameter structure of the palMifare component. */
	phhalHw_SamAV3_DataParams_t * pHalSamDataParams;    					/**< Pointer to the parameter structure of the hal of the SAM layer. */
	uint8_t bKeyModifier[PHAL_MFP_SIZE_KEYMODIFIER];    					/**< Key Modifier for MIFARE Plus SL2 authentication */
} phalMfp_SamAV3_NonX_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfp_SamAV3_NonX_Init(
		phalMfp_SamAV3_NonX_DataParams_t * pDataParams,						/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,											/**< [In] Specifies the size of the data parameter structure. */
		phhalHw_SamAV3_DataParams_t * pHalSamDataParams,					/**< [In] Pointer to the parameter structure of the hal of the SAM layer. */
		void * pPalMifareDataParams											/**< [In] Pointer to the parameter structure of the palMifare component. */
	);
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */

/**
 * end of group phalMfp_Sam_NonX
 * @}
 */
#endif /* NXPBUILD__PHAL_MFP_SAM_NONX */



/***************************************************************************************************************************************/
/* Sam X Dataparams and Initialization Interface.																				       */
/***************************************************************************************************************************************/

#ifdef NXPBUILD__PHAL_MFP_SAM_X

/** \defgroup phalMfp_Sam_X Component : Sam_X
 * @{
 */

#define PHAL_MFP_SAMAV2_X_ID										0x03	/**< ID for SamAV2_X MIFARE Plus layer. */

/** \brief MIFARE Plus SamAV2_X parameter structure */
typedef struct
{
	uint16_t wId;															/**< Layer ID for this component, NEVER MODIFY! */
	phhalHw_SamAV2_DataParams_t * pHalDataParams;							/**< Pointer to the parameter structure of the HAL layer of the SAM. */
	void * pPalMifareDataParams;											/**< Pointer to the parameter structure of the palMifare component. */
} phalMfp_SamAV2_X_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfp_SamAV2_X_Init(
		phalMfp_SamAV2_X_DataParams_t * pDataParams,						/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,											/**< [In] Specifies the size of the data parameter structure. */
		phhalHw_SamAV2_DataParams_t * pHalDataParams,						/**< [In] Pointer to the HAL parameter structure of the SAM. */
		void * pPalMifareDataParams											/**< [In] Pointer to the parameter structure of the underlying palMFIARE layer. */
	);

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_X
#define PHAL_MFP_SAMAV3_X_ID										0x05	/**< ID for SamAV3_X MIFARE Plus layer. */

/** \brief MIFARE Plus SamAV3_X parameter structure */
typedef struct
{
    uint16_t wId;															/**< Layer ID for this component, NEVER MODIFY! */
    phhalHw_SamAV3_DataParams_t * pHalDataParams;							/**< Pointer to the parameter structure of the HAL layer of the SAM. */
    void * pPalMifareDataParams;											/**< Pointer to the parameter structure of the palMifare component. */
} phalMfp_SamAV3_X_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfp_SamAV3_X_Init(
		phalMfp_SamAV3_X_DataParams_t * pDataParams,						/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,											/**< [In] Specifies the size of the data parameter structure. */
		phhalHw_SamAV3_DataParams_t * pHalDataParams,						/**< [In] Pointer to the HAL parameter structure of the SAM. */
		void * pPalMifareDataParams											/**< [In] Pointer to the parameter structure of the underlying palMFIARE layer. */
	);

#endif /* NXPBUILD__PHAL_MFP_SAMAV3_X */

/**
 * end of group phalMfp_Sam_X
 * @}
 */
#endif /* NXPBUILD__PHAL_MFP_SAM_X */



/***************************************************************************************************************************************/
/* Mifare Plus EV1 Generic command declarations.     																				   */
/***************************************************************************************************************************************/

#ifdef NXPBUILD__PHAL_MFP

/** \addtogroup phalMfp
 * @{
 */

/** \name Options to indicate the ISO14443 protocol layer to be used. */
/** @{ */
#define PHAL_MFP_ISO14443_L3										0x00U	/**< Option to use Iso14443 Layer 3 protocol. */
#define PHAL_MFP_ISO14443_L4										0x01U	/**< Option to use Iso14443 Layer 4 protocol. */
/** @} */

/** \name Options to indicate the communication mode. */
/** @{ */
#define PHAL_MFP_ENCRYPTION_OFF										0x00U	/**< Option to indicate the communication between PCD and PICC is plain. */
#define PHAL_MFP_ENCRYPTION_ON										0x01U	/**< Option to indicate the communication between PCD and PICC is encrypted. */
/** @} */

/** \name Options to indicate the communication mode as maced for PCD to PICC transfer. */
/** @{ */
#define PHAL_MFP_MAC_ON_COMMAND_OFF									0x00U	/**< Option to indicate the communication is not maced for PCD to PICC transfer. */
#define PHAL_MFP_MAC_ON_COMMAND_ON									0x01U	/**< Option to indicate the communication is maced for PCD to PICC transfer. */
/** @} */

/** \name Options to indicate the communication mode as maced for PICC to PCD transfer. */
/** @{ */
#define PHAL_MFP_MAC_ON_RESPONSE_OFF								0x00U	/**< Option to indicate the communication is not maced for PICC to PCD transfer. */
#define PHAL_MFP_MAC_ON_RESPONSE_ON									0x01U	/**< Option to indicate the communication is maced for PICC to PCD transfer. */
/** @} */

/* Mifare Plus personalization commands. --------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_Personalization Commands_Personalization
 * \brief These Components implements the MIFARE Plus personalization commands.
 * @{
 */

/**
 * \brief Performs a MIFARE Plus Write Perso command.
 *
 * The Write Perso command can be executed using the ISO14443-3 communication protocol (after layer 3 activation)\n
 * or using the ISO14443-4 protocol (after layer 4 activation)\n
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_WritePerso(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bLayer4Comm,												/**< [In] ISO14443 protocol to be used.
																			 *			\arg #PHAL_MFP_ISO14443_L3
																			 *			\arg #PHAL_MFP_ISO14443_L4
																			 */
		uint16_t wBlockNr,													/**< [In] Block number to be personalized. */
		uint8_t * pValue													/**< [In] The value for the block mentioned in BlockNr parameter. */
	);

/**
 * \brief Performs a MIFARE Plus Commit Perso command.
 *
 * The Commit Perso command can be executed using the ISO14443-3 communication protocol (after layer 3 activation)\n
 * or using the ISO14443-4 protocol (after layer 4 activation).\n
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_CommitPerso(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bLayer4Comm													/**< [In] ISO14443 protocol to be used.
																			 *			\arg #PHAL_MFP_ISO14443_L3
																			 *			\arg #PHAL_MFP_ISO14443_L4
																			 */
	);

/**
 * end of group phalMfp_Personalization
 * @}
 */


/* Mifare Plus authentication commands. ---------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_Authenticate Commands_Authenticate
 * \brief These Components implement the MIFARE Plus authentication commands.
 * @{
 */

/**
 * \brief Perform MIFARE(R) Authenticate command in Security Level 2 with MIFARE Picc.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_AuthenticateClassicSL2(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bBlockNo,													/**< [In] PICC block number to be used for authentication. */
		uint8_t bKeyType,													/**< [In] Authentication key type to be used.
																			 *			\arg #PHAL_MFP_KEYA
																			 *			\arg #PHAL_MFP_KEYB
																			 */
		uint16_t wKeyNumber,     											/**< [In] Key number to used from software or hardware keystore. */
		uint16_t wKeyVersion,    											/**< [In] Key version to used from software or hardware keystore. */
		uint8_t * pUid,														/**< [In] UID of the PICC received during anti-collision sequence. */
		uint8_t bUidLength													/**< [In] Length of the UID buffer. */
	);

#ifdef NXPBUILD__PHAL_MFP_NDA

/** \name Options to indicate the Authentication type to be performed. */
/** @{ */
#define PHAL_MFP_AUTHENTICATE_FIRST								0x01U	/**< Option to indicate the authenticate type as first. */
#define PHAL_MFP_AUTHENTICATE_NON_FIRST							0x00U	/**< Option to indicate the authenticate type as non-first or following. */
/** @} */

/**
 * \brief Performs a complete MIFARE Plus Authentication for Security Level 0.
 *
 * The high-level function performs a 2-step (in future applications also 3-step) authentication.\n
 * See the section Evolution on the Main Page regarding future changes in the key parameter.\n\n
 *
 * The following table shows which parameter is relevant depending on the parameters bLayer4Comm and bFirstAuth.\n
 * An "X" encodes that this parameter is relevant. A "-" encodes that this parameter is ignored (if it is an in-parameter) or that it shall be ignored (if it is an out-parameter).\n
 *
 * \verbatim
 * +-------------+-------+-------+
 * | bFirstAuth  |   0   |   1   |
 * +-------------+---+---+---+---+
 * | bLayer4Comm | 0 | 1 | 0 | 1 |
 * +-------------+---+---+---+---+
 * | wBlockNr    | X | X | X | X |
 * | pKx         | X | X | X | X |
 * | pRndA       | X | X | X | X |
 * | bLenPcdCap2 | - | - | - | X |
 * | pPcdCap2    | - | - | - | X |
 * | pPdCap2     | - | - | - | X |
 * +-------------+---+---+---+---+
 * \endverbatim
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_AuthenticateSL0(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bLayer4Comm,												/**< [In] ISO14443 protocol to be used.
																			 *			\arg #PHAL_MFP_ISO14443_L3
																			 *			\arg #PHAL_MFP_ISO14443_L4
																			 */
		uint8_t bFirstAuth,      											/**< [In] Type of authentication to be performed.
																			 *			\arg #PHAL_MFP_AUTHENTICATE_FIRST
																			 *			\arg #PHAL_MFP_AUTHENTICATE_NON_FIRST
																			 */
		uint16_t wBlockNr,       											/**< [In] PICC Block number to be used for authentication. */
		uint16_t wKeyNumber,     											/**< [In] Key number be to used from software or hardware keystore. */
		uint16_t wKeyVersion,    											/**< [In] Key version be to used from software or hardware keystore. */
		uint8_t bLenDivInput,    											/**< [In] Length of diversification input used to diversify the key.
																			 *		  If 0, no diversification is performed.
																			 */
		uint8_t * pDivInput,     											/**< [In] Diversification Input used to diversify the key. */
		uint8_t bLenPcdCap2,     											/**< [In] Length of the input PCD capabilities. */
		uint8_t * pPcdCap2In,    											/**< [In] Buffer containing the Input PCD Capabilities. \n
																			 *			\c If length is non zero, PCDCapabilites should be available. \n
																			 *			\c If length is zero, PCDCapabilites buffer should be NULL. \n
																			 */
		uint8_t * pPcdCap2Out,   											/**< [Out] Buffer containing the Output PCD capabilities. This will be of 6 bytes. */
		uint8_t * pPdCap2        											/**< [Out] Buffer containing the Output PD capabilities. This will be of 6 bytes. */
	);

/**
 * \brief Performs a complete MIFARE Plus Authentication for Security Level 1.
 *
 * The high-level function performs a 2-step (in future applications also 3-step) authentication.\n
 * See the section Evolution on the Main Page regarding future changes in the key parameter.\n\n
 *
 * The following table shows which parameter is relevant depending on the parameters bLayer4Comm and bFirstAuth.\n
 * An "X" encodes that this parameter is relevant. A "-" encodes that this parameter is ignored (if it is an in-parameter) or that it shall be ignored (if it is an out-parameter).\n
 *
 * \verbatim
 * +-------------+-------+-------+
 * | bFirstAuth  |   0   |   1   |
 * +-------------+---+---+---+---+
 * | bLayer4Comm | 0 | 1 | 0 | 1 |
 * +-------------+---+---+---+---+
 * | wBlockNr    | X | X | X | X |
 * | pKx         | X | X | X | X |
 * | pRndA       | X | X | X | X |
 * | bLenPcdCap2 | - | - | - | X |
 * | pPcdCap2    | - | - | - | X |
 * | pPdCap2     | - | - | - | X |
 * +-------------+---+---+---+---+
 * \endverbatim
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_AuthenticateSL1(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bLayer4Comm,												/**< [In] ISO14443 protocol to be used.
																			 *			\arg #PHAL_MFP_ISO14443_L3
																			 *			\arg #PHAL_MFP_ISO14443_L4
																			 */
		uint8_t bFirstAuth,      											/**< [In] Type of authentication to be performed.
																			 *			\arg #PHAL_MFP_AUTHENTICATE_FIRST
																			 *			\arg #PHAL_MFP_AUTHENTICATE_NON_FIRST
																			 */
		uint16_t wBlockNr,       											/**< [In] PICC Block number to be used for authentication. */
		uint16_t wKeyNumber,     											/**< [In] Key number to be used from software or hardware keystore. */
		uint16_t wKeyVersion,    											/**< [In] Key version to be used from software or hardware keystore. */
		uint8_t bLenDivInput,    											/**< [In] Length of diversification input used to diversify the key.
																			 *		  If 0, no diversification is performed.
																			 */
		uint8_t * pDivInput,     											/**< [In] Diversification Input used to diversify the key. */
		uint8_t bLenPcdCap2,     											/**< [In] Length of the input PCD capabilities. */
		uint8_t * pPcdCap2In,    											/**< [In] Buffer containing the Input PCD Capabilities. \n
																			 *			\c If length is non zero, PCDCapabilites should be available. \n
																			 *			\c If length is zero, PCDCapabilites buffer should be NULL. \n
																			 */
		uint8_t * pPcdCap2Out,   											/**< [Out] Buffer containing the Output PCD capabilities. This will be of 6 bytes. */
		uint8_t * pPdCap2        											/**< [Out] Buffer containing the Output PD capabilities. This will be of 6 bytes. */
	);

/**
 * \brief Performs a complete MIFARE Plus Authentication and Key Derivation for Security Level 2.
 *
 * The high-level function performs a 2-step (in future applications also 3-step) authentication.\n
 * The function provides the resulting MIFARE Sector Key Modifier.\n
 * If a MIFARE Classic authentication is performed afterwards, use the MIFARE Sector Key Modifier XOR the MIFARE Classic sector key as the key.\n
 * See the section Evolution on the Main Page regarding future changes in the key parameter.\n\n
 *
 * The following table shows which parameter is relevant depending on the parameters bLayer4Comm and bFirstAuth.\n
 * An "X" encodes that this parameter is relevant. A "-" encodes that this parameter is ignored (if it is an in-parameter) or that it shall be ignored (if it is an out-parameter).\n
 *
 * \verbatim
 * +-------------+-------+-------+
 * | bFirstAuth  |   0   |   1   |
 * +-------------+---+---+---+---+
 * | bLayer4Comm | 0 | 1 | 0 | 1 |
 * +-------------+---+---+---+---+
 * | wBlockNr    | X | X | X | X |
 * | pKx         | X | X | X | X |
 * | pRndA       | X | X | X | X |
 * | bLenPcdCap2 | - | - | - | X |
 * | pPcdCap2    | - | - | - | X |
 * | pPdCap2     | - | - | - | X |
 * | pKmf        | X | - | - | - |
 * +-------------+---+---+---+---+
 * \endverbatim
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_AuthenticateSL2(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bLayer4Comm,												/**< [In] ISO14443 protocol to be used.
																			 *			\arg #PHAL_MFP_ISO14443_L3
																			 *			\arg #PHAL_MFP_ISO14443_L4
																			 */
		uint8_t bFirstAuth,      											/**< [In] Type of authentication to be performed.
																			 *			\arg #PHAL_MFP_AUTHENTICATE_FIRST
																			 *			\arg #PHAL_MFP_AUTHENTICATE_NON_FIRST
																			 */
		uint16_t wBlockNr,       											/**< [In] PICC Block number to be used for authentication. */
		uint16_t wKeyNumber,     											/**< [In] Key number to be used from software or hardware keystore. */
		uint16_t wKeyVersion,    											/**< [In] Key version to be used from software or hardware keystore. */
		uint8_t bLenDivInput,    											/**< [In] Length of diversification input used to diversify the key.
																			 *		  If 0, no diversification is performed.
																			 */
		uint8_t * pDivInput,     											/**< [In] Diversification Input used to diversify the key. */
		uint8_t bLenPcdCap2,     											/**< [In] Length of the input PCD capabilities. */
		uint8_t * pPcdCap2In,    											/**< [In] Buffer containing the Input PCD Capabilities. \n
																			 *			\c If length is non zero, PCDCapabilites should be available. \n
																			 *			\c If length is zero, PCDCapabilites buffer should be NULL. \n
																			 */
		uint8_t * pPcdCap2Out,   											/**< [Out] Buffer containing the Output PCD capabilities. This will be of 6 bytes. */
		uint8_t * pPdCap2,        											/**< [Out] Buffer containing the Output PD capabilities. This will be of 6 bytes. */
		uint8_t * pKmf														/**< [Out] MIFARE Sector Key Modifier (6 bytes). */
	);

/**
 * \brief Performs a complete MIFARE Plus Authentication and Key Derivation for Security Level 3.
 *
 * The high-level function performs a 2-step (in future applications also 3-step) authentication.\n
 * The function computes the resulting session keys for encryption and MACing and stores them in the card state structure.\n
 * See the section Evolution on the Main Page regarding future changes in the key parameter.\n\n
 *
 * The following table shows which parameter is relevant depending on the parameters bLayer4Comm and bFirstAuth.\n
 * An "X" encodes that this parameter is relevant. A "-" encodes that this parameter is ignored (if it is an in-parameter) or that it shall be ignored (if it is an out-parameter).\n
 *
 * \verbatim
 * +-------------+-------+-------+
 * | bFirstAuth  |   0   |   1   |
 * +-------------+---+---+---+---+
 * | wBlockNr    | X | X | X | X |
 * | pKx         | X | X | X | X |
 * | pRndA       | X | X | X | X |
 * | bLenPcdCap2 | - | - | - | X |
 * | pPcdCap2    | - | - | - | X |
 * | pPdCap2     | - | - | - | X |
 * +-------------+---+---+---+---+
 * \endverbatim
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_AuthenticateSL3(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFirstAuth,      											/**< [In] Type of authentication to be performed.
																			 *			\arg #PHAL_MFP_AUTHENTICATE_FIRST
																			 *			\arg #PHAL_MFP_AUTHENTICATE_NON_FIRST
																			 */
		uint16_t wBlockNr,       											/**< [In] PICC Block number to be used for authentication. */
		uint16_t wKeyNumber,     											/**< [In] Key number to be used from software or hardware keystore. */
		uint16_t wKeyVersion,    											/**< [In] Key version to be used from software or hardware keystore. */
		uint8_t bLenDivInput,    											/**< [In] Length of diversification input used to diversify the key.
																			 *		  If 0, no diversification is performed.
																			 */
		uint8_t * pDivInput,     											/**< [In] Diversification Input used to diversify the key. */
		uint8_t bLenPcdCap2,     											/**< [In] Length of the input PCD capabilities. */
		uint8_t * pPcdCap2In,    											/**< [In] Buffer containing the Input PCD Capabilities. \n
																			 *			\c If length is non zero, PCDCapabilites should be available. \n
																			 *			\c If length is zero, PCDCapabilites buffer should be NULL. \n
																			 */
		uint8_t * pPcdCap2Out,   											/**< [Out] Buffer containing the Output PCD capabilities. This will be of 6 bytes. */
		uint8_t * pPdCap2        											/**< [Out] Buffer containing the Output PD capabilities. This will be of 6 bytes. */
	);
#endif /* NXPBUILD__PHAL_MFP_NDA */

/**
 * end of group phalMfp_Authenticate
 * @}
 */


/* Mifare Plus data operation commands. ---------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_DataOperation Commands_DataOperations
 * \brief These Components implement the MIFARE Plus data operation commands.
 * @{
 */

/**
 * \brief Performs a Multi Block Write command. This command is for to use in Security Level 2 only and with MFC Authenticated state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_MultiBlockWrite(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bBlockNr,													/**< [In] PICC block number to which the data should be written. */
		uint8_t bNumBlocks,													/**< [In] Number of blocks to write (must not be more than 3). */
		uint8_t * pBlocks													/**< [In] Block(s) (16 * bNumBlocks bytes). */
	);

/**
 * \brief Performs a Multi Block Read command. This command is for to use in Security Level 2 only and with MFC Authenticated state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_MultiBlockRead(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bBlockNr,													/**< [In] PICC block number from which the data should be read. */
		uint8_t bNumBlocks,													/**< [In] Number of blocks to read (must not be more than 3). */
		uint8_t * pBlocks													/**< [Out] Block(s) (16 * bNumBlocks bytes). */
	);


#ifdef NXPBUILD__PHAL_MFP_NDA
/**
 * \brief Performs a Write / Write MACed command. This command writes a 16 byte data to the PICC.
 * The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Write(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bEncrypted,													/**< [In] Type of communication to be used. Based on this flag the command code
																			 *		  will be updated.
																			 *			\arg #PHAL_MFP_ENCRYPTION_OFF
																			 *			\arg #PHAL_MFP_ENCRYPTION_ON
																			 */
		uint8_t bWriteMaced,												/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wBlockNr,													/**< [In] PICC block number to which the data should be written. */
		uint8_t bNumBlocks,													/**< [In] Number of blocks to write (must not be more than 3) */
		uint8_t * pBlocks													/**< [In] The data to be written. This buffer should have data equal to NumBlocks * 16,
																			 *		  where 16 is one block size.
																			 */
	);

/**
 * \brief Performs a Read / Read MACed command. The parameter Encrypted, ReadMaced and MacOnCmd are valid only
 * for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Read(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bEncrypted,													/**< [In] Type of communication to be used. Based on this flag the command code
																			 *		  will be updated.
																			 *			\arg #PHAL_MFP_ENCRYPTION_OFF
																			 *			\arg #PHAL_MFP_ENCRYPTION_ON
																			 */
		uint8_t bReadMaced,													/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint8_t bMacOnCmd,													/**< [In] Indicate whether the command should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_COMMAND_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_COMMAND_ON
																			 */
		uint16_t wBlockNr,													/**< [In] PICC block number from which the data should be read. */
		uint8_t bNumBlocks,													/**< [In] Number of blocks to be read (must not be more than 3) */
		uint8_t * pBlocks													/**< [Out] The data to be read. This buffer should have data equal to ((NumBlocks * 16) + 8),
																			 *		   where 16 is one block size and 8 bytes is for storing the MAC received form PICC.
																			 *		   The last 8 bytes will be cleared once returned to the application.
																			 */
	);
#endif /* NXPBUILD__PHAL_MFP_NDA */

/**
 * end of group phalMfp_DataOperation
 * @}
 */


#ifdef NXPBUILD__PHAL_MFP_NDA
/* Mifare Plus value operation commands. --------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_ValueOperation Commands_ValueOperations
 * \brief These Components implement the MIFARE Plus value operation commands.
 * @{
 */

/**
 * \brief Performs a Write / Write MACed command of a value.
 * The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_WriteValue(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bEncrypted,													/**< [In] Type of communication to be used. Based on this flag the command code
																			 *		  will be updated.
																			 *			\arg #PHAL_MFP_ENCRYPTION_OFF
																			 *			\arg #PHAL_MFP_ENCRYPTION_ON
																			 */
		uint8_t bWriteMaced,												/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wBlockNr,													/**< [In] PICC block number to which the value should be written. */
		uint8_t * pValue,													/**< [In] The value to be written. This buffer should have value equal to 4 bytes. */
		uint8_t bAddrData													/**< [In] The address to be written. */
	);

/**
 * \brief Performs a Read / Read MACed Value command.
 * The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP authenticated state and
 * not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_ReadValue(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bEncrypted,													/**< [In] Type of communication to be used. Based on this flag the command code
																			 *		  will be updated.
																			 *			\arg #PHAL_MFP_ENCRYPTION_OFF
																			 *			\arg #PHAL_MFP_ENCRYPTION_ON
																			 */
		uint8_t bReadMaced,													/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint8_t bMacOnCmd,													/**< [In] Indicate whether the command should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_COMMAND_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_COMMAND_ON
																			 */
		uint16_t wBlockNr,													/**< [In] PICC block number from which the value should be read. */
		uint8_t * pValue,													/**< The value read from the specified block number. The buffer will have 4 bytes
																			 *	  of value information.
																			 */
		uint8_t * pAddrData													/**< [Out] The address from the read value information. */
	);

/**
 * \brief Performs an Increment / Increment MACed command.
 * The parameter IncrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Increment(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bIncrementMaced,											/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wBlockNr,													/**< [In] PICC block number to be used for incrementing the value. */
		uint8_t * pValue													/**< [In] The value to be incremented. This buffer should have 4 bytes value information.
																			 *		  The value to be incremented should be LSB first order. \n
																			 *		  \c For Ex. If the value to be incremented is by 1 times then the pValue buffer should be, \n
																			 *			 0x01, 0x00, 0x00, 0x00.
																			 */
	);

/**
 * \brief Performs a Decrement / Decrement MACed command.
 * The parameter DecrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Decrement(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bDecrementMaced,											/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wBlockNr,													/**< [In] PICC block number to be used for decrementing the value. */
		uint8_t * pValue													/**< [In] The value to be decremented. This buffer should have 4 bytes value information.
																			 *		  The value to be decremented should be LSB first order. \n
																			 *		  \c For Ex. If the value to be decremented is by 1 times then the pValue buffer should be, \n
																			 *			 0x01, 0x00, 0x00, 0x00.
																			 */
	);

/**
 * \brief Performs an Increment Transfer / Increment Transfer MACed command.
 * The parameter IncrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_IncrementTransfer(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bIncrementTransferMaced,									/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wSourceBlockNr,											/**< [In] PICC block number to be used for incrementing the value. */
		uint16_t wDestinationBlockNr,										/**< [In] PICC block number to be used for transferring the value. */
		uint8_t * pValue													/**< [In] The value to be incremented and transferred. This buffer should have 4 bytes
																			 *		  value information. The value to be incremented and transferred should be LSB first order. \n
																			 *		  \c For Ex. If the value to be incremented is by 1 times then the pValue buffer should be, \n
																			 *			 0x01, 0x00, 0x00, 0x00.
																			 */
	);

/**
 * \brief Performs a Decrement Transfer / Decrement Transfer MACed command.
 * The parameter DecrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_DecrementTransfer(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bDecrementTransferMaced,									/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wSourceBlockNr,											/**< [In] PICC block number to be used for decrementing the value. */
		uint16_t wDestinationBlockNr,										/**< [In] PICC block number to be used for transferring the value. */
		uint8_t * pValue													/**< [In] The value to be decremented and transferred. This buffer should have 4 bytes
																			 *		  value information. The value to be decremented and transferred should be LSB first order. \n
																			 *		  \c For Ex. If the value to be decremented is by 1 times then the pValue buffer should be, \n
																			 *			 0x01, 0x00, 0x00, 0x00.
																			 */
	);

/**
 * \brief Performs a Transfer / Transfer MACed command.
 * The parameter TransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Transfer(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bTransferMaced,												/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wBlockNr													/**< [In] PICC block number to be used for transferring the value. */
	);

/**
 * \brief Performs a Restore / Restore MACed command.
 * The parameter RestoreMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Restore(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bRestoreMaced,												/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wBlockNr													/**< [In] PICC block number to be used for restoring the value. */
	);

/**
 * end of group phalMfp_ValueOperation
 * @}
 */
#endif /* NXPBUILD__PHAL_MFP_NDA */


/* Mifare Plus special commands. ----------------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_Special Commands_Special
 * \brief These Components implement the MIFARE Plus additional feature commands.
 * @{
 */

/**
 * \brief Performs a Reset Auth command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_ResetAuth(
		void * pDataParams													/**< [In] Pointer to this layer's parameter structure. */
	);

/**
 * \brief Reset the libraries internal secure messaging state.
 *
 * This function must be called before interacting with the PICC to set the libraries internal card-state back to default.\n
 * E.g. when an error occurred or after a reset of the field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_ResetSecMsgState(
		void * pDataParams													/**< [In] Pointer to this layer's parameter structure. */
	);

#ifdef NXPBUILD__PHAL_MFP_NDA
/**
 * \brief Performs a Key Change of a MIFARE Plus key. Same as phalMfp_Write, but diversification input can be provided.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_ChangeKey(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bChangeKeyMaced,											/**< [In] Indicate whether the response should be maced. Based on this flag the
																			 *		  command code will be updated.
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_OFF
																			 *			\arg #PHAL_MFP_MAC_ON_RESPONSE_ON
																			 */
		uint16_t wBlockNr,													/**< [In] PICC block number to which the key should be changed. */
		uint16_t wKeyNumber,     											/**< [In] Key number to be used from software or hardware keystore. */
		uint16_t wKeyVersion,    											/**< [In] Key version to be used from software or hardware keystore. */
		uint8_t bLenDivInput,    											/**< [In] Length of diversification input used to diversify the key.
																			 *		  If 0, no diversification is performed.
																			 */
		uint8_t * pDivInput     											/**< [In] Diversification Input used to diversify the key. */
	);

/** \name Options to indicate the generation of random C. */
/** @{ */
#define PHAL_MFP_RNDC_GENERATION_INCLUDED							0x00U	/**< Option to indicate the randon number should be generated by the user and exchanged. */
#define PHAL_MFP_RNDC_GENERATION_EXCLUDED							0x01U	/**< Option to indicate the randon number should not be generated by the user and exchanged. */
/** @} */

/**
 * \brief Performs the whole Proximity Check command chain. This command is supported in Authenticated state only.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_ProximityCheck(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bGenerateRndC,												/**< [In] Indicate the generation of RndC.
																			 *			\arg #PHAL_MFP_RNDC_GENERATION_INCLUDED
																			 *			\arg #PHAL_MFP_RNDC_GENERATION_EXCLUDED
																			 */
		uint8_t * pRndC,													/**< [In] Provided RndC (7 bytes), ignored if bGenerateRndC == 1. */
		uint8_t bPps1,														/**< [In] Communication Speed byte (PPS1). */
		uint8_t bNumSteps,													/**< [In] Number of ProximityCheck cycles; RFU, must be 1. */
		uint8_t * pUsedRndC													/**< [Out] Used RndC (7 bytes), can be NULL. */
	);
#endif /* NXPBUILD__PHAL_MFP_NDA */

/**
 * end of group phalMfp_Special
 * @}
 */

/**
 * end of group phalMfp
 * @}
 */

#endif /* NXPBUILD__PHAL_MFP */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHMFPLIB_H */
