/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Interface to I.PrivateCommands in P40
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALP40CMDPRIV_H
#define PHALP40CMDPRIV_H

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_P40CMDPRIV_SW

/** \defgroup phalP40CmdPriv_Sw Component : Software
* \brief Software implementation.
* @{
*/

#define PHAL_P40CMDPRIV_SW_ID   0x01    /**< ID for P40 CmdPriv software component. */

/**
* \brief Private parameter structure
*/
typedef struct
{
    uint16_t wId;   /**< Layer ID for this component, NEVER MODIFY! */
    void * pHal;    /**< Pointer to the parameter structure of the underlying hardware layer. */
    void * pKeyStoreDataParams;     /**< Pointer to the parameter structure of the KeyStore component. */
} phalP40CmdPriv_Sw_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPriv_Sw_Init(
                            phalP40CmdPriv_Sw_DataParams_t * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                            uint16_t wDataParamSize,                      /**< [In] Specifies the size of the data parameter structure. */
                            void * pHalDataParams,                         /**< [In] Pointer to the parameter structure of the underlying hardware layer. */
                            void * pKeyStoreDataParams               /**< [In] Pointer to the parameter structure of the KeyStore component. */
                                  );

/** @} */
#endif /* NXPBUILD__PHAL_P40CMDPRIV_SW */

#ifdef NXPBUILD__PHAL_P40CMDPRIV

/** \defgroup phalP40CmdPriv P40 Private Commands
* \brief This is the RdLib side of the private commands defined in the P40 CmdPriv.
*
* <b>Defined format:</b>
* \verbatim:
* --> [CMD] [OPT. PARAM], [DATAtx(0)]...[DATAtx(n)]
* <-- [DATArx(0)]...[DATArx(n)]
* \endverbatim
* \n
* The HW interface can be either contact or contactless. contact interface sends the formated\n
* message as defined. For contactless interface, odd parity bit is applied and CRC16 appended to\n
* command and response messages , as per ISO 14443 specifications.\n
* \n
* It is the responsibilty of the application to setup Hal parameters correctly for odd parrity and CRC.
* @{
*/

/** \name Custom Error Codes
*/
/*@{*/
#define PHAL_P40CMDPRIV_ERR_COMMAND         (PH_ERR_CUSTOM_BEGIN+0) /**< P40 bad command error. */
#define PHAL_P40CMDPRIV_ERR_PARAM           (PH_ERR_CUSTOM_BEGIN+1) /**< P40 bad parameter error. */
#define PHAL_P40CMDPRIV_ERR_FORMAT          (PH_ERR_CUSTOM_BEGIN+2) /**< P40 bad cmd format error. */
#define PHAL_P40CMDPRIV_ERR_EXECUTION       (PH_ERR_CUSTOM_BEGIN+3) /**< P40 execution error. */
#define PHAL_P40CMDPRIV_ERR_NOT_AUTH        (PH_ERR_CUSTOM_BEGIN+4) /**< P40 not authorized error. */
#define PHAL_P40CMDPRIV_ERR_RESP_LENGTH     (PH_ERR_CUSTOM_BEGIN+5) /**< P40 response length incorrect. */
#define PHAL_P40CMDPRIV_ERR_UNKNOWN         (PH_ERR_CUSTOM_BEGIN+6) /**< P40 unknown error returned . */
/*@}*/

/** \name Authentication keytypes
*/
/*@{*/
#define PHAL_P40CMDPRIV_AUTHKEY_WAFERTEST  0x00U   /**< Auth key for wafer test. */
#define PHAL_P40CMDPRIV_AUTHKEY_FINALTEST  0x01U   /**< Auth key for final test. */
/*@}*/

/** \name Random Data flags
*/
/*@{*/
#define PHAL_P40CMDPRIV_USE_RANDOM_OFF  0x00U   /**< Disable random data. */
#define PHAL_P40CMDPRIV_USE_RANDOM_ON   0xFFU   /**< Enable random data. */
/*@}*/

/** \name Checksum types
*/
/*@{*/
#define PHAL_P40CMDPRIV_CKSUM_CRC8      0x00U   /**< CRC-8 Checksum. */
#define PHAL_P40CMDPRIV_CKSUM_CRC16     0x01U   /**< CRC-16 Checksum. */
#define PHAL_P40CMDPRIV_CKSUM_CRC32     0x10U   /**< CRC-32 Checksum. */
/*@}*/

#define PHAL_P40CMDPRIV_VERSIONINFO_SIZE    0x0010U

/**
* \brief Send GetVersionP command to P40 I.PrivateCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPriv_GetVersionP(
                                       void * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                                       uint8_t * pVersionData   /**< [In] Pointer to store GetVersion return data. */
                                       );

/**
* \brief Send WriteMemoryP command to P40 I.PrivateCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPriv_WriteMemoryP(
                                        void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
                                        uint16_t wStartAddr,    /**< [In] Specify starting address to write. */
                                        uint8_t bUseRandom,     /**< [In] Specify if P40 should write random data to memory. */
                                        uint8_t bDataLength,    /**< [In] Number of bytes to write, 0 = 256 bytes. */
                                        uint8_t * pData         /**< [In] Bytes to write  if bUseRandomData = 0. */
                                        );

/**
* \brief Send ReadMemoryP command to P40 I.PrivateCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPriv_ReadMemoryP(
                                      void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                      uint16_t wStartAddr,      /**< [In] specify starting address to read. */
                                      uint8_t bDataLength,      /**< [In] Number of bytes to read, 0 = 256 bytes. */
                                      uint8_t * pData,          /**< [Out] Buffer to store read data. */
                                      uint16_t * pReadLength    /**< [Out] Actual number of bytes read. */
                                      );

/**
* \brief Send CalculateChecksumP command to P40 I.PrivateCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPriv_CalculateChecksumP(
    void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wStartAddr,    /**< [In] Starting address to begin computation. */
    uint16_t wLength,       /**< [In] Number of bytes to compute over. */
    uint8_t bCksumType,     /**< [In] Type of checksum to compute. */
    uint16_t wDestAddress,  /**< [In] Optional address to store computed checksum on chip. 0 = not specified. */
    uint8_t * pCrc          /**< [Out] Storage of returned calculation. */
    );

/**
* \brief Send ExecuteP command to P40 I.PrivateCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPriv_ExecuteP(
                                   void * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wExeAddr,       /**< [In] Address of executable function. */
                                   uint8_t bMagic,               /**< [In] 0xFF - jump to wExeAddr. 0x0-0x1C syscall, 0x80-0x9C- Usrcall*/
                                   uint8_t bParamLength,    /**< [In] Number of parameter bytes to send to function. */
                                   uint8_t * pParamBuffer,  /**< [In] Additional parameters for function. */
                                   uint16_t bRespSize,      /**< [In] Total size of response buffer. */
                                   uint8_t * pRespBuffer,   /**< [Out] Storage for function return data. */
                                   uint16_t * pRespLength   /**< [Out] Number of returned data bytes written or expected to write in case of overflow. */
                                   );


/**
* \brief Send AuthenticateP command to P40 I.PrivateCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPriv_AuthenticateP(
                                    void * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                                    uint8_t bKeyId,               /**< [In] Ident of Key to authenticate to. */
                                    uint16_t wKeyNumber,  /**< [In] Key number to be used in authentication. */
                                    uint16_t wKeyVersion    /**< [In] Key version to be used in authentication. */
                                    );

/**
* \brief Send FastTestP command to P40 I.PrivateCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b TestCode is invalid.
* \retval #PH_ERR_INTERNAL_ERROR Test execution failed.
*/
phStatus_t phalP40CmdPriv_FastTestP(
                                    void * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                    uint8_t bTestCode       /**< [In] Ident of subtest to perform. */
                                    );

/**
* \brief Send SwitchLifeCycleP command to P40 I.PrivateCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b bLifeCycle is invalid.
* \retval #PH_ERR_INTERNAL_ERROR Test execution failed.
*/
phStatus_t phalP40CmdPriv_SwitchLifeCycleP( void *    pDataParams,	/**< [In] Pointer to this layer's parameter structure. */
                                            uint8_t    bLifeCycle	/**< [In] Life cycle that shall be switched to. */
											);

/** @} */
#endif /* NXPBUILD__PHAL_P40CMDPRIV */

#endif /* PHALP40CMDPRIV_H */
