/*
 * Copyright 2013, 2017, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Amplifier Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLAMP_H
#define PHDLAMP_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHDL_AMP_AR75A250

/** \cond INCLUDE_INTERNALS \defgroup phdlAmp_AR75A250 Component : AR75A250
* @{
*/

#define PHDL_AMP_AR75A250_ID  0x01U    /**< ID for Hardware Amplifier AR75A250 Architecture layer. */

#define PHDL_AMP_AR75A250_MIN_GAIN  0000U    /**< Define the minimum allowed gain value for set gain. */
#define PHDL_AMP_AR75A250_MAX_GAIN  4095U    /**< Define the maximum allowed gain value for set gain. */

/**
* \brief Amp Software parameter structure
*/
typedef struct
{
    uint16_t wId;                               /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;					/**< Pointer to the parameter structure of the balReg component. */
    uint16_t wCurrentGain;						/**< Current gain value. */
    uint8_t bWait;                              /**< Defines whether the function only returns after the transients of setting the gain are finished. */
} phdlAmp_AR75A250_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlAmp_AR75A250_Init(
    phdlAmp_AR75A250_DataParams_t * pDataParams,	/**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,						/**< [In] Specifies the size of the data parameter structure. */
    void * pBalRegDataParams						/**< [In] Pointer to the parameter structure of the balReg component. */
    );
/** @} \endcond */
#endif /* NXPBUILD__PHDL_AMP_AR75A250 */

#ifdef NXPBUILD__PHDL_AMP_BBA100

/** \cond INCLUDE_INTERNALS \defgroup phdlAmp_BBA100 Component : BBA100
* @{
*/

#define PHDL_AMP_BBA100_ID  0x02U    /**< ID for Hardware Amplifier BBA100 Architecture layer. */

#define PHDL_AMP_BBA100_MIN_GAIN  0    /**< Define the minimum allowed gain value for set gain. */
#define PHDL_AMP_BBA100_MAX_GAIN  42   /**< Define the maximum allowed gain value for set gain. */

/**
* \brief Amp Software parameter structure
*/
typedef struct
{
    uint16_t wId;                               /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;                   /**< Pointer to the parameter structure of the balReg component. */
    uint16_t wCurrentGain;                      /**< Current gain value. */
    uint8_t bWait;                              /**< Defines whether the function only returns after the transients of setting the gain are finished. */
    uint8_t bSelectedPath;                      /**< Output path that is used with this config */
} phdlAmp_BBA100_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlAmp_BBA100_Init(
    phdlAmp_BBA100_DataParams_t * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,                     /**< [In] Specifies the size of the data parameter structure. */
    void * pBalRegDataParams                        /**< [In] Pointer to the parameter structure of the balReg component. */
    );
/** @} \endcond */
#endif /* NXPBUILD__PHDL_AMP_BBA100 */

#ifdef NXPBUILD__PHDL_AMP_BBA150

/** \cond INCLUDE_INTERNALS \defgroup phdlAmp_BBA150 Component : BBA150
* @{
*/

#define PHDL_AMP_BBA150_ID  0x04U    /**< ID for Hardware Amplifier BBA150 Architecture layer. */

#define PHDL_AMP_BBA150_MIN_GAIN  0    /**< Define the minimum allowed gain value for set gain. 0    .. -25dB*/
#define PHDL_AMP_BBA150_MAX_GAIN  4000 /**< Define the maximum allowed gain value for set gain. 4000 .. +15dB*/

/**
* \brief Amp Software parameter structure
*/
typedef struct
{
    uint16_t wId;                               /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;                   /**< Pointer to the parameter structure of the balReg component. */
    uint16_t wCurrentGain;                      /**< Current gain value. */
    uint8_t bWait;                              /**< Defines whether the function only returns after the transients of setting the gain are finished. */
    uint8_t bSelectedPath;                      /**< Output path that is used with this config */
} phdlAmp_BBA150_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlAmp_BBA150_Init(
    phdlAmp_BBA150_DataParams_t * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,                     /**< [In] Specifies the size of the data parameter structure. */
    void * pBalRegDataParams                        /**< [In] Pointer to the parameter structure of the balReg component. */
    );
/** @} \endcond */
#endif /* NXPBUILD__PHDL_AMP_BBA150 */

#ifdef NXPBUILD__PHDL_AMP_MPRFA

/** \cond INCLUDE_INTERNALS \defgroup phdlAmp_MPRFA Component : MPRFA
* @{
*/

#define PHDL_AMP_MPRFA_ID  0x03U       /**< ID for Hardware Amplifier MPRFA Architecture layer. */

#define PHDL_AMP_MPRFA_MIN_GAIN_V2  20     /**< Define the minimum allowed gain value for set gain. 8 because there are arrors at lower values*/
#define PHDL_AMP_MPRFA_MAX_GAIN_V2  1000   /**< Define the maximum allowed gain value for set gain. */
#define PHDL_AMP_MPRFA_MIN_GAIN_V3  20     /**< Define the minimum allowed gain value for set gain. 8 because there are arrors at lower values*/
#define PHDL_AMP_MPRFA_MAX_GAIN_V3  2000   /**< Define the maximum allowed gain value for set gain. */
#define PHDL_AMP_MPRFA_NOT_DEF_GAIN 0xFFFF /**< with this value the gain is not defined */

#define PHDL_AMP_MPRFA_VERSION_SIZE         20U         /**< The size of the version. */
#define PHDL_AMP_MPRFA_SERIAL_SIZE         13U         /**< The size of the serial number. */

/**
* \brief Amp Software parameter structure
*/
typedef struct
{
    uint16_t wId;                               /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;                   /**< Pointer to the parameter structure of the balReg component. */
    uint16_t wCurrentGain;                      /**< Current gain value. */
    uint16_t wMinGain;                          /**< The minumum gain that should be used for the amplfier */
    uint8_t bWait;                              /**< Defines whether the function only returns after the transients of setting the gain are finished. */
    uint8_t bOn;                                /**< us set to PH_ON if the amplifier is activated */
    uint8_t bVersionString[PHDL_AMP_MPRFA_VERSION_SIZE];                   /**< This value stores the version string of the amplifier */
    uint8_t bSerialString[PHDL_AMP_MPRFA_SERIAL_SIZE];                   /**< This value stores the serial string of the amplifier */
} phdlAmp_MPRFA_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlAmp_MPRFA_Init(
    phdlAmp_MPRFA_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,                     /**< [In] Specifies the size of the data parameter structure. */
    void * pBalRegDataParams                        /**< [In] Pointer to the parameter structure of the balReg component. */
    );
/** @} \endcond */
#endif /* NXPBUILD__PHDL_AMP_MPRFA */

#ifdef NXPBUILD__PHDL_AMP

/** \cond INCLUDE_INTERNALS \defgroup phdlAmp Amplifier
* \brief These Components implement the Amplifier.
* @{
*/

/**
* \name phdlAmp Amplifier Configs
*/
/*@{*/
#define PHDL_AMP_CONFIG_WAIT        0x0000U /**< Defines if the function waits until all changes are applied in the amplififer*/
#define PHDL_AMP_CONFIG_MIN_GAIN    0x0001U /**< Defines the minimum gain setting of the amplifier (read only except for MPRFA) */
#define PHDL_AMP_CONFIG_MAX_GAIN    0x0002U /**< Defines the maximum gain setting of the amplifier (read only) */
/*@}*/

/**
* \brief Set the power of the amplifier (ON/OFF)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlAmp_SetPower(
    void * pDataParams,		/**< [In] Pointer to this layer's parameter structure. */
    uint8_t bPowerOn			/**< [In] Specifies the power state of the amplifier (0=off, 1=on). */
    );

/**
* \brief Set the gain value of the amplifier (0-4095)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlAmp_SetGain(
    void * pDataParams,		/**< [In] Pointer to this layer's parameter structure. */
    uint16_t wGainValue		/**< [In] Specifies the gain value. */
    );

/**
* \brief Set Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlAmp_SetConfig(
    void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
    uint16_t wIdentifier,			/**< [In] Specifies parameter identifier. */
    uint16_t wValue				/**< [In] Specifies parameter value. */
    );

/**
* \brief Get Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlAmp_GetConfig(
    void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
    uint16_t wIdentifier,			/**< [In] Specifies parameter identifier. */
    uint16_t * pwValue				/**< [Out] Return parameter value. */
    );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_AMP*/

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHDLAMP_H */
