/*
 * Copyright 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */


/** \file
 * ISO3 FilterCalibration Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLISO3_FILTERCALIBRATION_CMD_H
#define PHDLISO3_FILTERCALIBRATION_CMD_H

#include <phdlISO3.h>
#include <ph_TypeDefs.h>


#define PHDL_ISO3_FILTERCALIBRATION_MAXIMUM_LENGTH_SEQUENCE_NUM_SAMPLES     0x80000  /*< Numb of samples for ml sequence. 2^19 = 524288 samples */



/**
* \brief Calibrate Filter Coefficients. Pass current time stamp and Antenna Type.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlISO3_FilterCalibration_Cmd_Calibrate(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bAntennaType,                                   /**< [In] Antenna type. */
    uint64_t qwCalibrationTimeStamp                         /**< [In] Current time stamp in UNIX format. */
    );

/**
* \brief Play Maximum Length Sequence.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlISO3_FilterCalibration_Cmd_PlayMaximumLengthSequence(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams   /**< [In] Pointer to this layer's parameter structure. */
    );

/**
* \brief Get maximum length sequence from ADC Input of FPGA.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlISO3_FilterCalibration_Cmd_GetMaximumLengthSequence(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
    uint8_t* pbMlsBuffer,                                   /**< [Out] Maximum length sequence. */
    uint32_t dwMlsBufferSizeBytes,                          /**< [In] Size of buffer. */
    uint32_t * pdwMlsBufferValidBytes                       /**< [Out] Size of maximum length sequence. */
    );

/**
* \brief Get system response after playing the maximum length sequence.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlISO3_FilterCalibration_Cmd_GetSystemResponse(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
    uint8_t* pbSystemResponseBuffer,                        /**< [Out] System response captured from RX. */
    uint32_t dwSystemResponseBufferSizeBytes,               /**< [In] Size of buffer. */
    uint32_t * pdwSystemResponseBufferValidBytes            /**< [Out] Size of system response. */
    );

/**
* \brief Get filter calibration and the associated calibration time time stamp (in UNIX format) for
         the specified antenna from PetaLinux server.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlISO3_FilterCalibration_Cmd_GetFilterCoefficients(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bAntennaType,                                   /**< [In] Antenna type. */
    IIRFilterCoeffs_t * coeffs,                             /**< [Out] Filter coefficients. */
    uint64_t * pqwCalibrationTimeStamp                      /**< [Out] Calibration time stamp of filter coefficients in UNIX format. */
    );

/**
* \brief Apply filter coefficients and store them and the associated calibration time stamp (in UNIX format)
         for the specified antenna type on the PetaLinux server.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlISO3_FilterCalibration_Cmd_SetFilterCoefficients(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bAntennaType,                                   /**< [In] Antenna type. */
    IIRFilterCoeffs_t coeffs,                               /**< [In] Filter coefficients. */
    uint64_t qwCalibrationTimeStamp                         /**< [In] Calibration time stamp of filter coefficients in UNIX format. */
    );



/**
* \brief Store data inside interal buffer of petalinux server application. Assign a buffer ID for futher usage.
*        Use this function to store the MLS and the System Response for Filter Calculation.
* \return Status code
*/
phStatus_t phdlISO3_FilterCalibration_Cmd_StoreInRemoteBuffer(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,  /**< [In] Pointer to an initialized HAL parameter structure.  */
    uint32_t dwRemoteBufferID,                              /**< [In] ID of external buffer. */
    uint8_t * pbData,                                       /**< [In] Array of data to be stored on server. */
    uint32_t dwDataNumBytes                                 /**< [In] Number of bytes of data. */
    );


/**
* \brief Calculate the filter coefficients by petalinux server application. Therefore data must be present
*        in remote buffers. These remote buffers are accessible by the server application. See
*        phdlISO3_FilterCalibration_Cmd_StoreInRemoteBuffer.
* \return Status code
*/
phStatus_t phdlISO3_FilterCalibration_Cmd_CalculateFilterCoefficients(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,  /**< [In] Pointer to an initialized HAL parameter structure. */
    uint8_t bAntennaType,                                   /**< [In] Antenna type */
    uint64_t qwCalibrationTimeStamp,                        /**< [In] Calibration time stamp */
    uint32_t dwBufferIDforMLS,                              /**< [In] Id of remote buffer where the Maximum Length Sequence is located */
    uint32_t dwBufferIDforSysRes                            /**< [In] Id of remote buffer where the System Response is located */
    );

#endif /* PHDLISO3_FILTERCALIBRATION_CMD_H */


