/*
 * Copyright 2013, 2017, 2019, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Master Amplifier Oscilloscope Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLMSTAMPOSC_H
#define PHDLMSTAMPOSC_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHDL_MSTAMPOSC_AR75A250

/** \cond INCLUDE_INTERNALS \defgroup phdlMstAmpOsc_AR75A250 Component : AR75A250
* \brief Specific Implementation of Master Amplifier Oscilloscope for the amplifier AR75A250. This implementation uses a log-interpolation to set the desired field strength.
    The accuracy of this interpolation is very limited due to the thermal properties of the amplifier. The only advantage of this implementation is the faster switching and the higher dynamic range.
* @{
*/

#define PHDL_MSTAMPOSC_AR75A250_ID            0x01U    /**< ID for Software Master Amplifier Oscilloscope device layer. */

/**
* \brief Master Amplifier Osci AR75A250 parameter structure
*/
typedef struct
{
    uint16_t wId;                                           /**< Layer ID for this component, NEVER MODIFY! */
	void * pDlAmpDataParams;                                /**< Pointer to the parameter structure of the dlAmp component. */
	void * pDlOsciDataParams;                               /**< Pointer to the parameter structure of the dlOsci component. */
	uint16_t wMinFieldStrengthAmp;                          /**< Min achieveable fieldstrength with amplifier. */
	uint16_t wMaxFieldStrengthAmp;                          /**< Max achieveable fieldstrength with amplifier. */
	uint8_t bChannel;                                       /**< Selected osci channel for field strength measurement. */
	uint8_t bPrecision;                                     /**< Set the required precision for set field strength. */
	uint8_t bCheckPrecision;                                /**< Flag indicating if precision has to be checked. */
	uint8_t bCal;                                           /**< Flag indicating if calibration has been performed. */
    uint16_t wMinGainVal;                                   /**< Determines the minimum value for the gain settings of the dlAmp. */
    uint16_t wMaxGainVal;                                   /**< Determines the maximum value for the gain settings of the dlAmp. */
} phdlMstAmpOsc_AR75A250_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlMstAmpOsc_AR75A250_Init(
                                 phdlMstAmpOsc_AR75A250_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wSizeOfDataParams,                    /**< [In] Specifies the size of the data parameter structure. */
                                 void * pDlAmpDataParams,                       /**< [In] Pointer to the parameter structure of the dlAmp component. */
                                 void * pDlOsciDataParams,                      /**< [In] Pointer to the parameter structure of the dlOsci component. */
                                 uint8_t bChannel                               /**< [In] Specifies the oscilloscope channel for the field strength signal. */
                                 );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_MSTAMPOSC_AR75A250 */

#ifdef NXPBUILD__PHDL_MSTAMPOSC_ISOSETUP

/** \cond INCLUDE_INTERNALS \defgroup phdlMstAmpOsc_ISOSetup Component : ISOSetup
* \brief Specific Implementation of Master Amplifier Oscilloscope for the ISO Setup. This implementation uses 3 (4) different
    gain values to set the fieldstrength inbetween it uses the potentiometer of the RdFpga to interpolate.
    With the configuration the configuration FIND_GAIN_VALS it can be defined, whether the optimum gain values shall be determined during calibration.
* @{
*/

#define PHDL_MSTAMPOSC_ISOSETUP_ID     0x02U    /**< ID for ISO Setup Master Amplifier Oscilloscope device layer. */

/**
* \name Master Amplifier Osci ISO Setup Configs
*/
/*@{*/
#define PHDL_MSTAMPOSC_ISOSETUP_CONFIG_FIND_GAIN_VALS      (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Configure if during calibration the maximum range gain values are determined. */
#define PHDL_MSTAMPOSC_ISOSETUP_CONFIG_USE_MAX_GAIN        (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Defines if the highest gain setting is used as well. */
#define PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_0              (PH_CONFIG_CUSTOM_BEGIN + 2)    /**< Defines the (initial) value for gain 0 (is modified if find gain vals is ON). */
#define PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_1              (PH_CONFIG_CUSTOM_BEGIN + 3)    /**< Defines the (initial) value for gain 0 (is modified if find gain vals is ON). */
#define PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_2              (PH_CONFIG_CUSTOM_BEGIN + 4)    /**< Defines the (initial) value for gain 0 (is modified if find gain vals is ON). */
#define PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_3              (PH_CONFIG_CUSTOM_BEGIN + 5)    /**< Defines the (initial) value for gain 0 (is modified if find gain vals is ON). */
#define PHDL_MSTAMPOSC_ISOSETUP_CONFIG_MAX_FIELD_FIND      (PH_CONFIG_CUSTOM_BEGIN + 6)    /**< Defines the minimum value for the maximum field strength that is tried to reach during find gain values. */
#define PHDL_MSTAMPOSC_ISOSETUP_CONFIG_MIN_FIELD_OVERLAP   (PH_CONFIG_CUSTOM_BEGIN + 7)    /**< Defines the minimum overlap of the field for the different gain values. */
/*@}*/

/* define the gain value identifiers for the different settings */
#define	PHDL_MSTAMPOSC_ISO_ID0      0x00U
#define	PHDL_MSTAMPOSC_ISO_ID1      0x01U
#define	PHDL_MSTAMPOSC_ISO_ID2      0x02U
#define	PHDL_MSTAMPOSC_ISO_ID3      0x03U
#define	PHDL_MSTAMPOSC_ISO_MIN_ID   PHDL_MSTAMPOSC_ISO_ID0
#define	PHDL_MSTAMPOSC_ISO_MAX_ID   PHDL_MSTAMPOSC_ISO_ID3

/**
* \brief Master Amplifier Osci ISO Setup parameter structure
*/
typedef struct
{
    uint16_t wId;                                               /**< Layer ID for this component, NEVER MODIFY! */
	void * pDlAmpDataParams;                                    /**< Pointer to the parameter structure of the dlAmp component. */
	void * pDlOsciDataParams;                                   /**< Pointer to the parameter structure of the dlOsci component. */
	void * pDlRdFpgaDataParams;                                 /**< Pointer to the parameter structure of the dlRdFpga component. */
	uint16_t wMinFieldStrengthLowField;                         /**< Min achieveable fieldstrength in low field region for this component. */
	uint16_t wMinFieldStrength[PHDL_MSTAMPOSC_ISO_MAX_ID + 1];  /**< Min achieveable fieldstrength with poti for amplifier setting[index]. */
	uint16_t wMaxFieldStrength[PHDL_MSTAMPOSC_ISO_MAX_ID + 1];  /**< Max achieveable fieldstrength with poti for amplifier setting[index]. */
	uint16_t wGainValue[PHDL_MSTAMPOSC_ISO_MAX_ID + 1];         /**< Gain value for different Gain Ids. */
	uint8_t bCurrGainId;                                        /**< Current set Gain Identifer. */
	uint8_t bUseMaxGain;                                        /**< Flag defining if the maximum gain ID is used (12 A/m). */
	uint8_t bChannel;                                           /**< Selected osci channel for field strength measurement. */
	uint8_t bPrecision;                                         /**< Set the required precision for set field strength. */
	uint8_t bCheckPrecision;                                    /**< Flag indicating if precision has to be checked. */
	uint8_t bCal;                                               /**< Flag indicating if calibration has been performed. */
	uint16_t wFieldRxGainLowerThresh;                           /**< Determines the field strength for the lower threshold of the Rx Gain (standart value -1). */
	uint16_t wFieldRxGainUpperThresh;                           /**< Determines the field strength for the upper threshold of the Rx Gain (standart value +1). */
	uint8_t bRxGainStandard;                                    /**< Sets the standart value for the RxGain. */
    uint8_t bFindGainVals;                                      /**< Determines whether the best gain values are determined during calibration. */
    uint16_t wMinPotiVal;                                       /**< Determines the minimum value for the poti settings of the dlRdFpga. */
    uint16_t wMaxPotiVal;                                       /**< Determines the maximum value for the poti settings of the dlRdFpga. */
    uint16_t wSNRLimitPotiVal;                                  /**< Determines the SNR limit value for the poti settings of the dlRdFpga. */
    uint16_t wMinGainVal;                                       /**< Determines the minimum value for the gain settings of the dlAmp. */
    uint16_t wMaxGainVal;                                       /**< Determines the maximum value for the gain settings of the dlAmp. */
    uint16_t wMaxFindFieldStrength;                             /**< Determines the maximum value that is tried to be reached during calibration if FIND_GAIN_VALS is active. */
    uint16_t wMinOverlap;                                       /**< Determines the minumum value for the field to overlap for the different gain values. */
} phdlMstAmpOsc_ISOSetup_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlMstAmpOsc_ISOSetup_Init(
                                 phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wSizeOfDataParams,                            /**< [In] Specifies the size of the data parameter structure. */
                                 void * pDlAmpDataParams,                               /**< [In] Pointer to the parameter structure of the dlAmp component. */
                                 void * pDlOsciDataParams,                              /**< [In] Pointer to the parameter structure of the dlOsci component. */
                                 void * pDlRdFpgaDataParams,                            /**< [In] Pointer to the parameter structure of the dlRdFpga component. */
                                 uint8_t bChannel,                                      /**< [In] Specifies the oscilloscope channel for the field strength signal. */
                                 uint16_t wFieldRxGainLowerThresh,                      /**< [In] Specifies the upper fieldstrenght threashold for the RxGain. */
                                 uint16_t wFieldRxGainUpperThresh,                      /**< [In] Specifies the Lower fieldstrenght threashold for the RxGain. */
                                 uint8_t bRxGainStandart				                /**< [In] Specifies the standart RxGain. */
                                 );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_MSTAMPOSC_ISOSETUP */

#ifdef NXPBUILD__PHDL_MSTAMPOSC_MP300

/** \defgroup phdlMstAmpOsc_Mp300 Component : Mp300
* @{
*/

#define PHDL_MSTAMPOSC_MP300_ID  0x03U    /**< ID for Software Master Amplifier Oscilloscope Architecture layer. */

/**
* \name Master Amplifier Osci ISO Setup Configs
*/
/*@{*/
#define PHDL_MSTAMPOSC_MP300_CONFIG_MAX_FIELDSTRENGTH_USED      (PH_CONFIG_CUSTOM_BEGIN) /**< Defines if gains higher than 8500 mA/m are used as well. */
#define PHDL_MSTAMPOSC_MP300_CONFIG_USE_FINE_CONTROL            (PH_CONFIG_CUSTOM_BEGIN+1) /**< Defines if the fine control is used */
#define PHDL_MSTAMPOSC_MP300_CONFIG_DEFAULT_MP300_GAIN          (PH_CONFIG_CUSTOM_BEGIN+2) /**< Defines the default gain that is used with MP300 */
#define PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MODE         (PH_CONFIG_CUSTOM_BEGIN+3) /**< Defines the rx gain target mode */
#define PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MIN_FS       (PH_CONFIG_CUSTOM_BEGIN+4) /**< Set the wMinMp300RxGainFS value */
#define PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MAX_FS       (PH_CONFIG_CUSTOM_BEGIN+5) /**< Set the wMaxMp300RxGainFS value */
#define PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MIN          (PH_CONFIG_CUSTOM_BEGIN+6) /**< Defines the rx gain min value */
#define PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MAX          (PH_CONFIG_CUSTOM_BEGIN+7) /**< Defines the rx gain max value */
#define PHDL_MSTAMPOSC_MP300_CONFIG_FAST_RECALIBRATE            (PH_CONFIG_CUSTOM_BEGIN+8) /**< Define if fast recalibrate is used */
#define PHDL_MSTAMPOSC_MP300_CONFIG_USE_NORMAL_CAL_IF_FAST_FAIL (PH_CONFIG_CUSTOM_BEGIN+9) /**< Define if a normal calibration should be used as fallback if fast calibration fails */
#define PHDL_MSTAMPOSC_MP300_CONFIG_SENSE_COIL_BALANCE_CHANNEL  (PH_CONFIG_CUSTOM_BEGIN+10) /**< configs the sense coil balance channel */
#define PHDL_MSTAMPOSC_MP300_CONFIG_CANCELLATION_TOKEN          (PH_CONFIG_CUSTOM_BEGIN+11) /**< configs the cancellation token */
/*@}*/

/* define the adjust mp300 rx gain mode */
#define	PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_DISABLED          0x00U
#define	PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_CONSTANT          0x01U
#define	PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR            0x02U
#define	PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT      0x03U
#define	PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT_UP   0x04U
#define	PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT_DOWN 0x05U

/**
* \brief Master Amplifier Osci Software parameter structure
*/
typedef struct
{
    uint16_t wId;                                           /**< Layer ID for this component, NEVER MODIFY! */
    void * pDlAmpDataParams;                                /**< Pointer to the parameter structure of the dlAmp component. */
    void * pDlOsciDataParams;                               /**< Pointer to the parameter structure of the dlOsci component. */
    void * pDlRdGainDataParams;                             /**< Pointer to the parameter structure of the device supporting setting gain of the reader component. */
    double * pdGainValues;                                  /**< Array with points where the FS is measured do interpolate the curve */
    double * pdFieldstrength;                               /**< FS for the Gain values. */
    double * pdbH2Gain;                                     /**< Value used for cubic spline interpolation */
    double * pdcH2Gain;                                     /**< Value used for cubic spline interpolation */
    double * pddH2Gain;                                     /**< Value used for cubic spline interpolation */
    double * pdbGain2H;                                     /**< Value used for cubic spline interpolation */
    double * pdcGain2H;                                     /**< Value used for cubic spline interpolation */
    double * pddGain2H;                                     /**< Value used for cubic spline interpolation */
    double * pdTempBuffer;                                  /**< Buffer used to store tmp values */
    uint16_t wMaxFieldstrengthUsed;                         /**< Max fieldstrength that is allowed (also used for calibrate). */
    uint8_t bNumberPoints;                                  /**< Number of supplied (and inserted) points maximal used for spline interpolation (full range) */
    uint8_t bMaxPoints;                                     /**< Number of points maximal used for spline interpolation (limited by buffer size) */
    uint8_t bFieldstrengthValuesSupplied;                   /**< If fieldstrength values are supplied this value will be set to 1 */
    uint8_t bMaxMeasuredInterpolationPoint;                 /**< Index of the last measured Interpolation point in calibration */
    uint8_t bChannel;                                       /**< Selected osci channel for field strength measurement. */
    uint8_t bPrecision;                                     /**< Set the required precision for set field strength. */
    uint8_t bCheckPrecision;                                /**< Flag indicating if precision has to be checked. */
    uint8_t bFastRecalibrateEnabled;                        /**< Flag indicating if recalibrate use a fast mode and did not switch if the RF field */
    uint8_t bUseNormalCalIfFastFail;                        /**< Flag indicating that in case of the fast recalibrate fails a normal calibration is performed */
    uint8_t bFineControl;                                   /**< Flag indicating if the scope should measure back the fs and adjust the reader to ajive better results */
    uint8_t bCal;                                           /**< Flag indicating if calibration has been performed. */
    uint16_t wDefaultMp300Gain;                             /**< Saves the default gain the MP300 use (in Permil). */
    uint16_t wCurrAmpGain;                                  /**< Saves the current gain the Amplifier. */
    uint16_t wMinAmpGain;                                   /**< Saves the minimal gain of the Amplifier. */
    uint8_t bRxGainMode;                                    /**< Mode used for set rx gain target. */
    uint16_t wMinMp300RxGainFS;                             /**< fs point were the wMinMp300RxGain value should be achived. */
    uint16_t wMaxMp300RxGainFS;                             /**< fs point were the wMaxMp300RxGain value should be achived. */
    uint16_t wMinMp300RxGain;                               /**< gain target at fs = wMinMp300RxGainFS. */
    uint16_t wMaxMp300RxGain;                               /**< gain target at fs = wMaxMp300RxGainFS. */
    uint8_t bSenseCoilBalanceChannel;                       /**< if the sense coil balance channel is set it is checked if the probes are connected in a wrong way. 0 indicated that this is not used. This is done by checking that the channel is not higher than wMaxFieldstrengthUsed and that the cal coil is > sense coil starting from bSenseCoilBalanceCheckLimit */
    uint16_t bSenseCoilBalanceCheckLimit;                   /**< starting from this fieldstrength it is checked that the sense coil < cal coil */
    uint8_t bCancellationToken;                             /**< if set to one the calibration operation should be canceled. */
} phdlMstAmpOsc_Mp300_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlMstAmpOsc_Mp300_Init(
                                 phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wSizeOfDataParams,                        /**< [In] Specifies the size of the data parameter structure. */
                                 void * pDlAmpDataParams,                           /**< [In] Pointer to the parameter structure of the dlAmp component. */
                                 void * pDlOsciDataParams,                          /**< [In] Pointer to the parameter structure of the dlOsci component. */
                                 void * pDlRdGainDataParams,                        /**< [In] Pointer to the parameter structure of the device supporting setting gain of the reader component. */
                                 uint8_t bChannel,                                  /**< [In] Specifies the oscilloscope channel for the field strength signal. */
                                 uint8_t bNumberPoints,                             /**< [In] Number of points unsed do interpolate the caracteristic curve */
                                 double *pGainValues,                               /**< [In] Gain values that should be used for calibrate or are the points where the fieldstrength is already measured. If it is NULL bNumberPoints will be used linear (this value can be deleted after init) */
                                 double *pFieldstrengthValues,                      /**< [In] Fieldstrenght values. If they are NULL a calibration will be done. Else this values will be used for interpolation (this value can be deleted after init) */
                                 double *pSplineParameterStorage,                   /**< [In] Array where all spline parameters are stored. This array must have a size of at least (18*bNumberPoints). (bNumberPoints*(Gain + FS + 6Parameter + Tmp)*2(to allow insert of new points) */
                                 uint16_t wSizeOfSplineParameterStorage             /**< [In] size of supplied storage (in bytes)*/
                                 );

/** @} */
#endif /* NXPBUILD__PHDL_MSTAMPOSC_MP300 */

#ifdef NXPBUILD__PHDL_MSTAMPOSC_ISO3

#define PHDL_MSTAMPOSC_ISO3_ID  0x04U    /**< ID for Software Master Amplifier Oscilloscope Architecture layer. */

/**
* \brief Master Osci Step Attenuator Configs
*/
#define	PHDL_MSTAMPOSC_ISO3_STEP_ATT_CONFIG_COUNT 6
enum phdl_StepAttConfig
{
	PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB        = 0,
	PHDL_MSTAMPOSC_ISO3_STEP_ATT_6DB        = 1,
	PHDL_MSTAMPOSC_ISO3_STEP_ATT_12DB       = 2,
	PHDL_MSTAMPOSC_ISO3_STEP_ATT_18DB       = 3,
	PHDL_MSTAMPOSC_ISO3_STEP_ATT_24DB       = 4,
	PHDL_MSTAMPOSC_ISO3_STEP_ATT_CURRENT    = 5
};

/**
* \name Master Amplifier Osci ISO3 Setup Configs
*/
/*@{*/
#define PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_0DB		            (PH_CONFIG_CUSTOM_BEGIN + 0)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_6DB		            (PH_CONFIG_CUSTOM_BEGIN + 1)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_12DB		        (PH_CONFIG_CUSTOM_BEGIN + 2)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_18DB		        (PH_CONFIG_CUSTOM_BEGIN + 3)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_24DB		        (PH_CONFIG_CUSTOM_BEGIN + 4)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT	            (PH_CONFIG_CUSTOM_BEGIN + 5)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_DAC_GAIN			            (PH_CONFIG_CUSTOM_BEGIN + 6)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_CUR_FIELD			        (PH_CONFIG_CUSTOM_BEGIN + 7)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_CUR_ANTENNA                  (PH_CONFIG_CUSTOM_BEGIN + 8)
#define PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MODE     (PH_CONFIG_CUSTOM_BEGIN + 9) /**< Defines the rx threshold target mode */
#define PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MIN_FS   (PH_CONFIG_CUSTOM_BEGIN + 10) /**< Set the wMinRxThresholdFS value */
#define PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MAX_FS   (PH_CONFIG_CUSTOM_BEGIN + 11) /**< Set the wMaxRxThresholdFS value */
#define PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MIN      (PH_CONFIG_CUSTOM_BEGIN + 12) /**< Defines the rx threshold min value */
#define PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MAX      (PH_CONFIG_CUSTOM_BEGIN + 13) /**< Defines the rx threshold max value */
/*@}*/

    /**
    * \brief define the adjust mp300 rx gain mode
    */
#define	PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_DISABLED          0x00U
#define	PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_CONSTANT          0x01U
#define	PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR            0x02U
#define	PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT      0x03U
#define	PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT_UP   0x04U
#define	PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT_DOWN 0x05U

    /**
    * \brief Antenna Types
    */
#define PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1           0x00000000U     /**< Antenna type 1 */
#define PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_2           0x00000001U     /**< Antenna type 2 */
#define PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_3           0x00000002U     /**< Antenna type 3 */

/**
* \name Master Amplifier Osci ISO Setup Default Values
*/
/*@{*/
#define PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_0DB  0x8040
#define PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_6DB  0x9048
#define PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_12DB 0x9060
#define PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_18DB 0xb058
#define PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_24DB 0xb070
/*@}*/

/**
* \brief Master Amplifier Osci Software parameter structure
*/
typedef struct
{
  uint16_t wId;                                                         /**< Layer ID for this component, NEVER MODIFY! */
  void * pHalDataParams;                                                /**< Pointer to the ISO3 hardware component */
  uint16_t wMaxFieldStrength;                                           /**< Max field strength in mA/m from last calibration or configured limit */
  uint16_t wMaxFieldStrengthConfigured;                                 /**< Max field strength in mA/m that was configurd as limit */
  uint16_t wMinFieldStrength;                                           /**< Min field strength in mA/m from last calibration */
  uint32_t dwDacGain;                                                   /**< Gain factor for envelope to control field strength */
  uint32_t dwFieldStrengthRatioA1;                                      /**< To convert measured amplitude into field strength values of antenna 1 */
  uint32_t dwFieldStrengthRatioA2;                                      /**< To convert measured amplitude into field strength values of antenna 2 */
  uint32_t dwFieldStrengthRatioA3;                                      /**< To convert measured amplitude into field strength values of antenna 3 */
  uint16_t wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_CONFIG_COUNT];  /**< set commands for every attenuation value */
  uint8_t  bCurentAntennaType;                                          /**< id of current antenna */
  uint8_t bRxThresholdMode;                                             /**< Mode used for set rx threshold target. */
  uint16_t wMinRxThresholdFS;                                           /**< fs point were the wMinRxThreshold value should be achived. */
  uint16_t wMaxRxThresholdFS;                                           /**< fs point were the wMaxRxThreshold value should be achived. */
  uint16_t wMinRxThreshold;                                             /**< threshold target at fs = wMinRxThresholdFS. */
  uint16_t wMaxRxThreshold;                                             /**< threshold target at fs = wMaxRxThresholdFS. */
} phdlMstAmpOsc_ISO3_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlMstAmpOsc_ISO3_Init(
                                 phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wSizeOfDataParams,                       /**< [In] Specifies the size of the data parameter structure. */
								 void * pHalDataParams	                           /**< [In] Pointer to the ISO3 hardware component */
                                 );

/** @} */
#endif /* NXPBUILD__PHDL_MSTAMPOSC_ISO3 */


#ifdef NXPBUILD__PHDL_MSTAMPOSC

/** \cond INCLUDE_INTERNALS \defgroup phdlMstAmpOsc Master Amplifier Oscilloscope
* \brief These Components implement the Master Amplifier Oscilloscope.
* @{
*/

/** \name Custom Error Codes
*/
/*@{*/
#define PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH                   (PH_ERR_CUSTOM_BEGIN+0) /**< Entered field strength too high Error. */
#define PHDL_MSTAMPOSC_ERR_FIELD_TOO_LOW                    (PH_ERR_CUSTOM_BEGIN+1) /**< Entered field strength too low Error. */
#define PHDL_MSTAMPOSC_ERR_FIELD_GAP                        (PH_ERR_CUSTOM_BEGIN+2) /**< Error in preset field strength values causing a gap in amplifiers chacracteristic. */
#define PHDL_MSTAMPOSC_ERR_FIELD_PRECISION_NOT_ACHIEVEABLE  (PH_ERR_CUSTOM_BEGIN+3) /**< Desired presision can not be reached, lower precision restriction. */
#define PHDL_MSTAMPOSC_ERR_NOT_CAL                          (PH_ERR_CUSTOM_BEGIN+4) /**< Desired presision can not be reached, lower precision restriction. */
#define PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS         (PH_ERR_CUSTOM_BEGIN+5) /**< Failed in finding the right gain values. */
#define PHDL_MSTAMPOSC_ERR_INVALID_ANTENNA_TYPE             (PH_ERR_CUSTOM_BEGIN+6) /**< Antenna type not supported */
#define PHDL_MSTAMPOSC_ERR_CHANNEL_VALUE_MISSMATCH          (PH_ERR_CUSTOM_BEGIN+7) /**< Indicates that the channel values did not match (Sense coil > Cal Coil) */
/*@}*/

/** \name Master Amplifier Oscilloscope Config
*/
/*@{*/
#define PHDL_MSTAMPOSC_CONFIG_PRECISION         0x00U  /**< Specifies the desired precision. */
#define PHDL_MSTAMPOSC_CONFIG_CHECK_PRECISION   0x01U  /**< Specifies if the desired precesion is checked,if the precion can't be reached a recalibration is done if this does not work out an error is thrown. */
#define PHDL_MSTAMPOSC_CONFIG_MIN_FIELD         0x02U  /**< Defines the minimum field strength that can be set (read only). */
#define PHDL_MSTAMPOSC_CONFIG_MAX_FIELD         0x03U  /**< Defines the maximum field strength that can be set (read only). */
#define PHDL_MSTAMPOSC_CONFIG_AMPLIFIER_GAIN    0x04U  /**< Used to read out the used gain  (read only). */
/*@}*/

#define PHDL_MSTAMPOSC_NUM_OF_RECAL                2U   /* define the number of recalibrations before a setfieldstrength fails */
#define PHDL_MSTAMPOSC_FIELD_STRENGTH_DIVISOR	1000U   /**< Divisor for resolution of field strength (1mA/m resolution == 1/1000). */

/**
* \brief Initializes all settings of the underlying components
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlMstAmpOsc_InitMstAmpOsc(
                                       void * pDataParams                 /**< [In] Pointer to this layer's parameter structure. */
                                       );

/**
* \brief Set config value of this component
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlMstAmpOsc_SetConfig(
                                   void * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wIdentifier,    /**< [In] Specifies parameter identifier. */
                                   uint16_t wValue          /**< [In] Specifies parameter value. */
                                   );
/**
* \brief Get config value of this component
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlMstAmpOsc_GetConfig(
                                   void * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wIdentifier,    /**< [In] Specifies parameter identifier. */
                                   uint16_t * pwValue       /**< [In] Specifies parameter value. */
                                   );

/**
* \brief calibration for the normal setting of the fieldstrenght
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlMstAmpOsc_Cal(
                             void * pDataParams /**< [In] Pointer to this layer's parameter structure. */
                             );
/**
* \brief Set the fieldstrength (mA/m) normal, this is done by varying the gain of the amplifier, the only advantage of
*\ this mode is the larger range that can be covered without switching between settings
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlMstAmpOsc_SetFieldStrength(
    void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wFieldStrength /**< [In] Specifies the field strength. */
    );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_MSTAMPOSC*/

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHDLMSTAMPOSC_H */
