/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic FPGA Box Device Layer Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLRDFPGA_H
#define PHDLRDFPGA_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHDL_RDFPGA_V6

/** \cond INCLUDE_INTERNALS \defgroup phdlRdFpga_V6 Component : V6
* @{
*/

#define PHDL_RDFPGA_ID  0x01U    /**< ID for Hardware Amplifier AR75A250 Architecture layer. */

#define PHDL_RDFPGA_V6_SET_POTI_TIME		       5U      /* define the time the poti needs to finish setting the gain */
#define PHDL_RDFPGA_V6_POTI_VAL_MIN             0000U      /* defines the mimum value for the poti */
#define PHDL_RDFPGA_V6_POTI_VAL_SNR_LIMT         300U      /* defines the limit of the poti value for acceptable SNR */
#define PHDL_RDFPGA_V6_POTI_VAL_MAX             1023U      /* defines the maximum value for the poti */

/** \name Custom Error Codes
*/
/*@{*/
#define PHDL_RDFPGA_ERR_DLL		(PH_ERR_CUSTOM_BEGIN+0)    /**< phCDllReader.dll threw an non standard error read wAddionalInfo to get error code */
/*@}*/

/**
* \brief dlRdFpga Software parameter structure
*/
typedef struct
{
    uint16_t wId;											/**< Layer ID for this component, NEVER MODIFY! */
    phbalReg_RdFpgaV6_DataParams_t * pBalRegDataParams;		/**< Pointer to the parameter structure of the balReg component. */
	uint16_t wAdditionalInfo;								/**< Storage for additional error information. */
} phdlRdFpga_V6_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlRdFpga_V6_Init(
							phdlRdFpga_V6_DataParams_t * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							uint16_t wSizeOfDataParams,							/**< [In] Specifies the size of the data parameter structure. */
							phbalReg_RdFpgaV6_DataParams_t * pBalRegDataParams	/**< [In] Pointer to the parameter structure of the balReg component. */
							);

/** @} \endcond */
#endif /* NXPBUILD__PHDL_RDFPGA_V6 */

#ifdef NXPBUILD__PHDL_RDFPGA

/** \cond INCLUDE_INTERNALS \defgroup phdlRdFpga Reader FPGA
* \brief These Components implement the Reader FPGA.
* @{
*/

/**
* \name Custom phdlRdFpga V6 Configs
*/
/*@{*/
#define PHDL_RDFPGA_CONFIG_ADDITIONAL_INFO  0x0000U    /**< get addional info (error code). */
#define PHDL_RDFPGA_CONFIG_MIN_POTI		    0x0001U   /**< get minimum poti value. */
#define PHDL_RDFPGA_CONFIG_MAX_POTI		    0x0002U   /**< get maximum poti value. */
#define PHDL_RDFPGA_CONFIG_SNR_LIMIT_POTI	0x0003U    /**< get the limit of the poti value for acceptable SNR. */
/*@}*/

/**
* \brief Set the potentiometer value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetPotiPos(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint16_t wPos					/**< [In] Poti Position. */
							 );

/**
* \brief Set the default potentiometer value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetDefaultPotiPos(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint16_t wPos					/**< [In] Poti Position. */
							 );
/**
* \brief Get the potentiometer value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetPotiPos(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint16_t * pwPos				/**< [Out] Poti Position. */
							 );
/**
* \brief increase the potentiometer value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_IncPotiPos(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );
/**
* \brief decrease the potentiometer value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_DecPotiPos(
                             void * pDataParams			/**< [In] Pointer to this layer's parameter structure. */
							 );

/**
* \brief Set the timeoutvalue
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetTimeOutVal(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t dwTimeOutVal			/**< [In] Timeout value. */
							 );

/**
* \brief Get the timeoutvalue
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetTimeOutVal(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t * pdwTimeOutVal		/**< [Out] Timeout value. */
							 );

/**
* \brief Load the ini file for both joiners
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_LoadJoinerIniFile(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint8_t * pFilename			/**< [In] Pointer to filename. */
							 );
/**
* \brief load a pattern to the box
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_LoadPattern(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint8_t * pFilename,			/**< [In] Pointer to filename. */
							 uint8_t bId					/**< [In] Pattern ID. */
							 );
/**
* \brief play a pattern
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_PlayPattern(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint8_t bId					/**< [In] Pattern ID. */
							 );
/**
* \brief play a sequence A
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_PlaySeqA(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );
/**
* \brief play a sequence B
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_PlaySeqB(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );
/**
* \brief play a sinus
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_PlaySinus(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );
/**
* \brief select the trigger source
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SelectTrigger(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint8_t bTriggerSource			/**< [In] Trigger Source. */
							 );

/**
* \brief perform a reset
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_LabelReset(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint16_t wRisingEdgeCnt,		/**< [In] Rising edge count. */
							 uint32_t dwSinus,				/**< [In] Sinus value. */
							 uint32_t dwHfPause				/**< [In] HF pause value. */
							 );

/**
* \brief perform a reset
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_LabelResetCounterReset(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint16_t wRisingEdgeCnt,		/**< [In] Rising edge count. */
							 uint32_t dwSinus,				/**< [In] Sinus value. */
							 uint32_t dwHfPause				/**< [In] HF pause value. */
							 );

/**
* \brief perform a reset
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_LabelResetCounterResetSingleShot(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint16_t wRisingEdgeCnt,		/**< [In] Rising edge count. */
							 uint32_t dwSinus,				/**< [In] Sinus value. */
							 uint32_t dwHfPause 			/**< [In] HF pause value. */
							 );
/**
* \brief perform a reset
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_LabelResetSingleShot(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint16_t wRisingEdgeCnt,		/**< [In] Rising edge count. */
							 uint32_t dwSinus,				/**< [In] Sinus value. */
							 uint32_t dwHfPause 			/**< [In] HF pause value. */
							 );

/**
* \brief perform a reset
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_LabelResetWithCommand(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t dwSinus				/**< [In] Sinus value. */
							 );

/**
* \brief activate the framedelay time measurement
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ActivateFDTMeasurement(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );

/**
* \brief activate the fast exchange mode
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ActivateFastExchange(
						void * pDataParams					/**< [In] Pointer to this layer's parameter structure. */
						);

/**
* \brief get the measured framedelay time value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetFDT(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t * pdwFdt				/**< [Out] Frame Delay Time. */
							 );

/**
* \brief get the measured framedelay time value for the fast mode
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetFDTFast(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t * pdwFdt				/**< [Out] Frame Delay Time. */
							 );

/**
* \brief activate the antitearing mode
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ActivateAntiTearingMode(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t dwSinus,				/**< [In] Sinus value. */
							 uint32_t dwHfPause				/**< [In] HF pause value. */
							 );

/**
* \brief perform the resonant frequency test
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ResFreqTest(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t dwHfPause,			/**< [In] HF pause value. */
							 uint8_t bRepeatFreqId			/**< [In] Repeat frequency ID. */
							 );

/**
* \brief activate the frame delay time measurement PICC to PCD
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ActivateFDTMeasurementPICCtoPCD(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t dwTime,				/**< [In] Time value. */
							 uint8_t bCheckResp,			/**< [In] Check response value. */
							 uint16_t wRxByteCnt,			/**< [In] Return byte count value. */
							 uint8_t bRxBitCnt,				/**< [In] Return bit count value. */
							 uint8_t * pRxExpected			/**< [In] Pointer to expected response. */
							 );

/**
* \brief activate the LMA measurement
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ActivateLMAMeasurement(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );
/**
* \brief activate the sequencer mode
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ActivateSequencerMode(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );

/**
* \brief deactivate all modes
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_DeactivateModes(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );

/**
* \brief set the rx gain of the joiner
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetJoinerRxGain(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint8_t bValue					/**< [In] RX Gain value. */
							 );
/**
* \brief get the rx gain of the joiner
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetJoinerRxGain(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint8_t * bValue				/**< [Out] RX Gain value. */
							 );
/**
* \brief set the FDT for the active DS
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetFDTPCDforActiveDS(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t dwValue				/**< [In] Delay value. */
							 );
/**
* \brief set the FDT for DS
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetFDTPCDforDS(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint32_t dwValue,				/**< [In] Delay value. */
							 uint8_t bId					/**< [In] Pattern ID. */
							 );

/**
* \brief Get Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetConfig(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint16_t wIdentifier,			/**< [In] Specifies parameter identifier. */
							 uint16_t * pwValue				/**< [Out] Return parameter value. */
							 );

/**
* \brief Get FPGA version
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetVersion(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
							 uint8_t * pwValue				/**< [Out] FPGA version value. */
							 );

/**
* \brief Get FPGA revision
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetRevision(
								void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                uint8_t * pbValue           /**< [Out] FPGA revision value. */
                                );

/**
* \brief Get FPGA error codes
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetErrorCodes(
								void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                uint8_t * pbErrors,         /**< [Out] Error values. */
                                uint8_t bMaxErrorCnt,       /**< [In] Specifies the maximum number of errors. */
                                uint8_t * pbErrorCnt        /**< [Out] Number of errors received. */
                                );

/**
* \brief Get FPGA revision
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetClockSource(
								void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                uint16_t wValue             /**< [In] Clock source value. */
                                );
/**
* \brief Get the first response for the FDT PICC to PCD
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetFirstResponse(
								void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                uint8_t * pbRxBuffer,       /**< [In] Pointer to the receive buffer. */
                                uint16_t wRxBufferSize,     /**< [In] Receive buffer size. */
                                uint16_t * pwRxBytes,       /**< [Out] Number of bytes received. */
                                uint8_t * pbNumValidRxBits, /**< [Out] Number of bits received. */
                                int16_t * pwError           /**< [Out] Protocol errors. */
                                );

/**
* \brief Get the response
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetResponse(
								void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                uint8_t * pbRxBuffer,       /**< [In] Pointer to the receive buffer. */
                                uint16_t wRxBufferSize,     /**< [In] Receive buffer size. */
                                uint16_t * pwRxBytes,       /**< [Out] Number of bytes received. */
                                uint8_t * pbNumValidRxBits, /**< [Out] Number of bits received. */
                                int16_t * pwError           /**< [Out] Protocol errors. */
                                );

/**
* \brief Get IQ timeout value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetIQTimeoutValue(
								void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                uint32_t * pdwValue         /**< [Out] IQ timeout value. */
                                );

/**
* \brief Set IQ timeout value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetIQTimeoutValue(
								void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                uint32_t dwValue            /**< [In] IQ timeout value. */
                                );

/**
* \brief Get FPGA terminal ID
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetTerminalId(
								void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                uint8_t * pbValue           /**< [Out] Terminal ID value. */
                                );

/**
* \brief Reset the Joiner
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ResetJoiner(
								void * pDataParams          /**< [In] Pointer to this layer's parameter structure. */
                                );

/**
* \brief Set config values (32 bit)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_SetConfig32(
                             void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
								uint32_t dwIdentifier,       /**< [In] Specifies parameter identifier. */
                                uint32_t dwValue,            /**< [In] Specifies parameter value. */
								uint8_t bParamSetMode        /**< [In] Specifies parameter set mode. */
							 );

/**
* \brief Get config values (32 bit)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_GetConfig32(
                                void * pDataParams,          /**< [In] Pointer to this layer's parameter structure. */
								uint32_t dwIdentifier,       /**< [In] Specifies parameter identifier. */
                                uint32_t * dwValue           /**< [Out] returns the parameter value. */
                               );


/**
* \brief Load the Filter file
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_LoadFilterFile(
                                        void * pDataParams,
                                        uint8_t * pFilename
                                        );

/**
* \brief Exchange FPGA Command
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/

phStatus_t phdlRdFpga_ExchangeFPGACommand(
											void * pDataParams,
				  							uint8_t * txData,
											uint32_t txLen,
											uint8_t * rxData,
											uint32_t maxRxLen,
											uint32_t * rxLen
											);

/**
* \brief activate the type B mode
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlRdFpga_ActivateTypeBMode(
                             void * pDataParams				/**< [In] Pointer to this layer's parameter structure. */
							 );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_RDFPGA*/

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHDLRDFPGA_H */
