/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Spectrum Analyzer R&S FSV Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLSPA_H
#define PHDLSPA_H

#define PHDLSPA_SERIAL_NUMBER_SIZE           20U                 /**< The size of the seriel number*/
#define PHDLSPA_SOFTWARE_VERSION_SIZE        8U                  /**< The size of the software number*/
#define PHDLSPA_DEVICE_NUMBER_SIZE           10U                 /**< The size of the Device string*/
#define PHDLSPA_ACTIVE_PROBE_NAME_SIZE       10U                 /**< The size of the Active Probe name string*/
#define PHDLSPA_ACTIVE_PROBE_SERIAL_ID_SIZE  30U                 /**< The size of the Active Probe seriel number string*/
#define PHDLSPA_ACTIVE_PROBE_PART_ID_SIZE    30U                 /**< The size of the Active Probe part number string*/

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHDL_SPA_FSV

/** \cond INCLUDE_INTERNALS \defgroup phdlSPA_FSV Component : R&S Spectrum Analyzer FSV
* @{
*/
#define PHDL_SPA_FSV_ID  0x01U                              /**< ID for R&S Spectrum Analyzer FSV device layer. */
/*@}*/

/**
* \name FSV specific configs
*/
/*@{*/
/**
* \brief Specific FSV Settings.
*/

#define PHDL_SPA_FSV_CONFIG_IQ_MEASUREMENT          (PH_CONFIG_CUSTOM_BEGIN)            /**< IQ measuremnt. */
#define PHDL_SPA_FSV_CONFIG_ACTIVE_TRACE            (PH_CONFIG_CUSTOM_BEGIN + 1)        /**< Activate multiple traces. */
#define PHDL_SPA_FSV_CONFIG_ACTIVE_PROBE            (PH_CONFIG_CUSTOM_BEGIN + 2)        /**< Configure the Active Probe*/

/**
* \brief FSV Software parameter structure
*/
typedef struct
{
    uint16_t wId;                                                   /**< Layer ID for this component, NEVER MODIFY! */
    phbalReg_Visa_DataParams_t * pBalRegDataParams;                 /**< Pointer to the parameter structure of the balReg component. */
    uint8_t bMaxFreq;                                               /**< Holds the maximum frequenzy value of the instrument*/
    uint64_t bMaxVideoBandwidth;                                    /**< Holds the maximum VBW value of the instrument*/
    uint8_t bUseHeaderForTrace;                                     /**< Holds the value if a header should be added to the trace file */
    uint8_t bDevId[PHDLSPA_SERIAL_NUMBER_SIZE];                     /**< Holds the device ID value of the instrument*/
    uint8_t bSwVersion[PHDLSPA_SOFTWARE_VERSION_SIZE];              /**< Holds the software version value of the instrument*/
    uint8_t bDevIdStr[PHDLSPA_DEVICE_NUMBER_SIZE];                  /**< Holds the device id string of the instrument*/
    uint16_t wActiveProbeState;                                     /**< Holds the Active Probe state*/
    uint8_t bActiveProbeName[PHDLSPA_ACTIVE_PROBE_NAME_SIZE];       /**< Holds the name of the Active Probe if connected*/
    uint8_t bActiveProbeSerNum[PHDLSPA_ACTIVE_PROBE_SERIAL_ID_SIZE];/**< Holds the seriel number of the Active Probe if connected*/
    uint8_t bActiveProbePartId[PHDLSPA_ACTIVE_PROBE_PART_ID_SIZE];  /**< Holds the seriel number of the Active Probe if connected*/
    uint16_t wMaxRfAttValue;                                        /**< Holds the maximum attenuation value*/
    int64_t qMaxRefLevel;                                           /**< Holds the maximum reference level value*/
} phdlSPA_FSV_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlSPA_FSV_Init(
    phdlSPA_FSV_DataParams_t * pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,                         /**< [In] Specifies the size of the data parameter structure. */
    phbalReg_Visa_DataParams_t * pBalRegDataParams      /**< [In] Pointer to the parameter structure of the balReg component. */
    );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_SPA_FSV */

#ifdef NXPBUILD__PHDL_SPA

/** \cond INCLUDE_INTERNALS \defgroup phdlSPA Spectrum Analyzer
* \brief These Components implement the Spectrum Analyzer
* @{
*/

/* name Custom Error Codes*/

/*@{*/
#define PHDL_SPA_ERR_SELF_TEST_ERROR                (PH_ERR_CUSTOM_BEGIN+0)	    /**< SPA self test failed. */
#define PHDL_SPA_ERR_CALIBRATION_ERROR              (PH_ERR_CUSTOM_BEGIN+1)	    /**< SPA self test failed. */
#define PHDL_SPA_ERR_ACTIVE_PROBE_NOT_CONNECTED     (PH_ERR_CUSTOM_BEGIN+2)	    /**< SPA self test failed. */

/* Command Errors */
/*#define PHDL_SPA_ERR_CMD_UNKNOWN_COMMAND                       (PH_ERR_CUSTOM_BEGIN+3)        /**< SPAloscope unknown command error. */

/* Execution Errors */
/*#define PHDL_SPA_ERR_EXE_ENVIRONMENT_ERROR                        (PH_ERR_CUSTOM_BEGIN+15)    /**< SPAloscope environment error. */
/*@}*/

/**
* \name SPA Configs
*/
/*@{*/
#define PHDL_SPA_CONFIG_TRIGGER_SOURCE              0x0000U /**< Select the trigger source S902*/
#define PHDL_SPA_CONFIG_TRIGGER_SLOPE               0x0001U /**< Select the trigger slope S902*/
#define PHDL_SPA_CONFIG_TRIGGER_LEVEL_IFPOWER       0x0002U /**< Select the IF power trigger level S900*/
#define PHDL_SPA_CONFIG_TRIGGER_LEVEL_AMABS         0x0003U /**< Select the AM power trigger level S900*/
#define PHDL_SPA_CONFIG_TRIGGER_LEVEL_AMREL         0x0004U /**< Select the AM relativ trigger level S900*/
#define PHDL_SPA_CONFIG_TRIGGER_LEVEL_EXTERNAL      0x0005U /**< Select the external trigger level S900*/
#define PHDL_SPA_CONFIG_TRIGGER_LEVEL_FMPOWER       0x0006U /**< Select the FM power trigger level S900*/
#define PHDL_SPA_CONFIG_TRIGGER_LEVEL_PMPOWER       0x0007U /**< Select the PM power trigger level S900*/
#define PHDL_SPA_CONFIG_TRIGGER_LEVEL_RFPOWER       0x0008U /**< Select the RF power trigger level S900*/
#define PHDL_SPA_CONFIG_TRIGGER_LEVEL_VIDEO         0x0009U /**< Select the video trigger level S900*/
#define PHDL_SPA_CONFIG_TRIGGER_CONDITION_MASKCOND  0x000AU /**< Select the mask condition for mask trigger S900*/
#define PHDL_SPA_CONFIG64_TRIGGER_OFFSET            0x000BU/**< Select the externelTrigger offset*/

#define PHDL_SPA_CONFIG_MODE                        0x000CU /**< Select the instrument mode */
#define PHDL_SPA_CONFIG_OUTPUT_FORMAT               0x000DU /**< Select the output format ASCII or Binary */

#define PHDL_SPA_CONFIG_FREQUENCY_MODE              0x000EU /**< Select the display trace configuration */
#define PHDL_SPA_CONFIG64_CENTER_FREQUENCY          0x000FU /**< Select the frequenzy configuration  */
#define PHDL_SPA_CONFIG64_FREQUENCY_SPAN            0x0010U /**< Select the frequenzy span configuration  */
#define PHDL_SPA_CONFIG64_FREQUENCY_OFFSET          0x0011U /**< Select the frequenzy offset configuration  */
#define PHDL_SPA_CONFIG64_FREQUENCY_START           0x0012U /**< Select the frequenzy start configuration  */
#define PHDL_SPA_CONFIG64_FREQUENCY_STOP            0x0013U /**< Select the frequenzy stop configuration  */

#define PHDL_SPA_CONFIG_SWEEP_MODE                  0x0014U /**< Select the sweep mode configuration  */
#define PHDL_SPA_CONFIG_SWEEP_TYPE                  0x0015U /**< Select the sweep type configuration  */
#define PHDL_SPA_CONFIG_SWEEP_COUNT                 0x0016U /**< Select the sweep  count configuration  */
#define PHDL_SPA_CONFIG_SWEEP_POINTS                0x0017U /**< Select the sweep points configuration  */

#define PHDL_SPA_CONFIG_AVERAGE_STATE               0x0018U /**< Set averaging to on or off mode */
#define PHDL_SPA_CONFIG_AVERAGE_COUNT               0x0019U /**< Set the count for the averaging mode */
#define PHDL_SPA_CONFIG_AVERAGE_TYPE                0x001AU /**< Set averaging type */

#define PHDL_SPA_CONFIG_INPUT_COUPLING              0x001BU /**< Select the input coupling configurations */
#define PHDL_SPA_CONFIG_INPUT_IMPEDANCE             0x001CU /**< Select the input impedanceconfigurations */
#define PHDL_SPA_CONFIG_INPUT_ATTENUATION_VALUE     0x001DU /**< Select the input attenuation value configurations */
#define PHDL_SPA_CONFIG_INPUT_ATTENUATION_AUTO      0x001EU /**< Select the input attenuation auto configurations */

#define PHDL_SPA_CONFIG64_BANDWIDTH_RESOLUTION      0x001FU /**< Select the bandwidth resultion configuration */
#define PHDL_SPA_CONFIG_BANDWIDTH_FILTER_TYPE       0x0020U /**< Select the bandwidth filter configuration */
#define PHDL_SPA_CONFIG64_BANDWIDTH_RESOLUTION_VIDEO 0x0021U /**< Select the bandwidth resultion configuration */

#define PHDL_SPA_CONFIG_TRACE_DETECTOR              0x0022U/**< Select the trace detector configuration */
#define PHDL_SPA_CONFIG_TRACE_MODE                  0x0023U/**< Select the trace mode */
#define PHDL_SPA_CONFIG_POWER_UNIT                  0x0024U/**< Select the unit for the power */
#define PHDL_SPA_CONFIG64_REFERENCE_LEVEL           0x0025U/**< Select the reference level */
#define PHDL_SPA_CONFIG_RUN_MODE                    0x0026U/**< Select the Run mode*/
#define PHDL_SPA_CONFIG64_SWEEP_TIME                0x0027U/**< Select the sweep time*/
#define PHDL_SPA_CONFIG_CONTINUE_MEAS               0x0028U/**< Continue Measuremnt*/
#define PHDL_SPA_CONFIG_OPERATION_COMPLETED_STATE   0x0029U/**< Get the Operation completed state*/
#define PHDL_SPA_CONFIG_PRESET                      0x002AU/**< Resets the SPA*/
#define PHDL_SPA_CONFIG64_REFERENCE_LEVEL_OFFSET    0x002BU/**< Set reference level offset*/
#define PHDL_SPA_CONFIG_MEASUREMENT                 0x002CU/**< Configure Measuemnt*/
/*@}*/

/** \name SPA Trigger Source (possible values for config PHDL_SPA_CONFIG_TRIGGER_SOURCE)
*/
/*@{*/
#define PHDL_SPA_TRIGGER_SOURCE_EXTERNAL            0x0000U /**< Set Trigger to external source */
#define PHDL_SPA_TRIGGER_SOURCE_IFPOWER             0x0001U /**< Set Trigger to secound im frequency source */
#define PHDL_SPA_TRIGGER_SOURCE_FREERUN             0x0002U /**< Set Trigger to free run source */
#define PHDL_SPA_TRIGGER_SOURCE_MASK                0x0003U /**< Set Trigger to mask source */
#define PHDL_SPA_TRIGGER_SOURCE_TIMEDOM             0x0004U /**< Set Trigger to time domain source */
#define PHDL_SPA_TRIGGER_SOURCE_TIME                0x0005U /**< Set Trigger to time source */
#define PHDL_SPA_TRIGGER_SOURCE_VIDEO               0x0006U /**< Set Trigger to video source only available in time domain */
#define PHDL_SPA_TRIGGER_SOURCE_AFPOWER             0x0007U /**< Set Trigger to AF power source */
#define PHDL_SPA_TRIGGER_SOURCE_FMPOWER             0x0008U /**< Set Trigger to FM power source */
#define PHDL_SPA_TRIGGER_SOURCE_AMPOWER             0x0009U /**< Set Trigger to AM power source */
#define PHDL_SPA_TRIGGER_SOURCE_AMRELATIV           0x000AU /**< Set Trigger to the corresponding AM signal source */
#define PHDL_SPA_TRIGGER_SOURCE_PMPOWER             0x000BU /**< Set Trigger to PM power source */
/*@}*/

/** \name SPA Trigger Slope (possible values for config PHDL_SPA_CONFIG_TRIGGER_SLOPE)
*/
/*@{*/
#define PHDL_SPA_TRIGGER_SLOPE_POSITIVE             0x0000U /**< Set Triggert to contiouse mode S897*/
#define PHDL_SPA_TRIGGER_SLOPE_NEGATIVE             0x0001U /**< Set Trigger to stop mode, stops measure after trigger S902*/
/*@}*/

/** \name SPA Trigger Mask conditions (possible values for config PHDL_SPA_CONFIG_TRIGGER_CONDITION_MASKCOND)
*/
/*@{*/
#define PHDL_SPA_TRIGGER_CONDITION_MASKCOND_ENTER   0x0000U /**< Set Trigger conditiom to enter S902*/
#define PHDL_SPA_TRIGGER_CONDITION_MASKCOND_LEAVE   0x0001U /**< Set Trigger condition to leave S902*/
#define PHDL_SPA_TRIGGER_CONDITION_MASKCOND_IN      0x0002U /**< Set Trigger condition to inside mask S902*/
#define PHDL_SPA_TRIGGER_CONDITION_MASKCOND_OUT     0x0003U /**< Set Trigger condition to outside mask S902*/
/*@}*/

/**
* \name SPA Mode Settings (possible values for config PHDL_SPA_CONFIG_MODE)
*/
/*@{*/
#define PHDL_SPA_MODE_SPECTRUM                      0x0000U /**< Set instrument to spectrum mode  S730*/
#define PHDL_SPA_MODE_REALTIME                      0x0001U /**< Set instrumentelt time mode S730*/
#define PHDL_SPA_MODE_ANALOGDEMOD                   0x0002U /**< Set instrument analog demodulation mode S730*/
/*@}*/

/**
* \name SPA Output format Settings (possible values for config PHDL_SPA_CONFIG_OUTPUT_FORMAT)
*/
/*@{*/
#define PHDL_SPA_OUTPUT_FORMAT_ASCII                0x0000U /**< Set the outputformat to ASCII S929*/
#define PHDL_SPA_OUTPUT_FORMAT_REAL                 0x0001U /**< Set the outputformat to Binary S929 */
#define PHDL_SPA_OUTPUT_FORMAT_REAL_HEADER          0x0002U /**< Set the outputformat to Binary S929 and add a header at the begin */
/*@}*/

/**
* \name SPA Frequency Mode Settings (possible values for config PHDL_SPA_CONFIG_FREQUENCY_MODE)
*/
/*@{*/
#define PHDL_SPA_FREQUENCY_MODE_CW                  0x0000U /**< Set frequency mode CW, Fixed, Sweep S082*/
#define PHDL_SPA_FREQUENCY_MODE_FIXED               0x0001U /**< Set frequency mode CW, Fixed, Sweep S082*/
#define PHDL_SPA_FREQUENCY_MODE_SWEEP               0x0002U /**< Set frequency mode CW, Fixed, Sweep S082*/
/*@}*/

/**
* \name SPA Sweep mode (possible values for config PHDL_SPA_CONFIG_SWEEP_MODE)
*/
/*@{*/
#define PHDL_SPA_SWEEP_MODE_AUTO                    0x0000U /**< Set the operation mode for the sweep S862*/
#define PHDL_SPA_SWEEP_MODE_SPECTRUM_EMMISOIN_MASK  0x0001U /**< Set the operation mode for the sweep S862*/
#define PHDL_SPA_SWEEP_MODE_SPURIOUS_EMMISIONS      0x0002U /**< Set the operation mode for the sweep S862*/
/*@}*/

/**
* \name SPA Sweep type (possible values for config PHDL_SPA_CONFIG_SWEEP_TYPE)
*/
/*@{*/
#define PHDL_SPA_SWEEP_TYPE_SWEEPLIST               0x0000U /**< Defines the Sweep type SWE, AUTO, FFT S863*/
#define PHDL_SPA_SWEEP_TYPE_AUTO                    0x0001U /**< Defines the Sweep type SWE, AUTO, FFT S863*/
#define PHDL_SPA_SWEEP_TYPE_FFT                     0x0002U /**< Defines the Sweep type SWE, AUTO, FFT S863*/
/*@}*/

/**
* \name SPA Averaging state (possible values for config PHDL_SPA_CONFIG_AVERAGE_STATE)
*/
/*@{*/
#define PHDL_SPA_AVERAGE_TYPE_VIDEO                 0x0000U /**< Set averaging Type Video S772*/
#define PHDL_SPA_AVERAGE_TYPE_LINEAR                0x0001U /**< Set averaging Type Linear S772*/
#define PHDL_SPA_AVERAGE_TYPE_POWER                 0x0002U /**< Set averaging Type Power S772*/
/*@}*/

/**
* \name SPA Input coupling Settings (possible values for config PHDL_SPA_CONFIG_INPUT_COUPLING)
*/
/*@{*/
#define PHDL_SPA_INPUT_COUPLING_AC                  0x0000U /**< Define RF input to AC mode S722*/
#define PHDL_SPA_INPUT_COUPLING_DC                  0x0001U /**< Define RF input to DC mode S722*/
/*@}*/

/**
* \name SPA Bandwidth filter Settings (possible values for config PHDL_SPA_CONFIG_BANDWIDTH_FILTER_TYPE)
*/
/*@{*/
#define PHDL_SPA_BWIDTH_FILTER_TYPE_NORMAL          0x0000U /**< Define the filter bandwidth Normal S775*/
#define PHDL_SPA_BWIDTH_FILTER_TYPE_CHANNEL         0x0001U /**< Define the filter bandwidth Channel S775*/
#define PHDL_SPA_BWIDTH_FILTER_TYPE_RRC             0x0002U /**< Define the filter bandwidth Normal S775*/
#define PHDL_SPA_BWIDTH_FILTER_TYPE_P5              0x0003U /**< Define the filter bandwidth Channel S775*/
/*@}*/

/**
* \SPA detector settings for traces (possible values for config PHDL_SPA_CONFIG_TRACE_DETECTOR)
*/
/*@{*/
#define PHDL_SPA_TRACE_DETECTOR_APEAK               0x0000U /**< Define the detector type to APEAK S864*/
#define PHDL_SPA_TRACE_DETECTOR_NEGATIVE            0x0001U /**< Define the detector type to Negative S864*/
#define PHDL_SPA_TRACE_DETECTOR_POSITIVE            0x0002U /**< Define the detector type to Positive S864*/
#define PHDL_SPA_TRACE_DETECTOR_SAMPLE              0x0003U /**< Define the detector type to Sample S864*/
#define PHDL_SPA_TRACE_DETECTOR_RMS                 0x0004U /**< Define the detector type to RMS S864*/
#define PHDL_SPA_TRACE_DETECTOR_AVERAGE             0x0005U /**< Define the detector type to Average S864*/
#define PHDL_SPA_TRACE_DETECTOR_QPEAK               0x0006U /**< Define the detector type to QPeak S864*/
#define PHDL_SPA_TRACE_DETECTOR_CAVERAGE            0x0007U /**< Define the detector type to CAverage S864*/
#define PHDL_SPA_TRACE_DETECTOR_CRMS                0x0008U /**< Define the detector type to CRMS S864*/
/*@}*/

/**
* \SPA mode settings for display (possible values for config PHDL_SPA_CONFIG_TRACE_MODE)
*/
/*@{*/
#define PHDL_SPA_TRACE_MODE_WRITE                   0x0000U /**< Define the trace type to Write S706*/
#define PHDL_SPA_TRACE_MODE_VIEW                    0x0001U /**< Define the trace type to View S706*/
#define PHDL_SPA_TRACE_MODE_AVERAGE                 0x0002U /**< Define the trace type to Average S706*/
#define PHDL_SPA_TRACE_MODE_MAXHOLD                 0x0003U /**< Define the trace type to Maxhold S706*/
#define PHDL_SPA_TRACE_MODE_MINHOLD                 0x0004U /**< Define the trace type to Minhold S706*/
#define PHDL_SPA_TRACE_MODE_BLANK                   0x0005U /**< Define the trace type to Blank S706*/
/*@}*/

/**
* \Unit for the power settings (possible values for config PHDL_SPA_CONFIG_POWER_UNIT)
*/
/*@{*/
#define PHDL_SPA_POWER_UNIT_DBM                     0x0000U/**< Define the unit for the power to dBm S695*/
#define PHDL_SPA_POWER_UNIT_DBMV                    0x0001U/**< Define the unit for the power to dBmV S695*/
#define PHDL_SPA_POWER_UNIT_DBUV                    0x0002U/**< Define the unit for the power to dBuV S695*/
#define PHDL_SPA_POWER_UNIT_DBUA                    0x0003U/**< Define the unit for the power to dBuA S695*/
#define PHDL_SPA_POWER_UNIT_DBPW                    0x0004U/**< Define the unit for the power to dBpW S695*/
#define PHDL_SPA_POWER_UNIT_VOLT                    0x0005U/**< Define the unit for the power to Volt S695*/
#define PHDL_SPA_POWER_UNIT_AMPERE                  0x0006U/**< Define the unit for the power to Ampere S695*/
#define PHDL_SPA_POWER_UNIT_WATT                    0x0007U/**< Define the unit for the power to Watt S695*/
/*@}*/

/**
* \Available run modes selection (possible values for config PHDL_SPA_CONFIG_RUN_MODE)
*/
/*@{*/
#define PHDL_SPA_RUN_CONTINOUS                      0x0000U/**< Define the run mode contiouse S793*/
#define PHDL_SPA_RUN_SINGLE                         0x0001U/**< Define the run mode single S793*/
#define PHDL_SPA_RUN_ABORT                          0x0002U/**< Cancel current run mode*/
/*@}*/

/**
* \Available run modes selection (possible values for config PHDL_SPA_CONFIG_MEASUREMENT)
*/
/*@{*/
#define PHDL_SPA_MEASURE_TYPE_PEAK                  0x0000U/**< Measure Peak*/
#define PHDL_SPA_MEASURE_TYPE_MEAN                  0x0001U/**< Measure Mean*/
#define PHDL_SPA_MEASURE_TYPE_RMS                   0x0002U/**< Measure RMS*/
#define PHDL_SPA_MEASURE_TYPE_SDEV                  0x0003U/**< Measure Standard deviation*/
/*@}*/

/**
* \Available run modes selection (possible values for config PHDL_SPA_FSV_CONFIG_ACTIVE_PROBE)
*/
/*@{*/
#define PHDL_SPA_FSV_ACTIVE_PROBE_DEACTIVATED       0x0000U/**< Activate Active Probe*/
#define PHDL_SPA_FSV_ACTIVE_PROBE_ACTIVATED         0x0001U/**< Deactivate Active Probe*/
#define PHDL_SPA_FSV_ACTIVE_PROBE_NOT_CONNECTED     0x0002U/**< Active Probe is not connected*/
#define PHDL_SPA_FSV_ACTIVE_PROBE_NOT_QUERIED       0x0003U/**< The state was not queried*/
/*@}*/

/**
* \brief Initialize Spectrum Analyzer settings
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlSPA_InitSPA(
        void * pDataParams                  /**< [In] Pointer to this layer's parameter structure. */
        );

/**
* \brief Set Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlSPA_SetConfig(
        void * pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,               /**< [In] Specifies parameter identifier. */
        uint16_t wValue                     /**< [In] Specifies parameter value. */
        );

/**
* \brief Get Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlSPA_GetConfig(
        void * pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,               /**< [In] Specifies parameter identifier. */
        uint16_t * pwValue                  /**< [Out] Return parameter value. */
        );

/**
* \Extended Set Config for 64 bit values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlSPA_SetConfig64(
        void * pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,               /**< [In] Specifies parameter identifier. */
        int64_t qValue                      /**< [In] Specifies parameter value. */
        );


/**
* \Extended Get Config for 64 bit values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlSPA_GetConfig64(
        void * pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,               /**< [In] Specifies parameter identifier. */
        int64_t * pqValue                   /**< [Out] Return parameter value. */
        );

/**
* \brief get the current displayed trace
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlSPA_GetTrace(
        void * pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel,                   /**< [In] Specifies the channel. */
        uint32_t dwTraceBufferSize,         /**< [In] Specifies the buffer size. */
        uint32_t * pdwTraceLength,          /**< [Out] Specifies the trace size. */
        uint8_t * pbTraceBuffer,            /**< [Out] Trace buffer  */
        uint8_t * pbHeaderOffset            /**< [Out] Header offset from SPA*/
        );

/**
* \Extended Set Config for 64 bit values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlSPA_GetMeasurement(
        void * pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wMeasureSourceType,        /**< [In] Specifies measured source type. */
        float64_t * fValue                  /**< [OUT] Pointer to measured value. */
        );
/** @} \endcond */
#endif /* NXPBUILD__PHDL_SPA */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHDLSPA_H */
