/*
 * Copyright 2013, 2017, 2019, 2024, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Thermostream Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLTHSTRM_H
#define PHDLTHSTRM_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHDL_THSTRM_TP04030

/** \cond INCLUDE_INTERNALS \defgroup phdlThstrm_TP04030 Component : TP04030
* @{
*/

#define PHDL_THSTRM_TP04030_ID  0x01U   /**< ID Hardware Thermostream TP04030 Architecture layer. */

/**
* \brief Thermostream Software parameter structure
*/
typedef struct
{
    uint16_t wId;                  /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;      /**< Pointer to the parameter structure of the balReg component. */
    uint16_t wWaitTempTimeoutSec;  /**< Timeout used while waiting for the Temperature to be reached. */
    uint8_t bCancellationToken;    /**< if set to one the waiting for temp reached should be canceled. */
} phdlThstrm_TP04030_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlThstrm_TP04030_Init(
        phdlThstrm_TP04030_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                      /**< [In] Specifies the size of the data parameter structure. */
        void * pBalRegDataParams                         /**< [In] Pointer to the parameter structure of the balReg component. */
        );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_THSTRM_TP04030 */

#ifdef NXPBUILD__PHDL_THSTRM_ATS545

	/** \cond INCLUDE_INTERNALS \defgroup phdlThstrm_ATS545 Component : ATS545
	* @{
	*/
	#define PHDL_THSTRM_ATS545_ID  0x02U   /**< ID Hardware Thermostream ATS545 Architecture layer. */

	/**
	* \brief Thermostream Software parameter structure
	*/
	typedef struct
	{
		uint16_t wId;                  /**< Layer ID for this component, NEVER MODIFY! */
		void * pBalRegDataParams;      /**< Pointer to the parameter structure of the balReg component. */
        uint16_t wWaitTempTimeoutSec;  /**< Timeout used while waiting for the Temperature to be reached. */
        uint8_t bCancellationToken;    /**< if set to one the waiting for temp reached should be canceled. */
	} phdlThstrm_ATS545_DataParams_t;

	/**
	* \brief Initialise this layer.
	* \return Status code
	* \retval #PH_ERR_SUCCESS Operation successful.
	*/
	phStatus_t phdlThstrm_ATS545_Init(
			phdlThstrm_ATS545_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
			uint16_t wSizeOfDataParams,                      /**< [In] Specifies the size of the data parameter structure. */
			void * pBalRegDataParams                         /**< [In] Pointer to the parameter structure of the balReg component. */
			);

	/** @} \endcond */
#endif /* NXPBUILD__PHDL_THSTRM_ATS545 */

#ifdef NXPBUILD__PHDL_THSTRM

/** \cond INCLUDE_INTERNALS \defgroup phdlThstrm Thermostream
* \brief These Components implement any Thermostream.
* @{
*/

/** \name Custom Error Codes
*/
/*@{*/
#define PHDL_THSTRM_ERR_NOT_READY                       (PH_ERR_CUSTOM_BEGIN+0)     /**< Thermostream Not Ready error. */
#define PHDL_THSTRM_ERR_QUERY_ERROR                     (PH_ERR_CUSTOM_BEGIN+1)     /**< Thermostream Query error. */
#define PHDL_THSTRM_ERR_EXECUTION_ERROR                 (PH_ERR_CUSTOM_BEGIN+2)     /**< Thermostream execution error. */
#define PHDL_THSTRM_ERR_COMMAND_ERROR                   (PH_ERR_CUSTOM_BEGIN+3)     /**< Thermostream command error. */
#define PHDL_THSTRM_ERR_DDE_OVERHEAT                    (PH_ERR_CUSTOM_BEGIN+4)     /**< Thermostream overheat error. */
#define PHDL_THSTRM_ERR_DDE_AIR_OPEN_LOOP               (PH_ERR_CUSTOM_BEGIN+5)     /**< Thermostream air open loop error. */
#define PHDL_THSTRM_ERR_DDE_SETP_OUT_OF_RANGE           (PH_ERR_CUSTOM_BEGIN+6)     /**< Thermostream setpoint out of range error. */
#define PHDL_THSTRM_ERR_DDE_LOW_FLOW                    (PH_ERR_CUSTOM_BEGIN+7)     /**< Thermostream air flow too low error. */
#define PHDL_THSTRM_ERR_DDE_LOW_AIR_PRESSURE            (PH_ERR_CUSTOM_BEGIN+8)     /**< Thermostream air pressure too low error. */
#define PHDL_THSTRM_ERR_DDE_NO_PURGE_FLOW               (PH_ERR_CUSTOM_BEGIN+9)     /**< Thermostream no purge flow error. */
#define PHDL_THSTRM_ERR_DDE_OPEN_PURGE_TEMP_SENSOR      (PH_ERR_CUSTOM_BEGIN+10)    /**< Thermostream open purge sensor error. */
#define PHDL_THSTRM_ERR_DDE_INTERNAL_ERROR              (PH_ERR_CUSTOM_BEGIN+11)    /**< Thermostream internal error. */
#define PHDL_THSTRM_ERR_DDE_DUT_OPEN_LOOP               (PH_ERR_CUSTOM_BEGIN+12)    /**< Thermostream DUT open loop error. */
#define PHDL_THSTRM_ERR_DDE_FLOW_SENSOR_HARDWARE_ERROR  (PH_ERR_CUSTOM_BEGIN+13)    /**< Thermostream flow sensor hardware error. */
#define PHDL_THSTRM_ERR_DDE_PURGE_HEAT_FAILURE          (PH_ERR_CUSTOM_BEGIN+14)    /**< Thermostream purge heat failure. */
#define PHDL_THSTRM_ERR_DDE_IMPROPER_SOFTWARE_VERSION   (PH_ERR_CUSTOM_BEGIN+15)    /**< Thermostream improper software version error. */
#define PHDL_THSTRM_ERR_DDE_NO_DUT_SENSOR_SELECTED      (PH_ERR_CUSTOM_BEGIN+16)    /**< Thermostream no DUT sensor selected error. */
#define PHDL_THSTRM_ERR_SP_WRONG_TEMP                   (PH_ERR_CUSTOM_BEGIN+17)    /**< Thermostream wrong temperature error. */
#define PHDL_THSTRM_ERR_HEAD_IS_STILL_UP                (PH_ERR_CUSTOM_BEGIN+18)    /**< Thermostream head is still up error. */
#define PHDL_THSTRM_ERR_INVALID_SETPOINT                (PH_ERR_CUSTOM_BEGIN+19)    /**< Thermostream invalid setpoint error. */
#define PHDL_THSTRM_ERR_SP_INVALID_SOAK_TIME            (PH_ERR_CUSTOM_BEGIN+20)    /**< Thermostream invalid soaktime error. */
#define PHDL_THSTRM_ERR_INVALID_WINDOW                  (PH_ERR_CUSTOM_BEGIN+21)    /**< Thermostream invalid temperature window error. */
#define PHDL_THSTRM_ERR_INVALID_FLOWRATE                (PH_ERR_CUSTOM_BEGIN+22)    /**< Thermostream invalid flow rate error. */
#define PHDL_THSTRM_ERR_TEMP_FORMAT                     (PH_ERR_CUSTOM_BEGIN+23)    /**< Thermostream invalid temperature format error. */
/*@}*/

/**
* \name Thermostream Configs
*/
/*@{*/
#define PHDL_THSTRM_CONFIG_SELECT_SETP              0x0000U /**< Select the setpoint. */
#define PHDL_THSTRM_CONFIG_SETP_TEMP                0x0001U /**< Select the setpoint temperature. */
#define PHDL_THSTRM_CONFIG_SETP_SOAKTIME            0x0002U /**< Select the setpoint soaktime. */
#define PHDL_THSTRM_CONFIG_SETP_WINDOW              0x0003U /**< Select the setpoint temperature window. */
#define PHDL_THSTRM_CONFIG_HEAD                     0x0004U /**< Select the head position. */
#define PHDL_THSTRM_CONFIG_FLOW                     0x0005U /**< Select the flow state. */
#define PHDL_THSTRM_CONFIG_FLOWRATE                 0x0006U /**< Select the flow rate. */
#define PHDL_THSTRM_CONFIG_SETP_TEMP_REACHED        0x0007U /**< Select the temperature reached flag (read only). */
#define PHDL_THSTRM_CONFIG_CURR_TEMP                0x0008U /**< Select the current temperature (read only). */
#define PHDL_THSTRM_CONFIG_COMPRESSOR               0x0009U /**< Select the compressor state (read only).*/
#define PHDL_THSTRM_CONFIG_HEADLOCKED               0x000AU /**< Select the locking state of the head*/
#define PHDL_THSTRM_CONFIG_WAITTEMP_TIMEOUT_SEC     0x000BU /**< Select the timeout to wait for temp to be reached */
#define PHDL_THSTRM_CONFIG_CANCELLATION_TOKEN       0x000CU /**< configs the cancellation token */
#define PHDL_THSTRM_CONFIG_EDIT_MODE                0x000DU /**< configs the edit mode for the setpoints */
#define PHDL_THSTRM_CONFIG_TEMPERATURE_PERFORMANCE_MODE 0x000EU /**< configs the Temperature Performance Mode */
/*@}*/

/**
* \name Thermostream Setpoints
*/
/*@{*/
#define PHDL_THSTRM_SETPOINT_HOT                    0x0000U /**< Setpoint Hot. */
#define PHDL_THSTRM_SETPOINT_AMBIENT                0x0001U /**< Setpoint Ambient. */
#define PHDL_THSTRM_SETPOINT_COLD                   0x0002U /**< Setpoint Cold. */
/*@}*/

#define PHDL_THSTRM_HEAD_UP                         0x0000U /**< Head up*/
#define PHDL_THSTRM_HEAD_DOWN                       0x0001U /**< Head down*/
/**
* \brief Initialize the settings of the thermostream
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlThstrm_InitThstrm(
        void * pDataParams /**< [In] Pointer to this layer's parameter structure. */
        );

/**
* \brief Set config value of this component
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlThstrm_SetConfig(
        void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,   /**< [In] Specifies parameter identifier. */
        uint16_t dwValue        /**< [In] Specifies parameter value. */
        );

/**
* \brief Get config value of this component
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlThstrm_GetConfig(
        void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,   /**< [In] Specifies parameter identifier. */
        uint16_t * pwValue      /**< [Out] Specifies parameter value. */
        );

/**
* \brief Wait until the desired temperature is reached
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlThstrm_WaitTemp(
        void * pDataParams   /**< [In] Pointer to this layer's parameter structure. */
        );

/**
* \brief switch this component off (on is not possible)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlThstrm_PowerOff(
        void * pDataParams   /**< [In] Pointer to this layer's parameter structure. */
        );

/**
* \brief clear errors of this component
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlThstrm_SoftReset(
        void * pDataParams /**< [In] Pointer to this layer's parameter structure. */
        );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_THSTRM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHDLTHSTRM_H */
