/*
 * Copyright 2013 - 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic HAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_H
#define PHHALHW_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_RC663

/**
 * \defgroup phhalHw_Rc663 Component : Rc663
 * \brief RC663 Reader HAL
 *
 * \b Note: This HAL takes care about the "Request Guard Time".
 * \see phpalI14443p3a_RequestA
 * \see phpalI14443p3a_WakeUpA
 * @{
 */

#define PHHAL_HW_RC663_ID                                                               0x01    /**< ID for Rc663 HAL component. */

#define PHHAL_HW_RC663_DEFAULT_TIMEOUT                                                  150U    /**< Default timeout in microseconds. */
#define PHHAL_HW_RC663_SHADOW_COUNT                                                     0x0010U /**< Number of shadowed configurations. */

/** \name Custom Configs */
/** @{ */
#define PHHAL_HW_RC663_CONFIG_FIFOSIZE                          (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Configure FIFO-Size. */
#define PHHAL_HW_RC663_CONFIG_LOADREG_MODE                      (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Enable (#PH_ON) or disable (#PH_OFF) LoadReg Mode.
                                                                                                 * If this mode is enabled (it's disabled by default) then LoadReg is used\n
                                                                                                 * if an operation mode changes instead of applying library constants.\n
                                                                                                 * This is useful if custom antenna configuration values are required to be applied.\n
                                                                                                 * The parameters for LoadReg are taken from \b pLoadRegConfig which is an input parameter\n
                                                                                                 * of \ref phhalHw_Rc663_Init. Refer there for a detailed description of \b pLoadRegConfig.
                                                                                                 */
/** @} */

/** \name Fifo Configuration Values */
/** @{ */
#define PHHAL_HW_RC663_VALUE_FIFOSIZE_255                                               0x0000U /**< Configure FIFO-Size to 256 bytes */
#define PHHAL_HW_RC663_VALUE_FIFOSIZE_512                                               0x0001U /**< Configure FIFO-Size to 512 bytes */
/** @} */

/** \name Custom Option Bits: Can be used for \ref phhalHw_Exchange, \ref phhalHw_Rc663_Cmd_Receive and \ref phhalHw_Rc663_Cmd_Transmit. */
/** @{ */
#define PHHAL_HW_RC663_OPTION_RXTX_TIMER_NOSTART                                        0x0000U /**< Do not start timer after transmission / before reception; (only for
                                                                                                 *   \ref phhalHw_Rc663_Cmd_Receive and \ref phhalHw_Rc663_Cmd_Transmit).
                                                                                                 */
#define PHHAL_HW_RC663_OPTION_RXTX_TIMER_START                                          0x0010U /**< Start timer after transmission / before reception, reset timer counter
                                                                                                 * value; (only for \ref phhalHw_Rc663_Cmd_Receive and \ref phhalHw_Rc663_Cmd_Transmit).
                                                                                                 */
/** @} */

/** \brief Rc663 HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pBalDataParams;                                                                     /**< Pointer to the lower layers parameter structure. */
    uint8_t bLoadRegMode;                                                                       /**< Whether LoadReg-Mode is enabled or not. */
    uint8_t * pLoadRegConfig;                                                                   /**< Pointer to configuration buffer for LoadReg mode; Refer to \ref phhalHw_Rc663_Init. */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wMaxPrecachedBytes;                                                                /**< Holds the max. number of bytes which are precached prior to command execution. */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
    uint16_t wCfgShadow[PHHAL_HW_RC663_SHADOW_COUNT];                                           /**< Configuration shadow; Stores configuration for current cardtype. */
    uint16_t wTimingMode;                                                                       /**< Current timing measurement mode. */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint32_t dwTimingUs;                                                                        /**< Current timing value. */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint8_t bSymbolStart;                                                                       /**< Preamble of a frame. */
    uint8_t bSymbolEnd;                                                                         /**< Trailer symbol of a frame. */
    uint8_t bFifoSize;                                                                          /**< Fifo-Size setting. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint8_t bBalConnectionType;                                                                 /**< Type of the underlying BAL connection. Refer to #PHHAL_HW_CONFIG_BAL_CONNECTION. */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC*/
} phhalHw_Rc663_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * A note on <b>pLoadRegConfig</b>:\n
 * This is used if #PHHAL_HW_RC663_CONFIG_LOADREG_MODE is enabled.\n
 * It is defined as follows:
 * \code
 * [MSB]                                              [LSB]
 * 31...............16..................8.................0
 * +------------------+------------------+----------------+
 * | wEEAddress (16b) | bRegAddress (8b) | bNumBytes (8b) |
 * +------------------+------------------+----------------+
 * \endcode
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Rc663_Init(
        phhalHw_Rc663_DataParams_t * pDataParams,                                               /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pLoadRegConfig,                                                               /**< [In] Pointer to configuration buffer for LoadReg mode; See description above. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. Specify the buffer +1 byte, because one byte is reserved for SPI communication. */
    );

#include <phhalHw_Rc663_Reg.h>
#include <phhalHw_Rc663_Cmd.h>

/**
 * end of group phhalHw_Rc663
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_RC663 */

#ifdef NXPBUILD__PHHAL_HW_RC523

/**
 * \defgroup phhalHw_Rc523 Component : Rc523
 * \brief RC523 Reader HAL
 *
 * Since RC523 and PN512 are quite compatible, the following PN512 features are also supported:\n
 * \li #PHHAL_HW_CARDTYPE_FELICA
 * \li #PHHAL_HW_CARDTYPE_I18092MPI
 * \li #PHHAL_HW_CARDTYPE_I18092MPT
 * \li \ref phhalHw_Rc523_Cmd_Config
 * \li \ref phhalHw_Rc523_Cmd_Autocoll
 *
 * If #PHHAL_HW_CARDTYPE_I18092MPT is chosen, Exchange performs the reception BEFORE the transmission.
 *
 * This makes it neccessary for the application to know what to send before actually having received\n
 * the request. Therefore using phhalHw_Exchange for PN512 operation is only useful for test applications.\n
 * For other applications it's recommended to use \ref phhalHw_Rc523_Cmd_Transmit and \ref phhalHw_Rc523_Cmd_Receive.
 *
 * @{
 */

#define PHHAL_HW_RC523_ID                                                               0x02    /**< ID for Rc523 HAL component. */

#define PHHAL_HW_RC523_SHADOW_COUNT                                                     0x000FU /**< Number of shadowed configurations. */

/** \name Custom Option Bits : Can be used for \ref phhalHw_Exchange, \ref phhalHw_Rc663_Cmd_Receive and \ref phhalHw_Rc663_Cmd_Transmit. */
/** @{ */

#define PHHAL_HW_RC523_OPTION_RXTX_TIMER_NOSTART                                        0x0000U /**< Do not start timer after transmission / before reception;
                                                                                                 *   (only for \ref phhalHw_Rc523_Cmd_Receive and \ref phhalHw_Rc523_Cmd_Transmit).
                                                                                                 */

#define PHHAL_HW_RC523_OPTION_RXTX_TIMER_START                                          0x0010U /**< Start timer after transmission / before reception, reset timer counter value;
                                                                                                 *   (only for \ref phhalHw_Rc523_Cmd_Receive and \ref phhalHw_Rc523_Cmd_Transmit).
                                                                                                 */
/** @} */

/** \brief Rc523 HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pBalDataParams;                                                                     /**< Pointer to the lower layers parameter structure. */
    void  * pKeyStoreDataParams;                                                                /**< Pointer to the parameter structure of the keyStore layer. */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wMaxPrecachedBytes;                                                                /**< Holds the max. number of bytes which are precached prior to command execution. */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
    uint16_t wCfgShadow[PHHAL_HW_RC523_SHADOW_COUNT];                                           /**< Configuration shadow; Stores configuration for current cardtype. */
    uint16_t wTimingMode;                                                                       /**< Current timing measurement mode. */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint32_t dwTimingUs;                                                                        /**< Current timing value. */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint8_t bBalConnectionType;                                                                 /**< Type of the underlying BAL connection. Refer to #PHHAL_HW_CONFIG_BAL_CONNECTION. */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC*/
} phhalHw_Rc523_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Rc523_Init(
        phhalHw_Rc523_DataParams_t * pDataParams,                                               /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        void * pKeyStoreDataParams,                                                             /**< [In] Pointer to the parameter structure of the keyStore layer. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. Specify the buffer +1 byte, because one byte is reserved for SPI communication. */
    );

#include <phhalHw_Rc523_Reg.h>
#include <phhalHw_Rc523_Cmd.h>

/**
 * end of group phhalHw_Rc523
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_RC523 */

#ifdef NXPBUILD__PHHAL_HW_SAMAV2

/**
 * \defgroup phhalHw_SamAV2 Component : SamAV2
 * \brief SamAV2 Reader HAL
 *
 * Also supports AV1 backwards compatibility mode.
 * @{
 */

#define PHHAL_HW_SAMAV2_ID                                                              0x03    /**< ID for SamAV2 HAL component. */

#define PHHAL_HW_SAMAV2_SHADOW_COUNT                                                    0x000FU /**< Number of shadowed configurations. */

#define PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN                                          6U      /**< Amount of needed and reserved memory for the protocol overhead. */
#define PHHAL_HW_SAMAV2_RESERVED_RX_BUFFER_LEN                                          2U      /**< Amount of needed and reserved memory for the protocol overhead. */

#define PHHAL_HW_SAMAV2_HC_SAM_UID_SIZE                                                 0x07U   /**< Length of the SAM UID */
#define PHHAL_HW_SAMAV2_KEYENTRY_SIZE_AV1                                               0x3CU   /**< Define the maximum possible size of a key entry in AV1. */
#define PHHAL_HW_SAMAV2_KEYENTRY_SIZE                                                   0x3DU   /**< Define the maximum possible size of a key entry. */

/** \name Operation modes */
/** @{ */
#define PHHAL_HW_SAMAV2_OPMODE_NON_X                                                    0x00    /**< Non-X operation mode. */
#define PHHAL_HW_SAMAV2_OPMODE_X_RC523                                                  0x01    /**< X-Mode operation (using Rc523). */
#define PHHAL_HW_SAMAV2_OPMODE_X_RC663                                                  0x02    /**< X-Mode operation (using Rc663). */
/** @} */

/** \name Host-Communication modes */
/** @{ */
#define PHHAL_HW_SAMAV2_HC_AV1_MODE                                                     0x01U   /**< Define the AV1 mode of the SAMAV2 */
#define PHHAL_HW_SAMAV2_HC_AV2_MODE                                                     0x02U   /**< Define the AV2 mode of the SAMAV2 */
/** @} */

/** \name Configs */
/** @{ */

#define PHHAL_HW_SAMAV2_CONFIG_HOSTMODE                         (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Get / Set the Hostmode; (e.g. #PHHAL_HW_SAMAV2_HC_AV1_MODE). */
#define PHHAL_HW_SAMAV2_CONFIG_DISABLE_NONX_CFG_MAPPING         (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Disables the mapping of standard-configs to the Reader HAL if set to #PH_ON;
                                                                                                 *   Default is #PH_OFF; Only applicable in NonX-Mode.
                                                                                                 */
/** @} */

/** \name phhalHw_SamAV2 Custom Exchange Option Bits*/
/** @{ */
#define PHHAL_HW_SAMAV2_EXCHANGE_NO_ENCIPHERING_BIT                                     0x0010U /**< Suppress MIFARE Crypto enciphering of transmitted data. */
#define PHHAL_HW_SAMAV2_EXCHANGE_NO_DECIPHERING_BIT                                     0x0020U /**< Suppress deciphering of received MIFARE Crypto streams. */
/** @} */

/** \name DES Key Options */
/** @{ */
#define PHHAL_HW_SAMAV2_DES_OPTION_DESFIRE4                                             0x00    /**< DESFire 4 compatibility mode. */
#define PHHAL_HW_SAMAV2_DES_OPTION_ISO_CRC16                                            0x01    /**< ISO 10116 mode with CRC16 protection and 4 bytes MAC. */
#define PHHAL_HW_SAMAV2_DES_OPTION_ISO_CRC32                                            0x02    /**< ISO 10116 mode with CRC32 protection and 8 bytes MAC. */
/** @} */

/** \brief SamAV2 HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pBalDataParams;                                                                     /**< Pointer to the lower layers parameter structure. */
    void * pReaderHalDataParams;                                                                /**< Pointer to the hal data params of a reader. NULL in case of X-Mode. */
    void * pKeyStoreDataParams;                                                                 /**< Pointer to the KeyStore used for Host Authentication. */
    void * pENCCryptoDataParams;                                                                /**< Pointer to the ENC crypto layers parameter structure. */
    void * pMACCryptoDataParams;                                                                /**< Pointer to the MAC crypto layers parameter structure. */
    void * pCryptoRngDataParams;                                                                /**< Pointer to the parameter structure of the CryptoRng layer. */
    uint32_t Cmd_Ctr;                                                                           /**< Command counter for Secure Messaging. */
    uint8_t bHostMode;                                                                          /**< Indicates, if we have AV1 or AV2 mode. */
    uint8_t bAuthType;                                                                          /**< The current Authentication Type used for SM. */
    uint8_t bPendingEncCmdData[16];                                                             /**< Command Data pending for encryption. */
    uint8_t bPendingEncCmdDataLength;                                                           /**< Length of pending command data to encrypt. */
    uint8_t bPendingCmdIv[16];                                                                  /**< Initvector for pending CMD-Data encryption. */
    uint8_t bPendingMacCmdData[16];                                                             /**< Command Data pending for MACing. */
    uint8_t bPendingMacCmdDataLength;                                                           /**< Length of pending command data to MAC. */
    uint8_t bPendingCmdMac[16];                                                                 /**< Initvector for pending CMD-Data MACing. */
    uint8_t bPendingMacRespData[16];                                                            /**< Response Data pending for MACing. */
    uint8_t bPendingMacRespDataLength;                                                          /**< Length of pending receive data to MAC. */
    uint8_t bPendingRespMac[16];                                                                /**< Intermediate response MAC. */
    uint8_t bPendingRespIv[16];                                                                 /**< Initvector for pending receive-Data decryption. */
    uint8_t bPendingRespData[16];                                                               /**< Pending (unreturned) response data. */
    uint8_t bPendingRespDataLength;                                                             /**< Length of pending response data. */
    uint8_t bCmdSM;                                                                             /**< Type of secure messaging for current command. */
    uint8_t bRespSM;                                                                            /**< Type of secure messaging for current response. */
    uint8_t bCommandChaining;                                                                   /**< Whether command chaining is active or not. */
    uint8_t bResponseChaining;                                                                  /**< Whether response chaining is active or not. */
    uint8_t bUid[PHHAL_HW_SAMAV2_HC_SAM_UID_SIZE];                                              /**< SAM UID. */
    uint8_t bMasterKeyCmacMode;                                                                 /**< Whether CMAC mode is enabled in the master key. */
    uint8_t bChangeEntryPlain;                                                                  /**< Whether the next ChangeKeyEntry / ChangeKUCEntry is plain. */
    uint8_t bOpMode;                                                                            /**< Operation mode; e.g. #PHHAL_HW_SAMAV2_OPMODE_NON_X. */
    uint8_t bLogicalChannel;                                                                    /**< Logical channel number to select for Authenthication*/
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes for exchange within the transmit buffer. */
    uint16_t wTxBufLen_Cmd;                                                                     /**< Number of valid bytes for other commands within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint16_t wCfgShadow[PHHAL_HW_SAMAV2_SHADOW_COUNT];                                          /**< Configuration shadow; Stores configuration for current cardtype. */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint16_t wAdditionalInfo;                                                                   /**< storage for additional error information. */
    uint16_t wTimingMode;                                                                       /**< Disables/Enables time measurement. */
    uint32_t dwTimingUs;                                                                        /**< Current timing value. */
    uint8_t bMifareCryptoDisabled;                                                              /**< Contains information about MIFARE Crypto enabled state. */
    uint8_t b2K3DESEncOption;                                                                   /**< Option for single DES and 2 Key Triple DES encryption keys. Can be set either to #PHHAL_HW_SAMAV2_DES_OPTION_DESFIRE4, #PHHAL_HW_SAMAV2_DES_OPTION_ISO_CRC16 or #PHHAL_HW_SAMAV2_DES_OPTION_ISO_CRC32 */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint8_t bDisableNonXCfgMapping;                                                             /**< Storage for #PHHAL_HW_SAMAV2_CONFIG_DISABLE_NONX_CFG_MAPPING setting. */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC*/
} phhalHw_SamAV2_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * The operation mode used is only dependend on the \b pReaderHalDataParams parameter.\n
 * If it's value is \b NULL X-Mode operation is performed, otherwise the HAL operates in Non-X Mode.\n
 * <b>Non-X Mode Specific:</b> It is \b not neccessary to do any (non-specific) calls to the linked Reader-HAL except an \b Init(),\n
 * the SamAV2 HAL will take over complete control of the linked Reader.\n
 * Specific initialisation like #PHHAL_HW_CONFIG_BAL_CONNECTION or #PHHAL_HW_CONFIG_SERIAL_BITRATE etc. though may be neccessary depending on the used HAL.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_SamAV2_Init(
        phhalHw_SamAV2_DataParams_t * pDataParams,                                              /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        void * pReaderHalDataParams,                                                            /**< [In] Pointer to a Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended. */
        void * pKeyStoreDataParams,                                                             /**< [In] Pointer to the KeyStore used for Host Authentication. */
        void * pCryptoENCDataParams,                                                            /**< [In] Pointer to the ENC crypto layers parameter structure. */
        void * pCryptoMACDataParams,                                                            /**< [In] Pointer to the MAC crypto layers parameter structure. */
        void * pCryptoRngDataParams,                                                            /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
        uint8_t bOpMode,                                                                        /**< [In] The desired operation mode; e.g. #PHHAL_HW_SAMAV2_OPMODE_NON_X. */
        uint8_t bLogicalChannel,                                                                /**< [In] The desired logical channel for this HAL. */
        uint8_t* pTxBuffer,                                                                     /**< [In] Pointer to global transmit buffer. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t* pRxBuffer,                                                                     /**< [In] Pointer to global receive buffer. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. */
    );

/**
 * \brief Detect UID, AV1/AV2 mode and HostAuth settings.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phhalHw_SamAV2_DetectMode(
        phhalHw_SamAV2_DataParams_t * pDataParams                                               /**< [In] Pointer to this layer's parameter structure. */
    );

#ifdef NXPBUILD__PHHAL_HW_RFIDLITE
#include <phhalHw_SamAV2_Rc523_Reg.h>
#include <phhalHw_SamAV2_Rc663_Reg.h>
#endif /* NXPBUILD__PHHAL_HW_RFIDLITE */
#include <phhalHw_SamAV2_Cmd.h>

/**
 * end of group phhalHw_SamAV2
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_SAMAV2 */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3

/**
 * \defgroup phhalHw_SamAV3 Component : SamAV3
 * \brief SamAV3 Reader HAL
 *
 * Also supports AV2 backwards compatibility mode.
 * @{
 */

#define PHHAL_HW_SAMAV3_ID                                                              0x10    /**< ID for SamAV3 HAL component. */

#define PHHAL_HW_SAMAV3_RESERVED_TX_BUFFER_LEN                                          6U      /**< Amount of needed and reserved memory for the protocol overhead. */
#define PHHAL_HW_SAMAV3_RESERVED_RX_BUFFER_LEN                                          2U      /**< Amount of needed and reserved memory for the protocol overhead. */

/** \name Operation modes */
/** @{ */
#define PHHAL_HW_SAMAV3_OPMODE_NON_X                                                    0x00    /**< Non-X operation mode. */
#define PHHAL_HW_SAMAV3_OPMODE_X_RC523                                                  0x01    /**< X-Mode operation (using Rc523). */
#define PHHAL_HW_SAMAV3_OPMODE_X_RC663                                                  0x02    /**< X-Mode operation (using Rc663). */
/** @} */

/** \name Host-Communication modes */
/** @{ */
#define PHHAL_HW_SAMAV3_HC_AV2_MODE                                                     0x02U   /**< Sam AV2 hardware is used */
#define PHHAL_HW_SAMAV3_HC_AV3_MODE                                                     0x03U   /**< Sam AV3 hardware is used with sam in Un-Activated or Activated state */
/** @} */

/** \name Configs */
/** @{ */
#define PHHAL_HW_SAMAV3_CONFIG_HOSTMODE                     (PH_CONFIG_CUSTOM_BEGIN + 0)        /**< Get / Set the Hostmode; (e.g. #PHHAL_HW_SAMAV3_HC_AV2_MODE). */
#define PHHAL_HW_SAMAV3_CONFIG_DISABLE_NONX_CFG_MAPPING     (PH_CONFIG_CUSTOM_BEGIN + 1)        /**< Disables the mapping of standard-configs to the Reader HAL if set to #PH_ON;
                                                                                                 * Default is #PH_OFF; Only applicable in NonX-Mode.
                                                                                                 */
/** @} */

/** \name phhalHw_SamAV3 Custom Exchange Option Bits */
/** @{ */
#define PHHAL_HW_SAMAV3_EXCHANGE_NO_ENCIPHERING_BIT                                 0x0010U     /**< Suppress MIFARE Crypto enciphering of transmitted data. */
#define PHHAL_HW_SAMAV3_EXCHANGE_NO_DECIPHERING_BIT                                 0x0020U     /**< Suppress deciphering of received MIFARE Crypto streams. */
/** @} */

/** \brief SamAV3 HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pBalDataParams;                                                                     /**< Pointer to the lower layers parameter structure. */
    void * pReaderHalDataParams;                                                                /**< Pointer to the hal data params of a reader. NULL in case of X-Mode. */
    void * pKeyStoreDataParams;                                                                 /**< Pointer to the KeyStore used for Host Authentication. */
    void * pENCCryptoDataParams;                                                                /**< Pointer to the ENC crypto layers parameter structure. */
    void * pMACCryptoDataParams;                                                                /**< Pointer to the MAC crypto layers parameter structure. */
    void * pCryptoRngDataParams;                                                                /**< Pointer to the parameter structure of the CryptoRng layer. */
    void * pPLUpload_ENCCryptoDataParams;                                                       /**< pointer to the ENC crypto layers parameter structure. This will be used for ProgrammableLogic feature only. */
    void * pPLUpload_MACCryptoDataParams;                                                       /**< pointer to the MAC crypto layers parameter structure. This will be used for ProgrammableLogic feature only. */
    uint32_t Cmd_Ctr;                                                                           /**< Command counter for Secure Messaging. */
    uint8_t bHostMode;                                                                          /**< Indicates #PHHAL_HW_SAMAV3_HC_AV2_MODE or #PHHAL_HW_SAMAV3_HC_AV3_MODE Host-Communication modes. */
    uint8_t bAuthType;                                                                          /**< The current Authentication Type used for SM. */
    uint8_t bKeyNo;                                                                             /**< Store the current authentication key. */
    uint8_t bPendingEncCmdData[16];                                                             /**< Command Data pending for encryption. */
    uint8_t bPendingEncCmdDataLength;                                                           /**< Length of pending command data to encrypt. */
    uint8_t bPendingCmdIv[16];                                                                  /**< Initvector for pending CMD-Data encryption. */
    uint8_t bPendingMacCmdData[16];                                                             /**< Command Data pending for MACing. */
    uint8_t bPendingMacCmdDataLength;                                                           /**< Length of pending command data to MAC. */
    uint8_t bPendingCmdMac[16];                                                                 /**< Initvector for pending CMD-Data MACing. */
    uint8_t bPendingMacRespData[16];                                                            /**< Response Data pending for MACing. */
    uint8_t bPendingMacRespDataLength;                                                          /**< Length of pending receive data to MAC. */
    uint8_t bPendingRespMac[16];                                                                /**< Intermediate response MAC. */
    uint8_t bPendingRespIv[16];                                                                 /**< Initvector for pending receive-Data decryption. */
    uint8_t bPendingRespData[16];                                                               /**< Pending (unreturned) response data. */
    uint8_t bPendingRespDataLength;                                                             /**< Length of pending response data. */
    uint8_t bCmdSM;                                                                             /**< Type of secure messaging for current command. */
    uint8_t bRespSM;                                                                            /**< Type of secure messaging for current response. */
    uint8_t bCommandChaining;                                                                   /**< Whether command chaining is active or not. */
    uint8_t bResponseChaining;                                                                  /**< Whether response chaining is active or not. */
    uint8_t bUid[0x07U];                                                                        /**< SAM UID. */
    uint8_t bMasterKeyCmacMode;                                                                 /**< Whether CMAC mode is enabled in the master key. */
    uint8_t bOpMode;                                                                            /**< Operation mode. One of the below values
                                                                                                 *      \arg #PHHAL_HW_SAMAV3_OPMODE_NON_X
                                                                                                 *      \arg #PHHAL_HW_SAMAV3_OPMODE_X_RC523
                                                                                                 *      \arg #PHHAL_HW_SAMAV3_OPMODE_X_RC663
                                                                                                 */
    uint8_t bLogicalChannel;                                                                    /**< Logical channel number to select for Authenthication*/
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes for exchange within the transmit buffer. */
    uint16_t wTxBufLen_Cmd;                                                                     /**< Number of valid bytes for other commands within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint16_t wCfgShadow[0x000FU];                                                               /**< Configuration shadow; Stores configuration for current cardtype. */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint16_t wTimingMode;                                                                       /**< Disables/Enables time measurement. */
    uint32_t dwTimingUs;                                                                        /**< Current timing value. */
    uint8_t bMifareCryptoDisabled;                                                              /**< Contains information about MIFARE Crypto enabled state. */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint8_t bDisableNonXCfgMapping;                                                             /**< Storage for #PHHAL_HW_SAMAV3_CONFIG_DISABLE_NONX_CFG_MAPPING setting. */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC*/
    uint8_t * pPLUploadBuf;                                                                     /**< Buffer to store the complete segment information of PLUpload command. The buffer should have a memory size equivalent to the complete PLUpload Segment(s) size. */
    uint16_t wPLUploadBufLen;                                                                   /**< The size of bytes available in PLUploadBuff buffer. */
    uint8_t aPLUploadSessMAC0[32];                                                              /**< Programmable Logic Initial session key for macing the data. */
    uint8_t bPLUploadKeyType;                                                                   /**< The current key type used for crypto operations. */
} phhalHw_SamAV3_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * The operation mode used is only dependend on the \b pReaderHalDataParams parameter.\n
 * If it's value is \b NULL X-Mode operation is performed, otherwise the HAL operates in Non-X Mode.\n
 * <b>Non-X Mode Specific:</b> It is \b not neccessary to do any (non-specific) calls to the linked Reader-HAL except an \b Init(),\n
 * the SamAV3 HAL will take over complete control of the linked Reader.\n
 * Specific initialisation like #PHHAL_HW_CONFIG_BAL_CONNECTION or #PHHAL_HW_CONFIG_SERIAL_BITRATE etc. though may be neccessary depending on the used HAL.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_SamAV3_Init(
        phhalHw_SamAV3_DataParams_t * pDataParams,                                              /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        void * pReaderHalDataParams,                                                            /**< [In] Pointer to a Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended. */
        void * pKeyStoreDataParams,                                                             /**< [In] Pointer to the KeyStore used for Host Authentication. */
        void * pCryptoENCDataParams,                                                            /**< [In] Pointer to the ENC crypto layers parameter structure. */
        void * pCryptoMACDataParams,                                                            /**< [In] Pointer to the MAC crypto layers parameter structure. */
        void * pCryptoRngDataParams,                                                            /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
        void * pPLUpload_CryptoENCDataParams,                                                   /**< Pointer to the ENC crypto layers parameter structure. This will be used for ProgrammableLogic feature only. */
        void * pPLUpload_CryptoMACDataParams,                                                   /**< Pointer to the MAC crypto layers parameter structure. This will be used for ProgrammableLogic feature only. */
        uint8_t bOpMode,                                                                        /**< [In] The desired operation mode
                                                                                                 *          \arg #PHHAL_HW_SAMAV3_OPMODE_NON_X.
                                                                                                 *          \arg #PHHAL_HW_SAMAV3_OPMODE_X_RC523.
                                                                                                 *          \arg #PHHAL_HW_SAMAV3_OPMODE_X_RC663.
                                                                                                 */
        uint8_t bLogicalChannel,                                                                /**< [In] The desired logical channel for this HAL. */
        uint8_t* pTxBuffer,                                                                     /**< [In] Pointer to global transmit buffer. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t* pRxBuffer,                                                                      /**< [In] Pointer to global receive buffer. */
        uint16_t wRxBufSize,                                                                    /**< [In] Size of the global receive buffer. */
        uint8_t* pPLUploadBuf                                                                   /**< [In] Pointer to global PLUpload buffer. */
    );

/**
 * \brief Detect UID, AV2/AV3 mode and HostAuth settings.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phhalHw_SamAV3_DetectMode(
        phhalHw_SamAV3_DataParams_t * pDataParams                                               /**< [In] Pointer to this layer's parameter structure. */
    );

#include <phhalHw_SamAV3_Cmd.h>

/**
 * end of group phhalHw_SamAV3
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

#ifdef NXPBUILD__PHHAL_HW_SAM

/**
 * \addtogroup phhalHw_Sam
 * \brief Specific SAM HAL component for SAM (AV4 and future SAM's)
 *  - Supports New features of SAM AV4.
 *  - Supports AV3 backwards compatibility mode.
 *  - Post \ref phhalHw_Sam_Init "Initialization of SAM", it's must to call \ref phhalHw_Sam_DetectMode "Detect Mode"
 *    interface to know the current state of SAM. Whether it's is in one of the following modes
 *      - \ref PHHAL_HW_SAM_HC_AV3_MODE "AV3 Activated or Un-Activated" mode
 *      - \ref PHHAL_HW_SAM_HC_AV4_MODE "AV4 Activated or Un-Activated" mode
 *  - SAM supports 4 logical channels.
 *      - Each Logical channel can have it's own authentication and session keys, and e.g. can have
 *        multiple PICC authentications at a time, on separate logical channels, basically running 2 independent
 *        secure messaging channels.
 *      - Other example would be that one logical channel for PICC communication and authentication, second one for an Offline
 *        Crypto key, that is used to additionally encrypt data that is written to a PICC with a separate key.
 *  - This component returns mapped error code for the actual error codes received from SAM. To get the actual error code
 *    returned by SAM call \ref phhalHw_GetConfig "Get Config" with configuration identifier as
 *    \ref PHHAL_HW_SAM_CONFIG_GET_SAM_ERROR_CODE "Actual SAM Error".
 *  - SAM Supports communication command exchange via ISO7816 T = 1 and I2C communication. To configure the communication type
 *    call \ref phhalHw_SetConfig "Set Config" with configuration identifier as \ref PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE
 *    "Communication Type"
 *  - To configure I2C Timeout, I2C BitRate and I2C Slave Address, call the below identifiers using \ref phhalHw_SetConfig
 *    "Set Config" interface
 *      - \ref PHHAL_HW_CONFIG_I2C_SLAVE_ADDRESS "slave Address"
 *      - \ref PHHAL_HW_CONFIG_I2C_TIMEOUT_MS "Timeout in ms"
 *      - \ref PHHAL_HW_CONFIG_I2C_BITRATE "BitRate". Supported ones are,
 *          - \ref PHHAL_HW_CONFIG_I2C_BITRATE_STANDARD_MODE "100 KBits/s"
 *          - \ref PHHAL_HW_CONFIG_I2C_BITRATE_FAST_MODE "400 KBits/s"
 *          - \ref PHHAL_HW_CONFIG_I2C_BITRATE_FAST_PLUS_MODE "1 MBits/s"
 *          - \ref PHHAL_HW_CONFIG_I2C_BITRATE_HIGH_SPEED_MODE "3.4 MBits/s"
 *  - Default 0x2B will be used as Slave Address and 200ms timeout for I2C Communication.
 * @{
 */

#define PHHAL_HW_SAM_ID                                                                 0x13U   /**< ID for SAM (AV4 and future SAM's) HAL component. */

/** \defgroup phhalHw_Sam_CommonDefs Global Defines
 * \brief Macro Definitions used across the whole SAM implementation
 * @{
 */

/** \defgroup phhalHw_Sam_Defines_ReservedLen Reserved Length
 * \brief Macro Definitions used Transmit and Response buffer reserved length
 * @{
 */
#define PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN                                             6U      /**< Amount of needed and reserved memory for the protocol overhead. */
#define PHHAL_HW_SAM_RESERVED_RX_BUFFER_LEN                                             2U      /**< Amount of needed and reserved memory for the protocol overhead. */
#define PHHAL_HW_SAM_ISO7816_EXCHANGE_RESPONSE_MAX                                      0xF8U   /**< Max Length of Transparent Exchange Response */
/**
 * end of group phhalHw_Sam_Defines_ReservedLen
 * @}
 */

/** \defgroup phhalHw_Sam_Defines_HCMode Host Communication Mode
 * \brief Macro Definitions used for Host communication mode
 * @{
 */
#define PHHAL_HW_SAM_HC_AV3_MODE                                                        0x03U  /**< Sam AV3 hardware is used with SAM in Un-Activated or Activated state */
#define PHHAL_HW_SAM_HC_AV4_MODE                                                        0x04U  /**< Sam AV4 hardware is used with SAM in Un-Activated or Activated state */
/**
 * end of group phhalHw_Sam_Defines_HCMode
 * @}
 */

/** \defgroup phhalHw_Sam_Defines_OpMode Operation Mode
 * \brief Macro Definitions used for operation mode. To be used during \ref phhalHw_Sam_Init
 * "Initialization of SAM" component.
 * @{
 */
#define PHHAL_HW_SAM_OPMODE_NON_X                                                       0x00    /**< Non-X operation mode. */
#define PHHAL_HW_SAM_OPMODE_X_RC523                                                     0x01U   /**< X-Mode operation (using Rc523). */
#define PHHAL_HW_SAM_OPMODE_X_RC663                                                     0x02U   /**< X-Mode operation (using Rc663). */
/**
 * end of group phhalHw_Sam_Defines_OpMode
 * @}
 */

/** \defgroup phhalHw_Sam_Defines_LC Logical Channel
 * \brief Macro Definitions used for Logical Channel. To be used during \ref phhalHw_Sam_Init
 * "Initialization of SAM" component.
 * @{
 */
#define PHHAL_HW_SAM_LOGICAL_CHANNEL_0                                                  0x00    /**< Logical Channel zero. */
#define PHHAL_HW_SAM_LOGICAL_CHANNEL_1                                                  0x01U   /**< Logical Channel one. */
#define PHHAL_HW_SAM_LOGICAL_CHANNEL_2                                                  0x02U   /**< Logical Channel two. */
#define PHHAL_HW_SAM_LOGICAL_CHANNEL_3                                                  0x03U   /**< Logical Channel three. */
/**
 * end of group phhalHw_Sam_Defines_LC
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_LFI Last Frame Indicator
 * \brief Sam Global macro Last Frame Indicator values that are used for most of the commands.
 * @{
 */
#define PHHAL_HW_SAM_ISO7816_LAST_FRAME                                                 0x00U   /**< Indication last frame in a cipher sequence */
#define PHHAL_HW_SAM_ISO7816_CHAINED_FRAME                                              0xAFU   /**< Indication intermediate frame in a cipher sequence */
/**
 * end of group phhalHw_Sam_Cmd_LFI
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_SessionKey SessionKey
 * \brief Sam Global macro Session Key values that are used for most of the commands.
 * @{
 */
#define PHHAL_HW_CMD_SAM_SESSION_KEY_ENC                                                0U      /**< Macro to represent the mode to generate Session ENC keys. */
#define PHHAL_HW_CMD_SAM_SESSION_KEY_MAC                                                1U      /**< Macro to represent the mode to generate Session MAC keys. */
/**
 * end of group phhalHw_Sam_Cmd_SessionKey
 * @}
 */

/** \defgroup phhalHw_Sam_Defines_Config Configuration
 * \brief Macro Definitions for \ref phhalHw_GetConfig "Get Config" and \ref phhalHw_SetConfig "Set Config" interfaces.
 * Refer Configuration under Generic HAL for all supported configuration identifiers.
 * @{
 */
#define PHHAL_HW_SAM_CONFIG_HOSTMODE                            (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Get / Set the HostMode; (e.g. \ref PHHAL_HW_SAM_HC_AV4_MODE "AV4" ). */
#define PHHAL_HW_SAM_CONFIG_DISABLE_NONX_CFG_MAPPING            (PH_CONFIG_CUSTOM_BEGIN + 1U)   /**< Disables the mapping of standard-configs to the Reader HAL if set to #PH_ON;
                                                                                                 *   Default is #PH_OFF; Only applicable in NonX-Mode.
                                                                                                 */
#define PHHAL_HW_SAM_CONFIG_GET_SAM_ERROR_CODE                  (PH_CONFIG_CUSTOM_BEGIN + 2U)   /**< Get the Actual Error Code returned by Sam. */
/**
 * end of group phhalHw_Sam_Defines_Config
 * @}
 */

/**
 * end of group phhalHw_Sam_CommonDefs
 * @}
 */

 /** \brief Data structure for HAL SAM (AV4 and future SAM's) layer implementation  */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pBalDataParams;                                                                     /**< Pointer to the lower layers parameter structure. */
    void * pReaderHalDataParams;                                                                /**< Pointer to the HAL data params of a reader. NULL in case of X-Mode. */
    void * pKeyStoreDataParams;                                                                 /**< Pointer to the KeyStore used for Host Authentication. */
    void * pENCCryptoDataParams;                                                                /**< Pointer to the ENC crypto layers parameter structure. */
    void * pMACCryptoDataParams;                                                                /**< Pointer to the MAC crypto layers parameter structure. */
    void * pCryptoRngDataParams;                                                                /**< Pointer to the parameter structure of the CryptoRng layer. */
    void * pPLUpload_ENCCryptoDataParams;                                                       /**< Pointer to the ENC crypto layers parameter structure. This will be used for
                                                                                                *    ProgrammableLogic feature only.
                                                                                                */
    void * pPLUpload_MACCryptoDataParams;                                                       /**< Pointer to the MAC crypto layers parameter structure. This will be used for
                                                                                                 * ProgrammableLogic feature only.
                                                                                                 */

    uint32_t wCmd_Ctr;                                                                          /**< Command counter for Secure Messaging. */
    uint8_t bHostMode;                                                                          /**< Indicates Host-Communication modes. One of the following,
                                                                                                 *      - \ref PHHAL_HW_SAM_HC_AV3_MODE "AV3"
                                                                                                 *      - \ref PHHAL_HW_SAM_HC_AV4_MODE "AV4"
                                                                                                 */
    uint8_t bAuthType;                                                                          /**< The current Authentication Type used for SM. */
    uint8_t bKeyNo;                                                                             /**< Store the current authentication key. */
    uint8_t bPendingEncCmdData[16U];                                                            /**< Command Data pending for encryption. */
    uint8_t bPendingEncCmdDataLength;                                                           /**< Length of pending command data to encrypt. */
    uint8_t bPendingCmdIv[16U];                                                                 /**< InitVector for pending CMD-Data encryption. */
    uint8_t bPendingMacCmdData[16U];                                                             /**< Command Data pending for MACing. */
    uint8_t bPendingMacCmdDataLength;                                                           /**< Length of pending command data to MAC. */
    uint8_t bPendingCmdMac[16U];                                                                /**< InitVector for pending CMD-Data MACing. */
    uint8_t bPendingMacRespData[16U];                                                           /**< Response Data pending for MACing. */
    uint8_t bPendingMacRespDataLength;                                                          /**< Length of pending receive data to MAC. */
    uint8_t bPendingRespMac[16U];                                                               /**< Intermediate response MAC. */
    uint8_t bPendingRespIv[16U];                                                                /**< InitVector for pending receive-Data decryption. */
    uint8_t bPendingRespData[16U];                                                              /**< Pending (unreturned) response data. */
    uint8_t bPendingRespDataLength;                                                             /**< Length of pending response data. */
    uint8_t bCmdSM;                                                                             /**< Type of secure messaging for current command. */
    uint8_t bRespSM;                                                                            /**< Type of secure messaging for current response. */
    uint8_t bCommandChaining;                                                                   /**< Whether command chaining is active or not. */
    uint8_t bResponseChaining;                                                                  /**< Whether response chaining is active or not. */
    uint8_t bUid[0x07U];                                                                        /**< SAM Unique Identifier. */
    uint8_t bOpMode;                                                                            /**< Operation mode. One of the below values
                                                                                                 *      - \ref PHHAL_HW_SAM_OPMODE_NON_X "Non-X Mode or S Mode"
                                                                                                 *      - \ref PHHAL_HW_SAM_OPMODE_X_RC523 "X Mode with RC523 Reader IC"
                                                                                                 *      - \ref PHHAL_HW_SAM_OPMODE_X_RC663 "X Mode with RC663 Reader IC"
                                                                                                 */
    uint8_t bLogicalChannel;                                                                    /**< Logical channel number to select for Authentication
                                                                                                 *      - \ref PHHAL_HW_SAM_LOGICAL_CHANNEL_0 "Logical Channel 0"
                                                                                                 *      - \ref PHHAL_HW_SAM_LOGICAL_CHANNEL_1 "Logical Channel 1"
                                                                                                 *      - \ref PHHAL_HW_SAM_LOGICAL_CHANNEL_2 "Logical Channel 2"
                                                                                                 *      - \ref PHHAL_HW_SAM_LOGICAL_CHANNEL_3 "Logical Channel 3"
                                                                                                 */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes for exchange within the transmit buffer. */
    uint16_t wTxBufLen_Cmd;                                                                     /**< Number of valid bytes for other commands within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b pTxBuffer
                                                                                                 *   equals \b pRxBuffer).
                                                                                                 */
    uint16_t wCfgShadow[0x000FU];                                                               /**< Configuration shadow; Stores configuration for current CardType. */
    uint8_t bCardType;                                                                          /**< Type of card for which the Hal is configured for. */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value
                                                                                                 *      - \ref PHHAL_HW_TIME_MICROSECONDS "Micro-Seconds"
                                                                                                 *      - \ref PHHAL_HW_TIME_MILLISECONDS "Milli-Seconds"
                                                                                                 */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint16_t wErrorCode;                                                                        /**< Actual Error Code received from SAM. */
    uint16_t wTimingMode;                                                                       /**< Disables/Enables time measurement. */
    uint32_t dwTimingUs;                                                                        /**< Current timing value. */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint8_t bDisableNonXCfgMapping;                                                             /**< Storage for #PHHAL_HW_SAM_CONFIG_DISABLE_NONX_CFG_MAPPING setting. */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC*/
    uint8_t * pPLUploadBuf;                                                                     /**< Buffer to store the complete segment information of PLUpload command.
                                                                                                 *   The buffer should have a memory size equivalent to the complete PLUpload Segment(s) size.
                                                                                                 */
    uint16_t wPLUploadBufLen;                                                                   /**< The size of bytes available in PLUploadBuff buffer. */
    uint8_t aPLUploadSessMAC0[32];                                                              /**< Programmable Logic Initial session key for macing the data. */
    uint8_t bPLUploadKeyType;                                                                   /**< The current key type used for crypto operations. */
    uint8_t * pCmdBuff;                                                                         /**< Pointer to global command buffer. This will be used to frame SAM Command. */
    uint16_t wCmdBuffSize;                                                                      /**< Size of the global command buffer. */
} phhalHw_Sam_DataParams_t;

/**
 * \brief Interface to initialize the SAM (AV4 and future SAM's) component.
 *  - \b pTxBuffer should be Size + \ref PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN "Reserved Tx Buffer Len".
 *    If \b pTxBuffer is 256, then it should be 256 + \ref PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN
 *    "Reserved Tx Buffer Len".
 *  - \b pRxBuffer should be Size + \ref PHHAL_HW_SAM_RESERVED_RX_BUFFER_LEN "Reserved Rx Buffer Len".
 *    If \b pRxBuffer is 256, then it should be 256 + \ref PHHAL_HW_SAM_RESERVED_RX_BUFFER_LEN
 *    "Reserved Rx Buffer Len".
 *  - The operation mode used is only depended on the \b pReaderHalDataParams parameter.
 *      - If \b pReaderHalDataParams parameter value is \b NULL X-Mode operation is performed, otherwise
 *        the HAL operates in S-Mode (Non-X) Mode.
 *      - <b>Non-X Mode Specific:</b> It is \b not necessary to do any (non-specific) calls to the linked
 *        Reader-HAL except an \b Init(), the Sam HAL will take over complete control of the linked Reader.
 *  - In case of S-Mode communication,
 *      - Pegoda - 3 reader (CLRD730): \b pBalDataParams parameter should be initialized with \ref phbalReg_Sam
 *        "BAL SAM" component, \ref phbalReg_Sam "BAL SAM's" \b pLowerBalDataParams parameter should be linked
 *        to one of the following,
 *          - \ref phbalReg_PcscWin "BAL PCSCWin" for PCSC interface.
 *          - \ref phbalReg_SerialWin "BAL SerialWin" for Serial (COMPORT) interface.
 *      - Pegoda - 2 reader (MFEV710)
 *          - SAM inserted in Reader slot: \b pBalDataParams parameter should be initialized with \ref phbalReg_Sam
 *            "BAL SAM" component and \ref phbalReg_Sam "BAL SAM's" \b pLowerBalDataParams parameter should be linked
 *            to \ref phbalReg_PcscWin "BAL PCSCWin".
 *          - SAM inserted in other contact based readers: \b pBalDataParams parameter should be initialized with
 *            \ref phbalReg_PcscWin "BAL PCSCWin" component.
 *  - In case of X-Mode communication for Pegoda - 2 reader (MFEV710, SAM inserted in Reader slot), \b pBalDataParams
 *    parameter should be initialized with \ref phbalReg_PcscWin "BAL PCSCWin" component.
 *  - SAM Supports communication command exchange via ISO7816 T = 1 and I2C communication. To configure the communication type
 *    call \ref phhalHw_SetConfig "Set Config" with configuration identifier as \ref PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE
 *    "Communication Type"
 *  - To configure I2C Timeout, I2C BitRate and I2C Slave Address, call the below identifiers using \ref phhalHw_SetConfig
 *    "Set Config" interface
 *      - \ref PHHAL_HW_CONFIG_I2C_SLAVE_ADDRESS "slave Address"
 *      - \ref PHHAL_HW_CONFIG_I2C_TIMEOUT_MS "Timeout in ms"
 *      - \ref PHHAL_HW_CONFIG_I2C_BITRATE "BitRate". Supported ones are,
 *          - \ref PHHAL_HW_CONFIG_I2C_BITRATE_STANDARD_MODE "100 KBits/s"
 *          - \ref PHHAL_HW_CONFIG_I2C_BITRATE_FAST_MODE "400 KBits/s"
 *          - \ref PHHAL_HW_CONFIG_I2C_BITRATE_FAST_PLUS_MODE "1 MBits/s"
 *          - \ref PHHAL_HW_CONFIG_I2C_BITRATE_HIGH_SPEED_MODE "3.4 MBits/s"
 *  - Default 0x2B will be used as Slave Address and 200ms timeout for I2C Communication.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *           - If the size of Sam context not matching with \b wSizeOfDataParams.
 *           - \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER
 *           - If the DataParams (other than \b pDataParams) are NULL.
 *           - If the pointers are NULL.
 *           - If \b wTxBufSize <= \ref PHHAL_HW_SAM_RESERVED_TX_BUFFER_LEN "Reserved Transmit Buffer Length"
 *           - If \b wRxBufSize <= \ref PHHAL_HW_SAM_RESERVED_RX_BUFFER_LEN "Reserved Response Buffer Length"
 *           - If \b bOpMode is not one of the following
 *               - \ref PHHAL_HW_SAM_OPMODE_NON_X "Non-X Mode or S Mode"
 *               - \ref PHHAL_HW_SAM_OPMODE_X_RC523 "X Mode with RC523 Reader IC"
 *               - \ref PHHAL_HW_SAM_OPMODE_X_RC663 "X Mode with RC663 Reader IC"
 *           - If \b bOpMode = \ref PHHAL_HW_SAM_OPMODE_NON_X "S Mode" and \b pReaderHalDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Init(
        phhalHw_Sam_DataParams_t * pDataParams,                                                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        void * pReaderHalDataParams,                                                            /**< [In] Pointer to a Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended. */
        void * pKeyStoreDataParams,                                                             /**< [In] Pointer to the KeyStore used for Host Authentication interfaces. */
        void * pCryptoENCDataParams,                                                            /**< [In] Pointer to the ENC crypto layers parameter structure. */
        void * pCryptoMACDataParams,                                                            /**< [In] Pointer to the MAC crypto layers parameter structure. */
        void * pCryptoRngDataParams,                                                            /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
        void * pPLUpload_CryptoENCDataParams,                                                   /**< Pointer to the ENC crypto layers parameter structure. This will be used for
                                                                                                 *   ProgrammableLogic feature only.
                                                                                                 */
        void * pPLUpload_CryptoMACDataParams,                                                   /**< Pointer to the MAC crypto layers parameter structure. This will be used for
                                                                                                 * ProgrammableLogic feature only.
                                                                                                 */
        uint8_t bOpMode,                                                                        /**< [In] The desired operation mode. One of the below values
                                                                                                 *          - \ref PHHAL_HW_SAM_OPMODE_NON_X "Non-X Mode or S Mode"
                                                                                                 *          - \ref PHHAL_HW_SAM_OPMODE_X_RC523 "X Mode with RC523 Reader IC"
                                                                                                 *          - \ref PHHAL_HW_SAM_OPMODE_X_RC663 "X Mode with RC663 Reader IC"
                                                                                                 */
        uint8_t bLogicalChannel,                                                                /**< [In] The desired logical channel for this HAL.
                                                                                                 *          - \ref PHHAL_HW_SAM_LOGICAL_CHANNEL_0 "Logical Channel 0"
                                                                                                 *          - \ref PHHAL_HW_SAM_LOGICAL_CHANNEL_1 "Logical Channel 1"
                                                                                                 *          - \ref PHHAL_HW_SAM_LOGICAL_CHANNEL_2 "Logical Channel 2"
                                                                                                 *          - \ref PHHAL_HW_SAM_LOGICAL_CHANNEL_3 "Logical Channel 3"
                                                                                                 */
        uint8_t* pTxBuffer,                                                                     /**< [In] Buffer for SAM command transmission. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t* pRxBuffer,                                                                     /**< [In] Buffer for SAM command Reception. */
        uint16_t wRxBufSize,                                                                    /**< [In] Size of the global receive buffer. */
        uint8_t* pPLUploadBuf,                                                                  /**< [In] Buffer for Programmable Logic operations. */
        uint8_t * pCmdBuff,                                                                     /**< [In] Pointer to global command buffer. This will be used to frame
                                                                                                 *        SAM Command internally by the library.
                                                                                                 */
        uint16_t wCmdBuffSize                                                                   /**< [In] Size of the global transmit buffer. */
    );

/**
 * \brief Saves UID and mode of SAM (AV3, AV4, future SAM's). This function should be called prior
 * to all other functions even before Host Authentication.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phhalHw_Sam_DetectMode(
        phhalHw_Sam_DataParams_t * pDataParams                                                  /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * end of group phhalHw_Sam
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef NXPBUILD__PHHAL_HW_RD70X

/**
 * \defgroup phhalHw_Rd70x Component : Rd70x
 * \brief RD70x (Pegoda1) Reader HAL
 *
 * It is neccessary to make use of #PHHAL_HW_RD70X_CONFIG_DISABLE_PCDRESET for this reader to support multiple cards.
 * @{
 */

#define PHHAL_HW_RD70X_ID                                                               0x04    /**< ID for Rd70x HAL component. */

/** \name Rd70x specific configs */
/** @{ */
/**<
 * Disable PcdReset call in \ref phhalHw_ApplyProtocolSettings function.
 *
 * Set this to #PH_ON on every HAL instance to prevent the reader from resetting powered cards\n
 * on an \ref phhalHw_ApplyProtocolSettings call.\n
 * Ensure that it is set to #PH_OFF when you set the protocol the first time to have essential parameters set in the firmware.
 */
#define PHHAL_HW_RD70X_CONFIG_DISABLE_PCDRESET  (PH_CONFIG_CUSTOM_BEGIN + 0)
/** @} */

#define PHHAL_HW_RD70X_SHADOW_COUNT                                                 0x000DU     /**< Number of shadowed configurations. */
#define PHHAL_HW_RD70X_RESERVED_BUFFER_LEN                                          32U         /**< Amount of needed and reserved memory for the protocol overhead. */
#define PHHAL_HW_RD70X_FIFOSIZE                                                     256U        /**< Size of RC internal FiFo buffer. */
#define PHHAL_HW_RD70X_DEFAULT_TIMEOUT_MS                                           5U          /**< Default timeout in milliseconds. */

/**
* \brief Rd70x HAL parameter structure
*/
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    phbalReg_Rd70xUsbWin_DataParams_t * pBalDataParams;                                         /**< pointer to the lower layers parameter structure */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
    uint16_t wCfgShadow[PHHAL_HW_RD70X_SHADOW_COUNT];                                           /**< Configuration shadow; Stores configuration for current cardtype. */
    uint16_t wTimingMode;                                                                       /**< Current timing measurement mode. */
    uint32_t dwTimeoutUs;                                                                       /**< TO value to be used with phhalHwRd700PiccExchgBitByte function. */
    uint32_t dwTimingUs;                                                                        /**< Current timing value. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint16_t wTxRxStartBackup;                                                                  /**< Absolute Start-Position for transmission / reception used by buffer preservation mechanism. */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint8_t bSymbolStart;                                                                       /**< Preamble of a frame. */
    uint8_t bSymbolEnd;                                                                         /**< Trailer symbol of a frame. */
    uint8_t bDisablePcdReset;                                                                   /**< Whether to disable PcdReset call in \ref phhalHw_ApplyProtocolSettings function or not. */
    uint8_t bFlags;                                                                             /**< Flags to be used with phhalHwRd700PiccExchgBitByte function. */
    uint8_t bProtBuffer[PHHAL_HW_RD70X_RESERVED_BUFFER_LEN];                                    /**< Protocol buffer. */
    uint8_t bProtBufferLen;                                                                     /**< Length of protocol data. */
    uint16_t wProtRxBufReadPos;                                                                 /**< Read position within \b pRxBuffer for protocol. */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint8_t bTimerControlPc;                                                                    /**< Backup for Timer control PC requirement */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC*/
} phhalHw_Rd70x_DataParams_t;

/**
* \brief Initialise the HAL component.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rd70x_Init(
        phhalHw_Rd70x_DataParams_t * pDataParams,                                               /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        phbalReg_Rd70xUsbWin_DataParams_t * pBalDataParams,                                     /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. */
    );

#include <phhalHw_Rd70x_Reg.h>
#include <phhalHw_Rd70x_Cmd.h>

/**
 * end of group phhalHw_Rd70x
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_RD70X */

#ifdef NXPBUILD__PHHAL_HW_RDCARDSIM

/**
 * \defgroup phhalHw_RdCardSim Component : RdCardSim
 * \brief RdCardSim Reader HAL
 *
 * This simulator works only with the digital simulator the was developed in the MF1C14 project.
 *
 * \b Restriction: This module only supports one card type at the same time.\n
 * To support more cards at the same time a shadow has to be implemented to shadow all configurations.\n
 * The phhalHw_RdCardSim_ApplyProtocolSettings function has to be extended to support the shadow and the default configuration.
 * @{
 */

#define PHHAL_HW_RDCARDSIM_ID                                                           0x05    /**< ID for Reader Card Simulation HAL component. */

/**
* \brief RdCardSim HAL parameter structure
*/
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalDataParams;                                                                      /**< pointer to the lower layers parameter structure */
    void * pKeyStoreDataParams;                                                                 /**< Pointer to the parameter structure of the keyStore layer. */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t bTxLastBits;                                                                        /**< Number of valid bits of the last byte in the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint8_t bRxAlign;                                                                           /**< Desired Bit-Alignment for reception. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
    uint16_t wRx2TxWait;                                                                        /**< Wait time in us for the reader model to wait between the Rx and TX frames. */
    uint8_t bTxBAUD;                                                                            /**< Storage for Tx baud rate setting. */
    uint8_t bRxBAUD;                                                                            /**< Storage for Rx baud rate setting. */
    uint8_t bTxCrc;                                                                             /**< Storage for Tx CRC setting. */
    uint8_t bRxCrc;                                                                             /**< Storage for Rx CRC setting. */
    uint8_t bParity;                                                                            /**< Storage for Parity setttings. */
    uint32_t dwResetWaitTime;                                                                   /**< Storage for reset wait time. */
    uint32_t dwResetRecoveryTime;                                                               /**< Storage for reset recovery time. */
    uint8_t * pIntBuffer;                                                                       /**< Internal buffer used as a scratch buffer. Size must be \b wIntBufferLen*wIntBufferNum. */
    uint16_t wIntBufferLen;                                                                     /**< Total length of internal buffer used as a scratch buffer. */
    uint16_t wIntBufferAlloc;                                                                   /**< Flag to show the status of buffer allocation. */
    uint16_t wTimingMode;                                                                       /**< Current timing measurement mode. */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint32_t dwTimingUs;                                                                        /**< Current timing value. */
    uint8_t bTPactive;                                                                          /**< Is Testpad communication active or not. */
    uint16_t wTimeT1;                                                                           /**< Optinal timeout T1 for Testpad communication. */
    uint16_t wTimeT2;                                                                           /**< Optinal timeout T2 for Testpad communication. */
    uint16_t wTimeT3;                                                                           /**< Optinal timeout T3 for Testpad communication. */
    uint16_t wTimeT4;                                                                           /**< Optinal timeout T4 for Testpad communication. */
    uint16_t wSyncOption;                                                                       /**< Syncronisation option for Testpad communication. */
    uint8_t bEnvEnable;
    uint8_t bC40ModeEnabled;                                                                    /**< Indicates if the C40 mode communication is enabled */
}phhalHw_RdCardSim_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_RdCardSim_Init(
        phhalHw_RdCardSim_DataParams_t * pDataParams,                                           /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        void * pKeyStoreDataParams,                                                             /**< [In] Pointer to the parameter structure of the keyStore layer. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize,                                                                   /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize,                                                                   /**< [In] Size of the global receive buffer. */
        uint8_t * pIntBuffer,                                                                   /**< [In] Internal buffer used as a scratch buffer. */
        uint16_t  wIntBufferLen                                                                 /**< [In] Length of innternal buffer used as a scratch buffer. */
    );

/**
 * end of group phhalHw_RdCardSim
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_RDCARDSIM */

#ifdef NXPBUILD__PHHAL_HW_LICENSEEWRAPPER
/**
 * \defgroup phhalHw_LicenseeWrapper Component : LicenseeWrapper
 * \brief Licensee Wrapper HAL
 *
 * This HAL component is intended to be used with the HAL layer developped by the Licensees
 * @{
 */

#define PHHAL_HW_LICENSEEWRAPPER_ID                                                     0x12    /**< ID for Licensee Wrapper HAL component. */

/** \brief Callback Delegate for \ref phhalHw_LicenseeWrapper_Init */
typedef phStatus_t ( *phhalHw_LicenseeWrapper_Init_t)(
        void ** pContext                                                                        /**< [In] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_LicenseeWrapper_Term */
typedef phStatus_t ( *phhalHw_LicenseeWrapper_Term_t)(
        void * pContext                                                                         /**< [In] User Defined Context. */
    );

/**  \brief Callback Delegate for \ref phhalHw_SetConfig */
typedef phStatus_t ( *phhalHw_LicenseeWrapper_SetConfig_t)(
        uint16_t  wConfig,                                                                      /**< [In] Configuration Identifier. */
        uint16_t wValue,                                                                        /**< [In] Configuration Value. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_GetConfig */
typedef phStatus_t (  *phhalHw_LicenseeWrapper_GetConfig_t)(
        uint16_t wConfig,                                                                       /**< [In] Configuration Identifier. */
        uint16_t * pValue,                                                                      /**< [Out] Configuration Value. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_FieldReset */
typedef phStatus_t (  *phhalHw_LicenseeWrapper_FieldReset_t)(
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_Wait */
typedef phStatus_t (  *phhalHw_LicenseeWrapper_Wait_t)(
        uint8_t bUnit,                                                                          /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
        uint16_t wTimeout,                                                                      /**< [In] Timeout value. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );


/** \brief Callback Delegate for \ref phhalHw_Exchange */
typedef phStatus_t (  *phhalHw_LicenseeWrapper_Exchange_t)(
        uint16_t wOption,                                                                       /**< [In] Option parameter. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Data to transmit. */
        uint16_t wTxLength,                                                                     /**< [In] Length of data to transmit. */
        uint8_t ** ppRxBuffer,                                                                  /**< [Out] Pointer to received data. */
        uint16_t * pRxLength,                                                                   /**< [Out] number of received data bytes. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
);

/** \brief Callback Delegate for \ref phhalHw_ApplyProtocolSettings */
typedef phStatus_t (  *phhalHw_LicenseeWrapper_ApplyProtocolSettings_t)(
        uint8_t bCardType,                                                                      /**< [In] Type of card for which the hal should be configured for. */
        void  * pContext                                                                        /**< [Out] User Defined Context. */
);

/** \brief Licensee HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    phhalHw_LicenseeWrapper_Init_t pCbInit;                                                     /**< Pointer to the Init callback. */
    phhalHw_LicenseeWrapper_Term_t pCbTerm;                                                     /**< Pointer to the Term callback. */
    phhalHw_LicenseeWrapper_SetConfig_t pCbSetConfig;                                           /**< Pointer to the SetConfig callback. */
    phhalHw_LicenseeWrapper_GetConfig_t pCbGetConfig;                                           /**< Pointer to the GetConfig callback. */
    phhalHw_LicenseeWrapper_FieldReset_t pCbFieldReset;                                         /**< Pointer to the FieldReset callback. */
    phhalHw_LicenseeWrapper_Wait_t pCbWait;                                                     /**< Pointer to the Wait callback. */
    phhalHw_LicenseeWrapper_Exchange_t pCbExchange;                                             /**< Pointer to the Exchange callback. */
    phhalHw_LicenseeWrapper_ApplyProtocolSettings_t pCbApplyProtocolSettings;                   /**< Pointer to the ApplyProtocolSettings callback. */
    void * LicenseeHalDll;                                                                      /**< Handle to the Licensee HAL DLL. (type HINSTANCE)*/
    void * pContext;                                                                            /**< Custom context which is passed back to each callback. */
} phhalHw_LicenseeWrapper_DataParams_t;


/**
 * \brief Initialise the Licensee Wrapper HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_LicenseeWrapper_Init(phhalHw_LicenseeWrapper_DataParams_t * UNALIGNED pDataParams, /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize,                                                                    /**< [In] Size of the global receive buffer. */
        int8_t * pDllFile                                                                       /**< [In] Path and name to the Licensee HAL DLL file. */
    );

/**
 * \brief De-initialise the Licensee Wrapper HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_LicenseeWrapper_Term(
        phhalHw_LicenseeWrapper_DataParams_t * pDataParams                                      /**< [In] Pointer to this layer's parameter structure. */
);

/**
 * end of group phhalHw_LicenseeWrapper
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_LICENSEEWRAPPER */

#ifdef NXPBUILD__PHHAL_HW_RC632

/**
 * \defgroup phhalHw_Rc632 Component : Rc632
 * \brief RC632 Reader HAL
 *
 * @{
 */
#define PHHAL_HW_RC632_ID                                                               0x09    /**< ID for Rc632 HAL component. */

#define PHHAL_HW_RC632_SHADOW_COUNT                                                     0x000DU /**< Number of shadowed configurations. */
#define PHHAL_HW_RC632_DEFAULT_TIMEOUT_MS                                               5U      /**< Default timeout in milliseconds. */

/** \brief Rc632 HAL parameter structure */
typedef struct
{
    unsigned short wId;                                                                         /**< Layer ID for this HAL component, NEVER MODIFY! */
    void  * pBalDataParams;                                                                     /**< Pointer to the lower layers parameter structure. */
    unsigned char * pTxBuffer;                                                                  /**< Pointer to global transmit buffer used by the Exchange() function. */
    unsigned short wTxBufSize;                                                                  /**< Size of the global transmit buffer. */
    unsigned short wTxBufLen;                                                                   /**< Number of valid bytes within the transmit buffer. */
    unsigned char * pRxBuffer;                                                                  /**< Pointer to global receive buffer used by the Exchange() function. */
    unsigned short wRxBufSize;                                                                  /**< Size of the global receive buffer. */
    unsigned short wRxBufLen;                                                                   /**< Number of valid bytes within the receive buffer. */
    unsigned short wRxBufStartPos;                                                              /**< Starting position within the global receive buffer. */
    unsigned short wTxBufStartPos;                                                              /**< Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). */
    unsigned short wMaxPrecachedBytes;                                                          /**< Holds the max. number of bytes which are precached prior to command execution. */
    unsigned char bCardType;                                                                    /**< Type of card for which the hal is configured for. */
    unsigned short wCfgShadow[0x10];                                                            /**< Configuration shadow; Stores configuration for current cardtype. */
    unsigned short wTimingMode;                                                                 /**< Current timing measurement mode. */
    unsigned char bTimeoutUnit;                                                                 /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    unsigned int dwTimingUs;                                                                    /**< Current timing value. */
    unsigned int dwTimeoutUs;                                                                   /**< Current timing value. */
    unsigned short wFieldOffTime;                                                               /**< Field-Off-Time in milliseconds. */
    unsigned short wFieldRecoveryTime;                                                          /**< Field-Recovery-Time in milliseconds. */
    unsigned char bSymbolStart;                                                                 /**< Preamble of a frame. */
    unsigned char bSymbolEnd;                                                                   /**< Trailer symbol of a frame. */
    unsigned char bDisablePcdReset;                                                             /**< Whether to disable PcdReset call in \ref phhalHw_ApplyProtocolSettings function or not. */
    unsigned char bFlags;                                                                       /**< Flags to be used.*/
    unsigned char bFifoSize;                                                                    /**< Fifo-Size setting. */
    unsigned short wAdditionalInfo;                                                             /**< Storage for additional error information. */
    unsigned char bBalConnectionType;                                                           /**< Type of the underlying BAL connection. Refer to #PHHAL_HW_CONFIG_BAL_CONNECTION. */
    unsigned char bRfResetAfterTo;                                                              /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint8_t bTimerControlPc;                                                                    /**< Backup for Timer control PC requirement */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC*/
} phhalHw_Rc632_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Rc632_Init(
        phhalHw_Rc632_DataParams_t * pDataParams,                                               /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t* pTxBuffer,                                                                     /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t* pRxBuffer,                                                                     /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. Specify the buffer +1 byte, because
                                                                                                 *        one byte is reserved for SPI communication.
                                                                                                 */
    );

#include <phhalHw_Rc632_Reg.h>
#include <phhalHw_Rc632_Cmd.h>

/**
 * end of group phhalHw_Rc632
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_RC632 */

#ifdef NXPBUILD__PHHAL_HW_CALLBACK

/**
 * \defgroup phhalHw_Callback Component : Callback
 * \brief Callback Reader HAL
 *
 * Virtual reader device which can be used to implement custom reader devices.
 * @{
 */

#define PHHAL_HW_CALLBACK_ID                                                            0xAB    /**< ID for Rc523 HAL component. */

/** \brief Callback Delegate for \ref phhalHw_Exchange */
typedef phStatus_t (*cbExchange)(
        uint16_t wOption,                                                                       /**< [In] Option parameter. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Data to transmit. */
        uint16_t wTxLength,                                                                     /**< [In] Length of data to transmit. */
        uint8_t ** ppRxBuffer,                                                                  /**< [Out] Pointer to received data. */
        uint16_t * pRxLength,                                                                   /**< [Out] number of received data bytes. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_WriteRegister */
typedef phStatus_t (*cbWriteRegister)(
        uint8_t bAddress,                                                                       /**< [In] Register Address. */
        uint8_t bValue,                                                                         /**< [In] Register Value. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/**  \brief Callback Delegate for \ref phhalHw_ReadRegister */
typedef phStatus_t (*cbReadRegister)(
        uint8_t bAddress,                                                                       /**< [In] Register Address. */
        uint8_t * pValue,                                                                       /**< [Out] Register Value. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_ApplyProtocolSettings */
typedef phStatus_t (*cbApplyProtocolSettings)(
        uint8_t bCardType,                                                                      /**< [In] Type of card for which the hal should be configured for. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_MfcAuthenticate */
typedef phStatus_t (*cbMfcAuthenticate)(
        uint8_t bBlockNo,                                                                       /**< [In] Blocknumber on card to authenticate to. */
        uint8_t bKeyType,                                                                       /**< [In] Either #PHHAL_HW_MFC_KEYA or #PHHAL_HW_MFC_KEYB. */
        uint16_t wKeyNo,                                                                        /**< [In] Key number to be used in authentication. */
        uint16_t wKeyVersion,                                                                   /**< [In] Key version to be used in authentication. */
        uint8_t * pUid,                                                                         /**< [In] Serial number of current cascade level; uint8_t[4]. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_MfcAuthenticateKeyNo */
typedef phStatus_t (*cbMfcAuthenticateKey)(
        uint8_t bBlockNo,                                                                       /**< [In] Blocknumber on card to authenticate to. */
        uint8_t bKeyType,                                                                       /**< [In] Either #PHHAL_HW_MFC_KEYA or #PHHAL_HW_MFC_KEYB */
        uint8_t * pKey,                                                                         /**< [In] Key to be used in authentication. */
        uint8_t * pUid,                                                                         /**< [In] Serial number of current cascade level; uint8_t[4]. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_SetConfig */
typedef phStatus_t (*cbSetConfig)(
        uint16_t wConfig,                                                                       /**< [In] Configuration Identifier. */
        uint16_t wValue,                                                                        /**< [In] Configuration Value. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_GetConfig */
typedef phStatus_t (*cbGetConfig)(
        uint16_t wConfig,                                                                       /**< [In] Configuration Identifier. */
        uint16_t * pValue,                                                                      /**< [Out] Configuration Value. */
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_FieldOn */
typedef phStatus_t (*cbFieldOn)(
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_FieldOff */
typedef phStatus_t (*cbFieldOff)(
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_FieldReset */
typedef phStatus_t (*cbFieldReset)(
        void * pContext                                                                         /**< [Out] User Defined Context. */
    );

/** \brief Callback Delegate for \ref phhalHw_Wait */
typedef phStatus_t (*cbWait)(
    uint8_t bUnit,                                                                              /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint16_t wTimeout,                                                                          /**< [In] Timeout value. */
    void * pContext                                                                             /**< [Out] User Defined Context. */
    );

/** \brief Callback HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    cbExchange pExchangeCb;                                                                     /**< Pointer to \ref cbExchange Callback. */
    cbWriteRegister pWriteRegisterCb;                                                           /**< Pointer to \ref cbWriteRegister Callback. */
    cbReadRegister pReadRegisterCb;                                                             /**< Pointer to \ref cbReadRegister Callback. */
    cbApplyProtocolSettings pApplyProtocolSettingsCb;                                           /**< Pointer to \ref cbApplyProtocolSettings Callback. */
    cbMfcAuthenticate pMfcAuthenticateCb;                                                       /**< Pointer to \ref cbMfcAuthenticate Callback. */
    cbMfcAuthenticateKey pMfcAuthenticateKeyCb;                                                 /**< Pointer to \ref cbMfcAuthenticateKey Callback. */
    cbSetConfig pSetConfigCb;                                                                   /**< Pointer to \ref cbSetConfig Callback. */
    cbGetConfig pGetConfigCb;                                                                   /**< Pointer to \ref cbGetConfig Callback. */
    cbFieldOn pFieldOnCb;                                                                       /**< Pointer to \ref cbFieldOn Callback. */
    cbFieldOff pFieldOffCb;                                                                     /**< Pointer to \ref cbFieldOff Callback. */
    cbFieldReset pFieldResetCb;                                                                 /**< Pointer to \ref cbFieldReset Callback. */
    cbWait pWaitCb;                                                                             /**< Pointer to \ref cbWait Callback. */
    void * pCbContext;                                                                          /**< Custom context which is passed back to each Callback. */
} phhalHw_Callback_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Callback_Init(
        phhalHw_Callback_DataParams_t * pDataParams,                                            /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        cbExchange pExchangeCb,                                                                 /**< [In] Pointer to \ref cbExchange Callback. */
        cbWriteRegister pWriteRegisterCb,                                                       /**< [In] Pointer to \ref cbWriteRegister Callback. */
        cbReadRegister pReadRegisterCb,                                                         /**< [In] Pointer to \ref cbReadRegister Callback. */
        cbApplyProtocolSettings pApplyProtocolSettingsCb,                                       /**< [In] Pointer to \ref cbApplyProtocolSettings Callback. */
        cbMfcAuthenticate pMfcAuthenticateCb,                                                   /**< [In] Pointer to \ref cbMfcAuthenticate Callback. */
        cbMfcAuthenticateKey pMfcAuthenticateKeyCb,                                             /**< [In] Pointer to \ref cbMfcAuthenticateKey Callback. */
        cbSetConfig pSetConfigCb,                                                               /**< [In] Pointer to \ref cbSetConfig Callback. */
        cbGetConfig pGetConfigCb,                                                               /**< [In] Pointer to \ref cbGetConfig Callback. */
        cbFieldOn pFieldOnCb,                                                                   /**< [In] Pointer to \ref cbFieldOn Callback. */
        cbFieldOff pFieldOffCb,                                                                 /**< [In] Pointer to \ref cbFieldOff Callback. */
        cbFieldReset pFieldResetCb,                                                             /**< [In] Pointer to \ref cbFieldReset Callback. */
        cbWait pWaitCb,                                                                         /**< [In] Pointer to \ref cbWait Callback. */
        void * pCallbackContext                                                                 /**< [In] Custom context which is passed back to each Callback. */
    );

/**
 * end of group phhalHw_Callback
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_CALLBACK */

#ifdef NXPBUILD__PHHAL_HW_MFCCRYPTO

/**
 * \defgroup phhalHw_MfcCrypto Component : MfcCrypto
 * \brief MfcCrypto Reader HAL
 *
 * Software Implementation of MIFARE Crypto1.
 * @{
 */

#define PHHAL_HW_MFCCRYPTO_ID                                                           0xAC    /**< ID for MIFARE Classic Crypto HAL component. */

#define PHHAL_HW_MFCCRYPTO_CONFIG_ENABLE                        (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Configuration to enable the MIFARE Crypto (Default: Off). Enabling the MIFARE Crypto block also disables the Crypto1, CRC's and parity of the hal below.*/

/** \brief MfcCrypto HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pHalDataParams;                                                                     /**< Pointer to the lower layers parameter structure. */
    uint8_t rA_buf[4];                                                                          /**< Initial RA value. */
    uint8_t rBp_buf[4];                                                                         /**< Initial RB value. */
    uint8_t cryptoReg[48];                                                                      /**< The CRYPTO LFSR. */
    uint8_t randReg[16];                                                                        /**< The PRNG LFSR for RB. */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t bCryptoState;                                                                       /**< Current crypto state. */
    uint8_t bActivated;                                                                         /**< Indicates whether this layer is activated or not. */
    uint32_t uIdForSpFeed1;                                                                     /**< UID to unit32. */
    uint8_t bEnabled;                                                                           /**< Disables/Enables Crypto. */
    uint8_t bRxBits;                                                                            /**< Number of valid bits in the receive buffer. */
} phhalHw_MfcCrypto_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_MfcCrypto_Init(
        phhalHw_MfcCrypto_DataParams_t * pDataParams,                                           /**< [In] Pointer to this layers parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pHalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize                                                                     /**< [In] Size of the global transmit buffer. */
    );

/**
 * end of group phhalHw_MfcCrypto
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_MFCCRYPTO */

#ifdef NXPBUILD__PHHAL_HW_RD710

/** \defgroup phhalHw_Rd710 Component : Rd710
* \brief RD710 (Pegoda2) Reader HAL
* @{
*/

#define PHHAL_HW_RD710_ID                                                               0x06    /**< ID for Rd710 HAL component. */

#define PHHAL_HW_RD710_RESERVED_BUFFER_LEN                                              8U      /**< Amount of needed and reserved memory for the protocol overhead. */

/** \brief Rd710 HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pBalDataParams;                                                                     /**< pointer to the lower layers parameter structure. */
    uint8_t bSlotNumber;                                                                        /**< Slot number. */
    uint8_t bTxLastBits;                                                                        /**< TxLastbits for ExchangeRaw function. */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint16_t wTxBufStart;                                                                       /**< Startposition in case of buffering within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC*/
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
} phhalHw_Rd710_DataParams_t;

/** \name PCSC Modes */
/** @{ */
#define PHHAL_HW_RD710_PCSCMODE_DEFAULT                                                 0x0000U /**< Default PC/SC operation. */
#define PHHAL_HW_RD710_PCSCMODE_DIRECT                                                  0x0001U /**< Direct PC/SC mode (Exchange raw reader commands). */
/** @} */

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Rd710_Init(
        phhalHw_Rd710_DataParams_t * pDataParams,                                               /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t bSlotNumber,                                                                    /**< [In] Slot number. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. */
    );

#include <phhalHw_Rd710_Cmd.h>

/**
 * end of group phhalHw_Rd710
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_RD710 */

#ifdef NXPBUILD__PHHAL_HW_RDFPGAV6

/**
 * \cond INCLUDE_INTERNALS
 * \defgroup phhalHw_RdFpgaV6 Component : RdFpga6
 * \brief RdFpga6 Reader HAL.
 * @{
 */

#define PHHAL_HW_RDFPGAV6_ID                                                            0x07    /**< ID for RdFpga HAL component. */

/** \name RdFpga specific configs */
/** @{ */
#define PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER                  (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Select the target joiner */
#define PHHAL_HW_RDFPGAV6_CONFIG_LOGGING                        (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Switch the logging on/off */
#define PHHAL_HW_RDFPGAV6_CONFIG_FIELD_OFF_TIME_CYCLE           (PH_CONFIG_CUSTOM_BEGIN + 2)    /**< Set the field off time for field-reset in clock cycles */
#define PHHAL_HW_RDFPGAV6_CONFIG_FIELD_OFF_TIME_US              (PH_CONFIG_CUSTOM_BEGIN + 3)    /**< Set the field off time for field-reset in us */
/** @} */

#define PHHAL_HW_RDFPGAV6_SHADOW_COUNT                                                  0x000DU /**< Number of shadowed configurations. */
#define PHHAL_HW_RDFPGAV6_DEFAULT_TIMEOUT_MS                                            5U      /**< Default timeout in milliseconds. */

/** \brief RdFpga HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this HAL component, NEVER MODIFY! */
    phbalReg_RdFpgaV6_DataParams_t * pBalDataParams;                                            /**< pointer to the lower layers parameter structure */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
    uint16_t wJoiner;                                                                           /**< Currently selected Joiner. */
    uint16_t wCfgShadow[PHHAL_HW_RDFPGAV6_SHADOW_COUNT];                                        /**< Configuration shadow; Stores configuration for current cardtype. */
    uint16_t wTimingMode;                                                                       /**< Current timing measurement mode. */
    uint32_t dwTimeoutUs;                                                                       /**< TO value to be used by exchange function. */
    uint32_t dwTimeoutETU;                                                                      /**< TO value in ISO ETU (128/fc) to be used by exchange function. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint16_t wTxRxStartBackup;                                                                  /**< Absolute Start-Position for transmission / reception used by buffer preservation mechanism. */
    uint32_t dwFieldOffTimeCycles;                                                              /**< Field-Off-Time in clock cycles. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint8_t bSymbolStart;                                                                       /**< Preamble of a frame. */
    uint8_t bSymbolEnd;                                                                         /**< Trailer symbol of a frame. */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint8_t bFieldState;                                                                        /**< Mirrored Field State. */
    uint8_t bTypeBModeActive;                                                                   /**< Flag indicating if TypeBMode is active */
} phhalHw_RdFpgaV6_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_RdFpgaV6_Init(
        phhalHw_RdFpgaV6_DataParams_t * pDataParams,                                            /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        phbalReg_RdFpgaV6_DataParams_t * pBalDataParams,                                        /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. */
    );

/**
 * end of group phhalHw_RdFpgaV6
 * @}
 * \endcond
 */
#endif /* NXPBUILD__PHHAL_HW_RDFPGAV6 */

#ifdef NXPBUILD__PHHAL_HW_MP300

/**
 * \cond INCLUDE_INTERNALS
 * \defgroup phhalHw_Mp300 Component : Mp300
 * \brief Mp300 (Micropross) Reader HAL
 * @{
 */

#define PHHAL_HW_MP300_ID                                                               0x08    /**< ID for Rd70x HAL component. */

/** \name Mp300 specific configs: Set Fieldstrength in percent of Maximum available Fieldstrength. */
/** @{ */
#define PHHAL_HW_MP300_CONFIG_FIELDSTRENGTH                     (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Configure Fieldstrength. */
#define PHHAL_HW_MP300_CONFIG_CARRIER_FREQUENCY_SHIFT           (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Configure Carrier frequency. */
#define PHHAL_HW_MP300_CONFIG_RX_CHANNEL                        (PH_CONFIG_CUSTOM_BEGIN + 2)    /**< Configure RX channel. */
#define PHHAL_HW_MP300_CONFIG_RELAY                             (PH_CONFIG_CUSTOM_BEGIN + 3)    /**< Configure Relay port. */
#define PHHAL_HW_MP300_CONFIG_FIELD_RISE_TIME                   (PH_CONFIG_CUSTOM_BEGIN + 4)    /**< Configure Field rise time(only after Fieldstrength 0%). */
#define PHHAL_HW_MP300_CONFIG_FALL_TIME                         (PH_CONFIG_CUSTOM_BEGIN + 5)    /**< Configure Fall time of modulation. */
#define PHHAL_HW_MP300_CONFIG_RISE_TIME                         (PH_CONFIG_CUSTOM_BEGIN + 6)    /**< Configure Rise time of modulation. */
#define PHHAL_HW_MP300_CONFIG_PAUSE_WIDTH                       (PH_CONFIG_CUSTOM_BEGIN + 7)    /**< Configure Pausewidth of modulation. */
#define PHHAL_HW_MP300_CONFIG_COUPLER_NUMBER                    (PH_CONFIG_CUSTOM_BEGIN + 8)    /**< Configure of Coupler. */
#define PHHAL_HW_MP300_CONFIG_RX_GAIN                           (PH_CONFIG_CUSTOM_BEGIN + 9)    /**< Configure of the gain of the receiver. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_SOF_LOW_CYCLES             (PH_CONFIG_CUSTOM_BEGIN + 10)   /**< Configure of the type B SOF low timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_SOF_HIGH_CYCLES            (PH_CONFIG_CUSTOM_BEGIN + 11)   /**< Configure of the type B SOF high timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_EGT_CYCLES                 (PH_CONFIG_CUSTOM_BEGIN + 12)   /**< Configure of the type B EGT timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_EOF_CYCLES                 (PH_CONFIG_CUSTOM_BEGIN + 13)   /**< Configure of the type B EOF timing. */
#define PHHAL_HW_MP300_CONFIG_ANTI_EMD                          (PH_CONFIG_CUSTOM_BEGIN + 14)   /**< Configure of the Anti EMD behavior. */
#define PHHAL_HW_MP300_CONFIG_FIELDSTRENGTH_THOUSAND            (PH_CONFIG_CUSTOM_BEGIN + 15)   /**< Configure Fieldstrength in Permil. */
#define PHHAL_HW_MP300_CONFIG_WAVEFORM_GENERATION_MODE          (PH_CONFIG_CUSTOM_BEGIN + 16)   /**< Configure the generation mode of the waveform. */
#define PHHAL_HW_MP300_CONFIG_RX_GAIN_TARGET                    (PH_CONFIG_CUSTOM_BEGIN + 17)   /**< Target value that should be reached with Adjust Rx (Default 750 but with card with low LMA and at high fieldstrength a higher value should be used */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_START_BIT_CYCLES           (PH_CONFIG_CUSTOM_BEGIN + 18)   /**< Configure of the type B Start Bit timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_0_CYCLES               (PH_CONFIG_CUSTOM_BEGIN + 19)   /**< Configure of the type B Bit 0 timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_1_CYCLES               (PH_CONFIG_CUSTOM_BEGIN + 20)   /**< Configure of the type B Bit 1 timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_2_CYCLES               (PH_CONFIG_CUSTOM_BEGIN + 21)   /**< Configure of the type B Bit 2 timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_3_CYCLES               (PH_CONFIG_CUSTOM_BEGIN + 22)   /**< Configure of the type B Bit 3 timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_4_CYCLES               (PH_CONFIG_CUSTOM_BEGIN + 23)   /**< Configure of the type B Bit 4 timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_5_CYCLES               (PH_CONFIG_CUSTOM_BEGIN + 24)   /**< Configure of the type B Bit 5 timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_6_CYCLES               (PH_CONFIG_CUSTOM_BEGIN + 25)   /**< Configure of the type B Bit 6 timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_7_CYCLES               (PH_CONFIG_CUSTOM_BEGIN + 26)   /**< Configure of the type B Bit 7 timing. */
#define PHHAL_HW_MP300_CONFIG_TYPE_B_STOP_BIT_CYCLES            (PH_CONFIG_CUSTOM_BEGIN + 27)   /**< Configure of the type B Stop Bit timing. */
#define PHHAL_HW_MP300_CONFIG_SOURCE_OF_CARRIER                 (PH_CONFIG_CUSTOM_BEGIN + 28)   /**< Configure of the carrier source. */
#define PHHAL_HW_MP300_CONFIG_MAX_FIELDSTRENGTH_THOUSAND        (PH_CONFIG_CUSTOM_BEGIN + 29)   /**< Get the maximum possible Fieldstrength. */
#define PHHAL_HW_MP300_CONFIG_FORCE_FDT_METHOD_1                (PH_CONFIG_CUSTOM_BEGIN + 30)   /**< Configure that always method one of FDT is used. If PH_OFF then for type a the method 2 is used for type A. */
#define PHHAL_HW_MP300_CONFIG_RFAFTERTEARING                    (PH_CONFIG_CUSTOM_BEGIN + 31)   /**< Configure of Rf after Tearing. */
#define PHHAL_HW_MP300_CONFIG_TEARING_EXCHANGE_COUNTER          (PH_CONFIG_CUSTOM_BEGIN + 32)   /**< Configure the exchange index for tearing functionality. */
/** @} */

/** \name Trigger Configuration Values */
/** @{ */
enum TriggerMode
{
    PHHAL_HW_MP300_TRIG_FORCE           = 0x01U,                                                /**< Force Triggerchannel */
    PHHAL_HW_MP300_TRIG_TX_ON           = 0x02U,                                                /**< The trigger out will be set to a logic 0 during the transmission of a frame to the card. */
    PHHAL_HW_MP300_TRIG_TX_OUT          = 0x03U,                                                /**< The trigger will represent the signal that is used to modulate the carrier. */
    PHHAL_HW_MP300_TRIG_RX_ON           = 0x06U,                                                /**< The trigger out will be set to a logic 0 during the reception of a frame coming from the card. */
    PHHAL_HW_MP300_TRIG_DELAY_AFTER_TX  = 0x07U,                                                /**< The Trigger is set to logic 0 after the transmission of the PCD and after a time specified with the Value Parameter it is set to logical 1 again. */
    PHHAL_HW_MP300_TRIG_CARRIER         = 0x08U,                                                /**< Trigger will represent the logic representation of the carrier. */
    PHHAL_HW_MP300_TRIG_EMD             = 0x13U                                                 /**< The trigger out will be set to a logic 0 during the reception of a frame coming from the card. Do supress EMD spikes it starts after receiving a few symbols
                                                                                                 * type A after 10 symbols
                                                                                                 * type B after 9 bits
                                                                                                 * type A HD after 8 bits
                                                                                                 * Felica after 8 bits (at least FPGA 33 has to be used
                                                                                                 */
};
/** @} */

/** \name CouplerResourceId */
/** @{ */
enum CouplerResourceId
{
    PHHAL_HW_MP300_RESSOURCE_TCL1 = 0x78U,                                                      /**< Id of a MP300 TCL1 */
    PHHAL_HW_MP300_RESSOURCE_TCL2 = 0x7AU,                                                      /**< Id of a MP300 TCL2 */
    PHHAL_HW_MP300_RESSOURCE_TCL3 = 0xF6U                                                       /**< Id of a MP500 TCL3 */
};
/** @} */

/** \name Carrier Source Configuration Values */
/** @{ */
enum SourceOfCarrier
{
    PHHAL_HW_MP300_CARRIER_INTERNAL         = 0x01U,                                            /**< Internal programmable carrier (default). */
    PHHAL_HW_MP300_CARRIER_EXTERNAL         = 0x02U,                                            /**< Take carrier from logical signal from trigger. */
    PHHAL_HW_MP300_CARRIER_INTERNAL_FIXED   = 0x03U                                             /**<  Fixed 13.56 MHz. */
};
/** @} */

/** \name Waveform generation modes */
/** @{ */
enum WaveformGenerationMode
{
    PHHAL_HW_MP300_LINEAR_MODULATION           = 0x01U,                                         /**< Waveform generation linear */
    PHHAL_HW_MP300_CUSTOMIZED_MODULATION       = 0x02U,                                         /**< Waveform generation with custom shape */
    PHHAL_HW_MP300_CUSTOMIZED_MODULATION_CLK   = 0x03U,                                         /**< Waveform generation with custom shape in clk step size */
    PHHAL_HW_MP300_CUSTOMIZED_MODULATION_10CLK = 0x04U                                          /**< Waveform generation with custom shape in 1/10 clk step size */
};
/** @} */

/** \name Tearing behaviour modes */
/** @{ */
enum TearingBehaviour
{
    PHHAL_HW_MP300_TEARING_FIELD_OFF               = 0x01U,                                     /**< Field is always off after exchange (either tearing occured or field is switched of after exchange */
    PHHAL_HW_MP300_TEARING_FIELD_ON                = 0x02U,                                     /**< Field is always in after exchange (either tearing occured and field is switched on again of a HF is performed after exchange */
    PHHAL_HW_MP300_TEARING_FIELD_OFF_AFTER_TEARING = 0x03U,                                     /**< Field stay if after tearing event, no change in the field when no tearing occure */
    PHHAL_HW_MP300_TEARING_FIELD_ON_AFTER_TEARING  = 0x04U                                      /**< Field is switched on after tearing event, no change in the field when no tearing occure */
};
/** @} */

#define PHHAL_HW_MP300_SHADOW_COUNT                                                     0x000FU /**< Number of shadowed configurations. */
#define PHHAL_HW_MP300_CRC_BUFFER_LEN                                                   2U      /**< Amount of needed and reserved memory for the CRC overhead. */
#define PHHAL_HW_MP300_FIFOSIZE                                                         256U    /**< Size of RC internal FiFo buffer. */
#define PHHAL_HW_MP300_DEFAULT_TIMEOUT_US                                               5000U   /**< Default timeout in milliseconds. */
#define PHHAL_HW_MP300_SERIAL_NUMBER_SIZE                                               16U     /**< The size of the serial number. */
#define PHHAL_HW_MP300_VERSION_STR_SIZE                                                 8U      /**< The size of the version strings. */
#define PHHAL_HW_MP300_COUPLER_NAME_SIZE                                                7U      /**< The size of the coupler name (longest = SWPSPY). */
#define PHHAL_HW_MP300_COUPLER_SERIAL_NUMBER_SIZE                                       9U      /**< The size of the coupler serial number. */
#define PHHAL_HW_MP300_COUPLER_REVISION_SIZE                                            5U      /**< The size of the coupler revision. */
#define PHHAL_HW_MP300_COUPLER_CALC_DATE_SIZE                                           4U      /**< The size used to store the calibration date + checksum */

/** \brief Callback Delegate for Bal timeout handling */
typedef phStatus_t (*cbBalTimeoutHandling)(void * pContext, uint8_t bCouplerNumber);

/** \brief Mp300 HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pBalDataParams;                                                                     /**< pointer to the lower layers parameter structure */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint8_t bCardType;                                                                          /**< Type of card for which the HAL is configured for. */
    uint16_t wCfgShadow[PHHAL_HW_MP300_SHADOW_COUNT];                                           /**< Configuration shadow; Stores configuration for current cardtype. */
    uint16_t wTimingMode;                                                                       /**< Current timing measurement mode. */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint64_t qwTiming_ns;                                                                       /**< Current timing value. */
    uint64_t qwCommunicationTimePCDtoPICCCyclesAll;                                             /**< Sum of time for communication between PCD and PICC. */
    uint64_t qwCommunicationTimePICCtoPCDCyclesAll;                                             /**< Sum of time for communication between PICC and PCD. */
    uint64_t qwCommunicationTimePCDtoPICCCyclesCurrent;                                         /**< Time for communication between PCD and PICC of the last exchange. */
    uint64_t qwCommunicationTimePICCtoPCDCyclesCurrent;                                         /**< Time for communication between PICC and PCD of the last exchange. */
    uint8_t bLastBit;                                                                           /**< Last bit exchanged (0 if last bit was Zero 1 if last bit was One. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint8_t bSymbolStart;                                                                       /**< Preamble of a frame. */
    uint8_t bSymbolEnd;                                                                         /**< Trailer symbol of a frame. */
    uint16_t wCurrentFieldStrength;                                                             /**< Current Fieldstrength in Permil. */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint8_t bRfAfterTearing;                                                                    /**< Storage for additional #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint16_t wTearingExchangeCounter;                                                           /**< Counter decreasing at each exchange until desired exchange is reached for tearing. */
    uint8_t * pIntBuffer;                                                                       /**< Internal buffer used as a scratch buffer. Size must be \b wIntBufferSize*wIntBufferNum. */
    uint16_t wIntBufferSize;                                                                    /**< Total length of internal buffer used as a scratch buffer. */
    uint16_t wIntBufferLen;                                                                     /**< Flag to show the status of buffer allocation. */
    uint16_t wIntBufferReadPos;                                                                 /**< Read position within \b pIntBuffer for protocol. */
    uint32_t dwCarrierFrequency;                                                                /**< Storage for #PHHAL_HW_MP300_CONFIG_CARRIER_FREQUENCY_SHIFT setting. */
    uint8_t bRxGain;                                                                            /**< Storage for Rx Gain setting. */
    uint8_t bRXChannel;                                                                         /**< Storage for the current RX Channel Setting. */
    uint8_t bCouplerNumber;                                                                     /**< Select the Coupler used for communication */
    uint16_t wFieldRiseTime;                                                                    /**< Storage for the field rise time. */
    uint8_t bRelay;                                                                             /**< Storage for Relay PH_ON or PH_OFF.*/
    uint8_t bModulationMode;                                                                    /**< Lower nibble stores the modulation mode upper nibble stores if the custom shape is loaded */
    uint16_t wFallTime;                                                                         /**< Storage of Fall time of modulation.*/
    uint16_t wRiseTime;                                                                         /**< Storage of Rise time of modulation.*/
    uint16_t wPauseWidth;                                                                       /**< Storage of Pause width of modulation(Type A only).*/
    uint8_t bSourceOfCarrier;                                                                   /**< Storage for the source of the carrier.*/
    uint8_t bSubCarrierMode;                                                                    /**< Storage for the current subcarrier mode for 15693.*/
    uint16_t wTypeBTimingParameter[14];                                                         /**< Storage of the Type B modulation timing parameter.*/
    uint16_t wCouplerResourceId;                                                                /**< Storage of the resource ID of the Coupler to know the type of Reader (TCL1 or TCL2 supported).*/
    uint8_t bAntiEMD;                                                                           /**< Storage for Anti EMD PH_ON or PH_OFF.*/
    uint8_t bForceFdtMethod1;                                                                   /**< Storage for Force Fdt Method 1 PH_ON or PH_OFF.*/
    uint16_t wTargetRxGainLevel;                                                                /**< Value that should be reached with Adjust Rx .. Default is 750 but if PICC has low LMA a higher value should be used in high field */
    uint8_t bSerialNumber[PHHAL_HW_MP300_SERIAL_NUMBER_SIZE];                                   /**< This value stores the serial number of the reader */
    uint8_t bDeviceFamily[PHHAL_HW_MP300_VERSION_STR_SIZE];                                     /**< This value stores the DeviceFamily of the reader */
    uint8_t bDeviceReferenceWithinFamily[PHHAL_HW_MP300_VERSION_STR_SIZE];                      /**< This value stores the DeviceReferenceWithinFamily of the reader */
    uint8_t bMotherboardSystemVersion[PHHAL_HW_MP300_VERSION_STR_SIZE];                         /**< This value stores the MotherboardSystemVersion of the reader */
    uint8_t bMotherboardBootVersion[PHHAL_HW_MP300_VERSION_STR_SIZE];                           /**< This value stores the MotherboardBootVersion of the reader */
    uint8_t bCouplerName[PHHAL_HW_MP300_COUPLER_NAME_SIZE];                                     /**< This value stores the CouplerName of the reader */
    uint8_t bCouplerSerialNumber[PHHAL_HW_MP300_COUPLER_SERIAL_NUMBER_SIZE];                    /**< This value stores the CouplerSerialNumber of the reader */
    uint8_t bCouplerRevision[PHHAL_HW_MP300_COUPLER_REVISION_SIZE];                             /**< This value stores the CouplerRevision of the reader */
    uint8_t bCouplerDriverVersion[PHHAL_HW_MP300_VERSION_STR_SIZE];                             /**< This value stores the CouplerDriverVersion of the reader */
    uint16_t wCouplerFPGAVersion;                                                               /**< This value stores the CouplerFPGAVersion of the reader */
    uint8_t bCouplerCalibrationDate[PHHAL_HW_MP300_COUPLER_CALC_DATE_SIZE];                     /**< Date where the till the coupler is calibrated */
    uint32_t dwScenarioID;                                                                      /**< ID of the current in use scenario */
    uint32_t dwScenarioPcdId;                                                                   /**< ID of the current in use scenario for card emulation */
    cbBalTimeoutHandling pBalTimeoutHandling;                                                   /**< Pointer to the handler function. */
    void * pBalTimeoutHandlingContext;                                                          /**< Context that can be used with handler function. */
    uint8_t bSpyFilename[256];                                                                  /**< Filename of the spied Tracefile */
    void *SpyThreadHandle;                                                                      /**< Handle of the SpyThread */
    void *GEvtFinished;                                                                         /**< Event to check if SpyThread is ready */
    void *GEvtReady;                                                                            /**< Event to check if SpyThread has finished */
    uint16_t wTriggerExchangeCounters[4];                                                       /**< Array of counters used to count the exchanges before trigger is set. */
    uint32_t dwTriggerConfigs[4];                                                               /**< Array of configs for each channel. */
    uint32_t dwTriggerValues[4];                                                                /**< Array of values for trigger configs. */
} phhalHw_Mp300_DataParams_t;

/**
* \brief Initialise the HAL component.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Mp300_Init(
        phhalHw_Mp300_DataParams_t * pDataParams,                                               /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize,                                                                    /**< [In] Size of the global receive buffer. */
        uint8_t * pIntBuffer,                                                                   /**< [In] Internal buffer used as a scratch buffer. */
        uint16_t  wIntBufferSize                                                                /**< [In] Length of innternal buffer used as a scratch buffer. */
    );

/**
 * end of group phhalHw_Mp300
 * @}
 * \endcond
 */
#endif /* NXPBUILD__PHHAL_HW_MP300 */

#ifdef NXPBUILD__PHHAL_HW_ISO3

/**
 * \cond INCLUDE_INTERNALS
 * \defgroup phhalHw_ISO3 Component : ISO3
 * \brief ISO3 Reader HAL
 * @{
 */
#define PHHAL_HW_ISO3_ID                                                                0x0A    /**< ID for ISO3 HAL component. */

#define PHHAL_HW_ISO3_SHADOW_COUNT                                                      0x0020U /**< Number of shadowed configurations. */
#define PHHAL_HW_ISO3_SHADOW_COUNT_CUSTOM                                               0x0030U /**< Number of shadowed configurations for custom configs. */
#define PHHAL_HW_ISO3_WAVESHAPE_BUFFER_CONFIG_COUNT                                     0x0006U /**< Number of waveshape buffer configurations */

#define PHHAL_HW_ISO3_EXCHANGE_COUNT_START                                              0x0001U /**< Start value for counting data exchanges */
#define PHHAL_HW_ISO3_DEFAULT_TIMEOUT_US                                                5000U   /**< Timeout for PICC response. Shadow struct default value */
#define PHHAL_HW_ISO3_CRC_BUFFER_LEN                                                    0x0002U /**< Length of CRC in bytes */
#define PHHAL_HW_ISO3_MAX_NUMB_WS_SLOTS                                                 0x1EU   /**< Max number of possible partitions of wave shape buffer */
#define PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_ID                                         0xFFFEU /**< Default SLOT ID for field on/off */
#define PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_NUMBER                                     0x0007U /**< Default partitioning of slots (match for type A and B)*/
#define PHHAL_HW_ISO3_OVS_FILTER_MAX_SLOT_NUM                                           0x0008U /**< Maximum number of memory slots for overshoot filter */
#define PHHAL_HW_ISO3_OVS_FILTER_COEFFICIENTS_PER_SLOT                                  0x0011U /**< Number of coefficients per overshoot filter (or slot) */

/** \name ISO3 HAL custom configs */
/** @{ */
#define PHHAL_HW_ISO3_CONFIG_FIELD_STATE                        (PH_CONFIG_CUSTOM_BEGIN + 16)   /**< Flag to show whether field is switched on. */
#define PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB               (PH_CONFIG_CUSTOM_BEGIN + 17)   /**< Number of partitions within the wave shape memory on the fpga */
#define PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_SIZE               (PH_CONFIG_CUSTOM_BEGIN + 18)   /**< Size in bytes of a single waveshape partition */
#define PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_ON            (PH_CONFIG_CUSTOM_BEGIN + 19)   /**< Slot ID for field on wave shape */
#define PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_OFF           (PH_CONFIG_CUSTOM_BEGIN + 20)   /**< Slot ID for field off wave shape */
#define PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_EXCHANGE            (PH_CONFIG_CUSTOM_BEGIN + 21)   /**< Slot ID for exchange wave shape */
#define PHHAL_HW_ISO3_CONFIG_FPGA_VERSION                       (PH_CONFIG_CUSTOM_BEGIN + 22)   /**< read only - get version of fpga bitfile */
#define PHHAL_HW_ISO3_CONFIG_BIT_BOUNDARY                       (PH_CONFIG_CUSTOM_BEGIN + 23)   /**< Set/Get bit boundary*/
#define PHHAL_HW_ISO3_CONFIG_EGT                                (PH_CONFIG_CUSTOM_BEGIN + 24)   /**< Set/Get Extra Guard Time (EGT)*/
#define PHHAL_HW_ISO3_CONFIG_DUTY_CYCLE                         (PH_CONFIG_CUSTOM_BEGIN + 25)   /**< Set/Get duty cycle*/
#define PHHAL_HW_ISO3_CONFIG_RX_NOISE_POWER_TH                  (PH_CONFIG_CUSTOM_BEGIN + 26)   /**< Set/Get the noise power threshold (18bits) of the receiver */
#define PHHAL_HW_ISO3_CONFIG_IIR_FILTER_CONTROL                 (PH_CONFIG_CUSTOM_BEGIN + 28)   /**< Enable / Disable IIR Filter */
#define PHHAL_HW_ISO3_CONFIG_DOUBLE_DATA_RATE                   (PH_CONFIG_CUSTOM_BEGIN + 29)
#define PHHAL_HW_ISO3_CONFIG_SEQ8_MODE                          (PH_CONFIG_CUSTOM_BEGIN + 30)   /**< Set/Get Sequence 8 buffer */
#define PHHAL_HW_ISO3_CONFIG_FIELD_OFF_TIME_US                  (PH_CONFIG_CUSTOM_BEGIN + 31)   /**< Set the field off time for field-reset in us */
#define PHHAL_HW_ISO3_CONFIG_DISABLE_AGC                        (PH_CONFIG_CUSTOM_BEGIN + 32)   /**< Disables Automatic Gain control for exchange, rf on/off, field reset */
#define PHHAL_HW_ISO3_CONFIG_IIR_FILTER_OVERFLOW                (PH_CONFIG_CUSTOM_BEGIN + 33)   /**< Flag to show whether IIR Filter overflow occured in adder or multiplexer */
#define PHHAL_HW_ISO3_CONFIG_FIELD_RECOVERY_TIME_US             (PH_CONFIG_CUSTOM_BEGIN + 34)   /**< Set the field recovery time in us (instead of ms) */
#define PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT0                      (PH_CONFIG_CUSTOM_BEGIN + 35)   /**< Sets the config for the Output 0 */
#define PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT1                      (PH_CONFIG_CUSTOM_BEGIN + 36)   /**< Sets the config for the Output 1 */
#define PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT2                      (PH_CONFIG_CUSTOM_BEGIN + 37)   /**< Sets the config for the Output 2 */
#define PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT3                      (PH_CONFIG_CUSTOM_BEGIN + 38)   /**< Sets the config for the Output 3 */
#define PHHAL_HW_ISO3_CONFIG_WAVESHAPE_PRESCALE_PERCENT         (PH_CONFIG_CUSTOM_BEGIN + 39)   /**< Sets the max pattern percent for waveshapes */
#define PHHAL_HW_ISO3_CONFIG_TX_LAST_BIT                        (PH_CONFIG_CUSTOM_BEGIN + 40)   /**< Gets last tx bit value */
#define PHHAL_HW_ISO3_CONFIG_PEAK_SENS_SIGNAL                   (PH_CONFIG_CUSTOM_BEGIN + 41)   /**< Gets SENS Signal peak value */
#define PHHAL_HW_ISO3_CONFIG_PEAK_CAL_SIGNAL                    (PH_CONFIG_CUSTOM_BEGIN + 42)   /**< Gets CAL Signal peak value */
#define PHHAL_HW_ISO3_CONFIG_OVS_FILTER_SLOT_ENABLE             (PH_CONFIG_CUSTOM_BEGIN + 43)   /**< Enable a certain overshoot filter slot. */
#define PHHAL_HW_ISO3_CONFIG_OVS_FILTER_SLOT_DISABLE            (PH_CONFIG_CUSTOM_BEGIN + 44)   /**< Disable a certain overshoot filter slot. */
#define PHHAL_HW_ISO3_CONFIG_MILLER_OFFSET                      (PH_CONFIG_CUSTOM_BEGIN + 45)   /**< Sets the config for Miller Min/Max test */
#define PHHAL_HW_ISO3_CONFIG_FIELD_OFF_TIME_CYCLES              (PH_CONFIG_CUSTOM_BEGIN + 46)   /**< sets the field of time in cw cycles */
/** @} */

/** \name Waveshape Buffer configs */
/** @{ */
enum WaveShapeBufferConfig
{
    PHHAL_HW_ISO3_WAVESHAPE_SLOT_SIZE         = 0,
    PHHAL_HW_ISO3_WAVESHAPE_SLOTS_NUMB        = 1,
    PHHAL_HW_ISO3_WAVESHAPE_SLOT_FIELD_ON     = 2,
    PHHAL_HW_ISO3_WAVESHAPE_SLOT_FIELD_OFF    = 3,
    PHHAL_HW_ISO3_WAVESHAPE_SLOT_EXCHANGE     = 4,
    PHHAL_HW_ISO3_WAVESHAPE_RFU               = 5
};
/** @} */

/** \name Trigger modes */
/** @{ */
enum TriggerModeIso3
{
    PHHAL_HW_ISO3_TRIG_MODE_TX_FIELD_OFF            = 0x0U,                                     /**< Trigger event when switching off carrier */
    PHHAL_HW_ISO3_TRIG_MODE_TX_IDLE                 = 0x1U,                                     /**< Trigger event when transmitter enters IDLE STATE */
    PHHAL_HW_ISO3_TRIG_MODE_TX_SOC                  = 0x2U,                                     /**< Trigger event when transmitter modulates SOC */
    PHHAL_HW_ISO3_TRIG_MODE_TX_LAST_BIT             = 0x3U,                                     /**< Trigger event when transmitter modulates last bit of a frame */
    PHHAL_HW_ISO3_TRIG_MODE_TX_RX_AFTER_LAST_TX_BIT = 0x4U,                                     /**< Trigger event when transmitter activates receiver after last TX bit */
    PHHAL_HW_ISO3_TRIG_MODE_TX_SOC_RX               = 0x5U,                                     /**< Trigger event when SOC was detected by receiver */
    PHHAL_HW_ISO3_TRIG_MODE_TX_LMA_RX_EVENT         = 0x6U,                                     /**< Trigger event when LMA event (8 subcarrier modulations) was detected */
    PHHAL_HW_ISO3_TRIG_MODE_RFU_1                   = 0x7U,                                     /**< RFU */
    PHHAL_HW_ISO3_TRIG_MODE_RFU_2                   = 0x8U,                                     /**< RFU */
    PHHAL_HW_ISO3_TRIG_MODE_RX_1ST_STATE_NOT_IDLE   = 0x9U,                                     /**< Trigger event when Receiver leaves IDLE STATE */
    PHHAL_HW_ISO3_TRIG_MODE_RFU_4                   = 0xAU,                                     /**< RFU */
    PHHAL_HW_ISO3_TRIG_MODE_RFU_5                   = 0xBU,                                     /**< RFU */
    PHHAL_HW_ISO3_TRIG_MODE_RX_1ST_MOD              = 0xCU,                                     /**< Trigger event when receiver detected first modulation */
    PHHAL_HW_ISO3_TRIG_MODE_RX_SOC                  = 0xDU,                                     /**< Trigger event when receiver detected SOC */
    PHHAL_HW_ISO3_TRIG_MODE_RX_BIT_NUM              = 0xEU,                                     /**< Trigger event when received a certain bit (see TRIGGER_CTRL1 */
    PHHAL_HW_ISO3_TRIG_MODE_RX_EOC                  = 0xFU                                      /**< Trigger event when recieved EOC */
};
/** @} */

/** \name Trigger 2 Signal ID */
/** @{ */
enum Trigger1SignalId
{
    PHHAL_HW_ISO3_TRIG1_SIGNAL_ADC_INPUT            = 0x0U,                                     /**< Signal on ADC input (@ 135.6 MHz) */
    PHHAL_HW_ISO3_TRIG1_SIGNAL_I_CHANNEL_DEMOD      = 0x1U,                                     /**< Signal after I channel demodulator */
};
/** @}*/

/** \name Trigger 2 Signal ID */
/** @{ */
enum Trigger2SignalId
{
    PHHAL_HW_ISO3_TRIG2_SIGNAL_ADC_INPUT            = 0x0U,                                     /**< Signal on ADC input (@ 135.6 MHz) */
    PHHAL_HW_ISO3_TRIG2_SIGNAL_Q_CHANNEL_DEMOD      = 0x1U,                                     /**< Signal after Q channel demodulator */
};
/** @} */

/** \name Duty cycle coding for Felica card type */
/** @{ */
enum TxDutyCycle
{
    PHHAL_HW_ISO3_TX_DUTY_CYCLE_25 = 0x0U,                                                      /**< 25 percent duty cycle */
    PHHAL_HW_ISO3_TX_DUTY_CYCLE_33 = 0x1U,                                                      /**< 33.33 percent duty cycle */
    PHHAL_HW_ISO3_TX_DUTY_CYCLE_42 = 0x2U,                                                      /**< 41.66 percent duty cycle */
    PHHAL_HW_ISO3_TX_DUTY_CYCLE_50 = 0x3U,                                                      /**< 50 percent duty cycle */
    PHHAL_HW_ISO3_TX_DUTY_CYCLE_58 = 0x4U,                                                      /**< 58.33 percent duty cycle */
    PHHAL_HW_ISO3_TX_DUTY_CYCLE_67 = 0x5U,                                                      /**< 66.66 percent duty cycle */
    PHHAL_HW_ISO3_TX_DUTY_CYCLE_75 = 0x6U                                                       /**< 75 percent duty cycle */
};
/** @} */

typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalDirectDataParams;                                                                /**< pointer to the lower layers parameter structure to communicate with the fpga input and output buffer */
    void * pBalDispatchDataParams;                                                              /**< pointer to the lower layers parameter structure to communicate with the server dispatcher */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint16_t wRxExpectedInputBytes;                                                             /**< Expected size of exchange response in bytes */
    uint8_t * pIntBuffer;                                                                       /**< Internal command buffer (i.e. fpga input buffer). */
    uint16_t wIntNumValidBytes;                                                                 /**< Number of valid bytes within the internal buffer */
    uint16_t wIntNumValidRxBytes;                                                               /**< Number of valid bytes within the internal buffer after bal receive */
    uint16_t wMaxIntBufferSizeBytes;                                                            /**< size of internal command buffer in bytes */
    uint8_t bCardType;                                                                          /**< Type of card for which the HAL is configured for. */
    uint16_t wCfgShadow[PHHAL_HW_ISO3_SHADOW_COUNT];                                            /**< Configuration shadow; Stores configuration for current cardtype. */
    uint16_t wCfgShadowCustom[PHHAL_HW_ISO3_SHADOW_COUNT_CUSTOM];                               /**< Configuration shadow for Custom configs; Stores configuration for current cardtype. */
    uint16_t wTimingMode;                                                                       /**< Current timing measurement mode. */
    uint8_t bLastBit;                                                                           /**< Last bit exchanged (0 if last bit was Zero 1 if last bit was One. */
    uint16_t wCurrentFieldStrength;                                                             /**< Current Fieldstrength in Permil. */
    uint32_t dwTxCtrl;                                                                          /**< Copy of the current state of the TX CTRL register. */
    uint32_t dwTxCtrl1;                                                                         /**< Copy of the current state of the TX CTRL 1 register. */
    uint32_t dwRxCtrl2;                                                                         /**< Copy of the current state of the RX CTRL 2 register. */
    uint32_t dwVenusClifRxConfigReg;                                                            /**< Copy of the current state of the PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG register. */
    uint32_t dwExchangeCounter;                                                                 /**< To assign every received packet to a transmitted packet */
    uint32_t dwNextTxBufferAddress;                                                             /**< either address for TX_DATA or TX_DATA_1. changes after every exchange! */
    uint32_t dwCurrentWaveShapeType;                                                            /**< id of curren wave shape, e.g. field on, field off or exchange*/
    uint16_t wLastFdtValueT1;                                                                   /**< measured time in fpga clocks for Trigger1 */
    uint16_t wLastFdtValueT2;                                                                   /**< measured time in fpga clocks for Trigger2 */
    uint32_t dwDacGain;                                                                         /**< Gain factor for envelope to control field strength */
    uint16_t wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_BUFFER_CONFIG_COUNT];              /**< Configs for fpga internal waveshape buffer */
    uint8_t bWaveShapeBufferSlotsBitmap[(PHHAL_HW_ISO3_MAX_NUMB_WS_SLOTS + 7) / 8];             /**< Bitmap indicates which slots are in use */
    uint8_t bCurrentWaveshapeSlot;                                                              /**< Slot ID of current wave shape */
    uint32_t dwIIRFilterCtrl;                                                                   /**< Register to control the IIR filter */
    uint32_t dwMlsWhiteNoiseCtrlReg;                                                            /**< Register to control the white noise generator extension in the transmitter */
    uint32_t dwRxStatReg;                                                                       /**< Register containing the receiver status */
    uint32_t dwDgrmSigDetTh;                                                                    /**< threshold of the signal detector in the venus receiver */
    uint32_t dwPicc2PcdDelay;                                                                   /**< Delay in ETUs {RX EOF, TX SOF} */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint8_t bRfResetAfterTo;                                                                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
    uint32_t dwFieldOffTimeCycles;                                                              /**< Field-Off-Time in clock cycles. */
    uint8_t bDisableAGC;                                                                        /**< Disables Automatic Gain control for exchange, rf on/off, field reset */
    uint32_t dwFieldRecoveryTimeCycles;                                                         /**< Field-Recovery time in clock cycles */
    uint8_t * pWaveShapeShadowBuffer;
    uint32_t dwWaveShapeShadowBufferSizeByte;
    uint32_t dwPcd2PiccTimeout;                                                                 /**< Maximum time {Last Tx bit, Rx EOF} in FPGA clock periods [1/135.6MHz] */
    uint16_t wDebugConfig0;                                                                     /**< Config for the debug outputs */
    uint16_t wDebugConfig1;                                                                     /**< Config for the debug outputs */
    uint16_t wDebugConfig2;                                                                     /**< Config for the debug outputs */
    uint16_t wDebugConfig3;                                                                     /**< Config for the debug outputs */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint16_t WaveShapePrescalePercent;                                                          /**< Sets the max pattern percent for waveshapes */
    uint32_t dwPeakSensSignal;                                                                  /**< Peak SENS signal value */
    uint32_t dwPeakCalSignal;                                                                   /**< Peak CAL signal value */
    uint8_t pbOvsFilterSlotsBitmap[(PHHAL_HW_ISO3_OVS_FILTER_MAX_SLOT_NUM + 7) / 8];            /**< Bitmap indicating which overshoot filter slots within FPGA memory are in use */
} phhalHw_ISO3_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_ISO3_Init(
        phhalHw_ISO3_DataParams_t * pDataParams,                                                /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,
        uint16_t wTxBufSize,
        uint8_t * pRxBuffer,
        uint16_t wRxBufSize,
        uint32_t * pIntBuffer,
        uint32_t wIntBufferSize,
        uint8_t * pWaveShapeShadowBuffer,
        uint32_t dwWaveShapeBufferSizeByte
    );
/**
 * end of group phhalHw_ISO3
 * @}
 * \endcond
 */
#endif /* NXPBUILD__PHHAL_HW_ISO3 */

#ifdef NXPBUILD__PHHAL_HW_PCSC

/**
 * \defgroup phhalHw_Pcsc Component : Pcsc
 * \brief PCSC HAL component
 *
 * @{
 */
#define PHHAL_HW_PCSC_ID                                                                0x11U   /**< ID for PCSC HAL component */

/** \name PCSC specific set/get configurations. */
/** @{ */
#define PHHAL_HW_PCSC_CONFIG_FSDI                           (PH_CONFIG_CUSTOM_BEGIN + 0x01U)    /**< Frame size for IFD Integer (FSDI). */
#define PHHAL_HW_PCSC_CONFIG_FSCI                           (PH_CONFIG_CUSTOM_BEGIN + 0x02U)    /**< Frame size for ICC Integer (FSCI). */
#define PHHAL_HW_PCSC_CONFIG_FWTI                           (PH_CONFIG_CUSTOM_BEGIN + 0x03U)    /**< Frame waiting Time Integer (FWTI) of the current ICC. */
#define PHHAL_HW_PCSC_CONFIG_MAX_SPEED                      (PH_CONFIG_CUSTOM_BEGIN + 0x04U)    /**< Maximum communication speed supported by the IFD. */
#define PHHAL_HW_PCSC_CONFIG_CURRENT_SPEED                  (PH_CONFIG_CUSTOM_BEGIN + 0x05U)    /**< Communication speed is set or will be set for the current ICC. */
#define PHHAL_HW_PCSC_CONFIG_MOD_INDEX                      (PH_CONFIG_CUSTOM_BEGIN + 0x06U)    /**< Modulation index (can be used for all ICCs, where modulation index is required). */
#define PHHAL_HW_PCSC_CONFIG_PCB                            (PH_CONFIG_CUSTOM_BEGIN + 0x07U)    /**< PCB for ISO/IEC 14443. */
#define PHHAL_HW_PCSC_CONFIG_CID                            (PH_CONFIG_CUSTOM_BEGIN + 0x08U)    /**< CID for ISO/IEC 14443. */
#define PHHAL_HW_PCSC_CONFIG_NAD                            (PH_CONFIG_CUSTOM_BEGIN + 0x09U)    /**< NAD for ISO/IEC 14443. */
#define PHHAL_HW_PCSC_CONFIG_DATA_CODING                    (PH_CONFIG_CUSTOM_BEGIN + 0x0BU)    /**< Data coding (IFD to ICC) for ISO/IEC 15693. */
#define PHHAL_HW_PCSC_CONFIG_EXCHANGE                       (PH_CONFIG_CUSTOM_BEGIN + 0x0CU)    /**< Set/Get HAL exchange.type; #PHHAL_HW_PCSC_TRANSPARENT or #PHHAL_HW_PCSC_NORMAL (default). */
#define PHHAL_HW_PCSC_CONFIG_STATUS_BYTES                   (PH_CONFIG_CUSTOM_BEGIN + 0x0DU)    /**< Get error status code (SW1 SW2). */
#define PHHAL_HW_PCSC_CONFIG_KEY_LOCATION                   (PH_CONFIG_CUSTOM_BEGIN + 0x0EU)    /**< Key location for PCSC load key command (#PHHAL_HW_PCSC_KEY_LOC_VOLATILE or #PHHAL_HW_PCSC_KEY_LOC_NONVOLATILE). */
/** @} */

/** \name PCSC Communication modes. */
/** @{ */
#define PHHAL_HW_PCSC_TRANSPARENT                                                       0U      /**< Exchange using PCSC transparent exchange command \ref phhalHw_Pcsc_Cmd_Transceive. */
#define PHHAL_HW_PCSC_NORMAL                                                            1U      /**< Exchange the data as it is; No formatting done by HAL; Just calls BAL exchange. */
/** @} */

/**
 * \brief PCSC HAL component.
 * This structure holds all the data that are required for the PCSC HAL
 * operations.
 */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY!. */
    void  * pBalDataParams;                                                                     /**< Pointer to the BAL parameter structure. */
    void  * pKeyStoreDataParams;                                                                /**< Pointer to the Key Store parameter structure. */
    uint8_t bTxLastBits;                                                                        /**< TxLastbits for ExchangeRaw function. */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC. */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for (if any). */
    uint16_t wErrorCode;                                                                        /**< SW1 SW2 error code received in response from reader. */
    uint8_t bRxLastBits;                                                                        /**< Number of valid bits in last byte received. */
    uint8_t bCollPos;                                                                           /**< Collision position in received data. */
    uint8_t bTransparentSession;                                                                /**< Flag to indicate transparent session is open. */
    uint32_t dwTimeoutUs;                                                                       /**< Receive timeout in microseconds. */
    uint8_t bSak;                                                                               /**< Type A SAK; Updated when switch protocol to layer 3A is success. */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint16_t wTxRxFlag;                                                                         /**< Transmission and Reception Flag to be used with \ref phhalHw_Pcsc_Cmd_TxRxFlag. */
    uint8_t bExchangeType;                                                                      /**< Type of HAL exchange to use. #PHHAL_HW_PCSC_TRANSPARENT (default) or #PHHAL_HW_PCSC_NORMAL. */
    uint16_t wAdditionalInfo;                                                                   /**< Additional Information. */
    uint8_t bKeyLocation;                                                                       /**< Key location for PCSC load key command (#PHHAL_HW_PCSC_KEY_LOC_VOLATILE or #PHHAL_HW_PCSC_KEY_LOC_NONVOLATILE). */
    uint8_t bTxDataRate;                                                                        /**< Type A TX data rate. */
    uint8_t bRxDataRate;                                                                        /**< Type A RX data rate. */
} phhalHw_Pcsc_DataParams_t;

/**
 * \brief Initializes the PCSC HAL component.
 *
 * This function has to be called as part of the initialization of the
 * HAL layer of the application that intends to use any PCSC reader.
 */
phStatus_t phhalHw_Pcsc_Init(
        phhalHw_Pcsc_DataParams_t * pDataParams,                                                /**<[In] Pointer to data parameter of this HAL layer. */
        uint16_t wSizeOfDataParams,                                                             /**<[In] Size of the data parameter passed. */
        void * pBalDataParams,                                                                  /**<[In] Pointer to BAL parameter structure. */
        void * pKeyStoreDataParams,                                                             /**<[In] Pointer to Key Store parameter structure. */
        uint8_t * pTxBuffer,                                                                    /**<[In] Pointer to global transmit buffer that will be used by Exchange(). */
        uint16_t wTxBufSize,                                                                    /**<[In] size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**<[In] Pointer to global receive buffer that will be used by Exchange(). */
        uint16_t wRxBufSize                                                                     /**<[In] Size of the global receive buffer. */
    );

/**
 * end of group phhalHw_Pcsc
 * @}
 */
#endif  /* NXPBUILD__PHHAL_HW_PCSC */

#ifdef NXPBUILD__PHHAL_HW_PROXILAB

/**
 * \cond INCLUDE_INTERNALS
 * \defgroup phhalHw_ProxiLAB Component : ProxiLAB
 * \brief ProxiLAB Reader HAL.
 * @{
 */

#define PHHAL_HW_PROXILAB_ID                                                            0x0C    /**< ID for ProxiLAB HAL component. */

/** \name ProxiLAB specific configs */
/** @{ */
#define PHHAL_HW_PROXILAB_CONFIG_DEMODULATORINPUT               (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Set the input of the demodulator one of ProxiLABDemodulatorInput */
#define PHHAL_HW_PROXILAB_CONFIG_POWERLEVEL                     (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Set the power level of the ProxiLab output (0..1023) */
#define PHHAL_HW_PROXILAB_CONFIG_DISPLAY_VERBOSE                (PH_CONFIG_CUSTOM_BEGIN + 2)    /**< Enable or disable the verbose output (default is enabled) */
#define PHHAL_HW_PROXILAB_CONFIG_CLOCK_SOURCE                   (PH_CONFIG_CUSTOM_BEGIN + 3)    /**< select the source of the clock. one of ProxiLABClockSource */
#define PHHAL_HW_PROXILAB_CONFIG_MODULATOR_OUTPUT               (PH_CONFIG_CUSTOM_BEGIN + 4)    /**< select the rf output mode. one of ProxiLABModulatorOutput */
#define PHHAL_HW_PROXILAB_CONFIG_DEMODULATOR_AUTO_THRESHOLD     (PH_CONFIG_CUSTOM_BEGIN + 5)    /**< Reader Rx DemodulatorAutoThreshold is enabled or disabled (Values PH_ON, PH_OFF) */
#define PHHAL_HW_PROXILAB_CONFIG_DEMODULATOR_THRESHOLD          (PH_CONFIG_CUSTOM_BEGIN + 6)    /**< Reader Rx DemodulatorThreshold from (0..255) */
/** @} */

/** \name Clock Source Configuration Values */
/** @{ */
enum ProxiLABClockSource
{
    PHHAL_HW_PROXILAB_CLOCK_SOURCE_INT =       0x00U,                                           /**< Use embedded PLL. */
    PHHAL_HW_PROXILAB_CLOCK_SOURCE_EXT_DIG =   0x01U,                                           /**< Use SMA3 as a clock input. The clock frequency is 2 times the carrier frequency.Ex : Fc = 13,56MHz => Digital Clock = 27,12MHz. */
    PHHAL_HW_PROXILAB_CLOCK_SOURCE_EXT_ANA =   0x02U                                            /**< Use SMA1 as a carrier input. The voltage range is [1V;1,5V]. There is no processing on this signal. */
};
/** @} */

/** \name Demodulator Input Configuration Values */
/** @{ */
enum ProxiLABDemodulatorInput
{
    PHHAL_HW_PROXILAB_DEMODULATOR_RF_OUT =     0x00U,                                           /**< Selects the RF OUT antenna (standard case: the antenna transmits and receives). */
    PHHAL_HW_PROXILAB_DEMODULATOR_RF_IN =      0x01U,                                           /**< Selects RF IN connector (the RF OUT antenna transmits, and any other probe is connected to RF IN to receive). */
    PHHAL_HW_PROXILAB_DEMODULATOR_BRIDGE =     0x03U,                                           /**< Selects RF IN HDMI connector (test assembly case: the RF OUT antenna transmits, and the bridge is connected to HDMI to receive). */
    PHHAL_HW_PROXILAB_DEMODULATOR_BRIDGE_SSB = 0x05U,                                           /**< Selects RF IN HDMI connector with SSB filter (test assembly case: the RF OUT antenna transmits, and the bridge is connected to HDMI to receive). */
    PHHAL_HW_PROXILAB_DEMODULATOR_SMA2 =       0x04U                                            /**< Selects SMA2 (TTL) connector. */
};
/** @} */

/** \name Modulator Output Configuration Values */
/** @{ */
enum ProxiLABModulatorOutput
{
    PHHAL_HW_PROXILAB_MODULATOR_RF_OUT             = 0x00U,                                     /**< Selects RF OUT as output. */
    PHHAL_HW_PROXILAB_MODULATOR_AWG_OUT            = 0x01U,                                     /**< Selects AWG OUT as output. */
    PHHAL_HW_PROXILAB_MODULATOR_RF_OUT_DUAL        = 0x02U,                                     /**< Enables both outputs. */
    PHHAL_HW_PROXILAB_MODULATOR_AWG_CARRIER_RF_OUT = 0x03U                                      /**< Selects AWG_CARRIER_RF_OUT. */
};
/** @} */

#define PHHAL_HW_PROXILAB_SHADOW_COUNT                                                  0x000FU /**< Number of shadowed configurations. */
#define PHHAL_HW_PROXILAB_DEFAULT_TIMEOUT_US                                            5000U   /**< Default timeout in milliseconds. */

/** \brief ProxiLAB HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this HAL component, NEVER MODIFY! */
    phbalReg_ProxiLAB_DataParams_t * pBalDataParams;                                            /**< pointer to the lower layers parameter structure */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint8_t bCardType;                                                                          /**< Type of card for which the hal is configured for. */
    uint8_t bTimeoutUnit;                                                                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint16_t wCfgShadow[PHHAL_HW_PROXILAB_SHADOW_COUNT];                                        /**< Configuration shadow; Stores configuration for current cardtype. */
    /*uint16_t wTimingMode;*/                                                                       /**< Current timing measurement mode. */
    uint16_t wAdditionalInfo;                                                                   /**< Storage for additional error information. */
    uint16_t wFieldOffTime;                                                                     /**< Field-Off-Time in milliseconds. */
    uint16_t wFieldRecoveryTime;                                                                /**< Field-Recovery-Time in milliseconds. */
    uint32_t dwPowerLevel;                                                                      /**< Power Level of RF Field */
} phhalHw_ProxiLAB_DataParams_t;

/**
 * \brief Initialise the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_ProxiLAB_Init(
        phhalHw_ProxiLAB_DataParams_t * pDataParams,                                            /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        phbalReg_ProxiLAB_DataParams_t * pBalDataParams,                                        /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. */
    );

/**
 * end of group phhalHw_ProxiLAB
 * @}
 * \endcond
 */
#endif /* NXPBUILD__PHHAL_HW_PROXILAB */

#ifdef NXPBUILD__PHHAL_HW_DUT

/**
 * \defgroup phhalHw_DUT Component : DUT
 * \brief any Reader HAL
 * @{
 */

#define PHHAL_HW_DUT_ID                                                                 0x0BU   /**< ID for DUT HAL component. */

#define PHHAL_HW_DUT_RESERVED_BUFFER_LEN                                                11U     /**< Amount of needed and reserved memory for the protocol overhead. */

/** \brief DUT HAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                               /**< Layer ID for this component, NEVER MODIFY!. */
    void  * pBalDataParams;                                                                     /**< pointer to the lower layers parameter structure. */
    uint8_t bSlotNumber;                                                                        /**< Slot number. */
    uint8_t bTxLastBits;                                                                        /**< TxLastbits for ExchangeRaw function. */
    uint8_t * pTxBuffer;                                                                        /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                        /**< Size of the global transmit buffer. */
    uint16_t wTxBufLen;                                                                         /**< Number of valid bytes within the transmit buffer. */
    uint16_t wTxBufStart;                                                                       /**< Start position in case of buffering within the transmit buffer. */
    uint8_t * pRxBuffer;                                                                        /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                        /**< Size of the global receive buffer. */
    uint16_t wRxBufLen;                                                                         /**< Number of valid bytes within the receive buffer. */
    uint16_t wRxBufStartPos;                                                                    /**< Starting position within the global receive buffer. */
    uint16_t wTxBufStartPos;                                                                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
    uint8_t bCardType;                                                                          /**< Type of card for which the HAL is configured for. */
    uint16_t dwFdtPc;                                                                           /**< Current timing value backup for PC */
} phhalHw_DUT_DataParams_t;

/**
 * \brief Initialize the HAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_DUT_Init(
        phhalHw_DUT_DataParams_t * pDataParams,                                                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t bSlotNumber,                                                                    /**< [In] Slot number. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                     /**< [In] Size of the global receive buffer. */
    );

/**
 * end of group phhalHw_DUT
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT */

#ifdef NXPBUILD__PHHAL_HW

/**
 * \defgroup phhalHw Hardware Abstraction Layer
 * \brief These are the Components which are used to abstract the
 * functionality of the physical reader device to a generic interface.
 * @{
 */

/** \name Communication Configs */
/** @{ */
#define PHHAL_HW_CONFIG_PARITY                                                          0x0000U /**< Enable or Disable Parity. */
#define PHHAL_HW_CONFIG_TXCRC                                                           0x0001U /**< Enable or Disable TxCrc. */
#define PHHAL_HW_CONFIG_RXCRC                                                           0x0002U /**< Enable or Disable RxCrc. */
#define PHHAL_HW_CONFIG_TXLASTBITS                                                      0x0003U /**< Set number of valid bits of last Tx-Byte. */
#define PHHAL_HW_CONFIG_RXLASTBITS                                                      0x0004U /**< Get number of valid bits of last Rx-Byte. */
#define PHHAL_HW_CONFIG_RXALIGN                                                         0x0005U /**< Set Rx-Aligned Bits. */
#define PHHAL_HW_CONFIG_RXDEAFBITS                                                      0x0006U /**< Configure Receiver Deaf-Time in ETUs. */
#define PHHAL_HW_CONFIG_TXWAIT_US                                                       0x0007U /**< Set TxWait (= time between last RxIrq and Tx of succeeding frame) in microseconds. */
#define PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL                                              0x0008U /**< Enable or Disable clearing of bits after coll. */
#define PHHAL_HW_CONFIG_TXDATARATE                                                      0x0009U /**< Configure Data-Rate for Transmission. */
#define PHHAL_HW_CONFIG_RXDATARATE                                                      0x000AU /**< Configure Data-Rate for Reception. */
#define PHHAL_HW_CONFIG_MODINDEX                                                        0x000BU /**< Set modulation index (unit and value are hardware-dependent). */
#define PHHAL_HW_CONFIG_ASK100                                                          0x000CU /**< Enable (#PH_ON) or disable (#PH_OFF) 100% modulation. */
#define PHHAL_HW_CONFIG_TIMEOUT_VALUE_US                                                0x000DU /**< Set RC Timeout (in [us]). */
#define PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS                                                0x000EU /**< Set RC Timeout (in [ms]). */
#define PHHAL_HW_CONFIG_SUBCARRIER                                                      0x000FU /**< Subcarrier setting for ISO 15693. */
#define PHHAL_HW_CONFIG_TIMING_MODE                                                     0x0010U /**< Set the timing mode. */
#define PHHAL_HW_CONFIG_TIMING_US                                                       0x0011U /**< Retrieve elapsed time of RC timer ([us]). */
#define PHHAL_HW_CONFIG_TIMING_MS                                                       0x0012U /**< Retrieve elapsed time of RC timer ([ms]). */
#define PHHAL_HW_CONFIG_FIELD_OFF_TIME                                                  0x0013U /**< Set the field off time for field-reset ([ms]); \b Note: A value of \c 0 is not allowed. */
#define PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME                                             0x0014U /**< Set the field recovery time for field-reset ([ms]); \b Note: For ISO15693 operation, this should be at least set to \c 1ms. */
#define PHHAL_HW_CONFIG_SYMBOL_START                                                    0x0015U /**< Disable / Set the SOF symbol of a frame. */
#define PHHAL_HW_CONFIG_SYMBOL_END                                                      0x0016U /**< Disable / Set the EOF symbol of a frame. */
#define PHHAL_HW_CONFIG_TIMER_DEVIATION_PROXIMITY_CHECK                                 0x8013U /**< Configure the Timer deviation percentage. This configuration should be used while performing ProximityCheck command. */
#define PHHAL_HW_CONFIG_ASYNCHRONOUS_IN_REPLY_PROCESS                                   0x8014U /**< Enable/Disable Handling of ISO-15693 ASynchronous In-Reply process command */
/** @} */

/**
 * \name
 * Sets the CRC type for Tx(#PHHAL_HW_CONFIG_TXCRC) and Rx(#PHHAL_HW_CONFIG_RXCRC).
 * Values can be one of\n
 * #PHHAL_HW_CRCTYPE_CRC5,\n
 * #PHHAL_HW_CRCTYPE_CRC16\n
 * optionally combined with the following inversion option.\n
 * #PHHAL_HW_CRCOPTION_INVERTED.
 */
/** @{ */
#define PHHAL_HW_CONFIG_CRCTYPE                                                         0x0017U
#define PHHAL_HW_CONFIG_FRAME_SYNC                                                      0x0017U /**< Set Frame Sync for Rc663 */
#define PHHAL_HW_CONFIG_TXDATANUM                                                       0x0018U /**< Tranmitter Data number register. */
#define PHHAL_HW_CONFIG_CARD_TYPE                                                       0x0019U /**< Getter for the current card type. */
#define PHHAL_HW_CONFIG_SET_READER_IC                                                   0x0020U /**< Gets or Set the Reader IC information. This will be helpful in Sam is connected in X mode. */
#define PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1                                              0x002EU /**< Disable MIFARE(R) Classic Crypto1. */
#define PHHAL_HW_CONFIG_ADDITIONAL_INFO                                                 0x002FU /**< Set / Get additional information. */
#define PHHAL_HW_CONFIG_RXBUFFER_STARTPOS                                               0x0030U /**< Start position of RX buffer to be used (never changed by HAL). */
#define PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE                                                0x0031U /**< Buffer size of RX buffer set at Init. */
#define PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE                                                0x0032U /**< Buffer size of TX buffer set at Init. */
#define PHHAL_HW_CONFIG_TXBUFFER_LENGTH                                                 0x0033U /**< Amount of valid bytes in TX buffer. */
#define PHHAL_HW_CONFIG_TXBUFFER                                                        0x0034U /**< Access the TxBuffer at the address defined by #PHHAL_HW_CONFIG_ADDITIONAL_INFO. */
#define PHHAL_HW_CONFIG_MAX_PRECACHED_BYTES                                             0x0035U /**< Configures the max. number of bytes which are precached prior to command execution. */
#define PHHAL_HW_CONFIG_BAL_CONNECTION                                                  0x0040U /**< Set the BAL connection type. The default value is always #PHHAL_HW_BAL_CONNECTION_RS232. */
#define PHHAL_HW_CONFIG_SERIAL_BITRATE                                                  0x0041U /**< Bitrate for serial communication. */
#define PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT                                              0x0050U /**< Perform an Rf-Reset in case of a timeout (only for \ref phhalHw_Exchange). */
#define PHHAL_HW_CONFIG_SETMINFDT                                                       0x0051U /**< Set/Reset minimum FDT (only for PC)*/
/** @} */

/** \name
 * Enable symbol send (#PH_ON).\n
 * ISO/IEC 18000p3m3 protocol will use data send (#PH_OFF).
 */
/** @{ */
#define PHHAL_HW_CONFIG_SET_SYMBOL_SEND                                                 0x005DU
#define PHHAL_HW_CONFIG_COMMUNICATION_CONFIG                                            0x0089U /**< Change the setting for the Socket communication (only for \ref phhalHw_Exchange in combination with RdCardSim). */
/** @} */

/** \name Configuration for I2C communication */
/** @{ */
#define PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE                                          0x0060U /**< Configure communication type.
                                                                                                 *      - \ref PHHAL_HW_SAM_COMMUNICATION_TYPE_TDA "TDA"
                                                                                                 *      - \ref PHHAL_HW_SAM_COMMUNICATION_TYPE_I2C "I2C"
                                                                                                 */
#define PHHAL_HW_CONFIG_I2C_SLAVE_ADDRESS                                               0x0061U /**< Config the I2C Slave address to be used. */
#define PHHAL_HW_CONFIG_I2C_BITRATE                                                     0x0062U /**< Config the Data-Rate for I2C communication.
                                                                                                 *   Supported values are
                                                                                                 *      - 0x00: 100 Kbits/s (Standard Mode)
                                                                                                 *      - 0x01: 400 Kbits/s (Fast Mode)
                                                                                                 *      - 0x02: 1 Mbits/s (Fast Mode Plus)
                                                                                                 *      - 0x03: 3.4 Mbits/s (High Speed Mode)
                                                                                                 */
#define PHHAL_HW_CONFIG_I2C_TIMEOUT_MS                                                  0x0063U /**< Configure I2C polling Timeout in terms of millisecond. */
/** @} */

/** \name Custom Error Codes (start with offset higher than max value of PHHAL_HW_SAMAV2_ERR/PHHAL_HW_SAMAV3_ERR error codes */
/** @{ */
#define PHHAL_HW_ISO3_ERR_ISO3                                      (PH_ERR_CUSTOM_BEGIN + 48)  /**< Non standard error. Read wAddionalInfo to get error code. */
#define PHHAL_HW_ISO3_ERR_IIR_FILTER_OVERFLOW                       (PH_ERR_CUSTOM_BEGIN + 49)  /**< IIR filter overflow error. */
#define PHHAL_HW_ISO3_ERR_INCOMPATIBLE_SERVER_VERSION               (PH_ERR_CUSTOM_BEGIN + 50)  /**< Incompatible server version for current ISO3 driver implementation */
#define PHHAL_HW_ISO3_ERR_INCOMPATIBLE_FPGA_VERSION                 (PH_ERR_CUSTOM_BEGIN + 51)  /**< Incompatible FPGA version for current ISO3 driver implementation */
#define PHHAL_HW_RDFPGAV6_ERR_DLL                                   (PH_ERR_CUSTOM_BEGIN + 52)  /**< phCDllReader.dll threw an non standard error read wAddionalInfo to get error code. */
#define PHHAL_HW_MP300_ERR_MP300                                    (PH_ERR_CUSTOM_BEGIN + 53)  /**< Non standard error. Read wAddionalInfo to get error code. */
#define PHHAL_HW_MP300_ERR_ADJUST_RX_CHANNEL_2                      (PH_ERR_CUSTOM_BEGIN + 54)  /**< Adjust rx Channel failed. */
#define PHHAL_HW_MP300_ERR_FDT_METHODE2_NOT_SUPPORTED               (PH_ERR_CUSTOM_BEGIN + 55)  /**< FDT Measurement methode 2 is not supported with the current response. */
/** @} */

/** \name Card Configs */
/** @{ */
#define PHHAL_HW_CARDTYPE_CURRENT                                                       0x0000U /**< Re-Apply settings for current Communication Mode. */
#define PHHAL_HW_CARDTYPE_ISO14443A                                                     0x0001U /**< ISO/IEC 14443A Mode. */
#define PHHAL_HW_CARDTYPE_ISO14443B                                                     0x0002U /**< ISO/IEC 14443B Mode. */
#define PHHAL_HW_CARDTYPE_FELICA                                                        0x0003U /**< Felica (JIS X 6319) Mode. */
#define PHHAL_HW_CARDTYPE_ISO15693                                                      0x0004U /**< ISO/IEC 15693 Mode. */
#define PHHAL_HW_CARDTYPE_ICODEEPCUID                                                   0x0005U /**< NXP I-Code EPC/UID Mode. */
#define PHHAL_HW_CARDTYPE_I18000P3M3                                                    0x0006U /**< ISO/IEC 18000-3 Mode3. */
#define PHHAL_HW_CARDTYPE_I18092MPI                                                     0x0007U /**< ISO/IEC ISO18092 (NFC) Passive Initiator Mode. */
#define PHHAL_HW_CARDTYPE_I18092MPT                                                     0x0008U /**< ISO/IEC ISO18092 (NFC) Passive Target Mode. */
/** @} */

/** \name CRC Values */
/** @{ */
#define PHHAL_HW_CRCTYPE_CRC5                                                           0x0001U /**< 5 bit length, polynomial x^5+x^3+1U, preset value 01001b, residue 00000b. */
#define PHHAL_HW_CRCTYPE_CRC16                                                          0x0002U /**< ISO/IEC 13239, 16 bit length, polynomial x^16+x^12+x^5+1U, preset value 0xFFFF, residue 0x1D0F. */
#define PHHAL_HW_CRCOPTION_INVERTED                                                     0x0100U /**< To indicate that the CRC generated should be inverted. This should be ORed with the CRCTYPEs above. */
/** @} */

/** \name Rf Datarate Values */
/** @{ */
#define PHHAL_HW_RF_DATARATE_106                                                        0x0000U /**< Data rate setting for 106 kBit/s. */
#define PHHAL_HW_RF_DATARATE_212                                                        0x0001U /**< Data rate setting for 212 kBit/s. */
#define PHHAL_HW_RF_DATARATE_424                                                        0x0002U /**< Data rate setting for 424 kBit/s. */
#define PHHAL_HW_RF_DATARATE_848                                                        0x0003U /**< Data rate setting for 848 kBit/s. */
#define PHHAL_HW_RF_DATARATE_1695                                                       0x0004U /**< Data rate setting for 1695 kBit/s. */
#define PHHAL_HW_RF_DATARATE_3390                                                       0x0005U /**< Data rate setting for 3390 kBit/s. */
#define PHHAL_HW_RF_DATARATE_6780                                                       0x0006U /**< Data rate setting for 6780 kBit/s. */
#define PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256                                            0x000AU /**< Data rate setting for ISO15693 '1 out of 256' mode. 1.66kb/s */
#define PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4                                              0x000BU /**< Data rate setting for ISO15693 '1 out of 4' mode. 26 kb/s */
#define PHHAL_HW_RF_RX_DATARATE_LOW                                                     0x000CU /**< Data rate setting for ISO15693 low datarate. */
#define PHHAL_HW_RF_RX_DATARATE_HIGH                                                    0x000DU /**< Data rate setting for ISO15693 high datarate. */
#define PHHAL_HW_RF_RX_DATARATE_FAST_LOW                                                0x000EU /**< Data rate setting for ISO15693 fast (low*2) datarate (FAST INVENTORY (PAGE) READ). */
#define PHHAL_HW_RF_RX_DATARATE_FAST_HIGH                                               0x000FU /**< Data rate setting for ISO15693 fast (high*2) datarate (FAST INVENTORY (PAGE) READ). */
#define PHHAL_HW_RF_DATARATE_EPCUID                                                     0x0010U /**< Data rate setting for EPC/UID. */
#define PHHAL_HW_RF_TX_DATARATE_I18000P3M3                                              0x0011U /**< Tx Data rate setting for ISO18000-3 Mode3. */
#define PHHAL_HW_RF_RX_DATARATE_I18000P3M3_MAN2                                         0x0012U /**< Rx Data rate setting for ISO18000-3 Mode3 Manchester 2SCP. */
#define PHHAL_HW_RF_RX_DATARATE_I18000P3M3_MAN4                                         0x0013U /**< Rx Data rate setting for ISO18000-3 Mode3 Manchester 4SCP. */
#define PHHAL_HW_RX_I18000P3M3_FL_423_MAN2                                              0x0013U /**< ICode ISO18000-3 Mode3 424 kBit/s (M=2). */
#define PHHAL_HW_RX_I18000P3M3_FL_847_MAN2                                              0x0014U /**< ICode ISO18000-3 Mode3 847 kBit/s (M=2). */
#define PHHAL_HW_RX_I18000P3M3_FL_423_MAN4                                              0x0015U /**< ICode ISO18000-3 Mode3 424 kBit/s (M=4). */
#define PHHAL_HW_RX_I18000P3M3_FL_847_MAN4                                              0x0016U /**< ICode ISO18000-3 Mode3 847 kBit/s (M=4). */
#define PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_53                                           0x0017U /**< Data rate setting for ISO15693 '1 out of 4' mode. 53 kb/s */
#define PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_106                                          0x0018U /**< Data rate setting for ISO15693 '1 out of 4' mode. 106 kb/s */
#define PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_212                                          0x0019U /**< Data rate setting for ISO15693 '1 out of 4' mode. 212 kb/s */
#define PHHAL_HW_RF_I15693_53KBPS_DATARATE                                              0x001BU /**< Data rate setting for 53 kbps. Used for Testing */
/** @} */

/** \name Rf Subcarrier Values */
/** @{ */
#define PHHAL_HW_SUBCARRIER_SINGLE                                                      0x0000U /**< Sub carrier setting for ISO15693 single subcarrier. */
#define PHHAL_HW_SUBCARRIER_DUAL                                                        0x0001U /**< Sub carrier setting for ISO15693 / EPC V2 dual subcarrier. */
#define PHHAL_HW_SUBCARRIER_QUAD                                                        0x0002U /**< Sub carrier setting for EPC V2 quad subcarrier. */
/** @} */

/** \name Time Units */
/** @{ */
#define PHHAL_HW_TIME_MICROSECONDS                                                      0x0000U /**< Time in microseconds (us). */
#define PHHAL_HW_TIME_MILLISECONDS                                                      0x0001U /**< Time in milliseconds (ms). */
/** @} */

/** \name Timing Mode settings */
/** @{ */
#define PHHAL_HW_TIMING_MODE_OFF                                                        0x0000U /**< Time Measurement disabled. */
#define PHHAL_HW_TIMING_MODE_FDT                                                        0x0001U /**< Measures time from Tx-End until Rx-Begin. */
#define PHHAL_HW_TIMING_MODE_COMM                                                       0x0002U /**< Measures the duration of the complete communication. */
#define PHHAL_HW_TIMING_MODE_OPTION_MASK                                                0xFF00U /**< Additional option-mask for time measurement. */
#define PHHAL_HW_TIMING_MODE_OPTION_DEFAULT                                             0x0000U /**< The timing value is cummulated on each Exchange until it gets retrieved by \ref phhalHw_GetConfig. */
#define PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR                                           0x0100U /**< Automatically resets the current timing value prior to any communication. */
/** @} */

/** \name Field Reset default times */
/** @{ */
#define PHHAL_HW_FIELD_OFF_DEFAULT                                                      0x0005U /**< Default Field-Off time. */
#define PHHAL_HW_FIELD_RECOVERY_DEFAULT                                                 0x0005U /**< Default Field-Recovery time. */
/** @} */

/** \name Frame Symbols (use #PH_OFF to disable) */
/** @{ */
#define PHHAL_HW_SYMBOL_ICODEEPCUID_SSOF                                                0x0001U /**< ICode EPC/UID Short Start Of Frame. */
#define PHHAL_HW_SYMBOL_ICODEEPCUID_LSOF                                                0x0002U /**< ICode EPC/UID Long Start Of Frame. */
#define PHHAL_HW_SYMBOL_ICODEEPCUID_CEOF                                                0x0003U /**< ICode EPC/UID Common End Of Frame. */
#define PHHAL_HW_SYMBOL_ICODEEPCUID_CS                                                  0x0004U /**< ICode EPC/UID Close Slot. */
#define PHHAL_HW_SYMBOL_I15693_SOF                                                      0x0005U /**< ISO15693 SOF (aumatically choosen between 1/256 and 1/4). */
#define PHHAL_HW_SYMBOL_I18000P3M3_PREAMBLE                                             0x0006U /**< ISO18000-3 Mode3 Preamble. */
#define PHHAL_HW_SYMBOL_I18000P3M3_FSYNC                                                0x0007U /**< ISO18000-3 Mode3 Frame-Sync. */
/** @} */

/** \name MIFARE Classic Authenticate Options */
/** @{ */
#define PHHAL_HW_MFC_KEYA                                                               0x0AU   /**< MIFARE Classic Key Type A. */
#define PHHAL_HW_MFC_KEYB                                                               0x0BU   /**< MIFARE Classic Key Type B. */
#define PHHAL_HW_MFC_KEY_LENGTH                                                         0x06U   /**< Length of a MIFARE Classic key (for completeness). */
/** @} */

/** \name BAL Connection types */
/** @{ */
#define PHHAL_HW_BAL_CONNECTION_RS232                                                   0x0000U /**< ReaderIC is connected via RS232. */
#define PHHAL_HW_BAL_CONNECTION_SPI                                                     0x0001U /**< ReaderIC is connected via SPI. */
#define PHHAL_HW_BAL_CONNECTION_I2C                                                     0x0002U /**< ReaderIC or SAM is connected via I2C. */
/** @} */

/**  \name RS232 Bitrate Values*/
/** @{ */
#define PHHAL_HW_RS232_BITRATE_7200                                                     0x0000U /**< 7.200 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_9600                                                     0x0001U /**< 9.600 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_14400                                                    0x0002U /**< 14.400 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_19200                                                    0x0003U /**< 19.200 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_38400                                                    0x0004U /**< 38.400 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_57600                                                    0x0005U /**< 57.500 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_115200                                                   0x0006U /**< 115.200 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_128000                                                   0x0007U /**< 128.000 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_230400                                                   0x0008U /**< 230.400 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_460800                                                   0x0009U /**< 460.800 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_921600                                                   0x000AU /**< 921.600 kBit/s. */
#define PHHAL_HW_RS232_BITRATE_1228800                                                  0x000BU /**< 1.228.800 kBit/s. */
/** @} */

/** \name Min FDT values */
/** @{ */
#define PHHAL_HW_MINFDT_106_US                                                      (91 + 10)   /**< (n * 128 + 84) / 13.56; n = 9. */
#define PHHAL_HW_MINFDT_212_US                                                      (87 + 40)   /**< (n * 128 + 148) / 13.56; n = 8. */
#define PHHAL_HW_MINFDT_424_US                                                      (84 + 35)   /**< (n * 128 + 116) / 13.56; n = 8. */
#define PHHAL_HW_MINFDT_848_US                                                      (83 + 45)   /**< (n * 128 + 100) / 13.56; n = 8. */
/** @} */


/** \name Parameter values for ISO/IEC 18000p3m3 HAL APIs \n */
/** @{ */
#define PHHAL_HW_I18000P3M3_GET_MAX_RESPS                                               0U      /**< Gets responses from more than one slots limited by the RX buffer size and number of slots. */
#define PHHAL_HW_I18000P3M3_ONE_TS_ONLY                                                 1U      /**< Gets response for only one time slot */
#define PHHAL_HW_I18000P3M3_GET_TAG_HANDLE                                              2U      /**< Gets response for only one time slot. Also sends a ReqRN to get the tag handle for this slot. */
#define PHHAL_HW_I18000P3M3_PERFORM_RESIZE                                              3U      /**< Perform Resize Round */
#define PHHAL_HW_I18000P3M3_SESSION_S0                                                  0x00U   /**< Session S0. */
#define PHHAL_HW_I18000P3M3_SESSION_S2                                                  0x02U   /**< Session S2. */
#define PHHAL_HW_I18000P3M3_TARGET_INVENTORIED_S0                                       0x00U   /**< INVENTORIED_S0 target. */
#define PHHAL_HW_I18000P3M3_TARGET_INVENTORIED_S2                                       0x02U   /**< INVENTORIED_S2 target. */
#define PHHAL_HW_I18000P3M3_TARGET_SL                                                   0x04U   /**< SL target. */
#define PHHAL_HW_I18000P3M3_UPDN_INCREMENT                                              0x06U   /**< Increment Q. */
#define PHHAL_HW_I18000P3M3_UPDN_NOCHANGE                                               0x00U   /**< Do not change Q. */
#define PHHAL_HW_I18000P3M3_UPDN_DECREMENT                                              0x03U   /**< Decrement Q. */
#define PHHAL_HW_I18000P3M3_CMD_RESIZE_ROUND                                            0x09U   /**< ResizeRound command code. */
/** @} */

/** \defgroup phhalHw_Cmd_CommType CommunicationType
 * \brief Sam macros to configure communication type. The macros to be used if configuration identifier
 * is \ref PHHAL_HW_CONFIG_SAM_COMMUNICATION_TYPE "Communication Type" for the below mentioned interfaces
 *  - \ref phhalHw_SetConfig "Set Config"
 *  - \ref phhalHw_GetConfig "Get Config"
 * @{
 */
#define PHHAL_HW_SAM_COMMUNICATION_TYPE_TDA                                             0x01U       /**< SAM Communication type as TDA. SAM APDU's will be exchanged
                                                                                                     *   via TDA interface.
                                                                                                     */
#define PHHAL_HW_SAM_COMMUNICATION_TYPE_I2C                                             0x02U       /**< SAM Communication type as I2C. SAM APDU's will be exchanged
                                                                                                     *   via I2C interface.
                                                                                                     */
/**
 * end of group phhalHw_Sam_Cmd_I2CBitRate
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_I2CBitRate I2C_BitRate
 * \brief Sam macros to configure I2C BitRate. The macros to be used if configuration identifier
 * is \ref PHHAL_HW_CONFIG_I2C_BITRATE "I2C BitRate" for the below mentioned interfaces
 *  - \ref phhalHw_SetConfig "Set Config"
 *  - \ref phhalHw_GetConfig "Get Config"
 * @{
 */
#define PHHAL_HW_CONFIG_I2C_BITRATE_STANDARD_MODE                                       0x00U   /**< I2C bit rate configuration value for 100 KBits/s. */
#define PHHAL_HW_CONFIG_I2C_BITRATE_FAST_MODE                                           0x01U   /**< I2C bit rate configuration value for 400 KBits/s. */
#define PHHAL_HW_CONFIG_I2C_BITRATE_FAST_PLUS_MODE                                      0x02U   /**< I2C bit rate configuration value for 1 MBits/s. */
#define PHHAL_HW_CONFIG_I2C_BITRATE_HIGH_SPEED_MODE                                     0x03U   /**< I2C bit rate configuration value for 3.4 MBits/s. */
/**
 * end of group phhalHw_Sam_Cmd_I2CBitRate
 * @}
 */

/**
 * \brief Perform Data Exchange with Picc.
 *
 * \b wOption can be one of:\n
 * \li #PH_EXCHANGE_DEFAULT
 *
 * \b wOption can be combined with:\n
 * \li #PH_EXCHANGE_BUFFERED_BIT
 * \li #PH_EXCHANGE_LEAVE_BUFFER_BIT
 *
 * \ref phhalHw_Rc523 <em>Special Behaviour:</em>\n
 * If #PHHAL_HW_CARDTYPE_I18092MPT is chosen, Exchange performs the reception BEFORE the transmission.\n
 * For details refer to \ref phhalHw_Rc523.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER \b wOption is invalid.
 * \retval #PH_ERR_SUCCESS_INCOMPLETE_BYTE Operation successful, incomplete byte received; Retrieve number of valid bits of last byte with #PHHAL_HW_CONFIG_RXLASTBITS.
 * \retval #PH_ERR_IO_TIMEOUT No response detected within the configured time frame.
 * \retval #PH_ERR_INTEGRITY_ERROR Response received but CRC or Parity is invalid.
 * \retval #PH_ERR_COLLISION_ERROR Response received but a collision occured; Retrieve number of valid bits of last byte with #PHHAL_HW_CONFIG_RXLASTBITS.
 * \retval #PH_ERR_BUFFER_OVERFLOW Internal receive buffer is too small for transmission or smaller than the response.
 * \retval #PH_ERR_FRAMING_ERROR Frame format is either invalid for configured protocol or corrupted.
 * \retval #PH_ERR_PROTOCOL_ERROR Frame format is definitely invalid for configured protocol.
 * \retval #PH_ERR_READ_WRITE_ERROR Hardware problem.
 * \retval #PH_ERR_TEMPERATURE_ERROR Hardware problem.
 * \retval #PH_ERR_RF_ERROR Hardware problem.
 * \retval #PH_ERR_INTERFACE_ERROR Hardware problem.
 */
phStatus_t phhalHw_Exchange(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                                       /**< [In] Option parameter. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Data to transmit. */
        uint16_t wTxLength,                                                                     /**< [In] Number of bytes to transmit. */
        uint8_t ** ppRxBuffer,                                                                  /**< [Out] Pointer to received data. */
        uint16_t * pRxLength                                                                    /**< [Out] Number of received data bytes. */
    );

/**
 * \brief Write value to register address.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Hardware problem.
 */
phStatus_t phhalHw_WriteRegister(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bAddress,                                                                       /**< [In] Register Address. */
        uint8_t bValue                                                                          /**< [In] Register Value. */
    );

/**
    * \brief Read value from register address.
    *
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_INTERFACE_ERROR Hardware problem.
    */
phStatus_t phhalHw_ReadRegister(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bAddress,                                                                       /**< [In] Register Address. */
        uint8_t * pValue                                                                        /**< [Out] Register Value. */
    );

/**
 * \brief Execute bCmd command with pTxBuffer parameters.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Hardware problem.
 */
phStatus_t phhalHw_ExecuteCmd(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCmd,                                                                           /**< [In] Command to be executed on the Reader IC. */
        uint16_t wOption,                                                                       /**< [In] Option field for exchange. */
        uint8_t bIrq0WaitFor,                                                                   /**< [In] IRQs on IRQ0 reg to be enabled. */
        uint8_t bIrq1WaitFor,                                                                   /**< [In] IRQs on IRQ1 reg to be enabled. */
        uint8_t * pTxBuffer,                                                                    /**< [In] Buffer to be put into the FIFO before executing the command. */
        uint16_t wTxLength,                                                                     /**< [In] Length of the Tx buffer. */
        uint16_t wRxBufferSize,                                                                 /**< [In] Length of Rx buffer. */
        uint8_t * pRxBuffer,                                                                    /**< [out] Receive Buffer. */
        uint16_t * pRxLength                                                                    /**< [out] Length of received data(valid). */
    );

/**
 * \brief Apply protocol settings.
 *
 * To keep the current card type, use #PHHAL_HW_CARDTYPE_CURRENT for bCardType.\n
 *
 * \ref phhalHw_Rc523 <em>Special Behaviour:</em>\n
 * If #PHHAL_HW_CARDTYPE_I18092MPT is chosen, Rx- and TxBuffers have to be different.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER \b bCardType is invalid or not supported.
 * \retval #PH_ERR_INTERFACE_ERROR Hardware problem.
 */
phStatus_t phhalHw_ApplyProtocolSettings(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCardType                                                                       /**< [In] Type of card for which the HAL should be configured for. */
    );

/**
 * \brief Perform MIFARE(R) Classic Authentication using a key number.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER \b bKeyType, \b wKeyNo or \b wKeyVersion is invalid.
 * \retval #PH_ERR_IO_TIMEOUT Error in authentication.
 * \retval #PH_ERR_AUTH_ERROR Error in authentication.
 */
phStatus_t phhalHw_MfcAuthenticateKeyNo(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bBlockNo,                                                                       /**< [In] Blocknumber on card to authenticate to. */
        uint8_t bKeyType,                                                                       /**< [In] Either #PHHAL_HW_MFC_KEYA or #PHHAL_HW_MFC_KEYB. */
        uint16_t wKeyNo,                                                                        /**< [In] Key number to be used in authentication. */
        uint16_t wKeyVersion,                                                                   /**< [In] Key version to be used in authentication. */
        uint8_t * pUid                                                                          /**< [In] Serial number of current cascade level; uint8_t[4]. */
    );

/**
 * \brief Perform MIFARE(R) Classic Authentication using a key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER \b bKeyType is invalid.
 * \retval #PH_ERR_IO_TIMEOUT Error in authentication.
 * \retval #PH_ERR_AUTH_ERROR Error in authentication.
 */
phStatus_t phhalHw_MfcAuthenticate(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bBlockNo,                                                                       /**< [In] Blocknumber on card to authenticate to. */
        uint8_t bKeyType,                                                                       /**< [In] Either #PHHAL_HW_MFC_KEYA or #PHHAL_HW_MFC_KEYB */
        uint8_t * pKey,                                                                         /**< [In] Key to be used in authentication. */
        uint8_t * pUid                                                                          /**< [In] Serial number of current cascade level; uint8_t[4]. */
    );

/**
 * \brief Set configuration parameter.
 *
 * phhalHw_SamAV2 <em>Special Behaviour:</em>\n
 * In NonX-Mode every setting (except PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1 and PHHAL_HW_SAMAV2_CONFIG_HOSTMODE)\n
 * is passed on to the linked Reader-HAL.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_SetConfig(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                                       /**< [In] Configuration Identifier. */
        uint16_t wValue                                                                         /**< [In] Configuration Value. */
    );

/**
 * \brief Get configuration parameter.
 *
 * phhalHw_SamAV2 <em>Special Behaviour:</em>\n
 * In NonX-Mode every query (except #PHHAL_HW_CONFIG_RXLASTBITS, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1 and PHHAL_HW_SAMAV2_CONFIG_HOSTMODE)\n
 * is passed on to the linked Reader-HAL.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_PARAMETER_OVERFLOW The parameter value is larger than the range of \b pValue.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_GetConfig(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                                       /**< [In] Configuration Identifier. */
        uint16_t * pValue                                                                       /**< [Out] Configuration Value. */
    );

/**
 * \brief Switch on the RF-field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_FieldOn(
        void * pDataParams                                                                      /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Switch off the RF-field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_FieldOff(
        void * pDataParams                                                                      /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Perform RF-Reset.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_FieldReset(
        void * pDataParams                                                                      /**< [In] Pointer to this layer's parameter structure. */
    );
/**
 * \brief Blocks until the time given by \b wTimeout elapsed.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER \b bUnit is invalid.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_Wait(
    void * pDataParams,                                                                         /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bUnit,                                                                              /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
    uint16_t wTimeout                                                                           /**< [In] Timeout value. */
    );


/**
 * \brief ISO 18000p3m3 Inventory command.
 *
 *  Note: This function does
 *  ISO18000p3m3 BeginRound
 *  Sends an Ack
 *  Optionally sends a ReqRN
 *  Stores the information related to tag reply in the HAL RX Buffer
 *  Continues with NextSlot or else pauses and returns based on the option specified
 *  Can return multi card information based on the configuration option and the size of
 *  HAL RX Buffer.
 *  The bTSprocessing (Time slot processing behavior) can take the following values
 *  \li #PHHAL_HW_I18000P3M3_GET_MAX_RESPS - Gets responses from more than one slot limited by the Rx Buffer size and number of slots specified in begin round.
 *  \li #PHHAL_HW_I18000P3M3_ONE_TS_ONLY - Gets response for only one time slot. Here the number of time slots should be zero in the begin round command.
 *  \li #PHHAL_HW_I18000P3M3_GET_TAG_HANDLE - Gets response for only one time slot. Also send a ReqRN to get the tag handle for this slot. Here the number of time slots should be zero in the begin round command.
 *
 *  The response is a Pointer to a buffer containing the result of the inventory.
 *  The buffer may contain the result of one or more timeslots. Each timeslot result has the following structure:
 *  1 Byte time slot status
 *      (
 *       0 ... Tag response available. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field' are present.
 *       1 ... Tag response available. In addition to three field above, 'Tag handle field' also present.
 *       2 ... No tag replied in timeslot. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field', 'Tag Handle' are not present.
 *       3 ... Two or more tags responded in the timeslot. (Collision). Tag Reply Length, valid bits in last byte and tag reply field not present.
 *      )
 *  1 Byte 'Tag reply' Length (1-66)
 *  1 Byte Valid bits in last Byte (0-7, 0 means all bits are valid)
 *  0-66 Bytes Tag reply
 *  0 or 2 Bytes Handle of the tag, in case the field
 *
 * A typical sequence in which this API is called is given below\n
 * \li BAL initialization (where applicable) - Example - phbalReg_Serial_Init()
 * \li HAL initialization - Example phhalHw_Rc663_Init();
 * \li Load ISO 1800p3m3 protocol - phhalHw_ApplyProtocolSettings(pHal, PHHAL_HW_CARDTYPE_I18000P3M3);
 * \li Turn ON the field - phhalHw_FieldOn();
 * \li Wait for initial guard time - phalHw_Wait(pHal, 5100, PHHAL_HW_TIME_MICROSECONDS);
 * \li Start inventory - phhalHw_I18000p3m3Inventory(pHal,pSelectFrame, bSelectFrameLen, bNumValidBits, pBeginRndFrame, bTSProcessing, &pResponse, &wRxLen);
 * \li Use phpalI18000p3m3 and phalI18000p3m3 APIs to further communicate with the tag.
 * \li Turn OFF the field - phhalHw_FieldOff();
 * \li Termination of the application flow.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_IO_TIMEOUT When there is no tag in the field.
 * \retval #PH_ERR_INVALID_PARAMETER If wrong value specified for bNumValidBits or pBeginRound.
 */
phStatus_t phhalHw_I18000p3m3Inventory(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pSelCmd,                                                                      /**< [In] ISO 18000p3m3 Select command frame. */
        uint8_t bSelCmdLen,                                                                     /**< [In] Select command length in bytes- 1 to 39 bytes. */
        uint8_t bNumValidBitsinLastByte,                                                        /**< [In] Number of valid bits in last byte of pSelCmd. */
        uint8_t * pBeginRndCmd,                                                                 /**< [In] ISO 18000p3m3 BeginRound command frame. This is 17bits i.e., 3 bytes are expected. CRC5 should not be provided. */
        uint8_t bTSprocessing,                                                                  /**< [In] TimeSlot processing behavior. */
        uint8_t ** ppRxBuffer,                                                                  /**< [Out] Pointer to HAL RX Buffer containing response from single/multiple tags. */
        uint16_t * wRxBufferLen                                                                 /**< [Out] Length of response in HAL RX Buffer. */
    );


/**
 * \brief ISO 18000p3m3 resume inventory command.
 *
 * Note: This function is to be called after phhalHw_I18000p3m3Inventory.
 * This internally calls the ISO 18000p3m3 NextSlot command once or multiple times to get responses from
 * remaining slots.
 *
 * Resume inventory also returns the response similar to phhalHw_I18000p3m3Inventory.
 *
 * A typical sequence in which this API is called is given below\n
 * \li BAL initialization (where applicable) - Example - phbalReg_Serial_Init()
 * \li HAL initialization - Example phhalHw_Rc663_Init();
 * \li Load ISO 1800p3m3 protocol - phhalHw_ApplyProtocolSettings(pHal, PHHAL_HW_CARDTYPE_I18000P3M3);
 * \li Turn ON the field - phhalHw_FieldOn();
 * \li Wait for initial guard time - phalHw_Wait(pHal, 5100, PHHAL_HW_TIME_MICROSECONDS);
 * \li Start inventory - phhalHw_I18000p3m3Inventory(pHal,pSelectFrame, bSelectFrameLen, bNumValidBits, pBeginRndFrame, bTSProcessing, &pResponse, &wRxLen);
 * \li Store the responses received for the tags.
 * \li If inventory has to be continued then issue the ResumeInventory command - phhalHw_I18000p3m3ResumeInventory(pHal, &pResponse, &wRxLen);
 * \li Use phpalI18000p3m3 and phalI18000p3m3 APIs to further communicate with the tag.
 * \li Turn OFF the field - phhalHw_FieldOff();
 * \li Termination of the application flow.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_IO_TIMEOUT When there is no tag in the field.
 * \retval #PH_ERR_USE_CONDITION If called when the phhalHw_I18000p3m3 was not called previously with bTSprocessing = PHHAL_HW_I18000P3M3_GET_MAX_RESPS.
 */
phStatus_t phhalHw_I18000p3m3ResumeInventory(
        void * pDataParams,                                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t ** ppRxBuffer,                                                                  /**< [Out] Pointer to HAL Rx Buffer containing response from single/multiple tags. */
        uint16_t * wRxBufferLen                                                                 /**< [Out] Length of response in hal Rx Buffer. */
    );

/**
 * \brief Reads the EEPROM information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_ReadEEPROM(
        void * pDataParams,                                             /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [IN] Options to be used for EEPROM Reading.
                                                                         *          - 0x0000: Only \b dwStartAddr and \b dwLength information
                                                                         *            will be exchagned to Reader.
                                                                         *          - \b dwStartAddr and \b dwLength along with the specified information
                                                                         *             to this paramter will be exchagned to Reader. These values are
                                                                         *             specific to Pegoda - 3 Reader.
                                                                         *              - 0x0010: Read information from User area
                                                                         *              - 0x0011: Read information from Secure Lib area
                                                                         *              - 0x0012: Read information from IC Configuration area
                                                                         *              - 0x0020: Read information from RF Configuration area
                                                                         *          - Values other that above specified ones.
                                                                         */
        uint32_t dwAddr_ProtIndex,                                      /**< [IN] The EEPROM address or Protocol Index from where the information
                                                                         *        should be read. One of the following,
                                                                         *          - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
                                                                         *            EEPROM address. This is supported by Pegoda - 3 Reader
                                                                         *            only.
                                                                         *          - If \b wOption = 0x0020, then 1 byte Protocol
                                                                         *            index of the RF Configuration. This is supported
                                                                         *            by Pegoda - 3 Reader only.
                                                                         *          - If \b wOption = 0x0000, then 4 byte EEPROM address.
                                                                         *            Reader other than Pegoda 3 reader.
                                                                         *
                                                                         */
        uint32_t dwNoOfBytesToRead,                                     /**< [IN] Number of bytes to be read from EEPROM starting from address specified.
                                                                         *        If Reader is Pegoda -3 and \b wOption = 0x0020, then zero should be
                                                                         *        provided.
                                                                         */
        uint8_t ** ppResponse,                                          /**< [OUT] EEPROM information from the mentioned start address (\b dwStartAddr)
                                                                         *         until the requested length. Make sure the buffer is allocated for the
                                                                         *         requested number of bytes (\b dwLength).
                                                                         */
        uint16_t * pRespLen                                             /**< [OUT] Length of bytes available in \b pResponse buffer */
    );

/**
 * \brief Writes the EEPROM information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_WriteEEPROM(
        void * pDataParams,                                             /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [IN] Options to be used for EEPROM Writing.
                                                                         *          - 0x0000: Only \b dwStartAddr and \b dwLength information
                                                                         *            will be exchagned to Reader.
                                                                         *          - \b dwStartAddr and \b dwLength along with the specified information
                                                                         *             will be exchagned to Reader. These values are specific to Pegoda - 3
                                                                         *             Reader.
                                                                         *              - 0x0010: Read information from User area
                                                                         *              - 0x0011: Read information from Secure Lib area
                                                                         *              - 0x0012: Read information from IC Configuration area
                                                                         *              - 0x0020: Read information from RF Configuration area
                                                                         */
        uint32_t dwStartAddr,                                           /**< [IN] The EEPROM address information to be written.
                                                                         *        One of the following,
                                                                         *          - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
                                                                         *            EEPROM address. This is supported by Pegoda - 3 Reader
                                                                         *            only.
                                                                         *          - If \b wOption = 0x0020, zero should be provided.
                                                                         *          - If \b wOption = 0x0000, then 4 byte EEPROM address.
                                                                         *            Reader other than Pegoda 3 reader.
                                                                         *
                                                                         */
        uint8_t * pData,                                                /**< [IN] Information to be written to EERPOM.
                                                                         *        One of the following,
                                                                         *          - If \b wOption = 0x0000, 0x0010, 0x0011 or 0x0012, then
                                                                         *            EEPROM value should be provided.
                                                                         *          - If \b wOption = 0x0020, then 1 byte Protocol index followed
                                                                         *            1 byte Register address followed by 4 byte Register Value.
                                                                         *            This is supported by Pegoda - 3 Reader only. \n
                                                                         *            Protocol Index1 ( 1 Byte) ||
                                                                         *            Reg Addr1 ( 1 Byte) || Reg Value1 ( 4 Bytes) ||
                                                                         *            Reg Addr2 ( 1 Byte) || Reg Value2 ( 4 Bytes) ||
                                                                         *            ...
                                                                         *            Reg AddrN ( 1 Byte) || Reg ValueN ( 4 Bytes)
                                                                         *
                                                                         */
        uint16_t wDataLen                                               /**< [IN] Length of bytes available in \b pData buffer. */
    );
/**
 * end of group phhalHw
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_H */
