/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Contact HAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHWCONTACT_H
#define PHHALHWCONTACT_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HWCONTACT
    /** \defgroup phhalHwContact Hardware Abstraction Layer for contact based reader
    * \brief These are the Components which are used to abstract the
    * functionality of the physical contact based reader device to a generic interface.
    * @{
    */

    /**
    * \name Communication Configs
    */
    /*@{*/
    /* Protocol independent Configs */
    #define PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE             0x0001U /**< Disabled or CRC or LRC. */
    #define PHHAL_HW_CONTACT_CONFIG_CONVENTION                  0x0002U /**< Set the convention. When inverse convention is set, the conveyed byte is equal to '3F'. When direct convention is set, the conveyed byte is equal to '3B'. */
    #define PHHAL_HW_CONTACT_CONFIG_FD                          0x0003U /**< Apply the F and D factor and compute the necessary wait and guard times. Param is a byte, its high order bits represent F, the low order bits D. */
    #define PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE                0x0004U /**< Gets the current selected protocol type. T=0 or T=1 */

    /* T=1 Configs */
    #define PHHAL_HW_CONTACT_CONFIG_BGT_CLK                     0x0005U /**< BGT is the minimum delay between the leading edges of two consecutive characters transmitted in opposite directions. */
    #define PHHAL_HW_CONTACT_CONFIG_BWT_CLK                     0x0006U /**< BWT is the maximum delay between the leading edge of the last character of the block received by the card and the leading edge of the first character of the next block transmitted by the card. */
    #define PHHAL_HW_CONTACT_CONFIG_CGT_CLK                     0x0007U /**< Character Guard Time for T=1 Protocool (CGT = GT) */
    #define PHHAL_HW_CONTACT_CONFIG_CWT_CLK                     0x0008U /**< CWT is the maximum delay between the leading edges of two consecutive characters in the block */
    #define PHHAL_HW_CONTACT_CONFIG_IFSC                        0x0009U /**< IFSC is the maximum size of information field of blocks that can be received by the card. */

    /* T=0 Configs */
    #define PHHAL_HW_CONTACT_CONFIG_WT_CLK                      0x000AU /**< The Waiting Time is the delay between the leading edge of any character transmitted by the card and the leading edge of the previous character, transmitted either by the card or by the interface device. */
    #define PHHAL_HW_CONTACT_CONFIG_GT_CLK                      0x000BU /**< Specify the value of the guard time in etu, and an additional value express in clock cycles */
    #define PHHAL_HW_CONTACT_CONFIG_EGT_CLK                     0x000CU /**< The Extra Guard Time indicates that, before being ready to receive the next character, the card requires the following delay from the leading edge of the previous character transmitted either by the card or by the interface device. */
    #define PHHAL_HW_CONTACT_CONFIG_RGT_CLK                     0x000DU /**< Any change of RGT implies the change of RGT1 and RGT2. */
    #define PHHAL_HW_CONTACT_CONFIG_RGT_1_CLK                   0x000EU /**< RGT1 is defined as the interval between the leading edge of the procedure byte sent by the card and the leading edge of the first data byte sent by the terminal. */
    #define PHHAL_HW_CONTACT_CONFIG_RGT_2_CLK                   0x000FU /**< RGT2 is defined as the interval between the leading edge of the last status byte sent by the card and the leading edge of the first byte of the next command sent by the terminal. */

    /* Other */
    #define PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR                 0x0010U /**< Defines the maximum number of clock cycles after the rising edge of reset before the card is seen as mute. */
    #define PHHAL_HW_CONTACT_CONFIG_PROTOCOL_TIMEOUT            0x0011U /**< Manages protocols timeouts. If Param is set to ON (default) all protocol timeouts applky (CTW, BWT ...). If Param is set to OFF, timeouts are not taken in account. Use with care, as a function may never return. */
    #define PHHAL_HW_CONTACT_CONFIG_ETU_WIDTH_CLK               0x0012U /**< Set etu width in clock cycles */
    #define PHHAL_HW_CONTACT_CONFIG_ETU_WIDTH_ATR_CLK           0x0013U /**< Set etu width in clock cycles when getting ATR */
    #define PHHAL_HW_CONTACT_CONFIG_INITIAL_WAITING_TIME_CLK    0x0014U /**< The Initial Waiting Time is the delay between the leading edges of two consecutive characters transmitted by the card during the answer to reset. */
    #define PHHAL_HW_CONTACT_CONFIG_PARITY                      0x0015U /**< Set the Iso character parity and enable/disable parity error detection. */
    #define PHHAL_HW_CONTACT_CONFIG_PARITY_CHECK                0x0016U /**< If enabled an error will occur when data is received with parity error. */
    #define PHHAL_HW_CONTACT_CONFIG_PARITY_ERROR_SIGNAL         0x0017U /**< Used to disable or enable the error signal handling. */
    #define PHHAL_HW_CONTACT_CONFIG_POWER_OFF_ON_ATR_ERROR      0x0018U /**< Manages power on ATR error. the deactivation sequence is called if the activation sequence returns an error (default) */
    #define PHHAL_HW_CONTACT_CONFIG_RESET_LOW_TIME_CLK          0x0019U /**< Defines the minimum number of clock cycles during the reset pin is forced low. The default value is 1 488 (4 etu). */
    #define PHHAL_HW_CONTACT_CONFIG_TIMEOUT_NULL_BYTE           0x001AU /**< In T=0 (respectively in T=1), a card may send a NULL byte (0x60) (respectively a S(WTX) block) without limit, then hanging the terminal. This value is used to set a time out, expressed in ms, while waiting the card response. */
    #define PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES          0x001BU /**< Number of expected bytes for Exchanges. */
    #define PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL       0x001CU /**< Configure Communication Channel. */
    #define PHHAL_HW_CONTACT_CONFIG_POLLING_INTERVAL_MS         0x001DU /**< Config the polling interval for I2C/SPI communication. */

    /* I/O Config */
    #define PHHAL_HW_CONTACT_CONFIG_VCC                         0x0020U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ          0x0021U /**< */
    /* I2C Config */
    #define PHHAL_HW_CONTACT_CONFIG_I2C_SLAVE_ADDRESS           0x002AU /**< Config Slave address for I2C communication. */
    #define PHHAL_HW_CONTACT_CONFIG_I2C_BITRATE_KBITSEC         0x002BU /**< Config the datarate for I2C communication. */
    #define PHHAL_HW_CONTACT_CONFIG_I2C_CLK_STRETCH             0x002CU /**< Config to enable and disable I2C clock stretching. */
    #define PHHAL_HW_CONTACT_CONFIG_I2C_HS_MODE                 0x002DU /**< Config to enabke and disable I2C Highspeed Mode. */
    /* SPI Config */
    #define PHHAL_HW_CONTACT_CONFIG_SPI_BITRATE_BITSEC          0x002EU /**< Config the datarate for SPI communication. */
    #define PHHAL_HW_CONTACT_CONFIG_SPI_RECV_NAD_VALUE          0x002FU /**< Config the expected NAD value for SPI Polling response detection. */
    #define PHHAL_HW_CONTACT_CONFIG_SPI_GP_MODE                 0x0024U /**< Config the global platform mode for SPI raw exchange length detection. */
    /* C1 .. Vcc */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_VCC_FALL_TIME           0x0030U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_VCC_RISE_TIME           0x0031U /**< */
    /* C2 .. Reset */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_RESET_FALL_TIME         0x0040U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_RESET_RISE_TIME         0x0041U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_RESET_OUTPUT_LOW        0x0042U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_RESET_OUTPUT_HIGH       0x0043U /**< */
    /* C3 .. Clock */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_FALL_TIME         0x0050U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_RISE_TIME         0x0051U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_OUTPUT_LOW        0x0052U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_OUTPUT_HIGH       0x0053U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_CLOCK_DUTY_CYCLE        0x0054U /**< */
    /* C7 .. I/O */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_IO_FALL_TIME            0x0060U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_IO_RISE_TIME            0x0061U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_IO_OUTPUT_LOW           0x0062U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_IO_OUTPUT_HIGH          0x0063U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_LOW            0x0065U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_IO_INPUT_HIGH           0x0066U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_IO_PULL_UP              0x0067U /**< */
    /* Measurement */
    #define PHHAL_HW_CONTACT_CONFIG_TIMING_MODE                 0x0070U /**< Set the timing mode. */
    #define PHHAL_HW_CONTACT_CONFIG_TIMING_US                   0x0071U /**< Retrieve elapsed time of timer ([us]). */
    #define PHHAL_HW_CONTACT_CONFIG_TIMING_MS                   0x0072U /**< Retrieve elapsed time of timer ([ms]). */
    /* Buffer */
    #define PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS           0x0080U /**< Start position of RX buffer to be used (never changed by HAL). */
    #define PHHAL_HW_CONTACT_CONFIG_RXBUFFER_BUFSIZE            0x0081U /**< Buffer size of RX buffer set at Init. */
    #define PHHAL_HW_CONTACT_CONFIG_TXBUFFER_BUFSIZE            0x0082U /**< Buffer size of TX buffer set at Init. */
    #define PHHAL_HW_CONTACT_CONFIG_TXBUFFER_LENGTH             0x0083U /**< Amount of valid bytes in TX buffer. */
    #define PHHAL_HW_CONTACT_CONFIG_TXBUFFER                    0x0084U /**< Access the TxBuffer at the address defined by #PHHAL_HW_CONFIG_ADDITIONAL_INFO. */
    #define PHHAL_HW_CONTACT_CONFIG_MAX_PRECACHED_BYTES         0x0085U /**< Configures the max. number of bytes which are precached prior to command execution. */
    /* Other */
    #define PHHAL_HW_CONTACT_CONFIG_ADDITIONAL_INFO             0x0090U /**< Set / Get additional information. */
    #define PHHAL_HW_CONTACT_CONFIG_RESET_ON_TIMEOUT            0x0091U /**< Perform an Rf-Reset in case of a timeout (only for \ref phhalHw_Exchange). */
    #define PHHAL_HW_CONTACT_CONFIG_I2C_RX_POLLING_TIMEOUT_MS   0x0092U /**< Timeout value in ms used by the I2C exchange function */


    /* Pin Configs */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_FALL_TIME               0x0000U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_RISE_TIME               0x0001U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_OUTPUT_LOW              0x0002U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_OUTPUT_HIGH             0x0003U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_DUTY_CYCLE              0x0004U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_INPUT_LOW               0x0005U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_INPUT_HIGH              0x0006U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_PULL_UP                 0x0007U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_NUMBER_OFFSET           0x0003U /**< */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_MASK                    0x00F0U /**< Mask to get the Pin Number from PinConfigs */
    #define PHHAL_HW_CONTACT_CONFIG_PIN_CONFIG_MASK             0x000FU /**< Mask to get the Config Param from PinConfigs */

    /* SPI GP Configs (Only supported with SCR reader) */
    #define PHHAL_HW_CONTACT_CONFIG_SPI_GP_MODE_T1SPI           0x0000U /**< T1SPI : Also known as "T=1 over SPI Interface", this is a quasi standard from NXP Semiconductors */
    #define PHHAL_HW_CONTACT_CONFIG_SPI_GP_MODE_GP_MODE_1       0x0001U /**< GP mode 1 : Global Platform specification "APDU Transport over SPI/I2C" up to version 0.0.0.45 */
    #define PHHAL_HW_CONTACT_CONFIG_SPI_GP_MODE_GP_MODE_2       0x0002U /**< GP mode 2 : Global Platform specification "APDU Transport over SPI/I2C", version 0.0.0.46 and newer */

    /* Maximum ATR Size */
    #define PHHAL_HW_CONTACT_MAX_ATR_SIZE                       0x0021U /**< Maximum size of the ATR according to ISO standards */
    #define PHHAL_HW_CONTACT_MAX_T1_BLOCK_SIZE                  0x0103U /**< Maximum size of one Block in T1 Protocol (1 NAD 1 PCB 1 LEN 254 INF 2 CRC) */
    #define PHHAL_HW_CONTACT_MAX_CRC_BUFFER_LEN                 0x0002U /**< Size of CRC that is used for the buffer. */
    /*@}*/

    /**
    * \name Pin Numbers
    */
    /*@{*/
    #define PHHAL_HW_PIN_C1_VCC                                 0x0000U /**< C1 used for VCC */
    #define PHHAL_HW_PIN_C2_RESET                               0x0001U /**< C2 used for Reset */
    #define PHHAL_HW_PIN_C3_CLOCK                               0x0002U /**< C3 used for Clock */
    #define PHHAL_HW_PIN_C7_IO                                  0x0006U /**< C7 used for IO */
    /*@}*/

    /**
    * \name Timing Mode settings
    */
    /*@{*/
    #define PHHAL_HW_TIMING_MODE_OFF                            0x0000U /**< Time Measurement disabled. */
    #define PHHAL_HW_TIMING_MODE_FDT                            0x0001U /**< Measures time from Tx-End until Rx-Begin. */
    #define PHHAL_HW_TIMING_MODE_COMM                           0x0002U /**< Measures the duration of the complete communication. */
    #define PHHAL_HW_TIMING_MODE_OPTION_MASK                    0xFF00U /**< Additional option-mask for time measurement. */
    #define PHHAL_HW_TIMING_MODE_OPTION_DEFAULT                 0x0000U /**< The timing value is cummulated on each Exchange until it gets retrieved by \ref phhalHw_GetConfig. */
    #define PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR               0x0100U /**< Automatically resets the current timing value prior to any communication. */
    /*@}*/


    /**
    * \name Protocol Types
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_PROTOCOLTYPE_CURRENT               0x0000U /**< Re-Apply settings for current Communication Mode. */
    #define PHHAL_HW_CONTACT_PROTOCOLTYPE_T0                    0x0001U /**< T=0 communication protocol */
    #define PHHAL_HW_CONTACT_PROTOCOLTYPE_T1                    0x0002U /**< T=1 communication protocol. */
    /*@}*/

    /**
    * \name Datarate Di Values
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_DI_1                               0x01U /**< Index for D=1. */
    #define PHHAL_HW_CONTACT_DI_2                               0x02U /**< Index for D=2. */
    #define PHHAL_HW_CONTACT_DI_4                               0x03U /**< Index for D=4. */
    #define PHHAL_HW_CONTACT_DI_8                               0x04U /**< Index for D=8. */
    #define PHHAL_HW_CONTACT_DI_16                              0x05U /**< Index for D=16. */
    #define PHHAL_HW_CONTACT_DI_32                              0x06U /**< Index for D=32. */
    #define PHHAL_HW_CONTACT_DI_64                              0x07U /**< Index for D=64. */
    #define PHHAL_HW_CONTACT_DI_12                              0x08U /**< Index for D=12. */
    #define PHHAL_HW_CONTACT_DI_20                              0x09U /**< Index for D=20. */
    /*@}*/

    /**
    * \name Datarate Fi Values
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_FI_372_4_MHZ                       0x00U /**< Index for F=372 with f(max) = 4MHz. */
    #define PHHAL_HW_CONTACT_FI_372_5_MHZ                       0x10U /**< Index for F=372 with f(max) = 5MHz. */
    /*@}*/

    #define PHHAL_HW_CONTACT_FD_DEFAULT                         0x11U       /**< Default Values for FD Byte */
    #define PHHAL_HW_CONTACT_CLK_DEFAULT                        5000000U    /**< Default Values for Clock */
    #define PHHAL_HW_CONTACT_MAX_CLK_ATR_DEFAULT                40000U      /**< Default Value for Max Clk Atr Setting */
    #define PHHAL_HW_CONTACT_POLLING_INTERVAL_MS_DEFAULT        1U          /**< Default polling interval for I2C/SPI */

    /**
    * \name Voltage Classes
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_CLASS_A                            5000U /**< Voltage for Class A. */
    #define PHHAL_HW_CONTACT_CLASS_B                            3000U /**< Voltage for Class B. */
    #define PHHAL_HW_CONTACT_CLASS_C                            1800U /**< Voltage for Class C. */
    #define PHHAL_HW_CONTACT_CLASS_A_MIN                        4500U /**< Minimum ISO Voltage for Class A. */
    #define PHHAL_HW_CONTACT_CLASS_B_MIN                        2700U /**< Minimum ISO Voltage for Class B. */
    #define PHHAL_HW_CONTACT_CLASS_C_MIN                        1620U /**< Minimum ISO Voltage for Class C. */
    #define PHHAL_HW_CONTACT_CLASS_A_MAX                        5500U /**< Maximum ISO Voltage for Class A. */
    #define PHHAL_HW_CONTACT_CLASS_B_MAX                        3300U /**< Maximum ISO Voltage for Class B. */
    #define PHHAL_HW_CONTACT_CLASS_C_MAX                        1980U /**< Maximum ISO Voltage for Class C. */
    /*@}*/

    /**
    * \name Clock Stop Modes
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_CLOCK_STOP_HIGH                    12U         /**< Clock stop at high. */
    #define PHHAL_HW_CONTACT_CLOCK_STOP_LOW                     13U         /**< Clock stop at low. */
    #define PHHAL_HW_CONTACT_CLOCK_STOP_RESUME                  14U         /**< Clock resume. */
    #define PHHAL_HW_CONTACT_CLOCK_STOP_APPLIED                 11U         /**< Clock applied. */
    #define PHHAL_HW_CONTACT_CLOCK_STOP_AUTO_OFF                0x80000000U /**< If ORED with Mode, the clock will be automatically switched off after each APDU. */
    /*@}*/

    /**
    * \name Time Units
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_TIME_MICROSECONDS                  0x0000U /**< Time in microseconds (us). */
    #define PHHAL_HW_CONTACT_TIME_MILLISECONDS                  0x0001U /**< Time in milliseconds (ms). */
    #define PHHAL_HW_CONTACT_TIME_ETU                           0x0002U /**< Time in ETU. */
    #define PHHAL_HW_CONTACT_TIME_CLK                           0x0003U /**< Time in clock cycles. */
    /*@}*/

    /**
    * \name Redundancy Code Modes
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_MODE_DISABLED                      0x0000U /**< Both CRC and LRC disabled */
    #define PHHAL_HW_CONTACT_MODE_TX_LRC                        0x0001U /**< LRC for TX */
    #define PHHAL_HW_CONTACT_MODE_RX_LRC                        0x0002U /**< LRC for RX */
    #define PHHAL_HW_CONTACT_MODE_LRC                           0x0003U /**< LRC for TX and RX */
    #define PHHAL_HW_CONTACT_MODE_TX_CRC                        0x0004U /**< CRC for TX */
    #define PHHAL_HW_CONTACT_MODE_RX_CRC                        0x0008U /**< CRC for RX */
    #define PHHAL_HW_CONTACT_MODE_CRC                           0x000CU /**< CRC for TX and RX */
    #define PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED                0x0014U /**< CRC swapped for TX */
    #define PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED                0x0028U /**< CRC swapped for RX */
    #define PHHAL_HW_CONTACT_MODE_CRC_SWAPPED                   0x003CU /**< CRC swapped for TX and RX */
    #define PHHAL_HW_CONTACT_MODE_MASK_TX                       0x0005U /**< Mask for Tx Settings. */
    #define PHHAL_HW_CONTACT_MODE_MASK_RX                       0x000AU /**< Mask for Rx Settings. */
    #define PHHAL_HW_CONTACT_MODE_MASK_TX_SWAPPED               0x0010U /**< Mask for Tx Swapped Settings. */
    #define PHHAL_HW_CONTACT_MODE_MASK_RX_SWAPPED               0x0020U /**< Mask for Rx Swapped Settings. */
    #define PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER                0x8000U /**< Flag that indicate if the Rx check should be performed on the whole rx buffer. If not set it starts from RxStartPos. */
    /*@}*/

    /**
    * \name Timing Mode settings
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_TIMING_MODE_OFF                    0x0000U /**< Time Measurement disabled. */
    #define PHHAL_HW_CONTACT_TIMING_MODE_FDT                    0x0001U /**< Measures time from Tx-End until Rx-Begin. */
    #define PHHAL_HW_CONTACT_TIMING_MODE_COMM                   0x0002U /**< Measures the duration of the complete communication. */
    #define PHHAL_HW_CONTACT_TIMING_MODE_OPTION_MASK            0xFF00U /**< Additional option-mask for time measurement. */
    #define PHHAL_HW_CONTACT_TIMING_MODE_OPTION_DEFAULT         0x0000U /**< The timing value is cummulated on each Exchange until it gets retrieved by \ref phhalHw_GetConfig. */
    #define PHHAL_HW_CONTACT_TIMING_MODE_OPTION_AUTOCLEAR       0x0100U /**< Automatically resets the current timing value prior to any communication. */
    /*@}*/

    /**
    * \name Convention Types
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_DIRECT_CONVENTION                  0x0001U /**< Both CRC and LRC disabled */
    #define PHHAL_HW_CONTACT_INVERSE_CONVENTION                 0x0002U /**< LRC Mode (1 Bytes). */
    /*@}*/

    /**
    * \name Communication Channels
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_UNDEFINED  0xFF /**< Channel is not selected */
    #define PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD   0U   /**< Communication Channel */
    #define PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C        1U   /**< Communication Channel for I2C */
    #define PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI        2U   /**< Communication Channel for SPI using read and write */
    #define PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI_EXCHANGE 3U   /**< Communication Channel for SPI using the polling of the reader */
    /*@}*/

    /**
    * \brief Perform Data Exchange with IFC.
    *
    * \b wOption can be one of:\n
    * \li #PH_EXCHANGE_DEFAULT
    *
    * \b wOption can be combined with:\n
    * \li #PH_EXCHANGE_BUFFERED_BIT
    * \li #PH_EXCHANGE_LEAVE_BUFFER_BIT
    *
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_INVALID_PARAMETER \b wOption is invalid.
    * \retval #PH_ERR_IO_TIMEOUT No response detected within the configured time frame.
    * \retval #PH_ERR_INTEGRITY_ERROR Response received but CRC or Parity is invalid.
    * \retval #PH_ERR_BUFFER_OVERFLOW Internal receive buffer is too small for transmission or smaller than the response.
    * \retval #PH_ERR_FRAMING_ERROR Frame format is either invalid for configured protocol or corrupted.
    * \retval #PH_ERR_PROTOCOL_ERROR Frame format is definitely invalid for configured protocol.
    * \retval #PH_ERR_READ_WRITE_ERROR Hardware problem.
    * \retval #PH_ERR_TEMPERATURE_ERROR Hardware problem.
    * \retval #PH_ERR_INTERFACE_ERROR Hardware problem.
    */
    phStatus_t phhalHwContact_Exchange(
        void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,       /**< [In] Option parameter. */
        uint8_t * pTxBuffer,    /**< [In] Data to transmit. */
        uint16_t wTxLength,     /**< [In] Number of bytes to transmit. */
        uint8_t ** ppRxBuffer,  /**< [Out] Pointer to received data. */
        uint16_t * pRxLength    /**< [Out] Number of received data bytes. */
        );

    /**
    * \brief Apply protocol settings.
    *
    * To keep the current protocol type, use #PHHAL_HW_CONTACT_PROTOCOLTYPE_CURRENT for bProtocolType.\n
    *
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_INVALID_PARAMETER \b bProtocolType is invalid or not supported.
    * \retval #PH_ERR_INTERFACE_ERROR Hardware problem.
    */
    phStatus_t phhalHwContact_ApplyProtocolSettings(
        void * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pAtr,       /**< [In] Atr Buffer with Settings to apply (if not Atr given default is applied). */
        uint16_t dwAtrLength, /**< [In] Length of the given Atr Buffer. */
        uint8_t bProtocolType /**< [In] Type of protocol for which the HAL should be configured for. */
        );

    /**
    * \brief Set configuration parameter.
    *
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
    * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
    * \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
    * \retval #PH_ERR_INTERFACE_ERROR Communication error.
    */
    phStatus_t phhalHwContact_SetConfig32(
        void * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,      /**< [In] Configuration Identifier. */
        uint32_t dwValue       /**< [In] Configuration Value. */
        );

    /**
    * \brief Get configuration parameter.
    *
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
    * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
    * \retval #PH_ERR_PARAMETER_OVERFLOW The parameter value is larger than the range of \b pValue.
    * \retval #PH_ERR_INTERFACE_ERROR Communication error.
    */
    phStatus_t phhalHwContact_GetConfig32(
        void * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,      /**< [In] Configuration Identifier. */
        uint32_t * pdwValue    /**< [Out] Configuration Value. */
        );

    /**
    * \brief Activates the contact based card.
    *
    * Performs a ISO 7816 activation of the card and returns the ATR.
    * If the card is already activated a warm reset and activation is perfomed
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_IO_TIMEOUT No card available.
    * \retval #PH_ERR_INTERFACE_ERROR Communication error.
    */
    phStatus_t phhalHwContact_Activate(
        void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pATRBuffer,   /**< [Out] Data to the received ATR. */
        uint16_t * pwATRLength  /**< [Out] Length of the reiceived ATR. */
        );

    /**
    * \brief Deactivaed a contact based card.
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_INTERFACE_ERROR Communication error.
    */
    phStatus_t phhalHwContact_Deactivate(
        void * pDataParams  /**< [In] Pointer to this layer's parameter structure. */
        );

    /**
    * \brief Performs a clock stop.
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_INTERFACE_ERROR Communication error.
    */
    phStatus_t phhalHwContact_ClockStop(
        void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wTgClockCount, /**< [In] Number of clock cycles while the clock must be kept running after the I/O line remains at Z state before stopping the clock (default is 1,860). Maximum value is 65,535 clock cycles. */
        uint16_t wThClockCount, /**< [In] Number of clock cycles before the exchange on the I/O line after a clock resume. */
        uint32_t dwPinState     /**< [In] The desired state of the clock. If CLOCK_STOP_HIGH or CLOCK_CTOP_LOW is ored with CLOCK_AUTO_OFF (0x80000000), the clock will be automatically switched off after each APDU. */
        );

    /**
    * \brief Blocks until the time given by \b wTimeout elapsed.
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_INVALID_PARAMETER \b bUnit is invalid.
    * \retval #PH_ERR_INTERFACE_ERROR Communication error.
    */
    phStatus_t phhalHwContact_Wait(
        void * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bUnit,      /**< [In] Unit of given timeout value (either #PHHAL_HW_CONTACT_TIME_MICROSECONDS or #PHHAL_HW_CONTACT_TIME_MILLISECONDS). */
        uint16_t wTimeout   /**< [In] Timeout value. */
        );

    /**
    * \brief Resets Reader to Default State and opens all needed Resources
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval #PH_ERR_INTERFACE_ERROR Communication error.
    */
    phStatus_t phhalHwContact_InitContactReader(
                                                void * pDataParams     /**< [In] Pointer to this layer's parameter structure. */
                                                );

    /** @} */
#endif /* NXPBUILD__PHHAL_HWCONTACT */

#ifdef NXPBUILD__PHHAL_HW_CONTACT_MICROPROSS
#define PHHAL_HW_CONTACT_MICROPROSS_ID               0x01    /**< ID for Micropross Contact HAL component. */

    /**
    * \name Micropross specific configs
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_COUPLER_NUMBER               (PH_CONFIG_CUSTOM_BEGIN)        /**< Configure of Coupler. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_ADDRESS_WIDTH  (PH_CONFIG_CUSTOM_BEGIN+1)      /**< Config I2C address width. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_RX_TIMEOUT     (PH_CONFIG_CUSTOM_BEGIN+2)      /**< Config I2C rx timeout. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TX_TIMEOUT     (PH_CONFIG_CUSTOM_BEGIN+3)      /**< Config I2C tx timeout. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_DAT        (PH_CONFIG_CUSTOM_BEGIN+4)      /**< Config I2C setup time data. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THD_DAT        (PH_CONFIG_CUSTOM_BEGIN+5)      /**< Config I2C hold time data. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_STA        (PH_CONFIG_CUSTOM_BEGIN+6)      /**< Config I2C setup time start. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THD_STA        (PH_CONFIG_CUSTOM_BEGIN+7)      /**< Config I2C hold time start. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TSU_STO        (PH_CONFIG_CUSTOM_BEGIN+8)      /**< Config I2C setup time stop. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_THIGH_SCL      (PH_CONFIG_CUSTOM_BEGIN+9)      /**< Config I2C clock high time. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_TLOW_SCL       (PH_CONFIG_CUSTOM_BEGIN+10)     /**< Config I2C clock low time. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CONFIG_I2C_PARAMETER_AUTO_NACK      (PH_CONFIG_CUSTOM_BEGIN+11)     /**< Config I2C auto NACK transmit. */

    #define PHHAL_HW_CONTACT_MICROPROSS_SHADOW_COUNT                    0x0068U     /**< Number of shadowed configurations. */
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_TIMEOUT_MS              5U          /**< Default timeout in milliseconds. */
    #define PHHAL_HW_CONTACT_MICROPROSS_SERIAL_NUMBER_SIZE              16U         /**< The size of the serial number. */
    #define PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE                8U          /**< The size of the version strings. */
    #define PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE               7U          /**< The size of the coupler name (longest = SWPSPY). */
    #define PHHAL_HW_CONTACT_MICROPROSS_COUPLER_SERIAL_NUMBER_SIZE      9U          /**< The size of the coupler serial number. */
    #define PHHAL_HW_CONTACT_MICROPROSS_COUPLER_REVISION_SIZE           5U          /**< The size of the coupler revision. */
    #define PHHAL_HW_CONTACT_MICROPROSS_COUPLER_CALC_DATE_SIZE          4U          /**< The size used to store the calibration date + checksum */
    #define PHHAL_HW_CONTACT_MICROPROSS_COUPER_NUMBER_DEFAULT           0xFFU       /**< Default Coupler Number (-1 means search for correct coupler) */

    /**
    * \name Micropross protocol parameter configs
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_IFSC                        1U          /**< Used to force IFSC value */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_SIM_ATR                     3U          /**< Used to force any parameters that can be defined with an ATR */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_INITIAL_ETU_WIDTH           6U          /**< Used to force the initial ETU when getting ATR. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_MAX_CLK_ATR                 10U         /**< Maximum number of clock cycles to wait for the ATR. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_STANDARD_USED               12U         /**< Apply standard timings for different standards. 0 -> ISO 7816. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_ERROR_SIGNAL                14U         /**< Used to disable or enable the error signal handling. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_CWT                         15U         /**< Used to force character waiting time with another value expressed in ETU. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_BWT                         16U         /**< Used to force block waiting time with another value expressed in ETU. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_WWT                         17U         /**< Work Waiting time */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_BGT                         18U         /**< Select a new value for the Block Guard Time in ETU. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_EGT                         19U         /**< Select a new value for the Extra Guard Time in ETU. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_RGT                         20U         /**< Select a new value for the Return Guard Time, expressed in ETU. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_CURRENT_F_D                 22U         /**< Apply the F and D factor. Param is a byte, high order bits -> F, low order bits  -> D. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_POWER_OFF_ON_ATR_ERROR      29U         /**< If value passed is OFF (0), voltages on all pin remain applied after activation sequence (MPS_OnCmm), even if the function return an error (time out, invalid ATR). If value passed is ON (1), the deactivation sequence is called if MPS_OnCmm returns an error. */
    #define PHHAL_HW_CONTACT_MICROPROSS_CPP_ETU_WIDTH                   34U         /**< Get the duration of one ETU in clock cycles. */

    /**
    * \name Date Formats
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_MICROPROSS_DATEFORMAT_EXTERNALCLK          1U /**< Date Format External Clock Mode */
    #define PHHAL_HW_CONTACT_MICROPROSS_DATEFORMAT_INTERNALCLK          2U /**< Date Format Internal Clock Mode */
    /*@}*/

    /**
    * \name Micropross Pin Config Default Values
    */
    /*@{*/

    /* C1 .. Vcc */
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_VCC_FALL_TIME                   10000U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_VCC_RISE_TIME                   10000U
    /* C2 .. Reset */
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_RESET_FALL_TIME                 10U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_RESET_RISE_TIME                 10U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_RESET_OUTPUT_LOW                0U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_RESET_OUTPUT_HIGH               100U
    /* C3 .. Clock */
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_FALL_TIME                 10U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_RISE_TIME                 10U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_OUTPUT_LOW                0U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_OUTPUT_HIGH               100U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_CLOCK_DUTY_CYCLE                50U
    /* C7 .. I/O */
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_FALL_TIME                    10U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_RISE_TIME                    10U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_OUTPUT_LOW                   0U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_OUTPUT_HIGH                  100U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_INPUT_LOW                    11U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_INPUT_HIGH                   25U
    #define PHHAL_HW_CONTACT_MICROPROSS_DEFAULT_PIN_IO_PULL_UP                      20000U

    /**
    * \name CouplerResourceId
    */
    /*@{*/
    enum ContactCouplerResourceId
    {
        PHHAL_HW_CONTACT_MICROPROSS_RESSOURCE_TC1 =     0xCAU,          /**< Id of a MP300 TCL1 */
        PHHAL_HW_CONTACT_MICROPROSS_RESSOURCE_TC2 =     0xCDU,          /**< Id of a MP300 TCL2 */
        PHHAL_HW_CONTACT_MICROPROSS_RESSOURCE_TC3 =     0x83U           /**< Id of a MP500 TCL3 */
    };
    /*@}*/

    /** \brief Callback Delegate for Bal timeout handling */
    typedef phStatus_t(*cbBalTimeoutHandling)(void * pContext, uint8_t bCouplerNumber);

    /**
    * \brief Micropross Contact HAL parameter structure
    */
    typedef struct
    {
        uint16_t wId;                                                                            /**< Layer ID for this component, NEVER MODIFY! */
        void  * pBalDataParams;                                                                  /**< pointer to the lower layers parameter structure */
        uint32_t  dwAtrLength;                                                                   /**< Length of the ATR stored internally in data params. */
        uint8_t pAtrBuffer[PHHAL_HW_CONTACT_MAX_ATR_SIZE];                                       /**< ATR of current card. */
        uint8_t * pTxBuffer;                                                                     /**< Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize;                                                                     /**< Size of the global transmit buffer. */
        uint16_t wTxBufLen;                                                                      /**< Number of valid bytes within the transmit buffer. */
        uint8_t * pRxBuffer;                                                                     /**< Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize;                                                                     /**< Size of the global receive buffer. */
        uint16_t wRxBufLen;                                                                      /**< Number of valid bytes within the receive buffer. */
        uint16_t wRxBufStartPos;                                                                 /**< Starting position within the global receive buffer. */
        uint16_t wTxBufStartPos;                                                                 /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
        uint32_t dwNumExpectedBytes;                                                             /**< Number of expected bytes for Exchanges. */
        uint8_t bProtocolType;                                                                   /**< Type of protocol for which the HAL is configured for. */
        uint32_t dwI2CRxPollingTimeoutMs;                                                        /**< Timeout Value in ms to be used by I2C exchange function. */
        uint32_t wCfgShadow[PHHAL_HW_CONTACT_MICROPROSS_SHADOW_COUNT];                           /**< Configuration shadow; Stores configuration for current protocol. */
        uint16_t wTimingMode;                                                                    /**< Current timing measurement mode. */
        uint8_t bTimeoutUnit;                                                                    /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
        uint64_t qwTiming_ns;                                                                    /**< Current timing value. */
        uint16_t wAdditionalInfo;                                                                /**< Storage for additional error information. */
        uint8_t bRfResetAfterTo;                                                                 /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
        uint8_t * pIntBuffer;                                                                    /**< Internal buffer used as a scratch buffer. Size must be \b wIntBufferSize*wIntBufferNum. */
        uint16_t wIntBufferSize;                                                                 /**< Total length of internal buffer used as a scratch buffer. */
        uint16_t wIntBufferLen;                                                                  /**< Flag to show the status of buffer allocation. */
        uint16_t wIntBufferReadPos;                                                              /**< Read position within \b pIntBuffer for protocol. */
        uint8_t bCouplerNumber;                                                                  /**< Select the Coupler used for communication */
        uint32_t dwGuardTimeEtu;                                                                 /**< Specify the value of the guard time in etu, and an additional value express in clock cycles */
        uint32_t dwGuardTimeClk;                                                                 /**< Use the specify the value of the guard time in etu and addan additional value express in clock cycles */
        uint16_t wCouplerResourceId;                                                             /**< Storage of the resource ID of the Coupler to know the type of Reader (TCL1 or TCL2 supported).*/
        uint8_t bSerialNumber[PHHAL_HW_CONTACT_MICROPROSS_SERIAL_NUMBER_SIZE];                   /**< This value stores the serial number of the reader */
        uint8_t bDeviceFamily[PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE];                     /**< This value stores the DeviceFamily of the reader */
        uint8_t bDeviceReferenceWithinFamily[PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE];      /**< This value stores the DeviceReferenceWithinFamily of the reader */
        uint8_t bMotherboardSystemVersion[PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE];         /**< This value stores the MotherboardSystemVersion of the reader */
        uint8_t bMotherboardBootVersion[PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE];           /**< This value stores the MotherboardBootVersion of the reader */
        uint8_t bCouplerName[PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE];                     /**< This value stores the CouplerName of the reader */
        uint8_t bCouplerSerialNumber[PHHAL_HW_CONTACT_MICROPROSS_COUPLER_SERIAL_NUMBER_SIZE];    /**< This value stores the CouplerSerialNumber of the reader */
        uint8_t bCouplerRevision[PHHAL_HW_CONTACT_MICROPROSS_COUPLER_REVISION_SIZE];             /**< This value stores the CouplerRevision of the reader */
        uint8_t bCouplerDriverVersion[PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE];             /**< This value stores the CouplerDriverVersion of the reader */
        uint16_t wCouplerFPGAVersion;                                                            /**< This value stores the CouplerFPGAVersion of the reader */
        uint8_t bCouplerCalibrationDate[PHHAL_HW_CONTACT_MICROPROSS_COUPLER_CALC_DATE_SIZE];     /**< Date where the till the coupler is calibrated */
        uint32_t dwScenarioID;                                                                   /**< ID of the current in use scenario */
        uint8_t bSpyFilename[256];                                                               /**< Filename of the spied Tracefile */
        void *SpyThreadHandle;                                                                   /**< Handle of the SpyThread */
        void *GEvtFinished;                                                                      /**< Event to check if SpyThread is ready */
        void *GEvtReady;                                                                         /**< Event to check if SpyThread has finished */
        uint8_t bCommunicationChannel;                                                           /**< Current Communication Channel - Standard or I2C. */
        uint16_t wSlaveAddress;                                                                  /**< Address of the Slave for I2C Communication. */
        uint32_t dwTHighScl;                                                                     /**< Duration of high level of I2C Clock. */
        uint32_t dwTLowScl;                                                                      /**< Duration of low level of I2C Clock. */
        uint32_t dwTSuDat;                                                                       /**< Duration of SDA Setup time for data. */
        uint32_t dwTHdDat;                                                                       /**< Duration of SDA Hold time for data. */
        cbBalTimeoutHandling pBalTimeoutHandling;                                                /**< Pointer to the handler function. */
        void * pBalTimeoutHandlingContext;                                                       /**< Context that can be used with handler function. */
    } phhalHwContact_Micropross_DataParams_t;

    /**
    * \brief Initialise the HAL component.
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval Other Depending on implementation and underlaying component.
    */
    phStatus_t phhalHwContact_Micropross_Init(
        phhalHwContact_Micropross_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                    /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize,                                    /**< [In] Size of the global receive buffer. */
        uint8_t * pIntBuffer,                                   /**< [In] Internal buffer used as a scratch buffer. */
        uint16_t  wIntBufferSize                                /**< [In] Length of innternal buffer used as a scratch buffer. */
        );

#endif /* NXPBUILD__PHHAL_HW_CONTACT_MICROPROSS */

#ifdef NXPBUILD__PHHAL_HW_CONTACT_TRANSPARENT
#define PHHAL_HW_CONTACT_TRANSPARENT_ID              0x02    /**< ID for Transparent Contact HAL component. */

    /**
    * \name Transparent specific configs
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_TRANSPARENT_SHADOW_COUNT           0x0068U     /**< Number of shadowed configurations. */
    #define PHHAL_HW_CONTACT_TRANSPARENT_WAIT_FOR_ATR           0x01U
    #define PHHAL_HW_CONTACT_TRANSPARENT_DONT_WAIT_FOR_ATR      0x00U
    #define PHHAL_HW_CONTACT_TRANSPARENT_DEFAULT_CLK            3600000U

    /**
    * \brief Transparent HAL parameter structure
    */
    typedef struct
    {
        uint16_t wId;                                                       /**< Layer ID for this component, NEVER MODIFY! */
        void    * pBalDataParams;                                           /**< pointer to the lower layers parameter structure */
        uint8_t * pTxBuffer;                                                /**< Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize;                                               /**< Size of the global transmit buffer. */
        uint16_t  wTxBufLen;                                                /**< Number of valid bytes within the transmit buffer. */
        uint8_t * pRxBuffer;                                                /**< Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize;                                               /**< Size of the global receive buffer. */
        uint16_t  wRxBufLen;                                                /**< Number of valid bytes within the receive buffer. */
        uint16_t  wRxBufStartPos;                                           /**< Starting position within the global receive buffer. */
        uint32_t  dwNumExpectedBytes;                                       /**< Number of expected bytes for Exchanges. */
        uint32_t  dwATRLength;                                              /**< Length of the ATR stored internally in data params. */
        uint8_t   pATR[PHHAL_HW_CONTACT_MAX_ATR_SIZE];                      /**< ATR of current card. */
        uint8_t   bProtocolType;                                            /**< Type of protocol for which the hal is configured for */
        uint32_t  wCfgShadow[PHHAL_HW_CONTACT_TRANSPARENT_SHADOW_COUNT];    /**< Configuration shadow; Stores configuration for current protocol. */
    }   phhalHwContact_Transparent_DataParams_t;

    /**
    * \brief Initialise the HAL component.
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval Other Depending on implementation and underlaying component.
    */
    phStatus_t phhalHwContact_Transparent_Init(
        phhalHwContact_Transparent_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize,                                   /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize                                    /**< [In] Size of the global receive buffer. */
        );

    /** @} \endcond */

#endif /* NXPBUILD__PHHAL_HW_CONTACT_TRANSPARENT */

#ifdef NXPBUILD__PHHAL_HW_CONTACT_RDSCR01
#define PHHAL_HW_CONTACT_RDSCR01_ID                    0x03    /**< ID for RDSCR01 Contact HAL component. */

    /**
    * \name RdScr01 specific configs
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_RDSCR01_SHADOW_COUNT           0x0068U     /**< Number of shadowed configurations. */

    typedef struct
    {
        uint16_t wId;                                                       /**< Layer ID for this component, NEVER MODIFY! */
        phbalReg_RdScr01_DataParams_t * pBalDataParams;                     /**< pointer to the lower layers parameter structure */
        uint8_t * pTxBuffer;                                                /**< Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize;                                               /**< Size of the global transmit buffer. */
        uint16_t  wTxBufLen;                                                /**< Number of valid bytes within the transmit buffer. */
        uint8_t * pRxBuffer;                                                /**< Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize;                                               /**< Size of the global receive buffer. */
        uint16_t  wRxBufLen;                                                /**< Number of valid bytes within the receive buffer. */
        uint16_t  wRxBufStartPos;                                           /**< Starting position within the global receive buffer. */
        uint32_t  dwNumExpectedBytes;                                       /**< Number of expected bytes for Exchanges. */
        uint32_t  dwAtrLength;                                              /**< Length of the ATR stored internally in data params. */
        uint8_t   pAtr[PHHAL_HW_CONTACT_MAX_ATR_SIZE];                      /**< ATR of current card. */
        uint8_t * pIntBuffer;                                               /**< Buffer used for exchange with max size Data length */
        uint32_t  dwIntBufferLen;                                           /**< Number of valid bytes within the internal temp buffer. */
        uint32_t  dwIntBufferSize;                                          /**< Size of the internal buffer. */
        uint16_t  wTimingMode;                                              /**< Timing mode used to measure FDT */
        uint64_t  qwTiming_ns;                                              /**< Current timing value. */
        uint8_t   bProtocolType;                                            /**< Type of protocol for which the hal is configured for */
        uint32_t  dwI2CRxPollingTimeoutMs;                                  /**< Timeout Value in ms to be used by I2C exchange function. */
        uint32_t  wCfgShadow[PHHAL_HW_CONTACT_RDSCR01_SHADOW_COUNT];        /**< Configuration shadow; Stores configuration for current protocol. */
        uint8_t   bCommunicationChannel;                                    /**< Current Communication Channel - Standard or I2C. */
        uint16_t  wSlaveAddress;                                            /**< Address of the Slave for I2C Communication. */
        uint32_t  dwBitrate;                                                /**< Bitrate used for I2C Communiation. */
        uint8_t   bClkStretch;                                              /**< Flag indicating clock streching state. */
        uint8_t   bHsMode;                                                  /**< Flag indicating the hs mode state. */
    }   phhalHwContact_RdScr01_DataParams_t;

    phStatus_t phhalHwContact_RdScr01_Init(
        phhalHwContact_RdScr01_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                         /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                              /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize,                               /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize,                               /**< [In] Size of the global receive buffer. */
        uint8_t * pIntBuffer,                               /**< [In] Pointer to internal buffer used by the Exchange() function. */
        uint32_t wIntBufferSize                             /**< [In] Size of the internal buffer. */
        );

#endif /* NXPBUILD__PHHAL_HW_CONTACT_RDSCR01 */

#ifdef NXPBUILD__PHHAL_HW_CONTACT_RDSCR2
#define PHHAL_HW_CONTACT_RDSCR2_ID                    0x07    /**< ID for RDSCR2 Contact HAL component. */

    /**
    * \name RdScr2 specific configs
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_RDSCR2_SHADOW_COUNT           0x0068U     /**< Number of shadowed configurations. */

    typedef struct
    {
        uint16_t wId;                                                       /**< Layer ID for this component, NEVER MODIFY! */
        phbalReg_RdScr2_DataParams_t * pBalDataParams;                     /**< pointer to the lower layers parameter structure */
        uint8_t * pTxBuffer;                                                /**< Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize;                                               /**< Size of the global transmit buffer. */
        uint16_t  wTxBufLen;                                                /**< Number of valid bytes within the transmit buffer. */
        uint8_t * pRxBuffer;                                                /**< Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize;                                               /**< Size of the global receive buffer. */
        uint16_t  wRxBufLen;                                                /**< Number of valid bytes within the receive buffer. */
        uint16_t  wRxBufStartPos;                                           /**< Starting position within the global receive buffer. */
        uint32_t  dwNumExpectedBytes;                                       /**< Number of expected bytes for Exchanges. */
        uint32_t  dwAtrLength;                                              /**< Length of the ATR stored internally in data params. */
        uint8_t   pAtr[PHHAL_HW_CONTACT_MAX_ATR_SIZE];                      /**< ATR of current card. */
        uint8_t * pIntBuffer;                                               /**< Buffer used for exchange with max size Data length */
        uint32_t  dwIntBufferLen;                                           /**< Number of valid bytes within the internal temp buffer. */
        uint32_t  dwIntBufferSize;                                          /**< Size of the internal buffer. */
        uint16_t  wTimingMode;                                              /**< Timing mode used to measure FDT */
        uint64_t  qwTiming_ns;                                              /**< Current timing value. */
        uint8_t   bProtocolType;                                            /**< Type of protocol for which the hal is configured for */
        uint32_t  dwI2CRxPollingTimeoutMs;                                  /**< Timeout Value in ms to be used by I2C exchange function. */
        uint32_t  wCfgShadow[PHHAL_HW_CONTACT_RDSCR2_SHADOW_COUNT];         /**< Configuration shadow; Stores configuration for current protocol. */
        uint8_t   bCommunicationChannel;                                    /**< Current Communication Channel - Standard or I2C. */
        uint16_t  wSlaveAddress;                                            /**< Address of the Slave for I2C Communication. */
        uint32_t  dwBitrate;                                                /**< Bitrate used for I2C Communiation. */
        uint8_t   bSpiRecvNad;                                              /**< Expected NAD for SPI polling. */
        uint32_t  dwSpiGpMode;                                              /**< Global Platform Mode used for raw SPI exchange length byte and NAD definition. */
        uint32_t  dwPollingIntervalMs;                                      /**< Polling interval for response detection (SPI/I2C). */
        uint8_t   bClkStretch;                                              /**< Flag indicating clock streching state. */
        uint8_t   bHsMode;                                                  /**< Flag indicating the hs mode state. */
    }   phhalHwContact_RdScr2_DataParams_t;

    phStatus_t phhalHwContact_RdScr2_Init(
        phhalHwContact_RdScr2_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                         /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                              /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize,                               /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize,                               /**< [In] Size of the global receive buffer. */
        uint8_t * pIntBuffer,                               /**< [In] Pointer to internal buffer used by the Exchange() function. */
        uint32_t wIntBufferSize                             /**< [In] Size of the internal buffer. */
        );

#endif /* NXPBUILD__PHHAL_HW_CONTACT_RDSCR2 */

#ifdef NXPBUILD__PHHAL_HW_CONTACT_KEOLABS
#define PHHAL_HW_CONTACT_KEOLABS_ID                  0x04    /**< ID for Keolabs Contact HAL component. */

    /**
    * \name Keolabs specific configs
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_KEOLABS_SHADOW_COUNT   0x0068U     /**< Number of shadowed configurations. */

    typedef struct
    {
        uint16_t wId;                                                       /**< Layer ID for this component, NEVER MODIFY! */
        phbalReg_ContactLAB_DataParams_t  * pBalDataParams;                 /**< pointer to the lower layers parameter structure */
        uint8_t * pTxBuffer;                                                /**< Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize;                                               /**< Size of the global transmit buffer. */
        uint16_t  wTxBufLen;                                                /**< Number of valid bytes within the transmit buffer. */
        uint8_t * pRxBuffer;                                                /**< Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize;                                               /**< Size of the global receive buffer. */
        uint16_t  wRxBufLen;                                                /**< Number of valid bytes within the receive buffer. */
        uint16_t  wRxBufStartPos;                                           /**< Starting position within the global receive buffer. */
        uint32_t  dwNumExpectedBytes;                                       /**< Number of expected bytes for Exchanges. */
        uint32_t  dwAtrLength;                                              /**< Length of the ATR stored internally in data params. */
        uint8_t   pAtr[PHHAL_HW_CONTACT_MAX_ATR_SIZE];                      /**< ATR of current card. */
        uint8_t * pIntBuffer;                                               /**< Buffer used for exchange with max size Data length */
        uint32_t  dwIntBufferLen;                                           /**< Number of valid bytes within the internal temp buffer. */
        uint32_t  dwIntBufferSize;                                          /**< Size of the internal buffer. */
        uint16_t  wTimingMode;                                              /**< Timing mode used to measure FDT */
        uint64_t  qwTiming_ns;                                              /**< Current timing value. */
        uint8_t   bProtocolType;                                            /**< Type of protocol for which the hal is configured for */
        uint32_t  wCfgShadow[PHHAL_HW_CONTACT_KEOLABS_SHADOW_COUNT];        /**< Configuration shadow; Stores configuration for current protocol. */
    }   phhalHwContact_Keolabs_DataParams_t;

    phStatus_t phhalHwContact_Keolabs_Init(
        phhalHwContact_Keolabs_DataParams_t * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize,                                   /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize,                                   /**< [In] Size of the global receive buffer. */
        uint8_t * pIntBuffer,                                   /**< [In] Pointer to internal buffer used by the Exchange() function. */
        uint32_t wIntBufferSize                                 /**< [In] Size of the internal buffer. */
        );

#endif /* NXPBUILD__PHHAL_HW_CONTACT_KEOLABS */

#ifdef NXPBUILD__PHHAL_HW_CONTACT_CALLBACK
#define PHHAL_HW_CONTACT_CALLBACK_ID                 0x05    /**< ID for Callback Contact HAL component. */

    /**
    * \brief Callback Delegate for \ref phhalHwContact_Exchange
    */
    typedef phStatus_t (*cbExchange)(
        uint16_t wOption,      /**< [In] Option parameter. */
        uint8_t * pTxBuffer,   /**< [In] Data to transmit. */
        uint16_t wTxLength,    /**< [In] Length of data to transmit. */
        uint8_t ** ppRxBuffer, /**< [Out] Pointer to received data. */
        uint16_t * pRxLength,  /**< [Out] number of received data bytes. */
        void * pContext        /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback Delegate for \ref phhalHwContact_ApplyProtocolSettings
    */
    typedef phStatus_t (*cbApplyProtocolSettingsAtr)(
        uint8_t * pAtr,        /**< [In] Atr Buffer with Settings to apply (if not Atr given default is applied). */
        uint16_t dwAtrLength,  /**< [In] Length of the given Atr Buffer. */
        uint8_t bProtocolType, /**< [In] Type of protocol for which the HAL should be configured for. */
        void * pContext        /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback Delegate for \ref phhalHwContact_SetConfig32
    */
    typedef phStatus_t (*cbSetConfig32)(
        uint16_t wConfig,      /**< [In] Configuration Identifier. */
        uint32_t dwValue,      /**< [In] Configuration Value. */
        void * pContext        /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback Delegate for \ref phhalHwContact_GetConfig32
    */
    typedef phStatus_t (*cbGetConfig32)(
        uint16_t wConfig,      /**< [In] Configuration Identifier. */
        uint32_t * pdwValue,   /**< [Out] Configuration Value. */
        void * pContext        /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback Delegate for \ref phhalHwContact_Activate
    */
    typedef phStatus_t (*cbActivate)(
        uint8_t * pATRBuffer,       /**< [Out] Data to the received ATR. */
        uint16_t * pwATRLength,     /**< [Out] Length of the reiceived ATR. */
        void * pContext             /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback Delegate for \ref phhalHwContact_Deactivate
    */
    typedef phStatus_t (*cbDeactivate)(
        void * pContext             /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback Delegate for \ref phhalHwContact_ClockStop
    */
    typedef phStatus_t (*cbClockStop)(
        uint16_t wTgClockCount,     /**< [In] Number of clock cycles while the clock must be kept running after the I/O line remains at Z state before stopping the clock (default is 1,860). Maximum value is 65,535 clock cycles. */
        uint16_t wThClockCount,     /**< [In] Number of clock cycles before the exchange on the I/O line after a clock resume. */
        uint32_t dwPinState,        /**< [In] The desired state of the clock. If CLOCK_STOP_HIGH or CLOCK_CTOP_LOW is ored with CLOCK_AUTO_OFF (0x80000000), the clock will be automatically switched off after each APDU. */
        void * pContext             /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback Delegate for \ref phhalHwContact_Wait
    */
    typedef phStatus_t (*cbWait)(
        uint8_t bUnit,              /**< [In] Unit of given timeout value (either #PHHAL_HW_CONTACT_TIME_MICROSECONDS or #PHHAL_HW_CONTACT_TIME_MILLISECONDS). */
        uint16_t wTimeout,          /**< [In] Timeout value. */
        void * pContext             /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback Delegate for \ref phhalHwContact_InitContactReader
    */
    typedef phStatus_t (*cbInitContactReader)(
        void * pContext             /**< [Out] User Defined Context. */
        );

    /**
    * \brief Callback HALContact parameter structure
    */
    typedef struct
    {
        uint16_t wId;                                           /**< Layer ID for this component, NEVER MODIFY! */
        cbExchange pExchangeCb;                                 /**< Pointer to \ref cbExchange Callback. */
        cbApplyProtocolSettingsAtr pApplyProtocolSettingsCb;    /**< Pointer to \ref cbApplyProtocolSettings Callback. */
        cbSetConfig32 pSetConfig32Cb;                           /**< Pointer to \ref cbSetConfig32 Callback. */
        cbGetConfig32 pGetConfig32Cb;                           /**< Pointer to \ref cbGetConfig32 Callback. */
        cbActivate pActivateCb;                                 /**< Pointer to \ref cbActivate Callback. */
        cbDeactivate pDeactivateCb;                             /**< Pointer to \ref cbDeactivate Callback. */
        cbClockStop pClockStopCb;                               /**< Pointer to \ref cbClockStop Callback. */
        cbWait pWaitCb;                                         /**< Pointer to \ref cbWait Callback. */
        cbInitContactReader pInitContactReader;                 /**< Pointer to \ref cbInitContactReader Callback. */
        void * pCbContext;                                      /**< Custom context which is passed back to each Callback. */
    } phhalHwContact_Callback_DataParams_t;

    /**
    * \brief Initialise the HALContact component.
    * \return Status code
    * \retval #PH_ERR_SUCCESS Operation successful.
    * \retval Other Depending on implementation and underlaying component.
    */
    phStatus_t phhalHwContact_Callback_Init(
        phhalHwContact_Callback_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                             /**< [In] Specifies the size of the data parameter structure. */
        cbExchange pExchangeCb,                                 /**< [In] Pointer to \ref cbExchange Callback. */
        cbApplyProtocolSettingsAtr pApplyProtocolSettingsCb,    /**< [In] Pointer to \ref cbApplyProtocolSettings Callback. */
        cbSetConfig32 pSetConfig32Cb,                           /**< [In] Pointer to \ref cbSetConfig32 Callback. */
        cbGetConfig32 pGetConfig32Cb,                           /**< [In] Pointer to \ref cbGetConfig32 Callback. */
        cbActivate pActivateCb,                                 /**< [In] Pointer to \ref cbActivate Callback. */
        cbDeactivate pDeactivateCb,                             /**< [In] Pointer to \ref cbDeactivate Callback. */
        cbClockStop pClockStopCb,                               /**< [In] Pointer to \ref cbClockStop Callback. */
        cbWait pWaitCb,                                         /**< [In] Pointer to \ref cbWait Callback. */
        cbInitContactReader pInitContactReader,                 /**< [In] Pointer to \ref cbInitContactReader Callback. */
        void * pCallbackContext                                 /**< [In] Custom context which is passed back to each Callback. */
        );

#endif /* NXPBUILD__PHHAL_HW_CONTACT_CALLBACK */

#ifdef NXPBUILD__PHHAL_HW_CONTACT_SMARTWARE
#define PHHAL_HW_CONTACT_SMARTWARE_ID                  0x06    /**< ID for Smartware Contact HAL component. */

    /**
    * \name Smartware specific configs
    */
    /*@{*/
    #define PHHAL_HW_CONTACT_SMARTWARE_SHADOW_COUNT           0x0068U     /**< Number of shadowed configurations. */

    typedef struct
    {
        uint16_t wId;                                                       /**< Layer ID for this component, NEVER MODIFY! */
        phbalReg_Smartware_DataParams_t * pBalDataParams;                   /**< pointer to the lower layers parameter structure */
        uint8_t * pTxBuffer;                                                /**< Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize;                                               /**< Size of the global transmit buffer. */
        uint16_t  wTxBufLen;                                                /**< Number of valid bytes within the transmit buffer. */
        uint8_t * pRxBuffer;                                                /**< Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize;                                               /**< Size of the global receive buffer. */
        uint16_t  wRxBufLen;                                                /**< Number of valid bytes within the receive buffer. */
        uint16_t  wRxBufStartPos;                                           /**< Starting position within the global receive buffer. */
        uint32_t  dwNumExpectedBytes;                                       /**< Number of expected bytes for Exchanges. */
        uint32_t  dwAtrLength;                                              /**< Length of the ATR stored internally in data params. */
        uint8_t   pAtr[PHHAL_HW_CONTACT_MAX_ATR_SIZE];                      /**< ATR of current card. */
        uint8_t * pIntBuffer;                                               /**< Buffer used for exchange with max size Data length */
        uint32_t  dwIntBufferLen;                                           /**< Number of valid bytes within the internal temp buffer. */
        uint32_t  dwIntBufferSize;                                          /**< Size of the internal buffer. */
        uint16_t  wTimingMode;                                              /**< Timing mode used to measure FDT */
        uint64_t  qwTiming_ns;                                              /**< Current timing value. */
        uint8_t   bProtocolType;                                            /**< Type of protocol for which the hal is configured for */
        uint32_t  wCfgShadow[PHHAL_HW_CONTACT_SMARTWARE_SHADOW_COUNT];      /**< Configuration shadow; Stores configuration for current protocol. */
    }   phhalHwContact_Smartware_DataParams_t;

    phStatus_t phhalHwContact_Smartware_Init(
        phhalHwContact_Smartware_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                             /**< [In] Specifies the size of the data parameter structure. */
        void * pBalDataParams,                                  /**< [In] Pointer to the lower layers parameter structure. */
        uint8_t * pTxBuffer,                                    /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t  wTxBufSize,                                   /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                    /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t  wRxBufSize,                                   /**< [In] Size of the global receive buffer. */
        uint8_t * pIntBuffer,                                   /**< [In] Pointer to internal buffer used by the Exchange() function. */
        uint32_t wIntBufferSize                                 /**< [In] Size of the internal buffer. */
        );

#endif /* NXPBUILD__PHHAL_HW_CONTACT_SMARTWARE */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHWCONTACT_H */
