/*
 * Copyright 2019 - 2020, 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_ISO3_CMD_H
#define PHHALHW_ISO3_CMD_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_ISO3

/**
* \name Minimum supported petalinux server version
*/
/*@{*/
#define PHHAL_HW_ISO3_PLS_CUR_VERSION                 2U
#define PHHAL_HW_ISO3_PLS_CUR_REVISION               20U
/*@}*/

/**
* \name Supported FPGA version
*/
/*@{*/
/*#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0dda          /**< 2020.02.12 add reset to fdt measurement and fixed setting and reseting of fdt trigger */
/*#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0ddb          /**< 2020.06.12 add functions for get las bit received and miller min/max test */
/*#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0ddc          /**< 2020.07.06 add for disable parity and fix timings */
/*#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0ddd          /**< 2020.08.11 trigger1&2 reset fix */
/*#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0dde          /**< 2020.10.28 add agc statistic, fpga dna, HF Pause and Startup granularity */
/*#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0ddf          /**< 2021.04.21 extend noise detection threshold to 16 bit */
/*#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0de0            /**< 2021.05.17 add error handling for buffer overflow in receiver */
/*#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0de1            /**< 2021.06.08 add Rx Watchdog to avoid reveiver stucking */
#define PHHAL_HW_ISO3_FPGA_VERSION                  0x0de2            /**< 2021.07.09 Trigger2 bugfix. For release 3.6.3  */
/*@}*/

/**
* \name ISO3 Server Application Commands
*/
/*@{*/
#define PHHAL_HW_ISO3_SERVER_CMD_INVALID            0x00000000U     /**< unknown server command */
#define PHHAL_HW_ISO3_SERVER_CMD_CALIBRATE          0x00000001U     /**< calibrate filter coefficients */
#define PHHAL_HW_ISO3_SERVER_CMD_GET_COEFF          0x00000002U     /**< get filter coefficients from ISO3 box */
#define PHHAL_HW_ISO3_SERVER_CMD_SET_COEFF          0x00000003U     /**< set filter coefficients located on server */
#define PHHAL_HW_ISO3_SERVER_CMD_GET_CONFIG         0x00000004U     /**< get configuration of server application */
#define PHHAL_HW_ISO3_SERVER_CMD_SET_CONFIG         0x00000005U     /**< set configuration of server application */
#define PHHAL_HW_ISO3_SERVER_CMD_READ_SOFTSCOPE     0x00000007U     /**< get data of soft scope buffer of FPGA */
#define PHHAL_HW_ISO3_SERVER_CMD_GET_VERSION        0x00000008U     /**< get version of PLS server running on the embedded linux */
#define PHHAL_HW_ISO3_SERVER_CMD_STORE_SOFTSCOPE    0x00000009U     /**< remotely buffer data from soft scope buffer */
#define PHHAL_HW_ISO3_SERVER_CMD_STORE_FROM_HOST    0x0000000AU     /**< remotely buffer data from host */
#define PHHAL_HW_ISO3_SERVER_CMD_SET_TIME           0x0000000BU     /**< set time of petalinux */
/*@}*/

/**
* \name Server Application Configs
*/
/*@{*/
#define PHHAL_HW_ISO3_SERVER_CONF_FIELD_STRENGTH    0x00000001U     /**< read write - field strength ratio of antenna type 1, 2 and 3 */
#define PHHAL_HW_ISO3_SERVER_CONF_MAC_ADDRESS       0x00000002U     /**< read only - mac address of server application */
/*@}*/

/**
* \name IIR Filter overflow Define
*/
/*@{*/
#define PHHAL_HW_ISO3_IIR_FILTER_OVERFLOW           0x00000001      /**< IIR Filter overflow in adder or multiplexer. see #PHHAL_HW_ISO3_REG_IIRFILTER_OVERFLOW */
/*@}*/

/**
* \name Maximal value on SENS and CAL ADC channels
*/
/*@{*/
#define PHDL_HW_ISO3_MAX_CAL_SENS_VALUE         0x1fff               /**< Maximal value on ADC converter for CAL and SENS. 14bit 2s complement 2Vpp. MSB is sign bit which this value does not take into account */
/*@}*/

/**
* \brief TODO: comment
* \return Status code
*/
phStatus_t phhalHw_ISO3_Cmd_SendServerCommand(
                        phhalHw_ISO3_DataParams_t * pDataParams,
                        uint32_t dwServerCommand,
                        uint8_t * pParams,
                        uint32_t wParamsSizeBytes
                        );

/**
* \brief TODO: comment
* \return Status code
*/
phStatus_t phhalHw_ISO3_Cmd_RecvServerResponse(
                        phhalHw_ISO3_DataParams_t * pDataParams,
                        uint8_t * pResp,
                        uint16_t wExpectedSizeBytes
                        );

/**
* \brief Write data to certain fpga register
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
returns the command execution status of the PCD. The PCD command execution status is mapped to a library status codes.
*/
phStatus_t phhalHw_ISO3_Cmd_WriteFpgaRegister(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint32_t dwAddr,                            /**< [In] Fpga Register for writing */
                            uint8_t * pValue,                           /**< [In] Transmitting data */
                            uint32_t dwSizeBytes                        /**< [In] Size in bytes of data */
                            );

/**
* \brief Read data located at certain fpga register
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
returns the command execution status of the PCD. The PCD command execution status is mapped to a library status codes.
*/
phStatus_t phhalHw_ISO3_Cmd_ReadFpgaRegister(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint32_t dwAddr,                            /**< [In] Fpga Register for reading */
                            uint8_t * pValue,                           /**< [OUT] Data located at fpga register */
                            uint32_t dwSizeBytes                        /**< [In] Size in bytes of expected data */
                            );

/**
* \brief Initialize fpga
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
returns the command execution status of the PCD. The PCD command execution status is mapped to a library status codes.
*/
phStatus_t phhalHw_ISO3_Cmd_InitFpga(
                            phhalHw_ISO3_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
                            );

/**
* \brief Read fpga signal data
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
returns the command execution status of the PCD. The PCD command execution status is mapped to a library status codes.
*/
phStatus_t phhalHw_ISO3_Cmd_ReadFpgaSignalData(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            int16_t * pwSig1Data,                       /**< [OUT] Signal data 1 */
                            int16_t * pwSig2Data,                       /**< [OUT] Signal data 2 */
                            uint16_t * pwSize                           /**< [OUT] Size in bytes of signal data */
                            );

/**
* \brief Write data directly to fpga output buffer
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
returns the command execution status of the PCD. The PCD command execution status is mapped to a library status codes.
*/
phStatus_t phhalHw_ISO3_Cmd_WriteFpgaOutBuffer(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint8_t * pTxData,                          /**< [In] Transmitting Data */
                            uint16_t  wTxLenghBytes,                    /**< [In] Size in Bytes of transmitting and receiving data */
                            uint8_t * ppRxData                          /**< [OUT] Receiving Data */
                            );

/**
* \brief Perform picc exchange
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
returns the command execution status of the PCD. The PCD command execution status is mapped to a library status codes.
*/
phStatus_t phhalHw_ISO3_Cmd_PiccExchange(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint8_t * pTxData,                          /**< [In] Transmitting data */
                            uint16_t  wTxLenghBytes,                    /**< [In] Size in bytes of transmitting data */
                            uint8_t ** ppRxData,                        /**< [OUT] Receiving data */
                            uint16_t * pRxLenghBytes                    /**< [OUT] Size in bytes of receiving data */
                            );

/**
* \brief Set step attenuator dampings
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
returns the command execution status of the PCD. The PCD command execution status is mapped to a library status codes.
*/
phStatus_t phhalHw_ISO3_Cmd_SetStepAttenuator(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint8_t bStepAttDb                          /**< [In] Set step attenuator to damping 0db, 6db, 12db or 24db */
                            );

/**
* \brief Load wave shape into certain slot within fpga's wave shape buffer
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
returns the command execution status of the PCD. The PCD command execution status is mapped to a library status codes.
*/
phStatus_t phhalHw_ISO3_Cmd_LoadWaveShape(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint16_t * pwWaveshape,                     /**< [In] Pointer to wave shape */
                            uint16_t wSizeBytes,                        /**< [In] Length of wave shape */
                            uint8_t wSlot                               /**< [In] Slot ID between 0 and NUMB_SLOTS (see configs) */
                            );

phStatus_t phhalHw_ISO3_Cmd_MeasureSystemResponse(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [in] Hal parameter struct */
                            int16_t * pwMlsTransmitted,                 /**< [out] destination pointer for returning the transmitted samples */
                            int16_t * pwMlsReceived,                    /**< [out] destination pointer for returning the received samples */
                            uint16_t * wNumSamples                      /**< [out] amount of samples. pwMlsTransmitted and pwMlsReceived have same length */
                            );

phStatus_t phhalHw_ISO3_Cmd_PlaySeq8(
                            phhalHw_ISO3_DataParams_t * pDataParams    /**< [In] Pointer to an initialized HAL parameter structure. */
                            );

struct tm;

/**
* \brief Set time of petalinux system
* \return Status code
*/
phStatus_t phhalHw_ISO3_Cmd_SetServerTime(
                            phhalHw_ISO3_DataParams_t * pDataParams,   /**< [In] Pointer to an initialized HAL parameter structure. */
                            struct tm* datetime                        /**< [In] Current timestamp */
                            );


/**
* \brief Get server version of the petalinux server application running within the embedded linux of the ISO3 Box.
* \return Status code
*/
phStatus_t phhalHw_ISO3_Cmd_GetPLSVersion(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint16_t * pwVersion,                       /**< [Out] Server Version */
                            uint16_t * pwRevision                       /**< [Out] Server Revision */
                            );

/**
* \brief lol
* \return Status code
*/
phStatus_t phhalHw_ISO3_Cmd_LoadOVSCoefficients(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint32_t * pCoefficients,                   /**< [In] Overshoot filter coefficients array */
                            uint16_t wSize,                             /**< [In] Size of overshoot filter coeffients array in bytes */
                            uint8_t bSlotID                             /**< [In] Slot ID within FPGA memory */
                            );

/**
* \brief Get the unique Device DNA of the FPGA board. This is 57 bit unique ID hardcoded for each Xilinx FPGA
* \return Status code
*/
phStatus_t phhalHw_ISO3_Cmd_GetDeviceDna(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint64_t *pdna                              /**< [Out] device dna value*/
                            );

/**
* \brief Get agc statistic
* \return Status code
*/
phStatus_t phhalHw_ISO3_Cmd_GetAgcStatistic(
                            phhalHw_ISO3_DataParams_t * pDataParams,    /**< [In] Pointer to an initialized HAL parameter structure. */
                            uint8_t bReset,                             /**  [In] If the is set to PH_ON the registers will be reseted after readout */
                            uint32_t * pdwMinAgc,                       /**< [Out] minimum value of agc control value */
                            uint32_t * pdwMaxAgc                        /**< [Out] maximum value of agc control value */
                            );

/**
 * \brief Set 32bit configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_ISO3_Cmd_SetConfig32(
        phhalHw_ISO3_DataParams_t * pDataParams,	   /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                              /**< [In] Configuration Identifier. */
        uint32_t dwValue                               /**< [In] Configuration Value. */
    );

/**
 * \brief Get 32bit configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_PARAMETER_OVERFLOW The parameter value is larger than the range of \b pValue.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_ISO3_Cmd_GetConfig32(
        phhalHw_ISO3_DataParams_t * pDataParams,                                                /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                                       /**< [In] Configuration Identifier. */
        uint32_t * pdwValue                                                                     /**< [Out] Configuration Value. */
    );



/** @} */
#endif /* NXPBUILD__PHHAL_HW_ISO3 */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_ISO3_CMD_H */
