/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic ISO14443-4 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHPALI7816P4_H
#define PHPALI7816P4_H

#include <ph_Status.h>
#include <phhalHwContact.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHPAL_I7816P4_SW

/** \defgroup phpalI7816p4_Sw Component : Software
* @{
*/

#define PHPAL_I7816P4_SW_ID                     0x01    /**< ID for Software ISO7816 layer */

#define PHPAL_I7816P4_SW_INT_BUFFER_SIZE        4096    /**< Size of the internally used buffer */
#define PHPAL_I7816P4_SW_INT_T0_HEADER_SIZE     5       /**< Size of the header for T=0 Protocol. */

/**
* \brief ISO7816 parameter structure
*/
typedef struct
{
    uint16_t  wId;                                              /**< Layer ID for this component, NEVER MODIFY! */
    void    * pHalDataParams;                                   /**< Pointer to the parameter structure of the underlying layer. */
    uint8_t   bT1MainState;                                     /**< Current Main Exchange-State for T=1 Protocol. */
    uint8_t   bT1TxState;                                       /**< Current TX Exchange-State for T=1 Protocol. */
    uint8_t   bT1RxState;                                       /**< Current RX Exchange-State for T=1 Protocol. */
    uint8_t   bT1SeqNumReceive;                                 /**< Sequence Number for received Block for T=1 Protocol. */
    uint8_t   bT1SeqNumSend;                                    /**< Sequence Number for sending Blocks for T=1 Protocol. */
    uint8_t   bT1CardIfs;                                       /**< Card sent IFS request containing the blocksize */
    uint8_t   bT1TxRetransmitCount;                             /**< Number of retransmits for Sending Blocks */
    uint8_t   bT1RxRetransmitCount;                             /**< Number of retransmits for receiving Block */
    uint8_t   pRxBuffer[PHPAL_I7816P4_SW_INT_BUFFER_SIZE];      /**< Buffer internally used as rx buffer */
    uint8_t   pTxBuffer[PHPAL_I7816P4_SW_INT_BUFFER_SIZE];      /**< Buffer internally used as tx buffer to collect buffered exchanges */
    uint16_t  wTxBufLen;                                        /**< Number of valid bytes within the transmit buffer. */
    uint16_t  wRxBufferLen;                                     /**< Length of the internally used buffer */
    uint16_t  wT0LengthExpected;                                /**< Length of the response expected, up to 256 for T=0 Protocol. */
    uint8_t   bT0CurrentCase;                                   /**< The currently used case for data exchange according to I7816. */
    uint8_t   pT0Header[PHPAL_I7816P4_SW_INT_T0_HEADER_SIZE];   /**< To save command header for T=0 Protocol. */
    uint8_t   bNad;                                             /**< Node Address; Ignored if bNadEnabled is equal '0'. */
    uint8_t   bIfsc;                                            /**< PICC Frame Size */
    uint8_t   bIfsd;                                            /**< PCD Frame Size */
    uint8_t   bAutoGetResponse;                                 /**< Automatic GetResponse on case 4S.3 */
    uint8_t   bAuto6cHandling;                                  /**< Automatic 6C Handling. */
    uint8_t   bProtocol;                                        /**< protocol T=0 or T=1 */
    uint8_t   pIntBuffer[PHPAL_I7816P4_SW_INT_BUFFER_SIZE];     /**< Buffer internally used as rx tx buffer */
    uint16_t  wIntBufferLen;                                    /**< Length of the internally used buffer */
    uint8_t * pIntPayload;                                      /**< Buffer internally used as payload buffer */
    uint16_t  wIntPayloadLen;                                   /**< Length of the internally used payload buffer */
    uint16_t  wI2CProtocol;                                     /**< Used Protocol for I2C communication */
    uint8_t   bBwtMultiplier;                                   /**< Multiplier for Bwt Timeout in case of WTX */
} phpalI7816p4_Sw_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI7816p4_Sw_Init(
                                phpalI7816p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                uint16_t wSizeOfDataParams,                    /**< [In] Specifies the size of the data parameter structure. */
                                void * pHalDataParams                          /**< [In] Pointer to the parameter structure of the underlying layer. */
                                );

/** @} */
#endif /* NXPBUILD__PHPAL_I7816P4_SW */

#ifdef NXPBUILD__PHPAL_I7816P4

/** \addtogroup ph_Error
* phpalI7816p4 Custom Errors
* @{
*/

/** \name phpalI7816p4 Error Codes
*/
/*@{*/
#define PHPAL_I7816P4_ERR_RECOVERY_FAILED  (PH_ERR_CUSTOM_BEGIN+0)     /**< ISO7816 error recovery failed. */
/*@}*/

/**
* \brief Set / Get Size of the Buffer the card is able to retrieve.
*/
#define PHPAL_I7816P4_CONFIG_IFSC               0x0001U

/**
* \brief Set / Get Node address.
*/
#define PHPAL_I7816P4_CONFIG_NAD                    0x0002U

/**
* \brief Set / Get Auto GetResponse
*/
#define PHPAL_I7816P4_CONFIG_AUTO_GETRESPONSE       0x0003U

/**
* \brief Set / Get protocol T=0 or T=1
*/
#define PHPAL_I7816P4_CONFIG_PROTOCOL               0x0004U

/**
* \brief Set / Get protocol T=0 or T=1
*/
#define PHPAL_I7816P4_CONFIG_AUTO_6C_HANDLING       0x0005U

/**
* \brief Set / Get TxRetransmit
*/
#define PHPAL_I7816P4_CONFIG_TX_RETRANSMIT_COUNT    0x0006U

/**
* \brief Set / Get RxRetransmit
*/
#define PHPAL_I7816P4_CONFIG_RX_RETRANSMIT_COUNT    0x0007U

/**
* \brief Set / Get Size of the Buffer the reader is able to retrieve.
*/
#define PHPAL_I7816P4_CONFIG_IFSD                   0x0008U

/**
* \brief Set / The Protocol used for I2C Communication.
*/
#define PHPAL_I7816P4_CONFIG_I2C_PROTOCOL           0x0009U

/*@}*/

/*I2C PROTOCOLS CONFIGS*/
#define PHPAL_I7816P4_I2C_PROTOCOL_T1OVERI2C_LEN_ONE_LRC     0x00
#define PHPAL_I7816P4_I2C_PROTOCOL_T1OVERI2C_LEN_ONE_CRC     0x02
#define PHPAL_I7816P4_I2C_PROTOCOL_GP_LEN_ONE_CRC            0x0A
#define PHPAL_I7816P4_I2C_PROTOCOL_GP_LEN_TWO_CRC            0x0B
#define PHPAL_I7816P4_I2C_PROTOCOL_GP_LEN_TWO_CRC_SWAPPED    0x0F

/*I2C SET CONFIG MASKS*/
#define PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE               0x01
#define PHPAL_I7816P4_SET_CONFIG_MASK_REDUNDANCY_MODE        0x02
#define PHPAL_I7816P4_SET_CONFIG_MASK_CRC_SWAPPED            0x04
#define PHPAL_I7816P4_SET_CONFIG_MASK_PROTOCOL               0x08

/**
* \brief Reset the ISO7816 protocol parameters.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI7816p4_ResetProtocol(
                                      void * pDataParams   /**< [In] Pointer to this layer's parameter structure. */
                                      );

/**
* \brief Deselect ISO7816 card.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI7816p4_Deselect(
                                 void * pDataParams    /**< [In] Pointer to this layer's parameter structure. */
                                 );

/**
* \brief Perform presence check for current card.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI7816p4_PresCheck(
                                  void * pDataParams   /**< [In] Pointer to this layer's parameter structure. */
                                  );

/**
* \brief Perform ISO7816 Data Exchange with Picc.
*
* \c wOption can be one of:\n
* \li #PH_EXCHANGE_DEFAULT
* \li #PH_EXCHANGE_TXCHAINING
* \li #PH_EXCHANGE_RXCHAINING
* \li #PH_EXCHANGE_RXCHAINING_BUFSIZE
*
* Additionally, the following options are also available:\n
* \li #PH_EXCHANGE_BUFFER_FIRST
* \li #PH_EXCHANGE_BUFFER_CONT
* \li #PH_EXCHANGE_BUFFER_LAST
*
* Alternatively to the FIRST/CONT/LAST options, the following bits can be combined:\n
* \li #PH_EXCHANGE_BUFFERED_BIT
* \li #PH_EXCHANGE_LEAVE_BUFFER_BIT
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI7816p4_Exchange(
                                 void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wOption,         /**< [In] Option parameter. */
                                 uint8_t * pTxBuffer,      /**< [In] Data to transmit. */
                                 uint16_t wTxLength,       /**< [In] Length of data to transmit. */
                                 uint8_t ** ppRxBuffer,    /**< [Out] Pointer to received data. */
                                 uint16_t * pRxLength      /**< [Out] number of received data bytes. */
                                 );

/**
* \brief Set configuration parameter.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI7816p4_SetConfig(
                                  void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                  uint16_t wConfig,    /**< [In] Configuration Identifier */
                                  uint16_t wValue      /**< [In] Configuration Value */
                                  );

/**
* \brief Get configuration parameter.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI7816p4_GetConfig(
                                  void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                  uint16_t wConfig,    /**< [In] Configuration Identifier */
                                  uint16_t * pValue    /**< [Out] Configuration Value */
                                  );

/** @} */
#endif /* NXPBUILD__PHPAL_I7816P4 */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHPALI7816P4_H */
