/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic ISO7816-4A Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHPALI7816P4A_H
#define PHPALI7816P4A_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHPAL_I7816P4A_SW

/** \defgroup phpalI7816p4a_Sw Component : Software
* @{
*/

#define PHPAL_I7816P4A_SW_ID   0x01    /**< ID for Software ISO7816-4A layer     */

/**
* \brief Software PAL-ISO7816P4A parameter structure
*/
typedef struct
{
    uint16_t wId;               /**< Layer ID for this component, NEVER MODIFY! */
    void  * pHalDataParams;     /**< Pointer to the parameter structure of the underlying layer. */
    uint8_t bIfsd;              /**< (Current) PCD Frame Size Integer; */
    uint8_t bCid;               /**< (Current) CID of the currently used Card; */
} phpalI7816p4a_Sw_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI7816p4a_Sw_Init(
                                 phpalI7816p4a_Sw_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure. */
                                 void * pHalDataParams                         /**< [In] Pointer to the parameter structure of the underlying layer.*/
                                 );

/** @} */
#endif /* NXPBUILD__PHPAL_I7816P4A_SW */


#ifdef NXPBUILD__PHPAL_I7816P4A

/** \defgroup phpalI7816p4a
* \brief These Components implement the additional ISO/IEC 7816-4 activation steps
* needed for ISO7816 cards.
* @{
*/

/**
* \brief Set / Get Size of the Buffer the card is able to retrieve.
*/
#define PHPAL_I7816P4A_CONFIG_IFSC               0x0001U

/**
* \brief Set / Get Size of the Buffer the reader is able to retrieve.
*/
#define PHPAL_I7816P4A_CONFIG_IFSD               0x0002U

/**
* \brief Perform a ISO7816-4A "IFS Request" command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Invalid response received.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI7816p4a_IfsRequest(
                                    void * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                    uint8_t bIfsd          /**< [In] Frame Size Integer. */
                                    );

/**
* \brief Perform a ISO7816-4A "Protocol and Parameter Selection" command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Invalid response received.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI7816p4a_Pps(
                             void * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                             uint8_t bProtocolType, /**< [In] Specifies the used Protocol Type */
                             uint8_t bFValue,       /**< [In] F Value for Datarate; */
                             uint8_t bDValue        /**< [In] D Value for Datarate; */
                             );

/**
* \brief Perform ISO7816-4A Activate, Ifs and Pps commands.
*
* \param bFsdi is only set when bProtocolType equals T=1
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Invalid response received.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI7816p4a_ActivateCard(
                                      void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                      uint8_t bProtocolType,    /**< [In] Specifies the used Protocol Type */
                                      uint8_t bFsdi,            /**< [In] Frame Size Integer; */
                                      uint8_t bFValue,          /**< [In] F Value for Datarate; */
                                      uint8_t bDValue,          /**< [In] D Value for Datarate; */
                                      uint8_t * pAtr,           /**< [Out] Cards ATR */
                                      uint16_t * pwATRLength    /**< [Out] Length of the reiceived ATR. */
                                      );

/**
* \brief Get configuration parameter.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI7816p4a_GetConfig(
                                   void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wConfig,    /**< [In] Configuration Identifier */
                                   uint16_t * pValue    /**< [Out] Configuration Value */
                                   );

#endif /* NXPBUILD__PHPAL_I7816P4A */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHPALI7816P4A_H */
