/*
 * Copyright 2009 - 2018, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software MIFARE(R) Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  CHu: Generated 31. July 2009
 *
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phalMfc.h>
#include <phpalMifare.h>
#include <ph_RefDefs.h>
#include <phKeyStore.h>

#ifdef NXPBUILD__PHAL_MFC_SW

#include "phalMfc_Sw.h"
#include "../phalMfc_Int.h"

phStatus_t phalMfc_Sw_Init(phalMfc_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pPalMifareDataParams,
	void * pKeyStoreDataParams)
{
	if (sizeof(phalMfc_Sw_DataParams_t) != wSizeOfDataParams)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFC);
	}

	PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFC);
	PH_ASSERT_NULL_PARAM (pPalMifareDataParams, PH_COMP_AL_MFC);

	/* init private data */
	pDataParams->wId                    = PH_COMP_AL_MFC | PHAL_MFC_SW_ID;
	pDataParams->pPalMifareDataParams   = pPalMifareDataParams;
	pDataParams->pKeyStoreDataParams    = pKeyStoreDataParams;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sw_Authenticate(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint16_t wKeyNo,
	uint16_t wKeyVersion, uint8_t * pUid, uint8_t bUidLength)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	phStatus_t  PH_MEMLOC_REM status;
	uint8_t     PH_MEMLOC_REM aKey[PH_KEYSTORE_SW_MAX_KEY_SIZE];
	uint8_t *   PH_MEMLOC_REM pKey;
	uint16_t    PH_MEMLOC_REM bKeystoreKeyType;

	/* Check if software key store is available. If software keystore is available, the key wil be
	   utilized form software one else the keys wuill be utilized from Reader. */
	if (pDataParams->pKeyStoreDataParams == NULL)
	{
		/* There is no software keystore available. */
		return phpalMifare_MfcAuthenticateKeyNo(
			pDataParams->pPalMifareDataParams,
			bBlockNo,
			bKeyType,
			wKeyNo,
			wKeyVersion,
			&pUid[bUidLength - 4]);
	}
	else
	{
		/* Bail out if we haven't got a keystore. */
		if (pDataParams->pKeyStoreDataParams == NULL)
		{
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFC);
		}

		/* Retrieve KeyA & KeyB from keystore. */
		PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
			pDataParams->pKeyStoreDataParams,
			wKeyNo,
			wKeyVersion,
			PH_KEYSTORE_KEY_TYPE_AES256_SIZE,
			aKey,
			&bKeystoreKeyType));

		/* Check key type */
		if (bKeystoreKeyType != PH_KEYSTORE_KEY_TYPE_MIFARE)
		{
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFC);
		}

		/* Evaluate which key to use */
		if ((bKeyType & 0x7F) == PHHAL_HW_MFC_KEYA)
		{
			/* Use KeyA */
			pKey = aKey;
		}
		else if ((bKeyType & 0x7F) == PHHAL_HW_MFC_KEYB)
		{
			/* Use KeyB */
			pKey = &aKey[PHHAL_HW_MFC_KEY_LENGTH];
		}
		else
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFC);
		}

		/* Exchage the information to PAL layer. */
		status = phpalMifare_MfcAuthenticate(
			pDataParams->pPalMifareDataParams,
			bBlockNo,
			bKeyType,
			pKey,
			&pUid[bUidLength - 4]);
		return status;
	}
}

phStatus_t phalMfc_Sw_Read(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pBlockData)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM aCommand[2];
	uint8_t *   PH_MEMLOC_REM pRxBuffer;
	uint16_t    PH_MEMLOC_REM wRxLength;

	/* Build command frame. */
	aCommand[0] = PHAL_MFC_CMD_READ;
	aCommand[1] = bBlockNo;

	/* Transmit the command frame. */
	PH_CHECK_SUCCESS_FCT(statusTmp, phpalMifare_ExchangeL3(
		pDataParams->pPalMifareDataParams,
		PH_EXCHANGE_DEFAULT,
		aCommand,
		2,
		&pRxBuffer,
		&wRxLength
		));

	/* Check received length. */
	if (wRxLength != PHAL_MFC_DATA_BLOCK_LENGTH)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFC);
	}

	/* Copy received data block. */
	memcpy(pBlockData, pRxBuffer, wRxLength);  /* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sw_ReadValue(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pValue,
	uint8_t * pAddrData)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bBuffer[PHAL_MFC_DATA_BLOCK_LENGTH];

	/* Perform read operation. */
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Sw_Read(pDataParams, bBlockNo, bBuffer));

	/* Check format of value block. */
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Int_CheckValueBlockFormat(bBuffer));

	/* Return received contents. */
	*pAddrData = bBuffer[12];
	memcpy(pValue, bBuffer, PHAL_MFC_VALUE_BLOCK_LENGTH);  /* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sw_Write(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pBlockData)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM aCommand[2];
	uint8_t *   PH_MEMLOC_REM pRxBuffer;
	uint16_t    PH_MEMLOC_REM wRxLength;

	/* Build command frame. */
	aCommand[0] = PHAL_MFC_CMD_WRITE;
	aCommand[1] = bBlockNo;

	/* Transmit the command frame (first part). */
	PH_CHECK_SUCCESS_FCT(statusTmp, phpalMifare_ExchangeL3(
		pDataParams->pPalMifareDataParams,
		PH_EXCHANGE_DEFAULT,
		aCommand,
		2,
		&pRxBuffer,
		&wRxLength
		));

	/* Transmit the data (second part). */
	PH_CHECK_SUCCESS_FCT(statusTmp, phpalMifare_ExchangeL3(
		pDataParams->pPalMifareDataParams,
		PH_EXCHANGE_DEFAULT,
		pBlockData,
		PHAL_MFC_DATA_BLOCK_LENGTH,
		&pRxBuffer,
		&wRxLength
		));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sw_WriteValue(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pValue,
	uint8_t bAddrData)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bBlockData[PHAL_MFC_DATA_BLOCK_LENGTH];

	/* Build a valid value block. */
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Int_CreateValueBlock(pValue, bAddrData,bBlockData ));

	/* Perform the write operation. */
	return phalMfc_Sw_Write(pDataParams, bBlockNo, bBlockData);
}

phStatus_t phalMfc_Sw_Increment(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pValue)
{
	return phalMfc_Int_Value(pDataParams->pPalMifareDataParams, PHAL_MFC_CMD_INCREMENT, bBlockNo, pValue);
}

phStatus_t phalMfc_Sw_Decrement(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pValue)
{
	return phalMfc_Int_Value(pDataParams->pPalMifareDataParams, PHAL_MFC_CMD_DECREMENT, bBlockNo, pValue);
}

phStatus_t phalMfc_Sw_Transfer(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo)
{
    return phalMfc_Int_Transfer(pDataParams->pPalMifareDataParams, bBlockNo);
}

phStatus_t phalMfc_Sw_Restore(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo)
{
	uint8_t PH_MEMLOC_REM bBuffer[PHAL_MFC_VALUE_BLOCK_LENGTH];

	/* restore needs four dummy bytes */
	memset(bBuffer, 0x00, PHAL_MFC_VALUE_BLOCK_LENGTH);  /* PRQA S 3200 */

	return phalMfc_Int_Value(pDataParams->pPalMifareDataParams, PHAL_MFC_CMD_RESTORE, bBlockNo, bBuffer);
}

phStatus_t phalMfc_Sw_IncrementTransfer(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo,
	uint8_t * pValue)
{
	phStatus_t PH_MEMLOC_REM statusTmp;
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Sw_Increment(pDataParams, bSrcBlockNo, pValue));
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Sw_Transfer(pDataParams, bDstBlockNo));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sw_DecrementTransfer(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo,
	uint8_t * pValue)
{
	phStatus_t PH_MEMLOC_REM statusTmp;
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Sw_Decrement(pDataParams, bSrcBlockNo, pValue));
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Sw_Transfer(pDataParams, bDstBlockNo));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sw_RestoreTransfer(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo)
{
	phStatus_t PH_MEMLOC_REM statusTmp;
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Sw_Restore(pDataParams, bSrcBlockNo));
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfc_Sw_Transfer(pDataParams, bDstBlockNo));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sw_PersonalizeUid(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bUidType)
{
	uint8_t     PH_MEMLOC_REM aCommand[2];
	uint8_t     PH_MEMLOC_REM * pRecv;
	uint16_t    PH_MEMLOC_REM wRxlen;

	/* Build command frame. */
	aCommand[0] = PHAL_MFC_CMD_PERSOUID;
	aCommand[1] = bUidType;

	/* Transmit the command frame. */
	return phpalMifare_ExchangeL3(
		pDataParams->pPalMifareDataParams,
		PH_EXCHANGE_DEFAULT,
		aCommand,
		2,
		&pRecv,
		&wRxlen
		);
}

phStatus_t phalMfc_Sw_GetVersion(phalMfc_Sw_DataParams_t * pDataParams, uint8_t * pVersion)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCommand;
	uint8_t *   PH_MEMLOC_REM pRxBuffer;
	uint16_t    PH_MEMLOC_REM wRxLength;

	/* Build command frame. */
	bCommand = PHAL_MFC_CMD_GETVERSION;

	/* Transmit the command frame. */
	PH_CHECK_SUCCESS_FCT(statusTmp, phpalMifare_ExchangeL3(
		pDataParams->pPalMifareDataParams,
		PH_EXCHANGE_DEFAULT,
		&bCommand,
		1,
		&pRxBuffer,
		&wRxLength
		));

	/* Check received length. */
	if (wRxLength != PHAL_MFC_VERSION_LENGTH)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFC);
	}

	/* Copy received data block. */
	memcpy(pVersion, pRxBuffer, wRxLength); /* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

#endif /* NXPBUILD__PHAL_MFC_SW */
