/*
 * Copyright 2013, 2017 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <string.h>
#include <phTools.h>
#include <phKeyStore.h>

#ifdef NXPBUILD__PH_CRYPTOSYM
#include <phCryptoSym.h>
#endif /* NXPBUILD__PH_CRYPTOSYM */
#ifdef NXPBUILD__PH_CRYPTORNG
#include <phCryptoRng.h>
#endif /* NXPBUILD__PH_CRYPTORNG */

#ifdef NXPBUILD__PHAL_MFDF_SW

#include "../phalMfdf_Int.h"
#include "phalMfdf_Sw.h"
#include "phalMfdf_Sw_Int.h"

phStatus_t phalMfdf_Sw_Init(phalMfdf_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pPalMifareDataParams, void * pKeyStoreDataParams,
	void * pCryptoDataParamsEnc, void * pCryptoRngDataParams, void * pHalDataParams)
{
	/* data param check */
	if (sizeof(phalMfdf_Sw_DataParams_t) != wSizeOfDataParams)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFDF);
	}
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFDF);
	PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_MFDF);


	/* init private data */
	pDataParams->wId = PH_COMP_AL_MFDF | PHAL_MFDF_SW_ID;
	pDataParams->pPalMifareDataParams = pPalMifareDataParams;
	pDataParams->pKeyStoreDataParams = pKeyStoreDataParams;
	pDataParams->pCryptoDataParamsEnc = pCryptoDataParamsEnc;
	pDataParams->pCryptoRngDataParams = pCryptoRngDataParams;
	pDataParams->bLastBlockIndex = 0;
	pDataParams->pHalDataParams = pHalDataParams;
	/* 2 Byte CRC initial value in Authenticate mode. */
	pDataParams->wCrc = PH_TOOLS_CRC16_PRESET_ISO14443A;

	/* 4 Byte CRC initial value in 0x1A, 0xAA mode. */
	pDataParams->dwCrc = PH_TOOLS_CRC32_PRESET_DF8;

	memset(pDataParams->bSessionKey, 0x00, 24);  /* PRQA S 3200 */
	pDataParams->bKeyNo = 0xFF; /* Set to invalid */
	memset(pDataParams->bIv, 0x00, 16); /* PRQA S 3200 */
	memset(pDataParams->pAid, 0x00, 3);  /* PRQA S 3200 */
	pDataParams->bAuthMode = PHAL_MFDF_NOT_AUTHENTICATED; /* Set to invalid */
	pDataParams->bWrappedMode = 0x00; /* Set to false */
	pDataParams->bCryptoMethod = 0xFF; /* No crypto just after init */
	pDataParams->wAdditionalInfo = 0x0000;
	pDataParams->wPayLoadLen = 0;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}




/* MIFARE DESFire security related commands. ----------------------------------------------------------------------------------------- */

phStatus_t phalMfdf_Sw_GetKeySettings(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t * pResponse)
{
	/**
	* This command can be issued without valid authentication
	*/
	uint16_t    PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen;
	uint8_t     PH_MEMLOC_REM *pRecv;

	/* form the command */
	bCmdBuff[0] = PHAL_MFDF_CMD_GET_KEY_SETTINGS;
	wCmdLen = 1;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(pDataParams,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));
	if (wRxlen != 0x02)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}
	memcpy(pResponse, pRecv, wRxlen); /* PRQA S 3200 */
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}


phStatus_t phalMfdf_Sw_GetKeyVersion(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t * pResponse)
{
	/**
	* This command can be issued without valid authentication
	*/
	uint16_t  PH_MEMLOC_REM statusTmp;
	uint8_t PH_MEMLOC_REM bCmdBuff[20];
	uint16_t PH_MEMLOC_REM wRxlen;
	uint16_t PH_MEMLOC_REM wCmdLen = 0;
	uint8_t PH_MEMLOC_REM *pRecv;

	if (bKeyNo > 0x0d)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_GET_KEY_VERSION;
	bCmdBuff[wCmdLen++] = bKeyNo;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(pDataParams,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));
	if (wRxlen != 0x01)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}
	memcpy(pResponse, pRecv, wRxlen); /* PRQA S 3200 */
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}




/* MIFARE DESFire PICC level commands. ----------------------------------------------------------------------------------------------- */
phStatus_t phalMfdf_Sw_CreateApplication(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pAid, uint8_t bKeySettings1,
	uint8_t bKeySettings2, uint8_t * pISOFileId, uint8_t * pISODFName, uint8_t bISODFNameLen)
{
	/*
	If (bKeySettings2 & 0x03)== 00 [DES, 3DES] then pDataParams->bAuthMode can be either
	0x0A or 0x1A.
	If (bKeySettings2 & 0x03)== 01 [3K3DES] then pDataParams->bAuthMode can only be 0x1A.
	If (bKeySettings2 & 0x03)== 10 [AES] then pDataParams->bAuthMode can only be 0xAA.
	*/
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if ((bISODFNameLen > 16) || (bOption > 0x03) || (bOption == 0x02))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_CREATE_APPLN;

	memcpy(&bCmdBuff[wCmdLen], pAid, 3); /* PRQA S 3200 */
	wCmdLen += 3;

	bCmdBuff[wCmdLen++] = bKeySettings1;
	bCmdBuff[wCmdLen++] = bKeySettings2;

	if (bOption & 0x01)
	{
		/* wISOFileId is present */
		bCmdBuff[wCmdLen++] = pISOFileId[0];
		bCmdBuff[wCmdLen++] = pISOFileId[1];
	}
	if (bOption & 0x02)
	{
		/* pISODFName is present */
		memcpy(&bCmdBuff[wCmdLen], pISODFName, bISODFNameLen); /* PRQA S 3200 */
		wCmdLen = wCmdLen + bISODFNameLen;
	}

	return phalMfdf_Sw_Int_Write_Plain(
		pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_DeleteApplication(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t * pAppId)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[10];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_DELETE_APPLN;
	memcpy(&bCmdBuff[1], pAppId, PHAL_MFDF_DFAPPID_SIZE); /* PRQA S 3200 */
	wCmdLen += PHAL_MFDF_DFAPPID_SIZE;

	return phalMfdf_Sw_Int_Write_Plain(
		pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_GetApplicationIDs(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t * pAidBuff, uint8_t * pNumAIDs)
{
	/**
	A PICC can store upto 28 applications. PICC will return all
	AIDs (3 byte/aid) in single response if the number of applications <= 20
	else, it will send the  remaining AIDs in the second transmission. The
	first response sent by PICC will have 0xAF in the status byte.
	*/
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen;
	uint8_t     PH_MEMLOC_REM *pRecv;

	/* Initialization */
	*pNumAIDs = 0;

	/* form the command */
	bCmdBuff[0] = PHAL_MFDF_CMD_GET_APPLN_IDS;
	wCmdLen = 1;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(pDataParams,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));
	memcpy(pAidBuff, pRecv, wRxlen); /* PRQA S 3200 */

	/* Update pNumAids and return  */
	*pNumAIDs = (uint8_t) (wRxlen / 3);

	/* Length should be a multiple of 3. Else return error */
	if (wRxlen % 3)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sw_GetDFNames(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pDFBuffer, uint8_t * pDFInfoLen)
{
	/**
	Returns AID(3B), FID(2B), DF-Name(1..16B) in one frame.

	Will return PH_EXCHANGE_RXCHAINING if more DF names are present.
	The caller has to call the function with option PH_EXCHANGE_RXCHAINING

	Will not work if authenticated in standard TDES or AES modes as per the
	Implementation Hints document.
	*/
	uint16_t    PH_MEMLOC_REM status;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen;
	uint8_t *   PH_MEMLOC_REM pRecv;

	/* form the command */
	if (bOption == PH_EXCHANGE_RXCHAINING)
	{
		bCmdBuff[0] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;
	}
	else if (bOption == PH_EXCHANGE_DEFAULT)
	{
		bCmdBuff[0] = PHAL_MFDF_CMD_GET_DF_NAMES;
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}
	wCmdLen = 1;

	if ((pDataParams->bAuthMode == PHAL_MFDF_AUTHENTICATEISO) ||
		(pDataParams->bAuthMode == PHAL_MFDF_AUTHENTICATEAES))
	{
		/* Should return, invalid scenario error. Card will be disabled
		in case this command is sent in these modes */
		return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_MFDF);
	}

	/* Send the command */
	status = phalMfdf_ExchangeCmd(
		pDataParams,
		pDataParams->pPalMifareDataParams,
		pDataParams->bWrappedMode,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	);

	/* check for protocol errors */
	if (((status & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING) &&
		(wRxlen == 0))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}
	/* Should return either zero bytes or more than 4 bytes. Anything inbetween
	is an error */
	if ((status == PH_ERR_SUCCESS) && (wRxlen != 0) && (wRxlen < 5))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}
	if (((status & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
		&& (wRxlen < 5))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}
	if (((status & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING) || (status == PH_ERR_SUCCESS))
	{
		memcpy(pDFBuffer, pRecv, wRxlen); /* PRQA S 3200 */
		*pDFInfoLen = (uint8_t) wRxlen;
	}
	return status;
}

phStatus_t phalMfdf_Sw_SelectApplication(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t * pAppId)
{
	uint16_t  PH_MEMLOC_REM statusTmp;
	uint8_t   PH_MEMLOC_REM bCmdBuff[10];
	uint16_t  PH_MEMLOC_REM wRxlen;
	uint16_t  PH_MEMLOC_REM wCmdLen;
	uint8_t * PH_MEMLOC_REM pRecv;

	/* form the command */
	bCmdBuff[0] = PHAL_MFDF_CMD_SELECT_APPLN;
	memcpy(&bCmdBuff[1], pAppId, PHAL_MFDF_DFAPPID_SIZE); /* PRQA S 3200 */
	wCmdLen = PHAL_MFDF_DFAPPID_SIZE + 1;


	/* Send the command */
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_ExchangeCmd(
		pDataParams,
		pDataParams->pPalMifareDataParams,
		pDataParams->bWrappedMode,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));

	/* Store the currently selected application Id */
	memcpy(pDataParams->pAid, pAppId, 3);  /* PRQA S 3200 */

	/* Select command never returns CMAC */
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}


phStatus_t phalMfdf_Sw_GetVersion(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t * pVerInfo)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t     PH_MEMLOC_REM *pRecv;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_GET_VERSION;
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(pDataParams,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));
	memcpy(pVerInfo, pRecv, wRxlen); /* PRQA S 3200 */

	if (wRxlen != 28)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sw_FreeMem(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t * pMemInfo)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[8];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t     PH_MEMLOC_REM *pRecv;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_FREE_MEM;
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(pDataParams,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));

	if (wRxlen != 3)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}

	memcpy(pMemInfo, pRecv, wRxlen); /* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}





/* MIFARE DESFire Application level commands. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfdf_Sw_GetFileIDs(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t * pFid, uint8_t * pNumFIDs)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wCmdLen;
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint8_t     PH_MEMLOC_REM * pRecv;

	/* form the command */
	bCmdBuff[0] = PHAL_MFDF_CMD_GET_FILE_IDS;
	wCmdLen = 1;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(
		pDataParams,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));

	memcpy(pFid, pRecv, wRxlen); /* PRQA S 3200 */
	*pNumFIDs = (uint8_t) wRxlen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sw_GetISOFileIDs(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t * pFidBuffer, uint8_t * pNumFIDs)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen;
	uint8_t     PH_MEMLOC_REM * pRecv;

	/* Initialization */
	*pNumFIDs = 0;

	/* form the command */
	bCmdBuff[0] = PHAL_MFDF_CMD_GET_ISO_FILE_IDS;
	wCmdLen = 1;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(
		pDataParams,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));
	/* Length should be multiple of 2 */
	if ((wRxlen != 0) && (wRxlen % 2))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}
	memcpy(pFidBuffer, pRecv, wRxlen); /* PRQA S 3200 */

	/* Update pNumAids and return  */
	*pNumFIDs = (uint8_t) (wRxlen / 2);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sw_GetFileSettings(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
	uint16_t    PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t     PH_MEMLOC_REM *pRecv;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_GET_FILE_SETTINGS;
	bCmdBuff[wCmdLen++] = bFileNo;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(
		pDataParams,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));

	/* 7 => Data files, 17 => value files, 13 => record files */
	if ((wRxlen != 7) && (wRxlen != 17) && (wRxlen != 13))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}

	memcpy(pFSBuffer, pRecv, wRxlen); /* PRQA S 3200 */

	/* Update pBufferLen and return  */
	*pBufferLen = (uint8_t) wRxlen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}


phStatus_t phalMfdf_Sw_CreateStdDataFile(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t  * pISOFileId, uint8_t bCommSett,
	uint8_t * pAccessRights, uint8_t * pFileSize)
{
	/*
	If (bKeySettings2 & 0x03)== 00 [DES, 3DES] then pDataParams->bAuthMode can be either
	0x0A or 0x1A.
	If (bKeySettings2 & 0x03)== 01 [3K3DES] then pDataParams->bAuthMode can only be 0x1A.
	If (bKeySettings2 & 0x03)== 10 [AES] then pDataParams->bAuthMode can only be 0xAA.
	*/
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if ((bFileNo > 0x1f) || (bOption > 0x01))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommSett != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_CREATE_STD_DATAFILE;

	/* File Number */
	bCmdBuff[wCmdLen++] = bFileNo;

	/* Copy ISO Filed ID if present */
	if (bOption == 0x01)
	{
		bCmdBuff[wCmdLen++] = pISOFileId[0];
		bCmdBuff[wCmdLen++] = pISOFileId[1];
	}

	/* Copy communication settings. communication settings in the first nibble so right shifting */
	bCmdBuff[wCmdLen++] = bCommSett >> 4;

	/* Copy Access rights */
	bCmdBuff[wCmdLen++] = pAccessRights[0];
	bCmdBuff[wCmdLen++] = pAccessRights[1];

	/* Copy File size supplied by the user */
	memcpy(&bCmdBuff[wCmdLen], pFileSize, 3); /* PRQA S 3200 */
	wCmdLen += 3;

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_CreateBackupDataFile(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t *pISOFileId, uint8_t bCommSett,
	uint8_t * pAccessRights, uint8_t * pFileSize)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if ((bFileNo > 0x1f) || (bOption > 0x01))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommSett != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_CREATE_BKUP_DATAFILE;

	/* File Number */
	bCmdBuff[wCmdLen++] = bFileNo;

	/* Copy ISO Filed ID if present */
	if (bOption == 0x01)
	{
		bCmdBuff[wCmdLen++] = pISOFileId[0];
		bCmdBuff[wCmdLen++] = pISOFileId[1];
	}

	/* Copy communication settings */
	bCmdBuff[wCmdLen++] = bCommSett >> 4;

	/* Copy Access rights */
	bCmdBuff[wCmdLen++] = pAccessRights[0];
	bCmdBuff[wCmdLen++] = pAccessRights[1];

	/* Copy File size supplied by the user */
	memcpy(&bCmdBuff[wCmdLen], pFileSize, 3); /* PRQA S 3200 */
	wCmdLen += 3;

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_CreateValueFile(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t bCommSett, uint8_t * pAccessRights, uint8_t * pLowerLmit,
	uint8_t * pUpperLmit, uint8_t * pValue, uint8_t bLimitedCredit)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommSett != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_CREATE_VALUE_FILE;
	bCmdBuff[wCmdLen++] = bFileNo;
	bCmdBuff[wCmdLen++] = bCommSett >> 4;

	memcpy(bCmdBuff + wCmdLen, pAccessRights, 2); /* PRQA S 3200 */
	wCmdLen = wCmdLen + 2;

	memcpy(bCmdBuff + wCmdLen, pLowerLmit, 4); /* PRQA S 3200 */
	wCmdLen = wCmdLen + 4;

	memcpy(bCmdBuff + wCmdLen, pUpperLmit, 4); /* PRQA S 3200 */
	wCmdLen = wCmdLen + 4;

	memcpy(bCmdBuff + wCmdLen, pValue, 4); /* PRQA S 3200 */
	wCmdLen = wCmdLen + 4;

	bCmdBuff[wCmdLen++] = bLimitedCredit;

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_CreateLinearRecordFile(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t  bFileNo, uint8_t  *pIsoFileId, uint8_t bCommSett,
	uint8_t * pAccessRights, uint8_t * pRecordSize, uint8_t * pMaxNoOfRec)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if ((bFileNo > 0x1f) || (bOption > 0x01))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommSett != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_CREATE_LINEAR_RECFILE;

	/* Copy the value information supplied by the user */
	bCmdBuff[wCmdLen++] = bFileNo;

	if (bOption == 1)
	{
		bCmdBuff[wCmdLen++] = pIsoFileId[0];
		bCmdBuff[wCmdLen++] = pIsoFileId[1];
	}
	bCmdBuff[wCmdLen++] = bCommSett >> 4;
	bCmdBuff[wCmdLen++] = pAccessRights[0];
	bCmdBuff[wCmdLen++] = pAccessRights[1];

	bCmdBuff[wCmdLen++] = pRecordSize[0];
	bCmdBuff[wCmdLen++] = pRecordSize[1];
	bCmdBuff[wCmdLen++] = pRecordSize[2];

	bCmdBuff[wCmdLen++] = pMaxNoOfRec[0];
	bCmdBuff[wCmdLen++] = pMaxNoOfRec[1];
	bCmdBuff[wCmdLen++] = pMaxNoOfRec[2];

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_CreateCyclicRecordFile(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t  bFileNo, uint8_t  *pIsoFileId, uint8_t bCommSett,
	uint8_t * pAccessRights, uint8_t * pRecordSize, uint8_t * pMaxNoOfRec)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if ((bFileNo > 0x1f) || (bOption > 0x01))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommSett != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_CREATE_CYCLIC_RECFILE;

	/* Copy the value information supplied by the user */
	bCmdBuff[wCmdLen++] = bFileNo;

	if (bOption == 1)
	{
		bCmdBuff[wCmdLen++] = pIsoFileId[0];
		bCmdBuff[wCmdLen++] = pIsoFileId[1];
	}
	bCmdBuff[wCmdLen++] = bCommSett >> 4;
	bCmdBuff[wCmdLen++] = pAccessRights[0];
	bCmdBuff[wCmdLen++] = pAccessRights[1];

	bCmdBuff[wCmdLen++] = pRecordSize[0];
	bCmdBuff[wCmdLen++] = pRecordSize[1];
	bCmdBuff[wCmdLen++] = pRecordSize[2];

	bCmdBuff[wCmdLen++] = pMaxNoOfRec[0];
	bCmdBuff[wCmdLen++] = pMaxNoOfRec[1];
	bCmdBuff[wCmdLen++] = pMaxNoOfRec[2];

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_DeleteFile(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bFileNo)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[10];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_DELETE_FILE;
	bCmdBuff[wCmdLen++] = bFileNo;

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}




/* MIFARE DESFire Data Manipulation commands. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfdf_Sw_ReadData(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppRxdata,
	uint16_t * pRxdataLen)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint16_t    PH_MEMLOC_REM wDataLen = 0;
	phStatus_t  PH_MEMLOC_REM status = 0;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
	{
		bCmdBuff[wCmdLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;
	}
	else if ((bOption & 0x0FU) == PH_EXCHANGE_DEFAULT)
	{
		bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_READ_DATA;
		bCmdBuff[wCmdLen++] = bFileNo;
		memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
		wCmdLen += 3;
		memcpy(&bCmdBuff[wCmdLen], pLength, 3); /* PRQA S 3200 */
		wCmdLen += 3;

		wDataLen = (uint16_t) pLength[1]; /* MSB */
		wDataLen <<= 8;
		wDataLen |= pLength[0]; /* LSB */
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

		if (
		((bOption & 0xF0U) == PHAL_MFDF_COMMUNICATION_PLAIN))
	{
		status = phalMfdf_Sw_Int_ReadData_Plain(
			pDataParams,
			bOption,
			bCmdBuff,
			wCmdLen,
			ppRxdata,
			pRxdataLen
		);
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((status == PH_ERR_SUCCESS) && (bCmdBuff[0] != PHAL_MFDF_RESP_ADDITIONAL_FRAME))
	{
		if ((wDataLen != *pRxdataLen) && (wDataLen != 0))
		{
			/* Reset authentication status */
			if ((pDataParams->bAuthMode == PHAL_MFDF_AUTHENTICATEISO) ||
				(pDataParams->bAuthMode == PHAL_MFDF_AUTHENTICATEAES))
			{
				phalMfdf_Sw_Int_ResetAuthStatus(pDataParams);
			}
			return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
		}
	}
	return status;
}

phStatus_t phalMfdf_Sw_WriteData(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[16];
	uint16_t    PH_MEMLOC_REM wDataLen;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommOption != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_WRITE_DATA;
	bCmdBuff[wCmdLen++] = bFileNo;
	memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
	wCmdLen += 3;
	memcpy(&bCmdBuff[wCmdLen], pDataLen, 3); /* PRQA S 3200 */
	wCmdLen += 3;

	/* Assuming here that the size can never go beyond FFFF.
	In fact it can never go beyond 8092 (1F9C) bytes */
	wDataLen = (uint16_t) pDataLen[1];
	wDataLen = wDataLen << 8;
	wDataLen |= pDataLen[0];

	{
		return phalMfdf_Sw_Int_Write_Plain(
			pDataParams,
			bCmdBuff,
			wCmdLen,
			bCommOption,
			pData,
			wDataLen
		);
	}
}

phStatus_t phalMfdf_Sw_GetValue(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{

	/* If not authenticated, send the data and get the value in plain.
	Else use the mode dictated by the caller of this API
	*/
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wRxlen = 0;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t     PH_MEMLOC_REM *pRecv = NULL;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommOption != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_GET_VALUE;
	bCmdBuff[wCmdLen++] = bFileNo;

	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Sw_Int_ReadData_Plain(
			pDataParams,
			bCommOption,
			bCmdBuff,
			wCmdLen,
			&pRecv,
			&wRxlen
		));
	}

	if (wRxlen != 4)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}
	memcpy(pValue, pRecv, wRxlen); /* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sw_Credit(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	phStatus_t status = 0;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommOption != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_CREDIT;
	bCmdBuff[wCmdLen++] = bFileNo;

	memcpy(&bCmdBuff[wCmdLen], pValue, 4); /* PRQA S 3200 */
	wCmdLen += 4;

	{
		/* COMMUNICATION IS PLAIN */
		status = phalMfdf_Sw_Int_Write_Plain(pDataParams,
			bCmdBuff,
			0x0002,
			bCommOption,
			&bCmdBuff[2],
			0x0004
		);
	}

	return status;
}

phStatus_t phalMfdf_Sw_Debit(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	phStatus_t status = 0;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommOption != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_DEBIT;
	bCmdBuff[wCmdLen++] = bFileNo;

	memcpy(&bCmdBuff[wCmdLen], pValue, 4); /* PRQA S 3200 */
	wCmdLen += 4;

	{
		/* COMMUNICATION IS PLAIN */
		status = phalMfdf_Sw_Int_Write_Plain(pDataParams,
			bCmdBuff,
			0x0002,
			bCommOption,
			&bCmdBuff[2],
			0x0004
		);
	}
	return status;
}

phStatus_t phalMfdf_Sw_LimitedCredit(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	phStatus_t status = 0;
	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommOption != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_LIMITED_CREDIT;
	bCmdBuff[wCmdLen++] = bFileNo;

	memcpy(&bCmdBuff[wCmdLen], pValue, 4); /* PRQA S 3200 */
	wCmdLen += 4;

	{
		/* COMMUNICATION IS PLAIN */
		status = phalMfdf_Sw_Int_Write_Plain(pDataParams,
			bCmdBuff,
			0x0002,
			bCommOption,
			&bCmdBuff[2],
			0x0004
		);
	}
	return status;
}

phStatus_t phalMfdf_Sw_WriteRecord(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData,
	uint8_t * pDataLen)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[16];
	uint16_t    PH_MEMLOC_REM wDataLen;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	phStatus_t status = 0;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommOption != PHAL_MFDF_COMMUNICATION_PLAIN)
		)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_WRITE_RECORD;
	bCmdBuff[wCmdLen++] = bFileNo;
	memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
	wCmdLen += 3;
	memcpy(&bCmdBuff[wCmdLen], pDataLen, 3); /* PRQA S 3200 */
	wCmdLen += 3;

	/* Assuming here that the size can never go beyond FFFF. In fact it can never go beyond 8092 (1F9C) bytes */
	wDataLen = (uint16_t) pDataLen[1];
	wDataLen = wDataLen << 8;
	wDataLen |= pDataLen[0];

	{
		status = phalMfdf_Sw_Int_Write_Plain(
			pDataParams,
			bCmdBuff,
			wCmdLen,
			bCommOption,
			pData,
			wDataLen
		);
	}

	return status;
}

phStatus_t phalMfdf_Sw_ReadRecords(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pNumRec,
	uint8_t * pRecSize, uint8_t ** ppRxdata, uint16_t * pRxdataLen)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint16_t    PH_MEMLOC_REM wRecLen = 0;
	uint16_t    PH_MEMLOC_REM wNumRec;
	phStatus_t  PH_MEMLOC_REM status = 0;

	/* Should also handle the scenario where 0xAF is returned by
	the PICC */
	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if ((bCommOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
	{
		bCmdBuff[wCmdLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;
	}
	else if ((bCommOption & 0x0FU) == PH_EXCHANGE_DEFAULT)
	{
		/* form the command */
		bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_READ_RECORDS;
		bCmdBuff[wCmdLen++] = bFileNo;

		/* Offset */
		memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
		wCmdLen += 3;

		/* Length */
		memcpy(&bCmdBuff[wCmdLen], pNumRec, 3); /* PRQA S 3200 */
		wCmdLen += 3;

		wRecLen = (uint16_t) pRecSize[1]; /* MSB */
		wRecLen <<= 8;
		wRecLen |= pRecSize[0]; /* LSB */

		wNumRec = (uint16_t) pNumRec[1]; /* MSB */
		wNumRec <<= 8;
		wNumRec |= pNumRec[0]; /* LSB */
		/* Total number of bytes to read */
		wRecLen = (uint16_t) wRecLen * wNumRec;
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

		if (
		((bCommOption & 0xF0U) == PHAL_MFDF_COMMUNICATION_PLAIN))
	{
		status = phalMfdf_Sw_Int_ReadData_Plain(
			pDataParams,
			bCommOption,
			bCmdBuff,
			wCmdLen,
			ppRxdata,
			pRxdataLen
		);
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}
	if ((status == PH_ERR_SUCCESS) && (bCmdBuff[0] != PHAL_MFDF_RESP_ADDITIONAL_FRAME))
	{
		/* Can check this case if user has not given num of records
		as 0x000000. If 0x000000, then all records are read */
		if ((wRecLen != *pRxdataLen) && (wRecLen != 0))
		{
			/* Reset authentication status */
			if ((pDataParams->bAuthMode == PHAL_MFDF_AUTHENTICATEISO) ||
				(pDataParams->bAuthMode == PHAL_MFDF_AUTHENTICATEAES))
			{
				phalMfdf_Sw_Int_ResetAuthStatus(pDataParams);
			}
			return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
		}
	}
	return status;
}

phStatus_t phalMfdf_Sw_ClearRecordFile(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bFileNo)
{
	uint8_t PH_MEMLOC_REM bCmdBuff[20];
	uint16_t PH_MEMLOC_REM wCmdLen = 0;

	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_CLEAR_RECORDS_FILE;
	bCmdBuff[wCmdLen++] = bFileNo;

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_CommitTransaction(phalMfdf_Sw_DataParams_t * pDataParams)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_COMMIT_TXN;

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfdf_Sw_AbortTransaction(phalMfdf_Sw_DataParams_t * pDataParams)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_ABORT_TXN;

	/* COMMUNICATION IS PLAIN */
	return phalMfdf_Sw_Int_Write_Plain(pDataParams,
		bCmdBuff,
		wCmdLen,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}




/* MIFARE DESFire ISO7816 commands. -------------------------------------------------------------------------------------------------- */
phStatus_t phalMfdf_Sw_IsoSelectFile(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname,
	uint8_t bDFnameLen, uint8_t ** ppRecv, uint16_t * pwRxlen)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bData[24];
	uint8_t     PH_MEMLOC_REM bLc = 0;

	if (bDFnameLen > 16)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}
	if ((bOption != 0x00) && (bOption != 0x0C))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	switch (bSelector)
	{
		case 0x00:
		case 0x02:
			/* Selection by EF Id*/
			/* Send MSB first to card */
			bData[0] = pFid[1];
			bData[1] = pFid[0];
			bLc = 2;
			break;

		case 0x04:
			/* Selection by DF Name */
			memcpy(bData, pDFname, bDFnameLen); /* PRQA S 3200 */
			bLc = bDFnameLen;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}


	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Int_Send7816Apdu(
		pDataParams,
		pDataParams->pPalMifareDataParams,
		0x03,
		PHAL_MFDF_CMD_ISO7816_SELECT_FILE,
		bSelector,
		bOption,
		bLc,
		bData,
		0x00,
		ppRecv,
		pwRxlen));

	/* ISO wrapped mode is on */
	pDataParams->bWrappedMode = 1;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sw_IsoReadBinary(phalMfdf_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid, uint8_t bBytesToRead,
	uint8_t ** ppRxBuffer, uint16_t * pBytesRead)
{
	uint8_t     PH_MEMLOC_REM bP1 = 0;
	uint8_t     PH_MEMLOC_REM bP2 = 0;
	uint8_t     PH_MEMLOC_REM bLe = 0;
	uint8_t     PH_MEMLOC_REM bCmdBuff[8];
	uint16_t    PH_MEMLOC_REM wOffset;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if (wOption == PH_EXCHANGE_DEFAULT)
	{
		if (bSfid & 0x80)
		{
			/* Short file id is supplied */
			if ((bSfid & 0x7FU) > 0x1F)
			{
				/* Error condition */
				return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
			}
			bP1 = bSfid;
			bP2 = bOffset;
		}
		else
		{
			/* P1 and P2 code the offset */
			wOffset = bP1 = bSfid;
			wOffset <<= 8; /* Left shift */
			wOffset |= bOffset;
			bP2 = bOffset;
		}
		bLe = bBytesToRead;
		bCmdBuff[wCmdLen++] = 0x00; /* Class */
		bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_ISO7816_READ_BINARY; /* Ins */
		bCmdBuff[wCmdLen++] = bP1;
		bCmdBuff[wCmdLen++] = bP2;
		bCmdBuff[wCmdLen++] = bBytesToRead; /* Le */
	}
	else if (wOption == PH_EXCHANGE_RXCHAINING)
	{
		wCmdLen = 0;
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	return phalMfdf_Sw_Int_IsoRead(
		pDataParams,
		wOption,
		bCmdBuff,
		wCmdLen,
		ppRxBuffer,
		pBytesRead
	);
}

phStatus_t phalMfdf_Sw_IsoUpdateBinary(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t * pData, uint8_t bDataLen)
{
	uint8_t     PH_MEMLOC_REM bP1 = 0;
	uint8_t     PH_MEMLOC_REM bP2 = 0;
	uint8_t     PH_MEMLOC_REM bLc = 0;
	uint16_t    PH_MEMLOC_REM wOffset;
	phStatus_t  PH_MEMLOC_REM status;

	if (bSfid & 0x80)
	{
		/* Short file id is supplied */
		if ((bSfid & 0x7FU) > 0x1F)
		{
			/* Error condition */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
		}
		bP2 = bOffset;
	}
	else
	{
		/* P1 and P2 code the offset */
		wOffset = bP1 = bSfid;
		wOffset <<= 8; /* Left shift */
		wOffset |= bOffset;
		bP2 = bOffset;
	}

	bLc = bDataLen;
	status = phalMfdf_Int_Send7816Apdu(
		pDataParams,
		pDataParams->pPalMifareDataParams,
		0x01,
		PHAL_MFDF_CMD_ISO7816_UPDATE_BINARY,
		bP1,
		bP2,
		bLc,
		pData,
		0x00,
		NULL,
		NULL);


	return status;
}

phStatus_t phalMfdf_Sw_IsoReadRecords(phalMfdf_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bRecNo, uint8_t bReadAllFromP1, uint8_t bSfid,
	uint8_t bBytesToRead, uint8_t ** ppRxBuffer, uint16_t * pBytesRead)
{
	uint8_t     PH_MEMLOC_REM bP1 = 0;
	uint8_t     PH_MEMLOC_REM bP2 = 0;
	uint8_t     PH_MEMLOC_REM bLe = 0;
	uint8_t     PH_MEMLOC_REM bCmdBuff[8];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	if (wOption == PH_EXCHANGE_DEFAULT)
	{
		if (bSfid > 0x1F)
		{
			/* Invalid Short File Id */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
		}
		else
		{
			/* Valid Sfid */
			bP1 = bRecNo;
			bP2 = bSfid;
			bP2 <<= 3; /* left shift by 3 bits to move SFID to bits 7 to 3 */
			if (bReadAllFromP1)
			{
				bP2 |= 0x05; /* Last three bits of P2 = 101 */
			}
			else
			{
				bP2 |= 0x04; /* Last three bits of P2 = 100. Read only record P1 */
			}
		}
		bLe = bBytesToRead;
		bCmdBuff[wCmdLen++] = 0x00; /* Class */
		bCmdBuff[wCmdLen++] = PHAL_MFDF_CMD_ISO7816_READ_RECORDS; /* Ins */
		bCmdBuff[wCmdLen++] = bP1;
		bCmdBuff[wCmdLen++] = bP2;
		bCmdBuff[wCmdLen++] = bBytesToRead; /* Le */
	}
	else if (wOption == PH_EXCHANGE_RXCHAINING)
	{
		wCmdLen = 0;
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	return phalMfdf_Sw_Int_IsoRead(
		pDataParams,
		wOption,
		bCmdBuff,
		wCmdLen,
		ppRxBuffer,
		pBytesRead
	);
}

phStatus_t phalMfdf_Sw_IsoAppendRecord(phalMfdf_Sw_DataParams_t * pDataParams, uint8_t bSfid, uint8_t * pData, uint8_t bDataLen)
{
	uint8_t     PH_MEMLOC_REM bP1 = 0;
	uint8_t     PH_MEMLOC_REM bP2 = 0;
	uint8_t     PH_MEMLOC_REM bLc = 0;
	phStatus_t  PH_MEMLOC_REM status;

	if (bSfid > 0x1F)
	{
		/* Invalid Short File Id */
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	bP2 = bSfid;
	bP2 <<= 3; /* left shift by 3 bits to move SFID to bits 7 to 3 */
	/* Last three bits of P2 = 000 */

	bLc = bDataLen;
	status = phalMfdf_Int_Send7816Apdu(
		pDataParams,
		pDataParams->pPalMifareDataParams,
		0x01,
		PHAL_MFDF_CMD_ISO7816_APPEND_RECORD,
		bP1,
		bP2,
		bLc,
		pData,
		0x00,
		NULL,
		NULL);
	if (status != PH_ERR_SUCCESS)
	{
		/* Reset authentication status */
		phalMfdf_Sw_Int_ResetAuthStatus(pDataParams);
	}
	return status;
}

phStatus_t phalMfdf_Sw_IsoGetChallenge(phalMfdf_Sw_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bLe, uint8_t * pRPICC1)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint16_t    PH_MEMLOC_REM wRxlen = 0;
	uint8_t     PH_MEMLOC_REM *pRecv;
	uint8_t     PH_MEMLOC_REM bData[24];
	uint16_t    PH_MEMLOC_REM wKeyType;

	PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
		pDataParams->pKeyStoreDataParams,
		wKeyNo,
		wKeyVer,
		sizeof(bData),
		bData,
		&wKeyType));

	if ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) || (wKeyType == PH_KEYSTORE_KEY_TYPE_3K3DES))
	{
		if (bLe != 0x10)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
		}
	}
	else if ((wKeyType == PH_KEYSTORE_KEY_TYPE_DES) || (wKeyType == PH_KEYSTORE_KEY_TYPE_2K3DES))
	{
		if (bLe != 0x08)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
		}
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfdf_Int_Send7816Apdu(
		pDataParams,
		pDataParams->pPalMifareDataParams,
		0x02,
		PHAL_MFDF_CMD_ISO7816_GET_CHALLENGE,
		0x00,
		0x00,
		0x00,
		NULL,
		bLe,
		&pRecv,
		&wRxlen));

	if (wRxlen != bLe)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}

	memcpy(pRPICC1, pRecv, wRxlen); /* PRQA S 3200 */

	/* Reset authentication status */
	phalMfdf_Sw_Int_ResetAuthStatus(pDataParams);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}


phStatus_t phalMfdf_Sw_GetConfig(phalMfdf_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
	switch (wConfig)
	{
		case PHAL_MFDF_ADDITIONAL_INFO:
			*pValue = pDataParams->wAdditionalInfo;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDF);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sw_SetConfig(phalMfdf_Sw_DataParams_t *pDataParams, uint16_t wConfig, uint16_t wValue)
{
	switch (wConfig)
	{
		case PHAL_MFDF_ADDITIONAL_INFO:
			pDataParams->wAdditionalInfo = wValue;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDF);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sw_ResetAuthStatus(phalMfdf_Sw_DataParams_t *pDataParams)
{
	phalMfdf_Sw_Int_ResetAuthStatus(pDataParams);
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

#endif /* NXPBUILD__PHAL_MFDF_SW */
