/*
 * Copyright 2013, 2016 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */


/** \file
 * Software MIFARE(R) Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phalMfp.h>
#include <phpalMifare.h>

#ifdef NXPBUILD__PH_CRYPTOSYM
#include <phCryptoSym.h>
#endif /* NXPBUILD__PH_CRYPTOSYM */

#ifdef NXPBUILD__PH_CRYPTORNG
#include <phCryptoRng.h>
#endif /* NXPBUILD__PH_CRYPTORNG */

#include <phKeyStore.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_MFP_SW

#include "../phalMfp_Int.h"
#include "phalMfp_Sw.h"
#include "phalMfp_Sw_Int.h"


phStatus_t phalMfp_Sw_Init(phalMfp_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pPalMifareDataParams, void * pKeyStoreDataParams, void * pCryptoDataParamsEnc,
	void * pCryptoDataParamsMac, void * pCryptoRngDataParams, void * pCryptoDiversifyDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus;

    /* data param check */
    if (sizeof(phalMfp_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFP);
    }

    PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFP);
    PH_ASSERT_NULL_PARAM (pPalMifareDataParams, PH_COMP_AL_MFP);


    /* Initialize the data parameters. */
    pDataParams->wId                    = PH_COMP_AL_MFP | PHAL_MFP_SW_ID;
    pDataParams->pPalMifareDataParams   = pPalMifareDataParams;
    pDataParams->pKeyStoreDataParams    = pKeyStoreDataParams;
    pDataParams->pCryptoDataParamsEnc   = pCryptoDataParamsEnc;
    pDataParams->pCryptoDataParamsMac   = pCryptoDataParamsMac;
    pDataParams->pCryptoRngDataParams   = pCryptoRngDataParams;
    pDataParams->pCryptoDiversifyDataParams  = pCryptoDiversifyDataParams;
    memset(pDataParams->bKeyModifier, 0x00, PHAL_MFP_SIZE_KEYMODIFIER); /* PRQA S 3200 */

    /* clear the secure messaging state */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sw_ResetSecMsgState(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}



/****************************************************************************************************************************************/
/* Mifare Plus command for personalization.																								*/
/****************************************************************************************************************************************/
phStatus_t phalMfp_Sw_WritePerso(phalMfp_Sw_DataParams_t * pDataParams, uint8_t bLayer4Comm, uint16_t wBlockNr, uint8_t * pValue)
{
    return phalMfp_Int_WritePerso(pDataParams->pPalMifareDataParams, bLayer4Comm, wBlockNr, pValue);
}

phStatus_t phalMfp_Sw_CommitPerso(phalMfp_Sw_DataParams_t * pDataParams, uint8_t bLayer4Comm)
{
    return phalMfp_Int_CommitPerso(pDataParams->pPalMifareDataParams, bLayer4Comm);
}



/****************************************************************************************************************************************/
/* Mifare Plus command for authentication.																								*/
/****************************************************************************************************************************************/
phStatus_t phalMfp_Sw_AuthenticateClassicSL2(phalMfp_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint16_t wKeyNumber,
	uint16_t wKeyVersion, uint8_t * pUid, uint8_t bUidLength)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aKey[PH_KEYSTORE_SW_MAX_KEY_SIZE];
    uint8_t *   PH_MEMLOC_REM pKey = NULL;
    uint16_t    PH_MEMLOC_REM bKeystoreKeyType = 0;
    uint8_t     PH_MEMLOC_REM bValue = 0;

    /* Bail out if we haven't got a keystore */
    if (pDataParams->pKeyStoreDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_HAL);
    }

    /* Retrieve KeyA & KeyB from keystore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNumber,
        wKeyVersion,
		PH_KEYSTORE_KEY_TYPE_AES256_SIZE,
        aKey,
        &bKeystoreKeyType));

    /* Check key type */
    if (bKeystoreKeyType != PH_KEYSTORE_KEY_TYPE_MIFARE)
    {
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_HAL);
    }

    /* Evaluate which key to use */
    if ((bKeyType & 0x7F) == PHHAL_HW_MFC_KEYA)
    {
        /* Use KeyA */
        pKey = aKey;
    }
    else if ((bKeyType & 0x7F) == PHHAL_HW_MFC_KEYB)
    {
        /* Use KeyB */
        pKey = &aKey[PHHAL_HW_MFC_KEY_LENGTH];
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Apply the key modifier if requested */
    for (bValue = 0; bValue < PHHAL_HW_MFC_KEY_LENGTH; bValue++)
    {
        pKey[bValue] ^= pDataParams->bKeyModifier[bValue];
    }

    return phpalMifare_MfcAuthenticate(
        pDataParams->pPalMifareDataParams,
        bBlockNo,
        bKeyType,
        pKey,
        &pUid[bUidLength - 4]);
}




/****************************************************************************************************************************************/
/* Mifare Plus command for data operations.																								*/
/****************************************************************************************************************************************/
phStatus_t phalMfp_Sw_MultiBlockWrite(phalMfp_Sw_DataParams_t * pDataParams, uint8_t bBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks)
{
    return phalMfp_Int_MultiBlockWrite(pDataParams->pPalMifareDataParams, bBlockNr, bNumBlocks, pBlocks);
}

phStatus_t phalMfp_Sw_MultiBlockRead(phalMfp_Sw_DataParams_t * pDataParams, uint8_t bBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks)
{
    return phalMfp_Int_MultiBlockRead(pDataParams->pPalMifareDataParams, bBlockNr, bNumBlocks, pBlocks);
}







/****************************************************************************************************************************************/
/* Mifare Plus command for special operations.																							*/
/****************************************************************************************************************************************/
phStatus_t phalMfp_Sw_ResetAuth(phalMfp_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM aCmd[1 /* command code */];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRxLength;

    /* command code */
    aCmd[0] = PHAL_MFP_CMD_RAUTH;

    /* command exchange */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        1,
        &pResponse,
        &wRxLength));

    /* check response */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(wRxLength, pResponse[0], 1));

    /* Reset the crypto layer */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_ResetSecMsgState(pDataParams));

    /* return exchange wStatus1 code */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sw_ResetSecMsgState(phalMfp_Sw_DataParams_t * pDataParams)
{
    pDataParams->wRCtr                          = 0;
    pDataParams->wWCtr                          = 0;
    pDataParams->bNumUnprocessedReadMacBytes    = 0;
    pDataParams->bFirstRead                     = 1;
    memset(pDataParams->bTi, 0x00, PHAL_MFP_SIZE_TI); /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sw_Cmd_PrepareProximityCheck(phalMfp_Sw_DataParams_t * pDataParams)
{
    return(phalMfp_Int_PrepareProximityCheck(pDataParams->pPalMifareDataParams));
}


#endif /* NXPBUILD__PHAL_MFP_SW */
