/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions for RD710Sam BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_RD710SAM

#include "phbalReg_Rd710Sam_Int.h"

phStatus_t phbalReg_Rd710Sam_CheckResponse(
    uint16_t wCmd,
    uint8_t * pRxBuffer,
    uint16_t wRxLength,
    uint8_t ** ppData,
    uint16_t * pDataLength
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint16_t    PH_MEMLOC_REM wDataLengthTmp;

    /* Reset data length */
    if (ppData != NULL)
    {
        *ppData = NULL;
    }
    if (pDataLength != NULL)
    {
        *pDataLength = 0;
    }

    /* Frame has to be at least 6 bytes */
    if (wRxLength < PHBAL_RD710SAM_FRAME_HEADER_LEN)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_BAL);
    }
    else
    {
        /* Retrieve length */
        wDataLengthTmp = pRxBuffer[PHBAL_RD710SAM_FRAME_LEN_POS];
        wDataLengthTmp |= ((uint16_t)pRxBuffer[PHBAL_RD710SAM_FRAME_LEN_POS+1] << 8);

        /* Length should match */
        if ((wRxLength - PHBAL_RD710SAM_FRAME_HEADER_LEN) != wDataLengthTmp)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_BAL);
        }
        else
        {
            /* Command Code should match */
            if ((pRxBuffer[PHBAL_RD710SAM_FRAME_CMD_POS] != (uint8_t)((uint16_t)wCmd >> 8)) ||
                (pRxBuffer[PHBAL_RD710SAM_FRAME_CMD_POS+1] != ((uint8_t)wCmd | 0x80)))
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_BAL);
            }
        }
    }

    /* Retrieve status */
    status = PH_ADD_COMPCODE(
        (uint16_t)pRxBuffer[PHBAL_RD710SAM_FRAME_STATUS_POS],
        ((uint16_t)pRxBuffer[PHBAL_RD710SAM_FRAME_STATUS_POS+1] << 8));

    /* Return data pointer */
    if (ppData != NULL)
    {
        *ppData = &pRxBuffer[PHBAL_RD710SAM_FRAME_PAYLOAD_POS];
    }
    /* Return data length  */
    if (pDataLength != NULL)
    {
        *pDataLength = wDataLengthTmp;
    }

    return status;
}

phStatus_t phbalReg_Rd710Sam_ParseAtr(
                                      uint8_t * pAtr,
                                      uint16_t wAtrLenght,
                                      uint8_t * pTa1,
                                      uint8_t * pSpecificMode
                                      )
{
    uint16_t PH_MEMLOC_REM wY;
    uint16_t PH_MEMLOC_REM wIndex = 0;

    /* Length Check */
    if (wAtrLenght < 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_BAL);
    }

    /* Skip TS */
    ++wIndex;

    /* Retrieve Y1 */
    wY = pAtr[wIndex++];

    /* Parse TA1 (Fi and Di) */
    if (wY & 0x10)
    {
        *pTa1 = pAtr[wIndex++];
    }
    /* Default Di and Di */
    else
    {
        *pTa1 = 0x11;
    }

    /* Ignore TB1 */
    if (wY & 0x20)
    {
        ++wIndex;
    }

    /* Ignore TC1 */
    if (wY & 0x40)
    {
        ++wIndex;
    }

    /* Parse TD1 */
    if (wY & 0x80)
    {
        /* Retrieve Y2 */
        wY = pAtr[wIndex++];

        /* Parse TA2 */
        if (wY & 0x10)
        {
            *pSpecificMode = (uint8_t)((pAtr[wIndex] & 0x80) ? 1 : 0);
        }
        else
        {
            *pSpecificMode = 0;
        }
    }
    else
    {
        *pSpecificMode = 0;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

#endif /* NXPBUILD__PHBAL_REG_RD710SAM */
