/*
 * Copyright 2013, 2015 - 2017, 2019, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/*
 * RdCardSim specific HAL-Component of Reader Library Framework.
 * $RCSfile $
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#include <phKeyStore.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_RDCARDSIM

#include "phhalHw_RdCardSim.h"
#include "phhalHw_RdCardSim_Int.h"
#include <phhalHw_RdCardSim_Cmd.h>

/* Default shadow for ISO14443-3A Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wRdCardSim_DefaultShadow_I14443a[][2] =
{
    {PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1,    PH_ON},
    {PHHAL_HW_CONFIG_PARITY,                PH_ON},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_RDCARDSIM_DEFAULT_TIMEOUT},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0x0000},
};

phStatus_t phhalHw_RdCardSim_Init(
                                  phhalHw_RdCardSim_DataParams_t * pDataParams,
                                  uint16_t wSizeOfDataParams,
                                  void    * pBalDataParams,
                                  void    * pKeyStoreDataParams,
                                  uint8_t * pTxBuffer,
                                  uint16_t  wTxBufSize,
                                  uint8_t * pRxBuffer,
                                  uint16_t  wRxBufSize,
                                  uint8_t * pIntBuffer,
                                  uint16_t  wIntBufferLen
                                  )
{
    phStatus_t  status;
    uint16_t wSize_RdCardSim_DataParams = sizeof(phhalHw_RdCardSim_DataParams_t);

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_Init");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSizeOfDataParams);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSize_RdCardSim_DataParams);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxBufSize);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRxBufSize);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIntBufferLen);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSizeOfDataParams_log, &wSizeOfDataParams);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSize_RdCardSim_DataParams_log, &wSize_RdCardSim_DataParams);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wTxBufSize_log, &wTxBufSize);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRxBufSize_log, &wRxBufSize);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wIntBufferLen_log, &wIntBufferLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    if (sizeof(phhalHw_RdCardSim_DataParams_t) != wSizeOfDataParams)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        status = PH_ERR_INVALID_DATA_PARAMS;
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pRxBuffer);
    PH_ASSERT_NULL (pIntBuffer);

    /* Init. private data */
    pDataParams->wId                  = PH_COMP_HAL | PHHAL_HW_RDCARDSIM_ID;
    pDataParams->pBalDataParams       = pBalDataParams;
    pDataParams->pKeyStoreDataParams  = pKeyStoreDataParams;
    pDataParams->pTxBuffer            = pTxBuffer;
    pDataParams->wTxBufSize           = wTxBufSize;
    pDataParams->wTxBufLen            = 0;
    pDataParams->bTxLastBits          = 0;
    pDataParams->pRxBuffer            = pRxBuffer;
    pDataParams->wRxBufSize           = wRxBufSize;
    pDataParams->wRxBufLen            = 0;
    pDataParams->wRxBufStartPos       = 0;
    pDataParams->bRxAlign             = 0;
    pDataParams->wAdditionalInfo      = 0;
    pDataParams->bCardType            = PHHAL_HW_CARDTYPE_ISO14443A;

    pDataParams->wRx2TxWait           = PHHAL_HW_RDCARDSIM_RX2TXWAIT_DEFAULT;
    pDataParams->bTxBAUD              = PHHAL_HW_RF_DATARATE_106;
    pDataParams->bRxBAUD              = PHHAL_HW_RF_DATARATE_106;
	pDataParams->bTxCrc               = 0;
    pDataParams->bRxCrc               = 0;
    pDataParams->bParity              = 1;
    pDataParams->dwResetWaitTime      = PHHAL_HW_RDCARDSIM_FIELD_OFF_DEFAULT;
    pDataParams->dwResetRecoveryTime  = PHHAL_HW_RDCARDSIM_FIELD_RECOVERY_DEFAULT;

    pDataParams->pIntBuffer           = pIntBuffer;
    pDataParams->wIntBufferLen        = wIntBufferLen;
    pDataParams->wIntBufferAlloc      = 0x00;
    memset(pDataParams->pIntBuffer, 0x00, wIntBufferLen); /* PRQA S 3200 */

    pDataParams->wTimingMode          = PHHAL_HW_TIMING_MODE_OFF;
    pDataParams->bTimeoutUnit         = PHHAL_HW_TIME_MICROSECONDS;
    pDataParams->dwTimingUs           = 0;

    /* test pad communication setup */
    pDataParams->bTPactive            = 0;
    pDataParams->wTimeT1              = 0;
    pDataParams->wTimeT2              = 0;
    pDataParams->wTimeT3              = 0;
    pDataParams->wTimeT4              = 0;
    pDataParams->wSyncOption          = 0;
    pDataParams->bEnvEnable           = 0;

    pDataParams->bC40ModeEnabled      = PH_OFF;

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    /* Verify exchange buffers */
    if ((wRxBufSize == 0) || (wIntBufferLen == 0))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_SetConfig(
                                       phhalHw_RdCardSim_DataParams_t * pDataParams,
                                       uint16_t  wConfig,
                                       uint16_t  wValue
                                       )
{
    phStatus_t  status, status2;
    uint8_t *   pTmpChar;


    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_SetConfig");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValue);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status2);
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(statusTmp);*/

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wConfig_log, &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wValue_log, &wValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    status  = PH_ERR_SUCCESS;
    status2 = PH_ERR_SUCCESS;

    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        if (pDataParams->bC40ModeEnabled == PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(status, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTmpChar));
            pDataParams->bParity  = (uint8_t)wValue;
            sprintf( (char*)pTmpChar, "CL_Parity %s", phhalHw_RdCardSim_ParityStr(pDataParams->bParity)); /* PRQA S 3200 */
            status = phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
            status2 = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_PARITY, (uint8_t)wValue);
            PH_CHECK_SUCCESS_FCT(status2, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTmpChar));
        }
        else
        {
            pDataParams->bParity  = (uint8_t)wValue;
            status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_PARITY, (uint8_t)wValue);
        }
        break;
    case PHHAL_HW_CONFIG_TXCRC:
        pDataParams->bTxCrc  = (uint8_t)wValue;
        status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_TX_CRC, (uint8_t)wValue);
        break;
    case PHHAL_HW_CONFIG_RXCRC:
        pDataParams->bRxCrc  = (uint8_t)wValue;
        status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_CRC, (uint8_t)wValue);
        break;
    case PHHAL_HW_CONFIG_TXLASTBITS:
        pDataParams->bTxLastBits = (uint8_t)wValue;
        break;
    case PHHAL_HW_CONFIG_RXLASTBITS:
        pDataParams->wAdditionalInfo = (uint8_t)wValue;
        break;
    case PHHAL_HW_CONFIG_RXALIGN:
        pDataParams->bRxAlign = (uint8_t)wValue;
        status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_OFFSET, pDataParams->bRxAlign);
        break;
    case PHHAL_HW_CONFIG_TXDATARATE:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTmpChar));
        pDataParams->bTxBAUD  = (uint8_t)wValue;
        sprintf( (char*)pTmpChar, "set rdr_context(pcd_picc) %sKBAUD", phhalHw_RdCardSim_BaudRateStr(pDataParams->bTxBAUD));
        status = phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
		if( pDataParams->bTxBAUD != 0 || pDataParams->bRxBAUD != 0)
		{
			sprintf( (char*)pTmpChar, "set rdr_context(fdt_n) 8");
			status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
			sprintf( (char*)pTmpChar, "set rdr_context(fdt_mode) GREATER");
			status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
		}
        status2 = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_TX_BAUDRATE, (uint8_t)wValue);
        PH_CHECK_SUCCESS_FCT(status2, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTmpChar));
        break;
    case PHHAL_HW_CONFIG_RXDATARATE:
        PH_CHECK_SUCCESS_FCT(status, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTmpChar));
        pDataParams->bRxBAUD  = (uint8_t)wValue;
        sprintf( (char*)pTmpChar, "set rdr_context(picc_pcd) %sKBAUD", phhalHw_RdCardSim_BaudRateStr(pDataParams->bRxBAUD));
        status = phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
        sprintf( (char*)pTmpChar, "set rdr_context(scheme) %s", phhalHw_RdCardSim_SchemeStr(pDataParams->bRxBAUD));
        status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
		if( pDataParams->bTxBAUD != 0 || pDataParams->bRxBAUD != 0)
		{
			sprintf( (char*)pTmpChar, "set rdr_context(fdt_n) 8");
			status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
			sprintf( (char*)pTmpChar, "set rdr_context(fdt_mode) GREATER");
			status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
		}
        status2 = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_BAUDRATE, (uint8_t)wValue);
        PH_CHECK_SUCCESS_FCT(status2, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTmpChar));
        break;
    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
        status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_US_TIMEOUT_LO, (uint8_t)(wValue & 0x00FF));
        status2 = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_US_TIMEOUT_HI, (uint8_t)((wValue & 0xFF00) >> 8));
		PH_CHECK_SUCCESS_FCT(status, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTmpChar));
		if( wValue > 300 )
		{
			sprintf( (char*)pTmpChar, "set rdr_context(fdt_n) 8");
			status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
			sprintf( (char*)pTmpChar, "set rdr_context(fdt_mode) GREATER");
			status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
		}
		else
		{
			sprintf( (char*)pTmpChar, "set rdr_context(fdt_n) 9");
			status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
			sprintf( (char*)pTmpChar, "set rdr_context(fdt_mode) EXACT");
			status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
		}
		PH_CHECK_SUCCESS_FCT(status2, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTmpChar));
        break;
    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MILLISECONDS;
        status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_MS_TIMEOUT_LO, (uint8_t)(wValue & 0x00FF));
        status2 = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_MS_TIMEOUT_HI, (uint8_t)((wValue & 0xFF00) >> 8));
		PH_CHECK_SUCCESS_FCT(status, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTmpChar));
		sprintf( (char*)pTmpChar, "set rdr_context(fdt_n) 8");
		status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
		sprintf( (char*)pTmpChar, "set rdr_context(fdt_mode) GREATER");
		status |= phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
		PH_CHECK_SUCCESS_FCT(status2, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTmpChar));
        break;
    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_DIS_MFCRYTO1, (uint8_t)wValue);
        break;
    case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:
        pDataParams->wTxBufSize  = wValue;
        break;
    case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:
        pDataParams->wRxBufSize  = wValue;
        break;
    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
        pDataParams->wAdditionalInfo  = (uint8_t)wValue;
        break;
    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
        pDataParams->wTxBufLen  = wValue;
        break;
    case PHHAL_HW_CONFIG_TXBUFFER:
        if (pDataParams->wAdditionalInfo >= pDataParams->wTxBufLen)
        {
            status = PH_ERR_PARAMETER_OVERFLOW;
        }
        pDataParams->pTxBuffer[pDataParams->wAdditionalInfo] = (uint8_t)wValue;
        break;
    case PHHAL_HW_CONFIG_RXDEAFBITS:

        /* Simulator testbench always receive the response from the card */
    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
        /* Simulator testbench does not detect anticollision */
        status = PH_ERR_SUCCESS;
        break;
    case PHHAL_HW_CONFIG_TIMING_MODE:
        /* Check supported option bits */
        switch (wValue & PHHAL_HW_TIMING_MODE_OPTION_MASK)
        {
        case PHHAL_HW_TIMING_MODE_OPTION_DEFAULT:
        case PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR:
            break;
        default:
            status = PH_ERR_INVALID_PARAMETER;
            break;
        }

        /* Check supported timing modes */
        switch (wValue & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK)
        {
        case PHHAL_HW_TIMING_MODE_OFF:
        case PHHAL_HW_TIMING_MODE_FDT:
            pDataParams->dwTimingUs = 0;
            pDataParams->wTimingMode = wValue;
            break;
        case PHHAL_HW_TIMING_MODE_COMM:
            status2 = PH_ERR_UNSUPPORTED_PARAMETER;
            break;
        default:
            status2 = PH_ERR_INVALID_PARAMETER;
            break;
        }
        break;
    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        pDataParams->dwResetWaitTime = (uint32_t)wValue;
        break;
    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        pDataParams->dwResetRecoveryTime = (uint32_t)wValue;
        break;
	case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        /* Boundary check */
        if (wValue > pDataParams->wRxBufSize)
        {
            status = PH_ERR_PARAMETER_OVERFLOW;
        }
        /* Set start position */
        pDataParams->wRxBufStartPos = wValue;
        break;
    case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
        if (pDataParams->bC40ModeEnabled == PH_OFF)
        {
            status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT, (uint8_t)wValue);
        }
        else
        {
            status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RF_RESET_ON_TIMEOUT, (uint8_t)wValue);
        }
        break;
	case PHHAL_HW_RDCARDSIM_CONFIG_COMMUNICATION:
        pDataParams->bTPactive = (uint8_t)wValue;
		status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_COMMUNICATION_INT, (uint8_t)wValue);
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_TIME_T1:
        pDataParams->wTimeT1 = wValue;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_TIME_T2:
        pDataParams->wTimeT2 = wValue;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_TIME_T3:
        pDataParams->wTimeT3 = wValue;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_TIME_T4:
        pDataParams->wTimeT4 = wValue;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_SYNC_OPTION:
        pDataParams->wSyncOption = wValue;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_ENV_ENABLE:
        if (wValue!= 0)
        {
            pDataParams->bEnvEnable = 1;
        }
        else
        {
            pDataParams->bEnvEnable = 0;
        }
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_FI:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_FI_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_DI:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_DI_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_ETU_CLK:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_ETU_CLK_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_STOPBITS:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_STOPBITS_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_CWI:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_CWI_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_BWI:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_BWI_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_FWI:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_FWI_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_WI:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_WI_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_T0:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_T0_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_T1:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_T1_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_RSTTYPE:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_RSTTYPE_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_CRC_LRC:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_CRC_LRC_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_N:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_N_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_PARITY:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_PARITY_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_INV:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_INV_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_PAROFF:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_PAROFF_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_PARERRCNT:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_PARERRCNT_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_PARERR:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_PARERR_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_SFRAME:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_SFRAME_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_CLKSTOP:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_CLKSTOP_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_CLASS:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_CLASS_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_TB_CLK:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_TB_CLK_INT, (uint8_t)wValue);
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_RBSYNC:
	    status = phhalHw_RdCardSim_WriteRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_RBSYNC_INT, (uint8_t)wValue);
	    break;
	case PHHAL_HW_RDCARDSIM_CONFIG_RX2TXWAIT:
		pDataParams->wRx2TxWait = wValue;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_ENV_C40_MODE:
        if (wValue == PH_OFF)
        {
            pDataParams->bC40ModeEnabled = PH_OFF;
        }
        else if (wValue == PH_ON)
        {
            pDataParams->bC40ModeEnabled = PH_ON;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        break;

    default:
        status = PH_ERR_UNSUPPORTED_PARAMETER;
        break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status2_log, &status2);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    /* Combine the status codes */
    if (status ==  PH_ERR_SUCCESS)
    {
        status = status2;
    }
    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_GetConfig(
                                       phhalHw_RdCardSim_DataParams_t * pDataParams,
                                       uint16_t   wConfig,
                                       uint16_t * pValue
                                       )
{

    phStatus_t  statusTmp;
    uint8_t    bBackup;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_GetConfig");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wConfig_log, &wConfig);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        *pValue = (uint16_t)(pDataParams->bParity);
        break;
    case PHHAL_HW_CONFIG_TXCRC:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister(pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_TX_CRC, &bBackup));
        /* During the merge it was observed that in C40 mode the value is applied after the check, so keep this for legacy */
        if (pDataParams->bC40ModeEnabled == PH_OFF)
        {
            *pValue = (uint16_t)bBackup;
        }
        if (bBackup != pDataParams->bTxCrc)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }
        *pValue = (uint16_t)bBackup;
        break;
    case PHHAL_HW_CONFIG_RXCRC:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister(pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_CRC, &bBackup));
        /* During the merge it was observed that in C40 mode the value is applied after the check, so keep this for legacy */
        if (pDataParams->bC40ModeEnabled == PH_OFF)
        {
            *pValue = (uint16_t)bBackup;
        }
        if (bBackup != pDataParams->bRxCrc)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }
        *pValue = (uint16_t)bBackup;
        break;
    case PHHAL_HW_CONFIG_TXLASTBITS:
        *pValue = (uint16_t)(pDataParams->bTxLastBits);
        break;
    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
    case PHHAL_HW_CONFIG_RXLASTBITS:
        *pValue = (uint16_t)(pDataParams->wAdditionalInfo);
        break;
    case PHHAL_HW_CONFIG_TXDATARATE:
        *pValue = (uint16_t)(pDataParams->bTxBAUD);
        break;
    case PHHAL_HW_CONFIG_RXDATARATE:
        *pValue = (uint16_t)(pDataParams->bRxBAUD);
        break;
    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_US_TIMEOUT_LO, &bBackup));
        *pValue = (uint16_t)bBackup;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_US_TIMEOUT_HI, &bBackup));
        *pValue = ((uint16_t)bBackup << 8) | *pValue;
		if (*pValue == 0)
		{
		  return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
		}
        break;
    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_MS_TIMEOUT_LO, &bBackup));
        *pValue = (uint16_t)bBackup;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_RX_MS_TIMEOUT_HI, &bBackup));
        *pValue = ((uint16_t)bBackup << 8) | *pValue;
        break;
    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_DIS_MFCRYTO1, &bBackup));
        *pValue = (uint16_t)bBackup;
        break;
    case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:
        *pValue = pDataParams->wTxBufSize;
        break;
    case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:
        *pValue = pDataParams->wRxBufSize;
        break;
    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
        *pValue = pDataParams->wTxBufLen;
        break;
    case PHHAL_HW_CONFIG_TXBUFFER:
        if (pDataParams->wAdditionalInfo >= pDataParams->wTxBufLen)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        *pValue = (uint16_t)pDataParams->pTxBuffer[pDataParams->wAdditionalInfo];
        break;
    case PHHAL_HW_CONFIG_TIMING_MODE:
        *pValue = pDataParams->wTimingMode;
        break;
    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        *pValue = (uint16_t)pDataParams->dwResetWaitTime;
        break;
    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        *pValue = (uint16_t)pDataParams->dwResetRecoveryTime;
        break;
    case PHHAL_HW_CONFIG_TIMING_US:
        if (pDataParams->dwTimingUs > 0xFFFF)
        {
            /* pDataParams->dwTimingUs = 0; */
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        else
        {
            *pValue = (uint16_t)pDataParams->dwTimingUs;
            pDataParams->dwTimingUs = 0;
        }
        break;
    case PHHAL_HW_CONFIG_TIMING_MS:
        if (pDataParams->dwTimingUs > (0xFFFF * 1000))
        {
            pDataParams->dwTimingUs = 0;
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        else
        {
            *pValue = (uint16_t)(pDataParams->dwTimingUs / 1000);
            pDataParams->dwTimingUs = 0;
        }
        break;
    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        *pValue = pDataParams->wRxBufStartPos;
        break;
    case PHHAL_HW_CONFIG_CARD_TYPE:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bCardType;
        break;
	  case PHHAL_HW_RDCARDSIM_CONFIG_COMMUNICATION:
        *pValue = (uint16_t)pDataParams->bTPactive;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_TIME_T1:
        *pValue = pDataParams->wTimeT1;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_TIME_T2:
        *pValue = pDataParams->wTimeT2;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_TIME_T3:
        *pValue = pDataParams->wTimeT3;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_TIME_T4:
        *pValue = pDataParams->wTimeT4;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_SYNC_OPTION:
        *pValue = pDataParams->wSyncOption;
        break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_FI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_FI_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_DI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_DI_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_ETU_CLK:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_ETU_CLK_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_STOPBITS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_STOPBITS_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_CWI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_CWI_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_BWI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_BWI_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_FWI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_FWI_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_WI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_WI_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_T0:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_T0_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_T1:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_T1_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_RSTTYPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_RSTTYPE_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_CRC_LRC:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_CRC_LRC_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_N:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_N_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_PARITY:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_PARITY_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_INV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_INV_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_PAROFF:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_PAROFF_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_PARERRCNT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_PARERRCNT_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_PARERR:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_PARERR_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_SFRAME:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_SFRAME_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_CLKSTOP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_CLKSTOP_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_CLASS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_CLASS_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_TB_CLK:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_TB_CLK_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_UT_RBSYNC:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister( pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_UT_RBSYNC_INT, &bBackup));
        *pValue = (uint16_t)bBackup;
	    break;
	case PHHAL_HW_RDCARDSIM_CONFIG_RX2TXWAIT:
		*pValue = pDataParams->wRx2TxWait;
	    break;
    case PHHAL_HW_RDCARDSIM_CONFIG_ENV_C40_MODE:
        *pValue = pDataParams->bC40ModeEnabled;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    statusTmp = PH_ERR_SUCCESS;
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wConfig_log, &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pValue_log, pValue);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &statusTmp);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_FieldOn(
                                     phhalHw_RdCardSim_DataParams_t * pDataParams
                                     )
{
    phStatus_t  status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_FieldOn");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    status = phhalHw_RdCardSim_Command_Int(pDataParams, (uint8_t *)"RST_POR");

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_FieldOff(
                                      phhalHw_RdCardSim_DataParams_t * pDataParams
                                      )
{
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_FieldOff");

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    /* satisfy compiler */
    pDataParams = NULL;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_FieldReset(
                                        phhalHw_RdCardSim_DataParams_t * pDataParams
                                        )
{
    phStatus_t  PH_MEMLOC_REM status = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t *   PH_MEMLOC_REM pTmpChar;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTmpChar));
    if (pDataParams->bC40ModeEnabled == PH_OFF || pDataParams->dwResetWaitTime > 0)
    {
        sprintf((char*)pTmpChar, "CL_Silence %lu CL_CONTINUE", (uint32_t)((pDataParams->dwResetWaitTime)*1000)); /* PRQA S 3200 */
        status = phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
    }
    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        sprintf((char*)pTmpChar, "RST_POR"); /* PRQA S 3200 */
        status = phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            if (pDataParams->bC40ModeEnabled == PH_OFF || pDataParams->dwResetRecoveryTime > 0)
            {
                sprintf((char*)pTmpChar, "CL_Silence %lu CL_CONTINUE", (uint32_t)((pDataParams->dwResetRecoveryTime)*1000)); /* PRQA S 3200 */
                status = phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
            }
        }
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTmpChar));

    return status;
}

phStatus_t phhalHw_RdCardSim_ReadEnvTiming(phhalHw_RdCardSim_DataParams_t * pDataParams, uint32_t* timing_val)
{
	phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t	    PH_MEMLOC_REM pTmpChar[12];
	uint8_t *   PH_MEMLOC_REM pFrame;
    uint16_t    PH_MEMLOC_REM wTxLength;
    uint16_t    PH_MEMLOC_REM wRxLength;
    /* uint8_t     PH_MEMLOC_REM bNBits; */
	uint8_t		PH_MEMLOC_REM aTmpStr[12];
	uint16_t	PH_MEMLOC_REM payload_length;

	memset(aTmpStr, 0x00, sizeof(aTmpStr));
	memset(pTmpChar,0x00, sizeof(pTmpChar));

    status = PH_ERR_SUCCESS;

	sprintf((char*)pTmpChar, "TIME_FDT_US");
    wTxLength = (uint16_t)strlen((char*)pTmpChar);
	PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_ENV, 0, 0, pTmpChar, pFrame));

	/* Append a newline character at the end as the TCL read function expects a newline */
    pFrame[strlen((char*)pFrame)]='\n';

    /* Update TxLength */
    wTxLength = (uint16_t)strlen((char*)pFrame);

	    /* Do the exchange with BAL Layer */
    status = phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pFrame,
        wTxLength,
        pDataParams->wIntBufferLen / PHHAL_HW_RDCARDSIM_INTBUFFER,
        pFrame,
        &wRxLength);

    /* Status check */
    if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));
        return status;
    }

	/* Length check */
    if (strlen((char*)pFrame) != (uint16_t)(wRxLength - 1))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        return status;
    }

	/* Remove the newline character from the input stream */
    pFrame[wRxLength - 1] = 0x00;

	/* result is in following format: ENVACKxxxx ddddd
	* xxxx is hex length of dddddd
	* copy the byte length, */
	strncpy((char*)aTmpStr, (char*)(pFrame+6), 4);
	/* note: strncpy might not add \0, see man page ! */
	aTmpStr[4] = 0;

	payload_length = (uint16_t)strtol((char*)aTmpStr, NULL, 16 /* hex */);
	if (payload_length > sizeof(aTmpStr)-1)
	{
		/* bug wrong payload_length, parse what fits */
		payload_length = sizeof(aTmpStr)-1;
	}

	/* copy up to palyload_length bytes.
	* Note: this also works when too many bytes are specified (ISS also counts the space at the moment) */
	strncpy((char*)aTmpStr, (char*)(pFrame+11), payload_length);
	aTmpStr[payload_length] = 0x0;
	*timing_val = (uint32_t)strtol((char*)aTmpStr,NULL,10);


	PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

    return status;
}

phStatus_t phhalHw_RdCardSim_Wait(
                                  phhalHw_RdCardSim_DataParams_t * pDataParams,
                                  uint8_t   bUnit,
                                  uint16_t  wTimeout
                                  )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint32_t    PH_MEMLOC_REM dwWaitns;
    uint8_t *   PH_MEMLOC_REM pTmpChar;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_Wait");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bUnit);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTimeout);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bUnit_log, &bUnit);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wTimeout_log, &wTimeout);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    status = PH_ERR_SUCCESS;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTmpChar));
    switch (bUnit)
    {
    case PHHAL_HW_TIME_MICROSECONDS:
        dwWaitns = (uint32_t)wTimeout*1000;
        break;
    case PHHAL_HW_TIME_MILLISECONDS:
        dwWaitns = (uint32_t)wTimeout*1000000;
        break;
    default:
        dwWaitns = 0;
        status = PH_ERR_INVALID_PARAMETER;
        break;
    }
    if (dwWaitns == 0)
    {
        dwWaitns = 1;
    }

    if ( status == PH_ERR_SUCCESS ) {
        sprintf((char*)pTmpChar, "CL_Silence %lu CL_CONTINUE",dwWaitns ); /* PRQA S 3200 */
        status = phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTmpChar));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_Exchange(
                                      phhalHw_RdCardSim_DataParams_t * pDataParams,
                                      uint16_t    wOption,
                                      uint8_t   * pTxBuffer,
                                      uint16_t    wTxLength,
                                      uint8_t  ** ppRxBuffer,
                                      uint16_t  * pRxLength
                                      )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bTmpBitLength;
    uint16_t    PH_MEMLOC_REM wTmpByteLength;
    uint8_t *   PH_MEMLOC_REM pFrame;
    uint16_t	PH_MEMLOC_REM wRxLengthTmp;
    uint8_t *	PH_MEMLOC_REM pRxBuffer;
    uint32_t    PH_MEMLOC_REM dwTmpTime = 0;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_Exchange");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(ppRxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(dwFdt);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_TEXT(TX_Frame, "TX_Frame = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(RX_Frame, "RX_Frame = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(FDT_Frame, "FDT      = ");

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wOption_log, &wOption);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wTxLength_log, &wTxLength);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pTxBuffer_log, pTxBuffer, wTxLength);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Check if caller has provided valid RxBuffer */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &wRxLengthTmp;
    }

    /* Reset received length */
    *pRxLength = 0;

    /* clear internal buffer if requested */
    if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        pDataParams->wTxBufLen = 0;
    }

    if (pDataParams->bTPactive== 1)
    {
        if (pDataParams->wTxBufLen == 0)
        {
            /* the first word is reserved for the lenght */
            pDataParams->wTxBufLen += 2;

            /* it is the first frame and we need to add the option bytes */
            pDataParams->pTxBuffer[pDataParams->wTxBufLen] = (uint8_t)(pDataParams->wTimeT1 & 0x00FF);
            pDataParams->pTxBuffer[pDataParams->wTxBufLen + 1] = (uint8_t)((pDataParams->wTimeT1  & 0xFF00) >> 8);
            /*pDataParams->wTxBufLen += 2;
            //pDataParams->pTxBuffer[pDataParams->wTxBufLen] = (uint8_t)(pDataParams->wTimeT2 & 0x00FF);
            //pDataParams->pTxBuffer[pDataParams->wTxBufLen + 1] = (uint8_t)((pDataParams->wTimeT2  & 0xFF00) >> 8);
            //pDataParams->wTxBufLen += 2;
            //pDataParams->pTxBuffer[pDataParams->wTxBufLen] = (uint8_t)(pDataParams->wTimeT3 & 0x00FF);
            //pDataParams->pTxBuffer[pDataParams->wTxBufLen + 1] = (uint8_t)((pDataParams->wTimeT3  & 0xFF00) >> 8);
            //pDataParams->wTxBufLen += 2;
            //pDataParams->pTxBuffer[pDataParams->wTxBufLen] = (uint8_t)(pDataParams->wTimeT4 & 0x00FF);
            //pDataParams->pTxBuffer[pDataParams->wTxBufLen + 1] = (uint8_t)((pDataParams->wTimeT4  & 0xFF00) >> 8);
            //pDataParams->wTxBufLen += 2;
            //pDataParams->pTxBuffer[pDataParams->wTxBufLen] = (uint8_t)(pDataParams->wSyncOption & 0x00FF);
            //pDataParams->pTxBuffer[pDataParams->wTxBufLen + 1] = (uint8_t)((pDataParams->wSyncOption  & 0xFF00) >> 8); */
            pDataParams->wTxBufLen += 2;

            if (!(wOption & PH_EXCHANGE_BUFFERED_BIT))
            {
                /* set the leave buffer option to append the rest of the data */
                wOption |= PH_EXCHANGE_LEAVE_BUFFER_BIT;
            }
        }
    }

    /* Buffer overflow check */
    if ((wOption & PH_EXCHANGE_BUFFERED_BIT) || (wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        if ((pDataParams->wTxBufLen + wTxLength) > pDataParams->wTxBufSize)
        {
            pDataParams->wTxBufLen = 0;
            /*
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
            status = PH_ERR_BUFFER_OVERFLOW;
            PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
            */
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Copy transmit buffer*/
        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pTxBuffer, wTxLength);  /* PRQA S 3200 */
        pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;

        /* Shall we already perform the Exchange? */
        if (wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }

        /* Set correct Tx pointer and length  */
        pTxBuffer = pDataParams->pTxBuffer;
        wTxLength = pDataParams->wTxBufLen;
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));

    if (pDataParams->bEnvEnable)
    {
      PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_ENV, wTxLength, pDataParams->bTxLastBits, pTxBuffer, pFrame));
    }
    else
    {
        if (pDataParams->bTPactive == 1)
        {
            /* write the correct length into the first word */
            pTxBuffer[0] = (uint8_t)wTxLength;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_WAF, wTxLength, pDataParams->bTxLastBits, pTxBuffer, pFrame));
        }
        else if (pDataParams->bTPactive == 2)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_PCC, wTxLength, pDataParams->bTxLastBits, pTxBuffer, pFrame));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_PIC, wTxLength, pDataParams->bTxLastBits, pTxBuffer, pFrame));
        }
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, TX_Frame);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pFrame);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* Append a newline character at the end as teh TCL read function expects a newline */
    pFrame[strlen((char*)pFrame)]='\n';

    /* Calculate how much bytes we can receive max. */
    if (12 + (pDataParams->wRxBufSize * 2) < 0xFFFF)
    {
        wRxLengthTmp = (uint16_t)(12 + (pDataParams->wRxBufSize * 2));
    }
    else
    {
        wRxLengthTmp = 0xFFFF;
    }

    /* Do the exchange with BAL Layer */
    status = phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pFrame,
        (uint16_t)strlen((char*)pFrame),
        wRxLengthTmp,
        pFrame,
        &wTmpByteLength);

    /* Length check */
    if (strlen((char*)pFrame) != (uint16_t)(wTmpByteLength - 1))
    {
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Remove the newline character from the input stream */
    pFrame[wTmpByteLength - 1] = 0x00;

    if (pDataParams->bC40ModeEnabled == PH_OFF)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, RX_Frame);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pFrame);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
    }

    /* Parse RxFrame */
    if (pDataParams->bEnvEnable)
    {
        status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_ENV, &wTmpByteLength, &bTmpBitLength, &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos], pDataParams->bC40ModeEnabled);
    }
	else
	{
		if (pDataParams->bTPactive == 1)
		{
			status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_WAF, &wTmpByteLength, &bTmpBitLength, &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos], pDataParams->bC40ModeEnabled);
		}
		else if (pDataParams->bTPactive == 2)
		{
			status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_PCC, &wTmpByteLength, &bTmpBitLength, &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos], pDataParams->bC40ModeEnabled);
		}
		else
		{
			status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_PIC, &wTmpByteLength, &bTmpBitLength, &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos], pDataParams->bC40ModeEnabled);
		}
	}

    if (pDataParams->bC40ModeEnabled == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_Wait( pDataParams, PHHAL_HW_TIME_MICROSECONDS, pDataParams->wRx2TxWait));
    }
    else
    {
        if (!pDataParams->bEnvEnable && (pDataParams->wRx2TxWait > 0))
        {
            /* for a 'real' communication, take care of the guard time */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_Wait( pDataParams, PHHAL_HW_TIME_MICROSECONDS, pDataParams->wRx2TxWait));
        }
    }

	/* TODO there was a if(status == OK) statement here that I removed SLe. I hope this is ok and has no side effects */
    pDataParams->wTxBufLen = 0;
    pDataParams->bTxLastBits = 0;
    pDataParams->wAdditionalInfo = bTmpBitLength;
    pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wTmpByteLength;
    *ppRxBuffer = pDataParams->pRxBuffer;
    *pRxLength = pDataParams->wRxBufLen;


    if ((status == PH_ERR_SUCCESS) && (pDataParams->wAdditionalInfo != 0))
    {
        if ( (pDataParams->wAdditionalInfo + pDataParams->bRxAlign) != 8 )
        {
            status = PH_ERR_SUCCESS_INCOMPLETE_BYTE;
        }
    }

    /* Timing */
    if ((pDataParams->bC40ModeEnabled == PH_OFF || !pDataParams->bEnvEnable) && (pDataParams->wTimingMode & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK) == PHHAL_HW_TIMING_MODE_FDT)
    {
        if (pDataParams->bC40ModeEnabled == PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadEnvTiming(pDataParams, (uint32_t*)&dwTmpTime));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister32(pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_FDT_US_TIME, &dwTmpTime));

            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, FDT_Frame);
            PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_INFO, dwFdt_log, &dwTmpTime);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }

        /* FDT single -> replace the previous value */
        if (pDataParams->wTimingMode & PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR)
        {
            pDataParams->dwTimingUs = dwTmpTime;
        }
        /* FDT -> add current value to current count */
        else
        {
            /* Check for overflow */
            if ((0xFFFFFFFF - pDataParams->dwTimingUs) < dwTmpTime)
            {
                pDataParams->dwTimingUs = 0xFFFFFFFF;
            }
            /* Update global timing value */
            else
            {
                pDataParams->dwTimingUs = pDataParams->dwTimingUs + dwTmpTime;
            }
        }
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    if (pDataParams->bC40ModeEnabled == PH_ON)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, RX_Frame);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pFrame);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    }

    if ( (status == PH_ERR_SUCCESS || status == PH_ERR_SUCCESS_INCOMPLETE_BYTE) && (*pRxLength > 0) )
    {
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pRxLength_log, pRxLength);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, ppRxBuffer_log, *ppRxBuffer, (*pRxLength-1));
    }
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_ApplyProtocolSettings(
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t bCardType
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t *  PH_MEMLOC_REM pShadowDefault;
    uint16_t    PH_MEMLOC_REM wShadowCount;
    uint16_t    PH_MEMLOC_COUNT wIndex;
    uint16_t    PH_MEMLOC_REM wConfig;
    uint8_t     PH_MEMLOC_REM bUseDefaultShadow;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_ApplyProtocolSettings");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bCardType);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bCardType_log, &bCardType);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);


    if (bCardType != PHHAL_HW_CARDTYPE_CURRENT)
    {
        bUseDefaultShadow = 1;
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
        pDataParams->bCardType = bCardType;
    }
    else
    {
        bUseDefaultShadow = 0;
    }

    switch (pDataParams->bCardType)
    {
        /* configure hardware for ISO 14443A */
    case PHHAL_HW_CARDTYPE_ISO14443A:
        pShadowDefault = (uint16_t*)wRdCardSim_DefaultShadow_I14443a;
        wShadowCount = sizeof(wRdCardSim_DefaultShadow_I14443a) / (sizeof(uint16_t) * 2);
        if (bUseDefaultShadow) {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig( pDataParams, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_ON));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig( pDataParams, PHHAL_HW_CONFIG_PARITY, PH_ON));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig( pDataParams, PHHAL_HW_CONFIG_TXCRC, PH_OFF));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig( pDataParams, PHHAL_HW_CONFIG_RXCRC, PH_OFF));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig( pDataParams, PHHAL_HW_CONFIG_TXDATARATE, PHHAL_HW_RF_DATARATE_106));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig( pDataParams, PHHAL_HW_CONFIG_RXDATARATE, PHHAL_HW_RF_DATARATE_106));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig( pDataParams, PHHAL_HW_CONFIG_TIMEOUT_VALUE_US, PHHAL_HW_RDCARDSIM_DEFAULT_TIMEOUT));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig( pDataParams, PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS, 0));
        }
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    /* Apply shadowed registers */
    for (wIndex = 0; wIndex < wShadowCount; ++wIndex)
    {
        /* Get wConfig */
        wConfig = pShadowDefault[wIndex << 1];

        /* Apply only one the correct timeout unit */
        if (!(((wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US) &&
            (pDataParams->bTimeoutUnit != PHHAL_HW_TIME_MICROSECONDS)) ||
            ((wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS) &&
            (pDataParams->bTimeoutUnit != PHHAL_HW_TIME_MILLISECONDS))))
        {
            /* Default shadow: */
            if (bUseDefaultShadow)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_SetConfig(pDataParams, wConfig, pShadowDefault[(wIndex << 1) + 1]));
            }
        }
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_MfcAuthenticateKeyNo(
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint16_t wKeyNo,
    uint16_t wKeyVersion,
    uint8_t * pUid
    )
{
#ifdef NXPBUILD__PH_KEYSTORE
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM aKey[PHHAL_HW_MFC_KEY_LENGTH * 2];
    uint8_t *   PH_MEMLOC_REM pKey;
    uint16_t    PH_MEMLOC_REM bKeystoreKeyType;

    /* Bail out if we haven't got a keystore */
    if (pDataParams->pKeyStoreDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* retrieve KeyA & KeyB from keystore */
    PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo,
        wKeyVersion,
        sizeof(aKey),
        aKey,
        &bKeystoreKeyType));

    /* check key type */
    if (bKeystoreKeyType != PH_KEYSTORE_KEY_TYPE_MIFARE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Evaluate which key to use */
    if (bKeyType == PHHAL_HW_MFC_KEYA)
    {
        /* Use KeyA */
        pKey = aKey;
    }
    else if (bKeyType == PHHAL_HW_MFC_KEYB)
    {
        /* Use KeyB */
        pKey = &aKey[PHHAL_HW_MFC_KEY_LENGTH];
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Call the authentication function using the key */
    return phhalHw_RdCardSim_MfcAuthenticate(pDataParams, bBlockNo, bKeyType, aKey, pUid);
#else
    /* satisfy compiler */
    if (pDataParams || bBlockNo || bKeyType || wKeyNo || wKeyVersion || pUid);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
#endif /* NXPBUILD__PH_KEYSTORE */
}

phStatus_t phhalHw_RdCardSim_MfcAuthenticate(
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint8_t * pKey,
    uint8_t * pUid
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_REM wTmp;
    uint8_t     PH_MEMLOC_REM bTmp;
    uint8_t     PH_MEMLOC_REM Xch[3];
    uint8_t     PH_MEMLOC_REM bIdx;
    uint8_t     PH_MEMLOC_REM bStrCnt;
    uint8_t     PH_MEMLOC_REM bCrypto_dis;
    uint8_t *   PH_MEMLOC_REM pPyld;
    uint8_t *   PH_MEMLOC_REM pFrame;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_MfcAuthenticate");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBlockNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeyType);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pKey);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bBlockNo_log, &bBlockNo);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bKeyType_log, &bKeyType);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pKey_log, pKey, (uint16_t)strlen((char*)pKey));
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pUid_log, pUid, (uint16_t)strlen((char*)pUid));
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    status = PH_ERR_SUCCESS;

    /* Check if MIFARE crypto is enabled */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister(pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_DIS_MFCRYTO1, &bCrypto_dis));
    /* Evaluate which key to use */

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pPyld));
    if (bCrypto_dis)
    {
        if ((bKeyType & 0x7F) == PHHAL_HW_MFC_KEYA)
        {
            /* MF_AUTHENT_A authetication */
            sprintf( (char*)pPyld, "MF_AUTHENT_A");  /* PRQA S 3200 */
        }
        else if ((bKeyType & 0x7F) == PHHAL_HW_MFC_KEYB)
        {
            /* MF_AUTHENT_B authetication */
            sprintf( (char*)pPyld, "MF_AUTHENT_B");  /* PRQA S 3200 */
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        bStrCnt = 12;
    }
    else
    {
        if ((bKeyType & 0x7F) == PHHAL_HW_MFC_KEYA)
        {
            /* MF_AUTHENT_A authetication */
            sprintf( (char*)pPyld, "MF_ReAUTHENT_A");  /* PRQA S 3200 */
        }
        else if ((bKeyType & 0x7F) == PHHAL_HW_MFC_KEYB)
        {
            /* MF_AUTHENT_B authetication */
            sprintf( (char*)pPyld, "MF_ReAUTHENT_B");  /* PRQA S 3200 */
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        bStrCnt = 14;
    }
    /* format the string testbench command to pass to the simulator */
    /* add a space */
    pPyld[bStrCnt++] = ' ';
    pPyld[bStrCnt++] = '4';
    pPyld[bStrCnt++] = '8';
    pPyld[bStrCnt++] = '\'';
    pPyld[bStrCnt++] = 'h';
    /* Copy key data */
    for ( bIdx = 0; bIdx < 6; bIdx++ )
    {
        memset( Xch, 0x00, 3);  /* PRQA S 3200 */
        sprintf((char*)Xch, "%02X\0", pKey[bIdx]); /* PRQA S 3200 */
        strcpy( (char*)&pPyld[bStrCnt + bIdx*2], (char*)Xch); /* PRQA S 3200 */
    }
    bStrCnt += 12;
    /* add a space */
    pPyld[bStrCnt++] = ' ';

    pPyld[bStrCnt++] = '3';
    pPyld[bStrCnt++] = '2';

    pPyld[bStrCnt++] = '\'';
    pPyld[bStrCnt++] = 'h';

    /* add UID*/
    for ( bIdx = 0; bIdx < 4; bIdx++ )
    {
        memset( Xch, 0x00, 3);  /* PRQA S 3200 */
        sprintf((char*)Xch, "%02X\0", pUid[bIdx]); /* PRQA S 3200 */
        strcpy( (char*)&pPyld[bStrCnt + bIdx*2], (char*)Xch); /* PRQA S 3200 */
    }

    bStrCnt += 8;
    /* add a space */
    pPyld[bStrCnt++] = ' ';
    pPyld[bStrCnt++] = '\'';
    pPyld[bStrCnt++] = 'h';
    /* add block Number */
    memset( Xch, 0x00, 3);  /* PRQA S 3200 */
    sprintf((char*)Xch, "%02X\0", bBlockNo); /* PRQA S 3200 */
    strcpy( (char*)&pPyld[bStrCnt], (char*)Xch); /* PRQA S 3200 */

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_ENV, 0, 0, pPyld, pFrame));

    /* Append a newline character at the end as teh TCL read function expects a newline */
    pFrame[strlen((char*)pFrame)]='\n';

    /* Do the exchenge with BAL Layer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pFrame,
        (uint16_t)strlen((char*)pFrame),
        pDataParams->wRxBufSize,
        pFrame,
        &wTmp));

    /* Length check */
    if (strlen((char*)pFrame) != (uint16_t)(wTmp - 1))
    {
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Remove the newline character from the input stream */
    pFrame[wTmp - 1] = 0x00;

    status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_ENV, &wTmp, &bTmp, pPyld, pDataParams->bC40ModeEnabled);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pPyld));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_WriteRegister(
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint8_t bValue
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_REM wTmp;
    uint8_t     PH_MEMLOC_REM bTmp;
    uint8_t *   PH_MEMLOC_REM pPyld;
    uint8_t *   PH_MEMLOC_REM pFrame;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_WriteRegister");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAddress);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bValue);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bAddress_log, &bAddress);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bValue_log, &bValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pPyld));

    status = PH_ERR_SUCCESS;
    sprintf( (char*)pPyld, "%02X%08X", bAddress, bValue);  /* PRQA S 3200 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_PMW, FRAME_PRMADDR_NCHAR+FRAME_PRMDATA_NCHAR, 0, pPyld, pFrame));


    /* Append a newline character at the end as teh TCL read function expects a newline */
    pFrame[strlen((char*)pFrame)]='\n';

    /* Do the exchenge with BAL Layer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pFrame,
        (uint16_t)strlen((char*)pFrame),
        pDataParams-> wIntBufferLen/PHHAL_HW_RDCARDSIM_INTBUFFER,
        pFrame,
        &wTmp));

    /* Length check */
    if (strlen((char*)pFrame) != (uint16_t)(wTmp - 1))
    {
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Remove the newline character from the input stream */
    pFrame[wTmp - 1] = 0x00;

    status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_PMW, &wTmp, &bTmp, pPyld, pDataParams->bC40ModeEnabled);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pPyld));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHw_RdCardSim_ReadRegister(
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint8_t * pValue
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_REM wTmp;
    uint8_t     PH_MEMLOC_REM bTmp;
    uint8_t *   PH_MEMLOC_REM pPyld;
    uint8_t *   PH_MEMLOC_REM pFrame;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_ReadRegister");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAddress);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bAddress_log, &bAddress);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pPyld));

    status = PH_ERR_SUCCESS;
    sprintf( (char*)pPyld, "%02X", bAddress);  /* PRQA S 3200 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_PMR, FRAME_PRMADDR_NCHAR, 0, pPyld, pFrame));

    /* Append a newline character at the end as teh TCL read function expects a newline */
    pFrame[strlen((char*)pFrame)]='\n';

    /* Do the exchenge with BAL Layer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pFrame,
        (uint16_t)strlen((char*)pFrame),
        pDataParams-> wIntBufferLen/PHHAL_HW_RDCARDSIM_INTBUFFER,
        pFrame,
        &wTmp));

    /* Length check */
    if (strlen((char*)pFrame) != (uint16_t)(wTmp - 1))
    {
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bAddress_log, &bAddress);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, pValue_log, pValue);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Remove the newline character from the input stream */
    pFrame[wTmp - 1] = 0x00;

    status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_PMR, &wTmp, &bTmp, pPyld, pDataParams->bC40ModeEnabled);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

    *pValue = (uint8_t)strtol( (char*)pPyld, NULL, 16);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pPyld));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bAddress_log, &bAddress);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, pValue_log, pValue);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}
phStatus_t phhalHw_RdCardSim_ReadRegister32(
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint32_t * pValue
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_REM wTmp;
    uint8_t     PH_MEMLOC_REM bTmp;
    uint8_t *  PH_MEMLOC_REM pPyld;
    uint8_t *   PH_MEMLOC_REM pFrame;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_ReadRegister32");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAddress);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bAddress_log, &bAddress);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pPyld));

    status = PH_ERR_SUCCESS;
	/* 8 bit register address */
    sprintf( (char*)pPyld, "%02X", bAddress);  /* PRQA S 3200 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_PMR, FRAME_PRMADDR_NCHAR, 0, pPyld, pFrame));

    /* Append a newline character at the end as teh TCL read function expects a newline */
    pFrame[strlen((char*)pFrame)]='\n';

    /* Do the exchange with BAL Layer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pFrame,
        (uint16_t)strlen((char*)pFrame),
        pDataParams->wRxBufSize,
        pFrame,
        &wTmp));

    /* Length check */
    if (strlen((char*)pFrame) != (uint16_t)(wTmp - 1))
    {
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bAddress_log, &bAddress);
        PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_INFO, pValue_log, pValue);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Remove the newline character from the input stream */
    pFrame[wTmp - 1] = 0x00;

    status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_PMR, &wTmp, &bTmp, pPyld, pDataParams->bC40ModeEnabled);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

	/* 32 bit response */
    *pValue = strtol( (char*)pPyld, NULL, 16);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pPyld));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bAddress_log, &bAddress);
    PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_INFO, pValue_log, pValue);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_RDCARDSIM */
