/*
 * Copyright 2013, 2015, 2018 - 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * RdCardSim specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phhalHw.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_RDCARDSIM

#include "phhalHw_RdCardSim.h"
#include "phhalHw_RdCardSim_Int.h"
#include <phhalHw_RdCardSim_Cmd.h>

phStatus_t phhalHw_RdCardSim_Command_Int (
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t   * pTxBuffer
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t *   PH_MEMLOC_REM pFrame;
    uint16_t    PH_MEMLOC_REM wTxLength;
    uint16_t    PH_MEMLOC_REM wRxLength;
    uint8_t     PH_MEMLOC_REM bNBits;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_Command_Int");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ALLOCATE_TEXT(TX_TbCmd, ", TX_TbCmd = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(TX_Frame, " TX_Frame = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(RX_Frame, " RX_Frame = ");

    /* Retrieve TxLength */
    wTxLength = (uint16_t)strlen((char*)pTxBuffer);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wTxLength_log, &wTxLength);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, TX_TbCmd);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pTxBuffer);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Generate CMD_ENV command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_ENV, 0, 0, pTxBuffer, pFrame));

    /* Append a newline character at the end as teh TCL read function expects a newline */
    pFrame[strlen((char*)pFrame)]='\n';

    /* Update TxLength */
    wTxLength = (uint16_t)strlen((char*)pFrame);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, TX_Frame);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pFrame);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* Do the exchenge with BAL Layer */
    status = phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pFrame,
        wTxLength,
        pDataParams->wIntBufferLen / PHHAL_HW_RDCARDSIM_INTBUFFER,
        pFrame,
        &wRxLength);

    /* Status check */
    if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Length check */
    if (strlen((char*)pFrame) != (uint16_t)(wRxLength - 1))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Remove the newline character from the input stream */
    pFrame[wRxLength - 1] = 0x00;

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, RX_Frame);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pFrame);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* Parse RxFrame */
    status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_ENV, &wRxLength, &bNBits, NULL, pDataParams->bC40ModeEnabled);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phhalHw_RdCardSim_Cmd_DownloadImage (
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint32_t dwAddress,
    uint8_t bMemType,
    uint16_t *pImageLen,
    uint8_t * pImage
    )
{
    uint8_t     bCh[3];
    uint16_t   wIdx;
    uint16_t   wTmpIdx;
    uint16_t   wImgBlockLen;

    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t     PH_MEMLOC_REM wTmp;
    uint8_t     PH_MEMLOC_REM bTmp;

    uint8_t *   PH_MEMLOC_REM pFrame;
    uint8_t *   PH_MEMLOC_REM pHdrFrame;
    uint16_t    PH_MEMLOC_REM wTxLength;
    uint16_t    PH_MEMLOC_REM wRxLength;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_Cmd_DownloadImage");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wTxLength_log, &wTxLength);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    /* Transmit image in blocks of max CMDMEM_MAX_XFER_SIZE bytes*/
    while (*pImageLen)
    {
        wImgBlockLen = (*pImageLen > CMDMEM_MAX_XFER_SIZE) ?  (uint16_t)CMDMEM_MAX_XFER_SIZE : *pImageLen;

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pHdrFrame));

        /* 14 byte header not including cmd*/
        sprintf((char*)pHdrFrame, "%08lX%02X%04X\0", dwAddress, bMemType, wImgBlockLen); /* PRQA S 3200 */
        wTmpIdx = (uint16_t)strlen ((char*)pHdrFrame);

        /*convert payload to ascii*/
        for (wIdx = 0; wIdx < wImgBlockLen; wIdx++)
        {
            memset(bCh, 0x00, 3); /* PRQA S 3200 */
            sprintf((char*)bCh, "%02X\0", pImage[wIdx]); /* PRQA S 3200 */
            strcpy((char*)&pHdrFrame[wTmpIdx], (char*)bCh); /* PRQA S 3200 */
            wTmpIdx += 2;
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_MEM, 0, 0, pHdrFrame, pFrame));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pHdrFrame));
		/* Append a newline character at the end as teh TCL read function expects a newline */
		pFrame[strlen((char*)pFrame)]='\n';
        /* Update TxLength */
        wTxLength = (uint16_t)strlen((char*)pFrame);

        /* Do the exchange with BAL Layer */
        PH_CHECK_SUCCESS_FCT(status, phbalReg_Exchange(
            pDataParams->pBalDataParams,
            PH_EXCHANGE_DEFAULT,
            pFrame,
            wTxLength,
            pDataParams->wIntBufferLen / PHHAL_HW_RDCARDSIM_INTBUFFER,
            pFrame,
            &wRxLength));

        /* Offset address, source pointer and byte count by transfered amount*/
        dwAddress += wImgBlockLen;
        *pImageLen = (uint16_t)(*pImageLen - wImgBlockLen);
        pImage += wImgBlockLen;
    }
    /* Remove the newline character from the input stream */
    pFrame[strlen((const char*)pFrame) - 1] = 0x00;
    status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_MEM, &wTmp, &bTmp, NULL, pDataParams->bC40ModeEnabled);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));
    return status;
}

phStatus_t phhalHw_RdCardSim_Cmd_UploadImage (
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint32_t dwAddress,
    uint8_t bMemType,
    uint16_t *pImageLen,
    uint8_t * pImage
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;

    uint8_t *   PH_MEMLOC_REM pFrame;

    status = PH_ERR_SUCCESS;
    dwAddress= 0;
    bMemType = 0;
    *pImage=0;
    *pImageLen= 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

    return status;
}

phStatus_t phhalHw_RdCardSim_Cmd_WaferTestExchange(
                                      phhalHw_RdCardSim_DataParams_t * pDataParams,
                                      uint16_t    wOption,
                                      uint8_t   * pTxBuffer,
                                      uint16_t    wTxLength,
                                      uint8_t  ** ppRxBuffer,
                                      uint16_t  * pRxLength
                                      )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bTmpBitLength;
    uint16_t    PH_MEMLOC_REM wTmpByteLength;
    uint8_t *   PH_MEMLOC_REM pFrame;
    uint16_t	PH_MEMLOC_REM wRxLengthTmp;
    uint8_t *	PH_MEMLOC_REM pRxBuffer;
    uint32_t    PH_MEMLOC_REM dwTmpTime;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_Cmd_WaferTestExchange");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(ppRxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_TEXT(TX_Frame, "TX_Frame = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(RX_Frame, "RX_Frame = ");

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wOption_log, &wOption);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wTxLength_log, &wTxLength);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pTxBuffer_log, pTxBuffer, wTxLength);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Check if caller has provided valid RxBuffer */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &wRxLengthTmp;
    }

    /* Reset received length */
    *pRxLength = 0;

    /* clear internal buffer if requested */
    if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        pDataParams->wTxBufLen = 0;
    }

    /* Buffer overflow check */
    if ((wOption & PH_EXCHANGE_BUFFERED_BIT) || (wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        if ((pDataParams->wTxBufLen + wTxLength) > pDataParams->wTxBufSize)
        {
            pDataParams->wTxBufLen = 0;
            /*
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
            status = PH_ERR_BUFFER_OVERFLOW;
            PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
            */
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Copy transmit buffer*/
        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pTxBuffer, wTxLength);  /* PRQA S 3200 */
        pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;

        /* Shall we already perform the Exchange? */
        if (wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }

        /* Set correct Tx pointer and length  */
        pTxBuffer = pDataParams->pTxBuffer;
        wTxLength = pDataParams->wTxBufLen;
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_WAF, wTxLength, pDataParams->bTxLastBits, pTxBuffer, pFrame));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, TX_Frame);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pFrame);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* Append a newline character at the end as teh TCL read function expects a newline */
    pFrame[strlen((char*)pFrame)]='\n';

    /* Calculate how much bytes we can receive max. */
    if (12 + (pDataParams->wRxBufSize * 2) < 0xFFFF)
    {
        wRxLengthTmp = (uint16_t)(12 + (pDataParams->wRxBufSize * 2));
    }
    else
    {
        wRxLengthTmp = 0xFFFF;
    }

    /* Do the exchenge with BAL Layer */
    status = phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pFrame,
        (uint16_t)strlen((char*)pFrame),
        wRxLengthTmp,
        pFrame,
        &wTmpByteLength);

    /* Length check */
    if (strlen((char*)pFrame) != (uint16_t)(wTmpByteLength - 1))
    {
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Remove the newline character from the input stream */
    pFrame[wTmpByteLength - 1] = 0x00;

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, RX_Frame);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pFrame);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* Parse RxFrame */
    status = phhalHw_RdCardSim_RxFrame(pFrame, CMD_WAF, &wTmpByteLength, &bTmpBitLength, &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos], pDataParams->bC40ModeEnabled);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pFrame));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_Wait( pDataParams, PHHAL_HW_TIME_MICROSECONDS, pDataParams->wRx2TxWait));

	/* TODO there was a if(status == OK) statement here that I removed SLe. I hope this is ok and has no side effects */
    pDataParams->wTxBufLen = 0;
    pDataParams->bTxLastBits = 0;
    pDataParams->wAdditionalInfo = bTmpBitLength;
    pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wTmpByteLength;
    *ppRxBuffer = pDataParams->pRxBuffer;
    *pRxLength = pDataParams->wRxBufLen;


    if ((status == PH_ERR_SUCCESS) && (pDataParams->wAdditionalInfo != 0))
    {
        if ( (pDataParams->wAdditionalInfo + pDataParams->bRxAlign) != 8 )
        {
            status = PH_ERR_SUCCESS_INCOMPLETE_BYTE;
        }
    }

    /* Timing */
    if ((pDataParams->wTimingMode & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK) == PHHAL_HW_TIMING_MODE_FDT)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister(pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_FDT_MS_TIME, (uint8_t*)&dwTmpTime));
        if (dwTmpTime == 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_ReadRegister(pDataParams, PHHAL_HW_RDCARDSIM_CONFIG_FDT_US_TIME, (uint8_t*)&dwTmpTime));
        }
        else
        {
            dwTmpTime = dwTmpTime * 1000;
        }
        /* FDT single -> replace the previous value */
        if (pDataParams->wTimingMode & PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR)
        {
            pDataParams->dwTimingUs = dwTmpTime;
        }
        /* FDT -> add current value to current count */
        else
        {
            /* Check for overflow */
            if ((0xFFFFFFFF - pDataParams->dwTimingUs) < dwTmpTime)
            {
                pDataParams->dwTimingUs = 0xFFFFFFFF;
            }
            /* Update global timing value */
            else
            {
                pDataParams->dwTimingUs = pDataParams->dwTimingUs + dwTmpTime;
            }
        }
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    if (status == PH_ERR_SUCCESS || status == PH_ERR_SUCCESS_INCOMPLETE_BYTE)
    {
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pRxLength_log, pRxLength);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, ppRxBuffer_log, *ppRxBuffer, (*pRxLength-1));
    }
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}


phStatus_t phhalHw_RdCardSim_Cmd_SetTestPadValue(
                                        phhalHw_RdCardSim_DataParams_t * pDataParams,
										uint8_t bValue
                                        )
{
  phStatus_t  PH_MEMLOC_REM status;
  phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t *   PH_MEMLOC_REM pTmpChar;

  statusTmp  = PH_ERR_SUCCESS;
  PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTmpChar));
	if(bValue == 0)
		sprintf((char*)pTmpChar, "TP_CLR: %d", (uint32_t)((pDataParams->dwResetWaitTime)*1000)); /* PRQA S 3200 */
	else
		sprintf((char*)pTmpChar, "TP_SET: %d", (uint32_t)((pDataParams->dwResetWaitTime)*1000)); /* PRQA S 3200 */
  status = phhalHw_RdCardSim_Command_Int(pDataParams, pTmpChar);

  PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTmpChar));

  return status;
}

#endif /* NXPBUILD__PHHAL_HW_RDCARDSIM */
