/*
 * Copyright 2013, 2017 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal definitions for ISO14443-4 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHPALI14443P4_SW_INT_H
#define PHPALI14443P4_SW_INT_H

#include <ph_Status.h>

/** \addtogroup ph_Private
* @{
*/

/** ISO 14443-4 Block Mask */
#define PHPAL_I14443P4_SW_BLOCK_MASK            0xC0U

/** ISO 14443-4 I-Block */
#define PHPAL_I14443P4_SW_I_BLOCK               0x00U
/** ISO 14443-4 I-Block RFU mask */
#define PHPAL_I14443P4_SW_I_BLOCK_RFU_MASK      0x22U
/** ISO 14443-4 I-Block RFU bits */
#define PHPAL_I14443P4_SW_I_BLOCK_RFU_BITS      0x02U

/** ISO 14443-4 R-Block */
#define PHPAL_I14443P4_SW_R_BLOCK               0x80U
/** ISO 14443-4 R-Block RFU mask */
#define PHPAL_I14443P4_SW_R_BLOCK_RFU_MASK      0x26U
/** ISO 14443-4 R-Block RFU bits */
#define PHPAL_I14443P4_SW_R_BLOCK_RFU_BITS      0x22U

/** ISO 14443-4 S-Block */
#define PHPAL_I14443P4_SW_S_BLOCK               0xC0U
/** ISO 14443-4 S-Block RFU mask */
#define PHPAL_I14443P4_SW_S_BLOCK_RFU_MASK      0x01U
/** ISO 14443-4 S-Block RFU bits */
#define PHPAL_I14443P4_SW_S_BLOCK_RFU_BITS      0x00U

/** Byte position of PCB byte */
#define PHPAL_I14443P4_SW_PCB_POS               0x00U

/** Blocknumber Bit within PCB */
#define PHPAL_I14443P4_SW_PCB_BLOCKNR           0x01U

/** Bit within PCB which signals presence of NAD byte */
#define PHPAL_I14443P4_SW_PCB_NAD_FOLLOWING     0x04U

/** Bit within PCB which signals presence of CID byte */
#define PHPAL_I14443P4_SW_PCB_CID_FOLLOWING     0x08U

/** Bit within PCB which signals chaining */
#define PHPAL_I14443P4_SW_PCB_CHAINING          0x10U

/** Bit within PCB which signals ACK */
#define PHPAL_I14443P4_SW_PCB_ACK               0x00U

/** Bit within PCB which signals NAK */
#define PHPAL_I14443P4_SW_PCB_NAK               0x10U

/** Bit mask within PCB for S block to get type */
#define PHPAL_I14443P4_SW_PCB_S_TYPE_MASK       0x32U

/** Bits within PCB which signal a Param command */
#define PHPAL_I14443P4_SW_PCB_PARAM             0x30U

/** Bits within PCB which signal a DESELECT command */
#define PHPAL_I14443P4_SW_PCB_DESELECT          0x02U

/** Bits within PCB which signal a WTX command */
#define PHPAL_I14443P4_SW_PCB_WTX               0x32U

/** WTXM Bits of WTX INF byte */
#define PHPAL_I14443P4_SW_WTXM_BITS             0x3FU

/** FWI default value */
#define PHPAL_I14443P4_SW_FWI_DEFAULT           0x04U

/** FSCI default value */
#define PHPAL_I14443P4_SW_FSCI_DEFAULT          0x02U

/** FTW min in microseconds */
#define PHPAL_I14443P4_SW_FWT_MIN_US            302

/** FTW max in microseconds */
#define PHPAL_I14443P4_SW_FWT_MAX_US            4949000

/** S(DESELECT) FTW in microseconds */
#define PHPAL_I14443P4_SW_FWT_DESELECT_US       4833

/** S(PARAM) FTW in microseconds */
#define PHPAL_I14443P4_SW_FWT_SPARAM_US         4833

/** Valid CID bits within CID field */
#define PHPAL_I14443P4_SW_CID_MASK              0x0FU

/** I-Block transmission mode  */
#define PHPAL_I14443P4_SW_STATE_I_BLOCK_TX      0x01U

/** I-Block reception mode  */
#define PHPAL_I14443P4_SW_STATE_I_BLOCK_RX      0x02U

/** Exchange finished mode  */
#define PHPAL_I14443P4_SW_STATE_FINISHED        0x0FU

/** An I-Block mode is retramsmitting */
#define PHPAL_I14443P4_SW_STATE_RETRANSMIT_BIT  0x40U

/** An I-Block mode is chaining */
#define PHPAL_I14443P4_SW_STATE_CHAINING_BIT    0x80U

/** Only 6 bits of the state are pure state codes */
#define PHPAL_I14443P4_SW_STATE_MASK            0x0FU

/** Default Maximum Retry count for ISO/IEC 14443-4:2008(E) Rule 4 and 5 */
#define PHPAL_I14443P4_SW_MAX_RETRIES_DEFAULT   2U

/** Time extension in microseconds for all commands.  */
#define PHPAL_I14443P4_EXT_TIME_US              60

/** S(Param) Frame code for standard frame */
#define PHPAL_I14443P4_SPARAM_FRAME_STANDARD             0x01

/** S(Param) Frame code for standard frame with error correction */
#define PHPAL_I14443P4_SPARAM_FRAME_WEC                  0x02

/** S(Param) Frameoption bit for Start/Stop bit suppression */
#define PHPAL_I14443P4_SPARAM_FRAMEOPTION_START_STOP_BIT 0x01

/** S(Param) Frameoption bit for Start/End of frame suppression */
#define PHPAL_I14443P4_SPARAM_FRAMEOPTION_SOF_EOF        0x02

/** S(Param) Frameoption bit for sync suppression */
#define PHPAL_I14443P4_SPARAM_FRAMEOPTION_SYNC           0x04

/** S(Param) Bitmask to check the RFU bits of the datarate */
#define PHPAL_I14443P4_SPARAM_DATARATE_RFU_MASK          0x80F0

/** S(Param) Expected RFU bits of the datarate */
#define PHPAL_I14443P4_SPARAM_DATARATE_RFU_BITS          0x0000

/** Tag for S(Param) block info */
#define PHPAL_I14443P4_SPARAM_BLOCK_INFO_TAG             0xA0
/** Tag for S(Param) bit rate request */
#define PHPAL_I14443P4_SPARAM_BIT_RATE_REQUEST           0xA1
/** Tag for S(Param) bit rate indication */
#define PHPAL_I14443P4_SPARAM_BIT_RATE_INDICATION        0xA2
/** Tag for S(Param) bit rate activation */
#define PHPAL_I14443P4_SPARAM_BIT_RATE_ACTIVATION        0xA3
/** Tag for S(Param) bit rate acknowledgment */
#define PHPAL_I14443P4_SPARAM_BIT_RATE_ACK               0xA4
/** Tag for S(Param) frame format request */
#define PHPAL_I14443P4_SPARAM_FRAME_FORMAT_REQUEST       0xA5
/** Tag for S(Param) frame format indication */
#define PHPAL_I14443P4_SPARAM_FRAME_FORMAT_INDICATION    0xA6
/** Tag for S(Param) frame format activation */
#define PHPAL_I14443P4_SPARAM_FRAME_FORMAT_ACTIVATION    0xA7
/** Tag for S(Param) frame format acknowledgment */
#define PHPAL_I14443P4_SPARAM_FRAME_FORMAT_ACK           0xA8

#define PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_PCD_TAG                        0x80
#define PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_PCD_TAG_LENGTH                 2
#define PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_PICC_TAG                       0x81
#define PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_PICC_TAG_LENGTH                2
#define PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_FRAME_OPTION_PICC_TAG          0x82
#define PHPAL_I14443P4_SPARAM_SUPPORTED_BIT_RATE_FRAME_OPTION_PICC_TAG_LENGTH   1

#define PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PCD_TAG                         0x83
#define PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PCD_TAG_LENGTH                  2
#define PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PICC_TAG                        0x84
#define PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_PICC_TAG_LENGTH                 2
#define PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_FRAME_OPTION_PICC_TAG           0x85
#define PHPAL_I14443P4_SPARAM_SELECTED_BIT_RATE_FRAME_OPTION_PICC_TAG_LENGTH    1

#define PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMES_PCD_TAG                          0x80
#define PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMES_PCD_TAG_LENGTH                   1
#define PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMES_PICC_TAG                         0x81
#define PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMES_PICC_TAG_LENGTH                  1
#define PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMING_OPTION_PCD_TAG                  0x82
#define PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMING_OPTION_PCD_TAG_LENGTH           1
#define PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMING_OPTION_PICC_TAG                 0x83
#define PHPAL_I14443P4_SPARAM_SUPPORTED_FRAMING_OPTION_PICC_TAG_LENGTH          1

#define PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PCD_TAG                           0x84
#define PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PCD_TAG_LENGTH                    1
#define PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PICC_TAG                          0x85
#define PHPAL_I14443P4_SPARAM_SELECTED_FRAMES_PICC_TAG_LENGTH                   1
#define PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PCD_TAG                   0x86
#define PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PCD_TAG_LENGTH            1
#define PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PICC_TAG                  0x87
#define PHPAL_I14443P4_SPARAM_SELECTED_FRAMING_OPTION_PICC_TAG_LENGTH           1


#define PHPAL_I14443P4_SPARAM_BIT_RATE_106                              0x0100
#define PHPAL_I14443P4_SPARAM_BIT_RATE_212                              0x0200
#define PHPAL_I14443P4_SPARAM_BIT_RATE_424                              0x0400
#define PHPAL_I14443P4_SPARAM_BIT_RATE_848                              0x0800
#define PHPAL_I14443P4_SPARAM_BIT_RATE_1695                             0x1000
#define PHPAL_I14443P4_SPARAM_BIT_RATE_3390                             0x2000
#define PHPAL_I14443P4_SPARAM_BIT_RATE_6780                             0x4000
#define PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_3FC4                         0x0001
#define PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_FC                           0x0002
#define PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_3FC2                         0x0004
#define PHPAL_I14443P4_SPARAM_BIT_RATE_PSK_2FC                          0x0008

#define PHPAL_I14443P4_SW_IS_PARAM(bPcb)                                  \
    (                                                                   \
    (((bPcb) & PHPAL_I14443P4_SW_PCB_S_TYPE_MASK) == PHPAL_I14443P4_SW_PCB_PARAM) \
    ? 1 : 0                                                             \
    )

#define PHPAL_I14443P4_SW_IS_INVALID_BLOCK_STATUS(bStatus)                  \
    ((                                                                      \
    (((bStatus) & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT) ||                     \
    (((bStatus) & PH_ERR_MASK) == PH_ERR_INTEGRITY_ERROR) ||                \
    (((bStatus) & PH_ERR_MASK) == PH_ERR_FRAMING_ERROR) ||                  \
    (((bStatus) & PH_ERR_MASK) == PH_ERR_COLLISION_ERROR))                  \
    ? 1 : 0                                                                 \
    )

typedef struct
{
    uint8_t bIndicated;
    uint8_t bSupplied;
    uint16_t wSuppliedValue;
    uint16_t wIndicatedValue;
} phpalI14443p4_Sw_SParam_SingleParam;

typedef struct
{
    phpalI14443p4_Sw_SParam_SingleParam BitRatePcd2Picc;
    phpalI14443p4_Sw_SParam_SingleParam BitRatePicc2Pcd;
    phpalI14443p4_Sw_SParam_SingleParam BitRateFramingOptionsPicc2Pcd;
    phpalI14443p4_Sw_SParam_SingleParam FramePcd2Picc;
    phpalI14443p4_Sw_SParam_SingleParam FramePicc2Pcd;
    phpalI14443p4_Sw_SParam_SingleParam FramingOptionsPcd2Picc;
    phpalI14443p4_Sw_SParam_SingleParam FramingOptionsPicc2Pcd;
} phpalI14443p4_Sw_SParam_Param;


/**
* \brief Build I-Block frame header
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_BuildIBlock(
                                        uint8_t bCidEnabled,    /**< [In] Set to != 0 to enable CID byte */
                                        uint8_t bCid,           /**< [In] CID byte */
                                        uint8_t bNadEnabled,    /**< [In] Set to != 0 to enable NAD byte */
                                        uint8_t bNad,           /**< [In] NAD byte */
                                        uint8_t bPcbBlockNum,   /**< [In] PCB block number bit */
                                        uint8_t bChaining,      /**< [In] Set to != 0 if chaining bit should be set */
                                        uint8_t * pTxBuffer,    /**< [Out] transmit buffer */
                                        uint16_t * pTxLength    /**< [Out] number of bytes to send */
                                        );

/**
* \brief Build R(ACK) / R(NAK) frames
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_BuildRBlock(
                                        uint8_t bCidEnabled,    /**< [In] Set to != 0 to enable CID byte */
                                        uint8_t bCid,           /**< [In] CID byte */
                                        uint8_t bPcbBlockNum,   /**< [In] PCB block number bit */
                                        uint8_t bIsAck,         /**< [In] Set to != 0 if an ACK should be generated */
                                        uint8_t * pTxBuffer,    /**< [Out] transmit buffer */
                                        uint16_t * pTxLength    /**< [Out] number of bytes to send */
                                        );

/**
* \brief Build S-Block frames
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_BuildSBlock(
                                        uint8_t bCidEnabled,    /**< [In] Set to != 0 to enable CID byte */
                                        uint8_t bCid,           /**< [In] CID byte */
                                        uint8_t bIsWtx,         /**< [In] Set to != 0 if a WTX frame should be generated */
                                        uint8_t bWtxm,          /**< [In] WTXM bits, used if /ref bIsWtx is set */
                                        uint8_t * pTxBuffer,    /**< [Out] transmit buffer */
                                        uint16_t * pTxLength    /**< [Out] number of bytes to send */
                                        );

/**
* \brief Build S-Param Block frames
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_BuildSParamBlock(
                                        uint8_t bCidEnabled,    /**< [In] Set to != 0 to enable CID byte */
                                        uint8_t bCid,           /**< [In] CID byte */
                                        uint8_t * pTxBuffer,    /**< [Out] transmit buffer */
                                        uint16_t * pTxLength    /**< [Out] number of bytes to send */
                                        );

/**
* \brief Check if received I-Block is valid.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Block invalid.
*/
phStatus_t phpalI14443p4_Sw_IsValidIBlock(
    uint8_t bCheckCid,      /**< [In] Set to != 0 to enable CID check */
    uint8_t bCid,           /**< [In] Current CID for comparison */
    uint8_t bCheckNad,      /**< [In] Set to != 0 to enable NAD check */
    uint8_t bNad,           /**< [In] Current NAD for comparison */
    uint8_t * pRxBuffer,    /**< [In] Received data */
    uint16_t wRxLength      /**< [In] Number of received data bytes */
    );

/**
* \brief Check if received R-Block is valid.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Block invalid.
*/
phStatus_t phpalI14443p4_Sw_IsValidRBlock(
    uint8_t bCheckCid,      /**< [In] Set to != 0 to enable CID check */
    uint8_t bCid,           /**< [In] Current CID for comparison */
    uint8_t * pRxBuffer,    /**< [In] Received data */
    uint16_t wRxLength      /**< [In] Number of received data bytes */
    );

/**
* \brief Check if received S-Block is valid.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Block invalid.
*/
phStatus_t phpalI14443p4_Sw_IsValidSBlock(
    uint8_t bCheckCid,      /**< [In] Set to != 0 to enable CID check */
    uint8_t bCid,           /**< [In] Current CID for comparison */
    uint8_t * pRxBuffer,    /**< [In] Received data */
    uint16_t wRxLength      /**< [In] Number of received data bytes */
    );

/**
* \brief Perform actual exchanging and take care about error handling etc.
*
* Checks response and performs state transition and complete error & WTX handling if neecessary.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_IsoHandling(
                                        phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                        uint16_t wOption,                               /**< [In] option parameter. */
                                        uint8_t bRetryCount,                            /**< [In] number of already performed retries. */
                                        uint8_t * pTxBuffer,                            /**< [In] data to transmit */
                                        uint16_t wTxLength,                             /**< [In] length of input data */
                                        uint8_t ** ppRxBuffer,                          /**< [Out] Pointer to received data */
                                        uint16_t * pRxLength                            /**< [In] Received data bytes */
                                        );

/**
* \brief Perform an exchange of an S(param) frame and take care about error handling etc.
*
* Checks response and performs retries if neecessary.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_ExchangeSParamFrame(
                                     phpalI14443p4_Sw_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                     uint8_t *pDataIn,                                  /**< [In] data that should be send to the PICC */
                                     uint16_t wDataInLength,                            /**< [In] length of data */
                                     uint8_t *pDataOut,                                 /**< [Out] buffer to store the data returned from the PICC */
                                     uint16_t wDataOutSize,                             /**< [In] Size of the return buffer */
                                     uint16_t *pwDataOutLength                          /**< [Out] number of bytes returned */
                                     );

/**
* \brief Parse the length of an TLV object.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_Parse_TLV_Len(
                                    uint8_t *pData,                                     /**< [In] Data to be parsed. */
                                    uint16_t wDataLen,                                  /**< [In] Length of the data to be parsed. */
                                    uint16_t wLengthOffset,                             /**< [In] Offset where the length intication should be start. */
                                    uint32_t *pdwLengthValue,                           /**< [Out] The parsed length */
                                    uint16_t * pwLengthLength                           /**< [Out] The length of the length element */
                                    );

/**
* \brief Attache the tag and the length object before the data.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_Attach_TLV_Header(
                                    uint8_t *pData,                                     /**< [In] Buffer where the data is written and the header should be written. */
                                    uint32_t wDataLen,                                  /**< [In] length of the data. */
                                    uint32_t wDataOffset,                               /**< [In] Offset where the data is located. */
                                    uint8_t bTag,                                       /**< [In] Tag that should be used */
                                    uint16_t *pwDataWritten                             /**< [Out] Number of bytes used for the header. */
                                    );

/**
* \brief Check the supplied parameters of an S(PARAM) request.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_SParamCheckSuppliedParameter(phpalI14443p4_Sw_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                                           phpalI14443p4_Sw_SParam_Param * pParameter                 /**< [In] Pointer to the struct with the supplied values. */
										   );


/**
* \brief Extract the indicated data of an S(PARAM) indication.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_SParamExtractIndicatedParam(phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                           phpalI14443p4_Sw_SParam_Param * pParameter,                  /**< [Out] Pointer to the struct with the supplied values. */
                                           uint8_t *pbSParamDataBuffer,                                 /**< [In] Buffer containing the returned values from the S(Param) request */
                                           uint16_t wSParamDataLen,                                     /**< [In] Length of the returned data */
                                           uint8_t bBitRateIndicationSupported,                         /**< [In] PH_ON if BitRate Indication should be checked */
                                           uint8_t bFrameFormatIndicationSupported                    /**< [In] PH_ON if FrameFormat Indication should be checked */
										   );

/**
* \brief Check if the supplied parameter are supported by the indicated parameter an S(PARAM) indication.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_SParamCheckSuppliedParameterAreSupported(
	phpalI14443p4_Sw_SParam_Param * pParameter /**< [In] Pointer to the struct with the supplied values. */
	);

/**
* \brief Automaticly gets the 'best' parameter for the S(Param) activation form the S(PARAM) indication.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_SParamGetAutoParameter(phpalI14443p4_Sw_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                                           phpalI14443p4_Sw_SParam_Param * pParameter                 /**< [In] Pointer to the struct with the supplied values. */
										   );

/**
* \brief Build the S(Param) activation command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_SParamBuildActivateData(phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                           phpalI14443p4_Sw_SParam_Param * pParameter,              /**< [In] Pointer to the struct with the supplied values. */
                                           uint8_t *pbSParamDataBuffer,                             /**< [Out] Buffer where the generated activation is written */
                                           uint16_t wSParamDataSize,                                /**< [In] Size of the Buffer */
                                           uint16_t *pwSParamDataStartPos,                          /**< [Out] Start index of the generated data within the array */
                                           uint8_t bBitRateIndicationSupported,                     /**< [In] PH_ON if BitRate Indication should be checked */
                                           uint8_t bFrameFormatIndicationSupported                /**< [In] PH_ON if FrameFormat Indication should be checked */
										   );

/**
* \brief Applies the activated parameters to the hardware.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4_Sw_Int_SParamApplyParameter(phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                           phpalI14443p4_Sw_SParam_Param * pParameter          /**< [In] Pointer to the struct with the supplied values. */
										   );



/**
* \brief Perform Data Exchange with Picc. If FWEC is enabled the frame will coded to FWEC
*
* \b wOption can be one of:\n
* \li #PH_EXCHANGE_DEFAULT
*
* \b wOption can be combined with:\n
* \li #PH_EXCHANGE_BUFFERED_BIT
* \li #PH_EXCHANGE_LEAVE_BUFFER_BIT
*
* \ref phhalHw_Rc523 <em>Special Behaviour:</em>\n
* If #PHHAL_HW_CARDTYPE_I18092MPT is chosen, Exchange performs the reception BEFORE the transmission.\n
* For details refer to \ref phhalHw_Rc523.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b wOption is invalid.
* \retval #PH_ERR_SUCCESS_INCOMPLETE_BYTE Operation successful, incomplete byte received; Retrieve number of valid bits of last byte with #PHHAL_HW_CONFIG_RXLASTBITS.
* \retval #PH_ERR_IO_TIMEOUT No response detected within the configured time frame.
* \retval #PH_ERR_INTEGRITY_ERROR Response received but CRC or Parity is invalid.
* \retval #PH_ERR_COLLISION_ERROR Response received but a collision occured; Retrieve number of valid bits of last byte with #PHHAL_HW_CONFIG_RXLASTBITS.
* \retval #PH_ERR_BUFFER_OVERFLOW Internal receive buffer is too small for transmission or smaller than the response.
* \retval #PH_ERR_FRAMING_ERROR Frame format is either invalid for configured protocol or corrupted.
* \retval #PH_ERR_PROTOCOL_ERROR Frame format is definitely invalid for configured protocol.
* \retval #PH_ERR_READ_WRITE_ERROR Hardware problem.
* \retval #PH_ERR_TEMPERATURE_ERROR Hardware problem.
* \retval #PH_ERR_RF_ERROR Hardware problem.
* \retval #PH_ERR_INTERFACE_ERROR Hardware problem.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_Exchange(
    phpalI14443p4_Sw_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wOption,       /**< [In] Option parameter. */
    uint8_t * pTxBuffer,    /**< [In] Data to transmit. */
    uint16_t wTxLength,     /**< [In] Number of bytes to transmit. */
    uint8_t ** ppRxBuffer,  /**< [Out] Pointer to received data. */
    uint16_t * pRxLength    /**< [Out] Number of received data bytes. */
    );

/**
* \brief Get the max possible buffer size for Rx and Tx buffer possible with current config and reader.
* If FWEC is enabled the buffer size if the FWEC Buffer is used
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
* \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
* \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
* \retval #PH_ERR_INTERFACE_ERROR Communication error.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_GetRxTxBufferSize(
    phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
    uint16_t *pwRxBufferSize,       /**< [Out] RxBufferSize. */
    uint16_t *pwTxBufferSize        /**< [Out] TxBufferSize. */
    );

/**
* \brief Get the number of preloaded bytes (either in FWEC buffer or in reader buffer)
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
* \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
* \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
* \retval #PH_ERR_INTERFACE_ERROR Communication error.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_GetNumberOfPreloadedBytes(
    phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
    uint16_t *pwNumberPreloadedBytes                /**< [Out] Number preloaded bytes. */
    );

/**
* \brief Get the stored PCB byte
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
* \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
* \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
* \retval #PH_ERR_INTERFACE_ERROR Communication error.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_GetTxPCB(
    phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
    uint16_t *pwPCB                                  /**< [Out] PCB byte. */
    );

/**
* \brief Change the stored PCB byte
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
* \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
* \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
* \retval #PH_ERR_INTERFACE_ERROR Communication error.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_SetTxPCB(
    phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bPCB                                    /**< [In] PCB byte. */
    );

/**
* \brief Change the TxBufferLen
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
* \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
* \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
* \retval #PH_ERR_INTERFACE_ERROR Communication error.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_SetTxBufferLen(
    phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wTxBufferLen                           /**< [In] TxBufferLen. */
    );

/**
* \brief Change Rx Buffer Start Pos
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
* \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
* \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
* \retval #PH_ERR_INTERFACE_ERROR Communication error.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_SetRxBufferStartPos(
    phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wRxBufferStartPos                      /**< [In] RxBufferStartPos. */
    );

/**
* \brief Get Rx Buffer Start Pos
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
* \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
* \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
* \retval #PH_ERR_INTERFACE_ERROR Communication error.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_GetRxBufferStartPos(
    phpalI14443p4_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
    uint16_t * pwRxBufferStartPos                   /**< [Out] RxBufferStartPos. */
    );

/**
* \brief Formats a given Buffer for FWEC Exchange
* \return Status code 0 or specific Errorcode
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_FormatBlockFWEC(
                                   uint8_t *pTxBuffer,              /**< [In] Data to be formatted for FWEC */
                                   uint16_t wTxSize,                /**< [In] Size of pTxBuffer (must be at least 4 bytes > than length to add CRC) */
                                   uint16_t wTxLen,                /**< [In] Length of pTxBuffer */
                                   uint8_t *pEnhancedBlock,         /**< [Out] Correctly formatted Block */
                                   uint16_t wEnhancedBlockSize,    /**< [In] Size of pEnhancedBlock */
                                   uint16_t *pwEnhancedBlockLen      /**< [Out] Actual size of pEnhancedBlock */
                                   );

/**
* \brief Converts a recieved Buffer from FWEC Format to normal data
* \return Status code 0 or specific Errorcode
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4_Sw_Int_VHBR_FormatRecvBlockFWEC(
                                   uint8_t *pRxEnhancedBlock,      /**< [In] Data to be converted */
                                   uint16_t wRxEnhancedBlockLen,   /**< [In] Length of Data to be converted */
                                   uint8_t *pRxBuffer,             /**< [Out] Converted Data */
                                   uint16_t wRxBufferSize,         /**< [In] Size of Buffer for converted Data */
                                   uint16_t *pwRxLen,              /**< [Out] Size of converted Data (including the offset) */
                                   uint16_t *pwRxStartPos          /**< [Out] Offset of the real data in the rx buffer */
                                   );

/** @}
* end of ph_Private
*/

#endif /* PHPALI14443P4_SW_INT_H */
