/*
 * Copyright 2017 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * PCSC MIFARE(R) Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHPAL_MIFARE_PCSC

#include <phhalHw_Pcsc_Cmd.h>
#include "phpalMifare_Pcsc.h"
#include "../phpalMifare_Int.h"
#include "phpalI14443p4.h"

phStatus_t phpalMifare_Pcsc_Init(
                                 phpalMifare_Pcsc_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 void * pHalDataParams,
                                 void * pPalI14443p4DataParams
                                 )
{
    if (sizeof(phpalMifare_Pcsc_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_MIFARE);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_PAL_MIFARE | PHPAL_MIFARE_PCSC_ID;
    pDataParams->pHalDataParams         = pHalDataParams;
	pDataParams->pPalI14443p4DataParams = pPalI14443p4DataParams;
    pDataParams->bWaitForWriteData      = 0U;
    pDataParams->bWaitForDecrementValue = 0U;
    pDataParams->bWaitForIncrementValue = 0U;
    pDataParams->wWriteAddress          = 0U;
    pDataParams->bExchangeType          = PHPAL_MIFARE_PCSC_NORMAL;

    return PH_ERR_SUCCESS;
}

phStatus_t phpalMifare_Pcsc_ExchangeL3(
                                       phpalMifare_Pcsc_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint8_t ** ppRxBuffer,
                                       uint16_t * pRxLength
                                       )
{
    phStatus_t  PH_MEMLOC_REM status = PH_ERR_INTERNAL_ERROR;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_REM wValidBits;
    uint16_t    PH_MEMLOC_REM wCrcIn;
    uint16_t    PH_MEMLOC_REM wCrcCalc;
    uint16_t    PH_MEMLOC_REM wRxLength;
    uint8_t *   PH_MEMLOC_REM pRxBuffer;
    uint8_t     PH_MEMLOC_REM bMfpDummyData = 0x0AU;

    /* Check if caller has provided valid RxBuffer */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &wRxLength;
    }

    if (pDataParams->bExchangeType == PHPAL_MIFARE_PCSC_NORMAL)
    {
        /* MFC / MFUL Read command */
        if ((pTxBuffer[0] == 0x30U) && (wTxLength == 2U))
        {
            /* Send read command */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_ReadBinary(
                (phhalHw_Pcsc_DataParams_t *)pDataParams->pHalDataParams,
                pTxBuffer[1],
                0x10U,
                ppRxBuffer,
                pRxLength));

            status = PH_ERR_SUCCESS;
        }
        /* MFC / MFUL Write command */
        else if ((((pTxBuffer[0] == 0xA0U) || (pTxBuffer[0] == 0xA2U)) && (wTxLength == 2U)) ||
            ((pDataParams->bWaitForWriteData == 1U) && ((wTxLength == 4U) || (wTxLength == 16U))))
        {
            if (pDataParams->bWaitForWriteData == 0U)
            {
                /* Data to write is send in next call to ExchangeL3 */
                pDataParams->bWaitForWriteData = 1U;

                /* Update Write Address */
                pDataParams->wWriteAddress = pTxBuffer[1];
            }
            else
            {
                /* Data to write received, send write command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_UpdateBinary(
                    (phhalHw_Pcsc_DataParams_t *)pDataParams->pHalDataParams,
                    pDataParams->wWriteAddress,
                    pTxBuffer,
                    (uint8_t)wTxLength));

                /* Reset flag */
                pDataParams->bWaitForWriteData = 0U;
            }

            /* For MFP */
            *pRxLength = 1U;
            *ppRxBuffer = &bMfpDummyData;

            status = PH_ERR_SUCCESS;
        }
        /* MFC Decrement */
        else if (((pTxBuffer[0] == 0xC0U) && (wTxLength == 2U)) ||
            ((pDataParams->bWaitForDecrementValue == 1U) && (wTxLength == 4U)))
        {
            if (pDataParams->bWaitForDecrementValue == 0U)
            {
                /* Data to write is send in next call to ExchangeL3 */
                pDataParams->bWaitForDecrementValue = 1U;

                /* Update Write Address */
                pDataParams->wWriteAddress = pTxBuffer[1];

                status = PH_ERR_SUCCESS;
            }
            else
            {
                /* Send decrement command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_Decrement(
                    (phhalHw_Pcsc_DataParams_t *)pDataParams->pHalDataParams,
                    (uint8_t)pDataParams->wWriteAddress,
                    pTxBuffer));

                /* Reset flag */
                pDataParams->bWaitForDecrementValue = 0U;

                /* MFC AL layer expect timeout */
                status = PH_ERR_IO_TIMEOUT;
            }

            /* For MFP */
            *pRxLength = 1U;
            *ppRxBuffer = &bMfpDummyData;
        }
        /* MFC Increment */
        else if (((pTxBuffer[0] == 0xC1U) && (wTxLength == 2U)) ||
            ((pDataParams->bWaitForIncrementValue == 1U) && (wTxLength == 4U)))
        {
            if (pDataParams->bWaitForIncrementValue == 0U)
            {
                /* Data to write is send in next call to ExchangeL3 */
                pDataParams->bWaitForIncrementValue = 1U;

                /* Update Write Address */
                pDataParams->wWriteAddress = pTxBuffer[1];

                status = PH_ERR_SUCCESS;
            }
            else
            {
                /* Send increment command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_Increment(
                    (phhalHw_Pcsc_DataParams_t *)pDataParams->pHalDataParams,
                    (uint8_t)pDataParams->wWriteAddress,
                    pTxBuffer));

                /* Reset flag */
                pDataParams->bWaitForIncrementValue = 0U;

                /* MFC AL layer expect timeout */
                status = PH_ERR_IO_TIMEOUT;
            }

            /* For MFP */
            *pRxLength = 1U;
            *ppRxBuffer = &bMfpDummyData;
        }
    }
    /* Tranparent (RAW) Exchange */
    else if (pDataParams->bExchangeType == PHPAL_MIFARE_PCSC_TRANSPARENT)
    {
        /* Switch CRC modes in case of first part of exchange. */
        if (!(wOption & PH_EXCHANGE_BUFFERED_BIT))
        {
            /* Enable TxCrc */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, PH_ON));

            /* Disable RxCrc */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, PH_OFF));
        }

        /* Perform Exchange */
        status = phhalHw_Exchange(
            pDataParams->pHalDataParams,
            wOption,
            pTxBuffer,
            wTxLength,
            ppRxBuffer,
            pRxLength);

        /* Return if no real exchange is done */
        if (wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            return status;
        }

        /* ACK/NAK Handling */
        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS_INCOMPLETE_BYTE)
        {
            /* Check for protocol error */
            if (*pRxLength != 1)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
            }

            /* Retrieve bitcount */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXLASTBITS, &wValidBits));

            /* Check for protocol error */
            if (wValidBits != 4)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
            }

            /* ACK/NAK Mapping */
            switch ((*ppRxBuffer)[0])
            {
                /* ACK -> everything OK */
            case PHPAL_MIFARE_RESP_ACK:
                status = PH_ERR_SUCCESS;
                break;
                /* Mapping of NAK codes: */
            case PHPAL_MIFARE_RESP_NAK0:
                status = PHPAL_MIFARE_ERR_NAK0;
                break;
            case PHPAL_MIFARE_RESP_NAK1:
                status = PHPAL_MIFARE_ERR_NAK1;
                break;
            case PHPAL_MIFARE_RESP_NAK4:
                status = PHPAL_MIFARE_ERR_NAK4;
                break;
            case PHPAL_MIFARE_RESP_NAK5:
                status = PHPAL_MIFARE_ERR_NAK5;
                break;
            case PHPAL_MIFARE_RESP_NAK6:
                status = PHPAL_MIFARE_ERR_NAK6;
                break;
            case PHPAL_MIFARE_RESP_NAK7:
                status = PHPAL_MIFARE_ERR_NAK7;
                break;
            case PHPAL_MIFARE_RESP_NAK9:
                status = PHPAL_MIFARE_ERR_NAK9;
                break;
            default:
                status = PH_ERR_PROTOCOL_ERROR;
                break;
            }
        }
        /* Normal data stream with CRC */
        else
        {
            /* Check status */
            PH_CHECK_SUCCESS(status);

            /* Check length (min. 1 byte + 2 byte CRC) */
            if (*pRxLength < 3)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
            }

            /* Retrieve CRC */
            wCrcIn  = (uint16_t)(((uint16_t)(*ppRxBuffer)[(*pRxLength) - 1]) << 8);
            wCrcIn |= (uint16_t)((*ppRxBuffer)[(*pRxLength) - 2]);

            /* Remove CRC from input data */
            *pRxLength -= 2;

            /* Calculate CRC */
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                PH_TOOLS_CRC16_PRESET_ISO14443A,
                PH_TOOLS_CRC16_POLY_ISO14443,
                *ppRxBuffer,
                *pRxLength,
                &wCrcCalc));

            /* CRC Check -> Compare input and calculated crc */
            if (wCrcIn == wCrcCalc)
            {
                status = PH_ERR_SUCCESS;
            }
            else
            {
                status = PH_ERR_INTEGRITY_ERROR;
            }
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_MIFARE);
    }

    return PH_ADD_COMPCODE(status, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_Pcsc_ExchangeL4(
                                       phpalMifare_Pcsc_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint8_t ** ppRxBuffer,
                                       uint16_t * pRxLength
                                       )
{
#ifdef NXPBUILD__PHPAL_I14443P4
    /* Used for Reception */
    uint16_t   PH_MEMLOC_REM RxLength;
    uint8_t *  PH_MEMLOC_REM pRxBuffer;

    /* Check if caller has provided valid RxBuffer */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &RxLength;
    }

    *pRxLength = 0;


	/* PAL 14443-4 Exchange to take care of NORMAL or TRANSPARENT exchange*/
    /* Wrapped mode send AS-IS */
	return phpalI14443p4_Exchange(
        pDataParams->pPalI14443p4DataParams,
        wOption,
        pTxBuffer,
        wTxLength,
        ppRxBuffer,
        pRxLength
		);
#else
    /* Satisfy compiler */
    if (pDataParams || wOption || pTxBuffer || wTxLength || ppRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
#endif
}

phStatus_t phpalMifare_Pcsc_ExchangePc(
                                       phpalMifare_Pcsc_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint8_t ** ppRxBuffer,
                                       uint16_t * pRxLength
                                       )
{
#if 0
/* #ifdef NXPBUILD__PHPAL_I14443P4 */
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bIsoFrame[3];
    uint16_t    PH_MEMLOC_REM wIsoFrameLen;
    uint16_t    PH_MEMLOC_REM wBlockNo;
    uint16_t    PH_MEMLOC_REM wCidConfig;
    uint16_t    PH_MEMLOC_REM wNadConfig;
    uint16_t    PH_MEMLOC_REM wCrcIn;
    uint16_t    PH_MEMLOC_REM wCrcCalc;

    /* Satisfy compiler */
    pDataParams = NULL;
    wBlockNo = 0;
    wCidConfig = 0;
    wNadConfig = 0;

    /* Ignore wOption byte */
    wOption = PH_EXCHANGE_DEFAULT;

    /* TODO: Disable TxCrc, Disable RxCrc */

    /* TODO: Retrieve ISO 14443-4 Protocol Parameters */

    /* Build ISO 14443-4 I-Block Frame */
    bIsoFrame[0] = (uint8_t)(0x02 | wBlockNo);
    wIsoFrameLen = 1;

    /* Append CID if needed */
    {
        bIsoFrame[0] |= 0x08;
        bIsoFrame[wIsoFrameLen++] = (uint8_t)(wCidConfig & 0x00FF);
    }

    /* Append NAD if needed */
    {
        bIsoFrame[0] |= 0x04;
        bIsoFrame[wIsoFrameLen++] = (uint8_t)(wNadConfig & 0x00FF);
    }

    /* Calculate CRC over the frame */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
        PH_TOOLS_CRC_OPTION_DEFAULT,
        PH_TOOLS_CRC16_PRESET_ISO14443A,
        PH_TOOLS_CRC16_POLY_ISO14443,
        bIsoFrame,
        wIsoFrameLen,
        &wCrcCalc));

    /* Calculate CRC over the data to send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
        PH_TOOLS_CRC_OPTION_DEFAULT,
        wCrcCalc,
        PH_TOOLS_CRC16_POLY_ISO14443,
        pTxBuffer,
        wTxLength,
        &wCrcCalc));

    /* TODO: Preload the frame */

    /* TODO: Perform Exchange */

    /* TODO: Check for success */

    /* Response length check */
    if (*pRxLength < 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
    }

    /* Retrieve CRC */
    wCrcIn  = (uint16_t)(((uint16_t)(*ppRxBuffer)[(*pRxLength) - 1]) << 8);
    wCrcIn |= (uint16_t)((*ppRxBuffer)[(*pRxLength) - 2]);

    /* Remove CRC from input data */
    *pRxLength -= 2;

    /* Calculate CRC over the received data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
        PH_TOOLS_CRC_OPTION_DEFAULT,
        wCrcCalc,
        PH_TOOLS_CRC16_POLY_ISO14443,
        *ppRxBuffer,
        *pRxLength,
        &wCrcCalc));

    /* CRC Check -> Compare input and calculated crc */
    if (wCrcIn != wCrcCalc)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_PAL_MIFARE);
    }

    /* TODO: Update ISO14443-4 Block Number */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_MIFARE);
#else
    /* Satisfy compiler */
    if (pDataParams || wOption || pTxBuffer || wTxLength || ppRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
#endif
}

phStatus_t phpalMifare_Pcsc_MfcAuthenticateKeyNo(
                                                 phpalMifare_Pcsc_DataParams_t * pDataParams,
                                                 uint8_t bBlockNo,
                                                 uint8_t bKeyType,
                                                 uint16_t wKeyNo,
                                                 uint16_t wKeyVersion,
                                                 uint8_t * pUid
                                                 )
{
    phStatus_t PH_MEMLOC_REM status;

    /* Execute HAL authenticate function */
    status = phhalHw_MfcAuthenticateKeyNo(
        pDataParams->pHalDataParams,
        bBlockNo,
        bKeyType,
        wKeyNo,
        wKeyVersion,
        pUid);

    /* Change component code for AUTH error and invalid paramter */
    if (((status & PH_ERR_MASK) == PH_ERR_AUTH_ERROR) ||
        ((status & PH_ERR_MASK) == PH_ERR_INVALID_PARAMETER))
    {
        status = PH_ADD_COMPCODE((status & PH_ERR_MASK), PH_COMP_PAL_MIFARE);
    }

    return status;
}

phStatus_t phpalMifare_Pcsc_MfcAuthenticate(
                                            phpalMifare_Pcsc_DataParams_t * pDataParams,
                                            uint8_t bBlockNo,
                                            uint8_t bKeyType,
                                            uint8_t * pKey,
                                            uint8_t * pUid
                                            )
{
    phStatus_t PH_MEMLOC_REM status;

    /* Execute HAL authenticate function */
    status = phhalHw_MfcAuthenticate(
        pDataParams->pHalDataParams,
        bBlockNo,
        bKeyType,
        pKey,
        pUid);

    /* Change component code for AUTH error and invalid paramter */
    if (((status & PH_ERR_MASK) == PH_ERR_AUTH_ERROR) ||
        ((status & PH_ERR_MASK) == PH_ERR_INVALID_PARAMETER))
    {
        status = PH_ADD_COMPCODE((status & PH_ERR_MASK), PH_COMP_PAL_MIFARE);
    }

    return status;
}

phStatus_t phpalMifare_Pcsc_SetConfig(
                                      phpalMifare_Pcsc_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t wValue
                                      )
{
	phStatus_t  PH_MEMLOC_REM status;

    switch (wConfig)
	{
    case PHPAL_MIFARE_PCSC_CONFIG_EXCHANGE:
        /* Parameter Check */
        if ((wValue != PHPAL_MIFARE_PCSC_NORMAL) && (wValue != PHPAL_MIFARE_PCSC_TRANSPARENT))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_MIFARE);
        }

		pDataParams->bExchangeType = (uint8_t)wValue;

		if(pDataParams->bExchangeType == PHPAL_MIFARE_PCSC_NORMAL)
		{
#ifdef NXPBUILD__PHPAL_I14443P4
			/* Configure PAL 14443-4 exchange to use normal exchange (not transparent) */
			PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_SetConfig(
				pDataParams->pPalI14443p4DataParams,
				PHPAL_I14443P4_PCSC_CONFIG_EXCHANGE,
				PHPAL_I14443P4_PCSC_NORMAL));

					/* Configure HAL exchange to use transparent exchange. */
			PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
				pDataParams->pHalDataParams,
				PHHAL_HW_PCSC_CONFIG_EXCHANGE,
				PHHAL_HW_PCSC_NORMAL));
#else
					/* Configure HAL exchange to use transparent exchange. */
			PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
				pDataParams->pHalDataParams,
				PHHAL_HW_PCSC_CONFIG_EXCHANGE,
				PHHAL_HW_PCSC_NORMAL));
#endif
			}
		else if(pDataParams->bExchangeType == PHPAL_MIFARE_PCSC_TRANSPARENT)
		{
#ifdef NXPBUILD__PHPAL_I14443P4
			/* Configure PAL 14443 exchange to use transparent exchange */
			PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_SetConfig(
				pDataParams->pPalI14443p4DataParams,
				PHPAL_I14443P4_PCSC_CONFIG_EXCHANGE,
				PHPAL_I14443P4_PCSC_TRANSPARENT));

			/* Configure HAL exchange to use transparent exchange. */
			PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
				pDataParams->pHalDataParams,
				PHHAL_HW_PCSC_CONFIG_EXCHANGE,
				PHHAL_HW_PCSC_TRANSPARENT));
#else
			/* Configure HAL exchange to use transparent exchange. */
			PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
				pDataParams->pHalDataParams,
				PHHAL_HW_PCSC_CONFIG_EXCHANGE,
				PHHAL_HW_PCSC_TRANSPARENT));
#endif
		}
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_MIFARE);
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phpalMifare_Pcsc_GetConfig(
                                      phpalMifare_Pcsc_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t * pValue
                                      )
{
    switch (wConfig)
    {
    case PHPAL_MIFARE_PCSC_CONFIG_EXCHANGE:
        *pValue = (uint16_t)pDataParams->bExchangeType;
        break;

    default:
        return phpalI14443p4_GetConfig(
        pDataParams->pPalI14443p4DataParams,
        wConfig,
        pValue);
    }

    return PH_ERR_SUCCESS;
}

#endif /* NXPBUILD__PHPAL_MIFARE_PCSC */
