/*
 * Copyright 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example program to demonstrate ASymmetric features.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/** Header for this file */
#include "Example-CryptoLibFootprint_ASym.h"

#ifdef NXPBUILD__PH_CRYPTOASYM

static uint8_t aPrivateKey[] =
{
    0x60, 0x0A, 0xC0, 0xBA, 0x0A, 0x5B, 0x79, 0xB6, 0x07, 0x25, 0xA9, 0xC8, 0xFF, 0xB3, 0xE7, 0x45, 0x4A, 0x7C, 0x70, 0x2D, 0xB5, 0x65, 0x3A, 0x8A, 0xA6, 0x0E, 0x2E, 0x4C, 0x25, 0x6F, 0xF0, 0x5D
};
static uint8_t aPublicKey[] =
{
    0x04, 0xC4, 0x18, 0x7C, 0x26, 0xA5, 0x95, 0xB1, 0x8B, 0x37, 0x03, 0x31, 0x2D, 0x80, 0x12, 0x99, 0x9B, 0x88, 0xEB, 0x28, 0xBD, 0xB4, 0x1F, 0x30, 0x11, 0x28, 0x3D, 0xD8, 0x06, 0xEC, 0xDB, 0x09,
    0x24, 0x1B, 0xA9, 0x9D, 0x61, 0x2D, 0xA5, 0x4A, 0x6E, 0xAF, 0x14, 0x0F, 0x80, 0xAB, 0x5B, 0x19, 0xE4, 0xBA, 0x66, 0x53, 0x5D, 0x12, 0xBE, 0x85, 0x3C, 0x98, 0x76, 0x7A, 0xDF, 0x7C, 0x13, 0x0C,
    0x1A
};

static void * pCryptoASym = NULL;
static uint8_t aSignature[64];
static uint16_t wSignLen = 0;
static uint8_t aInternalBuffer[2048];
static uint16_t wInternalBuffSize = (uint16_t)sizeof(aInternalBuffer);
static uint8_t aMessage[] = "ASymmetric Crypto Demonstration";



phStatus_t Demo_ASymmetricCrypto_Main()
{
    uint32_t dwChoice = 0;
    char aChoice[5];
    uint8_t bEndLoop = FALSE;

    do
    {
        dwChoice = 0;

        /* Clear the console. */
        system("cls");

        if(CRYPTOASYM_MAX > 2)
        {
#ifdef NXPBUILD__PH_CRYPTOASYM_MBEDTLS
            printf("Press %d to demonstrate ASymmetric Crypto functionality using external mBedTLS implementation. \n", CRYPTOASYM_MBEDTLS);
#endif /* NXPBUILD__PH_CRYPTOASYM_MBEDTLS */

            printf("\n");
            printf("Press x to exit the application.\n");
            printf("Press r to return to previous demo description.\n");
            printf("Enter the option and press Enter to perform the demo - ");
            scanf("%s", aChoice);

            /* Convert to Integer value. */
            ToInt(aChoice, dwChoice);

            /* Return to previous demo description if requested */
            RETURN_IF_REQUESTED(dwChoice);

            /* Exit if requested by user. */
            EXIT_IF_REQUESTED(dwChoice);
        }
        else
        {
            dwChoice = 1;
        }

        printf("\n\n");

        switch(dwChoice)
        {
#ifdef NXPBUILD__PH_CRYPTOASYM_MBEDTLS
            case CRYPTOASYM_MBEDTLS:
                /* Initialize CryptoASym component. */
                CHECK_SUCCESS(phCryptoASym_mBedTLS_Init(&stCryptoASym_mBedTLS, sizeof(phCryptoASym_mBedTLS_DataParams_t), pKeyStore,
                    aInternalBuffer, wInternalBuffSize), TRUE);

                pCryptoASym = &stCryptoASym_mBedTLS;

                /* End the loop */
                bEndLoop = TRUE;
                break;
#endif /* NXPBUILD__PH_CRYPTOASYM_MBEDTLS */

            default:
                printf("Invalid option selected.\n");
                printf("--------------------------------------------------------------------------------------------------- \n");
                printf("\n\n");
                printf("Press any key to continue with the demonstration.\n");
                printf("Press r to return to previous demo description.\n");
                printf("Press x to exit the demonstration.\n");
                dwChoice = _getch();

                /* Return to previous demo description if requested */
                RETURN_IF_REQUESTED(dwChoice);

                /* Exit if requested by user. */
                EXIT_IF_REQUESTED(dwChoice);
                break;
        }
    } while(!bEndLoop);

    CHECK_SUCCESS(Demo_ASymmetricCrypto_Features(), FALSE);

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_ASymmetricCrypto_Features()
{
    uint32_t dwChoice = 0;
    char aChoice[5];

    uint8_t IS_KEY_PAIR_AVAILABLE = FALSE;

#ifdef NXPBUILD__PH_KEYSTORE_ASYM
    CHECK_SUCCESS(UpdateKeystore_ASym(), TRUE);
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */

    /* Initialize Crypto ASymmetric component for logging. */
#ifdef NXPBUILD__PH_LOG
    CHECK_SUCCESS(phLog_Register(pCryptoASym, aLogEntries, (uint16_t) (sizeof(aLogEntries) / sizeof(phLog_LogEntry_t))), TRUE);
#endif /* NXPBUILD__PH_LOG */

    do
    {
        dwChoice = 0;

        /* Clear the console. */
        system("cls");

        printf("Press %d for Demo on ComputeHash command. \n", MESSAGE_DIGEST);
        printf("Press %d for Demo on Generate Key-Pair command. \n", GENERATE_KEYPAIR);
        printf("Press %d for Demo on Export Key-Pair command. \n", EXPORT_KEYS);
        printf("Press %d for Demo on Message Signing command. \n", SIGN_MESSAGE);
        printf("Press %d for Demo on Message Verification command. \n", VERIFY_MESSAGE);
        printf("Press %d for Demo on Shared Secret Computation (ECDH) command. \n", COMPUTE_SHARED_SECRET);
        printf("Press %d for Demo on Direct Loading of Key-Pair command in Un-compressed Binary Format. \n", LOAD_KEYPAIR_DIRECT);
#ifdef NXPBUILD__PH_KEYSTORE_ASYM
        printf("Press %d for Demo on Loading of Key-Pair from KeyStore command in Un-compressed Binary Format. \n", LOAD_KEYPAIR_KEYSTORE);
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */
        printf("Press %d for Demo to Verify Message-Signature using Public Key. \n", VERIFY_SIGNATURE);
        printf("\n");
        printf("Press x to exit the application.\n");
        printf("Press r to return to previous demo description.\n");
        printf("Enter the option and press Enter to perform the demo - ");
        scanf("%s", aChoice);

        /* Convert to Integer value. */
        ToInt(aChoice, dwChoice);

        printf("\n\n");

        /* Return to previous demo description if requested */
        RETURN_IF_REQUESTED(dwChoice);

        /* Exit if requested by user. */
        EXIT_IF_REQUESTED(dwChoice);

        switch(dwChoice)
        {
            case MESSAGE_DIGEST:
                printf("Demo on Hash Computation command ----------------------------------------------------------------- \n");

                CHECK_SUCCESS(MessageDigest(aMessage, (uint16_t) sizeof(aMessage)), TRUE);

                printf("\tSuccessfully Generated Hash on the Message.\n");
                break;

            case GENERATE_KEYPAIR:
                printf("Demo on Key-Pair Generation command --------------------------------------------------------------- \n");
                printf("Key-Pair are generated internally by CryptoASym component of Reader Library.\n\n");

                IS_KEY_PAIR_AVAILABLE = FALSE;
                CHECK_SUCCESS(GenerateKeyPair(), TRUE);
                IS_KEY_PAIR_AVAILABLE = TRUE;

                printf("\tSuccessfully Generated ECC Private / Public Key Pair.\n");
                break;

            case EXPORT_KEYS:
                printf("Demo on Key-Pair Export command ------------------------------------------------------------------- \n");
                printf("Key-Pair are generated internally by CryptoASym component of Reader Library.\n\n");

                /* Check if KeyPair is generated or Loaded. */
                if(!IS_KEY_PAIR_AVAILABLE)
                {
                    printf("\tNo Key available for exporting.\n");
                    printf("\tPerform GenerateKeyPair or Load Key pair features prior performing ExportKey.\n\n");
                    break;
                }

                CHECK_SUCCESS(ExportKey(), TRUE);

                printf("\tSuccessfully Exported ECC Private / Public Key Pair.\n");
                break;

            case SIGN_MESSAGE:
                printf("Demo on Message Signing command ----------------------------------------------------------------- \n\n");

                /* Check if KeyPair is generated or Loaded. */
                if(!IS_KEY_PAIR_AVAILABLE)
                {
                    printf("\tNo Key available for Message Signing.\n");
                    printf("\tPerform GenerateKeyPair or Load Key pair features prior performing Message Signing.\n\n");
                    break;
                }

                CHECK_SUCCESS(Message_Sign(aMessage, (uint16_t) sizeof(aMessage), TRUE), TRUE);

                printf("\tSuccessfully Signed using ECC Private / Public Key Pair.\n");
                break;

            case VERIFY_MESSAGE:
                printf("Demo on Message Verification command ------------------------------------------------------------ \n");
                printf("Message Signature will be performed first and then using the Message and Signature Verification will be performed.\n\n");

                /* Check if KeyPair is generated or Loaded. */
                if(!IS_KEY_PAIR_AVAILABLE)
                {
                    printf("\tNo Key available for exporting.\n");
                    printf("\tPerform GenerateKeyPair or Load Key pair features prior performing Signature Verification.\n\n");
                    break;
                }

                CHECK_SUCCESS(Message_Verify(aMessage, (uint16_t) sizeof(aMessage)), TRUE);
                break;

            case COMPUTE_SHARED_SECRET:
                printf("Demo on Computation of Shared Secret (ECDH) command --------------------------------------------------------- \n");
                printf("Computation of Shared Secret will be performed using Private Key on one part and Public Key of another party.\n\n");

                /* Check if KeyPair is generated or Loaded. */
                if(!IS_KEY_PAIR_AVAILABLE)
                {
                    printf("\tNo Key available for exporting.\n");
                    printf("\tPerform GenerateKeyPair or Load Key pair features prior performing Shared Secret Computation.\n\n");
                    break;
                }

                CHECK_SUCCESS(ComputeSharedSecret(), TRUE);

                printf("\tSuccessfully Computed the Shared secret between two parties.\n");
                break;

            case LOAD_KEYPAIR_DIRECT:
                printf("Demo on Loading of Key-Pair directly -------------------------------------------------------------- \n");
                printf("Key are loaded directly to CryptoASym component.\n\n");

                IS_KEY_PAIR_AVAILABLE = FALSE;
                CHECK_SUCCESS(LoadKeys_Direct(aPrivateKey, (uint16_t)sizeof(aPrivateKey), aPublicKey, (uint16_t)sizeof(aPublicKey)), TRUE);
                IS_KEY_PAIR_AVAILABLE = TRUE;

                printf("\tSuccessfully loaded ECC Private / Public Key Pair.\n");
                printf("\tPerform ExportKey, Verification and Shared Secret demos to check the loaded keys.\n\n");
                break;

#ifdef NXPBUILD__PH_KEYSTORE_ASYM
            case LOAD_KEYPAIR_KEYSTORE:
                printf("Demo on Loading of Key-Pair from KeyStore --------------------------------------------------------- \n");
                printf("Key are loaded from KeyStore component.\n\n");

                IS_KEY_PAIR_AVAILABLE = FALSE;
                CHECK_SUCCESS(LoadKeys(KEY_ADDRESS, KEY_PAIR_PRIVATE_KEY), TRUE);
                CHECK_SUCCESS(LoadKeys(KEY_ADDRESS, KEY_PAIR_PUBLIC_KEY), TRUE);
                IS_KEY_PAIR_AVAILABLE = TRUE;

                printf("\tSuccessfully loaded ECC Private / Public Key Pair.\n");
                printf("\tPerform ExportKey, Verification and Shared Secret demos to check the loaded keys.\n\n");
                break;
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */

            case VERIFY_SIGNATURE:
                printf("Demo to Verify Message-Signature using Public Key ------------------------------------------------- \n");

                IS_KEY_PAIR_AVAILABLE = FALSE;
                CHECK_SUCCESS(VerifySignature(), TRUE);
                break;

            default:
                printf("Invalid option selected.\n");
                break;
        }

        printf("--------------------------------------------------------------------------------------------------- \n");
        printf("\n\n");
        printf("Press any key to continue with the demonstration.\n");
        printf("Press r to return to previous demo description.\n");
        printf("Press x to exit the demonstration.\n");
        dwChoice = _getch();
    } while(END_LOOP(dwChoice));

    /* Return to previous demo description if requested */
    RETURN_IF_REQUESTED(dwChoice);

    /* Exit if requested by user. */
    EXIT_IF_REQUESTED(dwChoice);

    return PH_ERR_SUCCESS;
}

#ifdef NXPBUILD__PH_KEYSTORE_ASYM
phStatus_t UpdateKeystore_ASym()
{
    phStatus_t wStatus = 0;
    uint16_t wKeyInfo = 0;

    /* Load ECC Key-Pair in Binary format. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_FormatKeyEntry(pKeyStore, KEY_ADDRESS, PH_KEYSTORE_KEY_TYPE_ECC));

    wKeyInfo = (uint16_t) (PH_KEYSTORE_KEY_PAIR_PRIVATE | PH_KEYSTORE_CURVE_ID_SECP256R1);
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_SetKeyASym(pKeyStore, KEY_ADDRESS, KEY_PAIR_PRIVATE_KEY, PH_KEYSTORE_KEY_TYPE_ECC,
        wKeyInfo, aPrivateKey, (uint16_t) sizeof(aPrivateKey)));

    wKeyInfo = (uint16_t) (PH_KEYSTORE_KEY_PAIR_PUBLIC | PH_KEYSTORE_CURVE_ID_SECP256R1);
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_SetKeyASym(pKeyStore, KEY_ADDRESS, KEY_PAIR_PUBLIC_KEY, PH_KEYSTORE_KEY_TYPE_ECC,
        wKeyInfo, aPublicKey, (uint16_t) sizeof(aPublicKey)));

    return PH_ERR_SUCCESS;
}
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */

phStatus_t MessageDigest(uint8_t * pMessage, uint16_t wMsgLen)
{
    phStatus_t wStatus = 0;
    int8_t aStatus[256];
    uint16_t wStatMsgLen = sizeof(aStatus);
    int32_t dwStatusCode = 0;

    uint8_t aHash[64];
    uint16_t wHashLen = 0;

    (void) memset(aStatus, 0x00, wStatMsgLen);

    printf("\tMessage\n");
    printf("\t                 Hex: ");
    PrintData(pMessage, (uint16_t) (wMsgLen - 1), "%02X ", "\n");
    PrintASCII(pMessage, (uint16_t) wMsgLen, TRUE);
    printf("\n\n");

    wStatus = phCryptoASym_ComputeHash(pCryptoASym, PH_EXCHANGE_DEFAULT, PH_CRYPTOASYM_HASH_ALGO_SHA256, pMessage,
        wMsgLen, aHash, &wHashLen);

    /* Log the additional message. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        CHECK_SUCCESS(phCryptoASym_GetLastStatus(pCryptoASym, wStatMsgLen, aStatus, &dwStatusCode), FALSE);

        printf("\n");
        printf("StatusCode: %d\n", dwStatusCode);
        printf("%s", aStatus);
        printf("\n");
    }
    else
    {
        printf("\tHash\n");
        printf("\t                 Hex: ");
        PrintData(aHash, wHashLen, "%02X ", "\n");
        PrintASCII(aHash, (uint16_t) wHashLen, TRUE);
        printf("\n");
    }

    return wStatus;
}

phStatus_t GenerateKeyPair()
{
    phStatus_t wStatus = 0;
    int8_t aStatus[256];
    uint16_t wStatMsgLen = sizeof(aStatus);
    int32_t dwStatusCode = 0;

    (void) memset(aStatus, 0x00, wStatMsgLen);

    wStatus = phCryptoASym_ECC_GenerateKeyPair(pCryptoASym, PH_CRYPTOASYM_CURVE_ID_SECP256R1);

    /* Log the additional message. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        CHECK_SUCCESS(phCryptoASym_GetLastStatus(pCryptoASym, wStatMsgLen, aStatus, &dwStatusCode), FALSE);

        printf("\n");
        printf("StatusCode: %d\n", dwStatusCode);
        printf("%s", aStatus);
        printf("\n");
    }

    return wStatus;
}

phStatus_t ExportKey()
{
    uint8_t aKey[256];
    uint16_t wKeyLen = 0;
    uint16_t wOption = 0;
    uint8_t bCurveID = 0;

    /* Clear key buffer. */
    (void) memset(aKey, 0x00, sizeof(aKey));

    /* Export the Keys in Binary format -------------------------------------------------------------------------------------- */
    printf("\tExporting Keys in Uncompressed Binary format (BigEndian).\n");
    wOption = (uint16_t) PH_CRYPTOASYM_PRIVATE_KEY;
    CHECK_SUCCESS(phCryptoASym_ECC_ExportKey(pCryptoASym, wOption, (uint16_t)sizeof(aKey), &bCurveID, aKey, &wKeyLen), TRUE);
    printf("\t         Private Key: ");
    PrintData(aKey, wKeyLen, "%02X ", "\n\n");

    /* Clear key buffer. */
    (void) memset(aKey, 0x00, sizeof(aKey));

    wOption = (uint16_t) PH_CRYPTOASYM_PUBLIC_KEY;
    CHECK_SUCCESS(phCryptoASym_ECC_ExportKey(pCryptoASym, wOption, (uint16_t)sizeof(aKey), &bCurveID, aKey, &wKeyLen), TRUE);
    printf("\t          Public Key: ");
    PrintData(aKey, wKeyLen, "%02X ", "\n\n\n");

    /* Clear key buffer. */
    (void) memset(aKey, 0x00, sizeof(aKey));

    return PH_ERR_SUCCESS;
}

phStatus_t Message_Sign(uint8_t * pMessage, size_t wMsgLen, uint8_t bPrintLogs)
{
    if(bPrintLogs)
    {
        printf("\tMessage\n");
        printf("\t                 Hex: ");
        PrintData(pMessage, (uint16_t) (wMsgLen - 1), "%02X ", "\n");
        PrintASCII(pMessage, (uint16_t) wMsgLen, TRUE);
        printf("\n\n");
    }

    /* Clear the Buffers. */
    (void) memset(aSignature, 0x00, sizeof(aSignature));

    /*  Sign the message. */
    CHECK_SUCCESS(phCryptoASym_ECC_Sign(pCryptoASym, PH_EXCHANGE_DEFAULT, PH_CRYPTOASYM_HASH_ALGO_SHA256, pMessage, (uint16_t) wMsgLen,
        aSignature, &wSignLen), TRUE);

    if(bPrintLogs)
    {
        printf("\tSignature\n");
        printf("\t                 Hex: ");
        PrintData(aSignature, wSignLen, "%02X ", "\n");
        printf("\n");
    }

    return PH_ERR_SUCCESS;
}

phStatus_t Message_Verify(uint8_t * pMessage, size_t wMsgLen)
{
    phStatus_t wStatus = PH_ERR_SUCCESS;

    /*  Sign the message. */
    printf("\tSignature Generation");
    CHECK_SUCCESS(Message_Sign(pMessage, wMsgLen, FALSE), TRUE);
    printf(" Successful\n\n");

    /*  Verify the message. */
    printf("\tSignature Verification\n");

    printf("\tMessage\n");
    printf("\t                 Hex: ");
    PrintData(pMessage, (uint16_t) (wMsgLen - 1), "%02X ", "\n");
    PrintASCII(pMessage, (uint16_t) wMsgLen, TRUE);
    printf("\n\n");

    printf("\tSignature\n");
    printf("\t                 Hex: ");
    PrintData(aSignature, wSignLen, "%02X ", "\n");
    printf("\n");

    wStatus = phCryptoASym_ECC_Verify(pCryptoASym, PH_EXCHANGE_DEFAULT, PH_CRYPTOASYM_HASH_ALGO_SHA256, pMessage, (uint16_t) wMsgLen,
        aSignature, wSignLen);
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        printf("\tSignature Verification Failed\n");
        return wStatus;
    }
    printf("\tSignature Verification Successful\n");

    return PH_ERR_SUCCESS;
}

phStatus_t ComputeSharedSecret()
{
    phStatus_t wStatus = 0;
    int8_t aStatus[256];
    uint16_t wStatMsgLen = sizeof(aStatus);
    int32_t dwStatusCode = 0;

    uint16_t wPrivateKeyLen = (uint16_t)sizeof(aPrivateKey);
    uint16_t wPublicKeyLen = (uint16_t)sizeof(aPublicKey);

    uint16_t wOption = 0;
    uint8_t aSharesSecret[64];
    uint16_t wSharedSecretLen = 0;

    uint8_t bCurveID = 0;

    (void) memset(aStatus, 0x00, wStatMsgLen);
    (void) memset(aSharesSecret, 0x00, sizeof(aSharesSecret));

    wOption = (uint16_t) PH_CRYPTOASYM_PRIVATE_KEY;
    CHECK_SUCCESS(phCryptoASym_ECC_ExportKey(pCryptoASym, wOption, wPrivateKeyLen, &bCurveID, aPrivateKey, &wPrivateKeyLen), TRUE);

    printf("\t         Private Key: ");
    PrintData(aPrivateKey, wPrivateKeyLen, "%02X ", "\n\n");

    printf("\t          Public Key: ");
    PrintData(aPublicKey, wPublicKeyLen, "%02X ", "\n\n");

    wStatus = phCryptoASym_ECC_SharedSecret(pCryptoASym, PH_KEYSTORE_CURVE_ID_SECP256R1, aPublicKey, wPublicKeyLen, aSharesSecret,
        &wSharedSecretLen);

    /* Log the additional message. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        CHECK_SUCCESS(phCryptoASym_GetLastStatus(pCryptoASym, wStatMsgLen, aStatus, &dwStatusCode), FALSE);

        printf("\n");
        printf("StatusCode: %d\n", dwStatusCode);
        printf("%s", aStatus);
        printf("\n");
    }
    else
    {
        printf("\t       Shared Secret: ");
        PrintData(aSharesSecret, wSharedSecretLen, "%02X ", "\n\n\n");
    }

    return wStatus;
}

phStatus_t LoadKeys_Direct(uint8_t * pPrivateKey, uint16_t wPrivateKeyLen, uint8_t * pPublicKey, uint16_t wPublicKeyLen)
{
    uint16_t wOption = 0;

    printf("\tPrivate Key         : ");
    PrintData(pPrivateKey, wPrivateKeyLen, "%02X ", "\n");
    printf("\n");

    wOption = (uint16_t) ( PH_CRYPTOASYM_PRIVATE_KEY | PH_CRYPTOASYM_CURVE_ID_SECP256R1 );
    CHECK_SUCCESS(phCryptoASym_ECC_LoadKeyDirect(pCryptoASym, wOption, pPrivateKey, wPrivateKeyLen), TRUE);

    printf("\tPublic Key          : ");
    PrintData(pPublicKey, wPublicKeyLen, "%02X ", "\n");
    printf("\n");

    wOption = (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | PH_CRYPTOASYM_CURVE_ID_SECP256R1 );
    CHECK_SUCCESS(phCryptoASym_ECC_LoadKeyDirect(pCryptoASym, wOption, pPublicKey, wPublicKeyLen), TRUE);

    return PH_ERR_SUCCESS;
}

#ifdef NXPBUILD__PH_KEYSTORE_ASYM
phStatus_t LoadKeys(uint16_t wKeyNo, uint16_t wKeyPair)
{
    uint16_t wOption = 0;

    wOption = (uint16_t) ((wKeyPair + 1) << 12);
    CHECK_SUCCESS(phCryptoASym_ECC_LoadKey(pCryptoASym, wOption, wKeyNo, wKeyPair), TRUE);

    return PH_ERR_SUCCESS;
}
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */

phStatus_t VerifySignature()
{
    phStatus_t wStatus = PH_ERR_SUCCESS;
    uint16_t wOption = 0;

    uint8_t aPubKey[] =
    {
        0x04, 0xE9, 0x34, 0xC6, 0x1D, 0xD4, 0xCA, 0x2B, 0x84, 0xC7, 0xFF, 0x63, 0xED, 0x8D, 0x9B, 0x59,
        0x91, 0x61, 0xA0, 0x91, 0xF7, 0x73, 0xEF, 0x98, 0xFC, 0xD2, 0xB1, 0x46, 0xBF, 0xDE, 0xBC, 0x1E,
        0xB5, 0xFD, 0xF9, 0xB7, 0x0F, 0xC5, 0x4A, 0x08, 0x8E, 0x90, 0x4D, 0xC8, 0x97, 0xB5, 0x73, 0xC8,
        0xCE, 0x75, 0xEC, 0xC6, 0x3F, 0xB1, 0xDF, 0xE0, 0x5B, 0x07, 0xFD, 0x6C, 0xFF, 0x2C, 0xC7, 0x7A,
        0xBC
    };
    uint16_t wPubKeyLen = (uint16_t) sizeof(aPubKey);

    uint8_t aMsg[] = { 0x50, 0x2D, 0x32, 0x35, 0x36, 0x20, 0x45, 0x43, 0x43, 0x20, 0x43, 0x75, 0x72, 0x76, 0x65 };
    uint16_t wMsgLen_Tmp = (uint16_t) sizeof(aMsg);

    uint8_t aSign[] =
    {
        0xD9, 0xE2, 0x04, 0xD7, 0x4F, 0xAD, 0x45, 0xD1, 0x0C, 0x32, 0xBC, 0xB9, 0x91, 0xD8, 0x69, 0xA8,
        0x9D, 0xF6, 0x72, 0x1F, 0xAD, 0xE8, 0xCA, 0x68, 0x6F, 0xF1, 0xC1, 0xC1, 0x9B, 0x1A, 0xE8, 0xA7,
        0xE7, 0xFC, 0x21, 0x65, 0x5B, 0xFD, 0x5E, 0xD7, 0x51, 0x85, 0x4A, 0xAF, 0x4D, 0xE3, 0x16, 0x85,
        0xC1, 0x30, 0x66, 0x64, 0xC2, 0xA4, 0x12, 0xEE, 0x9E, 0xEE, 0x95, 0x9B, 0xFA, 0x4A, 0x2B, 0x13
    };
    wSignLen = (uint16_t) sizeof(aSign);

    printf("\tPublic Key          : ");
    PrintData(aPubKey, wPubKeyLen, "%02X ", "\n");
    printf("\n");

    wOption = (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | PH_CRYPTOASYM_CURVE_ID_SECP256R1);
    CHECK_SUCCESS(phCryptoASym_ECC_LoadKeyDirect(pCryptoASym, wOption, aPubKey, wPubKeyLen), TRUE);

    printf("\tMessage\n");
    printf("\t                 Hex: ");
    PrintData(aMsg, wMsgLen_Tmp, "%02X ", "\n");
    PrintASCII(aMsg, wMsgLen_Tmp, FALSE);
    printf("\n\n");

    printf("\tSignature\n");
    printf("\t                 Hex: ");
    PrintData(aSign, wSignLen, "%02X ", "\n");
    printf("\n");

    wStatus = phCryptoASym_ECC_Verify(pCryptoASym, PH_EXCHANGE_DEFAULT, PH_CRYPTOASYM_HASH_ALGO_NOT_APPLICABLE, aMsg, wMsgLen_Tmp,
        aSign, wSignLen);
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        printf("\tSignature Verification Failed\n");
        return wStatus;
    }
    printf("\tSignature Verification Successful\n");

    return PH_ERR_SUCCESS;
}

#endif /* NXPBUILD__PH_CRYPTOASYM */
