/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example application for MIFARE DUOX.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7230 $
 * $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
 */

#include "Example-MfDUOX.h"

#include <mbedtls/x509_crt.h>

#ifdef NXPBUILD__PH_LOG
    phLog_RegisterEntry_t stLogRegisterEntries[16];
    phLog_LogEntry_t stLogEntries[48];
#endif /* NXPBUILD__PH_LOG */

int __cdecl main()
{
    phStatus_t wStatus = 0;
    int32_t dwChoice = 0;

    uint8_t aAtr[256];
    uint8_t aSeed[16];
    uint8_t aTx_HalBuffer[512];
    uint8_t aRx_HalBuffer[512];

    uint8_t aPortName[] = { '1','2','7','.','0','.','0','.','1','\0','1','5','2','0','0','\0','0','0','0','2','\0' };
    uint8_t aTx_SocketBuffer[0x400];
    uint8_t aRx_SocketBuffer[0x200];
    uint8_t aTx_SimBuffer[0x400];
    uint8_t aRx_SimBuffer[0x400];
    uint8_t aInt_SimBuffer[0x400];

    uint8_t aComPort[20];
    uint8_t aEnumBuff[512];
    uint8_t aCmdBuff[512];
    uint8_t aPrsBuff[512];
    uint8_t aCryptoASym_Buffer[512];
    uint8_t aTmi_Buffer[255];

    phKeyStore_Sw_KeyEntry_t aKeyEntry[NUM_KEY_ENTRY];
    phKeyStore_Sw_KeyVersionPair_t aKeyVersion[NUM_KEY_ENTRY * NUM_KEY_VERSION_PAIR];
    phKeyStore_Sw_KUCEntry_t aKeyUsageCounter[NUM_KUC_ENTRY];

    phbalReg_SocketWin_DataParams_t stBal_Socket;
    phhalHw_RdCardSim_DataParams_t stHal_Sim;

    phbalReg_SerialWin_DataParams_t stBal_Serial;
    phhalHw_DUT_DataParams_t stHal_DUT;

    phbalReg_PcscWin_DataParams_t stBal_Pcsc;
    phhalHw_Rd710_DataParams_t    stHal_Rd710;

    phCryptoASym_mBedTLS_DataParams_t stCryptoASym;
    phCryptoSym_Sw_DataParams_t stCryptoEnc;
    phCryptoSym_Sw_DataParams_t stCryptoMac;
    phCryptoSym_Sw_DataParams_t stCryptoSymRng;
    phCryptoRng_Sw_DataParams_t stCryptoRng;

    phalVca_Sw_IidTableEntry_t stIidTableEntry[0x13];   /**< Pointer to the IID Table storage for the layer. */
    phalVca_Sw_CardTableEntry_t stCardTableEntry[0x13]; /**< Pointer to the Card Table storage for the layer. */

    printf("Press %d to connect with RD710 (Pegoda-2) reader.\n", RD710);
    printf("Press %d to connect with RD730 (Pegoda-3) reader.\n", RD730);
    printf("Press %d to connect with Simulator.\n", SIMULATOR);
    printf("Press x to exit the application.\n");
    printf("Enter the option and press Enter to connect with the selected reader - ");
    dwChoice = _getch();

    printf("%c\n\n", dwChoice);

    /* Exit if requested. */
    EXIT_IF_REQUESTED(dwChoice);

    /* Update Reader flag. */
    dwChoice = dwChoice - 0x30;

    switch(dwChoice)
    {
        case RD710:
            /* Initialize the reader BAL component. */
            wStatus = phbalReg_PcscWin_Init(&stBal_Pcsc, sizeof(stBal_Pcsc), aAtr, 256);
            CHECK_SUCCESS(wStatus);

            /* Set the BAL Communication Configuration values */
            wStatus = phbalReg_SetConfig(&stBal_Pcsc, PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,
                PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED);
            CHECK_SUCCESS(wStatus);

            wStatus = phbalReg_SetConfig(&stBal_Pcsc, PHBAL_REG_PCSCWIN_CONFIG_SHARE, PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);
            CHECK_SUCCESS(wStatus);

            /* Select Port to be used (no SAM reader) */
            wStatus = phbalReg_SetPort(&stBal_Pcsc, (uint8_t *) PCSC_READER_NO_SAM_NAME);
            CHECK_SUCCESS(wStatus);

            /* Open the PCSC port */
            wStatus = phbalReg_OpenPort(&stBal_Pcsc);
            CHECK_SUCCESS(wStatus);

            /* Initialize the Rd710 HAL component. */
            wStatus = phhalHw_Rd710_Init(&stHal_Rd710, sizeof(stHal_Rd710), &stBal_Pcsc, 0, aTx_HalBuffer, sizeof(aTx_HalBuffer),
                aRx_HalBuffer, sizeof(aRx_HalBuffer));
            CHECK_SUCCESS(wStatus);

            /* Standard Rd710 Command Init. */
            wStatus = (phStatus_t) phhalHw_Rd710_Cmd_InitReader(&stHal_Rd710);
            CHECK_SUCCESS(wStatus);

            /* Set default reader as RD710. */
            pHal_Generic = &stHal_Rd710;
            break;

        case RD730:
            /* Add escape sequence. */
            aComPort[0] = '\\'; aComPort[1] = '\\'; aComPort[2] = '.'; aComPort[3] = '\\';

            /* Get the Comport to use. */
            printf("\n");
            printf("Please provide Pegoda 3 Comport number to use - ");
            scanf("%s", &aComPort[4U]);

            /* Initialize and connect to Contactless Bal component. */
            wStatus = phbalReg_SerialWin_Init(&stBal_Serial, sizeof(phbalReg_SerialWin_DataParams_t),
                sizeof(aEnumBuff), aEnumBuff);
            CHECK_SUCCESS(wStatus);

            wStatus = phbalReg_SetPort(&stBal_Serial, aComPort);
            CHECK_SUCCESS(wStatus);

            wStatus = phbalReg_OpenPort(&stBal_Serial);
            CHECK_SUCCESS(wStatus);

            /* Set the BitRate for BAL. */
            wStatus = phbalReg_SetConfig(&stBal_Serial, PHBAL_REG_SERIALWIN_CONFIG_BITRATE,
                PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200);
            CHECK_SUCCESS(wStatus);

            /* Initialize the reader HAL component. */
            wStatus = phhalHw_DUT_Init(&stHal_DUT, sizeof(stHal_DUT), &stBal_Serial, 0, aTx_HalBuffer,
                sizeof(aTx_HalBuffer), aRx_HalBuffer, sizeof(aRx_HalBuffer));
            CHECK_SUCCESS(wStatus);

            wStatus = phhalHw_DUT_InitReader(&stHal_DUT, PHHAL_HW_DUT_INIT_CONTACTLESS);
            CHECK_SUCCESS(wStatus);

            /* Set default reader as RD730. */
            pHal_Generic = &stHal_DUT;
            break;

        case SIMULATOR:
            /* Initialize the reader BAL component */
            wStatus = phbalReg_SocketWin_Init(&stBal_Socket, sizeof(stBal_Socket), 0, sizeof(aTx_SocketBuffer),
                aTx_SocketBuffer, sizeof(aRx_SocketBuffer), aRx_SocketBuffer);
            CHECK_SUCCESS(wStatus);

            wStatus = phbalReg_SetPort(&stBal_Socket, aPortName);
            CHECK_SUCCESS(wStatus);

            /* Initialize the reader HAL component. */
            wStatus = phhalHw_RdCardSim_Init(&stHal_Sim, sizeof(stHal_Sim), &stBal_Socket, &stKeyStore, aTx_SimBuffer,
                sizeof(aTx_SimBuffer), aRx_SimBuffer, sizeof(aRx_SimBuffer), aInt_SimBuffer, sizeof(aInt_SimBuffer));
            CHECK_SUCCESS(wStatus);

            /* Open the reader port. */
            wStatus = phbalReg_OpenPort(&stBal_Socket);
            CHECK_SUCCESS(wStatus);

            /* Set the default reader as Rd70x. */
            pHal_Generic = &stHal_Sim;

            wIsSimulator = PH_ON;
            break;

        default:
            printf("\n\n");
            printf("Invalid option selected.\n");
            printf("Press any key to exit with the demonstration.\n");
            _getch();
            exit(0);
            break;
    }

    /* Initialize software KeyStore component. */
    wStatus = phKeyStore_Sw_Init(&stKeyStore, sizeof(stKeyStore), aKeyEntry, NUM_KEY_ENTRY, aKeyVersion,
        NUM_KEY_VERSION_PAIR, aKeyUsageCounter, NUM_KUC_ENTRY);
    CHECK_SUCCESS(wStatus);

    /* Initialize CryptoASym. */
    wStatus = phCryptoASym_mBedTLS_Init(&stCryptoASym, sizeof(stCryptoASym), &stKeyStore, aCryptoASym_Buffer,
        sizeof(aCryptoASym_Buffer));
    CHECK_SUCCESS(wStatus);

    /* Initialize CryptoSym for encryption. */
    wStatus = phCryptoSym_Sw_Init(&stCryptoEnc, sizeof(stCryptoEnc), &stKeyStore);
    CHECK_SUCCESS(wStatus);

    /* Initialize CryptoSym for macing. */
    wStatus = phCryptoSym_Sw_Init(&stCryptoMac, sizeof(stCryptoMac), &stKeyStore);
    CHECK_SUCCESS(wStatus);

    /* Initialize CryptoSym for random number generation. */
    wStatus = phCryptoSym_Sw_Init(&stCryptoSymRng, sizeof(stCryptoSymRng), &stKeyStore);
    CHECK_SUCCESS(wStatus);

    /* Initialize Crypto for random number generation. */
    wStatus = phCryptoRng_Sw_Init(&stCryptoRng, sizeof(stCryptoRng), &stCryptoSymRng);
    CHECK_SUCCESS(wStatus);

    /* Rest the seed buffer to initiate random number generation. */
    memset(aSeed, 0x00, sizeof(aSeed));

    /* Initiate seed for random number generation. */
    wStatus = phCryptoRng_Seed(&stCryptoRng, aSeed, sizeof(aSeed));
    CHECK_SUCCESS(wStatus);

    /* Initialize ISO14443-3A component. */
    wStatus = phpalI14443p3a_Sw_Init(&stI14443p3a, sizeof(stI14443p3a), pHal_Generic);
    CHECK_SUCCESS(wStatus);

    /* Initialize ISO14443-4A component. */
    wStatus = phpalI14443p4a_Sw_Init(&stI14443p4a, sizeof(stI14443p4a), pHal_Generic);
    CHECK_SUCCESS(wStatus);

    /* Initialize ISO14443-4 generic component. */
    wStatus = phpalI14443p4_Sw_Init(&stI14443p4, sizeof(stI14443p4), pHal_Generic);
    CHECK_SUCCESS(wStatus);

    /* Initialize MIFARE component. */
    wStatus = phpalMifare_Sw_Init(&stMifare, sizeof(stMifare), pHal_Generic, &stI14443p4);
    CHECK_SUCCESS(wStatus);

    /* Initialize TMI utility. */
    memset(&aTmi_Buffer[0], 0x00, sizeof(aTmi_Buffer));
    wStatus = phTMIUtils_Init(&stTmiUtils, &aTmi_Buffer[0], sizeof(aTmi_Buffer));
    CHECK_SUCCESS(wStatus);

    /* Initialize VCA */
    wStatus = phalVca_Sw_Init(&stVca, sizeof(stVca), &stMifare, &stKeyStore, &stCryptoEnc, &stCryptoRng,
        stIidTableEntry, (uint16_t) sizeof(stIidTableEntry), stCardTableEntry, (uint16_t) sizeof(stCardTableEntry));
    CHECK_SUCCESS(wStatus);

    /* Disable Timing Mode. */
    if(wIsSimulator == PH_ON)
    {
        wStatus = phalVca_SetConfig(&stVca, PHAL_VCA_TIMING_MODE, PH_OFF);
        CHECK_SUCCESS(wStatus);
    }

    /* Initialize MIFARE DUOX */
    wStatus = phalMfDuoX_Sw_Init(&stMfDuoX, sizeof(stMfDuoX), &stMifare, &stKeyStore, &stCryptoASym, &stCryptoEnc,
        &stCryptoMac, &stCryptoRng, &stTmiUtils, &stVca, aCmdBuff, (uint16_t) sizeof(aCmdBuff), aPrsBuff,
        (uint16_t) sizeof(aPrsBuff));
    CHECK_SUCCESS(wStatus);

        /* Initialize logging */
#ifdef NXPBUILD__PH_LOG
/* Register the component for logging */
    wStatus = phLog_Init(MfDuoX_Log, stLogRegisterEntries, (uint16_t) (sizeof(stLogRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
    CHECK_SUCCESS(wStatus);

    wStatus = phLog_Register(pHal_Generic, stLogEntries, (uint16_t) (sizeof(stLogEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(wStatus);

    wStatus = phLog_Register(&stMfDuoX, stLogEntries, (uint16_t) (sizeof(stLogEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(wStatus);

    wStatus = phLog_Register(&stVca, stLogEntries, (uint16_t) (sizeof(stLogEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(wStatus);
#endif /* NXPBUILD__PH_LOG */

    /* Configure Hardware for Type-A cards. */
    wStatus = phhalHw_ApplyProtocolSettings(pHal_Generic, PHHAL_HW_CARDTYPE_ISO14443A);
    CHECK_SUCCESS(wStatus);

    /* Perform a FieldOn. */
    wStatus = phhalHw_FieldOn(pHal_Generic);
    CHECK_SUCCESS(wStatus);

    /* Add keys to the software KeyStore component. */
    wStatus = UpdateKeystore();
    CHECK_SUCCESS(wStatus);

    do
    {
        /* Clear the console. */
        system("cls");

        printf("Pre-Requisites.\n");
        printf("    Make sure MIFARE DUOX Product is configured with below Key Configuration before performing any Demo routines.\n");
        printf("        PICC Master Key Type: AES 128-Bit\n");
        printf("        PICC Master Key     : ");
        PRINT_DATA_NEWLINE(aAES128_Key, sizeof(aAES128_Key));
        printf("\n");

        printf("Press %d for Demo on ISOGeneral Authentication.\n", ISO_GENERAL_AUTHENTICATION);
        printf("Press %d for Demo on ISOGeneral Authentication (Final).\n", ISO_GENERAL_AUTHENTICATION_FINAL);
        printf("Press %d for Demo on Symmetric Authentication with AES 128-Bit.\n", AES128_AUTHENTICATION);
        printf("Press %d for Demo on Symmetric Authentication with AES 256-Bit.\n", AES256_AUTHENTICATION);
        printf("Press %d for Demo on Standard File Operation (Data Read and Write). \n", STANDARD_FILE_OPERATION);
        printf("Press %d for Demo on Value File Operation (Value Credit and Debit). \n", VALUE_FILE_OPERATION);
        printf("Press %d for Demo on Record File Operation (Record Read and Write). \n", RECORD_FILE_OPERATION);
        printf("Press %d for Demo on Transaction MAC  (TMC and TMV) feature. \n", TRANSACTION_MAC);
        printf("Press %d for Demo on Transaction Signature  (TMC and TSV) feature. \n", TRANSACTION_SIGNATURE);
        printf("Press %d for Demo on Secure Dynamic Messaging feature (SDM MAC). \n", SECURE_DYNAMIC_MESSAGING_MAC);
        printf("Press %d for Demo on Secure Dynamic Messaging feature (SDM Signature). \n", SECURE_DYNAMIC_MESSAGING_SIG);
        printf("Press %d for Demo on Originality Check feature. \n", ORIGINALITY_CHECK);
        printf("Press %d for Demo on GetVersion. \n", GET_VERSION);
        printf("Press %d for Demo on Proximity Check feature. \n", PROXIMITY_CHECK);
        printf("Press %d for Demo on EV Charging feature. \n", EV_CHARGING);
        printf("Press x to exit the demonstration. \n");
        printf("Enter the option and press Enter to perform the demo - ");
        scanf_s("%d", &dwChoice);

        printf("\n\n");

        /* Exit if requested. */
        EXIT_IF_REQUESTED(dwChoice);

        switch(dwChoice)
        {
            case ISO_GENERAL_AUTHENTICATION:
                printf("Demo on ISOGeneral Authentication ---------------------------------------------------------------------\n\n");
                Demo_ECCAuthentication(PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE);
                break;

            case ISO_GENERAL_AUTHENTICATION_FINAL:
                printf("Demo on ISOGeneral Authentication (Final) -------------------------------------------------------------\n\n");
                Demo_ECCAuthentication(PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE_FINAL);
                break;

            case AES128_AUTHENTICATION:
                printf("Demo on Symmetric Authentication with AES 128-Bit -----------------------------------------------------\n\n");
                printf("\tThis demonstration routine uses PICC Master Key configured to AES128 Key type with Key\n");
                printf("\thaving all zeros. If MIFARE DUOX PICC with right configuration is placed, \n");
                printf("\tpress any key to continue.\n");
                dwChoice = _getch();

                printf("\n");

                Demo_AESAuthentication(PH_OFF);
                break;

            case AES256_AUTHENTICATION:
                printf("Demo on Symmetric Authentication with AES 256-Bit -----------------------------------------------------\n\n");
                printf("\tThis demonstration routine uses PICC Master Key for Authentication.\n");
                printf("\tBelow are the steps performed to change PICC Master Key from AES128 to AES256.\n");
                printf("\t\t1. Authentication using PICC Master Key is carried out with AES 128 bit Key.\n");
                printf("\t\t2. PICC Master Key is changed AES128 bit key type to AES256 bit key type.\n");
                printf("\t\t3. Authentication using PICC Master Key is carried out with AES 256 bit Key.\n");
                printf("\n");

                Demo_AESAuthentication(PH_ON);
                break;

            case STANDARD_FILE_OPERATION:
                printf("Demo on Standard File Operation (Data Read and Write) -------------------------------------------------\n\n");
                Demo_StandardFileOperation();
                break;

            case VALUE_FILE_OPERATION:
                printf("Demo on Value File Operation (Value Credit and Debit) -------------------------------------------------\n\n");
                Demo_ValueFileOperation();
                break;

            case RECORD_FILE_OPERATION:
                printf("Demo on Record File Operation (Record Read and Write) -------------------------------------------------\n\n");
                Demo_RecordFileOperation();
                break;

            case TRANSACTION_MAC:
                printf("Demo on Transaction MAC feature -----------------------------------------------------------------------\n");
                printf("Here Transaction Counter (TMC) and Transaction MAC (TMV) will be returned by PICC\n");
                Demo_TrasactionMAC();
                break;

            case TRANSACTION_SIGNATURE:
                printf("Demo on Transaction Signature feature -----------------------------------------------------------------\n\n");
                printf("Here Transaction Counter (TMC) and Transaction Signature (TSV) will be returned by PICC\n");
                Demo_TrasactionSIG();
                break;

            case SECURE_DYNAMIC_MESSAGING_MAC:
                printf("Demo on Secure Dynamic Messaging feature with SDM MAC returned by PICC --------------------------------\n\n");
                Demo_SecureDynamicMessaging_MAC();
                break;

            case SECURE_DYNAMIC_MESSAGING_SIG:
                printf("Demo on Secure Dynamic Messaging feature with SDM Signature returned by PICC --------------------------\n\n");
                Demo_SecureDynamicMessaging_SIG();
                break;

            case ORIGINALITY_CHECK:
                printf("Demo on Originality Check feature ---------------------------------------------------------------------\n\n");
                Demo_OriginalityCheck();
                break;

            case GET_VERSION:
                printf("Demo on Get Version -----------------------------------------------------------------------------------\n\n");
                Demo_GetVersion();
                break;

            case PROXIMITY_CHECK:
                printf("Demo on Proximity Check feature -----------------------------------------------------------------------\n\n");
                Demo_ProximityCheck();
                break;

            case EV_CHARGING:
                printf("Demo on EV Charging feature ---------------------------------------------------------------------------\n\n");
                Demo_EVCharging();
                break;

            default:
                printf("Invalid option selected.\n");
                break;
        }

        printf("\n\n");
        printf("Press any key to continue with the demonstration.\n");
        printf("Press x to exit the demonstration.\n");
        dwChoice = _getch();
    } while((dwChoice != 'x') && (dwChoice != 'X'));

    /* Close the connection. */
    if(stBal_Pcsc.wId == (uint16_t) (PH_COMP_BAL | PHBAL_REG_PCSCWIN_ID))
    {
        wStatus = phbalReg_ClosePort(&stBal_Pcsc);
        CHECK_SUCCESS(wStatus);
    }

    if(stBal_Serial.wId == (uint16_t) (PH_COMP_BAL | PHBAL_REG_SERIALWIN_ID))
    {
        wStatus = phbalReg_ClosePort(&stBal_Serial);
        CHECK_SUCCESS(wStatus);
    }

    if(stBal_Socket.wId == (uint16_t) (PH_COMP_BAL | PHBAL_REG_SOCKETWIN_ID))
    {
        wStatus = phbalReg_ClosePort(&stBal_Socket);
        CHECK_SUCCESS(wStatus);
    }

    return PH_ERR_SUCCESS;
}

phStatus_t AddKey_ASymm(uint8_t bKeyNo, uint8_t bKeyPos, uint16_t wKeyPair, uint8_t* pKey, uint8_t bKeyLen)
{
    phStatus_t wStatus = 0;
    uint16_t wKeyInfo = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_FormatKeyEntry(&stKeyStore, bKeyNo, PH_KEYSTORE_KEY_TYPE_ECC));

    wKeyInfo = (uint16_t) (wKeyPair | PH_KEYSTORE_CURVE_ID_SECP256R1);
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_SetKeyASym(&stKeyStore, bKeyNo, bKeyPos, PH_KEYSTORE_KEY_TYPE_ECC,
        wKeyInfo, pKey, bKeyLen));

    return PH_ERR_SUCCESS;

}

phStatus_t AddKey_Symm(uint8_t bKeyNo, uint8_t bKeyVer, uint8_t* pKey, uint16_t wKeyType)
{
    phStatus_t wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_FormatKeyEntry(&stKeyStore, bKeyNo, wKeyType));
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_SetKeyAtPos(&stKeyStore, bKeyNo, 0x00, wKeyType,
        pKey, bKeyVer));

    return PH_ERR_SUCCESS;
}

phStatus_t UpdateKeystore()
{
    phStatus_t wStatus = 0;

    /* Add ASymmetric Keys --------------------------------------------------------------------------------------------------- */

    /* CA Root Key ----------------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, AddKey_ASymm(ECC_P256_HOST_CA_PUBLIC_KEY_ADDRESS, ECC_P256_HOST_CA_PUBLIC_KEY_POSITION,
        PH_KEYSTORE_KEY_PAIR_PUBLIC, aP256_HOST_CA_PublicKey, (uint8_t) sizeof(aP256_HOST_CA_PublicKey)));

    PH_CHECK_SUCCESS_FCT(wStatus, AddKey_ASymm(ECC_P256_DEVICE_CA_PUBLIC_KEY_ADDRESS, ECC_P256_DEVICE_CA_PUBLIC_KEY_POSITION,
        PH_KEYSTORE_KEY_PAIR_PUBLIC, aP256_DEVICE_CA_PublicKey, (uint8_t) sizeof(aP256_DEVICE_CA_PublicKey)));





    /* Host key(P-256) ------------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, AddKey_ASymm(ECC_P256_HOST_PRIVATE_KEY_ADDRESS, ECC_P256_HOST_PRIVATE_KEY_POSITION,
        PH_KEYSTORE_KEY_PAIR_PRIVATE, aP256_HOST_PrivateKey, (uint8_t) sizeof(aP256_HOST_PrivateKey)));





    /* Device key(P-256) ----------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, AddKey_ASymm(ECC_P256_DEVICE_PRIVATE_KEY_ADDRESS, ECC_P256_DEVICE_PRIVATE_KEY_POSITION,
        PH_KEYSTORE_KEY_PAIR_PRIVATE, aP256_DEVICE_PrivateKey, (uint8_t) sizeof(aP256_DEVICE_PrivateKey)));





    /* Add Symmetric Keys ---------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, AddKey_Symm(AES128_KEY_ADDRESS, AES128_KEY_VERSION, aAES128_Key,
        PH_KEYSTORE_KEY_TYPE_AES128));

    PH_CHECK_SUCCESS_FCT(wStatus, AddKey_Symm(AES256_KEY_ADDRESS, AES256_KEY_VERSION, aAES256_Key,
        PH_KEYSTORE_KEY_TYPE_AES256));

    return PH_ERR_SUCCESS;
}

void ErrorMessage(phStatus_t wStatus)
{
    if((wStatus == 0x00) || (wStatus == 0x90))
    {
        printf("Error Message    : Successful operation\n");
        printf("PICC Error       : %02Xh\n", wStatus);
    }
    else
    {
        if((wStatus & PH_COMP_MASK) == PH_COMP_CRYPTOASYM)
        {
            switch((wStatus & PH_ERR_MASK))
            {
                case PH_ERR_UNSUPPORTED_CURVE_ID:
                    printf("Error Message    : Curve not supported\n");
                    break;

                case PH_ERR_UNSUPPORTED_KEY_PAIR_TYPE:
                    printf("Error Message    : Key Pair not supported\n");
                    break;

                case PH_ERR_VERIFICATION_FAILED:
                    printf("Error Message    : Message - Signature verification failed\n");
                    break;
            }
        }

        if((wStatus & PH_COMP_MASK) == PH_COMP_AL_VCA)
        {
            switch((wStatus & PH_ERR_MASK))
            {
                case PHAL_VCA_ERR_CMD_INVALID:
                    printf("Error Message    : Command Invalid\n");
                    printf("PICC Error       : 0Bh\n");
                    break;

                case PHAL_VCA_ERR_FORMAT:
                    printf("Error Message    : Format of the command is not correct (e.g. too many or ");
                    printf("too few bytes)\n");
                    printf("PICC Error       : 0Ch\n");
                    break;

                case PHAL_VCA_ERR_AUTH:
                    printf("Error Message    : MAC Verification failure\n");
                    break;
            }
        }

        if((wStatus & PH_COMP_MASK) == PH_COMP_AL_MFDUOX)
        {
            switch((wStatus & PH_ERR_MASK))
            {
                case PHAL_MFDUOX_ERR_OPERATION_OK_LIM:
                    printf("Error Message    : Successful operation with limited functionality\n");
                    printf("PICC Error       : 01h\n");
                    break;

                case PHAL_MFDUOX_ERR_CMD_OVERFLOW:
                    printf("Error Message    : Too many commands in the session or transaction\n");
                    printf("PICC Error       : 07h\n");
                    break;

                case PHAL_MFDUOX_ERR_BNR:
                    printf("Error Message    : Invalid Block number: not existing in the implementation");
                    printf(" or not valid to target with this command\n");
                    printf("PICC Error       : 09h\n");
                    break;

                case PHAL_MFDUOX_ERR_CMD_INVALID:
                    printf("Error Message    : Command Invalid\n");
                    printf("PICC Error       : 0Bh\n");
                    break;

                case PHAL_MFDUOX_ERR_FORMAT:
                    printf("Error Message    : Format of the command is not correct (e.g. too many or ");
                    printf("too few bytes)\n");
                    printf("PICC Error       : 0Ch\n");
                    break;

                case PHAL_MFDUOX_ERR_GEN_FAILURE:
                    printf("Error Message    : Failure in the operation of the PD\n");
                    printf("PICC Error       : 0Fh\n");
                    break;

                case PHAL_MFDUOX_ERR_NO_CHANGES:
                    printf("Error Message    : No changes done to backup files\n");
                    printf("PICC Error       : 0Ch\n");
                    break;

                case PHAL_MFDUOX_ERR_NOT_SUPPORTED:
                    printf("Error Message    : Not Supported Error\n");
                    printf("PICC Error       : 0Dh\n");
                    break;

                case PHAL_MFDUOX_ERR_OUT_OF_EEPROM:
                    printf("Error Message    : Insufficient NV-Memory\n");
                    printf("PICC Error       : 0Eh\n");
                    break;

                case PHAL_MFDUOX_ERR_PICC_CRYPTO:
                    printf("Error Message    : Crypto error returned by PICC\n");
                    printf("PICC Error       : 1Eh\n");
                    break;

                case PHAL_MFDUOX_ERR_WEAK_FIELD:
                    printf("Error Message    : Field strength not sufficient to enable power harvesting ");
                    printf("for the targeted current / voltage level\n");
                    printf("PICC Error       : 1Fh\n");
                    break;

                case 0x0C:
                    printf("Error Message    : Length of command string invalid\n");
                    printf("PICC Error       : 7Eh\n");
                    break;

                case PHAL_MFDUOX_ERR_NO_SUCH_KEY:
                    printf("Error Message    : Invalid key number specified\n");
                    printf("PICC Error       : 40h\n");
                    break;

                case PHAL_MFDUOX_ERR_PERMISSION_DENIED:
                    printf("Error Message    : Current configuration/wStatus does not allow ");
                    printf("the requested command\n");
                    printf("PICC Error       : 9Dh\n");
                    break;

                case PHAL_MFDUOX_ERR_PARAMETER_ERROR:
                    printf("Error Message    : Parameter value error returned by PICC\n");
                    printf("PICC Error       : 9Eh\n");
                    break;

                case PHAL_MFDUOX_ERR_APPLICATION_NOT_FOUND:
                    printf("Error Message    : Requested AID not found on PICC\n");
                    printf("PICC Error       : A0h\n");
                    break;

                case 0x07:
                    printf("Error Message    : Authentication error\n");
                    printf("PICC Error       : AEh or 06h\n");
                    break;

                case 0x71:
                    printf("Error Message    : Successful operation with chanining response\n");
                    printf("PICC Error       : AFh\n");
                    break;

                case PHAL_MFDUOX_ERR_BOUNDARY_ERROR:
                    printf("Error Message    : Attempt to read/write data from/to beyond the ");
                    printf("files/record's limits\n");
                    printf("PICC Error       : BEh\n");
                    break;

                case PHAL_MFDUOX_ERR_COMMAND_ABORTED:
                    printf("Error Message    : Previous command not fully completed. Not all ");
                    printf("frames were requested or provided by the PCD\n");
                    printf("PICC Error       : CAh\n");
                    break;

                case PHAL_MFDUOX_ERR_CERTIFICATE:
                    printf("Error Message    : Reader certificate or CertAccessRights related error\n");
                    printf("PICC Error       : CEh\n");
                    break;

                case PHAL_MFDUOX_ERR_DUPLICATE:
                    printf("Error Message    : File/Application with same number already exists.\n");
                    printf("PICC Error       : DEh\n");
                    break;

                case PHAL_MFDUOX_ERR_FILE_NOT_FOUND:
                    printf("Error Message    : Specified file number does not exist\n");
                    printf("PICC Error       : F0h\n");
                    break;

                case PHAL_MFDUOX_ERR_DF_GEN_ERROR:
                    printf("Error Message    : MIFARE DUOX Generic error\n");
                    printf("PICC Error       : %02Xh\n", stMfDuoX.wAdditionalInfo);
                    break;

                case PHAL_MFDUOX_ERR_DF_7816_GEN_ERROR:
                    printf("Error Message    : MIFARE DUOX ISO 7816 Generic error\n");
                    printf("PICC Error       : %04Xh\n", stMfDuoX.wAdditionalInfo);
                    break;
            }
        }
    }
}

void ToBinary(uint8_t *pData, uint16_t *pDataLen)
{
    uint32_t dwData = 0;
    uint16_t wIndex = 0;
    uint16_t wIndex1 = 0;
    uint16_t wRem = 0;
    uint8_t bLen = 4;
    uint8_t aChunk[4];

    wRem = *pDataLen;
    for(wIndex = 0; wIndex < *pDataLen; wIndex += 4)
    {
        memset(aChunk, 0x00, 4);
        memcpy(aChunk, &pData[wIndex], bLen);
        dwData = (uint16_t) strtol((char *) aChunk, NULL, 16);

        if(wRem > 2)
            pData[wIndex1++] = (uint8_t) (dwData >> 8);
        pData[wIndex1++] = (uint8_t) dwData;

        bLen = (uint8_t) ((wRem > bLen) ? bLen : wRem);
        wRem -= bLen;
    }

    *pDataLen = wIndex1;
}

void PrintData(unsigned char* pBuffer, unsigned int dwLength, uint16_t wAddNewLine)
{
    unsigned int dwSplitCount = 0;
    unsigned int dwIndex1;
    unsigned int dwIndex2;
    unsigned int dwCount;

    unsigned int dwRem;
    unsigned int dwSplitLen = 32;

    if((pBuffer == NULL) || (dwLength == 0))
        printf("NULL\n");

    else
    {
        dwCount = dwSplitLen;
        dwRem = dwLength;

        dwSplitCount = (dwLength / dwSplitLen) + ((dwLength % dwSplitLen) ? 1 : 0);
        dwCount = (dwLength > dwSplitLen) ? dwSplitLen : dwLength;

        for(dwIndex1 = 0; dwIndex1 < dwSplitCount; ++dwIndex1)
        {
            for(dwIndex2 = 0; dwIndex2 < dwCount; ++dwIndex2)
                printf("%02X ", pBuffer[(dwIndex1 * dwSplitLen) + dwIndex2]);

            if(wAddNewLine != 0)
                printf("\n");

            if((dwSplitCount > 1) && (dwIndex1 < (dwSplitCount - 1)))
                printf("                                                              ");

            dwRem -= dwSplitLen;
            dwCount = (dwRem >= dwSplitLen) ? dwSplitLen : dwRem;
        }
    }
}





phStatus_t ActivateCard()
{
    phStatus_t wStatus;

    uint8_t bCardPlaced = 0;
    uint8_t bMoreCardsAvailable;
    uint8_t bCidEnabled;
    uint8_t bCid = 1;
    uint8_t bNadSupported;
    uint8_t bFwi;
    uint8_t bFsdi = 8;
    uint8_t bFsci;

    uint8_t aAts[256];

    /* Reset the field */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_FieldReset(pHal_Generic));

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Wait(pHal_Generic, PHHAL_HW_TIME_MILLISECONDS, 5));

    /* Activate Layer 3 card. In loop till a card is detected. */
    do
    {
        wStatus = phpalI14443p3a_ActivateCard(&stI14443p3a, NULL, 0x00, aUid, &bUid_Len, &bSak,
            &bMoreCardsAvailable);

        if((wStatus != PH_ERR_SUCCESS) && (!bCardPlaced))
        {
            printf("Place NTAG X DNA tag into field.\n");
            bCardPlaced = 1;
        }
    } while(wStatus != PH_ERR_SUCCESS);
    CHECK_SUCCESS(wStatus);

    /* Activate L4 */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalI14443p4a_ActivateCard(&stI14443p4a, bFsdi, bCid,
        PHPAL_I14443P4A_DATARATE_106, PHPAL_I14443P4A_DATARATE_106, aAts));

    /* Retrieve the ISO14443-4A protocol parameters. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalI14443p4a_GetProtocolParams(&stI14443p4a, &bCidEnabled,
        &bCid, &bNadSupported, &bFwi, &bFsdi, &bFsci));

    /* Set the ISO14443-4 protocol parameters. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalI14443p4_SetProtocol(&stI14443p4, bCidEnabled, bCid,
        bNadSupported, 0, bFwi, bFsdi, bFsci));

    return PH_ERR_SUCCESS;
}

phStatus_t Prepare_ISOGeneralAuthenticate(uint8_t bAuthType, uint8_t bPICC_KeyNo, uint16_t wDisplayInfo)
{
    phStatus_t wStatus = 0;
    uint8_t aConfig[3];

    uint8_t aCertLen[3] = { 0x00, 0x00, 0x00 };
    uint32_t dwCertLen = 0;

    uint8_t aAccessRights[2] = { 0x0F, 0x00 };
    uint8_t aKeyPolicy[2] = { 0x00 , 0x00 };
    uint8_t aCRLFileID[3] = { 0x00, 0x00, 0x00 };
    uint8_t aIssuer[17] = { 0x30, 0x0F, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03,
        0x55, 0x04, 0x2D, 0x03, 0x04, 0x00, 0x01, 0x23, 0x47 };

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    /* Update KeyPolicy */
    aKeyPolicy[0] = 0x80;       /* ECC Based Mutual Authentication */
    aKeyPolicy[0] |= 0x40;      /* ECC Based Transaction Signature */
    aKeyPolicy[0] |= 0x20;      /* ECC Based Secure Dynamic Messaging */

    /* Create Application */
    if(wDisplayInfo == PH_ON)
        printf("    Application Creation with ECC Support\n");
    PH_CHECK_SUCCESS_FCT(wStatus, CreateSelectApplication(APPLICATION_TYPE_ECC, wDisplayInfo));

    /* Perform Authentication. */
    printf("        %s", AUTH_STATE(PHAL_MFDUOX_EV2_AUTHENTICATED));
    PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, APP_MASTER_KEY));
    printf(": SUCCESS\n");

    /* Enable Ephemeral Public Key */
    if(bAuthType == PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE_FINAL)
    {
        /* Get the current SM Configuration */
        if(wDisplayInfo == PH_ON)
        {
            printf("    Get the existing Secure Messaging Configuration\n");
            printf("        Options                                             : 04 (Secure Messaging Configuration)\n");
        }
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_GetConfiguration(&stMfDuoX,
            PHAL_MFDUOX_SET_CONFIGURATION_SM_CONFIGURATION, 1, &pResponse,
            &wRespLen));

        if(wDisplayInfo == PH_ON)
        {
            printf("        Response                                            : ");
            PRINT_DATA_NEWLINE(pResponse, wRespLen);
            printf("        Status                                              : SUCCESS\n\n");
        }

        memcpy(aConfig, pResponse, wRespLen);

        /* Enable ISOSelectFile integrated ECC-based authentication */
        aConfig[1] |= 0x08;
        aConfig[2] = 0x0C;
        if(wDisplayInfo == PH_ON)
        {
            printf("    Update Secure Messaging Configuration\n");
            printf("        Options                                             : 04 (Secure Messaging Configuration)\n");
            printf("        Data                                                : ");
            PRINT_DATA_NEWLINE(aConfig, 3);
        }
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_SetConfiguration(&stMfDuoX,
            PHAL_MFDUOX_SET_CONFIGURATION_SM_CONFIGURATION, aConfig, 3));

        if(wDisplayInfo == PH_ON)
            printf("        Status                                              : SUCCESS\n\n");
    }

    if(wDisplayInfo == PH_ON)
        printf("\n");

    /* Create standard File for storing Device Certificate */
    aAccessRights[0] = 0x00;
    aAccessRights[1] = 0xE0;
    if(wDisplayInfo == PH_ON)
        printf("    Standard Data for saving Device Certificate\n");
    PH_CHECK_SUCCESS_FCT(wStatus, CreateStandardFile(STANDARD_DATA_FILE_ECC, PH_ON,
        PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights, wDisplayInfo));

    /* Write Certificate */
    dwCertLen = sizeof(aP256_DEVICE_Cert);
    memcpy(aCertLen, (uint8_t *) &dwCertLen, 3);
    PH_CHECK_SUCCESS_FCT(wStatus, Write(STANDARD_DATA_FILE_ECC, PHAL_MFDUOX_COMMUNICATION_PLAIN,
        PHAL_MFDUOX_CHAINING_ISO, 0, aCertLen, (uint16_t)sizeof(aCertLen), wDisplayInfo));

    PH_CHECK_SUCCESS_FCT(wStatus, Write(STANDARD_DATA_FILE_ECC, PHAL_MFDUOX_COMMUNICATION_PLAIN,
        PHAL_MFDUOX_CHAINING_ISO, 3, aP256_DEVICE_Cert, (uint16_t)sizeof(aP256_DEVICE_Cert),
        wDisplayInfo));

    /* Perform ManageCARootKey operation to store CA Root PublicKey and Issuer. */
    aAccessRights[0] = 0x0F;
    aAccessRights[1] = 0x00;
    if(wDisplayInfo == PH_ON)
    {
        printf("    Writing Host CA Public Key and Issuer to Device for Host certificate validation (ManageCARootKey)\n");
        printf("        Communication Mode                                  : 30 (FULL)\n");
        printf("        Key Number                                          : %02X\n", bPICC_KeyNo);
        printf("        Curve ID                                            : 0C (NIST P-256)\n");
        printf("        Access Rights                                       : ");
        PRINT_DATA_NEWLINE(aAccessRights, 2);
        printf("        Write Access                                        : 3E\n");
        printf("        Read Access                                         : 3F\n");
        printf("        CRL File                                            : 00\n");
        printf("        CRL File AID                                        : ");
        PRINT_DATA_NEWLINE(aCRLFileID, sizeof(aCRLFileID));

        printf("        Public Key                                          : ");
        PRINT_DATA_NEWLINE(aP256_HOST_CA_PublicKey, sizeof(aP256_HOST_CA_PublicKey));

        printf("        Issuer                                              : ");
        PRINT_DATA_NEWLINE(aIssuer, sizeof(aIssuer));
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ManageCARootKey(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL,
        bPICC_KeyNo, PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256, aAccessRights, 0x3E, 0x3F, 0x00, aCRLFileID,
        ECC_P256_HOST_CA_PUBLIC_KEY_ADDRESS, ECC_P256_HOST_CA_PUBLIC_KEY_POSITION, aIssuer,
        sizeof(aIssuer)));

    if(wDisplayInfo == PH_ON)
        printf("        Status                                              : SUCCESS\n\n");

    /* Load Device Private Key. */
    if(wDisplayInfo == PH_ON)
    {
        printf("    Importing Private key to Device for Authentication message signing (ManageKeyPair)\n");
        printf("        Communication Mode                                  : 30 (FULL)\n");
        printf("        Key Number                                          : %02X\n", bPICC_KeyNo);
        printf("        Option                                              : 01 (Import Private Key)\n");
        printf("        Curve ID                                            : 0Ch (NIST-P256)\n");
        printf("        Key Policy                                          : ");
        PRINT_DATA_NEWLINE(aKeyPolicy, 2);
        printf("        Write Access                                        : 30\n");

        printf("        Private Key                                         : ");
        PRINT_DATA_NEWLINE(aP256_DEVICE_PrivateKey, sizeof(aP256_DEVICE_PrivateKey));
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ManageKeyPair(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL,
        bPICC_KeyNo, PHAL_MFDUOX_TARGET_ACTION_IMPORT_PRIVATE_KEY, PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256,
        aKeyPolicy, 0x30, 0, ECC_P256_DEVICE_PRIVATE_KEY_ADDRESS, ECC_P256_DEVICE_PRIVATE_KEY_POSITION,
        &pResponse, &wRespLen));

    if(wDisplayInfo == PH_ON)
    {
        printf("        Response                                            : ");
        PRINT_DATA_NEWLINE(pResponse, wRespLen);

        printf("        Status                                              : SUCCESS\n\n");
    }

    return wStatus;
}

phStatus_t ExtractPublicKey(uint8_t * pCertificate, uint16_t wCertLen, uint8_t * pPubKey, uint16_t * pPubKeyLen)
{
    int32_t dwStatus = 0;
    uint8_t bIndex = 0;
    uint8_t bLen = 0;

    char aErrorMsg[200];

    mbedtls_x509_crt stCert;
    mbedtls_asn1_buf stCert_PubKey;

    mbedtls_x509_crt_init(&stCert);

    do
    {
        /* Parse one X.509 certificate in DER format */
        dwStatus = mbedtls_x509_crt_parse_der(&stCert, pCertificate, wCertLen);
        CHECK_SUCCESS_MBEDTLS(dwStatus);

        /* Fetch subject Public key from X.509 certificate. */
        stCert_PubKey = stCert.pk_raw;
        if(stCert_PubKey.p[bIndex] == 0x30U)
        {
            bIndex += 5U;
            bLen = (uint8_t) (bIndex + stCert_PubKey.p[bIndex] + stCert_PubKey.p[bIndex +
                stCert_PubKey.p[bIndex] + 2U] + 3U);

            bIndex = bLen;
            if(stCert_PubKey.p[bIndex] == 0x03U)
            {
                bIndex++;
                *pPubKeyLen = (uint16_t) (stCert_PubKey.p[bIndex] - 1);
                memcpy(&pPubKey[0U], &stCert_PubKey.p[bIndex + 2U], *pPubKeyLen);
            }
        }
    } while(0);

    mbedtls_x509_crt_free(&stCert);

    return PH_ERR_SUCCESS;
}

phStatus_t Authenticate(uint8_t bAuthState, uint8_t bPICC_KeyNo)
{
    phStatus_t wStatus = 0;
    uint8_t bKeyNo = AES128_KEY_ADDRESS;
    uint8_t bKeyVer = AES128_KEY_VERSION;
    uint8_t aOptsA[] = { 0x80, 0x04, 0x80, 0x00, STANDARD_DATA_FILE_ECC, ECC_AUTHENTICATION_KEY };

    uint8_t aE_PubB[65];
    uint16_t wE_PubB = 65;

    uint8_t aPubKey[65];
    uint16_t wPubKeyLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    uint8_t * pOptsB = NULL;
    uint8_t bOptsBLen = 0;

    uint8_t * pCertB = NULL;
    uint16_t wCertBLen = 0;

    uint8_t * pSigB = NULL;
    uint16_t wSigBLen = 0;

    if(bAuthState == PHAL_MFDUOX_EV2_AUTHENTICATED)
    {
        if(wIsAES256KeyType == PH_ON)
        {
            bKeyNo = AES256_KEY_ADDRESS;
            bKeyVer = AES256_KEY_VERSION;
        }
        else
        {
            bKeyNo = AES128_KEY_ADDRESS;
            bKeyVer = AES128_KEY_VERSION;
        }

        /* Perform EV2 First Authentication */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_AuthenticateEv2(&stMfDuoX, PHAL_MFDUOX_AUTH_FIRST,
            PHAL_MFDUOX_NO_DIVERSIFICATION, bKeyNo, bKeyVer, bPICC_KeyNo, NULL, 0, aPcdCap2In,
            (uint8_t) sizeof(aPcdCap2In), aPcdCap2Out, aPdCap2));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ISOGeneralAuthenticate(&stMfDuoX, PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE,
            ECC_AUTHENTICATION_KEY, ECC_P256_HOST_PRIVATE_KEY_ADDRESS, ECC_P256_HOST_PRIVATE_KEY_POSITION,
            aOptsA, (uint8_t) sizeof aOptsA, aP256_HOST_Cert, (uint16_t) sizeof aP256_HOST_Cert,
            NULL, 0, aE_PubB, &wE_PubB, &pResponse, &wRespLen));

        /* Remove the Constants */
        pResponse += 2;
        wRespLen -= 2;

        /* Extract OptsB, CertificateB and SignatureB from Response */
        pOptsB = pResponse;
        bOptsBLen = (uint8_t) (sizeof aOptsA + 1/* Length of OptsB */);

        pCertB = &pResponse[bOptsBLen];
        wCertBLen = (uint16_t) (wRespLen - (bOptsBLen + 64));

        pSigB = &pResponse[wRespLen - 64];
        wSigBLen = 64;

        /* Extract Device Public key from received Certificate */
        ExtractPublicKey(pCertB, wCertBLen, aPubKey, &wPubKeyLen);

        /* Complete ISOGeneral Authentication. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ISOGeneralAuthenticateVerify(&stMfDuoX,
            aPubKey, wPubKeyLen, pOptsB, bOptsBLen, aE_PubB, wE_PubB, pSigB, wSigBLen));
    }

    return wStatus;
}

phStatus_t CreateSelectApplication(uint8_t bAppType, uint16_t wDisplayInfo)
{
    phStatus_t wStatus = 0;
    uint8_t bOption = 0;
    uint8_t bKeySett1 = 0x0F, bKeySett2 = 0x00, bKeySett3 = 0x00;

    uint8_t aISOFileId_EccApp[2] = { 0x11, 0xE1 };
    uint8_t aISOFileId_EvcApp[2] = { 0xDF, 0x00 };

    uint8_t aKeySetValues[] = { 0x00, 0x05, 0x10, 0x00 };
    uint8_t bKeySetValues_Len = 0;

    uint8_t * pISOFileId = NULL;
    uint8_t * pISODFName = NULL;
    uint8_t bISODFName_Len = 0;

    /* Configure KeySetting 2 */
    bKeySett2 = (uint8_t) ((wIsAES256KeyType == PH_ON) ? 0xC0 : 0x80);
    bKeySett2 |= 0x20;  /* 2 byte File Identifiers is present */
    bKeySett2 |= 0x10;  /* Key Setting 3 is present */
    bKeySett2 |= 5;     /* NoOfKeys */

    /* Activate the card. */
    printf("        PICC Activation");
    PH_CHECK_SUCCESS_FCT(wStatus, ActivateCard());
    printf("                                     : SUCCESS\n");

    /* Perform Authentication. */
    PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, PICC_MASTER_KEY));

    /* Format the PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Format(&stMfDuoX));

    /* Set ISOFileID and ISODFName pointers */
    if(bAppType == APPLICATION_TYPE_ECC)
    {
        bOption = PHAL_MFDUOX_ISO_FILE_ID_DF_NAME_AVAILABLE;

        pAID = aAppID_ECC;
        pISOFileId = aISOFileId_EccApp;
        pISODFName = aISODfName_EccApp;
        bISODFName_Len = (uint8_t) sizeof(aISODfName_EccApp);
    }

    /* Set ISOFileID and ISODFName pointers */
    if(bAppType == APPLICATION_TYPE_EV_CHARGING)
    {
        bOption = PHAL_MFDUOX_ISO_FILE_ID_DF_NAME_AVAILABLE;

        bKeySett1 = 0x09;
        bKeySett2 = 0xB5;
        bKeySett3 = 0x04;
        bKeySetValues_Len = 4;

        pAID = aAppID_EVC;
        pISOFileId = aISOFileId_EvcApp;
        pISODFName = aISODfName_EvcApp;
        bISODFName_Len = (uint8_t) sizeof(aISODfName_EvcApp);

        aKeySetValues[2] = (uint8_t) ((wIsAES256KeyType == PH_ON) ? 0x20U : 0x10U);
    }

    /* Application Creation. */
    if(wDisplayInfo == PH_ON)
    {
        printf("        AppID                                               : ");
        PRINT_DATA_NEWLINE(pAID, 3);

        printf("        ISOFileID and DFNname                               : %02X (%s)\n", bOption,
            ((bOption == PHAL_MFDUOX_ISO_FILE_ID_DF_NAME_AVAILABLE) ? "Present" : "Not Present"));

        printf("        Key Setting 1                                       : %02X\n", bKeySett1);
        printf("            AppKeySettings Changable                        : PH_ON\n");
        printf("            File create/delete configuration                : PH_ON\n");
        printf("            File directory access configuration             : PH_ON\n");
        printf("            AppMasterKey Changeable                         : PH_ON\n");
        printf("\n");

        printf("        Key Setting 2                                       : %02X\n", bKeySett2);
        printf("            KeyType                                         : %02X (AES %s-Bit)\n", (bKeySett2 & 0xC0),
            ((bKeySett2 & 0xC0) ? "128" : "256"));

        printf("            ISO File information                            : %02X (%s)\n", (bKeySett2 & 0x20),
            ((bKeySett2 & 0x20) ? "Present" : "Not Present"));
        printf("            Key Setting 3                                   : %02X (%s)\n", (bKeySett2 & 0x10),
            ((bKeySett2 & 0x10) ? "Present" : "Not Present"));

        printf("            Number of Keys                                  : 05h\n");
        printf("\n");

        printf("        Key Setting 3                                       : %02X\n", bKeySett3);
        printf("            Application deletion with AppMasterKey          : 00 (Depending on PICC Key Settings)\n");
        printf("            Application specific Capability data            : 00 (Disabled)\n");
        printf("            Application specific VCProximityKey             : 00 (Disabled)\n");
        printf("            Application key sets                            : 00 (Disabled)\n\n");

        printf("        ISO File Identifier                                 : ");
        PRINT_DATA_NEWLINE(pISOFileId, 2);

        printf("        ISO DF Name                                         : ");
        PRINT_DATA_NEWLINE(pISODFName, bISODFName_Len);
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_CreateApplication(&stMfDuoX, bOption, pAID, bKeySett1,
        bKeySett2, bKeySett3, aKeySetValues, bKeySetValues_Len, pISOFileId, pISODFName, bISODFName_Len));

    if(wDisplayInfo == PH_ON)
        printf("        Status                                              : SUCCESS\n\n");
    else
    {
        printf("        Application Creation (AID: ");
        PRINT_DATA(pAID, 3);
        printf(")");
        printf("               : SUCCESS\n");
    }

    /* Select the appplication */
    printf("        Application Selection (AID: ");
    PRINT_DATA(pAID, 3);
    printf(")");
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP,
        pAID, NULL));
    printf("              : SUCCESS\n\n");

    return wStatus;
}

phStatus_t DeleteFile(uint8_t bFileID)
{
    phStatus_t wStatus = 0;
    uint8_t bIndex = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    /* Get the File ID's */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_GetFileIDs(&stMfDuoX, &pResponse, &wRespLen));

    /* Delete the file if exists */
    for(bIndex = 0; bIndex < (uint8_t) wRespLen; bIndex++)
    {
        if(pResponse[bIndex] == bFileID)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_DeleteFile(&stMfDuoX, bFileID));
        }
    }

    return wStatus;
}

phStatus_t CreateStandardFile(uint8_t bFileNo, uint16_t wIsISOFileID_Present, uint8_t bFileOption,
    uint8_t *pAccessRights, uint16_t wDisplayInfo)
{
    phStatus_t wStatus = 0;
    uint32_t dwFileSize = 0;
    uint8_t bOption = 0;

    uint8_t aISOFileID[2] = { 0xEF, 0x00 };
    uint8_t aFileSize[3];

    uint8_t * pISOFileId = aISOFileID;

    aISOFileID[1] = bFileNo;
    bOption = (uint8_t) wIsISOFileID_Present;

    /* Delete the file if Exists */
    PH_CHECK_SUCCESS_FCT(wStatus, DeleteFile(bFileNo));

    /* Set FileOption, AccessRights and FileSize*/
    if(bFileNo == STANDARD_DATA_FILE_ECC)
    {
        dwFileSize = 3 /* First 3 bytes for Certificate Length */ + sizeof(aP256_DEVICE_Cert);
        memcpy(aFileSize, (uint8_t *) &dwFileSize, 3);
    }
    else
    {
        dwFileSize = 256;
        memcpy(aFileSize, (uint8_t *) &dwFileSize, 3);
    }

    if(wDisplayInfo == PH_ON)
    {
        printf("        File Identifier (FileID)                            : %02X\n", bFileNo);
        printf("        ISO File Identifier (ISOFileID)                     : ");
        PRINT_DATA_NEWLINE(pISOFileId, 2);

        printf("        File Option                                         : %02X (%s)\n", bFileOption,
            FILE_OPTION(bFileOption));

        printf("        Access Rights                                       : ");
        PRINT_DATA_NEWLINE(pAccessRights, 2);

        printf("        File Size                                           : ");
        PRINT_DATA_NEWLINE(aFileSize, 3);
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_CreateStdDataFile(&stMfDuoX, bOption, bFileNo,
        pISOFileId, bFileOption, pAccessRights, aFileSize));

    if(wDisplayInfo == PH_ON)
        printf("        Status                                              : SUCCESS\n\n");

    return wStatus;
}

phStatus_t CreateValueFile(uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights)
{
    phStatus_t wStatus = 0;
    uint8_t aLowerLimit[] = { 0x00, 0x00, 0x00, 0x00 };
    uint8_t aUpperLimit[] = { 0xFF, 0xFF, 0xFF, 0x7F };
    uint8_t aValue[] = { 0x10, 0x00, 0x00, 0x00 };

    /* Delete the file if Exists */
    PH_CHECK_SUCCESS_FCT(wStatus, DeleteFile(bFileNo));

    printf("        File Identifier (FileID)                            : %02X\n", bFileNo);
    printf("        File Option                                         : %02X (%s)\n", bFileOption,
        FILE_OPTION(bFileOption));

    printf("        Access Rights                                       : ");
    PRINT_DATA_NEWLINE(pAccessRights, 2);

    printf("        Lower Limit                                         : ");
    PRINT_DATA_NEWLINE(aLowerLimit, sizeof(aLowerLimit));

    printf("        Upper Limit                                         : ");
    PRINT_DATA_NEWLINE(aUpperLimit, sizeof(aUpperLimit));

    printf("        Value                                               : ");
    PRINT_DATA_NEWLINE(aValue, sizeof(aValue));

    printf("        Limited Credit                                      : 00 (Disabled)\n");

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_CreateValueFile(&stMfDuoX, bFileNo, bFileOption,
        pAccessRights, aLowerLimit, aUpperLimit, aValue, (uint8_t) PHAL_MFDUOX_LIMITED_CREDIT_DISABLED));

    printf("        Status                                              : SUCCESS\n\n");

    return wStatus;
}

phStatus_t CreateRecordFile(uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights)
{
    phStatus_t wStatus = 0;
    uint8_t aISOFileID[2] = { 0x00, 0x00 };

    uint8_t aRecordSize[] = { 0x10, 0x00, 0x00 };
    uint8_t aMaxNoOfRec[] = { 0x04, 0x00, 0x00 };

    aISOFileID[0] = bFileNo;

    /* Delete the file if Exists */
    PH_CHECK_SUCCESS_FCT(wStatus, DeleteFile(bFileNo));

    printf("        File Identifier (FileID)                            : %02X\n", bFileNo);
    printf("        File Option                                         : %02X (%s)\n", bFileOption,
        FILE_OPTION(bFileOption));

    printf("        Access Rights                                       : ");
    PRINT_DATA_NEWLINE(pAccessRights, 2);

    printf("        Record Size                                         : ");
    PRINT_DATA_NEWLINE(aRecordSize, sizeof(aRecordSize));

    printf("        Maximum Number of Records                           : ");
    PRINT_DATA_NEWLINE(aMaxNoOfRec, sizeof(aMaxNoOfRec));

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_CreateLinearRecordFile(&stMfDuoX, PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE,
        bFileNo, aISOFileID, bFileOption, pAccessRights, aRecordSize, aMaxNoOfRec));

    printf("        Status                                              : SUCCESS\n\n");

    return wStatus;
}

phStatus_t CreateTransactionMACFile(uint8_t bTMKeyOption)
{
    phStatus_t wStatus = 0;

    uint8_t aAccessRights[] = { 0xFE, 0xEF };

    /* Delete the file if Exists */
    PH_CHECK_SUCCESS_FCT(wStatus, DeleteFile(TRANSACTION_MAC_FILE));

    printf("    Transaction File creation\n");
    printf("        File Identifier (FileID)                            : %02X\n", TRANSACTION_MAC_FILE);
    printf("        File Option                                         : %02X (%s)\n", PHAL_MFDUOX_FILE_OPTION_PLAIN,
        FILE_OPTION(PHAL_MFDUOX_FILE_OPTION_PLAIN));

    printf("        Access Rights                                       : ");
    PRINT_DATA_NEWLINE(aAccessRights, 2);

    printf("        TMIExclFileMap                                      : NULL\n");
    printf("        TMKey Option                                        : %02X (AES128 KeyType, %s)\n",
        bTMKeyOption, ((bTMKeyOption & PHAL_MFDUOX_TMKEY_MODE_TRANSACTION_SIGNATURE)
            ? "Transaction Signature" : "Transaction MAC"));

    if(bTMKeyOption & PHAL_MFDUOX_TMKEY_MODE_TRANSACTION_SIGNATURE)
        printf("        Transation Signature Key Number                     : %02X\n", ECC_TSIG_KEY);
    else
    {
        printf("        Transation MAC Key                                  : ");
        PRINT_DATA_NEWLINE(aAES128_Key, sizeof(aAES128_Key));

        printf("        Transation MAC Key Version                          : 00\n");
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_CreateTransactionMacFile(&stMfDuoX,
        PHAL_MFDUOX_CREATE_TMAC_FILE_DIVERSIFICATION_OFF, TRANSACTION_MAC_FILE,
        PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights, AES128_KEY_ADDRESS,
        bTMKeyOption, aAES128_Key, AES128_KEY_VERSION, NULL, 0, NULL,
        ECC_TSIG_KEY));

    printf("        Status                                              : SUCCESS\n\n");

    return wStatus;
}

phStatus_t Write(uint8_t bFileNo, uint8_t bCommOption, uint8_t bChaining, uint32_t dwOffset,
    uint8_t * pData, uint32_t dwDataLen, uint16_t wDisplayInfo)
{
    phStatus_t wStatus = 0;

    uint8_t aOffset[3] = { 0x00, 0x00, 0x00 };
    uint8_t aDataLen[3] = { 0x00, 0x00, 0x00 };

    memcpy(aOffset, (uint8_t *) & dwOffset, 3);
    memcpy(aDataLen, (uint8_t *) & dwDataLen, 3);

    if(wDisplayInfo == PH_ON)
    {
        printf("    Write Operation\n");
        printf("        File Identifier (FileID)                            : %02X\n", bFileNo);
        printf("        Communication Mode                                  : %02X (%s)\n", bCommOption,
            COMM_OPTION(bCommOption));

        printf("        Chaining Mode                                       : %02X\n", bChaining);
        printf("        CRL Version                                         : FFFF\n");
        printf("        Offset                                              : ");
        PRINT_DATA_NEWLINE(aOffset, 3);

        printf("        Length of Data to Write                             : ");
        PRINT_DATA_NEWLINE(aDataLen, 3);

        printf("        Data to Write                                       : ");

        PRINT_DATA_NEWLINE(pData, dwDataLen);
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_WriteData(&stMfDuoX, bCommOption, bChaining, bFileNo,
        0xFFFF, aOffset, pData, aDataLen));

    if(wDisplayInfo == PH_ON)
        printf("        Status                                              : SUCCESS\n\n");

    return wStatus;
}

phStatus_t ReadData(uint8_t bFileNo, uint8_t bCommOption, uint8_t bChaining, uint32_t dwOffset,
    uint32_t dwDataLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint16_t wDisplayInfo)
{
    phStatus_t wStatus = 0;

    uint8_t aOffset[3] = { 0x00, 0x00, 0x00 };
    uint8_t aDataLen[3] = { 0x00, 0x00, 0x00 };

    memcpy(aOffset, (uint8_t *) & dwOffset, 3);
    memcpy(aDataLen, (uint8_t *) & dwDataLen, 3);

    if(wDisplayInfo == PH_ON)
    {
        printf("    Read Operation\n");
        printf("        File Identifier (FileID)                            : %02X\n", bFileNo);
        printf("        Communication Mode                                  : %02X (%s)\n", bCommOption,
            COMM_OPTION(bCommOption));

        printf("        Chaining Mode                                       : %02X\n", bChaining);
        printf("        Offset                                              : ");
        PRINT_DATA_NEWLINE(aOffset, 3);

        printf("        Length of Data to Read                              : ");
        PRINT_DATA_NEWLINE(aDataLen, 3);
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ReadData(&stMfDuoX, bCommOption, bChaining,
        bFileNo, aOffset, aDataLen, ppResponse, pRespLen));

    if(wDisplayInfo == PH_ON)
    {
        printf("        Data Read                                           : ");
        PRINT_DATA_NEWLINE(*ppResponse, *pRespLen);

        printf("        Status                                              : SUCCESS\n\n");
    }

    return wStatus;
}

phStatus_t ChangeFileSettings(uint8_t bFileOption, uint8_t *pAccessRights)
{
    phStatus_t wStatus = 0;
    uint8_t bOffset = 0;
    uint8_t bIsVCUID_Present = 0;
    uint8_t bIsSDMRdCtr_Present = 0;
    uint8_t bIsPICCData_Present = 0;
    uint8_t bIsSDMEncData_Present = 0;

    uint8_t aAddInfo[25];
    uint8_t bAddInfoLen = 0;

    /* Set the local variables */
    bIsVCUID_Present = (uint8_t) ((aSDMInfo[0] & 0x80) && (aSDMInfo[2] & 0xE0));
    bIsSDMRdCtr_Present = (uint8_t) ((aSDMInfo[0] & 0x40) && (aSDMInfo[2] & 0xE0));
    bIsPICCData_Present = (uint8_t) !((aSDMInfo[2] & 0xE0) || (aSDMInfo[2] & 0xF0));
    bIsSDMEncData_Present = (uint8_t) (aSDMInfo[0] & 0x10);

    /* Copy SDM File Option and SDM AccessRights */
    memcpy(&aAddInfo[bAddInfoLen], aSDMInfo, 3);
    bAddInfoLen += 3;

    /* Copy VCIID Offset */
    if(bIsVCUID_Present || bIsPICCData_Present)
    {
        memcpy(&aAddInfo[bAddInfoLen], &aSDMInfo[3], 3);
        bAddInfoLen += 3;
    }

    /* Copy SDM Read Ctr Offset */
    if(bIsSDMRdCtr_Present)
    {
        memcpy(&aAddInfo[bAddInfoLen], &aSDMInfo[6], 3);
        bAddInfoLen += 3;
    }

    /* Copy SDM MAC Input Offset */
    memcpy(&aAddInfo[bAddInfoLen], &aSDMInfo[9], 3);
    bAddInfoLen += 3;

    /* Copy SDM File Offset and SDM File Lenght */
    if(bIsSDMEncData_Present)
    {
        memcpy(&aAddInfo[bAddInfoLen], &aSDMInfo[12], 3);
        bAddInfoLen += 3;

        memcpy(&aAddInfo[bAddInfoLen], &aSDMInfo[15], 3);
        bAddInfoLen += 3;
    }

    /* Copy SDM MAC Offset */
    memcpy(&aAddInfo[bAddInfoLen], &aSDMInfo[18], 3);
    bAddInfoLen += 3;

    printf("        Change File Setting to enable SDM Feature\n");
    printf("            Option                                          : %02X (%s)\n",
        PHAL_MFDUOX_COMMUNICATION_PLAIN, COMM_OPTION(PHAL_MFDUOX_COMMUNICATION_PLAIN));

    printf("            File Identifier (FileID)                        : %02X\n", STANDARD_DATA_FILE);
    printf("            File Option                                     : %02X (%s, ",
        bFileOption, FILE_OPTION(bFileOption & 0x03));
    printf("Secure Dynamic Messaging and Mirroring = Enabled)\n");

    printf("            Access Rights                                   : ");
    PRINT_DATA_NEWLINE(pAccessRights, 2);

    printf("            Secure Dynamic Messaging Information            : ");
    PRINT_DATA_NEWLINE(aAddInfo, bAddInfoLen);

    printf("                                                              ");
    printf("SDM Options             : %02X (VCUID, SDM Read Ctr%sEnabled)\n", aAddInfo[bOffset++],
        (bIsSDMEncData_Present ? " and SDM EncFile Data " : " "));

    printf("                                                              ");
    printf("SDM AccessRights        : ");
    PRINT_DATA_NEWLINE(&aAddInfo[bOffset], 2);
    bOffset += 2;

    if(bIsVCUID_Present || bIsPICCData_Present)
    {
        printf("                                                              ");
        if(bIsVCUID_Present) printf("VCUID Offset            : ");
        if(bIsPICCData_Present) printf("PICC Data Offset        : ");
        PRINT_DATA_NEWLINE(&aAddInfo[bOffset], 3);
        bOffset += 3;
    }

    if(bIsSDMRdCtr_Present)
    {
        printf("                                                              ");
        printf("SDM ReadCtr Offset      : ");
        PRINT_DATA_NEWLINE(&aAddInfo[bOffset], 3);
        bOffset += 3;
    }

    printf("                                                              ");
    printf("SDM MAC Input Offset    : ");
    PRINT_DATA_NEWLINE(&aAddInfo[bOffset], 3);
    bOffset += 3;

    if(bIsSDMEncData_Present)
    {
        printf("                                                              ");
        printf("SDM EncFile Offset      : ");
        PRINT_DATA_NEWLINE(&aAddInfo[bOffset], 3);
        bOffset += 3;

        printf("                                                              ");
        printf("SDM EncFile Length      : ");
        PRINT_DATA_NEWLINE(&aAddInfo[bOffset], 3);
        bOffset += 3;
    }

    printf("                                                              ");
    printf("SDM MAC Offset          : ");
    PRINT_DATA_NEWLINE(&aAddInfo[bOffset], 3);
    bOffset += 3;

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ChangeFileSettings(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN,
        STANDARD_DATA_FILE, bFileOption, pAccessRights, aAddInfo, bAddInfoLen));
    printf("            Status                                          : SUCCESS\n");

    return wStatus;
}

phStatus_t EnableVCProximityKey()
{
    phStatus_t wStatus = 0;
    uint8_t bKeyNo = 0;
    uint8_t bPICCKeyNo = 0;

    /* Authenticate using PICC MasterKey */
    PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, PICC_MASTER_KEY));

    /* Enale VC Configuration Key */
    bKeyNo = (uint8_t) (wIsAES256KeyType ? AES256_KEY_ADDRESS : AES128_KEY_ADDRESS);
    bPICCKeyNo = (uint8_t) ((wIsAES256KeyType ? 0xC0 : 0x80) | VC_CONFIGURATION_KEY);
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ChangeKeyEv2(&stMfDuoX, PHAL_MFDUOX_CHGKEY_NO_DIVERSIFICATION,
        bKeyNo, 0, bKeyNo, 0, 0, bPICCKeyNo, NULL, 0));

    /* Authenticate uisng VC Configuration Key */
    PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, VC_CONFIGURATION_KEY));

    /* Enable VC ProximityCheck Key */
    bPICCKeyNo = (uint8_t) ((wIsAES256KeyType ? 0xC0 : 0x80) | VC_PROXIMITY_KEY);
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ChangeKeyEv2(&stMfDuoX, PHAL_MFDUOX_CHGKEY_NO_DIVERSIFICATION,
        bKeyNo, 0, bKeyNo, 0, 0, bPICCKeyNo, NULL, 0));

    return wStatus;
}





phStatus_t Demo_ECCAuthentication(uint8_t bAuthType)
{
    phStatus_t wStatus = 0;

    uint8_t aOptsA[] = { 0x80, 0x04, 0x80, 0x00, STANDARD_DATA_FILE_ECC, ECC_AUTHENTICATION_KEY };

    uint8_t aE_PubB[65];
    uint8_t * pE_PubB = aE_PubB;
    uint16_t wE_PubB = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    uint8_t aPubKey[65];
    uint16_t wPubKeyLen = 0;

    uint8_t * pOptsB = NULL;
    uint8_t bOptsBLen = 0;

    uint8_t * pCertB = NULL;
    uint16_t wCertBLen = 0;

    uint8_t * pSigB = NULL;
    uint16_t wSigBLen = 0;

    /* Clear Authentication State */
    bAuth_State = PHAL_MFDUOX_NOT_AUTHENTICATED;

    do
    {
        printf("Preparation for ISOGeneral Authenticate ---------------------------------------------------------------\n");

        /* Activate the card. */
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("    PICC Activation                                         : SUCCESS\n");

        wStatus = Prepare_ISOGeneralAuthenticate(bAuthType, ECC_AUTHENTICATION_KEY, PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n\n");

        printf("ISOGeneral Authentication -----------------------------------------------------------------------------\n");

        /* Perform ISOSelectFile in case of ISOGeneralAuthenticate Final */
        if(bAuthType == PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE_FINAL)
        {
            /* Activate the card. */
            wStatus = ActivateCard();
            CHECK_SUCCESS_BREAK(wStatus);
            printf("    PICC Activation                                         : SUCCESS\n");

            /* Perform ISOSelect File */
            printf("    ISOSelectFile to get the Ephemeral Public Key (Device, E.Pub.B)\n");
            printf("        Option                                              : 0x0C (FCI Returned)\n");
            printf("        Selector                                            : 0x04 (Select by DF name)\n");
            printf("        File Identifier                                     : NULL\n");
            printf("        DF Name                                             : ");
            PRINT_DATA_NEWLINE(aISODfName_EccApp, sizeof(aISODfName_EccApp));

            wStatus = phalMfDuoX_IsoSelectFile(&stMfDuoX, PHAL_MFDUOX_FCI_RETURNED, PHAL_MFDUOX_SELECTOR_4, NULL,
                aISODfName_EccApp, (uint8_t) sizeof(aISODfName_EccApp), 0, &pResponse, &wRespLen);
            CHECK_SUCCESS_BREAK(wStatus);

            printf("        Response                                            : ");
            PRINT_DATA_NEWLINE(pResponse, wRespLen);
            printf("        Status                                              : SUCCESS\n\n");

            /* Copy Ephemeral Public Key received from PICC */
            memcpy(aE_PubB, &pResponse[4], (wRespLen - 4));
            wE_PubB = (uint16_t) (wRespLen - 4);
        }

        /* Perform ISOGeneral Authenticate or ISOGeneral Authenticate Final */
        printf("    ISOGneral Authenticate %s- Mutual Authentication with Cert.A\n",
            ((bAuthType == PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE_FINAL) ? " Final" : ""));

        printf("        Authentication Type                                 : %02Xh (ISOGeneral Authenticate%s)\n",
            bAuthType, ((bAuthType == PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE_FINAL) ? " Final" : ""));

        printf("        CA Root Key Number                                  : %02Xh\n", ECC_AUTHENTICATION_KEY);
        printf("        OptsA                                               : ");
        PRINT_DATA_NEWLINE(aOptsA, sizeof(aOptsA));

        printf("        Certificate (Host, Cert.A)                          : ");
        PRINT_DATA_NEWLINE(aP256_HOST_Cert, (uint16_t) sizeof aP256_HOST_Cert);

        if(bAuthType == PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE_FINAL)
        {
            printf("        Ephemeral Public Key (Device, E.Pub.B)              : ");
            PRINT_DATA_NEWLINE(pE_PubB, wE_PubB);
        }

        wStatus = phalMfDuoX_ISOGeneralAuthenticate(&stMfDuoX, bAuthType, ECC_AUTHENTICATION_KEY,
            ECC_P256_HOST_PRIVATE_KEY_ADDRESS, ECC_P256_HOST_PRIVATE_KEY_POSITION, aOptsA,
            (uint8_t) sizeof aOptsA, aP256_HOST_Cert, (uint16_t) sizeof aP256_HOST_Cert,
            NULL, 0, pE_PubB, &wE_PubB, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);

        if(bAuthType == PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE)
        {
            printf("        Ephemeral Public Key (Device, E.Pub.B)              : ");
            PRINT_DATA_NEWLINE(pE_PubB, wE_PubB);
        }

        /* Remove the Constants */
        printf("        Response\n");
        pResponse += 2;
        wRespLen -= 2;

        /* Extract OptsB, CertificateB and SignatureB from Response */
        pOptsB = pResponse;
        bOptsBLen = (uint8_t) (sizeof aOptsA + 1/* Length of OptsB */);
        printf("            OptsB                                           : ");
        PRINT_DATA_NEWLINE(pOptsB, bOptsBLen);

        pCertB = &pResponse[bOptsBLen];
        wCertBLen = (uint16_t) (wRespLen - (bOptsBLen + 64));
        printf("            Certificate (Device, Cert.B)                    : ");
        PRINT_DATA_NEWLINE(pCertB, wCertBLen);

        pSigB = &pResponse[wRespLen - 64];
        wSigBLen = 64;
        printf("            Signature (Device, SigB)                        : ");
        PRINT_DATA_NEWLINE(pSigB, wSigBLen);

        printf("        Status                                              : SUCCESS\n\n");

        /* Extract Device Public key from received Certificate */
        ExtractPublicKey(pCertB, wCertBLen, aPubKey, &wPubKeyLen);

        /* Complete ISOGeneral Authentication. */
        printf("    ISOGneral Authenticate Verification- Message and Signature verification\n");
        printf("        Public Key (Device Public Key)                      : ");
        PRINT_DATA_NEWLINE(aPubKey, wPubKeyLen);

        printf("        OptsB (Message)                                     : ");
        PRINT_DATA_NEWLINE(pOptsB, bOptsBLen);

        printf("        Ephemeral Public Key (Device, E.Pub.B)              : ");
        PRINT_DATA_NEWLINE(pE_PubB, wE_PubB);

        printf("        Signature (Device, SigB)                            : ");
        PRINT_DATA_NEWLINE(pSigB, wSigBLen);

        wStatus = phalMfDuoX_ISOGeneralAuthenticateVerify(&stMfDuoX, aPubKey, wPubKeyLen, pOptsB, bOptsBLen,
            pE_PubB, wE_PubB, pSigB, wSigBLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        Status                                              : SUCCESS\n\n");

        /* Configure Authentication State */
        bAuth_State = PHAL_MFDUOX_ECC_AUTHENTICATED;

    } while(0);

    return wStatus;
}

phStatus_t Demo_AESAuthentication(uint16_t wIsKeyTypeAES256)
{
    phStatus_t wStatus = 0;

    /* Clear Authentication State */
    bAuth_State = PHAL_MFDUOX_NOT_AUTHENTICATED;

    do
    {
        /* Perform Application Selection. */
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        PICC Activation                             : SUCCESS\n");

        /* Select Master Application */
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP, aAppID_Master, NULL);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("        Perform %s Authentication\n", AUTH_KEY_TYPE(wIsAES256KeyType));
        printf("            Authentication Type                     : %02X (First Auth)\n", PHAL_MFDUOX_AUTH_FIRST);
        printf("            Key Diversification                     : %04X (Disabled)\n", PHAL_MFDUOX_NO_DIVERSIFICATION);
        printf("            Key Number of the tag                   : %02X\n", PICC_MASTER_KEY);
        printf("            PCD Input Capabilities                  : ");
        PRINT_DATA_NEWLINE(aPcdCap2In, sizeof(aPcdCap2In));

        /* Perform EV2 First Authentication */
        wStatus = phalMfDuoX_AuthenticateEv2(&stMfDuoX, PHAL_MFDUOX_AUTH_FIRST, PHAL_MFDUOX_NO_DIVERSIFICATION,
            AES128_KEY_ADDRESS, AES128_KEY_VERSION, PICC_MASTER_KEY, NULL, 0, aPcdCap2In, (uint8_t) sizeof(aPcdCap2In),
            aPcdCap2Out, aPdCap2);
        CHECK_SUCCESS_BREAK(wStatus);
        wIsAES256KeyType = PH_OFF;

        printf("            PCD Output Capabilities                 : ");
        PRINT_DATA_NEWLINE(aPcdCap2Out, sizeof(aPcdCap2Out));

        printf("            PD Output Capabilities                  : ");
        PRINT_DATA_NEWLINE(aPdCap2, sizeof(aPdCap2));

        printf("            Status                                  : SUCCESS\n\n");

        /* Change Application Master Key to AES256 */
        if(wIsKeyTypeAES256 == PH_ON)
        {
            printf("        Perform Change Key to Convert Application Master Key to AES256 KeyType\n");
            printf("            Key Diversification                     : %04X (Disabled)\n", PHAL_MFDUOX_CHGKEY_NO_DIVERSIFICATION);
            printf("            PICC Key Number                         : %02X (PICC Master Key | 0xC0 (AES256 KeyType))\n",
                (0xC0 | PICC_MASTER_KEY));
            wStatus = phalMfDuoX_ChangeKey(&stMfDuoX, PHAL_MFDUOX_CHGKEY_NO_DIVERSIFICATION,
                AES128_KEY_ADDRESS, AES128_KEY_VERSION, AES256_KEY_ADDRESS, AES256_KEY_VERSION,
                (uint8_t) (0xC0 | PICC_MASTER_KEY), NULL, 0);
            CHECK_SUCCESS_BREAK(wStatus);

            printf("            Status                                  : SUCCESS\n\n");

            /* Perform Authenticate using AES256 Key */
            printf("        Perform %s Authentication\n", (wIsKeyTypeAES256 ? "AES 256-Bit" : "AES 128-Bit"));
            printf("            Authentication Type                     : %02X (First Auth)\n", PHAL_MFDUOX_AUTH_FIRST);
            printf("            Key Diversification                     : %04X (Disabled)\n", PHAL_MFDUOX_NO_DIVERSIFICATION);
            printf("            Key Number of the tag                   : %02X\n", PICC_MASTER_KEY);
            printf("            PCD Input Capabilities                  : ");
            PRINT_DATA_NEWLINE(aPcdCap2In, sizeof(aPcdCap2In));

            /* Perform EV2 First Authentication */
            wStatus = phalMfDuoX_AuthenticateEv2(&stMfDuoX, PHAL_MFDUOX_AUTH_FIRST, PHAL_MFDUOX_NO_DIVERSIFICATION,
                AES256_KEY_ADDRESS, AES256_KEY_VERSION, PICC_MASTER_KEY, NULL, 0, aPcdCap2In, (uint8_t) sizeof(aPcdCap2In),
                aPcdCap2Out, aPdCap2);
            CHECK_SUCCESS_BREAK(wStatus);

            wIsAES256KeyType = PH_ON;

            printf("            PCD Output Capabilities                 : ");
            PRINT_DATA_NEWLINE(aPcdCap2Out, sizeof(aPcdCap2Out));

            printf("            PD Output Capabilities                  : ");
            PRINT_DATA_NEWLINE(aPdCap2, sizeof(aPdCap2));

            printf("            Status                                  : SUCCESS\n\n");
        }

        printf("\n");
        printf("\tNote: \n");
        printf("        Secure messaging using %s Bit Key will be used for the other \n", AUTH_KEY_TYPE(wIsAES256KeyType));
        printf("        Demo operation where ever Authentication is necessary\n\n");
    } while(0);

    /* Configure Authentication State */
    bAuth_State = PHAL_MFDUOX_EV2_AUTHENTICATED;

    return wStatus;
}

phStatus_t Demo_StandardFileOperation()
{
    phStatus_t wStatus = 0;
    uint8_t bFileNo = STANDARD_DATA_FILE;
    uint8_t aAccessRights[2] = { 0xEE, 0xEE };

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    do
    {
        if(bAuth_State == PHAL_MFDUOX_EV2_AUTHENTICATED)
        {
            /* Create and Select Application */
            wStatus = CreateSelectApplication(APPLICATION_TYPE_NORMAL, PH_OFF);
            CHECK_SUCCESS_BREAK(wStatus);
            printf("\n");
        }

        /* Standard File Creation with Plain communication, Free Access Rights and no Authentication */
        printf("    Standard File Creation with Plain Communication and Free Access Rights ----------------------------\n");
        wStatus = CreateStandardFile(bFileNo, PH_ON, PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights,
            PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Activate the card. */
        printf("        PICC Activation");
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                     : SUCCESS\n");

        /* Select Application */
        printf("        Application Selection");
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP, pAID, NULL);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                               : SUCCESS\n");

        /* Write some data to Standard File. */
        printf("        Write Data                                          : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = Write(bFileNo, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            0, aData, sizeof(aData), PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Read the written data from file */
        printf("        Read Data                                           : ");
        wStatus = ReadData(bFileNo, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            0, sizeof(aData), &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Standard File Creation with Plain Communication and APP_Master Key for Read and Write operations */
        printf("    Standard File Creation with Plain communication and App Master Key for Authentication -------------\n");
        aAccessRights[0] = 0x00; aAccessRights[1] = 0x00;
        wStatus = CreateStandardFile(bFileNo, PH_ON, PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights, PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE(bAuth_State));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_MASTER_KEY));
        printf(": SUCCESS\n");

        /* Write some data to Standard File. */
        printf("        Write Data                                          : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = Write(bFileNo, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE, 0,
            aData, sizeof(aData), PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Read the written data from file */
        printf("        Read Data                                           : ");
        wStatus = ReadData(bFileNo, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE, 0,
            sizeof(aData), &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Standard File Creation with MAC Communication and Key 1 for Read and Write operations */
        printf("    Standard File Creation with MAC communication and Key 1 for Authentication ------------------------\n");
        aAccessRights[0] = 0x10; aAccessRights[1] = 0x00;
        wStatus = CreateStandardFile(bFileNo, PH_ON, PHAL_MFDUOX_FILE_OPTION_MAC, aAccessRights, PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 1              "));
         PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_1));
        printf(": SUCCESS\n");

        /* Write some data to Standard File. */
        printf("        Write Data                                          : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = Write(bFileNo, PHAL_MFDUOX_COMMUNICATION_MAC, PHAL_MFDUOX_CHAINING_NATIVE, 0,
            aData, sizeof(aData), PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Read the written data from file */
        printf("        Read Data                                           : ");
        wStatus = ReadData(bFileNo, PHAL_MFDUOX_COMMUNICATION_MAC, PHAL_MFDUOX_CHAINING_NATIVE, 0,
            sizeof(aData), &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Standard File Creation with FULL Communication, Key 1 for Read and Key 2 for Write operations */
        printf("    Standard File Creation with Full communication, Key1 for Read and Key2 for Write ------------------\n");
        aAccessRights[0] = 0x00; aAccessRights[1] = 0x12;
        wStatus = CreateStandardFile(bFileNo, PH_ON, PHAL_MFDUOX_FILE_OPTION_FULL, aAccessRights, PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 2              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_2));
        printf(": SUCCESS\n");

        /* Write some data to Standard File. */
        printf("        Write Data                                          : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = Write(bFileNo, PHAL_MFDUOX_COMMUNICATION_FULL, PHAL_MFDUOX_CHAINING_NATIVE, 0,
            aData, sizeof(aData), PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 1              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_1));
        printf(": SUCCESS\n");

        /* Read the written data from file */
        printf("        Read Data                                           : ");
        wStatus = ReadData(bFileNo, PHAL_MFDUOX_COMMUNICATION_FULL, PHAL_MFDUOX_CHAINING_NATIVE, 0,
            sizeof(aData), &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");
    } while(0);

    return wStatus;
}

phStatus_t Demo_ValueFileOperation()
{
    phStatus_t wStatus = 0;
    uint8_t bFileNo = VALUE_FILE;

    uint8_t aAccessRights[2] = { 0xEE, 0xEE };
    uint8_t aValue[] = { 0x05, 0x00, 0x00, 0x00 };

    uint8_t * pValue = NULL;
    uint16_t wValueLen = 0;

    do
    {
        if(bAuth_State == PHAL_MFDUOX_EV2_AUTHENTICATED)
        {
            /* Create and Select Application */
            wStatus = CreateSelectApplication(APPLICATION_TYPE_NORMAL, PH_OFF);
            CHECK_SUCCESS_BREAK(wStatus);
            printf("\n");
        }

        /* Value File Creation with Plain communication, Free Access Rights and no Authentication */
        printf("    Value File Creation with Plain Communication and Free Access Rights -------------------------------\n");
        wStatus = CreateValueFile(bFileNo, PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Activate the card. */
        printf("        PICC Activation");
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                     : SUCCESS\n");

        /* Select Application */
        printf("        Application Selection");
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP,
            pAID, NULL);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                               : SUCCESS\n");

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Perform Credit operation */
        printf("        Credit Value                                        : ");
        PRINT_DATA_NEWLINE(aValue, wValueLen);
        wStatus = phalMfDuoX_Credit(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo, aValue);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Perform Debit operation */
        printf("        Debit Value                                         : ");
        PRINT_DATA_NEWLINE(aValue, wValueLen);
        wStatus = phalMfDuoX_Debit(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo, aValue);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Value File Creation with Plain Communication and APP_Master Key for Read and Write operations */
        printf("    Value File Creation with Plain communication and App Master Key for Authentication ----------------\n");
        aAccessRights[0] = 0x00; aAccessRights[1] = 0x00;
        wStatus = CreateValueFile(bFileNo, PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE(bAuth_State));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_MASTER_KEY));
        printf(": SUCCESS\n");

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Perform Credit operation */
        printf("        Credit Value                                        : ");
        PRINT_DATA_NEWLINE(aValue, wValueLen);
        wStatus = phalMfDuoX_Credit(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo, aValue);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Perform Debit operation */
        printf("        Debit Value                                         : ");
        PRINT_DATA_NEWLINE(aValue, wValueLen);
        wStatus = phalMfDuoX_Debit(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo, aValue);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Value File Creation with MAC Communication and Key 1 for Read and Write operations */
        printf("    Value File Creation with MAC communication and Key 1 for Authentication ---------------------------\n");
        aAccessRights[0] = 0x10; aAccessRights[1] = 0x00;
        wStatus = CreateValueFile(bFileNo, PHAL_MFDUOX_FILE_OPTION_MAC, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 1              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_1));
        printf(": SUCCESS\n");

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_MAC, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Perform Credit operation */
        printf("        Credit Value                                        : ");
        PRINT_DATA_NEWLINE(aValue, wValueLen);
        wStatus = phalMfDuoX_Credit(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_MAC, bFileNo, aValue);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_MAC, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Perform Debit operation */
        printf("        Debit Value                                         : ");
        PRINT_DATA_NEWLINE(aValue, wValueLen);
        wStatus = phalMfDuoX_Debit(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_MAC, bFileNo, aValue);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_MAC, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Value File Creation with FULL Communication, Key 1 for Read and Key 2 for Write operations */
        printf("    Value File Creation with Full communication, Key1 for Read and Key2 for Write ---------------------\n");
        aAccessRights[0] = 0x20; aAccessRights[1] = 0x10;
        wStatus = CreateValueFile(bFileNo, PHAL_MFDUOX_FILE_OPTION_FULL, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 1              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_1));
        printf(": SUCCESS\n");

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 2              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_2));
        printf(": SUCCESS\n");

        /* Perform Credit operation */
        printf("        Credit Value                                        : ");
        PRINT_DATA_NEWLINE(aValue, wValueLen);
        wStatus = phalMfDuoX_Credit(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, bFileNo, aValue);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 1              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_1));
        printf(": SUCCESS\n");

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
        printf("\n");

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 2              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_2));
        printf(": SUCCESS\n");

        /* Perform Debit operation */
        printf("        Debit Value                                         : ");
        PRINT_DATA_NEWLINE(aValue, wValueLen);
        wStatus = phalMfDuoX_Debit(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, bFileNo, aValue);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 1              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_1));
        printf(": SUCCESS\n");

        /* Get the Current Value */
        printf("        Get Current Value                                   : ");
        wStatus = phalMfDuoX_GetValue(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, bFileNo,
            &pValue, &wValueLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pValue, wValueLen);
    } while(0);

    return wStatus;
}

phStatus_t Demo_RecordFileOperation()
{
    phStatus_t wStatus = 0;
    uint8_t bFileNo = RECORD_FILE;

    uint8_t aAccessRights[2] = { 0xEE, 0xEE };
    uint8_t aOffset[] = { 0x00, 0x00, 0x00 };

    uint8_t aDataLen[3] = { 0x10, 0x00, 0x00 };
    uint8_t aRecordNo[] = { 0x00, 0x00, 0x00 };
    uint8_t aRecordCount[] = { 0x01, 0x00, 0x00 };
    uint8_t aRecordSize[] = { 0x10, 0x00, 0x00 };

    uint8_t * pTMC = NULL;
    uint16_t wTMCLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    do
    {
        if(bAuth_State == PHAL_MFDUOX_EV2_AUTHENTICATED)
        {
            /* Create and Select Application */
            wStatus = CreateSelectApplication(APPLICATION_TYPE_NORMAL, PH_OFF);
            CHECK_SUCCESS_BREAK(wStatus);
            printf("\n");
        }

        /* Linear Record File Creation with Plain communication, Free Access Rights and no Authentication */
        printf("    Linear Record File Creation with Plain Communication and Free Access Rights -----------------------\n");
        wStatus = CreateRecordFile(bFileNo, PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Activate the card. */
        printf("        PICC Activation");
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                     : SUCCESS\n");

        /* Select Application */
        printf("        Application Selection");
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP,
            pAID, NULL);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                               : SUCCESS\n");

        /* Write some data to Record File. */
        printf("        Write Record                                        : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = phalMfDuoX_WriteRecord(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            bFileNo, aOffset, aData, aDataLen);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Commit Transaction for the record to be written */
        printf("        Commit Transaction");
        wStatus = phalMfDuoX_CommitTransaction(&stMfDuoX, PHAL_MFDUOX_OPTION_TRANSACTION_INFO_NOT_RETURNED, &pTMC,
            &wTMCLen, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                  : SUCCESS\n");

        /* Read the written data from file */
        printf("        Read Record                                         : ");
        wStatus = phalMfDuoX_ReadRecords(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            bFileNo, aRecordNo, aRecordCount, aRecordSize, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Linear Record File Creation with Plain Communication and APP_Master Key for Read and Write operations */
        printf("    Linear Record File Creation with Plain communication and App Master Key for Authentication --------\n");
        aAccessRights[0] = 0x00; aAccessRights[1] = 0x00;
        wStatus = CreateRecordFile(bFileNo, PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE(bAuth_State));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_MASTER_KEY));
        printf(": SUCCESS\n");

        /* Write some data to Record File. */
        printf("        Write Record                                        : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = phalMfDuoX_WriteRecord(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            bFileNo, aOffset, aData, aDataLen);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Commit Transaction for the record to be written */
        printf("        Commit Transaction");
        wStatus = phalMfDuoX_CommitTransaction(&stMfDuoX, PHAL_MFDUOX_OPTION_TRANSACTION_INFO_NOT_RETURNED, &pTMC,
            &wTMCLen, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                  : SUCCESS\n");

        /* Read the written data from file */
        printf("        Read Record                                         : ");
        wStatus = phalMfDuoX_ReadRecords(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            bFileNo, aRecordNo, aRecordCount, aRecordSize, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Linear Record File Creation with MAC Communication and Key 1 for Read and Write operations */
        printf("    Linear Record File Creation with MAC communication and Key 1 for Authentication -------------------\n");
        aAccessRights[0] = 0x10; aAccessRights[1] = 0x00;
        wStatus = CreateRecordFile(bFileNo, PHAL_MFDUOX_FILE_OPTION_MAC, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 1              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_1));
        printf(": SUCCESS\n");

        /* Write some data to Record File. */
        printf("        Write Record                                        : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = phalMfDuoX_WriteRecord(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_MAC, PHAL_MFDUOX_CHAINING_NATIVE,
            bFileNo, aOffset, aData, aDataLen);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Commit Transaction for the record to be written */
        printf("        Commit Transaction");
        wStatus = phalMfDuoX_CommitTransaction(&stMfDuoX, PHAL_MFDUOX_OPTION_TRANSACTION_INFO_NOT_RETURNED, &pTMC,
            &wTMCLen, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                  : SUCCESS\n");

        /* Read the written data from file */
        printf("        Read Record                                         : ");
        wStatus = phalMfDuoX_ReadRecords(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_MAC, PHAL_MFDUOX_CHAINING_NATIVE,
            bFileNo, aRecordNo, aRecordCount, aRecordSize, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Increment File Number */
        bFileNo++;

        /* Linear Record File Creation with FULL Communication, Key 1 for Read and Key 2 for Write operations */
        printf("    Linear Record File Creation with Full communication, Key1 for Read and Key2 for Write -------------\n");
        aAccessRights[0] = 0x00; aAccessRights[1] = 0x12;
        wStatus = CreateRecordFile(bFileNo, PHAL_MFDUOX_FILE_OPTION_FULL, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 2              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_2));
        printf(": SUCCESS\n");

        /* Write some data to Record File. */
        printf("        Write Record                                        : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = phalMfDuoX_WriteRecord(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, PHAL_MFDUOX_CHAINING_NATIVE,
            bFileNo, aOffset, aData, aDataLen);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Commit Transaction for the record to be written */
        printf("        Commit Transaction");
        wStatus = phalMfDuoX_CommitTransaction(&stMfDuoX, PHAL_MFDUOX_OPTION_TRANSACTION_INFO_NOT_RETURNED, &pTMC,
            &wTMCLen, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                  : SUCCESS\n");

        /* Perform Authentication */
        printf("        %s", AUTH_STATE_MSG(bAuth_State, "Authentication using Application key 1              "));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_KEY_1));
        printf(": SUCCESS\n");

        /* Read the written data from file */
        printf("        Read Record                                         : ");
        wStatus = phalMfDuoX_ReadRecords(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, PHAL_MFDUOX_CHAINING_NATIVE,
            bFileNo, aRecordNo, aRecordCount, aRecordSize, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");
    } while(0);

    return wStatus;
}

phStatus_t Demo_TrasactionMAC()
{
    phStatus_t wStatus = 0;

    uint8_t * pTMC = 0;
    uint16_t wTMC_Len = 0;

    uint8_t aTMV[8];
    uint8_t * pTMV = NULL;
    uint16_t wTMV_Len = 0;

    uint8_t * pTMI = 0;
    uint32_t dwTMI_Len = 0;

    uint8_t aAccessRights[2] = { 0x00, 0x00 };

    do
    {
        if(bAuth_State == PHAL_MFDUOX_EV2_AUTHENTICATED)
        {
            /* Create and Select Application */
            wStatus = CreateSelectApplication(APPLICATION_TYPE_NORMAL, PH_OFF);
            CHECK_SUCCESS_BREAK(wStatus);
            printf("\n");
        }

        /* Create standard File for Read and Write operation */
        printf("    Standard File Creation with Plain Communication ---------------------------------------------------\n");
        wStatus = CreateStandardFile(STANDARD_DATA_FILE, PH_ON, PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights,
            PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Create Transaction Mac */
        wStatus = CreateTransactionMACFile((uint8_t)(PHAL_MFDUOX_KEY_TYPE_AES128 |
            PHAL_MFDUOX_TMKEY_MODE_TRANSACTION_MAC));
        CHECK_SUCCESS_BREAK(wStatus);

        /* Activate TMI Collection */
        wStatus = phTMIUtils_ActivateTMICollection(&stTmiUtils, PH_TMIUTILS_ACTIVATE_TMI);
        CHECK_SUCCESS_BREAK(wStatus);

        wStatus = phTMIUtils_ActivateTMICollection(&stTmiUtils, PH_TMIUTILS_RESET_TMI);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Authentication */
        printf("    %s", AUTH_STATE(bAuth_State));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_MASTER_KEY));
        printf("    : SUCCESS\n\n");

        /* Write some data to Standard File. */
        wStatus = Write(STANDARD_DATA_FILE, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            0, aData, sizeof(aData), PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Commit the Transaction */
        printf("    Commit Transaction\n");
        printf("        Option                                              : 01 (TMC / TMV Returned)\n");
        wStatus = phalMfDuoX_CommitTransaction(&stMfDuoX, PHAL_MFDUOX_OPTION_TRANSACTION_INFO_RETURNED,
            &pTMC, &wTMC_Len, &pTMV, &wTMV_Len);
        CHECK_SUCCESS_BREAK(wStatus);

        memcpy(aTMV, pTMV, wTMV_Len);

        printf("        Transaction MAC Counter (TMC)                       : ");
        PRINT_DATA_NEWLINE(pTMC, wTMC_Len);

        printf("        Transaction MAC Value (TMV, Received from PICC)     : ");
        PRINT_DATA_NEWLINE(aTMV, wTMV_Len);

        printf("        Status                                              : SUCCESS\n\n");

        /* Get the TMI Information */
        wStatus = phTMIUtils_GetTMI(&stTmiUtils, &pTMI, &dwTMI_Len);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Deactivate TMI Collection */
        wStatus = phTMIUtils_ActivateTMICollection(&stTmiUtils, PH_TMIUTILS_PAUSE_TMI);
        CHECK_SUCCESS_BREAK(wStatus);

        wStatus = phTMIUtils_ActivateTMICollection(&stTmiUtils, PH_TMIUTILS_RESET_TMI);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Calculate TMV */
        wStatus = phalMfDuoX_CalculateTMV(&stMfDuoX, PHAL_MFDUOX_TM_KEY_DIVERSIFICATION_OFF, AES128_KEY_ADDRESS,
            AES128_KEY_VERSION, 0, 0, NULL, 0, pTMC, aUid, bUid_Len, pTMI, (uint16_t) dwTMI_Len, &pTMV, &wTMV_Len);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("        Transaction MAC Value (TMV, Computed by Libray)     : ");
        PRINT_DATA_NEWLINE(pTMV, wTMV_Len);

        if(memcmp(aTMV, pTMV, wTMV_Len) == 0)
            printf("        Transaction MAC Verification                        : SUCCESS\n\n");
        else
            printf("        Transaction MAC Verification                        : Failed\n\n");

    } while(0);

    return wStatus;
}

phStatus_t Demo_TrasactionSIG()
{
    phStatus_t wStatus = 0;
    uint8_t aAccessRights[2] = { 0x00 , 0x00 };
    uint8_t aKeyPolicy[2] = { 0x40 , 0x00 };

    uint8_t aPubKey[65];
    uint8_t * pPubKey = NULL;
    uint16_t wPubKeyLen = 0;

    uint8_t * pTMC = 0;
    uint16_t wTMC_Len = 0;

    uint8_t * pTSV = NULL;
    uint16_t wTSV_Len = 0;

    uint8_t * pTMI = 0;
    uint32_t dwTMI_Len = 0;

    do
    {
        if(bAuth_State == PHAL_MFDUOX_EV2_AUTHENTICATED)
        {
            /* Create and Select Application */
            wStatus = CreateSelectApplication(APPLICATION_TYPE_NORMAL, PH_OFF);
            CHECK_SUCCESS_BREAK(wStatus);
            printf("\n");
        }

        /* Perform Authentication */
        printf("    %s", AUTH_STATE(bAuth_State));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_MASTER_KEY));
        printf("    : SUCCESS\n\n");

        /* Perform Manage KeyPair to generate a private key for Signature generation. */
        printf("    Generating Key Pair for signing operation in device (ManageKeyPair)\n");
        printf("        Communication Mode                                  : 30 (FULL)\n");
        printf("        Key Number                                          : %02X\n", ECC_TSIG_KEY);
        printf("        Option                                              : 00 (Generate KeyPair)\n");
        printf("        Curve ID                                            : 0Ch (NIST P-256)\n");
        printf("        Key Policy                                          : ");
        PRINT_DATA_NEWLINE(aKeyPolicy, 2);
        printf("        Write Access                                        : 30\n");

        wStatus = phalMfDuoX_ManageKeyPair(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL,
            ECC_TSIG_KEY, PHAL_MFDUOX_TARGET_ACTION_GENERATE_KEY_PAIR, PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256,
            aKeyPolicy, 0x30, 0, 0, 0, &pPubKey, &wPubKeyLen);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("        Response                                            : ");
        PRINT_DATA_NEWLINE(pPubKey, wPubKeyLen);

        printf("        Status                                              : SUCCESS\n\n");
        memcpy(aPubKey, pPubKey, wPubKeyLen);

        /* Create standard File for Read and Write operation */
        printf("    Standard File Creation with Plain Communication ---------------------------------------------------\n");
        wStatus = CreateStandardFile(STANDARD_DATA_FILE, PH_ON, PHAL_MFDUOX_FILE_OPTION_PLAIN, aAccessRights,
            PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Create Transaction Mac */
        wStatus = CreateTransactionMACFile((uint8_t) (PHAL_MFDUOX_TMKEY_MODE_TRANSACTION_SIGNATURE));
        CHECK_SUCCESS_BREAK(wStatus);

        /* Activate TMI Collection */
        wStatus = phTMIUtils_ActivateTMICollection(&stTmiUtils, PH_TMIUTILS_ACTIVATE_TMI);
        CHECK_SUCCESS_BREAK(wStatus);

        wStatus = phTMIUtils_ActivateTMICollection(&stTmiUtils, PH_TMIUTILS_RESET_TMI);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Write some data to Standard File. */
        wStatus = Write(STANDARD_DATA_FILE, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            0, aData, sizeof(aData), PH_ON);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Commit the Transaction */
        printf("    Commit Transaction\n");
        printf("        Option                                              : 01 (TMC / TMV Returned)\n");
        wStatus = phalMfDuoX_CommitTransaction(&stMfDuoX, PHAL_MFDUOX_OPTION_TRANSACTION_INFO_RETURNED,
            &pTMC, &wTMC_Len, &pTSV, &wTSV_Len);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("        Transaction MAC Counter (TMC)                       : ");
        PRINT_DATA_NEWLINE(pTMC, wTMC_Len);

        printf("        Transaction Sig Value (TSV, Received from PICC)     : ");
        PRINT_DATA_NEWLINE(pTSV, wTSV_Len);

        printf("        Status                                              : SUCCESS\n");

        /* Get the TMI Information */
        wStatus = phTMIUtils_GetTMI(&stTmiUtils, &pTMI, &dwTMI_Len);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Deactivate TMI Collection */
        wStatus = phTMIUtils_ActivateTMICollection(&stTmiUtils, PH_TMIUTILS_PAUSE_TMI);
        CHECK_SUCCESS_BREAK(wStatus);

        wStatus = phTMIUtils_ActivateTMICollection(&stTmiUtils, PH_TMIUTILS_RESET_TMI);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Verify Transaction Signature received from PICC */
        wStatus = phalMfDuoX_VerifyTransationSignature(&stMfDuoX, PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256,
            aPubKey, wPubKeyLen, pTMI, (uint16_t) dwTMI_Len, pTMC, pTSV, wTSV_Len);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            printf("        Transaction Signature Verification                  : SUCCESS\n\n");
        else
            printf("        Transaction Signature Verification                  : Failed\n\n");

    } while(0);

    return wStatus;
}

phStatus_t Demo_SecureDynamicMessaging_MAC()
{
    phStatus_t wStatus = 0;
    uint8_t bKeyNo = 0;

    uint8_t aAccessRights[2] = { 0xEE, 0xEE };

    uint8_t aVCUID_PICCData[32];
    uint16_t wVCUID_PICCDataLen = 0;

    uint8_t aSDMRdCtr[6];
    uint16_t wSDMRdCtrLen = 0;

    uint8_t aEncFileData[32];
    uint16_t wEncFileDataLen = 0;

    uint8_t aMac[16];
    uint16_t wMacLen = 0;

    uint8_t bFileOption = (uint8_t) (PHAL_MFDUOX_FILE_OPTION_PLAIN | PHAL_MFDUOX_FILE_OPTION_SDM_MIRRORING_ENABLED);

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    do
    {
        bKeyNo = (uint8_t) (wIsAES256KeyType ? AES256_KEY_ADDRESS : AES128_KEY_ADDRESS);

        if(bAuth_State == PHAL_MFDUOX_EV2_AUTHENTICATED)
        {
            /* Create and Select Application */
            wStatus = CreateSelectApplication(APPLICATION_TYPE_NORMAL, PH_OFF);
            CHECK_SUCCESS_BREAK(wStatus);
            printf("\n");
        }

        /* Standard File Creation with Plain communication, Free Access Rights and no Authentication */
        printf("    Standard File Creation with SDM Enabled");
        wStatus = CreateStandardFile(STANDARD_DATA_FILE, PH_ON, bFileOption, aAccessRights, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                 : SUCCESS\n");

        /* Read the data from file */
        printf("    Read Data (Without Secure Dynamic Feature Enabled)      : ");
        wStatus = ReadData(STANDARD_DATA_FILE, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            0, 128, &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        printf("    Enabling Secure Dynamic Feature (FileAR.SDMMetaRead = Plain PICC Data Mirroring) ------------------\n");

        /* Perform Authentication */
        printf("        %s", AUTH_STATE(bAuth_State));
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(bAuth_State, APP_MASTER_KEY));
        printf(": SUCCESS\n");

        /* Enable Secure Dynamic Feature */
        wStatus = ChangeFileSettings(bFileOption, aAccessRights);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Select Application */
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP, pAID, NULL);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        Application Selection                               : SUCCESS\n");

        /* Read the data from file */
        printf("        Read Data (With Secure Dynamic Feature Enabled)     : ");
        wStatus = ReadData(STANDARD_DATA_FILE, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            0, 256, &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Extract the information */
        memcpy(aVCUID_PICCData, &pResponse[aSDMInfo[3]], 14);
        wVCUID_PICCDataLen = 14;

        memcpy(aSDMRdCtr, &pResponse[aSDMInfo[6]], 6);
        wSDMRdCtrLen = 6;

        memcpy(aMac, &pResponse[aSDMInfo[18]], 16);
        wMacLen = 16;

        printf("        VCUID (Mirrored)                                    : ");
        PRINT_DATA_NEWLINE(aVCUID_PICCData, wVCUID_PICCDataLen);
        ToBinary(aVCUID_PICCData, &wVCUID_PICCDataLen);
        printf("        VCUID (Not Mirrored)                                : ");
        PRINT_DATA_NEWLINE(aVCUID_PICCData, wVCUID_PICCDataLen);
        printf("\n");

        printf("        SDM Read Counter (Mirrored)                         : ");
        PRINT_DATA_NEWLINE(aSDMRdCtr, wSDMRdCtrLen);
        ToBinary(aSDMRdCtr, &wSDMRdCtrLen);
        printf("        SDM Read Counter (Not Mirrored)                     : ");
        PRINT_DATA_NEWLINE(aSDMRdCtr, wSDMRdCtrLen);
        printf("\n");

        printf("        SDM MAC (Mirrored, Returned by PICC)                : ");
        PRINT_DATA_NEWLINE(aMac, wMacLen);
        ToBinary(aMac, &wMacLen);
        printf("        SDM MAC (Not Mirrored, Returned by PICC)            : ");
        PRINT_DATA_NEWLINE(aMac, wMacLen);
        printf("\n");





        printf("    Enabling Secure Dynamic Feature (SDMMetaRead = No PICC Data Mirroring, EncFile Data) --------------\n");

        /* Enable Secure Dynamic Feature */
        aSDMInfo[0] |= 0x10; /* SDM Enc File Data  */
        aSDMInfo[1] = 0xFE; /* SDMFileRead2 = 0x0F, SDMCtrRet = 0x0E */
        aSDMInfo[2] = 0x00; /* SDMMetaRead = 0xF0, SDMFileRead = 0x00 */
        aAccessRights[0] = 0xE0;
        aAccessRights[1] = 0x00;
        wStatus = ChangeFileSettings(bFileOption, aAccessRights);

        /* Restore SDMInfo for Next operation */
        aSDMInfo[0] = 0xC1;
        aSDMInfo[1] = 0xFE;
        aSDMInfo[2] = 0xE0;
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Read the data from file */
        printf("        Read Data (With Secure Dynamic Feature Enabled)     : ");
        wStatus = ReadData(STANDARD_DATA_FILE, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            0, 256, &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Extract the information */
        memcpy(aVCUID_PICCData, &pResponse[aSDMInfo[3]], 32);
        wVCUID_PICCDataLen = 32;

        memcpy(aEncFileData, &pResponse[aSDMInfo[12]], 32);
        wEncFileDataLen = aSDMInfo[15];

        memcpy(aMac, &pResponse[aSDMInfo[18]], 16);
        wMacLen = 16;

        printf("        PICC Data (Enc, Mirrored)                           : ");
        PRINT_DATA_NEWLINE(aVCUID_PICCData, 32);

        ToBinary(aVCUID_PICCData, &wVCUID_PICCDataLen);
        printf("        PICC Data (Enc, Not Mirrored)                       : ");
        PRINT_DATA_NEWLINE(aVCUID_PICCData, wVCUID_PICCDataLen);
        wStatus = phalMfDuoX_DecryptSDMPICCData(&stMfDuoX, bKeyNo, 0, aVCUID_PICCData, wVCUID_PICCDataLen,
            &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        PICC Data (Plain, Not Mirrored)                     : ");
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Get SDM Read Counter */
        wStatus = phalMfDuoX_GetFileCounters(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_PLAIN, STANDARD_DATA_FILE,
            &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        memcpy(aSDMRdCtr, pResponse, 3);
        aSDMRdCtr[3] = 0x00;

        printf("        SDM ENCFile Data (Enc, Mirrored)                    : ");
        PRINT_DATA_NEWLINE(aEncFileData, wEncFileDataLen);
        ToBinary(aEncFileData, &wEncFileDataLen);
        wStatus = phalMfDuoX_DecryptSDMENCFileData(&stMfDuoX, PHAL_MFDUOX_VCUID_RDCTR_PRESENT, bKeyNo, 0,
            0, 0, aUid, bUid_Len, aSDMRdCtr, aEncFileData, wEncFileDataLen, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        SDM ENCFile Data (Plain, Not Mirrored)              : ");
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        printf("        SDM MAC (Mirrored)                                  : ");
        PRINT_DATA_NEWLINE(aMac, wMacLen);
        ToBinary(aMac, &wMacLen);
        printf("        SDM MAC (Not Mirrored)                              : ");
        PRINT_DATA_NEWLINE(aMac, wMacLen);

    } while(0);

    return wStatus;
}

phStatus_t Demo_SecureDynamicMessaging_SIG()
{
    phStatus_t wStatus = 0;
    uint8_t bKeyNo = 0;

    uint8_t aAccessRights[2] = { 0xEE, 0xEF };
    uint8_t aKeyPolicy[2] = { 0x20 , 0x00 };

    uint8_t aPublicKey[65];
    uint16_t wPublicKeyLen = 0;

    uint8_t aVCUID_PICCData[32];
    uint16_t wVCUID_PICCDataLen = 0;

    uint8_t aSDMRdCtr[6];
    uint16_t wSDMRdCtrLen = 0;

    uint8_t aSig[128];
    uint16_t wSigLen = 0;

    uint8_t bFileOption = (uint8_t) (PHAL_MFDUOX_FILE_OPTION_PLAIN | PHAL_MFDUOX_FILE_OPTION_SDM_MIRRORING_ENABLED);

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    do
    {
        bKeyNo = (uint8_t) (wIsAES256KeyType ? AES256_KEY_ADDRESS : AES128_KEY_ADDRESS);

        if(bAuth_State == PHAL_MFDUOX_EV2_AUTHENTICATED)
        {
            /* Create and Select Application */
            wStatus = CreateSelectApplication(APPLICATION_TYPE_NORMAL, PH_OFF);
            CHECK_SUCCESS_BREAK(wStatus);
            printf("\n");
        }

        /* Standard File Creation with Plain communication, Free Access Rights and no Authentication */
        printf("    Standard File Creation with SDM Enabled");
        wStatus = CreateStandardFile(STANDARD_DATA_FILE, PH_ON, bFileOption, aAccessRights, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                 : SUCCESS\n");

        /* Read the data from file */
        printf("    Read Data (Without Secure Dynamic Feature Enabled)      : ");
        wStatus = ReadData(STANDARD_DATA_FILE, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_NATIVE,
            0, 128, &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Authenticate using Application Master Key */
        printf("    Authentication using Application Master key (0)         ");
        wStatus = Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, APP_MASTER_KEY);
        CHECK_SUCCESS_BREAK(wStatus);
        printf(": SUCCESS\n");

        /* Perform Manage KeyPair to generate a private key for Signature generation. */
        printf("    KeyPair Generation for SDM Signature                    ");
        wStatus = phalMfDuoX_ManageKeyPair(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, ECC_SDM_KEY,
            PHAL_MFDUOX_TARGET_ACTION_GENERATE_KEY_PAIR, PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256,
            aKeyPolicy, 0x30, 0, 0, 0, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf(": SUCCESS\n\n");

        /* Copy public key to local buffer */
        memcpy(aPublicKey, pResponse, wRespLen);
        wPublicKeyLen = wRespLen;

        /* Load Public Key To Keystore. */
        wStatus = AddKey_ASymm(SDM_PUBLIC_KEY_ADDRESS, SDM_PUBLIC_KEY_PSOITION,
            PH_KEYSTORE_KEY_PAIR_PUBLIC, pResponse, (uint8_t) wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("    Enabling Secure Dynamic Feature (FileAR.SDMMetaRead = Plain PICC Data Mirroring) ------------------\n");

        /* Enable Secure Dynamic Feature */
        aSDMInfo[1] = (uint8_t) ((ECC_SDM_KEY << 4) | 0x0E); /* SDMFileRead2 = Private Key, SDMCtrRet = 0x0E */
        aSDMInfo[2] = 0xEF; /* SDMMetaRead = 0xE0, SDMFileRead = 0x0F */
        wStatus = ChangeFileSettings(bFileOption, aAccessRights);

        /* Restore SDMInfo for Next operation */
        aSDMInfo[1] = 0xFE;
        aSDMInfo[2] = 0xE0;
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Select Application */
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP, pAID, NULL);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        Application Selection                               : SUCCESS\n");

        /* Read the data from file */
        printf("        Read Data (With Secure Dynamic Feature Enabled)     : ");
        wStatus = ReadData(STANDARD_DATA_FILE, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_ISO,
            0, 256, &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Extract the information */
        memcpy(aVCUID_PICCData, &pResponse[aSDMInfo[3]], 14);
        wVCUID_PICCDataLen = 14;

        memcpy(aSDMRdCtr, &pResponse[aSDMInfo[6]], 6);
        wSDMRdCtrLen = 6;

        memcpy(aSig, &pResponse[aSDMInfo[18]], 128);
        wSigLen = 128;

        printf("        VCUID (Mirrored)                                    : ");
        PRINT_DATA_NEWLINE(aVCUID_PICCData, wVCUID_PICCDataLen);
        ToBinary(aVCUID_PICCData, &wVCUID_PICCDataLen);
        printf("        VCUID (Not Mirrored)                                : ");
        PRINT_DATA_NEWLINE(aVCUID_PICCData, wVCUID_PICCDataLen);
        printf("\n");

        printf("        SDM Read Counter (Mirrored)                         : ");
        PRINT_DATA_NEWLINE(aSDMRdCtr, wSDMRdCtrLen);
        ToBinary(aSDMRdCtr, &wSDMRdCtrLen);
        printf("        SDM Read Counter (Not Mirrored)                     : ");
        PRINT_DATA_NEWLINE(aSDMRdCtr, wSDMRdCtrLen);
        printf("\n");

        /* Verify SDM Signature received from PICC */
        printf("        SDM Signature Verification\n");
        printf("            Input                                           : ");
        PRINT_DATA_NEWLINE(&pResponse[aSDMInfo[10]], (uint16_t) (aSDMInfo[18] - aSDMInfo[10]));

        printf("            PublicKey                                       : ");
        PRINT_DATA_NEWLINE(aPublicKey, wPublicKeyLen);

        printf("            SDM Signature (Mirrored)                        : ");
        PRINT_DATA_NEWLINE(aSig, wSigLen);
        ToBinary(aSig, &wSigLen);
        printf("            SDM Signature (Not Mirrored)                    : ");
        PRINT_DATA_NEWLINE(aSig, wSigLen);

        wStatus = phalMfDuoX_VerifySDMSignature(&stMfDuoX, SDM_PUBLIC_KEY_ADDRESS, SDM_PUBLIC_KEY_PSOITION,
            &pResponse[aSDMInfo[10]], (uint16_t) (aSDMInfo[18] - aSDMInfo[10]), aSig, wSigLen);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            printf("            Verification Status                             : SUCCESS\n\n");
        else
            printf("            Verification Status                             : Failed\n\n");





        printf("    Enabling Secure Dynamic Feature (SDMMetaRead = No PICC Data Mirroring) ----------------------------\n");

        /* Enable Secure Dynamic Feature */
        aSDMInfo[1] = (uint8_t) ((ECC_SDM_KEY << 4) | 0x0E); /* SDMFileRead2 = Private Key, SDMCtrRet = 0x0E */
        aSDMInfo[2] = 0x0F; /* SDMMetaRead = 0x00, SDMFileRead = 0x0F */
        aAccessRights[0] = 0xE0;
        aAccessRights[1] = 0x00;
        wStatus = ChangeFileSettings(bFileOption, aAccessRights);

        /* Restore SDMInfo for Next operation */
        aSDMInfo[1] = 0xFE;
        aSDMInfo[2] = 0xE0;
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Read the data from file */
        printf("        Read Data (With Secure Dynamic Feature Enabled)     : ");
        wStatus = ReadData(STANDARD_DATA_FILE, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_ISO,
            0, 256, &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");

        /* Extract the information */
        memcpy(aVCUID_PICCData, &pResponse[aSDMInfo[3]], 32);
        wVCUID_PICCDataLen = 32;

        memcpy(aSig, &pResponse[aSDMInfo[18]], 128);
        wSigLen = 128;

        printf("        SDM Signature Verification\n");
        printf("            Input                                           : ");
        PRINT_DATA_NEWLINE(&pResponse[aSDMInfo[10]], (uint16_t) (aSDMInfo[18] - aSDMInfo[10]));

        printf("            PublicKey                                       : ");
        PRINT_DATA_NEWLINE(aPublicKey, wPublicKeyLen);

        printf("            SDM Signature (Mirrored)                        : ");
        PRINT_DATA_NEWLINE(aSig, wSigLen);
        ToBinary(aSig, &wSigLen);
        printf("            SDM Signature (Not Mirrored)                    : ");
        PRINT_DATA_NEWLINE(aSig, wSigLen);

        wStatus = phalMfDuoX_VerifySDMSignature(&stMfDuoX, SDM_PUBLIC_KEY_ADDRESS, SDM_PUBLIC_KEY_PSOITION,
            &pResponse[aSDMInfo[10]], (uint16_t) (aSDMInfo[18] - aSDMInfo[10]), aSig, wSigLen);
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            printf("            Verification Status                             : SUCCESS\n\n");
        else
            printf("            Verification Status                             : Failed\n\n");

        printf("        PICC Data (Enc, Mirrored)                           : ");
        PRINT_DATA_NEWLINE(aVCUID_PICCData, 32);

        ToBinary(aVCUID_PICCData, &wVCUID_PICCDataLen);
        printf("        PICC Data (Enc, Not Mirrored)                       : ");
        PRINT_DATA_NEWLINE(aVCUID_PICCData, wVCUID_PICCDataLen);
        wStatus = phalMfDuoX_DecryptSDMPICCData(&stMfDuoX, bKeyNo, 0, aVCUID_PICCData, wVCUID_PICCDataLen,
            &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        PICC Data (Plain, Not Mirrored)                     : ");
        PRINT_DATA_NEWLINE(pResponse, wRespLen);
        printf("\n");
    } while(0);

    return wStatus;
}

phStatus_t Demo_OriginalityCheck()
{
    phStatus_t wStatus = 0;
    uint32_t dwCertLen = 0;

    uint8_t aOptsA[] = { 0x80, 0x00 };
    uint8_t aKeyPolicy[2] = { 0x00 , 0x01 };

    uint8_t aPubKey[65];
    uint16_t wPubKeyLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    uint8_t * pCert = NULL;
    uint16_t wCertLen = 0;

    do
    {
        printf("    Originality Signature verification at PICC level --------------------------------------------------\n");

        /* Activate the card. */
        printf("        PICC Activation");
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                     : SUCCESS\n");

        /* Read the Certificate from file */
        printf("        Certificate Read from PICC                          : ");
        wStatus = ReadData(STANDARD_DATA_FILE_ORIGINALITY_CHECK, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_ISO,
            0, 3, &pResponse, &wRespLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);

        dwCertLen = (pResponse[2] << 16) | (pResponse[1] << 8) | pResponse[0];
        wStatus = ReadData(STANDARD_DATA_FILE_ORIGINALITY_CHECK, PHAL_MFDUOX_COMMUNICATION_PLAIN, PHAL_MFDUOX_CHAINING_ISO,
            3, dwCertLen, &pCert, &wCertLen, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Removed the Length from Certificate */
        PRINT_DATA_NEWLINE(pCert, wCertLen);
        printf("\n");

        /* Extract Public key from received Certificate */
        ExtractPublicKey(pCert, wCertLen, aPubKey, &wPubKeyLen);

        /* Perform Iso Internal Authenticate */
        printf("        Originaly Signature Verification (ISOInternalAuthenticate)\n");
        printf("            Private Key Number                              : %02X\n", ECC_ORIGINALITY_CHECK_KEY);
        printf("            Curve                                           : %02X\n", PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256);
        printf("            Public Key                                      : ");
        PRINT_DATA_NEWLINE(aPubKey, wPubKeyLen);
        printf("\n");
        printf("            OptsA                                           : ");
        PRINT_DATA_NEWLINE(aOptsA, sizeof(aOptsA));

        wStatus = phalMfDuoX_ISOInternalAuthenticate(&stMfDuoX, ECC_ORIGINALITY_CHECK_KEY,
            PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256, aPubKey, wPubKeyLen, aOptsA,
            (uint8_t) sizeof(aOptsA), NULL, 0);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Status                                          : SUCCESS\n\n");





        printf("    Originality Signature verification at Application Level -------------------------------------------\n");

        /* Create and Select Application */
        wStatus = CreateSelectApplication(APPLICATION_TYPE_ECC, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Perform Authentication. */
        printf("        Authentication");
        wStatus = Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, APP_MASTER_KEY);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                      : SUCCESS\n\n");

        /* Generate KeyPair */
        printf("    Generating Key Pair for signing operation in device (ManageKeyPair)\n");
        printf("            Communication Mode                              : 30 (FULL)\n");
        printf("            Key Number                                      : %02X\n", ECC_ORIGINALITY_CHECK_KEY);
        printf("            Option                                          : 00 (Generate KeyPair)\n");
        printf("            Curve ID                                        : 0Ch (NIST-P256)\n");
        printf("            Key Policy                                      : ");
        PRINT_DATA_NEWLINE(aKeyPolicy, 2);
        printf("            Write Access                                    : 30\n");

        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_ManageKeyPair(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL,
            ECC_ORIGINALITY_CHECK_KEY, PHAL_MFDUOX_TARGET_ACTION_GENERATE_KEY_PAIR,
            PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256, aKeyPolicy, 0x30, 0, 0, 0,
            &pResponse, &wPubKeyLen));
        memcpy(aPubKey, pResponse, wPubKeyLen);

        printf("            Response                                        : ");
        PRINT_DATA_NEWLINE(aPubKey, wPubKeyLen);

        printf("            Status                                          : SUCCESS\n\n");

        /* Perform Iso Internal Authenticate */
        printf("        Originaly Signature Verification (ISOInternalAuthenticate)\n");
        printf("            Private Key Number                              : %02X\n", ECC_ORIGINALITY_CHECK_KEY);
        printf("            Curve                                           : %02X\n", PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256);
        printf("            Public Key                                      : ");
        PRINT_DATA_NEWLINE(aPubKey, wPubKeyLen);
        printf("\n");

        printf("            OptsA                                           : ");
        PRINT_DATA_NEWLINE(aOptsA, sizeof(aOptsA));
        printf("\n");

        /* Select Application */
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP,
            pAID, NULL);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Perform Iso Internal Authenticate */
        wStatus = phalMfDuoX_ISOInternalAuthenticate(&stMfDuoX, ECC_ORIGINALITY_CHECK_KEY,
            PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256, aPubKey, wPubKeyLen, aOptsA,
            (uint8_t) sizeof(aOptsA), NULL, 0);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Status                                          : SUCCESS\n\n");

    } while(0);

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_GetVersion()
{
    phStatus_t wStatus = 0;
    uint8_t bOffset = 0;
    uint8_t bUIDLen = 7;

    uint8_t * pVersion = NULL;
    uint16_t wVerLen = 0;

    do
    {
        /* Activate the card. */
        printf("        PICC Activation");
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                     : SUCCESS\n");

        /* Select Master Application */
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP, aAppID_Master, NULL);
        CHECK_SUCCESS_BREAK(wStatus);

        /* Get Version in Not Authenticate State ----------------------------------------------------------------------------- */
        printf("        Get Version in Not Authenticate State               : ");
        wStatus = phalMfDuoX_GetVersion(&stMfDuoX, PHAL_MFDUOX_GET_VERSION_RETURN_FAB_ID,
            &pVersion, &wVerLen);
        CHECK_SUCCESS_BREAK(wStatus);

        PRINT_DATA_NEWLINE(pVersion, wVerLen);
        printf("\n");

        /* Get Version in Authenticate State --------------------------------------------------------------------------------- */
        printf("        Authentication using PICC Master key                ");
        wStatus = Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, PICC_MASTER_KEY);
        CHECK_SUCCESS_BREAK(wStatus);
        printf(": SUCCESS\n");

        printf("        Get Version in Authenticate State                   : ");
        wStatus = phalMfDuoX_GetVersion(&stMfDuoX, PHAL_MFDUOX_GET_VERSION_RETURN_FAB_ID,
            &pVersion, &wVerLen);
        CHECK_SUCCESS_BREAK(wStatus);

        PRINT_DATA_NEWLINE(pVersion, wVerLen);
        printf("\n");

        printf("        Hardware Information\n");
        printf("            Vendor ID                   : %02Xh\n", pVersion[bOffset++]);
        printf("            Hardware Type               : %02Xh\n", pVersion[bOffset++]);
        printf("            Hardware SubType            : %02Xh\n", pVersion[bOffset++]);
        printf("            Hardware Major Version      : %02Xh\n", pVersion[bOffset++]);
        printf("            Hardware Minor Version      : %02Xh\n", pVersion[bOffset++]);
        printf("            Hardware Storage Size       : %02Xh\n", pVersion[bOffset++]);
        printf("            Hardware Protocol           : %02Xh\n\n", pVersion[bOffset++]);
        printf("        Software Information\n");
        printf("            Vendor ID                   : %02Xh\n", pVersion[bOffset++]);
        printf("            Software Type               : %02Xh\n", pVersion[bOffset++]);
        printf("            Software SubType            : %02Xh\n", pVersion[bOffset++]);
        printf("            Software Major Version      : %02Xh\n", pVersion[bOffset++]);
        printf("            Software Minor Version      : %02Xh\n", pVersion[bOffset++]);
        printf("            Software Storage Size       : %02Xh\n", pVersion[bOffset++]);
        printf("            Software Protocol           : %02Xh\n\n", pVersion[bOffset++]);
        printf("        Manufacturer Information\n");

        if(wVerLen > 29)
        {
            printf("            UID Format                  : %02Xh\n", pVersion[bOffset++]);

            bUIDLen = pVersion[bOffset++];
            printf("            UID Length                  : %02Xh\n", bUIDLen);
        }

        printf("            Unique Identifier           : ");
        PRINT_DATA_NEWLINE(&pVersion[bOffset], bUIDLen);
        bOffset += bUIDLen;

        printf("            Batch Number                : ");
        PRINT_DATA_NEWLINE(&pVersion[bOffset], 3);
        bOffset += 3;

        printf("            Type Identifier             : " );
        PRINT_DATA_NEWLINE(&pVersion[bOffset], 2);
        bOffset += 2;

        printf("            Calander Week of Production : %02Xh\n", pVersion[bOffset++]);
        printf("            Year of Production          : %02Xh\n", pVersion[bOffset++]);
        printf("            Fabrication Identifier      : %02Xh\n", pVersion[bOffset]);
        printf("\n");

    } while(0);

    return wStatus;
}

phStatus_t Demo_ProximityCheck()
{
    phStatus_t wStatus = 0;
    uint8_t bOption = 0;
    uint8_t bKeyNo = 0;

    uint8_t aPubRespTime[2] = { 0x00, 0x00 };
    uint8_t aCumRndRC[16];

    uint8_t aResponse[10];
    uint16_t wRespLen = 0;

    do
    {
        /* Activate the card. */
        printf("        PICC Activation");
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                     : SUCCESS\n");

        /* Select Master Application */
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP, aAppID_Master, NULL);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("        Enable VC Configuration and Proximity Key");
        wStatus = EnableVCProximityKey();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("           : SUCCESS\n\n");

        /* Evaluate KeyType to use. */
        bKeyNo = (uint8_t) (wIsAES256KeyType ? AES256_KEY_ADDRESS : AES128_KEY_ADDRESS);

        printf("    Proximity Check in Not Authenticate State ---------------------------------------------------------\n");

        /* Activate the card. */
        printf("        PICC Activation");
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                     : SUCCESS\n");

        /* Select Master Application */
        printf("        Master Application Selection (AID: 00 00 00)");
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP, aAppID_Master, NULL);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        : SUCCESS\n");

        printf("        Proximity Check (PPC, PC and VPC)\n");
        printf("            Random Number Generation                        : 01 (Rnd Generated by NXP Reader Library)\n");
        wStatus = phalVca_ProximityCheckNew(&stVca, PHAL_VCA_PC_RNDC_GENERATE, NULL, 8, bKeyNo,
            0, NULL, 0, &bOption, aPubRespTime, aResponse, &wRespLen, aCumRndRC);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("            Option                                          : %02X (%s Present)\n", bOption,
            ((bOption & 0x01) ? "PPS1" : "Active BitRate"));

        printf("            Published Response Time                         : ");
        PRINT_DATA_NEWLINE(aPubRespTime, 2);

        if((bOption & 0x01)) printf("            PPS1 (Protocol Parameter Selection 1)           : ");
        else printf("            Active Bit Rate                                 : ");
        PRINT_DATA_NEWLINE(aResponse, wRespLen);

        printf("            Status                                          : SUCCESS\n\n");





        printf("    Proximity Check in Authenticate State -------------------------------------------------------------\n");

        /* Authenticate using PICC MasterKey */
        printf("        Authentication using PICC Master Key");
        PH_CHECK_SUCCESS_FCT(wStatus, Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, PICC_MASTER_KEY));

        printf("                : SUCCESS\n");

        printf("        Proximity Check (PPC, PC and VPC)\n");
        printf("            Random Number Generation                        : 01 (Rnd Generated by NXP Reader Library)\n");
        wStatus = phalVca_ProximityCheckNew(&stVca, PHAL_VCA_PC_RNDC_GENERATE, NULL, 8, bKeyNo,
            0, NULL, 0, &bOption, aPubRespTime, aResponse, &wRespLen, aCumRndRC);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("            Option                                          : %02X (%s Present)\n", bOption,
            ((bOption & 0x01) ? "PPS1" : "Active BitRate"));

        printf("            Published Response Time                         : ");
        PRINT_DATA_NEWLINE(aPubRespTime, 2);

        if((bOption & 0x01)) printf("            PPS1 (Protocol Parameter Selection 1)           : ");
        else printf("            Active Bit Rate                                 : ");
        PRINT_DATA_NEWLINE(aResponse, wRespLen);

        printf("            Status                                          : SUCCESS\n\n");

    } while(0);

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_EVCharging()
{
    phStatus_t wStatus = 0;

    uint8_t aConfig[] = { 0x02, 0x0E };
    uint8_t aAccessRights[2] = { 0xE, 0x00 };
    uint8_t aKeyPolicy[2] = { 0x10, 0x00 };

    uint8_t aCertLen[3] = { 0x00, 0x00, 0x00 };
    uint32_t dwCertLen = 0;

    uint8_t aArbData[] =
    {
        0x5E, 0xA5, 0xC4, 0xD0, 0x8C, 0x33, 0xB7, 0xE4, 0xCA, 0xFD, 0x64, 0xEE, 0x37, 0x2D, 0x89, 0x0A,
        0x55, 0xCC, 0xB5, 0x8E, 0x7D, 0xCF, 0xBB, 0x66, 0x25, 0xD8, 0xAF, 0xE8, 0x84, 0xE2, 0x65, 0x8B
    };
    uint16_t wArbDataLen = sizeof(aArbData);

    uint8_t aPublicKey[65];
    uint16_t wPublicKeyLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    do
    {
        /* Activate the card. */
        printf("    PICC Activation");
        wStatus = ActivateCard();
        CHECK_SUCCESS_BREAK(wStatus);
        printf("                                        : SUCCESS\n\n");

        printf("    Personalization of EV Charging Feature ------------------------------------------------------------\n");

        /* Create and Select Application */
        wStatus = CreateSelectApplication(APPLICATION_TYPE_EV_CHARGING, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("\n");

        /* Perform Authentication. */
        printf("        Authentication using Application Master key (0)     ");
        wStatus = Authenticate(PHAL_MFDUOX_EV2_AUTHENTICATED, APP_MASTER_KEY);
        CHECK_SUCCESS_BREAK(wStatus);
        printf(": SUCCESS\n");

        printf("        Enabling EV Charging Feature                        ");
        wStatus = phalMfDuoX_GetConfiguration(&stMfDuoX, PHAL_MFDUOX_SET_CONFIGURATION_SM_CONFIGURATION, PH_ON,
            &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);

        memcpy(aConfig, pResponse, wRespLen);

        aConfig[1] |= 0x10;
        wStatus = phalMfDuoX_SetConfiguration(&stMfDuoX, PHAL_MFDUOX_SET_CONFIGURATION_SM_CONFIGURATION,
            aConfig, 2);
        CHECK_SUCCESS_BREAK(wStatus);
        printf(": SUCCESS\n");

        /* Create standard File for storing Certificate */
        printf("        Standard File Creation for storing Certificate      ");
        wStatus = CreateStandardFile(STANDARD_DATA_FILE_ECC, PH_ON, PHAL_MFDUOX_FILE_OPTION_FULL,
            aAccessRights, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf(": SUCCESS\n");

        /* Create standard File for storing Certificate */
        printf("        Standard File Creation for EV Charging Operation    ");
        aAccessRights[0] = 0xEE; aAccessRights[1] = 0xEE;
        wStatus = CreateStandardFile(STANDARD_DATA_FILE_EVC, PH_ON, PHAL_MFDUOX_FILE_OPTION_PLAIN,
            aAccessRights, PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf(": SUCCESS\n");

        /* Write Certificate */
        printf("        Writing EV Certificate to File\n");
        dwCertLen = sizeof(aP256_DEVICE_Cert);
        memcpy(aCertLen, (uint8_t *) &dwCertLen, 3);
        wStatus = Write(STANDARD_DATA_FILE_ECC, PHAL_MFDUOX_COMMUNICATION_FULL, PHAL_MFDUOX_CHAINING_ISO,
            0, aCertLen, (uint16_t)sizeof(aCertLen), PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Certificate Length                              : ");
        PRINT_DATA_NEWLINE(aCertLen, 3);

        wStatus = Write(STANDARD_DATA_FILE_ECC, PHAL_MFDUOX_COMMUNICATION_FULL, PHAL_MFDUOX_CHAINING_ISO,
            3, aP256_DEVICE_Cert, (uint16_t) sizeof(aP256_DEVICE_Cert), PH_OFF);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Certificate                                     : ");
        PRINT_DATA_NEWLINE(aP256_DEVICE_Cert, sizeof(aP256_DEVICE_Cert));

        /* Load Device Private Key. */
        printf("        Importing Private key to Device (ManageKeyPair)\n");
        printf("            Private Key                                     : ");
        PRINT_DATA_NEWLINE(aP256_DEVICE_PrivateKey, sizeof(aP256_DEVICE_PrivateKey));

        wStatus = phalMfDuoX_ManageKeyPair(&stMfDuoX, PHAL_MFDUOX_COMMUNICATION_FULL, ECC_EV_CHARGING_KEY,
            PHAL_MFDUOX_TARGET_ACTION_IMPORT_PRIVATE_KEY, PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256,
            aKeyPolicy, 0x3E, 0, ECC_P256_DEVICE_PRIVATE_KEY_ADDRESS,
            ECC_P256_DEVICE_PRIVATE_KEY_POSITION, &pResponse,
            &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Status                                          : SUCCESS\n\n");

        printf("        Enabling CryptoAPI for Signing                      ");
        wStatus = phalMfDuoX_GetConfiguration(&stMfDuoX, PHAL_MFDUOX_SET_CONFIGURATION_CRYPTO_API_MANAGEMENT,
            PH_ON, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);

        memcpy(aConfig, pResponse, wRespLen);

        aConfig[0] = 0x02;
        aConfig[1] = 0x0E;
        wStatus = phalMfDuoX_SetConfiguration(&stMfDuoX, PHAL_MFDUOX_SET_CONFIGURATION_CRYPTO_API_MANAGEMENT,
            aConfig, 2);
        CHECK_SUCCESS_BREAK(wStatus);
        printf(": SUCCESS\n\n");





        printf("    Execution of EV Charging Feature ------------------------------------------------------------------\n");

        /* Select Application */
        wStatus = phalMfDuoX_SelectApplication(&stMfDuoX, PHAL_MFDUOX_SELECT_PRIMARY_APP, aAppID_EVC, NULL);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("        Application Selection                               : SUCCESS\n");

        printf("        EV Charging Message Signing\n");
        printf("            Arbitrary data to sign                          : ");
        PRINT_DATA_NEWLINE(aArbData, wArbDataLen);
        wStatus = phalMfDuoX_VdeECDSASign(&stMfDuoX, 0, 0, aArbData, wArbDataLen, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Signature                                       : ");
        PRINT_DATA_NEWLINE(pResponse, wRespLen);

        printf("            Status                                          : SUCCESS\n\n");

        /* Extract public Key from Certificate. */
        printf("        Signature Versification\n");
        ExtractPublicKey(aP256_DEVICE_Cert, sizeof(aP256_DEVICE_Cert), aPublicKey, &wPublicKeyLen);
        wStatus = phCryptoASym_ECC_LoadKeyDirect(stMfDuoX.pCryptoDataParamsASym, (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY |
            PH_CRYPTOASYM_CURVE_ID_SECP256R1), aPublicKey, wPublicKeyLen);
        CHECK_SUCCESS_BREAK(wStatus);

        printf("            Public Key                                      : ");
        PRINT_DATA_NEWLINE(aPublicKey, wPublicKeyLen);

        printf("            Signature                                       : ");
        PRINT_DATA_NEWLINE(pResponse, wRespLen);

        printf("            Message                                         : ");
        PRINT_DATA_NEWLINE(aArbData, wArbDataLen);

        wStatus = phCryptoASym_ECC_Verify(stMfDuoX.pCryptoDataParamsASym, PH_EXCHANGE_DEFAULT, PH_CRYPTOASYM_HASH_ALGO_SHA256,
            aArbData, wArbDataLen, pResponse, wRespLen);
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            printf("            Verification Status                             : SUCCESS\n\n");
        else
            printf("            Verification Status                             : Failed\n\n");

        /* Perform VdeReadData on Standard Data File */
        printf("        VDE Read Data Operation\n");
        printf("            File Identifier (FileID)                        : %02X\n", STANDARD_DATA_FILE_EVC);
        wStatus = phalMfDuoX_VdeReadData(&stMfDuoX, PH_EXCHANGE_DEFAULT, STANDARD_DATA_FILE_EVC, 0, 0, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Response                                        : ");
        PRINT_DATA_NEWLINE(pResponse, wRespLen);

        printf("            Status                                          : SUCCESS\n\n");

        /* Perform VdeWriteData on Standard Data File */
        printf("        VDE Write Data Operation\n");
        printf("            File Identifier (FileID)                        : %02X\n", STANDARD_DATA_FILE_EVC);
        printf("            Data                                            : ");
        PRINT_DATA_NEWLINE(aData, sizeof(aData));
        wStatus = phalMfDuoX_VdeWriteData(&stMfDuoX, PHAL_MFDUOX_EV_OPERATION_WRITE, 0, aData, (uint16_t)sizeof(aData));
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Status                                          : SUCCESS\n\n");

        /* Perform VdeReadData on Standard Data File */
        printf("        VDE Read Data Operation\n");
        printf("            File Identifier (FileID)                        : %02X\n", STANDARD_DATA_FILE_EVC);
        wStatus = phalMfDuoX_VdeReadData(&stMfDuoX, PH_EXCHANGE_DEFAULT, STANDARD_DATA_FILE_EVC, 0, 0, &pResponse, &wRespLen);
        CHECK_SUCCESS_BREAK(wStatus);
        printf("            Response                                        : ");
        PRINT_DATA_NEWLINE(pResponse, wRespLen);

        printf("            Status                                          : SUCCESS\n\n");

    } while(0);

    return wStatus;
}

/* Logging function. */
#ifdef NXPBUILD__PH_LOG
void MfDuoX_Log(void* pDataParams, uint8_t bOption, phLog_LogEntry_t* pEntries, uint16_t wEntryCount)
{
    uint16_t wIteration;
    UNUSED_VARIABLE(bOption);
    UNUSED_VARIABLE(pDataParams);

    for(wIteration = 0; wIteration < wEntryCount; wIteration++)
    {
        if((strcmp(pEntries[0].pString, (char *) "phhalHw_Exchange") != 0) ||
            (strstr(pEntries[wIteration].pString, (char *) "CompId") != NULL) ||
            (strstr(pEntries[wIteration].pString, (char *) "Status") != NULL) ||
            (strstr(pEntries[wIteration].pString, (char *) "status") != NULL))
            continue;

        if(wIteration == 0)
            printf("\n");
        else
        {
            printf("        %s: ", pEntries[wIteration].pString);
        }

        if(pEntries[wIteration].wDataLen != 0)
            PRINT_DATA_NEWLINE((uint8_t *) (pEntries[wIteration].pData), pEntries[wIteration].wDataLen);
    }
}
#endif /* NXPBUILD__PH_LOG */
