/*
 * Copyright 2013 - 2015, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * TMIUtils Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHTMIUTILS_H
#define PHTMIUTILS_H

#include <ph_Status.h>
#include <ph_TypeDefs.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PH_TMIUTILS

/** \defgroup phTMIUtils TMI Utilities
 * \brief This layer provides Utilities for operating on transaction MAC.
 * @{
 */

/** \name Activate TMI options */
/** @{ */
#define PH_TMIUTILS_DEACTIVATE_TMI			0x00U	/**< Deactivate TMI collection. */
#define PH_TMIUTILS_ACTIVATE_TMI			0x01U	/**< Activate TMI collection. */
#define PH_TMIUTILS_PAUSE_TMI				0x02U	/**< Pause TMI collection. */
#define PH_TMIUTILS_RESUME_TMI				0x03U	/**< Resume Paused TMI collection. */
#define PH_TMIUTILS_RESET_TMI				0x04U	/**< Reset TMI collection Buffer Index to 0. */
/** @} */

/** \name Configuration Options */
/** @{ */
#define PH_TMIUTILS_TMI_STATUS				0x01U	/**< Config Option for TMI Status.*/
#define PH_TMIUTILS_TMI_OFFSET_LENGTH		0x02U	/**< Config option for Length offset in TMI Length */
#define PH_TMIUTILS_TMI_BUFFER_INDEX		0x04U	/**< Config option for buffer index */
#define PH_TMIUTILS_TMI_OFFSET_VALUE		0x08U	/**< Config option for Offset Value */
/** @} */

/** \name Flags for CollectTMI */
/** @{ */
#define PH_TMIUTILS_NO_PADDING				0x00U	/**< Flag indicating that padding is not required.*/
#define PH_TMIUTILS_READ_INS				0x01U	/**< Flag indicating that instruction is Read Operation.*/
#define PH_TMIUTILS_ZEROPAD_CMDBUFF			0x02U	/**< Flag to indicate zero padding after Cmd buffer */
#define PH_TMIUTILS_ZEROPAD_DATABUFF		0x04U	/**< Flag to indicate zero padding after Data buffer */
/** @} */

/** \brief Definition of a singe Log entry */
typedef struct
{
	uint8_t * pTMIBuffer;							/**< Pointer to TMI Buffer. */
	uint32_t dwTMIBufLen;							/**< Length of TMI Buffer. */
	uint32_t dwTMIbufIndex;							/**< Pointer indicating the TMI Buffer fill index. */
	uint8_t bTMIStatus;								/**< Indicates whether TMI collection is PH_ON or PH_OFF. */
	uint32_t dwOffsetInTMI;							/**< Indicates the offset in TMI buffer where the Length field is stored in case of unspecified length read. */
} phTMIUtils_t;

/**
 * \brief Initialize TMI Utils component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phTMIUtils_Init(
		phTMIUtils_t * pDataParams,					/**< [In] Pointer to this layers parameter structure. */
		uint8_t * pTMIBuffer,						/**< [In] Input buffer for collecting the command / data information.. */
		uint32_t dwBufLen							/**< [In] Size of input buffer. */
	);
/**
 * \brief Perfomrs Activation, Deactivation, Resume, Reset and Pause of TmiCollection.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phTMIUtils_ActivateTMICollection(
		phTMIUtils_t * pDataParams,					/**< [In] Pointer to this layers parameter structure. */
		uint8_t bOption								/**< [In] One of the below options.
													 *			\arg #PH_TMIUTILS_DEACTIVATE_TMI
													 *			\arg #PH_TMIUTILS_ACTIVATE_TMI
													 *			\arg #PH_TMIUTILS_PAUSE_TMI
													 *			\arg #PH_TMIUTILS_RESUME_TMI
													 *			\arg #PH_TMIUTILS_RESET_TMI
													 */
	);

/**
 * \brief Get TMI buffer
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phTMIUtils_GetTMI(
		phTMIUtils_t * pDataParams,					/**< [In] Pointer to this layers parameter structure. */
		uint8_t ** ppTMIBuffer,						/**< [Out] Get the collected command / data information. */
		uint32_t * dwTMILen							/**< [Out] Length of bytes available in ppTMIBuffer. */
	);

/**
 * \brief Performs collection of command / data information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phTMIUtils_CollectTMI(
		phTMIUtils_t * pDataParams,					/**< [In] Pointer to this layers parameter structure. */
		uint8_t bOption,							/**< [In] One of the below values.
													 *			\arg #PH_TMIUTILS_NO_PADDING
													 *			\arg #PH_TMIUTILS_READ_INS
													 *			\arg #PH_TMIUTILS_ZEROPAD_CMDBUFF
													 *			\arg #PH_TMIUTILS_ZEROPAD_DATABUFF
													 */
		uint8_t * pCmdBuffer,						/**< [In] Buffer containing the command information. */
		uint16_t wCmdBufLen,						/**< [In] Length of bytes available in pCmdBuffer. */
		uint8_t * pData,							/**< [In] Buffer containing the data information. */
		uint32_t dwDataLen,							/**< [In] Length of bytes available in pData. */
		uint16_t wBlockSize							/**< [In] Block size for appending padding bytes if #PH_TMIUTILS_NO_PADDING is not used. */
	);

/**
 * \brief Performs SetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phTMIUtils_SetConfig(
		phTMIUtils_t * pDataParams,					/**< [In] Pointer to this layers parameter structure. */
		uint16_t wConfig,							/**< [In] One of the Below Configurations.
													 *			\arg #PH_TMIUTILS_TMI_OFFSET_LENGTH
													 *			\arg #PH_TMIUTILS_TMI_OFFSET_VALUE
													 */
		uint32_t dwValue							/**< [In] The value for the given configuration. */
	);

/**
 * \brief Performs GetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phTMIUtils_GetConfig(
		phTMIUtils_t * pDataParams,					/**< [In] Pointer to this layers parameter structure. */
		uint16_t wConfig,							/**< [In] One of the Below Configurations.
												 	 *			\arg #PH_TMIUTILS_TMI_STATUS
													 *			\arg #PH_TMIUTILS_TMI_OFFSET_LENGTH
													 *			\arg #PH_TMIUTILS_TMI_BUFFER_INDEX
													 */
		uint32_t *pValue							/**< [Out] The value for the given configuration. */
	);


/**
 * end of group phTMIUtils
 * @}
 */
#endif /* NXPBUILD__PH_TMIUTILS */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif  /* PHTMIUTILS_H */
