/*
 * Copyright 2017 - 2020, 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic ICode Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  CHu: Generated 02. January 2017
 *
 */

#ifndef PHALICODE_H
#define PHALICODE_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef NXPBUILD__PHAL_ICODE
/**
 * \addtogroup phalICode_CommonDefs
 * @{
 */

/**
 * \defgroup phalICode_Defines_BufferSize Buffer Size
 * @{
 */
#define PHAL_ICODE_BLOCK_SIZE                                               4U  /**< Macro to represent the ICode block size. */
/**
 * end of group phalMfEcc_Defines_BufferSize
 * @}
 */

/**
 * end of group phalICode_CommonDefs
 * @}
 */
#endif /* NXPBUILD__PHAL_ICODE */

#ifdef NXPBUILD__PHAL_ICODE_SW
/***************************************************************************************************************************************/
/* Software DataParams and Initialization Interface.                                                                                   */
/***************************************************************************************************************************************/

/**
 * \defgroup phalICode_Sw Component : Software
 * \brief Sub-Component to initialize ICode Generic component as Software.
 * @{
 */

#define PHAL_ICODE_SW_ID                                                0x01U   /**< ID for ICode Software layer. */

/** \brief Data structure for ICode Software layer implementation  */
typedef struct
{
    uint16_t wId;                                                               /**< Layer ID for this component, NEVER MODIFY! */
    void * pPalSli15693DataParams;                                              /**< Pointer to the parameter structure of the underlying palSli15693 layer. */
    void * pCryptoDataParams;                                                   /**< Pointer to the parameter structure of the underlying Crypto layer for encryption / Decryption. */
    void * pCryptoRngDataParams;                                                /**< Pointer to the parameter structure of the underlying Crypto layer for random number generation. */
    void * pKeyStoreDataParams;                                                 /**< Pointer to the parameter structure of the underlying KeyStore layer. */
    uint8_t aRnd_Challenge[70];                                                 /**< Buffer to store the random number generated by Challenge command. */
    uint8_t bBuffering;                                                         /**< Enable or disable the data buffering. */
    uint8_t bChallengeType;                                                     /**< Type of Challenge processed by \ref phalICode_Challenge "Challenge" command.
                                                                                 *   This will help to update the timing information while reading the TResponse using
                                                                                 *   \ref phalICode_ReadBuffer "Read Buffer".
                                                                                 */
    uint16_t wChallange_Timeout;                                                /**< Timeout value to be configured for Challenge command. This will be the timing to wait for response. */
    uint16_t wChallange_Timeout_Unit;                                           /**< Timeout unit to be configured for Challenge command */
} phalICode_Sw_DataParams_t;

/**
 * \brief Initialization API for ICode software component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *          - If the input size doest not match the DataParams size of this component.
 *          - If any of the DataParams are null.
 */
phStatus_t phalICode_Sw_Init(
        phalICode_Sw_DataParams_t * pDataParams,                                /**< [In] Pointer to this layers parameter structure. */
        uint16_t wSizeOfDataParams,                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pPalSli15693DataParams,                                          /**< [In] Pointer to the parameter structure of the underlying palSli15693 layer. */
        void * pCryptoDataParams,                                               /**< [In] Pointer to the parameter structure of the underlying Crypto layer for encryption / Decryption. */
        void * pCryptoRngDataParams,                                            /**< [In] Pointer to the parameter structure of the underlying Crypto layer for random number generation. */
        void * pKeyStoreDataParams                                              /**< [In] Pointer to the parameter structure of the underlying KeyStore layer. */
    );

/**
 * end of group phalICode_Sw
 * @}
 */
#endif /* NXPBUILD__PHAL_ICODE_SW */










/***************************************************************************************************************************************/
/* ICode Generic interface declarations.                                                                                               */
/***************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_ICODE

/**
 * \defgroup phalICode ICode
 * \brief Generic component that provides the command description of ICode products.
 * @{
 */

/** \defgroup phalICode_Errors ErrorCodes
 * \brief Description of error codes returned by VICC and Reader Library.
 * @{
 */

/**
 * \defgroup phalICode_ErrorCodes_VICC ErrorCodes_VICC
 * \brief Description of actual error codes received from VICC. These codes are for reference purpose only.
 *
 * \note
 * All the interfaces will return codes mentioned in \ref phalICode_ErrorCodes_Mapped "Library Mapped ErrorCodes" and not these codes.
 * @{
 */
#define PHAL_ICODE_RESP_ERR_COMMAND_NOT_SUPPORTED                       0x01U   /**< The command is not supported, i.e. the request code is not recognized. */
#define PHAL_ICODE_RESP_ERR_COMMAND_NOT_RECOGNIZED                      0x02U   /**< The command is not recognized, for example: a format error occurred. */
#define PHAL_ICODE_RESP_ERR_COMMAND_OPTION_NOT_SUPPORTED                0x03U   /**< The command option is not supported. */
#define PHAL_ICODE_RESP_ERR_NO_INFORMATION                              0x0FU   /**< Error with no information given or a specific error code is not supported. */
#define PHAL_ICODE_RESP_ERR_BLOCK_NOT_AVAILABLE                         0x10U   /**< The specified block is not available (does not exist). */
#define PHAL_ICODE_RESP_ERR_BLOCK_LOCKED                                0x11U   /**< The specified block is already locked and thus cannot be locked again. */
#define PHAL_ICODE_RESP_ERR_CONTENT_CHANGE_FAILURE                      0x12U   /**< The specified block is locked and its content cannot be changed. */
#define PHAL_ICODE_RESP_ERR_BLOCK_PROGRAMMING_FAILURE                   0x13U   /**< The specified block was not successfully programmed. */
#define PHAL_ICODE_RESP_ERR_BLOCK_NOT_LOCKED                            0x14U   /**< The specified block was not successfully locked. */
#define PHAL_ICODE_RESP_ERR_BLOCK_PROTECTED                             0x15U   /**< The specified block is protected. */
#define PHAL_ICODE_RESP_ERR_GENERIC_CRYPTO_ERROR                        0x40U   /**< Generic cryptographic error. */
/**
 * end of group phalICode_ErrorCodes_VICC
 * @}
 */

/** \defgroup phalICode_ErrorCodes_Mapped ErrorCodes_Mapped
 * \brief Description of mapped error codes from Library with respect to VICC Error codes.
 * @{
 */
#define PHAL_ICODE_ERR_CUSTOM_COMMANDS_ERROR        (PH_ERR_CUSTOM_BEGIN + 0U)  /**< Custom commands Error codes. */
#define PHAL_ICODE_ERR_COMMAND_NOT_SUPPORTED        (PH_ERR_CUSTOM_BEGIN + 1U)  /**< The command is not supported, i.e. the request code is not recognized.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_COMMAND_NOT_SUPPORTED "Command Not Supported" error.
                                                                                 */
#define PHAL_ICODE_ERR_COMMAND_NOT_RECOGNIZED       (PH_ERR_CUSTOM_BEGIN + 2U)  /**< The command is not recognized, for example: a format error occurred.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_COMMAND_NOT_RECOGNIZED "Command not Recognized" error.
                                                                                 */
#define PHAL_ICODE_ERR_COMMAND_OPTION_NOT_SUPPORTED (PH_ERR_CUSTOM_BEGIN + 3U)  /**< The command option is not supported.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_COMMAND_OPTION_NOT_SUPPORTED "Command Option not Supported" error.
                                                                                 */
#define PHAL_ICODE_ERR_NO_INFORMATION               (PH_ERR_CUSTOM_BEGIN + 4U)  /**< Error with no information given or a specific error code is not supported.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_NO_INFORMATION "No Information" error.
                                                                                 */
#define PHAL_ICODE_ERR_BLOCK_NOT_AVAILABLE          (PH_ERR_CUSTOM_BEGIN + 5U)  /**< The specified block is not available (does not exist).
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_BLOCK_NOT_AVAILABLE "Not Available" error.
                                                                                 */
#define PHAL_ICODE_ERR_BLOCK_LOCKED                 (PH_ERR_CUSTOM_BEGIN + 6U)  /**< The specified block is already locked and thus cannot be locked again.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_BLOCK_LOCKED "Block Locked" error.
                                                                                 */
#define PHAL_ICODE_ERR_CONTENT_CHANGE_FAILURE       (PH_ERR_CUSTOM_BEGIN + 7U)  /**< The specified block is locked and its content cannot be changed.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_CONTENT_CHANGE_FAILURE "Content Change Failure" error.
                                                                                 */
#define PHAL_ICODE_ERR_BLOCK_PROGRAMMING_FAILURE    (PH_ERR_CUSTOM_BEGIN + 8U)  /**< The specified block was not successfully programmed.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_BLOCK_PROGRAMMING_FAILURE "Block Programming Failure" error.
                                                                                 */
#define PHAL_ICODE_ERR_BLOCK_NOT_LOCKED             (PH_ERR_CUSTOM_BEGIN + 9U)  /**< The specified block was not successfully locked.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_BLOCK_NOT_LOCKED "Block Not Locked" error.
                                                                                 */
#define PHAL_ICODE_ERR_BLOCK_PROTECTED              (PH_ERR_CUSTOM_BEGIN + 10U) /**< The specified block is protected.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_BLOCK_PROTECTED "Block Protected" error.
                                                                                 */
#define PHAL_ICODE_ERR_GENERIC_CRYPTO_ERROR         (PH_ERR_CUSTOM_BEGIN + 11U) /**< Generic cryptographic error.
                                                                                 *   This error represents VICC's \ref PHAL_ICODE_RESP_ERR_GENERIC_CRYPTO_ERROR "Generic Crypto Error" error.
                                                                                 */

/**
 * end of group phalICode_ErrorCodes_Mapped
 * @}
 */

/**
 * end of group phalICode_Errors
 * @}
 */

/**
 * \defgroup phalICode_CommonDefs Defines
 * \brief Describes about the macro definitions supported / to be used in the interfaces.
 * @{
 */

/** \defgroup phalICode_Flags Flags
 * \brief Description about the Request Flags.
 * @{
 */

/** \defgroup phalICode_Flags_Common Common
 * \brief Flags that are common and are part of bits 1 - 4 ( i.e. 0 - 3)\. These flags will be utilized
 * for the below mentioned interfaces.
 *  - \ref phalICode_Inventory "Inventory"
 *  - \ref phalICode_InventoryRead "InventoryRead"
 *  - \ref phalICode_InventoryReadExtended "InventoryReadExtended"
 *  - \ref phalICode_FastInventoryRead "FastInventoryRead"
 *  - \ref phalICode_InventoryPageRead "InventoryPageRead"
 *  - \ref phalICode_FastInventoryPageRead "FastInventoryPageRead"
 *  - \ref phalICode_GetConfig "GetConfig"
 *  - \ref phalICode_SetConfig "SetConfig"
 * @{
 */
#define PHAL_ICODE_FLAG_TWO_SUB_CARRIERS                                0x01U   /**< Sub-Carrier Flag. If set two sub-carriers shall be used (VICC to VCD). Else a single sub-carrier
                                                                                 *   frequency shall be used by the VICC.
                                                                                 */
#define PHAL_ICODE_FLAG_DATA_RATE                                       0x02U   /**< DataRate Flag. If set high data rate shall be used (VICC to VCD). Else low data rate shall
                                                                                 *   be used.
                                                                                 */
#define PHAL_ICODE_FLAG_INVENTORY                                       0x04U   /**< Inventory Flag. Shall be set for inventory request. Refer below sub flags in case of Inventory is set or not set.
                                                                                 *      - \ref phalICode_Flags_INV_Set "Flags if Inventory is Set."
                                                                                 *      - \ref phalICode_Flags_INV_NotSet "Flags if Inventory is Not Set."
                                                                                 */
#define PHAL_ICODE_FLAG_PROTOCOL_EXTENSION                              0x08U   /**< ProtocolExtension Flag. If set protocol format is extended. Reserved for future use. No supported
                                                                                 *   in current version.
                                                                                 */
#define PHAL_ICODE_FLAG_OPTION                                          0x40U   /**< Option Flag. Meaning is defined by the command description. It shall be set to 0 if not otherwise defined by
                                                                                 *   the command.
                                                                                 */
#define PHAL_ICODE_FLAG_FAST_DATA_RATE                                  0x80U   /**< RFU as per ISO15693 but this flag should be utilized for NTAG5 product only. This flag is used to
                                                                                 *   set the hardware RX DataRate to Fast_High (~53 kbps) for Fast_Low (~13 kbps). Also to note that this flag
                                                                                 *   should not be used for any Inventory commands. Currently the support is for
                                                                                 *      - \ref phalICode_FastReadMultipleBlocks "FastReadMultipleBlocks"
                                                                                 *      - \ref phalICode_ExtendedFastReadMultipleBlocks "ExtendedFastReadMultipleBlocks"
                                                                                 */

/**
 * end of group phalICode_Flags_Common
 * @}
 */

/** \defgroup phalICode_Flags_INV_NotSet Inventory_NotSet
 * \brief Flags when Inventory is NOT SET. These are part for bits 5 - 8 ( i.e\. 4 - 7)\. These flags will be utilized
 * for the below mentioned interfaces.
 *  - \ref phalICode_Inventory "Inventory"
 *  - \ref phalICode_InventoryRead "InventoryRead"
 *  - \ref phalICode_InventoryReadExtended "InventoryReadExtended"
 *  - \ref phalICode_FastInventoryRead "FastInventoryRead"
 *  - \ref phalICode_InventoryPageRead "InventoryPageRead"
 *  - \ref phalICode_FastInventoryPageRead "FastInventoryPageRead"
 *  - \ref phalICode_GetConfig "GetConfig"
 *  - \ref phalICode_SetConfig "SetConfig"
 * @{
 */
#define PHAL_ICODE_FLAG_SELECTED                                        0x10U   /**< Select Flag. If set only VICC in selected state shall respond. The \ref PHAL_ICODE_FLAG_ADDRESSED "Address Flag"
                                                                                 *   shall be set to 0 and the UID field will not be included in the request. If not set Request shall be executed
                                                                                 *   by any VICC according to the setting of \ref PHAL_ICODE_FLAG_ADDRESSED "Address Flag".
                                                                                 */
#define PHAL_ICODE_FLAG_ADDRESSED                                       0x20U   /**< Address Flag. If set request is addressed. UID field is included in request. It shall be executed only by the
                                                                                 *   VICC whose UID matches the UID specified in the request. If not set request is not addressed. UID field is not
                                                                                 *   included in request. It shall be executed by any VICC.
                                                                                 */
/**
 * end of group phalICode_Flags_INV_NotSet
 * @}
 */

/** \defgroup phalICode_Flags_INV_Set Inventory_Set
 * \brief Flags when Inventory is SET. These are part for bits 5 - 8 ( i.e\. 4 - 7)\. These flags will be utilized
 * for the below mentioned interfaces.
 *  - \ref phalICode_Inventory "Inventory"
 *  - \ref phalICode_InventoryRead "InventoryRead"
 *  - \ref phalICode_InventoryReadExtended "InventoryReadExtended"
 *  - \ref phalICode_FastInventoryRead "FastInventoryRead"
 *  - \ref phalICode_InventoryPageRead "InventoryPageRead"
 *  - \ref phalICode_FastInventoryPageRead "FastInventoryPageRead"
 *  - \ref phalICode_GetConfig "GetConfig"
 *  - \ref phalICode_SetConfig "SetConfig"
 * @{
 */
#define PHAL_ICODE_FLAG_AFI                                             0x10U   /**< AFI Flag. If set AFI field is present in request. If not set AFI field is not present in request. */
#define PHAL_ICODE_FLAG_NBSLOTS                                         0x20U   /**< NbSlots Flag. If set Use 1 slot instead of 16 slots for inventory. If not set use 16 slots. */
/**
 * end of group phalICode_Flags_INV_Set
 * @}
 */

/**
 * end of group phalICode_Flags
 * @}
 */

/** \defgroup phalICode_Option Options
 * \brief Description for Option definitions that will be used in \ref phalICode_Mandatory "Mandatory",
 * \ref phalICode_Optional "Optional" and \ref phalICode_Custom "Custom" set of commands. When these definitions are
 * used it will update the Flags 7th bit (#PHAL_ICODE_FLAG_OPTION)
 * @{
 */
#define PHAL_ICODE_OPTION_OFF                           PH_OFF                  /**< Option Bit is cleared. */
#define PHAL_ICODE_OPTION_ON                            PH_ON                   /**< Option Bit is set. */
#define PHAL_ICODE_OPTION_DEFAULT                       PHAL_ICODE_OPTION_OFF   /**< Default Option. This flag represents #PHAL_ICODE_OPTION_OFF value. */
/**
 * end of group phalICode_Option
 * @}
 */

/**
 * end of group phalICode_CommonDefs
 * @}
 */





/* iCode Mandatory commands ---------------------------------------------------------------------------------------- */
/**
 * \defgroup phalICode_Mandatory Commands_Mandatory
 * \brief Describes the Mandatory commands of ICode products.
 * @{
 */

/**
 * \brief Performs a Inventory command. This command performs the ISO15693 Anti-Collision sequence and detects one ISO15693 complaint
 * VICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_Inventory(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phalICode_Flags_Common "Common"
                                                                                 *          - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t * pDsfid,                                                       /**< [Out] 1 byte Data Storage Format Identifier. */
        uint8_t * pUid                                                          /**< [Out] 8 bytes of Unique identifier of the tag. */
    );

/**
 * \brief Performs a StayQuiet command. When receiving the Stay quiet command, the VICC shall enter the quiet state and shall
 * NOT send back a response. There is NO response to the Stay quiet command.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_StayQuiet(
        void * pDataParams                                                      /**< [In] Pointer to this layers parameter structure. */
    );

/**
 * end of group phalICode_Mandatory
 * @}
 */





/* iCode Optional commands ----------------------------------------------------------------------------------------- */
/**
 * \defgroup phalICode_Optional Commands_Optional
 * \brief Describes the Optional commands of ICode products.
 * @{
 */

/** \defgroup phalICode_Optional_Defines Defines
 * \brief Description of options to be used for Optional command sets.
 * @{
 */

/** \defgroup phalICode_Optional_Defines_TAM_AuthType TAM_AuthenticateType
 * \brief Options to be used for \ref phalICode_AuthenticateTAM "AuthenticateTAM" command.
 * @{
 */
#define PHAL_ICODE_AUTHENTICATE_TAM_1                                   0x00U   /**< Option to indicate Authentication type as TAM1.
                                                                                 *   The command framing will be according to ISO/IEC 29167 specification
                                                                                 */
#define PHAL_ICODE_AUTHENTICATE_TAM_2                                   0x04U   /**< Option to indicate Authentication type as TAM2.
                                                                                 *   The command framing will be according to ISO/IEC 29167 specification
                                                                                 */

/**
 * end of group phalICode_Optional_Defines_TAM_AuthType
 * @}
 */

/** \defgroup phalICode_Optional_Defines_Diversification Diversification
 * \brief Options to be used for \ref phalICode_AuthenticateTAM "AuthenticateTAM" command.
 * @{
 */
#define PHAL_ICODE_DIVERSIFICATION_DISABLED                             0x0000U /**< Option to indicate diversification is disabled. */
#define PHAL_ICODE_DIVERSIFICATION_ENABLED                              0x0010U /**< Option to indicate diversification is enabled. */
 /**
 * end of group phalICode_Optional_Defines_Challenge
 * @}
 */

/** \defgroup phalICode_Optional_Defines_Challenge Challenge
 * \brief Options to be used for below mentioned commands,
 *  - \ref phalICode_AuthenticateTAM "AuthenticateTAM"
 *  - \ref phalICode_Challenge "Challenge"
 * @{
 */
#define PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL                          0x0000U /**< Option to generate the random number internally. */
#define PHAL_ICODE_GENERATE_CHALLENGE_EXTERNAL                          0x0100U /**< Option to use the random number provided by user. */
/**
 * end of group phalICode_Optional_Defines_Challenge
 * @}
 */

 /** \defgroup phalICode_Optional_Defines_AUTH_MAM2 AuthenticateMAM
 * \brief Options to be used for \ref phalICode_AuthenticateMAM "AuthenticateMAM" command.
 * @{
 */
#define PHAL_ICODE_PURPOSE_MAM2_AUTHENTICATE                            0x00U   /**< Option to indicate PurposeMAM2 value as Standard usage, for Authentication. */
#define PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY_HF_RESET                0x08U   /**< Option to indicate PurposeMAM2 value as Privacy disable until HF reset. */
#define PHAL_ICODE_PURPOSE_MAM2_ENABLE_PRIVACY                          0x09U   /**< Option to indicate PurposeMAM2 value as Privacy enable. */
#define PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY                         0x0AU   /**< Option to indicate PurposeMAM2 value as Privacy disable. */
#define PHAL_ICODE_PURPOSE_MAM2_DESTROY                                 0x0BU   /**< Option to indicate PurposeMAM2 value as Destroy. */
/**
 * end of group phalICode_Optional_Defines_AUTH_MAM2
 * @}
 */

/** \defgroup phalICode_Optional_Defines_TAM_ChallengeType TAM_ChallengeType
 * \brief Options to be used for \ref phalICode_Challenge "Challenge" command.
 * @{
 */
#define PHAL_ICODE_CHALLENGE_TAM_1                                   0x00U   /**< Option to indicate Challenge type as TAM1.
                                                                              *   The command framing will be according to ISO/IEC 29167 specification
                                                                              */

 /**
 * end of group phalICode_Optional_Defines_TAM_ChallengeType
 * @}
 */

/** \defgroup phalICode_Optional_Defines_ReadBuffer ReadBuffer
 * \brief Options to be used for \ref phalICode_ReadBuffer "ReadBuffer" command.
 * @{
 */
#define PHAL_ICODE_READ_BUFFER_VERIFY_OFF                               PH_OFF  /**< Option to return the received response from VICC without verification and decryption. */
#define PHAL_ICODE_READ_BUFFER_VERIFY_ON                                PH_ON   /**< Option to return the received response from VICC with verification and decryption. */
/**
 * end of group phalICode_Optional_Defines_ReadBuffer
 * @}
 */

/** \addtogroup phalICode_Optional_Defines_InfoParams InfoParams
 * \brief Options to be used for \ref phalICode_ExtendedGetSystemInformation "ExtendedGetSystemInformation" command.
 * @{
 */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_DEFAULT                          0x10U   /**< Option to indicate the default value of VICC response with memory addressing. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_DSFID                            0x01U   /**< Option to indicate the return of DSFID information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_AFI                              0x02U   /**< Option to indicate the return of AFI information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_VICC_MEM_SIZE                    0x04U   /**< Option to indicate the return of VICC memory size information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_IC_REFERENCE                     0x08U   /**< Option to indicate the return of IC Reference information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_MOI                              0x10U   /**< Option to indicate the return of MOI (memory addressing) information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_COMMAND_LIST                     0x20U   /**< Option to indicate the return of Supported Command List information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_CSI_INFORMATION                  0x40U   /**< Option to indicate the return of CSI information from VICC. */
#define PHAL_ICODE_INFO_PARAMS_REQUEST_EXT_GET_SYS_INFO                 0x80U   /**< Option to indicate the return of Two byte length of Ext Get System information from VICC. */
/**
 * end of group phalICode_Defines_Optional
 * @}
 */

/**
 * end of group phalICode_Optional_Defines
 * @}
 */

/**
 * \brief Performs a Single block read command. When receiving the Read Single Block command, the VICC shall read the requested block and send
 * back its value in the response.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadSingleBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data to be read. */
        uint8_t ** ppData,                                                      /**< [Out] Information received from VICC
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, block Security Status information is not available. Only block data
                                                                                 *            is available. Format will be 4 byte data
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, both block Security Status information and Block Data is available.
                                                                                 *            Format of the response will be 1 byte Status + 4 byte data.
                                                                                 */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b ppData buffer. */
    );

/**
 * \brief Performs a Single block write command. When receiving the Write single block command, the VICC shall write the requested block with the
 * data contained in the request and report the success of the operation in the response.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_WriteSingleBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the write operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
        uint8_t bBlockNo,                                                       /**< [In] Block number to which the data should be written. */
        uint8_t * pData,                                                        /**< [In] Information to be written to the specified block number. */
        uint8_t bDataLen                                                        /**< [In] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Performs a Lock block command. When receiving the Lock block command, the VICC shall lock permanently the requested block.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_LockBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the lock operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
        uint8_t bBlockNo                                                        /**< [In] Block number which should be locked. */
    );

/**
 * \brief Performs a Multiple block read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s) and send
 * back its value in the response.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 * Note: The memory should be created and sent to the \b pData buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadMultipleBlocks(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data to be read. */
        uint8_t bNumBlocks,                                                     /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
                                                                                 *            is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
                                                                                 *            Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
                                                                                 *            Status N + 4 byte data N, Where 1, 2 ... N is the block number.
                                                                                 */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Performs a Select command.
 * When receiving the Select command,
 *  - If the UID is equal to its own UID, the VICC shall enter the selected state and shall send a response.
 *  - If it is different, the VICC shall return to the Ready state and shall not send a response.The Select command
 *    shall always be executed in Addressed mode. (The Select_flag is set to 0. The Address_flag is set to 1.)
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_Select(
        void * pDataParams                                                      /**< [In] Pointer to this layers parameter structure. */
    );

/**
 * \brief Performs a ResetToReady command. When receiving a Reset to ready command, the VICC shall return to the Ready state.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ResetToReady(
        void * pDataParams                                                      /**< [In] Pointer to this layers parameter structure. */
    );

/**
 * \brief Performs a WriteAFI command. When receiving the Write AFI request, the VICC shall write the AFI value into its memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_WriteAFI(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the write operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
        uint8_t bAfi                                                            /**< [In] Value of Application Family Identifier. */
    );

/**
 * \brief Performs a LockAFI command. When receiving the Lock AFI request, the VICC shall lock the AFI value permanently into its memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_LockAFI(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the lock operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
    );

/**
 * \brief Performs WriteDSFID command. When receiving the Write DSFID request, the VICC shall write the DSFID value into its memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_WriteDSFID(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the write operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
        uint8_t bDsfid                                                          /**< [In] Value of DSFID (data storage format identifier). */
    );

/**
 * \brief Performs LockDSFID command. When receiving the Lock DSFID request, the VICC shall lock the DSFID value permanently into its memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_LockDSFID(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the lock operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
    );

/**
 * \brief Performs GetSystemInformation command. This command allows for retrieving the system information value from the VICC.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.\n
 * Response will contain the below information.
 *          | **Bit** |   **Flag Name**  | **Value** |                                     **Description**                                     |
 *          |:-------:|:-----------------|:---------:|:----------------------------------------------------------------------------------------|
 *          |    b1   | DSFID            |     0     | DSFID is not supported. DSFID field is not present                                      |
 *          |    ^    | ^                |     1     | DSFID is supported. DSFID field is present                                              |
 *          |    b2   | AFI              |     0     | AFI is not supported. AFI field is not present                                          |
 *          |    ^    | ^                |     1     | AFI is supported. AFI field is present                                                  |
 *          |    b3   | VICC Memory Size |     0     | Information on VICC memory size is not supported. <br>Memory size field is not present. |
 *          |    ^    | ^                |     1     | Information on VICC memory size is supported. Memory<br>size field is present.          |
 *          |    b4   | IC Reference     |     0     | Information on IC reference is not supported. IC<br>reference field is not present.     |
 *          |    ^    | ^                |     1     | Information on IC reference is supported. IC reference<br>field is present.             |
 *          |    b5   | RFU              |     0     |                                                                                         |
 *          |    b6   | RFU              |     0     |                                                                                         |
 *          |    b7   | RFU              |     0     |                                                                                         |
 *          |    b8   | RFU              |     0     |                                                                                         |
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_GetSystemInformation(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppSystemInfo,                                                /**< [Out] The system information of the VICC. Refer description for more information. */
        uint16_t * pSystemInfoLen                                               /**< [Out] Length of bytes available in \b ppSystemInfo buffer. */
    );

/**
 * \brief Performs GetMultipleBlockSecurityStatus. When receiving the Get multiple block security status command, the VICC
 * shall send back the block security status.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 * Note: The memory should be created and sent to the Status buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_GetMultipleBlockSecurityStatus(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bBlockNo,                                                       /**< [In] Block number for which the status should be returned. */
        uint8_t bNoOfBlocks,                                                    /**< [In] Number of blocks to be used for returning the status. */
        uint8_t * pStatus,                                                      /**< [Out] The status of the block number mentioned in bBlockNo until bNumBlocks. */
        uint16_t * pStatusLen                                                   /**< [Out] Length of bytes available in \b pStatus buffer. */
    );

/**
 * \brief Performs a Multiple block fast read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s)
 * and send back its value in the response.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 * Note: The memory should be created and sent to the Data buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_FastReadMultipleBlocks(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data to be read. */
        uint8_t bNumBlocks,                                                     /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
                                                                                 *            is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
                                                                                 *            Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
                                                                                 *            Status N + 4 byte data N, Where 1, 2 ... N is the block number.
                                                                                 */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes. */
    );

/**
 * \brief Performs a Extended Single block read command. When receiving the Extended Read Single Block command, the VICC shall read the
 * requested block and send back its value in the response. If a VICC supports Extended read single block command, it shall also support
 * Read single block command for the first 256 blocks of memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ExtendedReadSingleBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint16_t wBlockNo,                                                      /**< [In] Block number from where the data to be read. */
        uint8_t ** ppData,                                                      /**< [Out] Information received from VICC
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, block Security Status information is not available. Only block data
                                                                                 *            is available. Format will be 4 byte data.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, both block Security Status information and Block Data is available.
                                                                                 *            Format of the response will be 1 byte Status + 4 byte data.
                                                                                 */
        uint16_t * pDataLen                                                     /**< [Out] Number of received data bytes. */
    );

/**
 * \brief Performs a Extended Single block Write command. When receiving the Extended write single block command, the VICC shall write the
 * requested block with the data contained in the request and report the success of the operation in the response. If a VICC supports
 * Extended write single block command, it shall also support Write single block command for the first 256 blocks of memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ExtendedWriteSingleBlock(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the write operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
        uint16_t wBlockNo,                                                      /**< [In] Block number to which the data should be written. */
        uint8_t * pData,                                                        /**< [In] Information to be written to VICC. */
        uint8_t bDataLen                                                        /**< [In] Number of bytes to be written. */
    );

/**
 * \brief Performs a Extended Lock block command. When receiving the Lock block command, the VICC shall lock permanently the requested
 * block. If a VICC supports Extended lock block command, it shall also support Lock block command for the first 256 blocks of memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ExtendedLockBlock (
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the lock operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
        uint16_t wBlockNo                                                       /**< [In] Block number which should be locked. */
    );

/**
 * \brief Performs a Extended Multiple block read command. When receiving the Read Multiple Block command, the VICC shall read the requested block(s)
 * and send back its value in the response. If a VICC supports Extended read multiple blocks command, it shall also support Read multiple blocks
 * command for the first 256 blocks of memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 * Note: The memory should be created and sent to the Data buffer..
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ExtendedReadMultipleBlocks(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint16_t wBlockNo,                                                      /**< [In] Block number from where the data to be read. */
        uint16_t wNumBlocks,                                                    /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
                                                                                 *            is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
                                                                                 *            Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
                                                                                 *            Status N + 4 byte data N, Where 1, 2 ... N is the block number.
                                                                                 */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

#ifdef NXPBUILD__PH_CRYPTOSYM
/**
 * \brief Performs TAM authentication with the tag. Support various authentication schemes of TAM authentication.
 * Refer description of \b bAuthType parameter for more information.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If \b wOption parameter is invalid.
 *          - If \b bAuthType parameter is invalid.
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 *          - If \b AuthType = \ref PHAL_ICODE_AUTHENTICATE_TAM_2 "TAM2 authentication as per ISO 2916 specification" and
 *            \b pAddInfo buffer is not holding 6 bytes of information.
 * \retval #PH_ERR_AUTH_ERROR           Difference in Random Challenge.
 * \retval #PH_ERR_PROTOCOL_ERROR
 *          - If KeyType is not AES128.
 *          - For invalid response format.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_AuthenticateTAM(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                       /**< [In] One of the below mentioned options.
                                                                                 *          - Option flag
                                                                                 *              - \ref PHAL_ICODE_OPTION_OFF "Option Flag is not set"
                                                                                 *              - \ref PHAL_ICODE_OPTION_ON "Option Flag is Set"
                                                                                 *
                                                                                 *          - Diversification option
                                                                                 *              - \ref PHAL_ICODE_DIVERSIFICATION_DISABLED "Key Diversification disabled"
                                                                                 *              - \ref PHAL_ICODE_DIVERSIFICATION_ENABLED "Key Diversification enabled"
                                                                                 *
                                                                                 *          - Challenge usage
                                                                                 *              - \ref PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL "Generate Challenge internally and exchange"
                                                                                 *              - \ref PHAL_ICODE_GENERATE_CHALLENGE_EXTERNAL "Exchange the Challenge provided by user"
                                                                                 */
        uint8_t bAuthType,                                                      /**< [In] Type of Authenticate. One of the below mentioned values,
                                                                                 *          - \ref PHAL_ICODE_AUTHENTICATE_TAM_1 "TAM1 authentication as per ISO 2916 specification"
                                                                                 *          - \ref PHAL_ICODE_AUTHENTICATE_TAM_2 "TAM2 authentication as per ISO 2916 specification"
                                                                                 *
                                                                                 */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store or SAM hardware KeyStore. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t bKeyID_CertNum,                                                 /**< [In] KeyID or Certificate number available in the tag.*/
        uint8_t * pDivInput,                                                    /**< [In] Diversification Input used to diversify the key. The diversification input is available in SAM mode only. */
        uint8_t bDivLen,                                                        /**< [In] Length of byte available in \b pDivInput buffer. If 0, no diversification is performed. */
        uint8_t * pChallenge,                                                   /**< [In] Challenge information to be exchanged with tag. Will be one of the following.
                                                                                 *          - If \b wOption has \ref PHAL_ICODE_GENERATE_CHALLENGE_EXTERNAL "Exchange the Challenge provided by user"
                                                                                 *              - \b bAuthType = \ref PHAL_ICODE_AUTHENTICATE_TAM_1 "TAM1 authentication as per ISO 2916 specification":
                                                                                 *                80 bit challenge (IChallenge_TAM1)
                                                                                 *
                                                                                 *
                                                                                 *              - \b bAuthType = \ref PHAL_ICODE_AUTHENTICATE_TAM_2 "TAM2 authentication as per ISO 2916 specification"
                                                                                 *                80 bit challenge (IChallenge_TAM2)
                                                                                 *
                                                                                 *          - If \b wOption has \ref PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL "Generate Challenge internally and exchange",
                                                                                 *            then, NULL should be provided.
                                                                                 */
        uint16_t wChallengeLen,                                                 /**< [In] Length of bytes available in \b pChallenge buffer. */
        uint8_t * pAddInfo,                                                     /**< [In] The information to be exchanged based on \b bAuthType parameter data.
                                                                                 *          - \b bAuthType = \ref PHAL_ICODE_AUTHENTICATE_TAM_1 "TAM1 authentication as per ISO 2916 specification":
                                                                                 *            Will be NULL as there is no additional information to be exchanged.
                                                                                 *
                                                                                 *
                                                                                 *          - \b bAuthType = \ref PHAL_ICODE_AUTHENTICATE_TAM_2 "TAM2 authentication as per ISO 2916 specification"
                                                                                 *            Will be BlockSize, Profile, Offset, BlockCount and ProtMode
                                                                                 *              - BlockSize: To select the size of custom data block to be used. The value should either be
                                                                                 *                0x00 for 16 bit block size or 0x01 for 64 bit block size, as per ISO 29167
                                                                                 *
                                                                                 *              - Profile: To select one of the memory profiles supported by the tag.
                                                                                 *                The Profile range is from 0 - 15, as per ISO 29167
                                                                                 *
                                                                                 *              - Offset: To set the offset for the specified profile. The Offset ranges form 0 - 4095, as per ISO 29167
                                                                                 *
                                                                                 *              - BlockCount: To select the custom data block to be used from the offset specified.
                                                                                 *                The BlockCount range is from 1 - 16.
                                                                                 *
                                                                                 *              - ProtMode: To specify the mode of operation to be used for encryption/decryption.
                                                                                 *                The ProtMode ranges form 0 - 3, as per ISO 29167
                                                                                 */
        uint16_t wAddInfoLen,                                                   /**< [In] Length of bytes available in \b pAddInfo buffer. */
        uint8_t ** ppResponse,                                                  /**< [Out] Response received from the Tag. Will be based on \b bAuthType parameter information. */
        uint16_t * pRespLen                                                     /**< [Out] Length of available bytes in \b ppResponse buffer. */
    );

/**
 * \deprecated
 * Refer \ref phalICode_AuthenticateTAM "TAM Authentication"
 *
 * \brief Performs TAM1 authentication with the tag.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_AUTH_ERROR           Difference in Random Challenge.
 * \retval #PH_ERR_PROTOCOL_ERROR
 *          - If KeyType is not AES128.
 *          - For invalid response format.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_AuthenticateTAM1(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store or SAM hardware KeyStore. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t bKeyNoCard,                                                     /**< [In] Block number of the AES key available in the tag.*/
        uint8_t * pDivInput,                                                    /**< [In] Diversification Input used to diversify the key. The diversification input is available in SAM mode only. */
        uint8_t bDivLen                                                         /**< [In] Length of byte available in \b pDivInput buffer. If 0, no diversification is performed. */
    );

/**
 * \deprecated
 * Refer \ref phalICode_AuthenticateTAM "TAM Authentication"
 *
 * \brief Performs TAM-2 authentication with the tag.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_AuthenticateTAM2(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store or SAM hardware KeyStore. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t bKeyNoCard,                                                     /**< [In] Block number of the AES key available in the tag.*/
        uint8_t * pDivInput,                                                    /**< [In] Diversification Input used to diversify the key. The diversification input is available in SAM mode only. */
        uint8_t bDivLen,                                                        /**< [In] Length of byte available in \b pDivInput buffer. If 0, no diversification is performed. */
        uint8_t bBlockSize,                                                     /**< [In] To select the size of custom data block to be used.
                                                                                          The value should either be 0x00 for 16 bit block size or 0x01 for 64 bit block size, as per ISO 29167 */
        uint8_t bBlockCount,                                                    /**< [In] To select the custom data block to be used from the offset specified.
                                                                                          The BlockCount range is from 1 - 16. */
        uint8_t bProfile,                                                       /**< [In] To select one of the memory profiles supported by the tag.
                                                                                          The Profile range is from 0 - 15, as per ISO 29167  */
        uint8_t bProtMode,                                                      /**< [In] To specify the mode of operation to be used for encryption/decryption.
                                                                                          The ProtMode ranges form 0 - 3, as per ISO 29167 */
        uint16_t wOffset,                                                       /**< [In] To set the offset for the specified profile. The Offset ranges form 0 - 4095, as per ISO 29167 */
        uint8_t * pCustomData,                                                  /**< [Out] The custom data returned by the tag. */
        uint16_t * pCustomDataLen                                               /**< [Out] Length of byte available in \b pCustomData buffer */
    );

/**
 * \brief Performs MAM authentication with the tag. Both the MAM part 1 and MAM part 2 authentication are
 * carried out internally by this interface.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_AUTH_ERROR           Difference in Random Challenge.
 * \retval #PH_ERR_PROTOCOL_ERROR
 *          - If KeyType is not AES128.
 *          - For invalid response format.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_AuthenticateMAM(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store or SAM hardware KeyStore. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t bKeyNoCard,                                                     /**< [In] Block number of the AES key available in the tag.*/
        uint8_t bPurposeMAM2,                                                   /**< [In] The purpose MAM2 data to be used.
                                                                                 *          - \ref PHAL_ICODE_PURPOSE_MAM2_AUTHENTICATE "Standard, For Authentication"
                                                                                 *          - \ref PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY_HF_RESET "Disable Privacy until HF Reset"
                                                                                 *          - \ref PHAL_ICODE_PURPOSE_MAM2_ENABLE_PRIVACY "Enable Privacy"
                                                                                 *          - \ref PHAL_ICODE_PURPOSE_MAM2_DISABLE_PRIVACY "Disable Privacy"
                                                                                 *          - \ref PHAL_ICODE_PURPOSE_MAM2_DESTROY "Destroy"
                                                                                 */
        uint8_t * pDivInput,                                                    /**< [In] Diversification Input used to diversify the key. The diversification input is available in SAM mode only. */
        uint8_t bDivLen                                                         /**< [In] Length of byte available in \b pDivInput buffer. If 0, no diversification is performed. */
    );


/**
 * \brief Performs tag authentication with the tag. This is another method of authenticating with the tag.
 * Here the challenge message is sent to the tag. The tag does not respond for this command. To verify
 * if this command was success the command \ref phalICode_ReadBuffer "ReadBuffer" should be called.
 *
 * - Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 * - Timeout can be set using \ref phalICode_SetConfig "SetConfig" utility interface with below mentioned
 *   configuration identifiers.
 *      - \ref PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US "Timeout in Microsecond" or
 *      - \ref PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS "Timeout in Millisecond"
 * - By Default Timeout \ref PHAL_ICODE_TIMEOUT_LONG_US "Long Timeout" is configured.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_PROTOCOL_ERROR       If there is a status other than IO_TIMEOUT.
 * \retval #PH_ERR_INVALID_PARAMETER    If \b bChallengeType parameter is invalid.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_Challenge(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                       /**< [In] One of the below mentioned options.
                                                                                 *              - \ref PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL "Generate Challenge internally and exchange"
                                                                                 *              - \ref PHAL_ICODE_GENERATE_CHALLENGE_EXTERNAL "Exchange the Challenge provided by user"
                                                                                 */
        uint8_t bChallengeType,                                                 /**< [In] Type of Challenge command to exchange. One of the below mentioned values,
                                                                                 *          - \ref PHAL_ICODE_CHALLENGE_TAM_1 "TAM1 Challenge communication as per ISO 2916 specification"
                                                                                 *
                                                                                 */
        uint8_t bKeyID_CertNum,                                                 /**< [In] KeyID or Certificate number available in the tag.*/
        uint8_t * pChallenge,                                                   /**< [In] Challenge information to be exchanged with tag. Will be one of the following.
                                                                                 *          - If \b wOption has \ref PHAL_ICODE_GENERATE_CHALLENGE_EXTERNAL "Exchange the Challenge provided by user"
                                                                                 *              - \b bChallengeType =\ref PHAL_ICODE_CHALLENGE_TAM_1 "TAM1 Challenge communication as per ISO 2916 specification":
                                                                                 *                80 bit challenge (IChallenge_TAM1)
                                                                                 *
                                                                                 *
                                                                                 *          - If \b wOption has \ref PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL "Generate Challenge internally and exchange",
                                                                                 *            then, NULL should be provided.
                                                                                 */
        uint16_t wChallengeLen                                                  /**< [In] Length of bytes available in \b pChallenge buffer. */
    );

/**
 * \brief Reads the crypto calculation result of previous Challenge command. If the Challenge Command was success,
 * Then the encrypted response will be returned. The response will be same as TAM1 response format. If verification
 * is enabled (i.e. bVerify = #PHAL_ICODE_READ_BUFFER_VERIFY_ON), The encrypted response will be decrypted and the
 * random number generated by the Challenge command will be compared against the received one. If fails
 * \ref PH_ERR_AUTH_ERROR "Authentication Error" will be returned.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_AUTH_ERROR           Difference in Random Challenge.
 * \retval #PH_ERR_PROTOCOL_ERROR
 *          - If KeyType is not AES128.
 *          - For invalid response format.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the verify values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadBuffer(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bVerify,                                                        /**< [In] To verify the received data with the random number generated by Challenge command.
                                                                                 *        - #PHAL_ICODE_READ_BUFFER_VERIFY_OFF
                                                                                 *        - #PHAL_ICODE_READ_BUFFER_VERIFY_ON : Applicable only if previous \ref phalICode_Challenge
                                                                                 *          "Challenge" command was performed for
                                                                                 *          \ref PHAL_ICODE_CHALLENGE_TAM_1 "TAM1 Challenge communication as per ISO 2916 specification"
                                                                                 */
        uint8_t bKeyNo,                                                         /**< [In] AES key address in software key store. */
        uint8_t bKeyVer,                                                        /**< [In] AES key version to be used. */
        uint8_t ** ppResponse,                                                  /**< [Out] Below are the outputs,
                                                                                 *          - If verification is enabled the decrypted response data will be available. Also
                                                                                 *            the response will be verified with the random number generated by
                                                                                 *            \ref phalICode_Challenge "Challenge" command.
                                                                                 *          - If verification is disabled the encrypted response data will be available.
                                                                                 */
        uint16_t * pRespLen                                                     /**< [Out] Length of available bytes in ppResponse buffer. */
    );
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * \brief Performs ExtendedGetSystemInformation command. This command allows for retrieving the system information value
 * from the VICC and shall be supported by the VICC if extended memory or security functionalities are supported by the VICC.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.\n
 * Response will contain the below information.
 *          | **Bit** |   **Flag Name**   | **Value** |                                     **Description**                                     |
 *          |:-------:|:------------------|:---------:|:----------------------------------------------------------------------------------------|
 *          |    b1   | DSFID             |     0     | DSFID is not supported. DSFID field is not present                                      |
 *          |    ^    | ^                 |     1     | DSFID is supported. DSFID field is present                                              |
 *          |    b2   | AFI               |     0     | AFI is not supported. AFI field is not present                                          |
 *          |    ^    | ^                 |     1     | AFI is supported. AFI field is present                                                  |
 *          |    b3   | VICC Memory Size  |     0     | Information on VICC memory size is not supported. Memory size field is not present.     |
 *          |    ^    | ^                 |     1     | Information on VICC memory size is supported. Memory size field is present.             |
 *          |    b4   | IC Reference      |     0     | Information on IC reference is not supported. IC reference field is not present.        |
 *          |    ^    | ^                 |     1     | Information on IC reference is supported. IC reference field is present.                |
 *          |    b5   | MOI               |     0     | 1 byte memory addressing                                                                |
 *          |    ^    | ^                 |     1     | 2 bytes memory addressing                                                               |
 *          |    b6   | VICC Command List |     0     | Data field of all supported commands is not present                                     |
 *          |    ^    | ^                 |     1     | Data field of all supported commands is present                                         |
 *          |    b7   | CSI Information   |     0     | CSI list is not present                                                                 |
 *          |    ^    | ^                 |     1     | CSI list is present                                                                     |
 *          |    b8   | Info Flag Field   |     0     | One byte length of Info flag field                                                      |
 *          |    ^    | ^                 |     1     | Two bytes length of Info flag field                                                     |
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ExtendedGetSystemInformation(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bInfoParams,                                                    /**< [In] Extend Get System Information parameter request fields.
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_DEFAULT
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_DSFID
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_AFI
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_VICC_MEM_SIZE
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_IC_REFERENCE
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_MOI
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_COMMAND_LIST
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_CSI_INFORMATION
                                                                                 *          - #PHAL_ICODE_INFO_PARAMS_REQUEST_EXT_GET_SYS_INFO
                                                                                 */
        uint8_t ** ppSystemInfo,                                                /**< [Out] The system information of the VICC. Refer Description for more information. */
        uint16_t * pSystemInfoLen                                               /**< [Out] Length of bytes available in \b ppSystemInfo buffer. */
    );

/**
 * \brief Performs ExtendedGetMultipleBlockSecurityStatus. When receiving the Extended Get multiple block security status
 * command, the VICC shall send back the block security status. The blocks are numbered from 0000 to FFFF (0 - 65535).
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 * Note: The memory should be created and sent to the Status buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - If \b wNoOfBlocks is zero.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ExtendedGetMultipleBlockSecurityStatus(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wBlockNo,                                                      /**< [In] Block number for which the status should be returned. */
        uint16_t wNoOfBlocks,                                                   /**< [In] Number of blocks to be used for returning the status. */
        uint8_t * pStatus,                                                      /**< [Out] The status of the block number mentioned in wBlockNo until wNumBlocks. */
        uint16_t * pStatusLen                                                   /**< [Out] Length of bytes available in \b pStatus buffer. */
    );

/**
 * \brief Performs a Extended Multiple block fast read command. When receiving the Read Multiple Block command, the VICC shall read the requested
 * block(s) and send back its value in the response. If a VICC supports Extended read multiple blocks command, it shall also support Read multiple
 * blocks command for the first 256 blocks of memory.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 * Note: The memory should be created and sent to the Data buffer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - If \b wNoOfBlocks is zero.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ExtendedFastReadMultipleBlocks(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint16_t wBlockNo,                                                      /**< [In] Block number from where the data to be read. */
        uint16_t wNumBlocks,                                                    /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
                                                                                 *            is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
                                                                                 *            Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
                                                                                 *            Status N + 4 byte data N, Where 1, 2 ... N is the block number.
                                                                                 */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * end of group phalICode_Optional
 * @}
 */





/* iCode Custom commands ------------------------------------------------------------------------------------------- */
/**
 * \defgroup phalICode_Custom Commands_Custom
 * \brief Describes the Custom commands of ICode products.
 * @{
 */

/** \defgroup phalICode_Custom_Defines Defines
 * \brief Description of options to be used for Custom command sets.
 * @{
 */

/** \defgroup phalICode_Custom_Defines_ExtFlags ExtendedInventoryFlags
 * \brief Options to be used with \ref phalICode_InventoryReadExtended "InventoryReadExtended" command.
 * @{
 */
#define PHAL_ICODE_INVENTORY_READ_EXT_DEFAULT                           0x00U   /**< Default option for InventoryReadExtended command. */
#define PHAL_ICODE_INVENTORY_READ_EXT_EAS_MODE                          0x01U   /**< Option to indicate that the VICC with EAS enabled will respond. */
#define PHAL_ICODE_INVENTORY_READ_EXT_UID_MODE                          0x02U   /**< Option to indicate complete UID transmission. */
#define PHAL_ICODE_INVENTORY_READ_EXT_CID_COMPARE                       0x04U   /**< Option to indicate transmission of 16 bit CID and only tags with the same CID will respond. */
#define PHAL_ICODE_INVENTORY_READ_EXT_CID_RESPONSE                      0x08U   /**< Option to indicate transmission of Custom ID. */
#define PHAL_ICODE_INVENTORY_READ_EXT_SKIP_DATA                         0x10U   /**< Option to indicate No user memory data are requested from the tag, first block number byte and
                                                                                 *   number of blocks byte shall not be transmitted in the command.
                                                                                 */
#define PHAL_ICODE_INVENTORY_READ_EXT_QUIET                             0x20U   /**< Option to indicate the state change to Quiet after response. */
#define PHAL_ICODE_INVENTORY_READ_EXT_PERSIST_QUIET                     0x40U   /**< Option to indicate the state change to Persistent Quiet after response. */
#define PHAL_ICODE_INVENTORY_READ_EXT_PERSIST_QUIET_RESPONSE            0x60U   /**< Option to indicate that tags in the PERSISTENT QUIET state will respond to the command. */
/**
 * end of group phalICode_Custom_Defines_ExtFlags
 * @}
 */

/** \defgroup phalICode_Custom_Defines_Password Password
 * \brief Password option for the below interfaces.
 *  - \ref phalICode_SetPassword "Set Password"
 *  - \ref phalICode_WritePassword "Write Password"
 *  - \ref phalICode_LockPassword "Lock Password"
 * @{
 */
#define PHAL_ICODE_SET_PASSWORD_READ                                    0x01U   /**< Option to indicate Password Identifier as Read. */
#define PHAL_ICODE_SET_PASSWORD_WRITE                                   0x02U   /**< Option to indicate Password Identifier as Write. */
#define PHAL_ICODE_SET_PASSWORD_PRIVACY                                 0x04U   /**< Option to indicate Password Identifier as Privacy. */
#define PHAL_ICODE_SET_PASSWORD_DESTROY                                 0x08U   /**< Option to indicate Password Identifier as Destroy. */
#define PHAL_ICODE_SET_PASSWORD_EAS_AFI                                 0x10U   /**< Option to indicate Password Identifier as EAS / AFI. */
/**
 * end of group phalICode_Custom_Defines_Password
 * @}
 */

/** \defgroup phalICode_Custom_Defines_ProtectPage ProtectPage
 * \brief Describes the definitions for \ref phalICode_ProtectPage "ProtectPage" command.
 * @{
 */
#define PHAL_ICODE_PROTECT_PAGE_PUBLIC                                  0x00U   /**< Option to indicate protection status as public. */
#define PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD                0x01U   /**< Option to indicate protection status as
                                                                                 *      - 32-bit Password Protection: Read and Write protection by Read password.
                                                                                 *      - 64-bit Password Protection: Read and Write protection by Read plus Write password.
                                                                                 *      - ICode DNA or other products that supports AES authentication, Mutual-Authentication
                                                                                 *        with a key with read privilege is required.
                                                                                 */
#define PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD                          0x10U   /**< Option to indicate protection status as
                                                                                 *      - 32-bit Password Protection: Write protection by Write password.
                                                                                 *      - 64-bit Password Protection: Write protection by Read plus Write password.
                                                                                 *      - ICode DNA or other products that supports AES authentication, Mutual-Authentication
                                                                                 *        with a key with write privilege is required.
                                                                                 */
#define PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE            0x11U   /**< Option to indicate protection status as
                                                                                 *      - 32-bit Password Protection: Read protected by Read and Write protection by Write password.
                                                                                 *      - 64-bit Password Protection: Read and Write protection by Read plus Write password.
                                                                                 *      - ICode DNA or other products that supports AES authentication, Mutual-Authentication
                                                                                 *        with a key with read and write privileges is required.
                                                                                 */
/**
 * end of group phalICode_Custom_Defines_ProtectPage
 * @}
 */

/** \defgroup phalICode_Custom_Defines_BitRates BitRates
 * \brief BitRate Options to be used for below interfaces.
 *  - \ref phalICode_ParameterRequest "Parameter Request"
 *  - \ref phalICode_ParameterSelect "Parameter Select"
 * @{
 */
#define PHAL_ICODE_PARAMETERS_BITRATE_26KBPS_BOTH_DIRECTIONS            0x00U   /**< Option to indicate the 26kbps BitRate support in both directions. */
#define PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VCD_VICC                   0x01U   /**< Option to indicate the 53kbps BitRate support from VCD to VICC. */
#define PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VCD_VICC                  0x02U   /**< Option to indicate the 106kbps BitRate support from VCD to VICC. */
#define PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VCD_VICC                  0x04U   /**< Option to indicate the 212kbps BitRate support from VCD to VICC. */
#define PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VICC_VCD                   0x10U   /**< Option to indicate the 53kbps BitRate support from VICC to VCD. */
#define PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VICC_VCD                  0x20U   /**< Option to indicate the 106kbps BitRate support from VICC to VCD. */
#define PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VICC_VCD                  0x40U   /**< Option to indicate the 212kbps BitRate support from VICC to VCD. */
/**
 * end of group phalICode_Custom_Defines_BitRates
 * @}
 */

/** \defgroup phalICode_Custom_Defines_Timing Timing
 * \brief Timing Options to be used for below interfaces.
 *  - \ref phalICode_ParameterRequest "Parameter Request"
 *  - \ref phalICode_ParameterSelect "Parameter Select"
 * @{
 */
#define PHAL_ICODE_PARAMETERS_TIMING_320_9_US                           0x00U   /**< Option to indicate that VICC supports 320.9us timing only. */
#define PHAL_ICODE_PARAMETERS_TIMING_160_5_US                           0x01U   /**< Option to indicate that VICC supports 160.5us timing only. */
#define PHAL_ICODE_PARAMETERS_TIMING_80_2_US                            0x02U   /**< Option to indicate that VICC supports 80.2us timing only. */
#define PHAL_ICODE_PARAMETERS_TIMING_SAME_BOTH_DIRECTIONS               0x10U   /**< Option to indicate that same timing on both the direction is compulsory. */
/**
 * end of group phalICode_Custom_Defines_Timing
 * @}
 */

/**
 * end of group phalICode_Custom_Defines
 * @}
 */

/**
 * \brief Perform ISO15693 InventoryRead command. When receiving the INVENTORY READ request, the ICODE IC performs the same as
 * the Anti-Collision sequence, with the difference that instead of the UID and the DSFID, the requested response is defined by
 * additional options.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_InventoryRead(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phalICode_Flags_Common "Common"
                                                                                 *          - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bBlockNo,                                                       /**< [In] Block number of first block to read. */
        uint8_t bNoOfBlocks,                                                    /**< [In] Number of blocks to read */
        uint8_t * pUid,                                                         /**< [Out] Received UID. */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUID buffer. */
        uint8_t * pData,                                                        /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ISO15693 InventoryReadExtended command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_InventoryReadExtended(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phalICode_Flags_Common "Common"
                                                                                 *          - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bExtendedOptions,                                               /**< [In] Refer \ref phalICode_Custom_Defines_ExtFlags "Extended Inventory Flags"
                                                                                 *        for supported values.
                                                                                 */
        uint8_t * pCID,                                                         /**< [In] Two byte CID -> if marked in extended options */
        uint8_t bBlockNo,                                                       /**< [In] Block Number from where start reading */
        uint8_t bNoOfBlocks,                                                    /**< [In] Number of blocks to read */
        uint8_t * pCDIDOut,                                                     /**< [Out] Received CID. */
        uint8_t * pUid,                                                         /**< [Out] Received UID. */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUID buffer. */
        uint8_t * pData,                                                        /**< [Out] Received Data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ISO15693 FastInventoryRead command. When receiving the FAST INVENTORY READ command the ICODE IC behaves the
 * same as the INVENTORY READ command with the following exceptions:
 *
 * - The data rate in the direction ICODE Tag to the interrogator is twice that defined in ISO/IEC 15693-3 depending on
 *   the Datarate_flag 53 kbit (high data rate) or 13 kbit (low data rate).
 *
 * - The data rate from the interrogator to the ICODE Tag and the time between the rising edge of the EOF from the
 *   interrogator to the ICODE Tag remain unchanged (stay the same as defined in ISO/IEC 15693-3).
 *
 * - In the ICODE Tag to the interrogator direction, only the single sub-carrier mode is supported.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_FastInventoryRead(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phalICode_Flags_Common "Common"
                                                                                 *          - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bBlockNo,                                                       /**< [In] Block number of first block to read. */
        uint8_t bNoOfBlocks,                                                    /**< [In] Number of blocks to read */
        uint8_t * pUid,                                                         /**< [Out] Received UID */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUID buffer. */
        uint8_t * pData,                                                        /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ISO15693 FastInventoryReadExtended command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_FastInventoryReadExtended(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phalICode_Flags_Common "Common"
                                                                                 *          - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bExtendedOptions,                                               /**< [In] Refer \ref phalICode_Custom_Defines_ExtFlags "Extended Inventory Flags"
                                                                                 *        for supported values.
                                                                                 */
        uint8_t * pCID,                                                         /**< [In] Two byte CID -> if marked in extended options */
        uint8_t bBlockNo,                                                       /**< [In] Block Number from where start reading */
        uint8_t bNoOfBlocks,                                                    /**< [In] Number of blocks to read */
        uint8_t * pCDIDOut,                                                     /**< [Out] Received CID. */
        uint8_t * pUid,                                                         /**< [Out] Received UID. */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUID buffer. */
        uint8_t * pData,                                                        /**< [Out] Received Data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief This command enables the EAS mode if the EAS mode is not locked. If the EAS mode is password protected
 * the EAS password has to be transmitted before with \ref phalICode_SetPassword "Set Password".
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_SetEAS(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
    );

/**
 * \brief This command disables the EAS mode if the EAS mode is not locked. If the EAS mode is password protected
 * the EAS password has to be transmitted before with \ref phalICode_SetPassword "Set Password".
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalICode_ResetEAS(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
    );

/**
 * \brief This command locks the current state of the EAS mode and the EAS ID. If the EAS mode is password protected
 * the EAS password has to be transmitted before with \ref phalICode_SetPassword "Set Password".
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER    For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_LockEAS(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
    );

/**
 * \brief This command returns the EAS sequence if the EAS mode is enabled.
 *
 *  - \c bOption disabled: \c bEasIdMaskLength and \c pEasIdValue are not transmitted, EAS Sequence is returned;
 *  - \c bOption enabled and \c bEasIdMaskLength = \c 0: EAS ID is returned;
 *  - \c bOption enabled and \c bEasIdMaskLength > \c 0: EAS Sequence is returned by ICs with matching \c pEasIdValue;
 *
 * If the EAS mode is disabled, the label remains silent.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_EASAlarm(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag;
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *              - EAS ID mask length and EAS ID value shall not be transmitted.
                                                                                 *              - If the EAS mode is enabled, the EAS response is returned from the ICODE IC.
                                                                                 *                This configuration is compliant with the EAS command of the ICODE IC
                                                                                 *          - #PHAL_ICODE_OPTION_ON.
                                                                                 *              - Within the command the EAS ID mask length has to be transmitted to identify how
                                                                                 *                many bits of the following EAS ID value are valid (multiple of 8-bits). Only those
                                                                                 *                ICODE ICs will respond with the EAS sequence which have stored the corresponding
                                                                                 *                data in the EAS ID configuration (selective EAS) and if the EAS Mode is set.
                                                                                 *              - If the EAS ID mask length is set to 0, the ICODE IC will answer with its EAS ID
                                                                                 */
        uint8_t * pEasIdValue,                                                  /**< [In] EAS ID; 0, 8 or 16 bits; optional. */
        uint8_t bEasIdMaskLen,                                                  /**< [In] 8 bits; optional. */
        uint8_t ** ppEas,                                                       /**< [Out] EAS ID (16 bits) or EAS Sequence (256 bits). */
        uint16_t * pEasLen                                                      /**< [Out] Length of bytes available in \b ppEas buffer. */
    );

/**
 * \brief This command enables the password protection for EAS. The EAS password has to be transmitted before with
 * \ref phalICode_SetPassword "Set Password".
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_PasswordProtectEAS(
        void * pDataParams                                                      /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief This command enables the password protection for AFI. The AFI password has to be transmitted before with
 * \ref phalICode_SetPassword "Set Password".
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_PasswordProtectAFI(
        void * pDataParams                                                      /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief With this command, a new EAS identifier is stored in the corresponding configuration memory. If the EAS mode
 * is password protected the EAS password has to be transmitted before with \ref phalICode_SetPassword "Set Password".
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_WriteEASID(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t * pEasIdValue                                                   /**< [In] EAS ID; 16 bits. */
    );

/**
 * \brief Performs ReadEPC command. On this command, the label will respond with it's EPC data.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadEPC(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t ** ppEpc,                                                       /**< [Out] EPC data; 96 bits. */
        uint16_t * pEpcLen                                                      /**< [Out] Length of bytes available in \b ppEpc buffer. */
    );

/**
 * \brief Perform GetNXPSystemInformation command. This command allows for retrieving the NXP system information value from the VICC.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_GetNXPSystemInformation(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppSystemInfo,                                                /**< [Out] The NXP system information of the VICC. Refer specific
                                                                                 *         product data-sheet for more information.
                                                                                 *         Response will contain the below informations.
                                                                                 *              - Byte 1: 1 byte of Protection Pointer address
                                                                                 *              - Byte 2: 1 byte of Protection Pointer Conditions
                                                                                 *              - Byte 3: 1 byte of Lock Bit settings
                                                                                 *              - Byte 4: 4 bytes of Supported commands and Features
                                                                                 */
        uint16_t * pSystemInfoLen                                               /**< [Out] Length of bytes available in \b ppSystemInfo buffer. */
    );

/**
 * \brief Perform InventoryPageRead command. When receiving the Inventory Page Read request, the ICODE IC performs the same
 * as in the anti-collision sequence, with the difference that instead of the UID and the DSFID the requested memory content
 * is re-transmitted from the ICODE IC. \n
 * \n
 * If the Option flag is set to 0, N pages of data including page protection status (password protection condition) are
 * re-transmitted. If the option flag is set to 1, N pages (4 blocks = 16 byte) of data including page protection status
 * (password protection condition) and the part of the UID which is not part of the mask are re-transmitted.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_InventoryPageRead(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phalICode_Flags_Common "Common"
                                                                                 *          - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bPageNo,                                                        /**< [In] Block number of first page to read. */
        uint8_t bNoOfPages,                                                     /**< [In] Number of pages to read */
        uint8_t ** ppUid,                                                       /**< [Out] Received UID */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b ppUid buffer. */
        uint8_t ** ppData,                                                      /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b ppData buffer. */
    );

/**
 * \brief Perform FastInventoryPageRead command. When receiving the FAST INVENTORY PAGE READ command the ICODE IC behaves the
 * same as the INVENTORY PAGE READ command with the following exceptions:
 *
 * - The data rate in the direction ICODE Tag to the interrogator is twice that defined in ISO/IEC 15693-3 depending on
 *   the Datarate_flag 53 kbit (high data rate) or 13 kbit (low data rate).
 *
 * - The data rate from the interrogator to the ICODE Tag and the time between the rising edge of the EOF from the
 *   interrogator to the ICODE Tag remain unchanged (stay the same as defined in ISO/IEC 15693-3).
 *
 * - In the ICODE Tag to the interrogator direction, only the single sub-carrier mode is supported.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_FastInventoryPageRead(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phalICode_Flags_Common "Common"
                                                                                 *          - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bPageNo,                                                        /**< [In] Block number of first page to read. */
        uint8_t bNoOfPages,                                                     /**< [In] Number of pages to read */
        uint8_t ** ppUid,                                                       /**< [Out] Received UID */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b ppUid buffer. */
        uint8_t ** ppData,                                                      /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b ppData buffer. */
    );

/**
 * \brief Performs a GetRandomNumber command. On this command, the label will respond with a random number.
 * The received random number shall be used to diversify the password for the \ref phalICode_SetPassword "Set Password"
 * command.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_GetRandomNumber(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppRnd,                                                       /**< [Out] Random number; 16 bits. */
        uint16_t * pRndLen                                                      /**< [Out] Number of bytes in \b ppRnd buffer. */
    );

/**
 * \brief Performs SetPassword command. With this command the different passwords can be transmitted to the label.
 *
 * This command has to be executed just once for the related passwords if the label is powered.
 *
 * \verbatim
 * [XOR password calculation example]
 * pXorPwd[0] = pPassword[0] ^ pRnd[0];
 * pXorPwd[1] = pPassword[1] ^ pRnd[1];
 * pXorPwd[2] = pPassword[2] ^ pRnd[0];
 * pXorPwd[3] = pPassword[3] ^ pRnd[1];
 * \endverbatim
 *
 * \b Remark: This command can only be executed in addressed or selected mode except of Privacy Password.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_SetPassword(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPwdIdentifier,                                                 /**< [In] Password Identifier;
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_READ
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_WRITE
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_PRIVACY
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_DESTROY
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_EAS_AFI
                                                                                 */
        uint8_t * pXorPwd                                                       /**< [In] XOR Password; 32 bits. */
    );

/**
 * \brief Performs WritePassword command. With this command, a new password is written into the related memory. Note that the
 * old password has to be transmitted before with \ref phalICode_SetPassword "Set Password". The new password takes effect
 * immediately which means that the new password has to be transmitted with \ref phalICode_SetPassword "Set Password" to
 * get access to protected blocks/pages.
 * \b Remark: This command can only be executed in addressed or selected mode.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_WritePassword(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPwdIdentifier,                                                 /**< [In] Password Identifier;
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_READ
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_WRITE
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_PRIVACY
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_DESTROY
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_EAS_AFI
                                                                                 */
        uint8_t * pPwd                                                          /**< [In] Plain Password; 32 bits. */
    );

/**
 * \brief Performs LockPassword command. This command locks the addressed password. Note that the addressed password
 * has to be transmitted before with \ref phalICode_SetPassword "Set Password". A locked password can not be changed
 * any longer.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_LockPassword(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPwdIdentifier                                                  /**< [In] Password Identifier;
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_READ
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_WRITE
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_PRIVACY
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_DESTROY
                                                                                 *          - #PHAL_ICODE_SET_PASSWORD_EAS_AFI
                                                                                 */
    );

/**
 * \brief Performs Page protection command. This command changes the protection status of a page. Note that the related
 * passwords have to be transmitted before with \ref phalICode_SetPassword "Set Password" if the page is not public.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface
 *
 *  - Protection status options for the products that do not have pages characterized as high and Low (<b>Protection status bits definition</b>).
 *      - Protection status bits definition in plain password mode (**where x denotes L (Low) or H (High)**)
 *          | **Wx** | **Rx** | **32-bit Protection**                                                                                                                              |
 *          |:------:|:------:|:---------------------------------------------------------------------------------------------------------------------------------------------------|
 *          |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                                                       |
 *          |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protected by the Read password"                                              |
 *          |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protected by the Write password"                                                                |
 *          |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read protected by the Read password and Write protected by the Read and Write password" |
 *          \n
 *
 *          | **Wx** | **Rx** | **64-bit Protection**                                                                                             |
 *          |:------:|:------:|:------------------------------------------------------------------------------------------------------------------|
 *          |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                      |
 *          |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protection by Read plus Write password"     |
 *          |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protection by Read plus Write password"                        |
 *          |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read and Write protection by Read plus Write password" |
 *          \n
 *
 *      - Protection status bits definition in AES mode (**where x denotes L (Low) or H (High)**)
 *          | **Wx** | **Rx** | **Protection**                                                                                                                                                   |
 *          |:------:|:------:|:-----------------------------------------------------------------------------------------------------------------------------------------------------------------|
 *          |    0   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_PUBLIC "Public"                                                                                                                     |
 *          |    0   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_READ_PASSWORD "Read and Write protected: Mutual-Authentication with a key with read privilege is required"                |
 *          |    1   |    0   | \ref PHAL_ICODE_PROTECT_PAGE_WRITE_PASSWORD "Write protected: Mutual-Authentication with a key with write privilege is required"                                  |
 *          |    1   |    1   | \ref PHAL_ICODE_PROTECT_PAGE_READ_WRITE_PASSWORD_SEPERATE "Read and Write protected: Mutual-Authentication with a key with read and write privilege is required"  |
 * \n\n
 *
 *  - Extended Protection status options for the products that have pages characterized as high and Low (<b>Extended Protection status byte</b>)
 *    | **Bit**  | **Name** | **Value** |        **Description**        |
 *    |:--------:|:---------|:---------:|:------------------------------|
 *    | b1 (LSB) | RL       |     0     | Page L is not read protected  |
 *    |    ^     | ^        |     1     | Page L is read protected      |
 *    |    b2    | WL       |     0     | Page L is not write protected |
 *    |    ^     | ^        |     1     | Page L is write protected     |
 *    |    b3    |          |     0     | RFU                           |
 *    |    b4    |          |     0     | RFU                           |
 *    |    b5    | RH       |     0     | Page H is not read protected  |
 *    |    ^     | ^        |     1     | Page H is read protected      |
 *    |    b6    | WH       |     0     | Page H is not write protected |
 *    |    ^     | ^        |     1     | Page H is write protected     |
 *    |    b7    |          |     0     | RFU                           |
 *    | b8 (MSB) |          |     0     | RFU                           |
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ProtectPage(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPPAdd_PageNo,                                                  /**< [In] Page number to be protected in case of products that do not have pages
                                                                                 *        characterized as high and Low. \n
                                                                                 *        Block number to be protected in case of products that have pages
                                                                                 *        characterized as high and Low.
                                                                                 */
        uint8_t bProtectionStatus                                               /**< [In] Refer description for supported values. */
    );

/**
 * \brief Perform LockPageProtectionCondition command. This command permanently locks the protection status of a page.
 * Note that the related passwords have to be transmitted before with \ref phalICode_SetPassword "Set Password" if
 * the page is not public.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_LockPageProtectionCondition(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bPageNo                                                         /**< [In] Page number to be protected. */
    );

/**
 * \brief Perform GetMultipleBlockProtectionStatus command. This instructs the label to return the block protection
 * status of the requested blocks.
 *
 * \b Remark: If \c bBlockNo + \c bNoOfBlocks exceeds the total available number of user blocks, the number of received
 * status bytes is less than the requested number. This means that the last returned status byte corresponds to the
 * highest available user block.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface
 * Note: The memory should be created and sent to the ProtectionStates buffer.
 *
 * Each byte of the response will contain the below information.
 *  |    **Bit**    |         **Name**         | **Value** |           **Description**            |
 *  |:-------------:|:-------------------------|:---------:|:-------------------------------------|
 *  | b1 (LSB)      | Lock bit (WAC)           |     0     | Block is not locked                  |
 *  | ^             | ^                        |     1     | Block is locked (Lock Block command) |
 *  | b2            | Read password protected  |     0     | Disabled                             |
 *  | ^             | ^                        |     1     | Enabled                              |
 *  | b3            | Write password protected |     0     | Disabled                             |
 *  | ^             | ^                        |     1     | Enabled                              |
 *  | b4            | Page protection lock     |     0     | Not Locked                           |
 *  | ^             | ^                        |     1     | Locked                               |
 *  | b5 - b8 (MSB) |                          |     0     |                                      |
 *  | **WAC - Write AccessCondition** ||||
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - If \b bNoOfBlocks is zero.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_GetMultipleBlockProtectionStatus(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bBlockNo,                                                       /**< [In] First Block number. */
        uint8_t bNoOfBlocks,                                                    /**< [In] First Block number. */
        uint8_t * pProtectionStates,                                            /**< [Out] Protection states of requested blocks. Response will be
                                                                                 *         repeated based on \b bNoOfBlocks information.
                                                                                 */
        uint16_t * pNumReceivedStates                                           /**< [Out] Length of bytes available in \b pProtectionStates buffer. */
    );

/**
 * \brief Performs Destroy command. This command permanently destroys the label.
 *
 * The Destroy password has to be transmitted before with \ref phalICode_SetPassword "Set Password".\n
 * \b Remark:
 *  - This command is irreversible and the label will never respond to any command again.
 *  - This command can only be executed in addressed or selected mode.
 *
 * Note: This command is not valid for iCode DNA product as the Destroy feature is part of Mutual
 * Authentication command (refer \ref phalICode_AuthenticateMAM).
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_Destroy(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t * pXorPwd                                                       /**< [In] XOR Password; 32 bits. Pass the password for the ICODE products that
                                                                                 *        supports and NULL for the products that do not support.
                                                                                 */
    );

/**
 * \brief Performs EnablePrivacy command. This command instructs the label to enter privacy mode.
 *
 * In privacy mode, the label will only respond to \ref phalICode_GetRandomNumber "Challenge" and \ref phalICode_SetPassword "Set Password" commands.
 * To get out of the privacy mode, the Privacy password has to be transmitted before with \ref phalICode_SetPassword "Set Password".
 *
 * Note: This command is not valid for iCode DNA product as the privacy enabling disabling feature is part of Mutual
 * Authentication command (refer \ref phalICode_AuthenticateMAM "Authenticate MAM").
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_EnablePrivacy(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t * pXorPwd                                                       /**< [In] XOR Password; 32 bits. Pass the password for the iCode products
                                                                                 *        that supports and NULL for the products that do not support.
                                                                                 */
    );

/**
 * \brief Perform 64-BitPasswordProtection command. This instructs the label that both of the Read and Write passwords
 * are required for protected access.
 *
 * Note that both the Read and Write passwords have to be transmitted before with \ref phalICode_SetPassword "Set Password".
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_64BitPasswordProtection(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption                                                         /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
    );

/**
 * \brief When receiving the STAY QUIET PERSISTENT command, the label IC enters the persistent quiet state and
 * will not send back a response.
 *
 * \b Remark: The STAY QUIET PERSISTENT command provides the same behavior as the mandatory STAY QUIET command
 * with the only difference at a reset (power off). The label IC will turn to the ready state, if the power off
 * time is exceeding the persistent time.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_StayQuietPersistent(
        void * pDataParams                                                      /**< [In] Pointer to this layers parameter structure. */
    );

/**
 * \brief Performs ReadSignature command. On this command, the label will respond with the signature value.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadSignature(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppSign,                                                      /**< [Out] The originality signature returned by the VICC. */
        uint16_t * pSignLen                                                     /**< [Out] Length of originality signature buffer. */
    );

/**
 * \brief Reads a multiple 4 byte(s) data from the mentioned configuration block address. Here the starting address of the
 * configuration block should be given in the parameter wBlockAddr and the number of blocks to read from the starting
 * block should be given in the parameter bNoOfBlocks.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bBlockAddr,                                                     /**< [In] Configuration block address. */
        uint8_t bNoOfBlocks,                                                    /**< [In] The n block(s) to read the configuration data. */
        uint8_t ** ppData,                                                      /**< [Out] Multiple of 4 (4 * No Of Blocks) byte(s) of data read from the
                                                                                 *         mentioned configuration block address.
                                                                                 */
        uint16_t * pDataLen                                                     /**< [Out]  Number of received configuration data bytes. */
    );

/**
 * \brief Writes a 4 byte data to the mentioned configuration block address.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_WriteConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bBlockAddr,                                                     /**< [In] Configuration block address. */
        uint8_t * pData                                                         /**< [In] A 4 byte data to be written to the mentioned configuration
                                                                                 *        block address.
                                                                                 */
    );

/**
 * \brief Enables the random ID generation in the tag. This interfaces is used to instruct the tag to generate
 * a random number in privacy mode.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_PickRandomID(
        void * pDataParams                                                      /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Provides the tag tamper status.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface
 *
 * Response will contain the below information.
 *  - Byte1: TagTamper Current State
 *  - Byte2: TagTamper Stored State
 *  | **TT Current State**                       | **TT Stored State**                        |
 *  |:-------------------------------------------|:-------------------------------------------|
 *  | O: Open                                    | O: Open                                    |
 *  | C: Closed                                  | C: Closed                                  |
 *  | I: Invalid                                 | E: Electrical Error                        |
 *  | E: Electrical Error                        | Z: Backup Management Error - Invalid Data  |
 *  | 0: Feature not active or info not provided | 0: Feature not active or info not provided |
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If any of the DataParams are null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - For the option values that are not supported.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadTT(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag as per ISO15693
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON.
                                                                                 */
        uint8_t ** ppResponse,                                                  /**< [Out] Two bytes of Tag Tamper status. */
        uint16_t * pRspLen                                                      /**< [Out]  Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Performs Parameter Request command. When receiving VICC PARAMETER REQUEST, NTAG5 returns all supported bit rates
 * and timing information.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ParameterRequest(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pBitRate,                                                     /**< [Out] One byte buffer containing the supported BitRates.
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_26KBPS_BOTH_DIRECTIONS
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VCD_VICC
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VCD_VICC
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VCD_VICC
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VICC_VCD
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VICC_VCD
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VICC_VCD
                                                                                 */
        uint8_t * pTiming                                                       /**< [Out] One byte buffer containing the supported timing information.
                                                                                 *          - #PHAL_ICODE_PARAMETERS_TIMING_320_9_US
                                                                                 *          - #PHAL_ICODE_PARAMETERS_TIMING_160_5_US
                                                                                 *          - #PHAL_ICODE_PARAMETERS_TIMING_80_2_US
                                                                                 *          - #PHAL_ICODE_PARAMETERS_TIMING_SAME_BOTH_DIRECTIONS
                                                                                 */
    );

/**
 * \brief Performs Parameter Select command. PARAMETER SELECT command is used to activate one bit rate combination and the T1
 * timing indicated in PARAMETER REQUEST response. Only one option in each direction shall be chosen. After the response to PARAMETER
 * SELECT command, new parameters are valid.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ParameterSelect(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bBitRate,                                                       /**< [In] One byte buffer containing the supported BitRates.
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_26KBPS_BOTH_DIRECTIONS
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VCD_VICC
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VCD_VICC
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VCD_VICC
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VICC_VCD
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VICC_VCD
                                                                                 *          - #PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VICC_VCD
                                                                                 */
        uint8_t bTiming                                                         /**< [In] One byte buffer containing the supported timing information.
                                                                                 *          - #PHAL_ICODE_PARAMETERS_TIMING_320_9_US
                                                                                 *          - #PHAL_ICODE_PARAMETERS_TIMING_160_5_US
                                                                                 *          - #PHAL_ICODE_PARAMETERS_TIMING_80_2_US
                                                                                 *          - #PHAL_ICODE_PARAMETERS_TIMING_SAME_BOTH_DIRECTIONS
                                                                                 */
    );

/**
 * \brief Performs a SRAM Read command.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - If \b bNoOfBlocks is zero.
 *          - If \b bNoOfBlocks + \b bBlockNo is greater than 3Fh.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_ReadSRAM(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag. Supported values are,
                                                                                 *          - #PHAL_ICODE_OPTION_OFF
                                                                                 *          - #PHAL_ICODE_OPTION_ON
                                                                                 */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data to be read. */
        uint8_t bNumBlocks,                                                     /**< [In] Total number of block to read. */
        uint8_t * pData,                                                        /**< [Out] Information received from VICC
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, Block Security Status information is not available. Only block data
                                                                                 *            is available. Format will be 4 byte data 1, 4 byte data 2, ... 4 byte data N
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, both Block Security Status information and Block Data is available.
                                                                                 *            Format of the response will be Status 1 + 4 byte data 1, Status 2 + 4 byte data 2, ...
                                                                                 *            Status N + 4 byte data N, Where 1, 2 ... N is the block number.
                                                                                 */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Performs a SRAM Write command.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER
 *          - If the buffers are null.
 *          - If \b bNoOfBlocks is zero.
 *          - If \b bNoOfBlocks + \b bBlockNo is greater than 3Fh.
 *          - For the option values that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_WriteSRAM(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the write operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
        uint8_t bBlockNo,                                                       /**< [In] Block number from where the data should be written. */
        uint8_t bNumBlocks,                                                     /**< [In] Total number of block to be written. */
        uint8_t * pData,                                                        /**< [In] Information to be written to VICC. */
        uint16_t wDataLen                                                       /**< [In] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Performs a I2CM Read command. This command is used to read from any I2C slave connected to NTAG5 I2C Host.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PARAMETER_SIZE       If the value of \b wDataLen is higher than 256.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_I2CMRead (
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bI2CParam,                                                      /**< [In] I2C Slave address from which the data should be read and the information
                                                                                 *        to set the Stop bit.
                                                                                 *          Bits 0 - 6: 7-bit I2C slave address of connected slave
                                                                                 *          Bit 7     : Disable STOP condition
                                                                                 *                      0b: STOP condition will be generated at the end of transaction
                                                                                 *                      1b: STOP condition will be omitted at the end of transaction
                                                                                 */
        uint16_t wDataLen,                                                     /**< [In] Total Number of data bytes to be read. If 1 byte has to be read then the
                                                                                 *        length will be 1.
                                                                                 */
        uint8_t * pData                                                         /**< [Out] Information read from the VICC. */
    );

/**
 * \brief Performs a I2CM Write command. This command is used to write to any I2C slave connected to NTAG5 I2C Host.
 *
 * Flag can be set using \ref phalICode_SetConfig "SetConfig" utility interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PARAMETER_SIZE       If the value of \b wDataLen is higher than 256.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_I2CMWrite(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option flag, Supported values are,
                                                                                 *          - If #PHAL_ICODE_OPTION_OFF, the VICC shall return its response when it has completed the write operation
                                                                                 *            starting after t1nom [4352/fc (320,9 us)] + a multiple of 4096/fc (302 us) with a total tolerance
                                                                                 *            of &plusmn;32/fc and latest after 20 ms upon detection of the rising edge of the EOF of the VCD request.
                                                                                 *          - If #PHAL_ICODE_OPTION_ON, the VICC shall wait for the reception of an EOF from the VCD and upon such
                                                                                 *            reception shall return its response.
                                                                                 */
        uint8_t * pData,                                                        /**< [In] Information to be written to the VICC. */
        uint16_t wDataLen                                                       /**< [In] Total Number of data bytes to be written. If 1 byte has to be written then the
                                                                                 *        length will be 1.
                                                                                 */
    );


/**
 * end of group phalICode_Custom
 * @}
 */





/* iCode Utility commands ------------------------------------------------------------------------------------------ */
/**
 * \defgroup phalICode_Utils Utilities
 * \brief Describes the utility interfaces. These interfaces are actually not of ISO15693 or any of the ICode
 * product families. These are generic interfaces as part of reader library for internal purpose.
 * @{
 */

/** \defgroup phalICode_Utility_Defines Defines
 * \brief Describes the options to be used for Utility interfaces.
 * @{
 */

/** \defgroup phalICode_Utility_Defines_Configuration Configuration
 * \brief Describes the supported options for \ref phalICode_SetConfig "SetConfig" and \ref phalICode_GetConfig "GetConfig" interfaces.
 * @{
 */
#define PHAL_ICODE_CONFIG_FLAGS                                         0x00U   /**< Option to Get / Set the Flags information. */
#define PHAL_ICODE_CONFIG_ADD_INFO                                      0x01U   /**< Option to Get the additional information. */
#define PHAL_ICODE_CONFIG_TIMEOUT_US                                    0x02U   /**< Option to Get the timeout value in milliseconds. */
#define PHAL_ICODE_CONFIG_TIMEOUT_MS                                    0x03U   /**< Option to Get the timeout value in microseconds. */
#define PHAL_ICODE_CONFIG_ENABLE_BUFFERING                              0x04U   /**< Option to Get / Set the buffering of response data to be enabled or disabled.
                                                                                 *   By default this flag will be disabled.
                                                                                 */
#define PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US                          0x05U   /**< Option to Get / Set the timeout value in microseconds for Challenge command. */
#define PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS                          0x06U   /**< Option to Get / Set the timeout value in milliseconds for Challenge command. */

#define PHAL_ICODE_DISABLE                                              0x00U   /**< Option to disable the buffering of response data. */
#define PHAL_ICODE_ENABLE                                               0x01U   /**< Option to enable the buffering of response data. */
#define PHAL_ICODE_TIMEOUT_SHORT_US                                     324U    /**< Timeout T1 in us (ISO/IEC15693, 9.1.4.). */
#define PHAL_ICODE_TIMEOUT_LONG_US                                      20000U  /**< Extended timeout for some commands. */
/**
 * end of group phalICode_Utility_Defines_Configuration
 * @}
 */

/** \defgroup phalICode_Utility_Defines_TagType TagType
 * \brief Describes the supported options for \ref phalICode_GetTagType "GetTagType" interface.
 * @{
 */
#define PHAL_ICODE_TAG_TYPE_UNKNOWN                                     0xFFFFU /**< Option to Get the tag type as Unknown. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI                                   0x0001U /**< Option to Get the tag type as ICode SLI. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI_S                                 0x0002U /**< Option to Get the tag type as ICode SLI-S. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI_L                                 0x0003U /**< Option to Get the tag type as ICode SLI-L. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLIX                                  0x5001U /**< Option to Get the tag type as ICode SLIX. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLIX_S                                0x5002U /**< Option to Get the tag type as ICode SLIX-S. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLIX_L                                0x5003U /**< Option to Get the tag type as ICode SLIX-L. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI_X2                                0x0801U /**< Option to Get the tag type as ICode SLIX2. */
#define PHAL_ICODE_TAG_TYPE_ICODE_SLI_X3                                0x2001U /**< Option to Get the tag type as ICode SLIX3. */
#define PHAL_ICODE_TAG_TYPE_ICODE_DNA                                   0x1801U /**< Option to Get the tag type as ICode DNA. */
#define PHAL_ICODE_TAG_TYPE_ICODE_NTAG_5                                0x5801U /**< Option to Get the tag type as ICode NTAG5. */
/**
 * end of group phalICode_Utility_Defines_TagType
 * @}
 */

/**
 * end of group phalICode_Utility_Defines
 * @}
 */

/**
 * \brief Gets the UID of the tag.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_GetSerialNo(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t ** ppUid,                                                       /**< [Out] 8 byte UID of the tag. */
        uint16_t * pUidLen                                                      /**< [Out] Length of the UID buffer. */
    );

/**
 * \brief Sets the UID of the tag.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_SetSerialNo(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t * pUid,                                                         /**< [In] 8 byte UID of the tag. */
        uint8_t bUidLen                                                         /**< [In] Length of the UID buffer. */
    );

/**
 * \brief Get the configuration settings.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    For configuration values (\b wConfig) that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_GetConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration to read.
                                                                                 *          \arg #PHAL_ICODE_CONFIG_FLAGS
                                                                                 *          \arg #PHAL_ICODE_CONFIG_ADD_INFO
                                                                                 *          \arg #PHAL_ICODE_CONFIG_TIMEOUT_US
                                                                                 *          \arg #PHAL_ICODE_CONFIG_TIMEOUT_MS
                                                                                 *          \arg #PHAL_ICODE_CONFIG_ENABLE_BUFFERING
                                                                                 *          \arg #PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US
                                                                                 *          \arg #PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS
                                                                                 */
        uint16_t * pValue                                                       /**< [In] The value for the mentioned configuration information in wConfig parameter.
                                                                                 *          - #PHAL_ICODE_CONFIG_FLAGS Refer below set of flags
                                                                                 *              - \ref phalICode_Flags_Common "Common"
                                                                                 *              - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *              - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 *          \n
                                                                                 *          - #PHAL_ICODE_CONFIG_ADD_INFO, any additional information like error code will be available.
                                                                                 *          \n
                                                                                 *          - #PHAL_ICODE_CONFIG_ENABLE_BUFFERING
                                                                                 *              - #PHAL_ICODE_DISABLE
                                                                                 *              - #PHAL_ICODE_ENABLE
                                                                                 */
    );

/**
 * \brief Set the configuration settings.
 *
 * \b NOTE: Both the flags #PHAL_ICODE_FLAG_DATA_RATE and #PHAL_ICODE_FLAG_FAST_DATA_RATE should not be combined,
 *          it should be passed separately along with other flag.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    For configuration values (\b wConfig) that are not supported.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phalICode_SetConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration to update.
                                                                                 *          \arg #PHAL_ICODE_CONFIG_FLAGS
                                                                                 *          \arg #PHAL_ICODE_CONFIG_ENABLE_BUFFERING
                                                                                 *          \arg #PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US
                                                                                 *          \arg #PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS
                                                                                 */
        uint16_t wValue                                                         /**< [In] The value for the mentioned configuration information in wConfig parameter.
                                                                                 *          - #PHAL_ICODE_CONFIG_FLAGS Refer below set of flags
                                                                                 *              - \ref phalICode_Flags_Common "Common"
                                                                                 *              - \ref phalICode_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *              - \ref phalICode_Flags_INV_Set "Inventory_Set"
                                                                                 *          \n
                                                                                 *          - #PHAL_ICODE_CONFIG_ENABLE_BUFFERING
                                                                                 *              - #PHAL_ICODE_DISABLE
                                                                                 *              - #PHAL_ICODE_ENABLE
                                                                                 */
    );

/**
 * \brief Get the type of Tag
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PHAL_ICODE_TAG_TYPE_UNKNOWN For tag types that are not supported.
 */
phStatus_t phalICode_GetTagType(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t * pTagType                                                     /**< [In] The type of ICode tag.
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_UNKNOWN
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_SLI
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_SLI_S
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_SLI_L
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_SLIX
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_SLIX_S
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_SLIX_L
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_SLI_X2
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_SLI_X3
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_DNA
                                                                                 *          - #PHAL_ICODE_TAG_TYPE_ICODE_NTAG_5
                                                                                 */
    );

/**
 * end of group phalICode_Utils
 * @}
 */

/**
 * end of group phalICode
 * @}
 */
#endif /* NXPBUILD__PHAL_ICODE */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHALICODE_H */
