/*
 * Copyright 2013 - 2020, 2022, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic MIFARE(R) Ultralight and NTAG Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALMFUL_H
#define PHALMFUL_H

#include <ph_Status.h>
#include <phpalMifare.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHAL_MFUL_SW
/***************************************************************************************************************************************/
/* Software DataParams and Initialization Interface.                                                                                   */
/***************************************************************************************************************************************/

/** \defgroup phalMful_Sw Component : Software
 * \brief Software implementation of the MIFARE Ultralight / NTAG commands. Here the MIFARE Ultralight / NTAG commands
 * are framed and exchanged to PICC.
 * @{
 */

#define PHAL_MFUL_SW_ID                                     0x01    /**< ID for Software MIFARE Ultralight layer */

/** \brief Data structure for MIFARE Ultralight / NTAG Software layer implementation  */
typedef struct
{
    uint16_t wId;                                                   /**< Layer ID for this component, NEVER MODIFY! */
    void * pPalMifareDataParams;                                    /**< Pointer to palMifare parameter structure. */
    void * pKeyStoreDataParams;                                     /**< Pointer to phKeystore parameter structure. */
    void * pCryptoDataParams;                                       /**< Pointer to phCrypto data parameters structure. */
    void * pCryptoRngDataParams;                                    /**< Pointer to the parameter structure of the CryptoRng layer. */
    uint8_t bCMACReq;                                               /**< Get/Set Config to include CMAC for required Commands */
    uint16_t wCmdCtr;                                               /**< Command count within transaction. */
    uint8_t bAuthMode;                                              /**< Authentication PWD [0x1B] or AES [0x1A] */
    uint8_t bAdditionalInfo;                                        /**< Specific error codes for MFUL generic errors or to get the response length of some commands. */
} phalMful_Sw_DataParams_t;

/**
 * \brief Initialization API for MIFARE Ultralight / NTAG software component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *          - If the input size doest not match the DataParams size of this component.
 *          - If any of the DataParams are null.
 */
phStatus_t phalMful_Sw_Init(
        phalMful_Sw_DataParams_t * pDataParams,                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                 /**< [In] Specifies the size of the data parameter structure. */
        void * pPalMifareDataParams,                                /**< [In] Pointer to palMifare parameter structure. */
        void * pKeyStoreDataParams,                                 /**< [In] Pointer to phKeystore parameter structure. */
        void * pCryptoDataParams,                                   /**< [In] Pointer to phCrypto data parameters structure. */
        void * pCryptoRngDataParams                                 /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
    );

/**
 * end of phalMful_Sw
 * @}
 */
#endif /* NXPBUILD__PHAL_MFUL_SW */















/************************************************************************************************************************************/
/* MIFARE Ultralight / NTAG Generic functions                                                                                       */
/************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_MFUL

/** \addtogroup phalMful
 * \brief This is only a wrapper layer to abstract the different MIFARE Ultralight / NTAG implementations.
 * With this wrapper it is possible to support more than one MIFARE Ultralight / NTAG implementation
 * in parallel, by adapting this wrapper.
 *
 * \note
 *      - Below are the initializations supported by this component.
 *          - \ref phalMful_Sw_Init "Software Initialization"
 *      - MIFARE Ultralight AES product supports exchanging CMAC for commands post successful Authentication.
 *          - If SEC_MSG_ACT bit is set, enable CMAC feature by configuring \ref PHAL_MFUL_CMAC_STATUS "Configure CMAC"
 *            identifier using \ref phalMful_SetConfig "Set Configuration" interface and value as \ref PH_ON "Enable".
 *            Post configuring this identifier in library, library will append CMAC for each command exchange and
 *            will verify the CMAC received form PICC also.
 *          - If SEC_MSG_ACT bit is not set, disable CMAC feature by configuring \ref PHAL_MFUL_CMAC_STATUS "Configure CMAC"
 *            identifier using \ref phalMful_SetConfig "Set Configuration" interface and value as \ref PH_OFF "Disable".
 *            Post configuring this identifier in library, library will not append CMAC for each command exchange.
 *
 * @{
 */


/**
 * \defgroup phalMful_CommonDefs Defines
 * @{
 */

/**
 * \defgroup phalMful_CommonDefs_BufferSize Buffer Size
 * @{
 */
#define PHAL_MFUL_READ_BLOCK_LENGTH                             16U /**< Length of MIFARE(R) Ultralight / NTAG read data block. */
#define PHAL_MFUL_WRITE_BLOCK_LENGTH                            4U  /**< Length of MIFARE(R) Ultralight / NTAG write data block. */
#define PHAL_MFUL_COMPWRITE_BLOCK_LENGTH                        16U /**< Length of MIFARE(R) Ultralight / NTAG compatibility write data block. */
#define PHAL_MFUL_COUNTER_RD_VALUE_LENGTH                       3U  /**< Length of MIFARE(R) Ultralight / NTAG Read counter. */
#define PHAL_MFUL_COUNTER_WR_VALUE_LENGTH                       4U  /**< Length of MIFARE(R) Ultralight / NTAG Write counter. */
#define PHAL_MFUL_PACK_LENGTH                                   2U  /**< Length of MIFARE(R) Ultralight / NTAG PACK (Password ACK) block. */
#define PHAL_MFUL_VERSION_LENGTH                                8U  /**< Length of MIFARE(R) Ultralight / NTAG version information. */
#define PHAL_MFUL_SIG_LENGTH_32                                32U  /**< Length of MIFARE(R) Ultralight / NTAG signature information. */
#define PHAL_MFUL_SIG_LENGTH_48                                48U  /**< Length of MIFARE(R) Ultralight / NTAG signature information. */
#define PHAL_MFUL_DES_BLOCK_SIZE                                8U  /**< Length of MIFARE(R) Ultralight / NTAG DES block. */
#define PHAL_MFUL_AES_BLOCK_SIZE                                16U /**< Length of MIFARE(R) Ultralight / NTAG AES block. */
#define PHAL_MFUL_DES_KEY_LENGTH                                16U /**< Length of MIFARE(R) Ultralight / NTAG key. */
/**
 * end of group phalMful_CommonDefs_BufferSize
 * @}
 */

/**
 * \defgroup phalMful_CommonDefs_Diversification Diversification
 * \brief Diversification options to be used with below mentioned interface(s)
 *  - \ref phalMful_UlcAuthenticate "Authentication using DES Key"
 *  - \ref phalMful_AuthenticateAES "Authentication using AES Key"
 *  - \ref phalMful_PwdAuth "Authentication using Password and Pack combination"
 * @{
 */
#define PHAL_MFUL_CMD_UL_AUTHENTICATE_DIV_OFF                   0U  /**< Option to indicate the diversification is disabled. */
#define PHAL_MFUL_CMD_UL_AUTHENTICATE_DIV_ON                    1U  /**< Option to indicate the diversification is enabled. */
/**
 * end of group phalMful_CommonDefs_AuthType
 * @}
 */

/**
 * end of group phalMful_CommonDefs
 * @}
 */

#ifdef NXPBUILD__PH_CRYPTOSYM
/**
 * \brief Perform MIFARE(R) Ultralight-C Authenticate command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_AUTH_ERROR           Authentication failure.
 * \retval #PH_ERR_PROTOCOL_ERROR       Unexpected PICC response length or unexpected first byte received from PICC.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_UlcAuthenticate(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                            /**< [In] Option for specifying the diversification input.
                                                                     *           Only valid for Sam AV3
                                                                     *          - \ref PHAL_MFUL_CMD_UL_AUTHENTICATE_DIV_OFF "Disable Diversification"
                                                                     *          - \ref PHAL_MFUL_CMD_UL_AUTHENTICATE_DIV_ON "Enable Diversification"
                                                                     */
        uint16_t wKeyNo,                                            /**< [In] Key number to be used in authentication. */
        uint16_t wKeyVer,                                           /**< [In] Key version to be used in authentication. */
        uint8_t * pDivInput,                                        /**< [In] Diversification input for key diversification. (1 to 31 byte(s) input). */
        uint8_t bDivInputLen                                        /**< [In] Length of diversification input. */
    );

/**
 * \brief Perform MIFARE(R) Ultralight / NTAG AES Authentication with PICC.
 * This will be using the AES128 keys and will generate and verify the contents based
 * on generic AES algorithm.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_AUTH_ERROR           Authentication failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_AuthenticateAES (
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                            /**< [In] Option for specifying the diversification input.
                                                                     *          - \ref PHAL_MFUL_CMD_UL_AUTHENTICATE_DIV_OFF "Disable Diversification"
                                                                     *          - \ref PHAL_MFUL_CMD_UL_AUTHENTICATE_DIV_ON "Enable Diversification"
                                                                     */
        uint16_t wKeyNo,                                            /**< [In] Key number to be used in authentication. */
        uint16_t wKeyVer,                                           /**< [In] Key version to be used in authentication. */
        uint8_t bRamKeyNo,                                          /**< [In] Key number of Destination Key where the computed session key will be stored.
                                                                     *        To be used for SAM AV3 only.
                                                                     */
        uint8_t bRamKeyVer,                                         /**< [In] Key version of Destination Key where the computed session key will be stored.
                                                                     *        To be used for SAM AV3 only.
                                                                     */
        uint8_t bKeyNoCard,                                         /**< [In] Key number on card. */
        uint8_t * pDivInput,                                        /**< [In] Diversification input. Can be NULL. */
        uint8_t bDivLen                                             /**< [In] Length of diversification input max 31B. */
    );
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * \brief Perform MIFARE(R) Ultralight / NTAG Read command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_Read(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bAddress,                                           /**< [In] Address on PICC to read from. */
        uint8_t * pData                                             /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the PICC. */
    );

/**
 * \brief Perform MIFARE(R) Ultralight / NATG Write command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_Write(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bAddress,                                           /**< [In] Address on PICC to write to. */
        uint8_t * pData                                             /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the PICC. */
    );

/**
 * \brief Perform NATG Fast Write command which writes 64 bytes from 0xF0 to 0xFF address blocks.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_FastWrite(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pData                                             /**< [In] Buffer containing the block data to be written to the PICC. */
    );
/**
 * \brief Perform MIFARE(R) Ultralight / NATG Compatibility-Write command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_CompatibilityWrite(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bAddress,                                           /**< [In] Address on PICC to write to. */
        uint8_t * pData                                             /**< [In] pData[#PHAL_MFUL_COMPWRITE_BLOCK_LENGTH] containing block to be written to the PICC. */
    );

/**
 * \brief Perform MIFARE(R) Ultralight Increment count command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_IncrCnt(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCntNum,                                            /**< [In] One byte counter number 00 to 02 */
        uint8_t * pCnt                                              /**< [In] pCnt[4] counter value LSB first. Only first three data bytes are used by the PICC. The 4th byte is ignored */
    );

/**
 * \brief Perform MIFARE(R) Ultralight / NATG Read count command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_ReadCnt(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCntNum,                                            /**< [In] One byte counter number 00 to 02 */
        uint8_t * pCntValue                                         /**< [Out] pCntValue[3] counter value. LSB first. Three bytes counter value returned from PICC. */
    );

/**
 * \addtogroup phalMful_CommonDefs_Diversification
 * @{
 */
#define PHAL_MFUL_CMD_PWD_AUTH_DIV_OFF                      0x00U   /**< Option to indicate the diversification is disabled. */
#define PHAL_MFUL_CMD_PWD_AUTH_DIV_OFF_INPUT_NOT_AVAILABLE  0x01U   /**< Option to indicate the diversification is disabled but the diversification should be performed
                                                                     *   without DivInput. In this cases the Div algorithm will be CMAC(Kx, Const | Padding)
                                                                     */
#define PHAL_MFUL_CMD_PWD_AUTH_DIV_ON                       0x02U   /**< Option to indicate the diversification is enabled and the diversification should be performed
                                                                     *   with DivInput. In this cases the Div algorithm will be CMAC(Kx, Const | M | Padding), where
                                                                     *   M is diversification input (1 - 31 bytes)
                                                                     */
/**
 * end of group phalMful_CommonDefs_Diversification
 * @}
 */

/**
 * \brief Perform MIFARE(R) Ultralight / NATG Password auth command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_AUTH_ERROR           Authentication failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_PwdAuth(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                            /**< [In] One of the below options should be used. This option byte is supported for Sam component only.
                                                                     *          - \ref PHAL_MFUL_CMD_PWD_AUTH_DIV_OFF "Diversification Disabled"
                                                                     *          - \ref PHAL_MFUL_CMD_PWD_AUTH_DIV_OFF_INPUT_NOT_AVAILABLE "Diversification Enabled with no Input"
                                                                     *          - \ref PHAL_MFUL_CMD_PWD_AUTH_DIV_ON "Diversification Enabled"
                                                                     */
        uint8_t bKeyNo,                                             /**< [In] Key number to be used in Sam hardware. Only valid for Sam AV3 NonX and X mode. */
        uint8_t bKeyVer,                                            /**< [In] Key version to be used in Sam hardware. Only valid for Sam AV3 NonX and X mode. */
        uint8_t * pDivInput,                                        /**< [In] Diversification input. Only valid for Sam AV3 NonX and X mode. */
        uint8_t bDivInputLen,                                       /**< [In] Diversification input length. Only valid for Sam AV3 NonX and X mode. */
        uint8_t * pPwd,                                             /**< [In] Four byte array of type uint8_t containing four password bytes. \n
                                                                     *          - For software layer, the user has to pass the password information.\n
                                                                     *          - For Sam AV3 NonX layer, the parameter will be updated with the password generated
                                                                     *             by Sam hardware.
                                                                     */
        uint8_t * pPack                                             /**< [Out] Two byte array of type uint8_t containing two password acknowledge bytes \n
                                                                     *         For software and Sam AV3 NonX layer, the user will be provided with the Pack
                                                                     *         information returned by the card.
                                                                     */
    );

/**
 * \brief Perform MIFARE(R) Ultralight / NATG get version command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_GetVersion(
            void * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
            uint8_t * pVersion  /**< [Out] An eight byte array containing version information bytes */
        );

/**
 * \brief Perform MIFARE(R) Ultralight / NATG Fast read command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_FastRead(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t  bStartAddr,                                        /**< [In] One byte start address */
        uint8_t bEndAddr,                                           /**< [In] One byte end address */
        uint8_t ** pData,                                           /**< [Out] Pointer to the data read from the card. The bytes are stored on the HAL's Rx buffer. */
        uint16_t * wNumBytes                                        /**< [Out] Contains number of bytes read from the card */
    );
/**
 * \brief Perform NTAG Sector Select within PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_SectorSelect(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bSecNo                                              /**< [In] The number of the sector to be selected. */
    );

/**
 * \brief Perform MIFARE(R) Ultralight / NATG read signature command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_ReadSign(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bAddr,                                              /**< [In] Value is always 00. Present for forward compatibility reasons */
        uint8_t ** pSignature                                       /**< [Out] Pointer to a 32/48 byte signature read from the card */
    );

/**
 * \brief Perform MIFARE(R) Ultralight EV1 check tearing event command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_ChkTearingEvent(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCntNum,                                            /**< [In] Value specifying the counter number 00 to 02 */
        uint8_t * pValidFlag                                        /**< [Out] Address of one byte containing the valid flag byte */
    );
/**
 * \brief Perform MIFARE(R) Ultralight / NATG Write Signature command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_WriteSign(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bAddress,                                           /**< [In] Relative page location of the signature part to be written. */
        uint8_t * pSignature                                        /**< [Out] 4 bytes of signature value to be written to the specified relative page address. */
    );

/**
 * \addtogroup phalMful_CommonDefs
 * @{
 */

/**
 * \defgroup phalMful_CommonDefs_Sign Lock Signature
 * \brief Options to be used with \ref phalMful_LockSign "Signature Locking" interface
 * @{
 */
#define PHAL_MFUL_CMD_UL_LOCK_SIGN_UNLOCK                       0U  /**< Option to unlock the signature. */
#define PHAL_MFUL_CMD_UL_LOCK_SIGN_LOCK                         1U  /**< Option to temporary lock the signature. */
#define PHAL_MFUL_CMD_UL_LOCK_SIGN_PERMANENT_LOCK               2U  /**< Option to permanently lock the signature. */
/**
 * end of group phalMful_CommonDefs_Sign
 * @}
 */

/**
 * end of group phalMful_CommonDefs
 * @}
 */

/**
 * \brief Perform MIFARE(R) Ultralight / NATG Lock Signature command with PICC.
 * This command locks the signature temporarily or permanently based on the information provided in the API. The locking and unlocking of the
 * signature can be performed using this command if the signature is not locked or temporary locked. If the signature is permanently locked,
 * then the unlocking cannot be done.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_LockSign(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bLockMode                                           /**< [In] Configuration to Lock, Unlock and Permanently Lock the signature data.
                                                                     *          - \ref PHAL_MFUL_CMD_UL_LOCK_SIGN_UNLOCK "Un-Lock Signature"
                                                                     *          - \ref PHAL_MFUL_CMD_UL_LOCK_SIGN_LOCK "Lock Signature"
                                                                     *          - \ref PHAL_MFUL_CMD_UL_LOCK_SIGN_PERMANENT_LOCK "Lock Signature Permanently"
                                                                     */
    );

/**
 * \brief Perform MIFARE(R) Ultralight Virtual Card Select command with PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_VirtualCardSelect(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pVCIID,                                           /**< [In] The Virtual Card Installation Identifier data for selecting the VC. */
        uint8_t bVCIIDLen,                                          /**< [In] Length of the VCIID data. */
        uint8_t * pVCTID                                            /**< [Out] 1 byte of Virtual card type identifier data. */
    );

/**
 * \brief Perform NTAG Tag Temper Status Read.
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMful_ReadTTStatus(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t  bAddr,                                             /**< [In] Value is always 00. Present for forward compatibility reason */
        uint8_t * pData                                             /**< [Out] 5 bytes of Tag Temper Status. */
    );





/***************************************************************************************************************************************/
/* MIFARE Ultralight / NTAG utility functions                                                                                          */
/***************************************************************************************************************************************/
/**
 * \addtogroup phalMful_CommonDefs
 * @{
 */

/**
 * \defgroup phalMful_CommonDefs_Config Configuration
 * \brief Configuration Identifier Options for \ref phalMful_GetConfig "Get Config" and \ref phalMful_SetConfig "Set Config" interfaces.
 * @{
 */
#define PHAL_MFUL_CMAC_STATUS                               0xA1U   /**< Option to Get/Set for inclusion of CMAC data for command and response. */
#define PHAL_MFUL_ADDITIONAL_INFO                           0xA2U   /**< Option to Get/Set additional info of a generic error or length of response data. */
/**
 * end of group phalMful_CommonDefs_Config
 * @}
 */

/**
 * end of group phalMful_CommonDefs
 * @}
 */

/**
 * \brief Perform a GetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS      If DataParams is null.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    If the configuration identifier is not supported.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_GetConfig(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                           /**< [In] Configuration to get the status of CMAC inclusion in command / response.
                                                                     *          - \ref #PHAL_MFUL_CMAC_STATUS "CMAC Status"
                                                                     *          - \ref #PHAL_MFUL_ADDITIONAL_INFO "Additional Info"
                                                                     */
        uint16_t * pValue                                           /**< [Out] The value for the mentioned configuration. */
    );

/**
 * \brief Perform a SetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS      If DataParams is null.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    If the configuration identifier is not supported.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_SetConfig(
        void * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                           /**< [In] Configuration to set the status of CMAC inclusion in command / response.
                                                                     *          - \ref #PHAL_MFUL_CMAC_STATUS "CMAC Status"
                                                                     *          - \ref #PHAL_MFUL_ADDITIONAL_INFO "Additional Info"
                                                                     */
        uint16_t wValue                                             /**< [In] The value for the mentioned configuration. */
    );

/**
 * \brief Calculate SUNCMAC.
 *
 * \note
 *      For the parameter pInData, Caller has to remove the ASCII formated data before passing to this interface. \n
 *      If the information is having TT status, User has to convert the ASCII bytes representation to NonASCII format. \n
 *          - 43h should be converted to 03h \n
 *          - 4Fh should be converted to 0Fh \n
 *          - 49h should be converted to 09h \n
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffer is null.
 * \retval #PH_ERR_INTEGRITY_ERROR      CMAC verification failure.
 * \retval XXXX
 *                                      - Depending on status codes return by PICC.
 *                                      - Other Depending on implementation and underlying component.
 */
phStatus_t phalMful_CalculateSunCMAC(
        void * pDataParams,                                         /**< [In] Pointer to this layers parameter structure. */
        uint16_t wKeyNo,                                            /**< [In] Key number to be used from software or hardware KeyStore. */
        uint16_t wKeyVer,                                           /**< [In] Key version to be used from software or hardware KeyStore. */
        uint8_t * pInData,                                          /**< [In] Input Data for which the MAC to be calculated. */
        uint16_t wInDataLen,                                        /**< [In] Length of bytes available in InData buffer. */
        uint8_t * pRespMac                                          /**< [Out] Computed SUN CMAC. */
    );

/**
 * end of phalMful
 * @}
 */
#endif /* NXPBUILD__PHAL_MFUL */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHALMFUL_H */
