/*
 * Copyright 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hal DUT specific declaration commands.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7463 $
 * $Date: 2025-08-29 15:56:52 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *   Created On 6 July, 2022
 */

#ifndef PHHALHW_DUT_CMD_H
#define PHHALHW_DUT_CMD_H

#include <ph_Status.h>

#ifdef NXPBUILD__PHHAL_HW_DUT
#include <phhalHw.h>

#define PHHALHW_TAG_EEPROM_USER_AREA                            0x0010U /**< Tag value to indiate EEPROM Read from User Area. */
#define PHHALHW_TAG_EEPROM_SECURE_LIB_AREA                      0x0011U /**< Tag value to indiate EEPROM Read from Secure Lib Area. */
#define PHHALHW_TAG_EEPROM_IC_CONFIG_AREA                       0x0012U /**< Tag value to indiate EEPROM Read from IC Config Area. */
#define PHHALHW_TAG_EEPROM_PROTOCOL_AREA                        0x0020U /**< Tag value to indiate EEPROM Read from RF Protocol Area. */

/** \defgroup phhalHw_DUT_Cmd Commands
 * \brief Hal DUT specific declaration for the following features
 * - Contactless. Refer \ref phhalHw_DUT_Cmd_RDOperations "Reader Operations" and \ref phhalHw_DUT_Cmd_HALOperations "HAL Operations"
 *   commands.
 * @{
 */

/* Initialization Commands ------------------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_Init Command_Initialization
 * \brief Reader Initialization commands.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_Init_CommandCodes Command_Codes
 * \brief Reader Initialization Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_INIT_CONTACTLESS                           0xB001U /**< Initialize the reader to perform Contactless operation. */
/**
 * end of phhalHw_DUT_Cmd_Init_CommandCodes group
 * @}
 */

/**
 * \brief Performs the initialization of different Components.
 *
 * To initialize the communication interface of the reader below information should be used for \b wCmd parameter.
 * - \ref PHHAL_HW_DUT_INIT_CONTACTLESS "Contactless via RF Interface"
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_InitReader(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wCmd                                                   /**< [IN] Command to initialize the Reader. Refer description for
                                                                         *        values and combination to use.
                                                                         */
    );
/**
 * end of phhalHw_DUT_Cmd_Init group
 * @}
 */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_VERSION
/* Version Commands -------------------------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_Version Command_Version
 * \brief Version Command of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_Version_CommandCodes Command_Codes
 * \brief Version Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_GETVERSION                             0x3201U /**< Complete Version information of NFC Controller's firmware or Host
                                                                         *   Controller's firmware.
                                                                         */
#define PHHAL_HW_DUT_CMD_GETVERSION_NFC                         0x3234U /**< Complete Version information of NFC Controller IC firmware. */
#define PHHAL_HW_DUT_CMD_GETVERSION_DATE_TIME                   0x3240U /**< Get the date time of compilation the NFC Controller's firmware or
                                                                         *   Host Controller's firmware.
                                                                         */
#define PHHAL_HW_DUT_CMD_GETVERSION_READER_IC_TYPE              0x3250U /**< Get the FrontEnd (Reader IC) or NFC Controller IC for which the code
                                                                         *   is compiled.
                                                                         */
/**
 * end of phhalHw_DUT_Cmd_Version_CommandCodes group
 * @}
 */

/**
 * \brief The function gets the Version of the Firmware of DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_GetVersion(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pProductVer,                                          /**< [OUT] Product Version. */
        uint16_t * pMajor,                                              /**< [OUT] Major Version. */
        uint8_t * pMinor,                                               /**< [OUT] Minor Version. */
        uint16_t * pPatch_Dev,                                          /**< [OUT] Development/Patch Version. */
        uint8_t * pVersionString,                                       /**< [OUT] Complete version information in form of ASCII bytes.
                                                                         *         Should be minimum 32 bytes long.
                                                                         */
        uint16_t * pVersionLen                                          /**< [In] Length of bytes allocated for \b pVersionString buffer.
                                                                         *   [Out] Length of bytes available in \b pVersionString buffer.
                                                                         */
    );

/**
 * \brief The function gets the Version of the NFC Controller connected to the DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_GetVersion_NFC(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pProductVer,                                          /**< [OUT] Product Version. */
        uint16_t * pMajor,                                              /**< [OUT] Major Version. */
        uint8_t * pMinor,                                               /**< [OUT] Minor Version. */
        uint16_t * pPatch_Dev,                                          /**< [OUT] Development/Patch Version. */
        uint8_t * pVersionString,                                       /**< [OUT] Complete version information in form of ASCII bytes.
                                                                         *         Should be minimum 32 bytes long.
                                                                         */
        uint16_t * pVersionLen                                          /**< [In] Length of bytes allocated for \b pVersionString buffer.
                                                                         *   [Out] Length of bytes available in \b pVersionString buffer.
                                                                         */
    );

/**
 * \brief The function gets the date and time at which the firmware code is compiled.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_GetVersion_CompiledInfo(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pResponse,                                            /**< [OUT] Complete version information in form of ASCII bytes.
                                                                         *         Should be minimum 128 bytes long.
                                                                         */
        uint16_t * pRspLen                                              /**< [In] Length of bytes allocated for \b pResponse buffer.
                                                                         *   [Out] Length of bytes available in \b pResponse buffer.
                                                                         */
    );

/**
 * \brief The function gets the type of NFC Reader IC being used.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_GetVersion_ReaderIcType(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pResponse,                                            /**< [OUT] NFCController / FrontEnd type in form of ASCII.
                                                                         *         Should be minimum 16 bytes long.
                                                                         */
        uint16_t * pRspLen                                              /**< [In] Length of bytes allocated for \b pResponse buffer.
                                                                         *   [Out] Length of bytes available in \b pResponse buffer.
                                                                         */
    );
/**
 * end of phhalHw_DUT_Cmd_Version group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_VERSION */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_RDOPS
/* Reader Operation Commands ----------------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_RDOperations Command_ReaderOps
 * \brief Reader Operation Command of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_RdOps_CommandCodes Command_Codes
 * \brief Reader Operation Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_RD_OPS_GET_READER_STATUS               0xA006U /**< Gets the Status of Reader that included Reader Firmware Version. */
#define PHHAL_HW_DUT_CMD_RD_OPS_READ_REGISTER                   0xA007U /**< Perform Register Read. */
#define PHHAL_HW_DUT_CMD_RD_OPS_WRITE_REGISTER                  0xA008U /**< Perform Register Write. */
#define PHHAL_HW_DUT_CMD_RD_OPS_FIELD_ON                        0xA009U /**< Switch on the RF-field. */
#define PHHAL_HW_DUT_CMD_RD_OPS_FIELD_OFF                       0xA00AU /**< Switch off the RF-field. */
#define PHHAL_HW_DUT_CMD_RD_OPS_FIELD_RESET                     0xA00BU /**< Perform RF-Reset. */
#define PHHAL_HW_DUT_CMD_RD_OPS_READ_EEPROM                     0xA00EU /**< Perform EEPROM Read. */
#define PHHAL_HW_DUT_CMD_RD_OPS_WRITE_EEPROM                    0xA00FU /**< Perform EEPROM Write. */
/**
 * end of phhalHw_DUT_Cmd_RdOps_CommandCodes group
 * @}
 */

/**
 * \brief Gets the Status and Firmware Version of the Reader.
 *  - A total of 9 bytes will be returned or lesser. Below is the description of bytes.
 *      - Byte[0]: ReaderIC.
 *          - 0x01: RC523
 *          - 0x02: SAM
 *          - 0x03: RC663
 *          - 0x04: PN7642
 *      - Byte[1]: Configuration
 *      - Byte[2]: PCSC Mode
 *      - Byte[3]: Pegoda Reader's Firmware Major Version
 *      - Byte[4]: Pegoda Reader's Firmware Minor Version
 *      - Byte[5]: Pegoda Reader's Firmware Build Version
 *      - Byte[6]: Unknown
 *      - Byte[7]: Unknown
 *      - Byte[8]: Unknown
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_GetReaderStatus(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t ** ppReaderStatus,                                      /**< [OUT] The Status and Firmware Version information returned by the reader. */
        uint16_t * pReaderStatusLen                                     /**< [OUT] Length of bytes available in \b ppReaderStatus buffer. */
    );

/**
 * \brief Reads the Register information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_Read_Register(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint32_t dwRegAddr,                                             /**< [IN] The Register address from where the information should
                                                                         *        be read.
                                                                         */
        uint32_t * pRegData                                             /**< [OUT] The Register data for the provided Address. */
    );

/**
 * \brief Reads the Register information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_Write_Register(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint32_t dwRegAddr,                                             /**< [IN] The Register address to which the information should be
                                                                         *        written.
                                                                         */
        uint32_t dwRegData                                              /**< [IN] The Register data for the provided Address. */
    );

/**
 * \brief Switches ON the RF Field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_FieldOn(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [IN] Pointer to this layer parameter structure. */
    );

/**
 * \brief Switches OFF the RF Field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_FieldOff(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [IN] Pointer to this layer parameter structure. */
    );

/**
 * \brief Resets the RF Field
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_FieldReset(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [IN] Pointer to this layer parameter structure. */
    );

/**
 * \brief Reads the EEPROM information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_Read_EEPROM(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [IN] Options to be used for EEPROM Reading.
                                                                         *          - 0x0000: Only \b dwStartAddr and \b dwLength information
                                                                         *            will be exchagned to Reader.
                                                                         *          - \b dwStartAddr and \b dwLength along with the specified information
                                                                         *             to this paramter will be exchagned to Reader. These values are
                                                                         *             specific to Pegoda - 3 Reader.
                                                                         *              - 0x0010: Read information from User area
                                                                         *              - 0x0011: Read information from Secure Lib area
                                                                         *              - 0x0012: Read information from IC Configuration area
                                                                         *              - 0x0020: Read information from RF Configuration area
                                                                         *          - Values other that above specified ones.
                                                                         */
        uint32_t dwAddr_ProtIndex,                                      /**< [IN] The EEPROM address or Protocol Index from where the information
                                                                         *        should be read. One of the following,
                                                                         *          - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
                                                                         *            EEPROM address. This is supported by Pegoda - 3 Reader
                                                                         *            only.
                                                                         *          - If \b wOption = 0x0020, then 1 byte Protocol
                                                                         *            index of the RF Configuration. This is supported
                                                                         *            by Pegoda - 3 Reader only.
                                                                         *          - If \b wOption = 0x0000, then 4 byte EEPROM address.
                                                                         *            Reader other than Pegoda 3 reader.
                                                                         *
                                                                         */
        uint32_t dwNoOfBytesToRead,                                     /**< [IN] Number of bytes to be read from EEPROM starting from address specified.
                                                                         *        If Reader is Pegoda -3 and \b wOption = 0x0020, then zero should be
                                                                         *        provided.
                                                                         */
        uint8_t ** ppResponse,                                          /**< [OUT] EEPROM information from the mentioned start address (\b dwStartAddr)
                                                                         *         until the requested length. Make sure the buffer is allocated for the
                                                                         *         requested number of bytes (\b dwLength).
                                                                         */
        uint16_t * pRespLen                                             /**< [OUT] Length of bytes available in \b pResponse buffer */
    );

/**
 * \brief Writes the EEPROM information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_Write_EEPROM(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [IN] Options to be used for EEPROM Writing.
                                                                         *          - 0x0000: Only \b dwStartAddr and \b dwLength information
                                                                         *            will be exchagned to Reader.
                                                                         *          - \b dwStartAddr and \b dwLength along with the specified information
                                                                         *             will be exchagned to Reader. These values are specific to Pegoda - 3
                                                                         *             Reader.
                                                                         *              - 0x0010: Read information from User area
                                                                         *              - 0x0011: Read information from Secure Lib area
                                                                         *              - 0x0012: Read information from IC Configuration area
                                                                         *              - 0x0020: Read information from RF Configuration area
                                                                         */
        uint32_t dwStartAddr,                                           /**< [IN] The EEPROM address information to be written.
                                                                         *        One of the following,
                                                                         *          - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
                                                                         *            EEPROM address. This is supported by Pegoda - 3 Reader
                                                                         *            only.
                                                                         *          - If \b wOption = 0x0020, zero should be provided.
                                                                         *          - If \b wOption = 0x0000, then 4 byte EEPROM address.
                                                                         *            Reader other than Pegoda 3 reader.
                                                                         *
                                                                         */
        uint8_t * pData,                                                /**< [IN] Information to be written to EERPOM.
                                                                         *        One of the following,
                                                                         *          - If \b wOption = 0x0000, 0x0010, 0x0011 or 0x0012, then
                                                                         *            EEPROM value should be provided.
                                                                         *          - If \b wOption = 0x0020, then 1 byte Protocol index followed
                                                                         *            1 byte Register address followed by 4 byte Register Value.
                                                                         *            This is supported by Pegoda - 3 Reader only. \n
                                                                         *            Protocol Index1 ( 1 Byte) ||
                                                                         *            Reg Addr1 ( 1 Byte) || Reg Value1 ( 4 Bytes) ||
                                                                         *            Reg Addr2 ( 1 Byte) || Reg Value2 ( 4 Bytes) ||
                                                                         *            ...
                                                                         *            Reg AddrN ( 1 Byte) || Reg ValueN ( 4 Bytes)
                                                                         *
                                                                         */
        uint16_t wDataLen                                               /**< [IN] Length of bytes available in \b pData buffer. */
    );

/**
 * end of phhalHw_DUT_Cmd_RDOperations group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_RDOPS */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_HAL
/* HAL Operation Commands -------------------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_HALOperations Command_HALOps
 * \brief HAL Operation Command of HAL DUT Command dispatcher frame.
 *
 * \note
 *      - These command codes will be used to communicate to a contactless device connected via RF only.
 *      - To perfom PICC / Tag exchanges via I2C use I2C sepcific commands.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_HALOps_CommandCodes Command_Codes
 * \brief HAL Operation Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_HAL_EXCHANGE                           0xB002U /**< Perform Data Exchange with PICC. */
#define PHHAL_HW_DUT_CMD_HAL_SET_CONFIG                         0xB003U /**< Set configuration parameter for the firmware. */
#define PHHAL_HW_DUT_CMD_HAL_GET_CONFIG                         0xB004U /**< Get configuration parameter of the firmware. */
#define PHHAL_HW_DUT_CMD_HAL_APPLY_PROT_SETTINGS                0xB005U /**< Perform Apply protocol settings. */
#define PHHAL_HW_DUT_CMD_HAL_WAIT                               0xB006U /**< Blocks until the time given by \c wTimeout elapsed. */
#define PHHAL_HW_DUT_CMD_HAL_MFC_AUTH                           0xB007U /**< Perform MIFARE(R) Classic contactless IC Authentication using a key. */
#define PHHAL_HW_DUT_CMD_HAL_EXECUTE                            0xB008U /**< Execute a RC command. */
#define PHHAL_HW_DUT_CMD_HAL_MFC_AUTH_KEY_NO                    0xB009U /**< Perform MIFARE(R) Classic Authentication using a key number. */
#define PHHAL_HW_DUT_CMD_HAL_TRANSMIT                           0xB00AU /**< Perform Data Transmit to PICC. */
#define PHHAL_HW_DUT_CMD_HAL_RECEIVE                            0xB00BU /**< Perform Data Receive from PICC. */
#define PHHAL_HW_DUT_CMD_HAL_I18000P3M3_INVENTORY               0xB00CU /**< Perform Inventory command for 18000p3m3. */
#define PHHAL_HW_DUT_CMD_HAL_I18000P3M3_RESUME_INVENTORY        0xB00DU /**< Perform Inventory command for 18000p3m3. */
/**
 * end of phhalHw_DUT_Cmd_HALOps_CommandCodes group
 * @}
 */

/**
 * \brief Perform Data Exchange with a PICC. This command is used to exchange information to PICC and at the same
 *  time receives the payload from PICC if available including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to PICC and also receives the payload from PICC
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of PICC. Here the information is not
 *        exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for PICC (if any). Here the
 *        information is not exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for PICC (if any). Here the complete
 *        buffered information is exchanged to PICC and also receives the information from PICC if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_TXCHAINING "Chained Transmission"
 *          - To send any pending payload to DUT.
 *          - If DUT requires more data, then DUT will respond with \ref PH_ERR_SUCCESS_CHAINING "Require More Data" as status for the current
 *            exchange call. The interface should be called for this flag until DUT returns \ref PH_ERR_SUCCESS "Complete" as status.
 *        \n
 *      - \ref PH_EXCHANGE_RXCHAINING "Chained Reception"
 *          - To receive any pending payload from DUT.
 *          - To start using this flag, DUT will return \ref PH_ERR_SUCCESS_CHAINING "More Data Available" as status during previous reception.
 *            Payload should be received until \ref PH_ERR_SUCCESS "Complete" as status is returned by DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Exchange(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint8_t * pTxBuff,                                              /**< [In] Payload information to be exchanged / buffers based on
                                                                         *        \b wOption information.
                                                                         */
        uint16_t wTxLen,                                                /**< [In] Length of bytes available in \b pTxBuff buffer. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by PICC. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_SetConfig(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier. */
        uint16_t wValue                                                 /**< [In] Configuration Value. */
    );

/**
 * \brief Get configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_GetConfig(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier. */
        uint16_t * pValue                                               /**< [Out] Configuration Value. */
    );

/**
 * \brief Apply protocol settings.
 * To keep the current card type, use #PHHAL_HW_CARDTYPE_CURRENT for bCardType.\n
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_INVALID_PARAMETER \b bCardType is invalid or not supported.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_ApplyProtocolSettings(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bCardType                                               /**< [In] Type of card for which the HAL should be configured for.
                                                                         *        One for the below values.
                                                                         *          - \ref PHHAL_HW_CARDTYPE_CURRENT "Current Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_ISO14443A "ISO14443 Type-A Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_ISO14443B "ISO14443 Type-B Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_FELICA "FELICA Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_ISO15693 "ISO15693 Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_I18000P3M3 "ISO18000 p3m3 Configuration"
                                                                         */
    );

/**
 * \brief Blocks until the time given by \b wTimeout elapsed.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_INVALID_PARAMETER \b bUnit is invalid.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Wait(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bUnit,                                                  /**< [In] Unit of given timeout value. Supported values are
                                                                         *        - \ref PHHAL_HW_TIME_MICROSECONDS "MicroSeconds"
                                                                         *        - \ref PHHAL_HW_TIME_MILLISECONDS "MilliSeconds"
                                                                         */
        uint16_t wTimeout                                               /**< [In] Timeout value based on \b bUnit information. */
    );

/**
 * \brief Perform MIFARE(R) Classic Authentication using a key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_INVALID_PARAMETER \b bKeyType is invalid.
 * \retval #PH_ERR_IO_TIMEOUT Error in Authentication.
 * \retval #PH_ERR_AUTH_ERROR Error in Authentication.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_MfcAuthenticate(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bBlockNo,                                               /**< [In] Block Number on card to Authenticate to. */
        uint8_t bKeyType,                                               /**< [In] Key Type to use for Authentication.
                                                                         *          - \ref PHHAL_HW_MFC_KEYA "Key-Type A"
                                                                         *          - \ref PHHAL_HW_MFC_KEYB "Key-Type B"
                                                                         */
        uint8_t * pKey,                                                 /**< [In] Key to be used in authentication. */
        uint8_t * pUid                                                  /**< [In] 4 byte Serial number of current cascade level */
    );

/**
 * \brief Perform Command Execution with a PICC. This command is used to exchange information to PICC and at the same
 *  time receives the payload from PICC if available including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to PICC and also receives the payload from PICC
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of PICC. Here the information is not
 *        exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for PICC (if any). Here the
 *        information is not exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for PICC (if any). Here the complete
 *        buffered information is exchanged to PICC and also receives the information from PICC if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_TXCHAINING "Chained Transmission"
 *          - To send any pending payload to DUT.
 *          - If DUT requires more data, then DUT will respond with \ref PH_ERR_SUCCESS_CHAINING "Require More Data" as status for the current
 *            exchange call. The interface should be called for this flag until DUT returns \ref PH_ERR_SUCCESS "Complete" as status.
 *        \n
 *      - \ref PH_EXCHANGE_RXCHAINING "Chained Reception"
 *          - To receive any pending payload from DUT.
 *          - To start using this flag, DUT will return \ref PH_ERR_SUCCESS_CHAINING "More Data Available" as status during previous reception.
 *            Payload should be received until \ref PH_ERR_SUCCESS "Complete" as status is returned by DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Execute(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bCmd,                                                   /**< [IN] Command to be executed on the Reader IC. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint8_t * pTxBuff,                                              /**< [In] Payload information to be exchanged / buffers based on
                                                                         *        \b wOption information.
                                                                         */
        uint16_t wTxLen,                                                /**< [In] Length of bytes available in \b pTxBuff buffer. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by PICC. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief Perform MIFARE(R) Classic Authentication using a key number.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_INVALID_PARAMETER \b bKeyType, \b wKeyNo or \b wKeyVersion is invalid.
 * \retval #PH_ERR_IO_TIMEOUT Error in Authentication.
 * \retval #PH_ERR_AUTH_ERROR Error in Authentication.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_MfcAuthenticateKeyNo(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bBlockNo,                                               /**< [In] Block Number on card to Authenticate to. */
        uint8_t bKeyType,                                               /**< [In] Key Type to use for Authentication.
                                                                         *          - \ref PHHAL_HW_MFC_KEYA "Key-Type A"
                                                                         *          - \ref PHHAL_HW_MFC_KEYB "Key-Type B"
                                                                         */
        uint16_t wKeyNo,                                                /**< [In] Key number to be used in authentication. */
        uint16_t wKeyVer,                                               /**< [In] Key version to be used in authentication. */
        uint8_t * pUid                                                  /**< [In] 4 byte Serial number of current cascade level */
    );

/**
 * \brief Perform Transmission of data with a PICC. This command is used to exchange information to PICC and at the same
 *  time receives the payload from PICC if available including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to PICC and also receives the payload from PICC
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of PICC. Here the information is not
 *        exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for PICC (if any). Here the
 *        information is not exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for PICC (if any). Here the complete
 *        buffered information is exchanged to PICC and also receives the information from PICC if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_TXCHAINING "Chained Transmission"
 *          - To send any pending payload to DUT.
 *          - If DUT requires more data, then DUT will respond with \ref PH_ERR_SUCCESS_CHAINING "Require More Data" as status for the current
 *            exchange call. The interface should be called for this flag until DUT returns \ref PH_ERR_SUCCESS "Complete" as status.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Transmit(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint8_t * pTxBuff,                                              /**< [In] Payload information to be exchanged / buffers based on
                                                                         *        \b wOption information.
                                                                         */
        uint16_t wTxLen                                                 /**< [In] Length of bytes available in \b pTxBuff buffer. */
    );

/**
 * \brief Perform Reception of Data from a PICC. This command is used to exchange information to PICC and at the same
 *  time receives the payload from PICC if available including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to PICC and also receives the payload from PICC
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of PICC. Here the information is not
 *        exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for PICC (if any). Here the
 *        information is not exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for PICC (if any). Here the complete
 *        buffered information is exchanged to PICC and also receives the information from PICC if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_RXCHAINING "Chained Reception"
 *          - To receive any pending payload from DUT.
 *          - To start using this flag, DUT will return \ref PH_ERR_SUCCESS_CHAINING "More Data Available" as status during previous reception.
 *            Payload should be received until \ref PH_ERR_SUCCESS "Complete" as status is returned by DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Receive(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by PICC. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief ISO 18000p3m3 Inventory command.
 *
 *  Note: This function does
 *  ISO18000p3m3 BeginRound
 *  Sends an ACK
 *  Optionally sends a ReqRN
 *  Stores the information related to tag reply in the HAL RX Buffer
 *  Continues with NextSlot or else pauses and returns based on the option specified
 *  Can return multi card information based on the configuration option and the size of HAL RX Buffer.
 *  The bTSprocessing (Time slot processing behavior) can take the following values
 *      - #PHHAL_HW_I18000P3M3_GET_MAX_RESPS - Gets responses from more than one slot limited by the Rx Buffer size and number of slots specified in begin round.
 *      - #PHHAL_HW_I18000P3M3_ONE_TS_ONLY - Gets response for only one time slot. Here the number of time slots should be zero in the begin round command.
 *      - #PHHAL_HW_I18000P3M3_GET_TAG_HANDLE - Gets response for only one time slot. Also send a ReqRN to get the tag handle for this slot. Here the number of time
 *        slots should be zero in the begin round command.
 *
 *  The response is a Pointer to a buffer containing the result of the inventory.
 *  The buffer may contain the result of one or more timeslots. Each timeslot result has the following structure:
 *  1 Byte time slot status
 *  (
 *      0 ... Tag response available. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field' are present.
 *      1 ... Tag response available. In addition to three field above, 'Tag handle field' also present.
 *      2 ... No tag replied in timeslot. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field', 'Tag Handle' are not present.
 *      3 ... Two or more tags responded in the timeslot. (Collision). Tag Reply Length, valid bits in last byte and tag reply field not present.
 *  )
 *  1 Byte 'Tag reply' Length (1-66)
 *  1 Byte Valid bits in last Byte (0-7, 0 means all bits are valid)
 *  0-66 Bytes Tag reply
 *  0 or 2 Bytes Handle of the tag, in case the field
 *
 * A typical sequence in which this API is called is given below
 *  - \ref phbalReg_SerialWin_Init "BAL initialization"
 *  - \ref phhalHw_DUT_Init "HAL initialization"
 *  - \ref phhalHw_ApplyProtocolSettings "Load ISO 1800p3m3 protocol" with Card Type as #PHHAL_HW_CARDTYPE_I18000P3M3
 *  - \ref phhalHw_FieldOn "Turn ON the field"
 *  - \ref phhalHw_Wait "Wait for initial 5000us guard time"
 *  - \ref phhalHw_I18000p3m3Inventory "Start inventory"
 *  - Use \ref phpalI18000p3m3 and \ref phalI18000p3m3 APIs to further communicate with the tag.
 *  - \ref phhalHw_FieldOff "Turn OFF the field"
 *  - Termination of the application flow.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval #PH_ERR_IO_TIMEOUT When there is no tag in the field.
 * \retval #PH_ERR_INVALID_PARAMETER If wrong value specified for bNumValidBits or pBeginRound.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_I18000p3m3Inventory(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pSelCmd,                                              /**< [In] ISO 18000p3m3 Select command frame. */
        uint8_t bSelCmdLen,                                             /**< [In] Select command length in bytes- 1 to 39 bytes. */
        uint8_t bNumValidBitsinLastByte,                                /**< [In] Number of valid bits in last byte of \b pSelCmd. */
        uint8_t * pBeginRndCmd,                                         /**< [In] ISO 18000p3m3 BeginRound command frame. This is 17bits
                                                                         *        i.e., 3 bytes are expected. CRC5 should not be provided.
                                                                         */
        uint8_t bTSprocessing,                                          /**< [In] TimeSlot processing behavior. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by ISO18000 tag. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief ISO 18000p3m3 resume inventory command.
 *
 * Note: This function is to be called after \ref phhalHw_I18000p3m3Inventory "I18000p3m3 Inventory".
 * This internally calls the ISO 18000p3m3 NextSlot command once or multiple times to get responses from
 * remaining slots.
 *
 * Resume inventory also returns the response similar to \ref phhalHw_I18000p3m3Inventory "I18000p3m3 Inventory".
 *
 * A typical sequence in which this API is called is given below
 *  - \ref phbalReg_SerialWin_Init "BAL initialization"
 *  - \ref phhalHw_DUT_Init "HAL initialization"
 *  - \ref phhalHw_ApplyProtocolSettings "Load ISO 1800p3m3 protocol" with Card Type as #PHHAL_HW_CARDTYPE_I18000P3M3
 *  - \ref phhalHw_FieldOn "Turn ON the field"
 *  - \ref phhalHw_Wait "Wait for initial 5000us guard time"
 *  - \ref phhalHw_I18000p3m3Inventory "Start inventory"
 *  - Store the responses received for the tags.
 *  - If inventory has to be continued then issue the \ref phhalHw_I18000p3m3ResumeInventory "ResumeInventory" command
 *  - Use \ref phpalI18000p3m3 and \ref phalI18000p3m3 APIs to further communicate with the tag.
 *  - \ref phhalHw_FieldOff "Turn OFF the field"
 *  - Termination of the application flow.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval #PH_ERR_IO_TIMEOUT When there is no tag in the field.
 * \retval #PH_ERR_USE_CONDITION If called when the phhalHw_I18000p3m3 was not called previously with bTSprocessing = PHHAL_HW_I18000P3M3_GET_MAX_RESPS.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_I18000p3m3ResumeInventory(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided ISO18000 tag. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * end of phhalHw_DUT_Cmd_HALOperations group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_HAL */































/**
 * end of phhalHw_DUT_Cmd group
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_DUT */

#endif /* PHHALHW_DUT_CMD_H */
