/*
 * Copyright 2013, 2015, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Command functions for Rc523 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_RC523_CMD_H
#define PHHALHW_RC523_CMD_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef  NXPBUILD__PHHAL_HW_RC523

/** \defgroup phhalHw_Rc523_Cmd Commands
* \brief Supported ReaderIC commands.
* @{
*/

/**
* \brief Transfers either 25 bytes into internal buffer or returns 25 bytes from internal buffer.
*
* If \c bDataLength equals zero, 25 bytes from the internal buffer are put into \c pDataOut.\n
* If \c bDataLength is 25, 25 bytes from \c pDataIn are copied into the internal buffer.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \c bDataLength is neither zero nor 25.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rc523_Cmd_Mem(
                                 phhalHw_Rc523_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                 uint8_t * pDataIn,                         /**< [In] 25 bytes In-data or ignored if \c bDataLength equals zero. */
                                 uint8_t bDataLength,                       /**< [In] Data length. Either zero or 25. */
                                 uint8_t * pDataOut                         /**< [Out] 25 bytes Out-data or ignored if \c bDataLength is nonzero. */
                                 );

/**
* \brief Stores data for \ref phhalHw_Rc523_Cmd_Autocoll command into internal buffer.
*
* \b Note: Only applicable if in Target mode (e.g. #PHHAL_HW_CARDTYPE_I18092MPT).
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \c bDataLength is neither zero nor 25.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rc523_Cmd_Config(
                                    phhalHw_Rc523_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                                    uint8_t * pSensRes,                         /**< [In] 2 bytes. */
                                    /** [In] 3 bytes; The first NfcId1 byte is fixed to \c 08h and the check byte is calculated automatically. */
                                    uint8_t * pNfcId1,
                                    uint8_t SelRes,                             /**< [In] 1 byte. */
                                    /** [In] 2 bytes (shall be 01h, FEh) + 6 bytes NfcId2 + 8 bytes Pad + 2 bytes SystemCode. */
                                    uint8_t * pPollingResp,
                                    uint8_t bNfcId3                             /**< [In] 1 byte. */
                                    );

/**
* \brief Generates a 10 byte random ID.
*
* The generated RandomId is stored in the internal buffer (the first 10 bytes get overridden).\n
* It can be retrieved using the function \ref phhalHw_Rc523_Cmd_Mem.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rc523_Cmd_GenerateRandomId(
    phhalHw_Rc523_DataParams_t * pDataParams    /**< [In] Pointer to this layer's parameter structure. */
    );

/**
* \brief Calculates a CRC over the input data and returns the result.
*
* Note: The CRC calculation parameters are taken from the respective registers.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rc523_Cmd_CalcCrc(
                                     phhalHw_Rc523_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                     uint8_t * pData,                           /**< [In] CRC input data. */
                                     uint16_t wDataLength,                      /**< [In] Length of input data. */
                                     uint16_t * pCrcResult                      /**< [Out] Calculated CRC. */
                                     );

/**
* \brief Transmit bytes to card without starting receiver afterwards.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rc523_Cmd_Transmit(
                                      phhalHw_Rc523_DataParams_t * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                      uint16_t wOption,                         /**< [In] Option parameter. */
                                      uint8_t * pTxBuffer,                      /**< [In] Data to transmit. */
                                      uint16_t wTxLength                        /**< [In] Length of data to transmit. */
                                      );

/**
* \brief Start receiver and wait until reception or timeout.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_IO_TIMEOUT No response received.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rc523_Cmd_Receive(
                                     phhalHw_Rc523_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                     uint16_t wOption,                          /**< [In] Option parameter. */
                                     uint8_t ** ppRxBuffer,                     /**< [Out] Pointer to received data. */
                                     uint16_t * pRxLength                       /**< [Out] Number of received data bytes. */
                                     );

/**
* \brief Handles Felica polling or MIFARE anticollision.
*
* Only applicable if in Target mode (e.g. #PHHAL_HW_CARDTYPE_I18092MPT).\n
* \b Note: Ensure that the correct configuration has been loaded using the \ref phhalHw_Rc523_Cmd_Config command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_IO_TIMEOUT No response received.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rc523_Cmd_Autocoll(
                                      phhalHw_Rc523_DataParams_t * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                      uint8_t ** ppRxBuffer,                    /**< [Out] Pointer to received data. */
                                      uint16_t * pRxLength                      /**< [Out] Number of received data bytes. */
                                      );

/**
* \brief Performs a Softreset.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Rc523_Cmd_SoftReset(
                                       phhalHw_Rc523_DataParams_t * pDataParams /**< [In] Pointer to this layer's parameter structure. */
                                       );

/** @} */
#endif /* NXPBUILD__PHHAL_HW_RC523 */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_RC523_CMD_H */
